/*
 * radius.c	Functions to send/receive radius packets.
 *
 * Version:	@(#)radius.c  2.35  19-Jul-1999  miquels@cistron.nl
 *
 */

char radius_sccsid[] =
"@(#)radius.c 	2.35 Copyright 1998-1999 Cistron Internet Services B.V.";

#include	"autoconf.h"

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<time.h>
#include	<ctype.h>

#include	"radiusd.h"

#if HAVE_MALLOC_H
#  include	<malloc.h>
#endif

/*
 *	Make sure our buffer is aligned.
 */
static int	i_send_buffer[1024];
static char	*send_buffer = (char *)i_send_buffer;

/*
 *	Reply to the request.  Also attach
 *	reply attribute value pairs and any user message provided.
 */
int rad_send_reply(int code, REQUEST *request, VALUE_PAIR *oreply,
			char *msg, int activefd)
{
	AUTH_HDR		*auth;
	u_short			total_length;
	struct	sockaddr	saremote;
	struct	sockaddr_in	*sin;
	u_char			*ptr, *length_ptr;
	char			*what;
	int			len;
	UINT4			lvalue;
	u_char			digest[16];
	int			secretlen;
	VALUE_PAIR		*reply;
	int			vendorcode, vendorpec;

	auth = (AUTH_HDR *)send_buffer;
	reply = oreply;

	switch(code) {
		case PW_PASSWORD_REJECT:
		case PW_AUTHENTICATION_REJECT:
			what = "Reject";
			/*
			 *	Also delete all reply attributes
			 *	except proxy-pair and port-message.
			 */
			reply = NULL;
			pairmove2(&reply, &oreply, PW_REPLY_MESSAGE);
			pairmove2(&reply, &oreply, PW_PROXY_STATE);
			break;
		case PW_ACCESS_CHALLENGE:
			what = "Challenge";
			break;
		case PW_AUTHENTICATION_ACK:
			what = "Ack";
			break;
		case PW_ACCOUNTING_RESPONSE:
			what = "Accounting Ack";
			break;
		default:
			what = "Reply";
			break;
	}

	/*
	 *	Build standard header
	 */
	auth->code = code;
	auth->id = request->packet->id;
	memcpy(auth->vector, request->packet->vector, AUTH_VECTOR_LEN);

	DEBUG("Sending %s of id %d to %s (nas %s)",
		what, request->packet->id, ip_ntoa(NULL, request->packet->src_ipaddr),
		nas_name2(request));

	total_length = AUTH_HDR_LEN;

	/*
	 *	Load up the configuration values for the user
	 */
	ptr = auth->data;
	while(reply != (VALUE_PAIR *)NULL) {
		debug_pair(stdout, reply);

		/*
		 *	This could be a vendor-specific attribute.
		 */
		length_ptr = NULL;
		if ((vendorcode = VENDOR(reply->attribute)) > 0 &&
		    (vendorpec  = dict_vendorpec(vendorcode)) > 0) {
			*ptr++ = PW_VENDOR_SPECIFIC;
			length_ptr = ptr;
			*ptr++ = 6;
			lvalue = htonl(vendorpec);
			memcpy(ptr, &lvalue, 4);
			ptr += 4;
			total_length += 6;
		} else if (reply->attribute > 0xff) {
			/*
			 *	Ignore attributes > 0xff
			 */
			reply = reply->next;
			continue;
		} else
			vendorpec = 0;

#ifdef ATTRIB_NMC
		if (vendorpec == VENDORPEC_USR) {
			lvalue = htonl(reply->attribute & 0xFFFF);
			memcpy(ptr, &lvalue, 4);
			total_length += 2;
			*length_ptr  += 2;
			ptr          += 4;
		} else
#endif
		*ptr++ = (reply->attribute & 0xFF);

		switch(reply->type) {

		case PW_TYPE_STRING:
			/*
			 *	FIXME: this is just to make sure but
			 *	should NOT be needed. In fact I have no
			 *	idea if it is needed :)
			 */
			if (reply->length == 0 && reply->strvalue[0] != 0)
				reply->length = strlen(reply->strvalue);

			len = reply->length;
			if (len >= AUTH_STRING_LEN) {
				len = AUTH_STRING_LEN - 1;
			}
#ifdef ATTRIB_NMC
			if (vendorpec != VENDORPEC_USR)
#endif
				*ptr++ = len + 2;
			if (length_ptr) *length_ptr += len + 2;
			memcpy(ptr, reply->strvalue,len);
			ptr += len;
			total_length += len + 2;
			break;

		case PW_TYPE_INTEGER:
		case PW_TYPE_IPADDR:
#ifdef ATTRIB_NMC
			if (vendorpec != VENDORPEC_USR)
#endif
				*ptr++ = sizeof(UINT4) + 2;
			if (length_ptr) *length_ptr += sizeof(UINT4)+ 2;
			if (reply->type != PW_TYPE_IPADDR)
				lvalue = htonl(reply->lvalue);
			else
				lvalue = reply->lvalue;
			memcpy(ptr, &lvalue, sizeof(UINT4));
			ptr += sizeof(UINT4);
			total_length += sizeof(UINT4) + 2;
			break;

		default:
			break;
		}

		reply = reply->next;
	}

	/*
	 *	Append the user message
	 *	FIXME: add multiple PW_REPLY_MESSAGEs if it
	 *	doesn't fit into one.
	 */
	if(msg != NULL) {
		len = strlen(msg);
		if (len > 0 && len < AUTH_STRING_LEN) {
			*ptr++ = PW_REPLY_MESSAGE;
			*ptr++ = len + 2;
			memcpy(ptr, msg, len);
			ptr += len;
			total_length += len + 2;
		}
	}

	auth->length = htons(total_length);

	/*
	 *	Append secret and calculate the response digest
	 */
	secretlen = strlen(request->packet->secret);
	memcpy(send_buffer + total_length, request->packet->secret, secretlen);
	md5_calc(digest, (char *)auth, total_length + secretlen);
	memcpy(auth->vector, digest, AUTH_VECTOR_LEN);
	memset(send_buffer + total_length, 0, secretlen);

	sin = (struct sockaddr_in *) &saremote;
        memset ((char *) sin, '\0', sizeof (saremote));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = request->packet->src_ipaddr;
	sin->sin_port = htons(request->packet->src_port);

	/*
	 *	Send it to the user
	 */
	sendto(activefd, (char *)auth, (int)total_length, (int)0,
			&saremote, sizeof(struct sockaddr_in));

	/*
	 *	Just to be tidy move pairs back.
	 */
	if (reply != oreply) {
		pairmove2(&oreply, &reply, PW_PROXY_STATE);
		pairmove2(&oreply, &reply, PW_REPLY_MESSAGE);
	}

	return 0;
}

/*
 *	Validates the requesting client NAS.  Calculates the
 *	digest to be used for decrypting the users password
 *	based on the clients private key.
 */
int calc_digest(u_char *digest, REQUEST *request)
{
	u_char	buffer[128];
	int	secretlen;
	CLIENT	*cl;

	/*
	 *	See if we know this client.
	 */
	if ((cl = client_find(request->packet->src_ipaddr)) == NULL) {
		log(L_ERR, "request from unknown client: %s",
			client_name(request->packet->src_ipaddr));
		return -1;
	}

	/*
	 *	Use the secret to setup the decryption digest
	 */
	secretlen = strlen(cl->secret);
	strcpy(buffer, cl->secret);
	memcpy(buffer + secretlen, request->packet->vector, AUTH_VECTOR_LEN);
	md5_calc(digest, buffer, secretlen + AUTH_VECTOR_LEN);
	strcpy(request->packet->secret, cl->secret);
	memset(buffer, 0, sizeof(buffer));

	return(0);
}

/*
 *	Validates the requesting client NAS.  Calculates the
 *	signature based on the clients private key.
 */
int calc_acctdigest(u_char *digest, REQUEST *request)
{
	int	secretlen;
	CLIENT	*cl;
	char zero[AUTH_VECTOR_LEN];
	char	* recvbuf = request->packet->data;
	int	len = request->packet->data_len;

	/*
	 *	See if we know this client.
	 */
	if ((cl = client_find(request->packet->src_ipaddr)) == NULL) {
		log(L_ERR, "request from unknown client: %s",
			client_name(request->packet->src_ipaddr));
		return -1;
	}

	/*
	 *	Copy secret into request->packet->secret so that we can
	 *	use it with send_acct_reply()
	 */
	secretlen = strlen(cl->secret);
	strcpy(request->packet->secret, cl->secret);

	/*
	 *	Older clients have the authentication vector set to
	 *	all zeros. Return `1' in that case.
	 */
	memset(zero, 0, sizeof(zero));
	if (memcmp(request->packet->vector, zero, AUTH_VECTOR_LEN) == 0)
		return 1;

	/*
	 *	Zero out the auth_vector in the received packet.
	 *	Then append the shared secret to the received packet,
	 *	and calculate the MD5 sum. This must be the same
	 *	as the original MD5 sum (request->packet->vector).
	 */
	memset(recvbuf + 4, 0, AUTH_VECTOR_LEN);
	memcpy(recvbuf + len, cl->secret, secretlen);
	md5_calc(digest, recvbuf, len + secretlen);

	/*
	 *	Return 0 if OK, 2 if not OK.
	 */
	return memcmp(digest, request->packet->vector, AUTH_VECTOR_LEN) ? 2 : 0;
}

/*
 *	Receive UDP client requests, build an authorization request
 *	structure, and attach attribute-value pairs contained in
 *	the request to the new structure.
 */
REQUEST	*radrecv(UINT4 host, u_short udp_port,
			u_char *buffer, int length)
{
	u_char		*ptr;
	AUTH_HDR	*auth;
	int		totallen;
	int		attribute;
	int		attrlen;
	int		vendorlen;
	DICT_ATTR	*attr;
	UINT4		lvalue;
	UINT4		vendorcode;
	UINT4		vendorpec;
	VALUE_PAIR	*first_pair;
	VALUE_PAIR	*prev;
	VALUE_PAIR	*pair;
	REQUEST	*request;

	/*
	 *	Pre-allocate the new request data structure
	 */
	if ((request = malloc(sizeof(REQUEST))) == NULL) {
		log(L_ERR|L_CONS, "no memory");
		exit(1);
	}
	memset(request, 0, sizeof(REQUEST));
	if ((request->packet = malloc(sizeof(RADIUS_PACKET))) == NULL) {
		log(L_ERR|L_CONS, "no memory");
		exit(1);
	}
	memset(request->packet, 0, sizeof(RADIUS_PACKET));

#if 1 /* FIXME: do we still need this? */
	if ((request->packet->data = malloc(length)) == NULL) {
		log(L_ERR|L_CONS, "no memory");
		exit(1);
	}
	request->packet->data_len = length;
	memcpy(request->packet->data, buffer, length);
#endif

	auth = (AUTH_HDR *)buffer;
	totallen = ntohs(auth->length);
	memset(request->packet, 0, sizeof(RADIUS_PACKET));
	if (length > totallen) length = totallen;

	DEBUG("radrecv: Request from host %s code=%d, id=%d, length=%d",
			ip_ntoa(NULL, host), auth->code, auth->id, totallen);

	/*
	 *	Fill header fields
	 */
	request->packet->src_ipaddr = host;
	request->packet->src_port = udp_port;
	request->packet->id = auth->id;
	request->packet->code = auth->code;
	memcpy(request->packet->vector, auth->vector, AUTH_VECTOR_LEN);

	/*
	 *	Extract attribute-value pairs
	 */
	ptr = auth->data;
	length -= AUTH_HDR_LEN;
	first_pair = (VALUE_PAIR *)NULL;
	prev = (VALUE_PAIR *)NULL;

	vendorcode = 0;
	vendorlen  = 0;

	while(length > 0) {

		if (vendorlen > 0) {
			attribute = *ptr++ | (vendorcode << 16);
			attrlen   = *ptr++;
		} else {
			attribute = *ptr++;
			attrlen   = *ptr++;
		}
		if (attrlen < 2) {
			length = 0;
			continue;
		}
		attrlen -= 2;
		length  -= 2;

		/*
		 *	This could be a Vendor-Specific attribute.
		 *
		 */
		if (vendorlen <= 0 &&
		    attribute == PW_VENDOR_SPECIFIC && attrlen > 6) {
			memcpy(&lvalue, ptr, 4);
			vendorpec = ntohl(lvalue);
			if ((vendorcode = dict_vendorcode(vendorpec))
			    != 0) {
#ifdef ATTRIB_NMC
				if (vendorpec == VENDORPEC_USR) {
					ptr += 4;
					memcpy(&lvalue, ptr, 4);
					/*printf("received USR %04x\n", ntohl(lvalue));*/
					attribute = (ntohl(lvalue) & 0xFFFF) |
							(vendorcode << 16);
					ptr += 4;
					attrlen -= 8;
					length -= 8;
				} else
#endif
				{
					ptr += 4;
					vendorlen = attrlen - 4;
					attribute = *ptr++ | (vendorcode << 16);
					attrlen   = *ptr++;
					attrlen -= 2;
					length -= 6;
				}
			}
		}

		if ((attr = dict_attrget(attribute)) == (DICT_ATTR *)NULL) {
			DEBUG("Received unknown attribute %d", attribute);
		}
		else if ( attrlen >= AUTH_STRING_LEN ) {
			DEBUG("attribute %d too long, %d >= %d", attribute,
				attrlen, AUTH_STRING_LEN);
		}
		else if ( attrlen > length ) {
			DEBUG("attribute %d longer as buffer left, %d > %d",
				attribute, attrlen, length);
		}
		else {
			if ((pair = malloc(sizeof(VALUE_PAIR))) == NULL) {
				log(L_ERR|L_CONS, "no memory");
				exit(1);
			}
			strcpy(pair->name, attr->name);
			pair->attribute = attr->value;
			pair->type = attr->type;
			pair->length = attrlen;
			pair->next = (VALUE_PAIR *)NULL;

			switch (attr->type) {

			case PW_TYPE_STRING:
				/* attrlen always < AUTH_STRING_LEN */
				memset(pair->strvalue, 0, AUTH_STRING_LEN);
				memcpy(pair->strvalue, ptr, attrlen);
				debug_pair(stdout, pair);
				if(first_pair == (VALUE_PAIR *)NULL) {
					first_pair = pair;
				}
				else {
					prev->next = pair;
				}
				prev = pair;
				break;
			
			case PW_TYPE_INTEGER:
			case PW_TYPE_DATE:
			case PW_TYPE_IPADDR:
				memcpy(&lvalue, ptr, sizeof(UINT4));
				if (attr->type != PW_TYPE_IPADDR)
					pair->lvalue = ntohl(lvalue);
				else
					pair->lvalue = lvalue;
				debug_pair(stdout, pair);
				if(first_pair == (VALUE_PAIR *)NULL) {
					first_pair = pair;
				}
				else {
					prev->next = pair;
				}
				prev = pair;
				break;
			
			default:
				DEBUG("    %s (Unknown Type %d)",
					attr->name,attr->type);
				free(pair);
				break;
			}

		}
		ptr += attrlen;
		length -= attrlen;
		if (vendorlen > 0) vendorlen -= (attrlen + 2);
	}
	request->packet->vps = first_pair;
	return(request);
}


#if 0 /* Not used anymore */
/*
 *	Reply to the request with a CHALLENGE.  Also attach
 *	any user message provided and a state value.
 */
static void send_challenge(REQUEST *request, char *msg,
				char *state, int activefd)
{
	AUTH_HDR		*auth;
	struct	sockaddr	saremote;
	struct	sockaddr_in	*sin;
	char			digest[AUTH_VECTOR_LEN];
	int			secretlen;
	int			total_length;
	u_char			*ptr;
	int			len;

	auth = (AUTH_HDR *)send_buffer;

	/*
	 *	Build standard response header
	 */
	auth->code = PW_ACCESS_CHALLENGE;
	auth->id = request->packet->id;
	memcpy(auth->vector, request->packet->vector, AUTH_VECTOR_LEN);
	total_length = AUTH_HDR_LEN;

	/*
	 *	Append the user message
	 */
	if (msg != (char *)NULL) {
		len = strlen(msg);
		if(len > 0 && len < AUTH_STRING_LEN) {
			ptr = auth->data;
			*ptr++ = PW_REPLY_MESSAGE;
			*ptr++ = len + 2;
			memcpy(ptr, msg, len);
			ptr += len;
			total_length += len + 2;
		}
	}

	/*
	 *	Append the state info
	 */
	if((state != (char *)NULL) && (strlen(state) > 0)) {
		len = strlen(state);
		*ptr++ = PW_STATE;
		*ptr++ = len + 2;
		memcpy(ptr, state, len);
		ptr += len;
		total_length += len + 2;
	}

	/*
	 *	Set total length in the header
	 */
	auth->length = htons(total_length);

	/*
	 *	Calculate the response digest
	 */
	secretlen = strlen(request->packet->secret);
	memcpy(send_buffer + total_length, request->packet->secret, secretlen);
	md5_calc(digest, (char *)auth, total_length + secretlen);
	memcpy(auth->vector, digest, AUTH_VECTOR_LEN);
	memset(send_buffer + total_length, 0, secretlen);

	sin = (struct sockaddr_in *) &saremote;
        memset ((char *) sin, '\0', sizeof (saremote));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = request->packet->src_ipaddr;
	sin->sin_port = htons(request->packet->src_port);

	DEBUG("Sending Challenge of id %d to s (nas %s)",
		request->packet->id, ip_ntoa(NULL, request->packet->src_ipaddr),
		nas_name2(request));
	
	/*
	 *	Send it to the user
	 */
	sendto(activefd, (char *)auth, (int)total_length, (int)0,
			&saremote, sizeof(struct sockaddr_in));
}
#endif

/*
 *	Free an AUTHREQ struct.
 */
void request_free(REQUEST *request)
{
	if (request->packet) {
		pairfree(request->packet->vps);
		if (request->packet->data)
			free(request->packet->data);
		memset(request->packet, 0, sizeof(RADIUS_PACKET));
		free(request->packet);
	}
	if (request->proxy) {
		pairfree(request->proxy->server_reply);
		free(request->proxy);
	}
	free(request);
}

