/*
 * auth.c	Authentication module.
 *
 * Author:	Copyright (C) 2001  Miquel van Smoorenburg.
 *		See also the file LICENSE.
 *
 */
char radius_auth_rcsid[] =
"$Id: auth.c,v 1.5 2001/02/08 16:02:00 miquels Exp $";

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"

#include "radius.h"

typedef struct radius_config_struct {
	char			*secret;
	struct sockaddr_in	ip_server1;
	struct sockaddr_in	ip_server2;
	struct sockaddr_in	ip_source;
	table			*ignore_users;
	int			radius_authoritative;
	int			radius_enabled;
	/* Should we move this stuff out of here? */
	time_t			lastreq;
	RAD_REQUEST		*req;
	RAD_REPLY		*rep;
} radius_config_rec;


/*
 *	Create initial configuration.
 */
static void *create_radius_dir_config(pool *p, char *d)
{
	radius_config_rec *sec;

	sec = (radius_config_rec *) ap_pcalloc(p, sizeof(radius_config_rec));
	sec->radius_authoritative = 999;
	sec->radius_enabled = 999;
	sec->req = (RAD_REQUEST *)ap_pcalloc(p, sizeof(RAD_REQUEST));
	sec->rep = (RAD_REPLY *)ap_pcalloc(p, sizeof(RAD_REPLY));
	sec->ignore_users = ap_make_table(p, 4);
	return sec;
}

/*
 *	Merge directory configs. For example, the config as set in
 *	access.conf and a config from .htaccess.
 */
static void *merge_radius_dir_config (pool *p, void *pa, void *s)
{
	radius_config_rec *rec, *prnt, *sub;

	rec = (radius_config_rec *) ap_pcalloc(p, sizeof(radius_config_rec));
	prnt = (radius_config_rec *)pa;
	sub = (radius_config_rec *)s;

	/* Secret */
	if (sub->secret || prnt->secret)
		rec->secret = ap_pstrdup(p,
			sub->secret ? sub->secret : prnt->secret);

	/* Servers and source IP */
	memcpy(&(rec->ip_server1), sub->ip_server1.sin_addr.s_addr ?
		&(sub->ip_server1) : &(prnt->ip_server1),
		sizeof(struct sockaddr_in));
	memcpy(&(rec->ip_server2), sub->ip_server2.sin_addr.s_addr ?
		&(sub->ip_server2) : &(prnt->ip_server2),
		sizeof(struct sockaddr_in));
	memcpy(&(rec->ip_source), sub->ip_source.sin_addr.s_addr ?
		&(sub->ip_source) : &(prnt->ip_source),
		sizeof(struct sockaddr_in));

	/* Combine the ignore_users. */
	rec->ignore_users = ap_overlay_tables(p, sub->ignore_users,
						prnt->ignore_users);

	/* Only copy authoritative setting if it was set at all */
	rec->radius_authoritative = 999;
	if (prnt->radius_authoritative < 999)
		rec->radius_authoritative = prnt->radius_authoritative;
	if (sub->radius_authoritative < 999)
		rec->radius_authoritative = sub->radius_authoritative;

	/* Ditto for enabled setting. */
	rec->radius_enabled = 999;
	if (prnt->radius_enabled < 999)
		rec->radius_enabled = prnt->radius_enabled;
	if (sub->radius_enabled < 999)
		rec->radius_enabled = sub->radius_enabled;

	/*
	 *	Now the only thing is, what req and rep we copy. It's
	 *	probably a good idea to just take those from the parent
	 *	so they are global, but that is NOT thread safe!
	 *
	 *	In apache 2.0, we should probably protect these
	 *	with a mutex or something while we're using them.
	 *
	 *	Or, just move them out of the config struct.
	 */
	rec->req = prnt->req;
	rec->rep = prnt->rep;

	return rec;
}

/*
 *	Process the RadiusIgnoreUser statement.
 */
static const char *set_ignore_user(cmd_parms *cmd, radius_config_rec *sec,
					char *arg)
{
	const char	*usernames = arg;
	char		*user;

	while (*usernames) {
		user = ap_getword_conf(cmd->pool, &usernames);
		if (*user)
			ap_table_setn(sec->ignore_users, user, "ignore");
	}

	return NULL;
}

/*
 *	Resolve hostname and put it in the config struct.
 */
static const char *set_radius_server(cmd_parms *cmd, char *ptr,
					char *host, char *port)
{
	struct sockaddr_in	*ip;
	struct in_addr		p;
	struct hostent		*h;
	int			offset;

	offset = (int) (long) cmd->info;

	ip = (struct sockaddr_in *)(ptr + offset);
	if (port) ip->sin_port = htons(atoi(port));
	if (inet_aton(host, &p)) {
		ip->sin_addr.s_addr = p.s_addr;
		return NULL;
	}
	if ((h = gethostbyname(host)) == NULL)
		return ap_pstrcat(cmd->pool, "Invalid hostname: ", host, NULL);
	memcpy(&(ip->sin_addr.s_addr), &(h->h_addr[0]), sizeof(struct in_addr));

	return NULL;
}

/*
 *	Save the secret into the config struct.
 */
static const char *set_radius_secret(cmd_parms *cmd, char *ptr, char *secret)
{
	char	**sptr;
	int	offset;

	offset = (int) (long) cmd->info;
	sptr = (char **)(ptr + offset);

	*sptr = ap_pstrdup(cmd->pool, secret);
	return NULL;
}

/*
 *	Table of configuration directives.
 */
static const command_rec auth_cmds[] =
{
    {"RadiusServer1", set_radius_server,
     (void *) XtOffsetOf(radius_config_rec, ip_server1), OR_AUTHCFG, TAKE12,
     "Primary radius server and optional port"},
    {"RadiusServer2", set_radius_server,
     (void *) XtOffsetOf(radius_config_rec, ip_server2), OR_AUTHCFG, TAKE12,
     "Secondary radius server and optional port"},
    {"RadiusSecret", set_radius_secret,
     (void *) XtOffsetOf(radius_config_rec, secret), OR_AUTHCFG, TAKE1,
     "text file containing group names and member user IDs"},
    {"RadiusSourceIP", set_radius_server,
     (void *) XtOffsetOf(radius_config_rec, ip_source), OR_AUTHCFG, TAKE12,
     "Source IP address to use for radius requests"},
    {"RadiusIgnoreUser", set_ignore_user,
     NULL, OR_AUTHCFG, RAW_ARGS,
     "Usernames to ignore for Radius authentication (fall-through)"},
    {"RadiusAuthoritative", ap_set_flag_slot,
     (void *) XtOffsetOf(radius_config_rec, radius_authoritative),
     OR_AUTHCFG, FLAG,
     "Set to 'off' to allow access control to be passed along to lower modules if the UserID is not known to this module"},
    {"RadiusEnabled", ap_set_flag_slot,
     (void *) XtOffsetOf(radius_config_rec, radius_enabled),
     OR_AUTHCFG, FLAG,
     "Set to 'off' to disable Radius authentication altogether"},
    {NULL}
};
module MODULE_VAR_EXPORT radius_auth_module;

/*
 *	Is the radius module enabled etc
 */
static int radius_enabled(request_rec *r, radius_config_rec *sec)
{
	conn_rec		*c;

	c = r->connection;
	if (sec->ip_server1.sin_addr.s_addr == 0 || sec->secret == NULL)
		return DECLINED;
	if (!sec->radius_enabled)
		return DECLINED;
	if (ap_table_get(sec->ignore_users, c->user) != NULL)
		return DECLINED;
	return OK;
}

/*
 *	HTTP basic authentication. Usually (always?) called first.
 */
static int authenticate_basic_user(request_rec *r)
{
    radius_config_rec	*sec;
    conn_rec		*c;
    time_t		now;
    const char		*sent_pw;
    char		*real_pw;
    char		*invalid_pw;
    int			res;

    c = r->connection;

    /*
     *	See if we want to process this user.
     */
    sec = (radius_config_rec *) ap_get_module_config(r->per_dir_config,
		&radius_auth_module);
    if ((res = ap_get_basic_auth_pw(r, &sent_pw)))
	return res;
    if ((res = radius_enabled(r, sec)) != OK)
	return res;

    /*
     *	Authenticate user through radius. Perhaps we still have
     * the authentication info cached.
     */
    now = time(NULL);
    if (now >= sec->lastreq + 10 ||
	strcmp(sec->req->user_name, c->user) != 0 ||
        strcmp(sec->req->password, sent_pw) != 0) {

	memset(sec->req, 0, sizeof(RAD_REQUEST));
	memset(sec->rep, 0, sizeof(RAD_REPLY));
	strncpy(sec->req->user_name, c->user, sizeof(sec->req->user_name) - 1);
	strncpy(sec->req->password, sent_pw, sizeof(sec->req->password) - 1);
	strncpy(sec->req->secret, sec->secret, sizeof(sec->req->secret) - 1);
	strncpy(sec->req->nas_id, r->hostname, sizeof(sec->req->nas_id) - 1);
	sec->req->nas_ip  = c->local_addr.sin_addr.s_addr;
	sec->req->server1 = sec->ip_server1.sin_addr.s_addr;
	sec->req->port1   = sec->ip_server1.sin_port;
	sec->req->server2 = sec->ip_server2.sin_addr.s_addr;
	sec->req->port2   = sec->ip_server2.sin_port;
	sec->req->source  = sec->ip_source.sin_addr.s_addr;

	if (rad_client(sec->req, sec->rep) < 0) {
		/*
		 *	Protocol failure somewhere....
		 */
		memset(sec->req, 0, sizeof(RAD_REQUEST));
		memset(sec->rep, 0, sizeof(RAD_REPLY));
		sec->lastreq = 0;
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
		"Radius user \"%s\": radius protocol failed for \"%s\"",
			c->user, r->uri);
		ap_note_basic_auth_failure(r);
		return AUTH_REQUIRED;
	}
	sec->lastreq = time(NULL);
    }

    if (sec->rep->code != RAD_AUTHACK) {
	if (!(sec->radius_authoritative))
	    return DECLINED;
	ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
		    "user %s: authentication failure for \"%s\"",
			c->user, r->uri);
	ap_note_basic_auth_failure(r);
	return AUTH_REQUIRED;
    }
    return OK;
}

static int group_in_list(char *list, const char *group)
{
	char	*e;
	int	len;

	len = strlen(group);

	while (*list) {
		for (e = list; *e && *e != ','; e++)
			;
		if (e - list == len && strncmp(list, group, len) == 0)
			return 1;
		while (*e == ',') e++;
		list = e;
	}
	return 0;
}

/*
 *	Now we have to check the _type_ of access.
 *	Meaning, "require valid-user|user foo|group foo" etc.
 *
 *	We'll have to query the radius server _again_ to get
 *	the list of groups, that's why we cache the radius
 *	reply in global variables.
 */
static int check_user_access(request_rec *r)
{
	radius_config_rec	*sec;
	table			*grpstatus;
	require_line		*reqs;
	const array_header	*reqs_arr;
	const char		*t, *w;
	char			*user;
	int			m, x;
	int			method_restricted = 0;
	int			did_radius = 0;

	/*
	 *	Set up a bunch of variables.
	 */
	sec = (radius_config_rec *) ap_get_module_config(r->per_dir_config,
					&radius_auth_module);
	if ((x = radius_enabled(r, sec)) != OK)
		return x;
	user = r->connection->user;
	m = r->method_number;
	reqs_arr = ap_requires(r);
	if (!reqs_arr)
		return DECLINED;
	reqs = (require_line *) reqs_arr->elts;

	/*
	 *	Loop over the requirements.
	 */
	for (x = 0; x < reqs_arr->nelts; x++) {

		/*
		 *	Is this a restricted method ?
		 */
		if (!(reqs[x].method_mask & (1 << m)))
			continue;
		method_restricted = 1;

		/*
		 *	What exactly is the requirement.
		 */
		t = reqs[x].requirement;
		w = ap_getword_white(r->pool, &t);

		if (!strcmp(w, "valid-user"))
			return OK;

		if (strcmp(w, "user") == 0) {
			while (t[0]) {
				w = ap_getword_conf(r->pool, &t);
				if (!strcmp(user, w))
					return OK;
			}
		}
		else if (strcmp(w, "group") == 0) {
			/*
			 *	See if user is in certain group.
			 */
			if (did_radius == 0) {
				if ((x = authenticate_basic_user(r)) != OK)
					return x;
				did_radius = 1;
			}
			while (t[0]) {
				w = ap_getword_conf(r->pool, &t);
				if (group_in_list(sec->rep->groups, w))
					return OK;
	    		}
		} else if (sec->radius_authoritative) {
			/*
			 *	We're authoritative, but we don't know this
			 *	require directive. Scream loudly.
			 */
			ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
		    "access to %s failed, reason: unknown require directive:"
		    "\"%s\"", r->uri, reqs[x].requirement);
		}
	}

	if (!method_restricted)
		return OK;

	if (!(sec->radius_authoritative))
		return DECLINED;

	ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
		"access to %s failed, reason: user %s not allowed access",
		r->uri, user);
	ap_note_basic_auth_failure(r);

	return AUTH_REQUIRED;
}

module MODULE_VAR_EXPORT radius_auth_module =
{
    STANDARD_MODULE_STUFF,
    NULL,			/* initializer */
    create_radius_dir_config,	/* dir config creater */
    merge_radius_dir_config,	/* dir merger --- default is to override */
    NULL,			/* server config */
    NULL,			/* merge server config */
    auth_cmds,			/* command table */
    NULL,			/* handlers */
    NULL,			/* filename translation */
    authenticate_basic_user,	/* check_user_id */
    check_user_access,		/* check auth */
    NULL,			/* check access */
    NULL,			/* type_checker */
    NULL,			/* fixups */
    NULL,			/* logger */
    NULL,			/* header parser */
    NULL,			/* child_init */
    NULL,			/* child_exit */
    NULL			/* post read-request */
};
