/*
 * radclient.c	Small implementation of a radiusclient,
 *		meant to be used with the apache radius module.
 *
 * Author:      Copyright (C) 2001  Miquel van Smoorenburg.
 *              See also the file LICENSE.
 */

char radius_rcsid[] =
"$Id: radius.c,v 1.3 2001/02/07 20:43:02 miquels Exp $";

#ifdef TARGET
#include "httpd.h"
#include "util_md5.h"
#else
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <openssl/md5.h>
#define AP_MD5_CTX   MD5_CTX
#define ap_MD5Init   MD5_Init
#define ap_MD5Update MD5_Update
#define ap_MD5Final  MD5_Final
#endif

#include "radius.h"

static int rad_id;

static void md5_calc(unsigned char *out, unsigned char *in, int len)
{
	AP_MD5_CTX	my_md5;

	ap_MD5Init(&my_md5);
	ap_MD5Update(&my_md5, in, len);
	ap_MD5Final(out, &my_md5);
}

/*
 *	Generate random bytes - multiple of (sizeof(int))
 */
static void rad_vector(unsigned char *vector, int vlen)
{
	static int	init = 0;
	int		i, r;

	if (!init) {
		srand(time(0) + getpid());
		init++;
	}
	for (i = 0; i < vlen; i += sizeof(int)) {
		r = rand();
		memcpy(vector + i, &r, sizeof(int));
	}
}

/*
 *	Build a complete radius packet.
 */
RAD_PACKET *rad_build_packet(char *buf, int *blen, RAD_REQUEST *req)
{
	RAD_PACKET	*rp;
	char		passbuf[16];
	char		md5buf[128];
	char		*ptr;
	unsigned int	x;
	int		i, len, tot_len;
	int		slen;

	/* Fill packet header. */
	rp = (RAD_PACKET *)buf;
	rp->code = RAD_AUTHREQ;
	rp->id = (rad_id++) & 0xff;
	rad_vector(rp->vector, sizeof(rp->vector));
	tot_len = 20;
	ptr = rp->data;

	/*
	 *	Add username
	 */
	len = strlen(req->user_name);
	*ptr++ = RAD_USER_NAME;
	*ptr++ = len + 2;
	memcpy(ptr, req->user_name, len);
	ptr     += len;
	tot_len += len + 2;

	/*
	 *	Add password. FIXME: only the simple algorithm
	 *	is implemented, so max password length is 16 chars.
	 */
	*ptr++ = RAD_PASSWORD;
	*ptr++ = 18;
	memset(passbuf, 0, 16);
	if ((len = strlen(req->password)) > 16)
		len = 16;
	memcpy(passbuf, req->password, len);

	/* Calculate the MD5 Digest */
	slen = strlen(req->secret);
	if (slen > sizeof(md5buf) - 16)
		slen = sizeof(md5buf) - 16;
	strncpy(md5buf, req->secret, slen);
	memcpy(md5buf + slen, rp->vector, 16);
	md5_calc(ptr, md5buf, slen + 16);

	/* Xor the password into the MD5 digest */
	for (i = 0; i < 16; i++)
		*ptr++ ^= passbuf[i];
	tot_len += 18;

	/*
	 *	Add Nas-Id.
	 */
	if ((len = strlen(req->nas_id)) > 0) {
		*ptr++ = RAD_NAS_IDENTIFIER;
		*ptr++ = len + 2;
		memcpy(ptr, req->nas_id, len);
		tot_len += len + 2;
		ptr += len;
	}

	/*
	 *	Add Nas-IP-Address.
	 */
	if (req->nas_ip != 0) {
		*ptr++ = RAD_NAS_IP_ADDRESS;
		*ptr++ = 6;
		memcpy(ptr, &(req->nas_ip), 4);
		ptr += 4;
		tot_len += 6;
	}

	/*
	 *	Add Service-Type = Authenticate-Only.
	 */
	*ptr++ = RAD_SERVICE_TYPE;
	*ptr++ = 6;
	x = htonl(RAD_AUTHENTICATE_ONLY);
	memcpy(ptr, &x, 4);
	ptr += 4;
	tot_len += 6;

	rp->len = htons(tot_len);

	*blen = tot_len;

	return rp;
}

static void append(char *str, int sz, int sep, char *word, int wlen)
{
	char			*p;
	int			l;

	l = strlen(str);
	if (l + wlen + 3 > sz)
		return;

	for (p = str; *p; p++)
		;
	if (p > str && p[-1] != sep)
		*p++ = sep;
	memcpy(p, word, wlen);
	p[wlen] = 0;
}

/*
 *	Decode a radius packet.
 */
int rad_decode(char *buf, int rlen, RAD_REQUEST *req, RAD_REPLY *rep)
{
	RAD_PACKET		*rp;
	int			len;
	int			alen;
	char			*ptr;

	rp = (RAD_PACKET *)buf;
	len = ntohs(rp->len);
	if (rlen < len) return -1;
	ptr = rp->data;
	rep->code = rp->code;

	while (ptr < buf + rlen) {
		alen = ptr[1];
		if (ptr[0] == RAD_REPLY_MESSAGE) {
			append(rep->reply, sizeof(rep->reply),
				0, ptr + 2, alen - 2);
		} else if (ptr[0] == RAD_GROUP) {
			append(rep->groups, sizeof(rep->groups),
				',', ptr + 2, alen - 2);
		}
		ptr += alen;
	}

	return 0;
}

int rad_client(RAD_REQUEST *req, RAD_REPLY *rep)
{
	static int		sockfd = -1;
	static unsigned int	sockaddr = 0;
	RAD_PACKET		*rp;
	struct sockaddr_in	sin, saremote;
	struct timeval		tv;
	fd_set			readfds;
	char			buffer[1024];
	char			rbuf[1024];
	int			i, blen, len, port;

	/*
	 *	Build request.
	 */
	memset(rep, 0, sizeof(RAD_REPLY));
	if (req->user_name[0] == 0) {
		rep->code = RAD_AUTHREJ;
		strcpy(rep->reply, "Empty username");
		return 0;
	}
	blen = sizeof(buffer);
	rp = rad_build_packet(buffer, &blen, req);

	/*
	 *	Try to re-use open socket, allocate if not possible.
	 */
	if (sockfd >= 0 && sockaddr != req->source) {
		close(sockfd);
		sockfd = -1;
		sockaddr = 0;
	}
	if (sockfd < 0)
		sockfd = socket(AF_INET, SOCK_DGRAM, 0);

	/*
	 *	Bind socket to local address.
	 */
	if (req->source && sockaddr == 0) {
		memset(&sin, 0, sizeof(sin));
		sin.sin_family = AF_INET;
		sin.sin_addr.s_addr = req->source;
		for (port = 2000; port < 64000; port++) {
			sin.sin_port = htons(port);
			if (bind(sockfd, (struct sockaddr *)&sin,
			    sizeof(sin)) == 0)
				break;
		}
		if (port >= 64000) {
			close(sockfd);
			sockfd = -1;
			return -1;
		}
		sockaddr = sin.sin_addr.s_addr;
	}

	/*
	 *	Send the request, wait for an answer.
	 */
	len = 0;
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	for (i = 0; i < 10; i++) {
		sin.sin_addr.s_addr = req->server1;
		sin.sin_port = req->port1;
		if (i >= 3 && (i & 1) && req->server2) {
			sin.sin_addr.s_addr = req->server2;
			sin.sin_port = req->port2;
		}
		if (sin.sin_port == 0)
			sin.sin_port = htons(RAD_AUTH_PORT);

		sendto(sockfd, (char *)rp, blen, 0, (struct sockaddr *)&sin,
			sizeof(sin));

		tv.tv_sec = 3;
		tv.tv_usec = 0;
		FD_ZERO(&readfds);
		FD_SET(sockfd, &readfds);
		if (select(sockfd + 1, &readfds, NULL, NULL, &tv) == 0)
			continue;
		len = recvfrom(sockfd, rbuf, sizeof(rbuf), 0, NULL, 0);
		if (len >= 20 && ((RAD_PACKET *)rbuf)->id == rp->id)
			break;
	}
	if (i >= 10)
		return -1;

	/*
	 *	Right, we have an answer! Decode it.
	 */
	return rad_decode(rbuf, len, req, rep);
}
