/*
 * eapclient.c	General radius packet debug tool.
 *
 * Version:	$Id: eapclient.c,v 1.42 2002/01/13 11:12:04 fcusack Exp $
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright 2000  The FreeRADIUS server project
 * Copyright 2000  Miquel van Smoorenburg <miquels@cistron.nl>
 * Copyright 2000  Alan DeKok <aland@ox.org>
 * Copyright 2002  hereuare Communications <raghud@hereuare.com>
 */

static const char rcsid[] = "$Id: eapclient.c,v 1.42 2002/01/13 11:12:04 fcusack Exp $";

#include "autoconf.h"
#include "libradius.h"

#include <stdio.h>
#include <stdlib.h>

#if HAVE_UNISTD_H
#	include <unistd.h>
#endif

#include <string.h>
#include <ctype.h>
#include <netdb.h>
#include <sys/socket.h>

#if HAVE_NETINET_IN_H
#	include <netinet/in.h>
#endif

#if HAVE_SYS_SELECT_H
#	include <sys/select.h>
#endif

#if HAVE_GETOPT_H
#	include <getopt.h>
#endif

#include "conf.h"
#include "radpaths.h"
#include "missing.h"

#include "mytls.h"

static tls_stuff_t tls;

static int retries = 10;
static float timeout = 3;
static const char *secret = NULL;
static int do_output = 1;
static int do_summary = 0;
static int filedone = 0;
static int totalapp = 0;
static int totaldeny = 0;
static char filesecret[256];

typedef struct md5_packet_t {
  uint8_t       value_size;
  uint8_t       value[1];
  uint8_t       name[1];
} md5_packet_t;

typedef struct eap_packet_t {
  uint8_t       code;
  uint8_t       id;
  uint8_t       length[2];
  uint8_t       data[1];
} eap_packet_t;

int acct_start(RADIUS_PACKET *req, RADIUS_PACKET **rep, VALUE_PAIR *veepee);
static int send_packet(RADIUS_PACKET *req, RADIUS_PACKET **rep, VALUE_PAIR* veepee);

/*
 *	Read valuepairs from the fp up to End-Of-File.
 */
static VALUE_PAIR *readvp(FILE *fp)
{
	char buf[8192];
	LRAD_TOKEN last_token;
	char *p;
	VALUE_PAIR *vp;
	VALUE_PAIR *list;
	int error = 0;

	list = NULL;

	while (!error && fgets(buf, sizeof(buf), fp) != NULL) {

		p = buf;

		/* If we get a '\n' by itself, we assume that's the end of that VP */
		if((buf[0] == '\n') && (list)) {
			return error ? NULL: list;
		} 
		if((buf[0] == '\n') && (!list)) {
			continue;
		} else {
			do {
				if ((vp = pairread(&p, &last_token)) == NULL) {
					librad_perror("eapclient:");
					error = 1;
					break;
				}
				pairadd(&list, vp);
			} while (last_token == T_COMMA);
		}
	}
	filedone = 1;
	return error ? NULL: list;
}

static void usage(void)
{
	fprintf(stderr, "Usage: eapclient [-c count] [-d raddb] [-f file] [-r retries] [-t timeout]\n"
			"[-i id] [-qvxS] server acct|auth [<secret>]\n");
	
	fprintf(stderr, " -c count    Send each packet 'count' times.\n");
	fprintf(stderr, " -d raddb    Set dictionary directory.\n");
	fprintf(stderr, " -f file     Read packets from file, not stdin.\n");
	fprintf(stderr, " -r retries  If timeout, retry sending the packet 'retires' times.\n");
	fprintf(stderr, " -t timeout  Wait 'timeout' seconds before retrying.\n");
	fprintf(stderr, " -i id       Set request id to 'id'.  Values may be 0..255\n");
	fprintf(stderr, " -S file     read secret from file, not command line.\n");
	fprintf(stderr, " -q          Do not print anything out.\n");
	fprintf(stderr, " -s          Print out summary information of auth results.\n");
	fprintf(stderr, " -v          Show program version information.\n");
	fprintf(stderr, " -x          Debugging mode.\n");

	exit(1);
}

static int getport(const char *name)
{
	struct	servent		*svp;

	svp = getservbyname (name, "udp");
	if (!svp) {
		return 0;
	}

	return ntohs(svp->s_port);
}

void eap_start(VALUE_PAIR *eap_msg)
{
	if (eap_msg == NULL)
		return;
	memset(eap_msg->strvalue, 0, 254);
	eap_msg->length = 2;
}

VALUE_PAIR *eap_identity(VALUE_PAIR *eap_msg, VALUE_PAIR *user_name)
{
	uint8_t *ptr;
	u_short len;

	if (eap_msg == NULL) return NULL;
	if (user_name == NULL) return NULL;
	len = htons(user_name->length + 5);
	ptr = eap_msg->strvalue;
	*ptr++ = 2/*EAP-Response*/;
	*ptr++ = 45 & 0xff;
	memcpy(ptr, &len, sizeof(u_short));
	ptr += 2;
	*ptr++ = 1/*EAP-Type=EAP-Identity*/;

	memcpy(ptr, user_name->strvalue, user_name->length);
	eap_msg->length = user_name->length + 5;

	return eap_msg;
}

VALUE_PAIR *eap_md5_challenge_response(VALUE_PAIR *eap_msg, VALUE_PAIR *password)
{
	uint8_t 	*ptr;
	u_short 	len;
	eap_packet_t 	*eap_p;
	md5_packet_t 	*md5_p;
	char 		string[MAX_STRING_LEN];
	char 		output[MAX_STRING_LEN];

	memset(output, 0, MAX_STRING_LEN);
	memset(string, 0, MAX_STRING_LEN);

	if (eap_msg == NULL)
		return NULL;

	eap_p = (eap_packet_t *)eap_msg->strvalue;
	md5_p = (md5_packet_t *)(eap_p->data + 1);

	/*
	 * get the challenge eap_md5 value and MD5 it with our password
	 * MD5(id + password + challenge)
	 */
	len = 1 + password->length + md5_p->value_size;

	ptr = string;
	*ptr++ = eap_p->id;
	memcpy(ptr, password->strvalue, password->length);
	ptr += password->length;
	memcpy(ptr, md5_p->value, md5_p->value_size);

	librad_md5_calc(output, string, len);

	/*
	 * Frame eap in the eap_message attr and send it
	 * len = EAP header + type + value_size + value
	 *     = 4 + 1 + 1 + 16
	 */
	len = htons(4+1 + 1+16);
	ptr = eap_msg->strvalue;
	*ptr++ = 2/*EAP-Response*/;
	*ptr++ = eap_p->id;
	memcpy(ptr, &len, sizeof(u_short));
	ptr += 2;
	*ptr++ = 4/*EAP-Type=EAP-MD5*/;
	*ptr++ = 16/*Value-Size*/;
	memcpy(ptr, output, 16);
	eap_msg->length = 22/*EAP-Length*/;
	
	return eap_msg;
}

/*
 * Handles multiple EAP-Message attrs
 * ie concatenates all to get the complete EAP packet
 */
eap_packet_t *get_eapmsg_attr(VALUE_PAIR *vps)
{
	VALUE_PAIR *vp_list, *i;
	eap_packet_t *eap_packet;
	uint8_t *ptr;
	int len;
	int total_len;

	vp_list = paircopy2(vps, PW_EAP_MESSAGE);
	if (vp_list == NULL) {
	printf("EAP_Message not found");
		return NULL;
	}

	/* Get the Actual length from the EAP packet */
	memcpy(&len, vp_list->strvalue+2, sizeof(u_short));
	len = ntohs(len);

	eap_packet = malloc(len);
	if (!eap_packet) {
		printf("out of memory");
		pairfree(&vp_list);
		return NULL;
	}
	ptr = (uint8_t *)eap_packet;
	memcpy(ptr, vp_list->strvalue, vp_list->length);
	ptr += vp_list->length;
	total_len = vp_list->length;

	if (vp_list->next == NULL) {
		printf("Only one EAP_Message attribute found\n");
		pairfree(&vp_list);
		return eap_packet;
	}

	printf("Multiple EAP_Message attributes found\n");

	/* RADIUS ensures order of attrs, so just concatenate all */
	for (i = vp_list->next; i; i = i->next) {
		if (total_len + i->length > len) {
			printf(" Malformed EAP, lengths mismatch\n");
		}
		total_len += i->length;
		memcpy(ptr, i->strvalue, i->length);
		ptr += i->length;
	}
	printf("Total length obtained -- %d\n", total_len);
	pairfree(&vp_list);
	return eap_packet;
}

int compose(RADIUS_PACKET *req, eap_packet_t *eap_packet)
{
	int eap_len, len;
	VALUE_PAIR *eap_msg;
	unsigned char *ptr;

	memcpy(&eap_len, &(eap_packet->length), sizeof(u_short));
	len = eap_len = ntohs(eap_len);
	ptr = (unsigned char *)eap_packet;

	do {
		/*
		if (eap_len > MAX_STRING_LEN) {
			len = MAX_STRING_LEN;
			eap_len -= MAX_STRING_LEN;
		*/
		if (eap_len > 252) {
			len = 252;
			eap_len -= 252;
		} else {
			len = eap_len;
			eap_len = 0;
		}

		/* create a value pair & append it to the request reply list */
		eap_msg = paircreate(PW_EAP_MESSAGE, PW_TYPE_STRING);
		memcpy(eap_msg->strvalue, ptr, len);
		eap_msg->length = len;
		pairadd(&(req->vps), eap_msg);
		ptr += len;
		eap_msg = NULL;
	} while (eap_len);

	return 0;
}

void eap_tls_world(RADIUS_PACKET *req, session_t *ssn)
{
	VALUE_PAIR *eap_msg;
	eap_packet_t *eap_received;
	eap_packet_t *eap_tosend;
	unsigned char flags;
	unsigned char *data = NULL;
	unsigned int data_len = 0;
	unsigned short recv_len = 0;
	unsigned short send_len = 0;
	unsigned short acklen = 5/*code+id+length+EAPtype*/;

	eap_received = get_eapmsg_attr(req->vps);
	pairdelete(&(req->vps), PW_EAP_MESSAGE);

	memcpy(&recv_len, eap_received->length, sizeof(unsigned short));
	recv_len = ntohs(recv_len);

	if (recv_len == acklen) {
		/* Its ack, send next fragment */
		printf("Its Ack, send the next fragment\n");
		goto eaptls_pack;
	}

	/* TLS-start received, send Client hello */
	if ((eap_received->data[1] & 0x20) == 0x20) {
		/* Start Client Hello */
		printf("Its start, Start with Client Hello\n");
		new_tls_connection(&tls, &ssn);
		tls.ssn = ssn;
		goto eaptls_pack;
	}
	
	/* No Flags set ? */
	if ((eap_received->data[1] & 0xc0) == 0x00) {
		data_len = recv_len - 6/*code+id+length+EAPtype+flags*/;
		data = eap_received->data + 2/*EAPtype+flags*/;
	}
	/* Length included ? */
	if ((eap_received->data[1] & 0x80) == 0x80) {
		memcpy(&data_len, &eap_received->data[2], sizeof(uint32_t));
		data_len = ntohl(data_len);
		data = eap_received->data + 6/*EAPtype+flags+TLS-Length*/;
	}

	/* More Fragments to receive, send an ack */
	if ((eap_received->data[1] & 0x40) == 0x40) {
		if (!data_len) {
			data_len = recv_len - 6/*code+id+length+EAPtype+flags*/;
			data = eap_received->data + 2/*EAPtype+flags*/;
		}
		goto eaptls_acknowledge;
	}

	if (!data_len) {
		printf(" Something is seriously broken check it\n");
		return;
	} else {
		/* Its time to answer the EAPTLS request */
		memcpy(ssn->dirty_in.data, data, data_len);
		ssn->dirty_in.used = data_len;
		tls_read(ssn);
		if (!ssn->type[3]/*isClient*/ && 
			(ssn->type[2] >= 22 || ssn->type[0] == 20)) {

			printf("Handshake is done successfully\n");
			close_session(ssn);
			free(ssn);
			ssn = NULL;

			goto eaptls_acknowledge;
		}
		/* If clean_out data is present then process
		 * and use tls_write();
		 */
		goto eaptls_pack;
	}

eaptls_acknowledge:
	/*store the data & send ack */
	eap_tosend = (eap_packet_t *)malloc(acklen);
	eap_tosend->code = 2/*EAP-Response*/;
	eap_tosend->id = eap_received->id;
	acklen = htons(acklen);
	memcpy(eap_tosend->length, &acklen, sizeof(unsigned short));
	eap_tosend->data[0/*EAPtype*/] = (0x0d & 0xff)/*EAP-Type=TLS*/;
	acklen = ntohs(acklen);

	eap_msg = paircreate(PW_EAP_MESSAGE, PW_TYPE_STRING);
	memcpy(eap_msg->strvalue, (unsigned char *)eap_tosend, acklen);
	eap_msg->length = acklen;
	pairadd(&(req->vps), eap_msg);
	if (eap_tosend) free(eap_tosend);
	if (eap_received) free(eap_received);
	return;

eaptls_pack:
	flags = 0x00;
	printf(" length before fragmentation = %d\n", ssn->dirty_out.used);
	if (ssn->dirty_out.used == 0) {
		printf(" Client Error: Something is seriously broken check it\n");
		return;
	} else {
		if (ssn->dirty_out.used > FRAGMENT_SIZE) {
		printf(" M bit set\n");
			data_len = FRAGMENT_SIZE;
			flags |= 0x40/*set M bit*/;
		} else {
			printf(" Single eaptls packet \n");
			data_len = ssn->dirty_out.used;
		}
	}
	flags |= 0x80/*set L bit*/;
	send_len = data_len + 10/*code+id+length+EAPtype+flags+TLSlength*/;
	eap_tosend = (eap_packet_t *)malloc(send_len);
	eap_tosend->code = 2/*EAP-Response*/;
	eap_tosend->id = eap_received->id;
	send_len = htons(send_len);
	memcpy(eap_tosend->length, &send_len, sizeof(unsigned short));
	eap_tosend->data[0/*EAPtype*/] = (0x0d & 0xff)/*EAP-TLS*/;
	eap_tosend->data[1/*flags*/] = flags;
	data_len = htonl(data_len);
	memcpy(eap_tosend->data+2/*EAPtype+flags*/, &data_len, sizeof(unsigned int));
	data_len = ntohl(data_len);
	from_record(&ssn->dirty_out, eap_tosend->data+6/*EAPtype+flags+TLSlength*/, data_len);
	compose(req, eap_tosend);
	if (eap_tosend) free(eap_tosend);
	if (eap_received) free(eap_received);
}

static int  reply_challenge(RADIUS_PACKET *req, RADIUS_PACKET **rep, VALUE_PAIR *veepee)
{
	VALUE_PAIR *eap_msg;
	eap_packet_t *eap_p;

	if ((eap_msg = pairfind((*rep)->vps, PW_EAP_MESSAGE)) == NULL)
		return 0;

	pairdelete(&(req->vps), PW_EAP_MESSAGE);
	pairdelete(&(req->vps), PW_STATE);
	pairmove2(&(req->vps), &((*rep)->vps), PW_EAP_MESSAGE);
	pairmove2(&(req->vps), &((*rep)->vps), PW_STATE);

	eap_p = (eap_packet_t *)eap_msg->strvalue;

	if (eap_p->code != 1) {
		printf(" Its not EAP request, Don't respond\n");
		return 0;
	}

	/*
	 * Identify the type of EAP reply
	 */
	switch (eap_p->data[0]) {
	case 1:
		printf("REQUEST FOR IDENTITY\n");
		eap_msg = eap_identity(eap_msg, pairfind(req->vps, PW_USER_NAME));
		//eap_msg = eap_identity(eap_msg, pairfind(veepee, PW_USER_NAME));
		break;

	case  4:
		printf("MD5 CHALLENGE RESPONSE\n");
		return 0;
		eap_msg = eap_md5_challenge_response(eap_msg, pairfind(veepee, PW_PASSWORD));
		break;

	case  13:
		printf("EAP TLS RESPONSE\n");
		eap_tls_world(req, tls.ssn);
		break;

	default:
		printf("Unknown EAP-Type\n");
		return 0;
	}

	/* keep req packet, modify eap_msg attr and send it back */
	if (req->data) {
		free(req->data);
		req->data = NULL;
		req->data_len = 0;
	}
	req->id++;

	/* clear reply radius packet */
	if((*rep)->vps) pairfree(&((*rep)->vps));
	if((*rep)->data) free((*rep)->data);
	(*rep)->data_len = 0;
	free(*rep);
	*rep = NULL;
	return send_packet(req, rep, veepee);
}

static int send_packet(RADIUS_PACKET *req, RADIUS_PACKET **rep, VALUE_PAIR *veepee)
{
	int i;
	struct timeval	tv;

	for (i = 0; i < retries; i++) {
		fd_set		rdfdesc;

		rad_send(req, NULL, secret);

		/* And wait for reply, timing out as necessary */
		FD_ZERO(&rdfdesc);
		FD_SET(req->sockfd, &rdfdesc);

		tv.tv_sec = (int)timeout;
		tv.tv_usec = 1000000 * (timeout - (int)timeout);

		/* Something's wrong if we don't get exactly one fd. */
		if (select(req->sockfd + 1, &rdfdesc, NULL, NULL, &tv) != 1) {
			continue;
		}

		*rep = rad_recv(req->sockfd);
		if (*rep != NULL) {
			break;
		} else {	/* NULL: couldn't receive the packet */
			librad_perror("eapclient:");
			exit(1);
		}
	}

	/* No response or no data read (?) */
	if (i == retries) {
		fprintf(stderr, "eapclient: no response from server\n");
		exit(1);
	}

	if (rad_decode(*rep, req, secret) != 0) {
		librad_perror("rad_decode");
		exit(1);
	}
	
	/* libradius debug already prints out the value pairs for us */
	if (!librad_debug && do_output) {
		printf("Received response ID %d, code %d, length = %d\n",
			(*rep)->id, (*rep)->code, (*rep)->data_len);
		vp_printlist(stdout, (*rep)->vps);
	}
	if((*rep)->code == PW_AUTHENTICATION_ACK) {
		printf("Sending Account Start Packet\n");
		totalapp++;
		//acct_start(req, rep, veepee);
	} else {
		totaldeny++;
	}

	if ((*rep)->code == PW_ACCESS_CHALLENGE) {
		printf("Received response ID %d, code %d, length = %d\n",
			(*rep)->id, (*rep)->code, (*rep)->data_len);
		reply_challenge(req, rep, veepee);
	}
	return 0;
}

int acct_start(RADIUS_PACKET *req, RADIUS_PACKET **rep, VALUE_PAIR *veepee)
{
	VALUE_PAIR *vp = NULL;
	time_t time_now;
	char now[MAX_STRING_LEN];

	time_now = time(NULL);
	sprintf(now, "%ld", time_now);

	printf(" Acct-Session-Id = %s\n", now);

	pairdelete(&(req->vps), PW_EAP_MESSAGE);
	pairdelete(&(req->vps), PW_MESSAGE_AUTHENTICATOR);

	vp = pairmake("Acct-Session-Id", now, T_OP_EQ);
	pairadd(&(req->vps), vp);
	vp = pairmake("Acct-Status-Type", "Start", T_OP_EQ);
	pairadd(&(req->vps), vp);

	req->dst_port++;
	req->code = PW_ACCOUNTING_REQUEST;
	if (req->data) {
	free(req->data);
		req->data = NULL;
		req->data_len = 0;
	}
	req->id++;
	/* clear reply radius packet */
	if((*rep)->vps) pairfree(&((*rep)->vps));
	if((*rep)->data) free((*rep)->data);
	(*rep)->data_len = 0;
	free(*rep);
	*rep = NULL;
	send_packet(req, rep, veepee);
	req->dst_port--;
	req->code = PW_AUTHENTICATION_REQUEST;
	return 1;
}

int main(int argc, char **argv)
{
	RADIUS_PACKET *req;
	RADIUS_PACKET *rep = NULL;
	char *p;
	int c;
	int port = 0;
	const char *radius_dir = RADDBDIR;
	VALUE_PAIR *veepee = NULL;
	VALUE_PAIR *vp = NULL;
	char password[256];
	char *filename = NULL;
	FILE *fp;
	int count = 1;
	int loop;
	int id;

	id = ((int)getpid() & 0xff);

	while ((c = getopt(argc, argv, "c:d:f:hi:qst:r:S:xv")) != EOF) switch(c) {
		case 'c':
			if (!isdigit(*optarg)) 
				usage();
			count = atoi(optarg);
			break;
		case 'd':
			radius_dir = optarg;
			break;
		case 'f':
			filename = optarg;
			break;
		case 'q':
			do_output = 0;
			break;
		case 'x':
			librad_debug = 1;
			break;
		case 'r':
			if (!isdigit(*optarg)) 
				usage();
			retries = atoi(optarg);
			break;
		case 'i':
			if (!isdigit(*optarg)) 
				usage();
			id = atoi(optarg);
			if ((id < 0) || (id > 255)) {
				usage();
			}
			break;
		case 's':
			do_summary = 1;
			break;
		case 't':
			if (!isdigit(*optarg)) 
				usage();
			timeout = atof(optarg);
			break;
		case 'v':
			printf("eapclient: $Id: eapclient.c,v 1.42 2002/01/13 11:12:04 fcusack Exp $ built on " __DATE__ " at " __TIME__ "\n");
			exit(0);
			break;
               case 'S':
		       fp = fopen(optarg, "r");
                       if (!fp) {
                               fprintf(stderr, "eapclient: Error opening %s: %s\n",
                                       optarg, strerror(errno));
                               exit(1);
                       }
                       if (fgets(filesecret, sizeof(filesecret), fp) == NULL) {
                               fprintf(stderr, "eapclient: Error reading %s: %s\n",
                                       optarg, strerror(errno));
                               exit(1);
                       }
		       fclose(fp);

                       /* truncate newline */
		       p = filesecret + strlen(filesecret) - 1;
		       while ((p >= filesecret) &&
			      (*p < ' ')) {
			       *p = '\0';
			       --p;
		       }

                       if (strlen(filesecret) < 2) {
                               fprintf(stderr, "eapclient: Secret in %s is too short\n", optarg);
                               exit(1);
                       }
                       secret = filesecret;
		       break;
		case 'h':
		default:
			usage();
			break;
	}
	argc -= (optind - 1);
	argv += (optind - 1);

	if ((argc < 3)  ||
	    ((secret == NULL) && (argc < 4))) {
		usage();
	}

	if (dict_init(radius_dir, RADIUS_DICTIONARY) < 0) {
		librad_perror("eapclient");
		return 1;
	}

	if ((req = rad_alloc(1)) == NULL) {
		librad_perror("eapclient");
		exit(1);
	}

	req->id = id;

	/*
	 *	Strip port from hostname if needed.
	 */
	if ((p = strchr(argv[1], ':')) != NULL) {
		*p++ = 0;
		port = atoi(p);
	}

	/*
	 *	See what kind of request we want to send.
	 */
	if (strcmp(argv[2], "auth") == 0) {
		if (port == 0) port = getport("radius");
		if (port == 0) port = PW_AUTH_UDP_PORT;
		req->code = PW_AUTHENTICATION_REQUEST;
	} else if (strcmp(argv[2], "acct") == 0) {
		if (port == 0) port = getport("radacct");
		if (port == 0) port = PW_ACCT_UDP_PORT;
		req->code = PW_ACCOUNTING_REQUEST;
		do_summary = 0;
	} else if (isdigit(argv[2][0])) {
		if (port == 0) port = getport("radius");
		if (port == 0) port = PW_AUTH_UDP_PORT;
		req->code = atoi(argv[2]);
	} else {
		usage();
	}

	/*
	 *	Resolve hostname.
	 */
	req->dst_port = port;
	req->dst_ipaddr = ip_getaddr(argv[1]);
	if (req->dst_ipaddr == INADDR_NONE) {
		librad_perror("eapclient: %s: ", argv[1]);
		exit(1);
	}

	/*
	 *	Add the secret.
	 */
	if (argv[3]) secret = argv[3];

	/*
	 *	Read valuepairs.
	 *	Maybe read them, from stdin, if there's no
	 *	filename, or if the filename is '-'.
	 */
	if (filename && (strcmp(filename, "-") != 0)) {
		fp = fopen(filename, "r");
		if (!fp) {
			fprintf(stderr, "eapclient: Error opening %s: %s\n",
				filename, strerror(errno));
			exit(1);
		}
	} else {
		fp = stdin;
	}
	
	/*
	 *	Send request.
	 */
	if ((req->sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("eapclient: socket: ");
		exit(1);
	}

    tls.tls_ctx = init_tls_ctx(KEYFILE, PASSWORD);
    tls.ssn = 0x00; 
	while(!filedone) {
		if(req->vps) pairfree(&req->vps);

		if ((req->vps = readvp(fp)) == NULL) {
			break;
		}
	
		if (veepee != NULL)
			pairfree(&veepee);

		if (pairfind(req->vps, PW_EAP_MESSAGE) != NULL) {
			pairmove2(&veepee, &(req->vps), PW_PASSWORD);
			//pairmove2(&veepee, &(req->vps), PW_USER_NAME);
		} else {
			/*
		 	 *	Keep a copy of the the User-Password attribute.
			 */
			if ((vp = pairfind(req->vps, PW_PASSWORD)) != NULL) {
				strNcpy(password, (char *)vp->strvalue, sizeof(vp->strvalue));

			/*
			 *      Otherwise keep a copy of the CHAP-Password attribute.
			 */
			} else if ((vp = pairfind(req->vps, PW_CHAP_PASSWORD)) != NULL) {
				strNcpy(password, (char *)vp->strvalue, sizeof(vp->strvalue));
			} else {
				*password = '\0';
			}
		}
	
		/*
		 *	Loop, sending the packet N times.
		 */
		for (loop = 0; loop < count; loop++) {
			req->id++;
	
			/*
			 *	If we've already sent a packet, free up the old
			 *	one, and ensure that the next packet has a unique
			 *	ID and authentication vector.
			 */
			if (req->data) {
				free(req->data);
				req->data = NULL;
			}

			librad_md5_calc(req->vector, req->vector,
						   sizeof(req->vector));
                        if (*password != '\0') {
                                vp = pairfind(req->vps, PW_PASSWORD);
                                if (vp) {
                                        strNcpy((char *)vp->strvalue, password, vp->length + 1);
                                        vp->length = strlen(password);

                                        rad_pwencode((char *)vp->strvalue,
                                                     &(vp->length),
                                                     secret, (char *)req->vector);
                                } else if ((vp = pairfind(req->vps, PW_CHAP_PASSWORD)) != NULL) {
                                        strNcpy((char *)vp->strvalue, password, vp->length + 1);
                                        vp->length = strlen(password);

                                        rad_chap_encode(req, (char *) vp->strvalue, req->id, vp);
                                        vp->length = 17;
                                }
			} /* there WAS a password */

			/*
			eap_start(pairfind(req->vps, PW_EAP_MESSAGE));
			*/
			eap_identity(pairfind(req->vps, PW_EAP_MESSAGE), pairfind(req->vps, PW_USER_NAME));
			send_packet(req, &rep, veepee);
			rad_free(&rep);
		}
	}
	if(do_summary) {
		printf("\n\t   Total approved auths:  %d\n", totalapp);
		printf("\t     Total denied auths:  %d\n", totaldeny);
	}
	return 0;
}
