/*
 * flowmux.c	
 *
 * Version:	$Id:$
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright 2000  The FreeRADIUS server project
 * Copyright 2001  Mark Fullmer and The Ohio State University
 * Copyright 2002  Nikolay P. Romanyuk <mag@vtelecom.ru>
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/uio.h>

#include <netinet/in.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stddef.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <syslog.h>
#include <unistd.h>

#include "flowmux.h"

int debug = 0;
struct ip_param *pdu = NULL;
struct ip_param *exp = NULL;
struct ip_param *local = NULL;
struct ip_count *counts = NULL;
int client[MAX_CONNS];

void usage(char *name)
{
	printf("Usage: %s [-h] [-d] -p local/port -i local/remote/port [-e local/remote/port]\n", name);
	printf("\t-h - help. This.\n");
	printf("\t-d - debug mode, no daemonize.\n");
	printf("\t-p local/port - IP-addr/port for incoming RADIUS queries.\n");
	printf("\t-i local/remote/port - Flow import parameters.\n");
	printf("\t-e local/remote/port - Flow export parameters.\n");
}

void flowlog(const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	if ( debug == 0 ) {
		vsyslog(LOG_NOTICE, fmt, ap);
	} else {
		vprintf(fmt, ap);
	}
	va_end(ap);
}

/* Main idea from Flow-Tools. See: http://www.splintered.net/sw/flow-tools/ */
in_addr_t get_ip(char *name)
{
	struct hostent *h;
	struct in_addr *in;
	u_long addr = 0;
	u_int n;
	int dns = 0;
	char *s;

	for ( s = name; *s; s++ ) {
		if ( isalpha(*s) ) {
			dns = 1;
			break;
		}
	}
	if ( dns ) {
		h = gethostbyname(s);
		if ( h == NULL ) goto num;
		if ( h->h_addrtype != PF_INET ) goto num;
		if ( h->h_length != sizeof(u_int32_t) ) goto num;
		in = (struct in_addr *) *h->h_addr_list;
		return (in->s_addr);
	}
num:
	s = name;
	while (1) {
		n = 0;
		while (*s && (*s != '.') && (*s != ' ') && (*s != '\t'))
			n = n * 10 + *s++ - '0';
		addr <<=8;
		addr |= n & 0xff;
		if ((!*s) || (*s == ' ') || (*s == '\t'))
			return(htonl(addr));
		++s;
	}
}

/* Main idea from Flow-Tools. See: http://www.splintered.net/sw/flow-tools/ */
struct ip_param *
conf_ip(char *arg, int flag)
{
	struct ip_param *p;
	char *s, *locip, *remip, *port, *s1;

	locip = remip = port = NULL;
	s = malloc(strlen(arg) + 1);
	if ( s == NULL ) {
		flowlog("malloc: %s", strerror(errno));
		exit(EX_UNAVAILABLE);
	}
	s1 = s;
	strcpy(s, arg);

	locip = s;
	if ( flag ) {
		for (; *s && *s != '/'; ++s );
		if ( *s ) {
			*s = '\0';
			remip = ++s;
		}
	}

	for (; *s && *s != '/'; ++s );
	if ( *s ) {
		*s = '\0';
		port = ++s;
	}

	p = malloc(sizeof(struct ip_param));
	if ( p == NULL ) {
		flowlog("malloc: %s", strerror(errno));
		exit(EX_UNAVAILABLE);
	}
	bzero(p, sizeof(struct ip_param));

	if ( locip ) p->loc_ip = get_ip(locip);
	else return(NULL);
	
	if ( remip ) p->rem_ip = get_ip(remip);
	else if ( flag ) return(NULL);

	if ( port ) p->port = atoi(port);
	else return(NULL);

	free(s1);

	return(p);
}

/* Main idea from Flow-Tools. See: http://www.splintered.net/sw/flow-tools/ */
int pdu_decode(struct msgbuf *buf)
{
	struct pdu_header *ph;
	int size, agg_metod, agg_version, n;
	u_int32_t ip, octets;
	struct pdu_v1 *pdu_v1;
	struct pdu_v5 *pdu_v5;
	struct pdu_v6 *pdu_v6;
	struct pdu_v7 *pdu_v7;
	struct pdu_v8_8 *pdu_v8_8;
	struct ip_count *c;
	u_int32_t net_c;
	u_char addr;

	if ( buf->inb < 4 ) return(-1);

	ph = (struct pdu_header*)&buf->buf;
#if BYTE_ORDER == LITTLE_ENDIAN
	SWAPINT16(ph->version);
	SWAPINT16(ph->count);
	SWAPINT32(ph->unix_secs);
#endif /* LITTLE_ENDIAN */
	switch ( ph->version ) {
	case 1:
		if ( ph->count > FT_PDU_V1_MAXFLOWS )
			goto bad;
		size = offsetof(struct pdu_v1, records) +
				ph->count * sizeof (struct rec_v1);
		if ( size != buf->inb )
			goto bad;
		pdu_v1 = (struct pdu_v1*)&buf->buf;
		for ( n = 0; n < ph->count; n++ ) {
			ip = pdu_v1->records[n].dstaddr;
			octets = pdu_v1->records[n].dOctets;
#if BYTE_ORDER == LITTLE_ENDIAN
			SWAPINT32(octets);
#endif /* LITTLE_ENDIAN */
			net_c = ip & 0xFFFFFF00;
			addr = ip | 0xFFFFFF00;

			for ( c = counts; c; c = c->next ) {
				if ( c->addr == net_c &&
					c->rec[addr].state == FLOW_ACCT_ON ) {
					if ( ph->unix_secs >= c->rec[addr].timestamp ) {
						c->rec[addr].count += octets;
						break;
					} else {
						flowlog("timestamp: acct %ld, packet %ld", c->rec[addr].timestamp, ph->unix_secs);
					}
				}
			}
		}
		break;
	case 5:
		if ( ph->count > FT_PDU_V5_MAXFLOWS )
			goto bad;
		size = offsetof(struct pdu_v5, records) +
			ph->count * sizeof (struct rec_v5);
		if ( size != buf->inb )
			goto bad;
		pdu_v5 = (struct pdu_v5*)&buf->buf;
		for ( n = 0; n < ph->count; n++ ) {
			ip = pdu_v5->records[n].dstaddr;
			octets = pdu_v5->records[n].dOctets;
#if BYTE_ORDER == LITTLE_ENDIAN
			SWAPINT32(octets);
#endif /* LITTLE_ENDIAN */
			net_c = ip & 0xFFFFFF00;
			addr = ip | 0xFFFFFF00;

			for ( c = counts; c; c = c->next ) {
				if ( c->addr == net_c &&
					c->rec[addr].state == FLOW_ACCT_ON ) {
					if ( ph->unix_secs >= c->rec[addr].timestamp ) {
						c->rec[addr].count += octets;
						break;
					} else {
						flowlog("timestamp: acct %ld, packet %ld", c->rec[addr].timestamp, ph->unix_secs);
					}
				}
			}
		}
		break;
	case 6:
		if ( ph->count > FT_PDU_V6_MAXFLOWS )
			goto bad;
		size = offsetof(struct pdu_v6, records) +
			ph->count * sizeof (struct rec_v6);
		if ( size != buf->inb )
			goto bad;
		pdu_v6 = (struct pdu_v6*)&buf->buf;
		for ( n = 0; n < ph->count; n++ ) {
			ip = pdu_v6->records[n].dstaddr;
			octets = pdu_v6->records[n].dOctets;
#if BYTE_ORDER == LITTLE_ENDIAN
			SWAPINT32(octets);
#endif /* LITTLE_ENDIAN */
			net_c = ip & 0xFFFFFF00;
			addr = ip | 0xFFFFFF00;

			for ( c = counts; c; c = c->next ) {
				if ( c->addr == net_c &&
					c->rec[addr].state == FLOW_ACCT_ON ) {
					if ( ph->unix_secs >= c->rec[addr].timestamp ) {
						c->rec[addr].count += octets;
						break;
					} else {
						flowlog("timestamp: acct %ld, packet %ld", c->rec[addr].timestamp, ph->unix_secs);
					}
				}
			}
		}
		break;
	case 7:
		if ( ph->count > FT_PDU_V7_MAXFLOWS )
			goto bad;
		size = offsetof(struct pdu_v6, records) +
			ph->count * sizeof (struct rec_v7);
		if ( size != buf->inb )
			goto bad;
		pdu_v7 = (struct pdu_v7*)&buf->buf;
		for ( n = 0; n < ph->count; n++ ) {
			ip = pdu_v7->records[n].dstaddr;
			octets = pdu_v7->records[n].dOctets;
#if BYTE_ORDER == LITTLE_ENDIAN
			SWAPINT32(octets);
#endif /* LITTLE_ENDIAN */
			net_c = ip & 0xFFFFFF00;
			addr = ip | 0xFFFFFF00;

			for ( c = counts; c; c = c->next ) {
				if ( c->addr == net_c &&
					c->rec[addr].state == FLOW_ACCT_ON ) {
					if ( ph->unix_secs >= c->rec[addr].timestamp ) {
						c->rec[addr].count += octets;
						break;
					} else {
						flowlog("timestamp: acct %ld, packet %ld", c->rec[addr].timestamp, ph->unix_secs);
					}
				}
			}
		}
		break;
	case 8:
		if ( buf->inb < (offsetof(struct pdu_v8_gen, agg_version) +
				sizeof ((struct pdu_v8_gen *)0)->agg_version))
			goto bad;
		agg_metod = ((struct pdu_v8_gen *)&buf->buf)->aggregation;
		agg_version = ((struct pdu_v8_gen *)&buf->buf)->agg_version;

		/* XXX Juniper hack */
		if ( agg_version == 0 ) agg_version = 2;

		/* can only decode version 2 aggregation method packets */
		if ( agg_version != 2 )
			goto bad;

		switch ( agg_metod ) {
		case 1:
			if ( ph->count > FT_PDU_V8_1_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_1, records) +
				ph->count * sizeof (struct rec_v8_1);
			if ( size != buf->inb )
				goto bad;
			break;
		case 2:
			if ( ph->count > FT_PDU_V8_2_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_2, records) +
				ph->count * sizeof (struct rec_v8_2);
			if ( size != buf->inb )
				goto bad;
			break;
		case 3:
			if ( ph->count > FT_PDU_V8_3_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_3, records) +
				ph->count * sizeof (struct rec_v8_3);
			if ( size != buf->inb )
				goto bad;
			break;
		case 4:
			if ( ph->count > FT_PDU_V8_4_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_4, records) +
				ph->count * sizeof (struct rec_v8_4);
			if ( size != buf->inb )
				goto bad;
			break;
		case 5:
			if ( ph->count > FT_PDU_V8_5_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_5, records) +
				ph->count * sizeof (struct rec_v8_5);
			if ( size != buf->inb )
				goto bad;
			break;
		case 6:
			if ( ph->count > FT_PDU_V8_6_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_6, records) +
				ph->count * sizeof (struct rec_v8_6);
			if ( size != buf->inb )
				goto bad;
			break;
		case 7:
			if ( ph->count > FT_PDU_V8_7_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_7, records) +
				ph->count * sizeof (struct rec_v8_7);
			if ( size != buf->inb )
				goto bad;
			break;
		case 8:
			if ( ph->count > FT_PDU_V8_8_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_8, records) +
				ph->count * sizeof (struct rec_v8_8);
			if ( size != buf->inb )
				goto bad;
			pdu_v8_8 = (struct pdu_v8_8*)&buf->buf;
			for ( n = 0; n < ph->count; n++ ) {
				ip = pdu_v8_8->records[n].dstaddr;
				octets = pdu_v8_8->records[n].dOctets;
#if BYTE_ORDER == LITTLE_ENDIAN
				SWAPINT32(octets);
#endif /* LITTLE_ENDIAN */
				net_c = ip & 0xFFFFFF00;
				addr = ip | 0xFFFFFF00;

				for ( c = counts; c; c = c->next ) {
					if ( c->addr == net_c &&
						c->rec[addr].state == FLOW_ACCT_ON ) {
						if ( ph->unix_secs >= c->rec[addr].timestamp ) {
							c->rec[addr].count += octets;
							break;
						} else {
							flowlog("timestamp: acct %ld, packet %ld", c->rec[addr].timestamp, ph->unix_secs);
						}
					}
				}
			}
			break;
		case 9:
			if ( ph->count > FT_PDU_V8_9_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_9, records) +
				ph->count * sizeof (struct rec_v8_9);
			if ( size != buf->inb )
				goto bad;
			break;
		case 10:
			if ( ph->count > FT_PDU_V8_10_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_10, records) +
				ph->count * sizeof (struct rec_v8_10);
			if ( size != buf->inb )
				goto bad;
			break;
		case 11:
			if ( ph->count > FT_PDU_V8_11_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_11, records) +
				ph->count * sizeof (struct rec_v8_11);
			if ( size != buf->inb )
				goto bad;
			break;
		case 12:
			if ( ph->count > FT_PDU_V8_12_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_12, records) +
				ph->count * sizeof (struct rec_v8_12);
			if ( size != buf->inb )
				goto bad;
			break;
		case 13:
			if ( ph->count > FT_PDU_V8_13_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_13, records) +
				ph->count * sizeof (struct rec_v8_13);
			if ( size != buf->inb )
				goto bad;
			break;
		case 14:
			if ( ph->count > FT_PDU_V8_14_MAXFLOWS )
				goto bad;
			size = offsetof(struct pdu_v8_14, records) +
				ph->count * sizeof (struct rec_v8_14);
			if ( size != buf->inb )
				goto bad;
			break;
		}		
		break;
	default:
		goto bad;
	}
#if BYTE_ORDER == LITTLE_ENDIAN
	SWAPINT16(ph->version);
	SWAPINT16(ph->count);
	SWAPINT32(ph->unix_secs);
#endif /* LITTLE_ENDIAN */
	return(0);
bad:
	return(-1);
}

int main(int ac, char **av)
{
	int i, len, len1, maxfd;
	struct timeval t;
	struct msgbuf buf;
	fd_set rfd, efd;
	struct cmd cmd;
	struct ip_count *c;
	u_int32_t net_c;
	u_char addr;
	int found;

	while ( (i = getopt(ac, av, "dh?p:i:e:")) != -1 ) {
		switch (i) {
		case 'd':
			debug = 1;
			break;
		case 'p':
			local = conf_ip(optarg, 0);	
			break;
		case 'i':
			pdu = conf_ip(optarg, 1);
			break;
		case 'e':
			exp = conf_ip(optarg, 1);
			break;
		case '?':
		case 'h':
		default:
			usage(av[0]);
			exit(EX_USAGE);
		}
	}
	if ( local == NULL ) {
		usage(av[0]);
		exit(EX_USAGE);
	}
	if ( pdu == NULL ) {
		usage(av[0]);
		exit(EX_USAGE);
	}

	/* Daemonize */
	if ( debug == 0 ) {
		found = fork();
		if ( found < 0 ) {
			perror("fork");
			exit(EX_OSERR);
		}
		if ( found > 0 ) {
			exit(EX_OK);
		}
		for ( i = 0; i < 20; i++ ) close(i);
		setsid();
		openlog("flowmux", LOG_PID, LOG_LOCAL0);
	}

	/* make socket for incoming RADIUS/STAT query */
	bzero(&local->loc_addr, sizeof(local->loc_addr));
	local->loc_addr.sin_family = PF_INET;
	local->loc_addr.sin_port = htons(local->port);
	local->loc_addr.sin_addr.s_addr = local->loc_ip;
	local->sock = socket(PF_INET, SOCK_STREAM, 0);
	if ( local->sock < 0 ) {
		flowlog("local socket: %s", strerror(errno));
		exit(EX_UNAVAILABLE);
	}
	len1 = 1;
	if ( setsockopt(local->sock, SOL_SOCKET, SO_REUSEADDR, (char *)&len1,
			sizeof(len1)) < 0 ) {
		flowlog("setsockopt(SO_REUSEADDR): %s", strerror(errno));
		exit(EX_UNAVAILABLE);
	}

	len = sizeof(local->loc_addr);
	if ( bind(local->sock, (struct sockaddr *)&local->loc_addr, len) < 0 ) {
		flowlog("local bind: %s", strerror(errno));
		exit(EX_UNAVAILABLE);
	}
	listen(local->sock, 5);
	if ( fcntl(local->sock, F_SETFL, O_NONBLOCK) < 0 ) {
		flowlog("fcntl: %s", strerror(errno));
		exit(EX_UNAVAILABLE);
	}

	/* make socket for incoming Flow data */
	bzero(&pdu->loc_addr, sizeof(pdu->loc_addr));
	pdu->loc_addr.sin_family = PF_INET;
	pdu->loc_addr.sin_port = htons(pdu->port);
	pdu->loc_addr.sin_addr.s_addr = pdu->loc_ip;
	pdu->sock = socket(PF_INET, SOCK_DGRAM, 0);
	if ( pdu->sock < 0 ) {
		flowlog("import socket: %s", strerror(errno));
		exit(EX_UNAVAILABLE);
	}
	len = sizeof(pdu->loc_addr);
	if ( bind(pdu->sock, (struct sockaddr *)&pdu->loc_addr, len) < 0 ) {
		flowlog("import bind: %s", strerror(errno));
		exit(EX_UNAVAILABLE);
	}

/* make socket for export Flow data */
	if ( exp ) {
		bzero(&exp->loc_addr, sizeof(exp->loc_addr));
		bzero(&exp->rem_addr, sizeof(exp->rem_addr));
		exp->loc_addr.sin_family = PF_INET;
		exp->loc_addr.sin_addr.s_addr = exp->loc_ip;
		exp->sock = socket(PF_INET, SOCK_DGRAM, 0);
		if ( exp->sock < 0 ) {
			flowlog("export socket: %s", strerror(errno));
			exit(EX_UNAVAILABLE);
		}
		len = sizeof(exp->loc_addr);
		if ( bind(exp->sock, (struct sockaddr *)&exp->loc_addr,
						len) < 0 ) {
			flowlog("export bind: %s", strerror(errno));
			exit(EX_UNAVAILABLE);
		}
		len = sizeof(exp->rem_addr);
		exp->rem_addr.sin_family = PF_INET;
		exp->rem_addr.sin_port = htons(exp->port);
		exp->rem_addr.sin_addr.s_addr = exp->rem_ip;
	}

	buf.iov[0].iov_len = sizeof(buf.buf);
	buf.iov[0].iov_base = (char *)&buf.buf;
	buf.msg.msg_iov = (struct iovec *)&buf.iov;
	buf.msg.msg_iovlen = 1;
	buf.msg.msg_name = &pdu->rem_addr;
	buf.msg.msg_namelen = sizeof(pdu->rem_addr);
	buf.msg.msg_control = &buf.msgip;
	buf.msg.msg_controllen = sizeof(buf.msgip);

	len1 = sizeof(local->rem_addr);

	/* main loop */
	while (1) {
		bzero(&t, sizeof(t));
		t.tv_sec = SELECT_TIMEOUT;

		FD_ZERO(&rfd);
		FD_ZERO(&efd);

		FD_SET(local->sock, &rfd);
		FD_SET(pdu->sock, &rfd);

		maxfd = MYMAX(pdu->sock, local->sock);

		/* add clients sockets into rfd and efd */
		for ( i = 0; i < MAX_CONNS; i++ ) {
			if ( client[i] ) {
				maxfd = MYMAX(maxfd, client[i]);
				FD_SET(client[i], &rfd);
				FD_SET(client[i], &efd);
			}
		}

		if ( select(maxfd + 1, &rfd, (fd_set *)NULL, &efd, &t) < 0 ) {
			if ( errno == EINTR ) {
				FD_ZERO(&rfd);
			} else {
				flowlog("select(poll)");
			}
/* XXX ??? */		continue;
		}

		if ( FD_ISSET(local->sock, &rfd) ) {
			/* incoming client (RADIUSd/STATd) connection */
			for ( i = 0; i < MAX_CONNS; i++ ) {
				if ( client[i] == 0 ) {
					client[i] = accept(local->sock, (struct sockaddr *)&local->rem_addr, &len1);
					fcntl(client[i], F_SETFL, O_NONBLOCK);
					if ( client[i] < 0 ) {
						client[i] = 0;
						flowlog("accept: %s", strerror(errno));
					}
					break;
				}
			}
			if ( i == MAX_CONNS ) {
				flowlog("Many connections: %d", i);
			}
		}

		for ( i = 0; i < MAX_CONNS; i++ ) {
			if ( client[i] ) {
				if ( FD_ISSET(client[i], &efd) ) {
					close(client[i]);
					client[i] = 0;
				}
				if ( FD_ISSET(client[i], &rfd) ) {
					if ( read(client[i], &cmd, sizeof(cmd)) <= 0 ) {
						close(client[i]);
						client[i] = 0;
						break;
					}
					net_c = cmd.addr & 0xFFFFFF00;
					addr = cmd.addr | 0xFFFFFF00;

					switch ( cmd.cmd ) {
					case FLOW_ACCT_START:
						found = 0;
						for ( c = counts; c; c = c->next ) {
							if ( c->addr == net_c ) {
								c->rec[addr].count = 0;
							/* XXX 1 sec lag for PW_ACCT_START */
								c->rec[addr].timestamp = time(NULL) - 1;
								c->rec[addr].state = FLOW_ACCT_ON;
								found = 1;
								break;
							}
						}
						if ( found == 0 ) {
							if ( counts == NULL ) {
								counts = malloc(sizeof(struct ip_count));
								if ( counts == NULL ) {
									flowlog("malloc %s", strerror(errno));
									break;
								}
								c = counts;
								goto init;
							}
							for ( c = counts; c->next; c = c->next ) ;
							c->next = malloc(sizeof(struct ip_count));
							if ( c->next == NULL ) {
								flowlog("malloc %s", strerror(errno));
								break;
							}
							c = c->next;
init:
							bzero(c, sizeof(struct ip_count));
							c->addr = net_c;
							c->rec[addr].state = FLOW_ACCT_ON;
						}
						break;
					case FLOW_ACCT_SEND:
						for ( c = counts; c; c = c->next ) {
							if ( c->addr == net_c ) {
								write(client[i], (char *)&c->rec[addr].count, sizeof(c->rec[addr].count));
								break;
							}
						}
						break;
					case FLOW_ACCT_STOP:
						for ( c = counts; c; c = c->next ) {
							if ( c->addr == net_c ) {
								write(client[i], (char *)&c->rec[addr].count, sizeof(c->rec[addr].count));
								c->rec[addr].state = FLOW_ACCT_OFF;
								c->rec[addr].count = 0;
								break;
							}
						}
						break;
					}
				}
			}
		}
		
		if ( FD_ISSET(pdu->sock, &rfd) ) {
			if ( (buf.inb = recvmsg(pdu->sock, &buf.msg, 0)) < 0 ) {
				flowlog("recvmsg %s", strerror(errno));
				continue;
			}
			/* Check pdu src IP address, if need */
			if ( pdu->rem_ip &&
				pdu->rem_addr.sin_addr.s_addr != pdu->rem_ip ) {
				flowlog("Wrong importer src IP! %X", pdu->rem_addr.sin_addr.s_addr);
				continue;
			}
			/* Check packet */
			if ( pdu_decode(&buf) ) {
				flowlog("Wrong packet!");
				continue;
			}
			if ( exp ) {
				sendto(exp->sock, buf.buf, buf.inb, 0,
				(struct sockaddr *)&exp->rem_addr, len);
			}
		}
	}
	exit(EX_OK);
}
