static const char rcsid[] =
	"$Id: rlm_sql.c,v 1.68 2001/11/09 00:53:11 kkalev Exp $";

#include "autoconf.h"

#include <stdio.h>
#include <sys/stat.h>
#include <stdlib.h>

#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <string.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <xmlrpc.h>
#include <xmlrpc_client.h>

#include "radiusd.h"
#include "modules.h"
#include "conffile.h"
#include "rlm_xmlrpc.h"

#define RLM_XMLRPC_NAME "XML-RPC Radius Accounting Module"
#define RLM_XMLRPC_VERSION "0.1"


static CONF_PARSER module_config[] = {
	{"rpcuri",PW_TYPE_STRING_PTR, offsetof(xmlrpc_instance,rpcuri), NULL, "http://localhost/"},
	{"rpcagent", PW_TYPE_STRING_PTR, offsetof(xmlrpc_instance,rpcagent), NULL, "AcctServer.Process"},
	{"login", PW_TYPE_STRING_PTR, offsetof(xmlrpc_instance,login), NULL, ""},
	{"password", PW_TYPE_STRING_PTR, offsetof(xmlrpc_instance,password), NULL, ""},
	{"start_encryption", PW_TYPE_INTEGER, offsetof(xmlrpc_instance,start_encryption), NULL, "0"},
	{"encryption_key", PW_TYPE_STRING_PTR, offsetof(xmlrpc_instance,encryption_key), NULL, ""},
	{"debug", PW_TYPE_INTEGER, offsetof(xmlrpc_instance,debug), NULL, "0"},
	{"encryption_mode", PW_TYPE_INTEGER, offsetof(xmlrpc_instance,encryption_mode), NULL, "0"},
	{"xmlrpc_debug", PW_TYPE_INTEGER, offsetof(xmlrpc_instance,xmlrpc_debug), NULL, "0"},
	{"timeout", PW_TYPE_INTEGER, offsetof(xmlrpc_instance,timeout.tv_sec), NULL, "20"},

	{NULL, -1, 0, NULL, NULL}
};

int init_xmlrpc_request(xmlrpc_env *env,xmlrpc_value **req_struct);


/***********************************************************************
 * start of main routines
 ***********************************************************************/
static int rlm_xmlrpc_init(void) {

	/*
	 * FIXME:
	 * We should put the sqlsocket array here once
	 * the module code is reworked to not unload
	 * modules on HUP.  This way we can have
	 * persistant connections.  -jcarneal
	 */
	return 0;
}


static int rlm_xmlrpc_instantiate(CONF_SECTION * conf, void **instance) 
{

	xmlrpc_instance  *inst;

	inst = rad_malloc(sizeof *inst);

	if (cf_section_parse(conf, inst, module_config) < 0) {
		free(inst);
		radlog(L_ERR,"error parsing configuration file");
		return -1;
	}

	if (inst->rpcuri == NULL) {
		radlog(L_ERR, "rlm_xmlrpc: missing 'rpcuri' directive.");
		free(inst);
		return -1;
	}

	if (inst->rpcagent == NULL) {
		radlog(L_ERR, "rlm_xmlrpc: missing 'rpcagent' directive.");
		free(inst);
		return -1;
	}
 
	inst->timeout.tv_usec = 0;

	*instance = inst;

	return 0;
}

static int rlm_xmlrpc_destroy(void) {

	return 0;
}

static int rlm_xmlrpc_detach(void *instance) {

	xmlrpc_instance *inst = instance;


	free(inst);

	return 0;
}

void rlm_xmlrpc_free(void *instance,xmlrpc_env *env)
{
    xmlrpc_env_clean(env);
    /* Shutdown our XML-RPC client library. */
    xmlrpc_client_cleanup();
    free(env);
}



/*
 *	Accounting: save the account data to our sql table
 */
int check_xmlrpc_fault(xmlrpc_env *env)
{
    if (env->fault_occurred) {
        radlog(L_ERR, "RLM-XML-RPC Fault: %s (%d)\n",
                env->fault_string, env->fault_code);
		return 1;
    }
	return 0;

}

static int rlm_xmlrpc_preaccounting(void *instance, REQUEST * request) 
{
	return 0;
}

static int rlm_xmlrpc_accounting(void *instance, REQUEST * request) 
{

	VALUE_PAIR *pair;
	xmlrpc_instance *inst = instance;
	int     acctstatustype = 0;
    xmlrpc_value *result;
    xmlrpc_value *strct;
    xmlrpc_value *item;
	xmlrpc_env *env;
	int     acctsessiontime = 0;
	char    logstr[1024];
	char buffer[MAX_STRING_LEN * 4];
	int res;
	
  /* Start up our XML-RPC client library. */
    xmlrpc_client_init(XMLRPC_CLIENT_NO_FLAGS, RLM_XMLRPC_NAME, RLM_XMLRPC_VERSION);

   /* Initialize our error-handling environment. */
    env=(xmlrpc_env *)rad_malloc(sizeof(xmlrpc_env));
    xmlrpc_env_init(env);


	/*
	 * Find the Acct Status Type
	 */
	if ((pair = pairfind(request->packet->vps, PW_ACCT_STATUS_TYPE)) != NULL) 
	{
		acctstatustype = pair->lvalue;
	} 
	else 
	{
		radius_xlat(logstr, 1024, "rlm_xmlrpc:  packet has no account status type.  [user '%{User-Name}', nas '%{NAS-IP-Address}']", request, NULL);
		radlog(L_ERR, logstr);
		rlm_xmlrpc_free(instance,env);
		return RLM_MODULE_INVALID;
	}

	/*
	 * If stop but zero session length AND no previous
	 * session found, drop it as in invalid packet 
	 * This is to fix CISCO's aaa from filling our  
	 * table with bogus crap
	 */
	if ((pair = pairfind(request->packet->vps, PW_ACCT_SESSION_TIME)) != NULL)
		acctsessiontime = pair->lvalue;

	if ((acctsessiontime <= 0) && (acctstatustype == PW_STATUS_STOP)) {
		radius_xlat(logstr, 1024, "rlm_xmlrpc:  Stop packet with zero session length.  (user '%{User-Name}', nas '%{NAS-IP-Address}')", request, NULL);
		radlog(L_ERR, logstr);
		rlm_xmlrpc_free(instance,env);
		return RLM_MODULE_FAIL;
	}

	strct=xmlrpc_struct_new(env);
	if(check_xmlrpc_fault(env))
	{
		rlm_xmlrpc_free(instance,env);
		return RLM_MODULE_FAIL;
	}
	init_xmlrpc_request(env,&strct);

	item=xmlrpc_build_value(env, "i",acctstatustype);
	if(check_xmlrpc_fault(env))
	{
		rlm_xmlrpc_free(instance,env);
		return RLM_MODULE_FAIL;
	}
	xmlrpc_struct_set_value (env,strct, "ACCTSTATUSTYPE",item);
	xmlrpc_DECREF(item);
	if(check_xmlrpc_fault(env))
	{
		rlm_xmlrpc_free(instance,env);
		return RLM_MODULE_FAIL;
	}
	if(acctstatustype==PW_STATUS_START)
	{
		item=xmlrpc_build_value(env, "i",(xmlrpc_int32)time(0));
		if(check_xmlrpc_fault(env))
		{
			rlm_xmlrpc_free(instance,env);
			return RLM_MODULE_FAIL;
		}
		xmlrpc_struct_set_value (env,strct, "ACCTSTARTTIME",item);
		xmlrpc_DECREF(item);
		if(check_xmlrpc_fault(env))
		{
			rlm_xmlrpc_free(instance,env);
			return RLM_MODULE_FAIL;
		}
	}
	else
	if(acctstatustype==PW_STATUS_STOP)
	{
		item=xmlrpc_build_value(env, "i",(xmlrpc_int32)time(0));
		if(check_xmlrpc_fault(env))
		{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
		xmlrpc_struct_set_value (env,strct, "ACCTSTOPTIME",item);
		xmlrpc_DECREF(item);
		if(check_xmlrpc_fault(env))
			{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
	}

	pair=request->packet->vps;
	while(pair)
	{
		switch(pair->attribute)
		{
		case PW_ACCT_SESSION_ID:
			item=xmlrpc_build_value(env, "s",pair->strvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "ACCTSESSIONID",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }

		case PW_ACCT_UNIQUE_SESSION_ID:
			item=xmlrpc_build_value(env, "s",pair->strvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "ACCTUNIQUEID",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_USER_NAME:
			item=xmlrpc_build_value(env, "s",pair->strvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "USERNAME",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_REALM:
			item=xmlrpc_build_value(env, "s",pair->strvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "REALM",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_NAS_IP_ADDRESS:
			item=xmlrpc_build_value(env, "s",pair->strvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "NASIPADDRESS",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_NAS_PORT_ID:
			vp_prints_value(buffer, sizeof(buffer), pair, 0);
			item=xmlrpc_build_value(env, "s", buffer);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "NASPORTID",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_NAS_PORT_TYPE:
			vp_prints_value(buffer, sizeof(buffer), pair, 0);
			item=xmlrpc_build_value(env, "s",buffer);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "NASPORTTYPE",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_ACCT_SESSION_TIME:
			item=xmlrpc_build_value(env, "i", (xmlrpc_int32)pair->lvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "ACCTSESSIONTIME",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_ACCT_AUTHENTIC:
			vp_prints_value(buffer, sizeof(buffer), pair, 0);
			item=xmlrpc_build_value(env, "s",buffer);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "ACCTAUTHENTIC",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_CONNECT_INFO:
			if(acctstatustype==PW_STATUS_START)
			{
				item=xmlrpc_build_value(env, "s", pair->strvalue);
				if(check_xmlrpc_fault(env))
					{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
				xmlrpc_struct_set_value (env,strct, "CONNECTINFO_START",item);
				xmlrpc_DECREF(item);
				if(check_xmlrpc_fault(env))
					{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			}
			else
			if(acctstatustype==PW_STATUS_STOP)
			{
				item=xmlrpc_build_value(env, "s", pair->strvalue);
				if(check_xmlrpc_fault(env))
					{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
				xmlrpc_struct_set_value (env,strct, "CONNECTINFO_STOP",item);
				xmlrpc_DECREF(item);
				if(check_xmlrpc_fault(env))
					{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			}
			break;
		case PW_ACCT_INPUT_OCTETS:
			item=xmlrpc_build_value(env, "i", (xmlrpc_int32)pair->lvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "ACCTINPUTOCTETS",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_ACCT_OUTPUT_OCTETS:
			item=xmlrpc_build_value(env, "i",(xmlrpc_int32)pair->lvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "ACCTOUTPUTOCTETS",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_CALLED_STATION_ID:
			item=xmlrpc_build_value(env, "s", pair->strvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "CALLEDSTATIONID",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_CALLING_STATION_ID:
			item=xmlrpc_build_value(env, "s",pair->strvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "CALLINGSTATIONID",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_ACCT_TERMINATE_CAUSE:
			vp_prints_value(buffer, sizeof(buffer), pair, 0);
			item=xmlrpc_build_value(env, "s",buffer);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "ACCTTERMINATECAUSE",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_SERVICE_TYPE:
			vp_prints_value(buffer, sizeof(buffer), pair, 0);
			item=xmlrpc_build_value(env, "s",buffer);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "SERVICETYPE",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_FRAMED_PROTOCOL:
			vp_prints_value(buffer, sizeof(buffer), pair, 0);
			item=xmlrpc_build_value(env, "s", buffer);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "FRAMEDPROTOCOL",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_FRAMED_IP_ADDRESS:
			item=xmlrpc_build_value(env, "s", pair->strvalue);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			xmlrpc_struct_set_value (env,strct, "FRAMEDIPADDRESS",item);
			xmlrpc_DECREF(item);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			break;
		case PW_ACCT_DELAY_TIME:
			if(acctstatustype==PW_STATUS_START)
			{
				item=xmlrpc_build_value(env, "i", (xmlrpc_int32)pair->lvalue);
				if(check_xmlrpc_fault(env))
					{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
				xmlrpc_struct_set_value (env,strct,"ACCTSTARTDELAY", item);
				xmlrpc_DECREF(item);
				if(check_xmlrpc_fault(env))
					{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			}
			else
			if(acctstatustype==PW_STATUS_STOP)
			{
				item=xmlrpc_build_value(env, "i", (xmlrpc_int32)pair->lvalue);
				if(check_xmlrpc_fault(env))
					{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
				xmlrpc_struct_set_value (env,strct, "ACCTSTOPDELAY",item);
				xmlrpc_DECREF(item);
				if(check_xmlrpc_fault(env))
					{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
			}
			break;
		}/*end switch(pair->attribute)*/
		pair = pair->next;
	}//end while(pair)

	radlog(L_DBG,"calling XMLRPC Server:%s,%s\n",inst->rpcuri,inst->rpcagent);
	if((result =xmlrpc_client_call(env,inst->rpcuri,inst->rpcagent, "(V)", strct))==NULL)
	{
	 	radlog(L_ERR,"Error occured while calling XMLRPC Server");
		xmlrpc_DECREF(strct);
		rlm_xmlrpc_free(instance,env);
		return RLM_MODULE_FAIL;
	}
	xmlrpc_DECREF(strct);
	if(check_xmlrpc_fault(env))
		{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
 	radlog(L_DBG,"called XMLRPC Server");
   
    if(result)
	{
	 		radlog(L_DBG,"XMLRPC Returned Result is OK");
			xmlrpc_parse_value(env, result, "(i)", &res);
			xmlrpc_DECREF(result);
			if(check_xmlrpc_fault(env))
				{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }
	 		radlog(L_DBG,"XMLRPC Check Result Passed");
	}
	else
		{ rlm_xmlrpc_free(instance,env); return RLM_MODULE_FAIL; }

	rlm_xmlrpc_free(instance,env);

	radlog(L_DBG,"Finished sending the accounting query via XMLRPC");
	return res ? RLM_MODULE_OK:RLM_MODULE_FAIL;
	
}




module_t rlm_xmlrpc = {
	"XMLRPC",
	RLM_TYPE_THREAD_SAFE,	/* type: reserved */
	rlm_xmlrpc_init,		/* initialization */
	rlm_xmlrpc_instantiate,	/* instantiation */
	{
		NULL,	/* authentication */
		NULL,	/* authorization */
		rlm_xmlrpc_preaccounting,			/* preaccounting */
		rlm_xmlrpc_accounting,	/* accounting */
		NULL			/* checksimul */
	},
	rlm_xmlrpc_detach,		/* detach */
	rlm_xmlrpc_destroy,	/* destroy */
};

int init_xmlrpc_request(xmlrpc_env *env,xmlrpc_value **req_struct)
{
	xmlrpc_value *int_item,*str_item;
	xmlrpc_value *tmp_struct=*req_struct;
	int_item=xmlrpc_build_value(env, "i",(xmlrpc_int32)0);
	str_item=xmlrpc_build_value(env, "s","");

	xmlrpc_struct_set_value (env,tmp_struct, "ACCTSTATUSTYPE",int_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTSTARTTIME",int_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTSTOPTIME",int_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTSESSIONID",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTUNIQUEID",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "USERNAME",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "REALM",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "NASIPADDRESS",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "NASPORTID",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "NASPORTTYPE",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTSESSIONTIME",int_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTAUTHENTIC",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "CONNECTINFO_START",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "CONNECTINFO_STOP",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTINPUTOCTETS",int_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTOUTPUTOCTETS",int_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "CALLEDSTATIONID",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "CALLINGSTATIONID",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTTERMINATECAUSE",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "SERVICETYPE",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "FRAMEDPROTOCOL",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "FRAMEDIPADDRESS",str_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct,"ACCTSTARTDELAY", int_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	xmlrpc_struct_set_value (env,tmp_struct, "ACCTSTOPDELAY",int_item);
	if(check_xmlrpc_fault(env))
		return RLM_MODULE_FAIL;
	return RLM_MODULE_OK;
}
