/*
 * acct.c	Accounting routines.
 *
 */
char acct_sccsid[] =
"@(#)acct.c	2.11  Copyright 1994 Livingston Enterprises Inc\n"
"		     Copyright 1998 Cistron Internet Services B.V.";


#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/file.h>
#include	<sys/stat.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<time.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<signal.h>
#include	<errno.h>
#include	<sys/wait.h>

#include	"conf.h"

#include	"radiusd.h"
#include	"radutmp.h"

/*
 *	FIXME: this should be configurable.
 */
int	doradwtmp = 1;


static char trans[64] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
#define ENC(c) trans[c]

static char porttypes[] = "ASITX";


#define LOCK_LEN sizeof(struct radutmp)

/*
 *	UUencode 4 bits base64. We use this to turn a 4 byte field
 *	(an IP adres) into 6 bytes of ASCII. This is used for the
 *	wtmp file if we didn't find a short name in the naslist file.
 */
char *uue(void *in)
{
	int i;
	static unsigned char res[7];
	unsigned char *data = (char *)in;

	res[0] = ENC( data[0] >> 2 );
	res[1] = ENC( ((data[0] << 4) & 060) + ((data[1] >> 4) & 017) );
	res[2] = ENC( ((data[1] << 2) & 074) + ((data[2] >> 6) & 03) );
	res[3] = ENC( data[2] & 077 );

	res[4] = ENC( data[3] >> 2 );
	res[5] = ENC( (data[3] << 4) & 060 );
	res[6] = 0;

	for(i = 0; i < 6; i++) {
		if (res[i] == ' ') res[i] = '`';
		if (res[i] < 32 || res[i] > 127)
			printf("uue: protocol error ?!\n");
	}
	return res;
}


/*
 *	Convert a struct radutmp to a normal struct utmp
 *	as good as we possibly can.
 */
static void make_wtmp(struct radutmp *ut, struct utmp *wt, int status)
{
	char		buf[32];
	NAS		*cl;
	char		*s;

	/*
	 *	Fill out the UTMP struct for the radwtmp file.
	 *	(this one must be "last" - compatible).
	 */
#ifdef __linux__
	/*
	 *	Linux has a field for the client address.
	 */
	wt->ut_addr = ut->framed_address;
#endif
	/*
	 *	We use the tty field to store the terminal servers' port
	 *	and address so that the tty field is unique.
	 */
	s = "";
	if ((cl = nas_find(ntohl(ut->nas_address))) != NULL)
		s = cl->shortname;
	if (s == NULL || s[0] == 0) s = uue(&(ut->nas_address));
#if UT_LINESIZE > 9
	sprintf(buf, "%03d:%s", ut->nas_port, s);
#else
	sprintf(buf, "%02d%s", ut->nas_port, s);
#endif
	strncpy(wt->ut_line, buf, UT_LINESIZE);

	/*
	 *	We store the dynamic IP address in the hostname field.
	 */
#ifdef UT_HOSTSIZE
	if (ut->framed_address) {
		ipaddr2str(buf, ntohl(ut->framed_address));
		strncpy(wt->ut_host, buf, UT_HOSTSIZE);
	}
#endif
#ifdef __svr4__
	wt->ut_xtime = ut->time;
#else
	wt->ut_time = ut->time;
#endif
#ifdef USER_PROCESS
	/*
	 *	And we can use the ID field to store
	 *	the protocol.
	 */
	if (ut->proto == P_PPP)
		strcpy(wt->ut_id, "P");
	else if (ut->proto == P_SLIP)
		strcpy(wt->ut_id, "S");
	else
		strcpy(wt->ut_id, "T");
	wt->ut_type = status == PW_STATUS_STOP ? DEAD_PROCESS : USER_PROCESS;
#endif
	if (status == PW_STATUS_STOP)
		wt->ut_name[0] = 0;
}


/*
 *	Zap a user, or all users on a NAS, from the radutmp file.
 */
int radzap(UINT4 nasaddr, int port, char *user, time_t t)
{
	struct radutmp	u;
	struct utmp	wt;
	FILE		*fp;
	int		fd;
	UINT4		netaddr;

	if (t == 0) time(&t);
	fp = fopen(RADWTMP, "a");
	netaddr = htonl(nasaddr);

	if ((fd = open(RADUTMP, O_RDWR|O_CREAT, 0644)) >= 0) {
		int r;

		/*
		 *	Lock the utmp file, prefer lockf() over flock().
		 */
#if defined(F_LOCK) && !defined(BSD)
		(void)lockf(fd, F_LOCK, LOCK_LEN);
#else
		(void)flock(fd, LOCK_EX);
#endif
	 	/*
		 *	Find the entry for this NAS / portno combination.
		 */
		r = 0;
		while (read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (((nasaddr != 0 && netaddr != u.nas_address) ||
			      (port >= 0   && port    != u.nas_port) ||
			      (user != NULL && strcmp(u.login, user) != 0) ||
			       u.type != P_LOGIN))
				continue;
			/*
			 *	Match. Zap it.
			 */
			if (lseek(fd, -(off_t)sizeof(u), SEEK_CUR) < 0) {
				log(L_ERR, "Accounting: radzap: negative lseek!\n");
				lseek(fd, (off_t)0, SEEK_SET);
			}
			u.type = P_IDLE;
			u.time = t;
			write(fd, &u, sizeof(u));

			/*
			 *	Add a logout entry to the wtmp file.
			 */
			if (fp != NULL)  {
				make_wtmp(&u, &wt, PW_STATUS_STOP);
				fwrite(&wt, sizeof(wt), 1, fp);
			}
		}
		close(fd);
	}
	if (fp) fclose(fp);

	return 0;
}


/*
 *	Store logins in the RADIUS utmp file.
 */
int rad_accounting_new(AUTH_REQ *authreq, int dowtmp) {
	struct radutmp	ut, u;
	struct utmp	wt;
	VALUE_PAIR	*vp;
	int		rb_record = 0;
	int		status = -1;
	int		nas_address = 0;
	int		framed_address = 0;
	int		protocol = -1;
	FILE		*fp;
	time_t		t;
	int		fd;
	int		ret = 0;
	int		just_an_update = 0;
	int		port_seen = 0;
	int		nas_port_type = 0;
	int		off;

	/*
	 *	Which type is this.
	 */
	if ((vp = pairfind(authreq->request, PW_ACCT_STATUS_TYPE)) == NULL) {
		log(L_ERR, "Accounting: no Accounting-Status-Type record.");
		return -1;
	}
	status = vp->lvalue;
	if (status == PW_STATUS_ACCOUNTING_ON ||
	    status == PW_STATUS_ACCOUNTING_OFF) rb_record = 1;

	if (!rb_record &&
	    (vp = pairfind(authreq->request, PW_USER_NAME)) == NULL) do {
		int check1 = 0;
		int check2 = 0;

		/*
		 *	ComOS (up to and including 3.5.1b20) does not send
		 *	standard PW_STATUS_ACCOUNTING_XXX messages.
		 *
		 *	Check for:  o no Acct-Session-Time, or time of 0
		 *		    o Acct-Session-Id of "00000000".
		 *
		 *	We could also check for NAS-Port, that attribute
		 *	should NOT be present (but we don't right now).
	 	 */
		if ((vp = pairfind(authreq->request, PW_ACCT_SESSION_TIME))
		     == NULL || vp->lvalue == 0)
			check1 = 1;
		if ((vp = pairfind(authreq->request, PW_ACCT_SESSION_ID))
		     != NULL && vp->length == 8 &&
		     memcmp(vp->strvalue, "00000000", 8) == 0)
			check2 = 1;
		if (check1 == 0 || check2 == 0) {
#if 0 /* Cisco sometimes sends START records without username. */
			log(L_ERR, "Accounting: no username in record");
			return -1;
#else
			break;
#endif
		}
		log(L_INFO, "Accounting: converting reboot records.");
		if (status == PW_STATUS_STOP)
			status = PW_STATUS_ACCOUNTING_OFF;
		if (status == PW_STATUS_START)
			status = PW_STATUS_ACCOUNTING_ON;
		rb_record = 1;
	} while(0);

#ifdef NT_DOMAIN_HACK
        if (!rb_record && vp) {
		char buffer[AUTH_STRING_LEN];
		char *ptr;
		if ((ptr = strchr(vp->strvalue, '\\')) != NULL) {
			strncpy(buffer, ptr + 1, sizeof(buffer));
			buffer[sizeof(buffer) - 1] = 0;
			strcpy(vp->strvalue, buffer);
		}
	}
#endif

	/*
	 *	Add any specific attributes for this username.
	 */
	if (!rb_record && vp != NULL) {
		hints_setup(authreq->request);
		presuf_setup(authreq->request);
	}
	time(&t);
	memset(&ut, 0, sizeof(ut));
	memset(&wt, 0, sizeof(wt));
	ut.porttype = 'A';

	/*
	 *	First, find the interesting attributes.
	 */
	for (vp = authreq->request; vp; vp = vp->next) {
		switch (vp->attribute) {
			case PW_USER_NAME:
				strncpy(ut.login, vp->strvalue, RUT_NAMESIZE);
				strncpy(wt.ut_name, vp->strvalue, UT_NAMESIZE);
				break;
			case PW_LOGIN_IP_HOST:
			case PW_FRAMED_IP_ADDRESS:
				framed_address = vp->lvalue;
				ut.framed_address = htonl(vp->lvalue);
				break;
			case PW_FRAMED_PROTOCOL:
				protocol = vp->lvalue;
				break;
			case PW_NAS_IP_ADDRESS:
				nas_address = vp->lvalue;
				ut.nas_address = htonl(vp->lvalue);
				break;
			case PW_NAS_PORT_ID:
				ut.nas_port = vp->lvalue;
				port_seen = 1;
				break;
			case PW_ACCT_DELAY_TIME:
				ut.delay = vp->lvalue;
				break;
			case PW_ACCT_SESSION_ID:
				/*
				 *	If length > 8, only store the
				 *	last 8 bytes.
				 */
				off = vp->length - sizeof(ut.session_id);
				if (off < 0) off = 0;
				memcpy(ut.session_id, vp->strvalue + off,
					sizeof(ut.session_id));
				break;
			case PW_NAS_PORT_TYPE:
				if (vp->lvalue >= 0 && vp->lvalue <= 4)
					ut.porttype = porttypes[vp->lvalue];
				nas_port_type = vp->lvalue;
				break;
		}
	}

	/*
	 *	If we didn't find out the NAS address, use the
	 *	originator's IP address.
	 */
	if (nas_address == 0) {
		nas_address = authreq->ipaddr;
		ut.nas_address = htonl(nas_address);
	}

	if (protocol == PW_PPP)
		ut.proto = 'P';
	else if (protocol == PW_SLIP)
		ut.proto = 'S';
	else
		ut.proto = 'T';
	ut.time = t - ut.delay;
	make_wtmp(&ut, &wt, status);

	/*
	 *	See if this was a portmaster reboot.
	 */
	if (status == PW_STATUS_ACCOUNTING_ON && nas_address) {
		log(L_INFO, "NAS %s restarted (Accounting-On packet seen)",
			nas_name(nas_address));
		radzap(nas_address, -1, NULL, ut.time);
		return 0;
	}
	if (status == PW_STATUS_ACCOUNTING_OFF && nas_address) {
		log(L_INFO, "NAS %s rebooted (Accounting-Off packet seen)",
			nas_name(nas_address));
		radzap(nas_address, -1, NULL, ut.time);
		return 0;
	}

	/*
	 *	If we don't know this type of entry pretend we succeeded.
	 */
	if (status != PW_STATUS_START &&
	    status != PW_STATUS_STOP &&
	    status != PW_STATUS_ALIVE) {
		log(L_ERR, "NAS %s port %d unknown packet type %d)",
			nas_name(nas_address), ut.nas_port, status);
		return 0;
	}

	/*
	 *	Perhaps we don't want to store this record into
	 *	radutmp/radwtmp. We skip records:
	 *
	 *	- without a NAS-Port-Id (telnet / tcp access)
	 *	- with the username "!root" (console admin login)
	 *	- with Port-Type = Sync (leased line up/down)
	 */
	if (!port_seen ||
	    strncmp(ut.login, "!root", RUT_NAMESIZE) == 0
#if 0 /* I HATE Ascend - they label ISDN as sync */
	    || nas_port_type == PW_NAS_PORT_SYNC
#endif
	)
		return 0;

	/*
	 *	Enter into the radutmp file.
	 */
	if ((fd = open(RADUTMP, O_RDWR|O_CREAT, 0644)) >= 0) {
		int r;

		/*
		 *	Lock the utmp file, prefer lockf() over flock().
		 */
#if defined(F_LOCK) && !defined(BSD)
		(void)lockf(fd, F_LOCK, LOCK_LEN);
#else
		(void)flock(fd, LOCK_EX);
#endif
	 	/*
		 *	Find the entry for this NAS / portno combination.
		 */
		r = 0;
		while (read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (u.nas_address != ut.nas_address ||
			    u.nas_port    != ut.nas_port)
				continue;

			if (status == PW_STATUS_STOP &&
			    strncmp(ut.session_id, u.session_id, 
			     sizeof(u.session_id)) != 0) {
				/*
				 *	Don't complain if this is not a
				 *	login record (some clients can
				 *	send _only_ logout records).
				 */
				if (u.type == P_LOGIN)
					log(L_ERR,
		"Accounting: logout: entry for NAS %s port %d has wrong ID",
					nas_name(nas_address), u.nas_port);
				r = -1;
				break;
			}

			if (status == PW_STATUS_START &&
			    strncmp(ut.session_id, u.session_id, 
			     sizeof(u.session_id)) == 0  &&
			    u.time >= ut.time) {
				if (u.type == P_LOGIN) {
					log(L_INFO,
		"Accounting: login: entry for NAS %s port %d duplicate",
					nas_name(nas_address), u.nas_port);
					r = -1;
					dowtmp = 0;
					break;
				}
				log(L_ERR,
		"Accounting: login: entry for NAS %s port %d wrong order",
				nas_name(nas_address), u.nas_port);
				r = -1;
				break;
			}

			/*
			 *	FIXME: the ALIVE record could need
			 *	some more checking, but anyway I'd
			 *	rather rewrite this mess -- miquels.
			 */
			if (status == PW_STATUS_ALIVE &&
			    strncmp(ut.session_id, u.session_id, 
			     sizeof(u.session_id)) == 0  &&
			    u.type == P_LOGIN) {
				/*
				 *	Keep the original login time.
				 */
				ut.time = u.time;
				if (u.login[0] != 0)
					just_an_update = 1;
			}

			if (lseek(fd, -(off_t)sizeof(u), SEEK_CUR) < 0) {
				log(L_ERR, "Accounting: negative lseek!\n");
				lseek(fd, (off_t)0, SEEK_SET);
			}
			r = 1;
			break;
		}

		if (r >= 0 &&  (status == PW_STATUS_START ||
				status == PW_STATUS_ALIVE)) {
			ut.type = P_LOGIN;
			write(fd, &ut, sizeof(u));
		}
		if (status == PW_STATUS_STOP) {
			if (r > 0) {
				u.type = P_IDLE;
				u.time = ut.time;
				u.delay = ut.delay;
				write(fd, &u, sizeof(u));
			} else if (r == 0) {
				log(L_ERR,
		"Accounting: logout: login entry for NAS %s port %d not found",
				nas_name(nas_address), ut.nas_port);
				r = -1;
			}
		}
		close(fd);
	} else {
		log(L_ERR, "Accounting: %s: %s", RADUTMP, strerror(errno));
		ret = -1;
	}

	/*
	 *	Don't write wtmp if we don't have a username, or
	 *	if this is an update record and the original record
	 *	was already written.
	 */
	if ((status != PW_STATUS_STOP && wt.ut_name[0] == 0) || just_an_update)
		dowtmp = 0;

	/*
	 *	Write a RADIUS wtmp log file.
	 */
	if (dowtmp && (fp = fopen(RADWTMP, "a")) != NULL) {
		ret = 0;
		fwrite(&wt, sizeof(wt), 1, fp);
		fclose(fp);
	}


	return ret;
}



int rad_accounting_orig(AUTH_REQ *authreq, int authtype, char *f, SQLSOCK *socket) {
	time_t		nowtime;
	struct tm	*tim;
        char		datebuf[20];
	int		*sqlpid;
	int		sqlstatus;
	FILE		*backupfile;
	struct stat	backup;
	char		*valbuf;
	char		username[AUTH_STRING_LEN*2+1];
	SQLREC sqlrecord = {"", "", "", "", 0, "", "", 0, "", 0, "", "", 0, 0, "", "", "", "", "", "", 0};
	SQLREC backuprecord = {"", "", "",  "", 0, "", "", 0, "", 0, "", "", 0, 0, "", "", "", "", "", "", 0};
	VALUE_PAIR	*pair;
	int		ret = 0;
	int		length;

	pair = authreq->request;
	strncpy(sqlrecord.Realm, authreq->realm, SQLBIGREC);
	while(pair != (VALUE_PAIR *)NULL) {

           /* Check the pairs to see if they are anything we are interested in. */
            switch(pair->attribute) {
                case PW_ACCT_SESSION_ID:
                	strncpy(sqlrecord.AcctSessionId, pair->strvalue, SQLBIGREC);
                	break;
                	
                case PW_USER_NAME:
			if (strlen(pair->strvalue) > AUTH_STRING_LEN)
				length = AUTH_STRING_LEN;
			else
				length = strlen(pair->strvalue);
			sql_escape_string(username, pair->strvalue, length);
                	strncpy(sqlrecord.UserName, username, SQLBIGREC);
                	break;
                	
                case PW_NAS_IP_ADDRESS:
			ipaddr2str(sqlrecord.NASIPAddress, pair->lvalue);
                	break;

                case PW_NAS_PORT_ID:
                	sqlrecord.NASPortId = pair->lvalue;
                	break;

                case PW_NAS_PORT_TYPE:
			valbuf = (char *)dict_valgetname(pair->lvalue, pair->name);
			if(valbuf != (char *)NULL) {
               			strncpy(sqlrecord.NASPortType, valbuf, SQLBIGREC);
			}
			break;

                case PW_ACCT_STATUS_TYPE:
       			sqlrecord.AcctStatusTypeId = pair->lvalue;
			valbuf = (char *)dict_valgetname(pair->lvalue, pair->name);
			if(valbuf != (char *)NULL) {
                		strncpy(sqlrecord.AcctStatusType, valbuf, SQLBIGREC);
			}
			break;

                case PW_ACCT_SESSION_TIME:
                	sqlrecord.AcctSessionTime = pair->lvalue;
                	break;

                case PW_ACCT_AUTHENTIC:
			valbuf = (char *)dict_valgetname(pair->lvalue, pair->name);
			if(valbuf != (char *)NULL) {
                		strncpy(sqlrecord.AcctAuthentic, valbuf, SQLBIGREC);
			}
			break;

                case PW_CONNECT_INFO:
                	strncpy(sqlrecord.ConnectInfo, pair->strvalue, SQLBIGREC);
                	break;

                case PW_ACCT_INPUT_OCTETS:
                	sqlrecord.AcctInputOctets = pair->lvalue;
                	break;

                case PW_ACCT_OUTPUT_OCTETS:
                	sqlrecord.AcctOutputOctets = pair->lvalue;
                	break;

                case PW_CALLED_STATION_ID:
                	strncpy(sqlrecord.CalledStationId, pair->strvalue, SQLLILREC);
                	break;

                case PW_CALLING_STATION_ID:
                	strncpy(sqlrecord.CallingStationId, pair->strvalue, SQLLILREC);
                	break;

                case PW_ACCT_TERMINATE_CAUSE:
			valbuf = (char *)dict_valgetname(pair->lvalue, pair->name);
			if(valbuf != (char *)NULL) {
                		strncpy(sqlrecord.AcctTerminateCause, valbuf, SQLBIGREC);
			}
			break;

                case PW_SERVICE_TYPE:
			valbuf = (char *)dict_valgetname(pair->lvalue, pair->name);
			if(valbuf != (char *)NULL) {
                		strncpy(sqlrecord.ServiceType, valbuf, SQLBIGREC);
			}
			break;

                case PW_FRAMED_PROTOCOL:
			valbuf = (char *)dict_valgetname(pair->lvalue, pair->name);
			if(valbuf != (char *)NULL) {
                		strncpy(sqlrecord.FramedProtocol, valbuf, SQLBIGREC);
			}
			break;

                case PW_FRAMED_IP_ADDRESS:
			ipaddr2str(sqlrecord.FramedIPAddress, pair->lvalue);
                	break;

                case PW_ACCT_DELAY_TIME:
                	sqlrecord.AcctDelayTime = pair->lvalue;
                	break;

                default:
                	break;
		}

		pair = pair->next;
	}


        nowtime = time(0) - sqlrecord.AcctDelayTime;
        tim = localtime(&nowtime);
        strftime(datebuf, sizeof(datebuf), "%Y%m%d%H%M%S", tim);

        strncpy(sqlrecord.AcctTimeStamp, datebuf, 20);
       

	/* If backup file exists we know the database was down */
	if(stat(MYSQLBACKUP, &backup) == 0) {
		if(backup.st_size > 0) {

			/* We'll fork a child to load records in the backup file */
			(pid_t)sqlpid = fork();
			if(sqlpid > 0) {

				/* suspend the parent while child reads records */
				while(waitpid((pid_t)sqlpid, &sqlstatus, 0) != (pid_t)sqlpid);
			}
			/* Child Process */
			if(sqlpid == 0) {
				if((backupfile = fopen(MYSQLBACKUP, "rwb")) == (FILE *)NULL) {
					log(L_ERR, "Acct: (Child) Couldn't open file %s", MYSQLBACKUP);
					exit(1);
				}

				/* Lock the mysql backup file, prefer lockf() over flock(). */
#if defined(F_LOCK) && !defined( BSD)
				(void)lockf((int)backupfile, (int)F_LOCK, (off_t)SQL_LOCK_LEN);
#else
				(void)flock(backupfile, SQL_LOCK_EX);
#endif  

				log(L_INFO, "Acct:  Clearing out sql backup file - %s", MYSQLBACKUP);

				while(!feof(backupfile)) {
					if(fread(&backuprecord, sizeof(SQLREC), 1, backupfile) == 1) {

						/* pass our filled structure to the
					 	   function that will write to the database */
						ret = sql_save_acct(socket, &backuprecord);

					}

				}
				unlink((const char *)MYSQLBACKUP);
				exit(0);
			}
		}
	}
	ret = sql_save_acct(socket, &sqlrecord);

	return ret;
}


/*
 *	rad_accounting: call both the old and new style accounting functions.
 */
int rad_accounting(AUTH_REQ *authreq, int activefd, SQLSOCK *socket)
{
	int reply = 0;
	int auth;
	char pw_digest[16];

	/*
	 *	See if we know this client, then check the
	 *	request authenticator.
	 */
	auth = calc_acctdigest(pw_digest, authreq);

	if (auth < 0) {
		authfree(authreq);
		return -1;
	}

	if (log_stripped_names) {
		/*
		 *	rad_accounting_new strips authreq for us. If
		 *	we run it first, the stripped info will also
		 *	get into the "details" file.
		 */
		if (rad_accounting_new(authreq, doradwtmp) == 0)
			reply = 1;
		if (rad_accounting_orig(authreq, auth, NULL, socket) == 0)
			reply = 1;
	} else {
		/*
		 *	First log into the details file, before the
		 *	username gets stripped by rad_accounting_new.
		 */
		if (rad_accounting_orig(authreq, auth, NULL, socket) == 0)
			reply = 1;
		if (rad_accounting_new(authreq, doradwtmp) == 0)
			reply = 1;
	}

	if (reply) {
		/*
		 *	Now send back an ACK to the NAS.
		 */
		rad_send_reply(PW_ACCOUNTING_RESPONSE,
			authreq, NULL, NULL, activefd);
	}

	authfree(authreq);

	return reply ? 0 : -1;
}

