#!/usr/bin/perl -Tw
#####################################################################################
#
# This program is a complete rewrite of the original radius.cgi written by
# Mike Machado for distribution with ICRADIUS.  Please send bugs to
# drew@innercite.com.
#
# Currently there are three ways to authenticate users.
# 1. direct query to radius to authenticate user and check for Radius-Operator
#	attribute in radcheck
# 2. like 1 but does a query to a radius daemon instead
# 3. authenticates off IC-ISP, our ISP billing and management package.
# Both types of radius queries (1 and 2) support encrypted passwords.
# Also, the 'use Authen::Radius' line needs to be uncommented if you are using
# method 2. (and dont forget to install it :)).
#
# There is a known bug in this program that I didnt have time to fix before this
# release.  I wrote some custom SQL functions in the module in order to make life
# easier.  There is one more step missing though, and that is the 'escaping' of
# values passed to these functions.  To make a long story short, if you have the
# special characters '&', '|' or '!' as the first letter of a attribute or value,
# those characters will not be inserted correctly, and the SQL may not behave as you
# would expect it to.  Anywhere that will be fixed soon.
#
# This script uses the new IC::Radius module which provides an API to communicate
# with the ICRADIUS tables without actually having to know any of the table names
# or columns. You can get a copy of IC::Radius from ftp.innercite.com/pub/perl
#
# Change the $radius-init() line below so that the first two parameters
# are the username and password to access your radius database
#
#					Author		: Drew Wilder-Goodwin
#							  <drew@innercite.com>
#					Release Date	: 2001-06-14
#
#	This program is released under the same terms as the ICRADIUS pacakage
#
####################################################################################

use strict;

use IC::Radius;
my $radius = new IC::Radius;
$radius->init('radius', 'radiuspass', 'localhost');

use CGI;
my $query = new CGI;
my $cgi = $query->url;
my $action = $query->param('action') || 'main';

use Date::Calc;
use MD5;
use Fcntl ':flock';

my $logfile		= "/var/log/radius.log";				# location of radius log
my $sessions_file	= "/usr/local/apache/cgi-bin/icradiusweb.sessions";	# location of session file
my $session_timeout	= 7200;							# session timeout (in seconds)
my $authentication_type	= 1;							# 1 = local, 2 = radius daemon, 3 = IC-ISP
#	use Authen::Radius;							# uncommect line if above is set to 2
#	use lib qw(/usr/local/custdb2/lib);					# uncomment line if above is set to 3
#	use Custdb::Admin;							# uncomment line if above is set to 3
#	use Custdb::DB;								# uncomment line if above is set to 3
my $cookie_name		= 'icradiusweb';					# name of session cookie
my $cookie_domain	= '.mydomain.com';					# cookie domain
my $tmp_sessions_file	= "/tmp/icradiusweb.sessions.$$";			# location of temporary sessions file
my $radhost		= 'radius.mydomain.com';				# radius host (if $authentication_type is 2)
my $radsecret		= '';							# radius secret (if $authentication_type is 2)
my $highlight		= 'lightblue';						# highlight color of tables

my $OK = 1;

my $actions = {
	main			=> 'main',

	user_management		=> 'user_management',
		add_user	=> 'add_user',
		delete_user	=> 'delete_user',
		edit_user	=> 'edit_user',
		user_summary	=> 'user_summary',

		remove_user_attributes	=> 'remove_user_attributes',
		remove_user_groups	=> 'remove_user_groups',
		add_user_attribute	=> 'edit_user',
		add_user_group		=> 'edit_user',
		edit_user_attribute	=> 'edit_user',

	group_management	=> 'group_management',
		add_group	=> 'add_group',
		delete_group	=> 'delete_group',
		edit_group	=> 'edit_group',
		group_summary	=> 'group_summary',

		remove_group_attributes	=> 'remove_group_attributes',
		add_group_attribute	=> 'edit_group',
		edit_group_attribute	=> 'edit_group',

	realm_management	=> 'realm_management',
		add_realm	=> 'add_realm',
		delete_realm	=> 'delete_realm',
		edit_realm	=> 'edit_realm',

	nas_management		=> 'nas_management',
		add_nas		=> 'add_nas',
		delete_nas	=> 'delete_nas',
		edit_nas	=> 'edit_nas',

	usage_report		=> 'usage_report',

	login_report		=> 'login_report',
		remove_sessions	=> 'login_report',
		session_detail	=> 'session_detail',

	summary_report		=> 'summary_report',

	radius_log		=> 'radius_log',
		scrolling_log	=> 'scrolling_log',
};

if (&authenticate) {
	&print_header unless $action eq 'scrolling_log';
	no strict 'refs'; &{$actions->{$action}}; use strict 'refs';
	&print_footer unless $action eq 'scrolling_log';
}

sub main {
	print $action;
}

#################################################
#		User management			#
#################################################

sub user_management {
	&edit_user;
}

sub add_user {
	my $username = $query->param('username') || '';
	my $password = $query->param('password') || '';
	my $encrypted = $query->param('encrypted') ? 'CHECKED' : '';

	ADD_USER_EXEC: {
		if ($username) {
			if (&user_exists($username)) {
				$radius->set_error('ITEM_EXISTS');
				safe_error('User Already Exists');
				&edit_user;
				return;
			}
			unless ($password) {
				$radius->set_error('MISSING_REQUIRED');
				safe_error('Blank password field');
				last ADD_USER_EXEC;
			}
			$encrypted = $encrypted eq 'CHECKED' ? 1 : 0;
			$radius->insert_user($username, $password, $encrypted);
			&edit_user;
			return;
		}
	};

	&print_user_options;

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">Username</td></tr>
			<tr><td align="center"><input type="text" name="username" value="$username"></td></tr>
			<tr><td><br></td></tr>
			<tr><td align="center">Password</td></tr>
			<tr><td align="center"><input type="password" name="password" value="$password"></td></tr>
			<tr><td><br></td></tr>
			<tr><td align="center"><input type="checkbox" name="encrypted" value="1" $encrypted>Encrypted Password?</td></tr>
			<tr><td><br></td></tr>
			<tr><td width="100%" align="center"><input type="submit" value="Add User"></td></tr>
		</table>
		<input type="hidden" name="action" value="add_user">
		</form>
	~;
}

sub delete_user {
	my $username = $query->param('username') || '';
	my $confirmation = $query->param('confirm') || '';

	DELETE_USER_EXEC: {
		if ($username) {
			unless (&user_exists($username)) {
				$radius->set_error('ITEM_NOT_FOUND');
				safe_error('User Not Found');
				last DELETE_USER_EXEC;
			}
			unless ($confirmation eq 'yes') {
				print_user_options($username);
				print qq~
					<form action="$cgi" method="post">
					<table width="100%" cellspacing="2" cellpadding="2" border="0">
						<tr><td align="center">Are you sure you want to delete user '$username'</td></tr>
						<tr><td><br></td></tr>
						<tr><td align="center"><input type="submit" value="Confirm Deletion">
					</table>
					<input type="hidden" name="action" value="delete_user">
					<input type="hidden" name="confirm" value="yes">
					<input type="hidden" name="username" value="$username">
					</form>
				~;
				return;
			}
			$radius->delete_user($username);
			&print_user_options;
			print qq!<center>User '$username' has been deleted</center>\n!;
			return;
		}
	}

	print_user_options($username);

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">Username</td></tr>
			<tr><td align="center"><input type="text" name="username" value="$username"></td></tr>
			<tr><td><br></td></tr>
			<tr><td width="100%" align="center"><input type="submit" value="Delete User"></td></tr>
		</table>
		<input type="hidden" name="action" value="delete_user">
		</form>
	~;
}

sub edit_user {
	my $username = $query->param('username') || '';

	EDIT_USER_EXEC: {
		if ($username) {
			last EDIT_USER_EXEC if (&user_exists($username));

			my $users = user_search($username);
			&print_user_options;
			print qq!<center>\n!;
			print qq!No match for user '$username'!;
			if (scalar(@$users)) {
				print qq!, however, I found the following similar users:<br><br>\n!;
				foreach my $user (@$users) {
					print qq!<a href="$cgi?action=edit_user&username=$user">$user</a><br>\n!;
				}
			} else {
				print qq!.<br><br>\n!;
			}
			print qq!</center>\n!;
			return;
		} else {
			&print_user_options;
			print qq~
				<form action="$cgi" method="post">
				<table width="100%" cellspacing="2" cellpadding="0" border="0">
					<tr><td align="center">Username</td></tr>
					<tr><td align="center"><input type="text" name="username"></td></tr>
					<tr><td><br></td></tr>
					<tr><td width="100%" align="center"><input type="submit" value="Edit User"></td></tr>
				</table>
				<input type="hidden" name="action" value="edit_user">
				</form>
			~;
			return;
		}
	}

	print_user_options($username, '<hr>');

	&add_user_attribute if ($action eq 'add_user_attribute');
	&add_user_group if ($action eq 'add_user_group');
	&edit_user_attribute if ($action eq 'edit_user_attribute');

	print qq~
		<table width="100%" cellspacing="2" cellpadding="4" border="0" cols="6">
		<tr>
			<td colspan="3">username: <b>$username</b></td>
			<td align="center"><a href="#check_items">check items</a></td>
			<td align="center"><a href="#reply_items">reply items</a></td>
			<td align="center"><a href="#user_groups">user groups</a></td>
		</tr>
		</table>

		<center>
		<table width="90%" border="0">
		<tr><td>
	~;

	my $groups = $radius->fetch_user_groups($username);
	my $reply_items = $radius->fetch_user_reply_items($username);
	my $check_items = $radius->fetch_user_check_items($username);

	###################
	### CHECK ITEMS ###
	###################
	print qq~
		<a name="check_items">
		<form action="$cgi" method="post"><table cellspacing="2" cellpadding="4" border="1" width="100%" cols="4">
		<caption>check items</caption>
		<tr>
			<th bgcolor="$highlight">attribute</th>
			<th bgcolor="$highlight">value</th>
			<th bgcolor="$highlight">remove</th>
			<th bgcolor="$highlight">group</th>
		</tr>
	~;
	foreach my $attribute (sort keys %$check_items) {
		$check_items->{$attribute} = [$check_items->{$attribute}] unless ref $check_items->{$attribute};
		foreach my $value (@{$check_items->{$attribute}}) {
			print qq~
				<tr>
					<td align="center">
						<a href="$cgi?action=edit_user_attribute&username=$username&attribute=$attribute&ovalue=$value&type=check">$attribute</a>
					</td>
					<td align="center">$value</td>
					<td align="center"><input type="checkbox" name="attributes" value="$attribute:$value"></td>
					<td align="center">-</td>
				</tr>
			~;
		}
	}
	foreach my $group (@$groups) {
		my $check_items = $radius->fetch_group_check_items($group);
		foreach my $attribute (sort keys %$check_items) {
			$check_items->{$attribute} = [$check_items->{$attribute}] unless ref $check_items->{$attribute};
			foreach my $value (@{$check_items->{$attribute}}) {
				print qq~
					<tr>
						<td align="center">$attribute</td>
						<td align="center">$value</td>
						<td align="center">-</td>
						<td align="center"><a href="$cgi?action=edit_group&groupname=$group">$group</a></td>
					</tr>
				~;
			}
		}
	}
	print qq~
		</table>
		<table cellspacing="2" cellpadding="2" border="0" width="100%" cols="2">
		<tr>
			<input type="hidden" name="action" value="remove_user_attributes">
			<input type="hidden" name="username" value="$username">
			<input type="hidden" name="type" value="check">
			<td align="left"><input type="submit" value="Remove selected items"></td>
			</form>
			<td align="right">
				<form action="$cgi" method="post">
				<input type="submit" value=" Add a check item ">
				<input type="hidden" name="action" value="add_user_attribute">
				<input type="hidden" name="type" value="check">
				<input type="hidden" name="username" value="$username">
				</form>
			</td>
		</tr>
		</table>
		</a>
		<br>
	~;

	###################
	### REPLY ITEMS ###
	###################
	print qq~
		<a name="reply_items">
		<form action="$cgi" method="post"><table cellspacing="2" cellpadding="4" border="1" width="100%" cols="4">
		<caption>reply items</caption>
		<tr>
			<th bgcolor="$highlight">attribute</th>
			<th bgcolor="$highlight">value</th>
			<th bgcolor="$highlight">remove</th>
			<th bgcolor="$highlight">group</th>
		</tr>
	~;
	foreach my $attribute (sort keys %$reply_items) {
		$reply_items->{$attribute} = [$reply_items->{$attribute}] unless ref $reply_items->{$attribute};
		foreach my $value (@{$reply_items->{$attribute}}) {
			print qq~
				<tr>
					<td align="center">
						<a href="$cgi?action=edit_user_attribute&username=$username&attribute=$attribute&ovalue=$value&type=reply">$attribute</a>
					</td>
					<td align="center">$value</td>
					<td align="center"><input type="checkbox" name="attributes" value="$attribute:$value"></td>
					<td align="center">-</td>
				</tr>
			~;
		}
	}
	foreach my $group (@$groups) {
		my $reply_items = $radius->fetch_group_reply_items($group);
		foreach my $attribute (sort keys %$reply_items) {
			$reply_items->{$attribute} = [$reply_items->{$attribute}] unless ref $reply_items->{$attribute};
			foreach my $value (@{$reply_items->{$attribute}}) {
				print qq~
					<tr>
						<td align="center">$attribute</td>
						<td align="center">$value</td>
						<td align="center">-</td>
						<td align="center"><a href="$cgi?action=edit_group&groupname=$group">$group</a></td>
					</tr>
				~;
			}
		}
	}
	print qq~
		</table>
		<table cellspacing="2" cellpadding="2" border="0" width="100%" cols="2">
		<tr>
			<input type="hidden" name="action" value="remove_user_attributes">
			<input type="hidden" name="username" value="$username">
			<input type="hidden" name="type" value="reply">
			<td align="left"><input type="submit" value="Remove selected items"></td>
			</form>
			<td align="right">
				<form action="$cgi" method="post">
				<input type="submit" value=" Add a reply item ">
				<input type="hidden" name="action" value="add_user_attribute">
				<input type="hidden" name="type" value="reply">
				<input type="hidden" name="username" value="$username">
				</form>
			</td>
		</tr>
		</table>
		</a>
		<br>
	~;

	###################
	##### GROUPS ######
	###################
	print qq~
		<a name="user_groups">
		<form action="$cgi" method="post"><table cellspacing="2" cellpadding="4" border="1" width="100%" cols="2">
		<caption>user groups</caption>
		<tr>
			<th bgcolor="$highlight">group</th>
			<th bgcolor="$highlight">remove</th>
		</tr>
	~;
	foreach my $group (@$groups) {
		print qq~
			<tr>
				<td align="center"><a href="$cgi?action=edit_group&groupname=$group">$group</a></td>
				<td align="center"><input type="checkbox" name="groups" value="$group"></td>
			</tr>
		~;
	}
	print qq~
		</table>
		<table cellspacing="2" cellpadding="2" border="0" width="100%" cols="2">
		<tr>
			<input type="hidden" name="action" value="remove_user_groups">
			<input type="hidden" name="username" value="$username">
			<td align="left"><input type="submit" value="Remove selected groups"></td>
			</form>
			<td align="right">
				<form action="$cgi" method="post">
				<input type="submit" value=" Add to group(s) ">
				<input type="hidden" name="action" value="add_user_group">
				<input type="hidden" name="username" value="$username">
				</form>
			</td>
		</tr>
		</table>
		</a>
		<br>
	~;



	print qq~
		</td></tr>
		</table>
		</center>
	~;
}

sub user_summary {
	&print_user_options;

	my $users = $radius->fetch_user_list;

	print qq~
		<center>
		<table cellspacing="2" cellpadding="2" border="0">
			<tr><td bgcolor="$highlight">Username</td><td bgcolor="$highlight" colspan="2">Manage</td></tr>
	~;
	foreach my $user (@$users) {
		print qq~
			<tr bgcolor="white">
				<td>$user</td>
				<td><a href="$cgi?action=edit_user&username=$user">edit</a></td>
				<td><a href="$cgi?action=delete_user&username=$user">delete</a></td>
			</tr>
		~;
	}
	print qq~
		</table>
		</center>
	~;
}

sub user_search {
	my $username = shift;

	my $users = {};
	my $users_list = [];

	my $sql = "SELECT DISTINCT UserName FROM radcheck WHERE UserName like '%$username%'";
	$radius->print_debug($sql, 'SQL');
	my $sth = $radius->{dbh}->prepare($sql);
	$sth->execute || &terminal_error;
	while (my ($user) = $sth->fetchrow_array) {
		$users->{$user} = 1;
	}
	$sth->finish;

	$sql = "SELECT DISTINCT UserName FROM radreply WHERE UserName like '%$username%'";
	$radius->print_debug($sql, 'SQL');
	$sth = $radius->{dbh}->prepare($sql);
	$sth->execute || &terminal_error;
	while (my ($user) = $sth->fetchrow_array) {
		$users->{$user} = 1;
	}
	$sth->finish;

	foreach my $user (sort keys %$users) {
		push @$users_list, $user;
	}

	return $users_list;
}

sub user_exists {
	my $username = shift;
	return 1 if (	$radius->select_count('radreply', {UserName => $username}) or
			$radius->select_count('radcheck', {UserName => $username}) or
			$radius->select_count('usergroup', {UserName => $username}));
	return 0;
}

sub remove_user_attributes {
	my $username = $query->param('username');
	my @attributes = $query->param('attributes');
	my $type = $query->param('type') || '';

	unless ($username) {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Missing username');
		return;
	}
	unless ($type eq 'check' or $type eq 'reply') {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid type');
		&edit_user;
		return;
	}

	foreach (@attributes) {
		my ($attribute, $value) = /^(.*?):(.*)$/;
		$radius->delete_user_attribute($username, $type, $attribute, $value);
	}

	unless (user_exists($username)) {
		&print_user_options;
		print "<center>All attributes/groups for $username have been deleted.</center>\n";
		return;
	}

	&edit_user;
}

sub remove_user_groups {
	my $username = $query->param('username');
	my @groups = $query->param('groups');

	unless ($username) {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Missing username');
		return;
	}

	foreach my $group (@groups) {
		$radius->delete_user_group($username, $group);
	}

	unless (user_exists($username)) {
		&print_user_options;
		print "<center>All attributes/groups for $username have been deleted.</center>\n";
		return;
	}

	&edit_user;
}

sub add_user_attribute {
	my $username = $query->param('username');
	my $type = $query->param('type') || '';
	my $attribute = $query->param('attribute') || '';
	my $value = $query->param('value');
	my $save = $query->param('save') || '';

	unless ($type eq 'check' or $type eq 'reply') {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid type');
		return;
	}

	if ($save eq 'yes') {
		$radius->insert_user_attribute($username, $type, $attribute, $value);
		return;
	}

	my $attributes = $radius->fetch_attribute_list;

	print qq~
		<center>
		<table width="90%" border="0" cellspacing="2" cellpadding="4" cols="2">
		<tr>
		<td align="center">
			<form action="$cgi" method="post" name="attribute_form">
			<select name="attribute" onChange="javascript: attribute_form.submit()">
	~;

	foreach my $attrib (@$attributes) {
		$attribute = $attrib->{Attribute} unless $attribute;
		print qq!<option value="$attrib->{Attribute}"!;
		print q! selected! if $attribute eq $attrib->{Attribute};
		print qq!>$attrib->{Attribute}\n!;
	}

	print qq~
			</select>
			<input type="hidden" name="action" value="add_user_attribute">
			<input type="hidden" name="username" value="$username">
			<input type="hidden" name="type" value="$type">
			<input type="submit" value="Go">
			</form>
		</td>
		<td align="center">
			<form action="$cgi" method="post" name="value_form">
	~;

	my $attrib = $radius->fetch_attribute_info($attribute);	

	unless (ref $attrib) {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid attribute');
		return;
	}

	$_ = $attrib->{Format};
	SWITCH: {
		if (/^string$/)	{ print qq!<input type="textbox" name="value">\n!; last SWITCH; };
		if (/^ipaddr$/)	{ print qq!<input type="textbox" name="value">\n!; last SWITCH; };
		if (/^date$/)	{ print qq!<input type="textbox" name="value">\n!; last SWITCH; };
		if (/^integer$/) {
			my $values = $radius->fetch_attribute_values($attrib->{Attribute});
			if ($values) {
				print qq!<select name="value">\n!;
				foreach my $value (@$values) {
					print qq!<option value="$value">$value\n!;
				}
				print qq!</select>\n!;
			} else {
				print qq!<input type="textbox" name="value">\n!;
			}
			last SWITCH;
		};
	}

	print qq~
		<input type="submit" value="Add">
		</td>
		<input type="hidden" name="action" value="add_user_attribute">
		<input type="hidden" name="username" value="$username">
		<input type="hidden" name="attribute" value="$attribute">
		<input type="hidden" name="type" value="$type">
		<input type="hidden" name="save" value="yes">
		</form></table>
		<hr>
		</center>
	~;
}

sub add_user_group {
	my $username = $query->param('username');
	my $submit = $query->param('submit') || '';

	if ($submit eq 'yes') {
		my @groups = $query->param('groups');
		foreach my $group (@groups) {
			$radius->insert_user_group($username, $group);
		}
		return;
	}

	my $groups = $radius->fetch_group_list;

	my $size = (scalar(@$groups) <= 5 ? scalar(@$groups) : 5);

	print qq~
		<center>
		<form action="$cgi" method="post">
		<select name="groups" size="$size" multiple>
	~;

	foreach my $group (@$groups) {
		print qq!<option value="$group">$group\n!;
	}

	print qq~
		</select>
		<br>
		<input type="submit" value="Add to user">
		<input type="hidden" name="action" value="add_user_group">
		<input type="hidden" name="username" value="$username">
		<input type="hidden" name="submit" value="yes">
		</form>
		<hr>
		</center>
	~;
}

sub edit_user_attribute {
	my $username = $query->param('username');
	my $type = $query->param('type') || '';
	my $attribute = $query->param('attribute') || '';
	my $ovalue = $query->param('ovalue');
	my $nvalue = $query->param('nvalue');
	my $save = $query->param('save');

	unless ($type eq 'check' or $type eq 'reply') {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid type');
		return;
	}

	$attribute = $radius->fetch_attribute_info($attribute);	

	unless (ref $attribute) {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid attribute');
		return;
	}

	if ($save eq 'yes') {
		$radius->update_user_attribute($username, $type, $attribute->{Attribute}, $ovalue, $nvalue);
		return;
	}

	print qq~
		<center>
		<script language="JavaScript">
		</script>
		<table width="90%" border="0" cellspacing="2" cellpadding="4" cols="2"><form action="$cgi" method="post">
		<tr>
		<td align="center">
			Attribute: $attribute->{Attribute}
		</td>
		<td>
	~;

	$_ = $attribute->{Format};
	SWITCH: {
		if (/^string$/)	{ print qq!<input type="textbox" name="nvalue" value="$ovalue">\n!; last SWITCH; };
		if (/^ipaddr$/)	{ print qq!<input type="textbox" name="nvalue" value="$ovalue">\n!; last SWITCH; };
		if (/^date$/)	{ print qq!<input type="textbox" name="nvalue" value="$ovalue">\n!; last SWITCH; };
		if (/^integer$/) {
			my $values = $radius->fetch_attribute_values($attribute->{Attribute});
			if ($values) {
				print qq!<select name="nvalue">\n!;
				foreach my $value (@$values) {
					print qq!<option value="$value"!;
					print q! selected! if ($value eq $ovalue);
					print qq!>$value\n!;
				}
				print qq!</select>\n!;
			} else {
				print qq!<input type="textbox" name="nvalue" value="$ovalue">\n!;
			}
			last SWITCH;
		};
	}

	print qq~
		<input type="submit" value="Save">
		</td>
		<input type="hidden" name="action" value="edit_user_attribute">
		<input type="hidden" name="username" value="$username">
		<input type="hidden" name="type" value="$type">
		<input type="hidden" name="attribute" value="$attribute->{Attribute}">
		<input type="hidden" name="ovalue" value="$ovalue">
		<input type="hidden" name="save" value="yes">
		</form></table>
		<hr>
		</center>
	~;
}

#################################################
#		Group management		#
#################################################

sub group_management {
	&edit_group;
}

sub add_group {
	my $groupname = $query->param('groupname') || '';

	ADD_GROUP_EXEC: {
		if ($groupname) {
			if (&group_exists($groupname)) {
				$radius->set_error('ITEM_EXISTS');
				safe_error('Group Already Exists');
			}
			&edit_group;
			return;
		}
	};

	&print_group_options;

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">Groupname</td></tr>
			<tr><td align="center"><input type="text" name="groupname" value="$groupname"></td></tr>
			<tr><td><br></td></tr>
			<tr><td width="100%" align="center"><input type="submit" value="Add Group"></td></tr>
		</table>
		<input type="hidden" name="action" value="add_group">
		</form>
	~;
}

sub delete_group {
	my $groupname = $query->param('groupname') || '';
	my $confirmation = $query->param('confirm') || '';

	DELETE_GROUP_EXEC: {
		if ($groupname) {
			unless (&group_exists($groupname)) {
				$radius->set_error('ITEM_NOT_FOUND');
				safe_error('Group Not Found');
				last DELETE_GROUP_EXEC;
			}
			unless ($confirmation eq 'yes') {
				print_group_options($groupname);
				print qq~
					<form action="$cgi" method="post">
					<table width="100%" cellspacing="2" cellpadding="2" border="0">
						<tr><td align="center">Are you sure you want to delete group '$groupname'</td></tr>
						<tr><td><br></td></tr>
						<tr><td align="center"><input type="submit" value="Confirm Deletion">
					</table>
					<input type="hidden" name="action" value="delete_group">
					<input type="hidden" name="confirm" value="yes">
					<input type="hidden" name="groupname" value="$groupname">
					</form>
				~;
				return;
			}
			$radius->delete_group($groupname);
			&print_group_options;
			print qq!<center>Group '$groupname' has been deleted</center>\n!;
			return;
		}
	}

	print_group_options($groupname);

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">Groupname</td></tr>
			<tr><td align="center"><input type="text" name="groupname" value="$groupname"></td></tr>
			<tr><td><br></td></tr>
			<tr><td width="100%" align="center"><input type="submit" value="Delete Group"></td></tr>
		</table>
		<input type="hidden" name="action" value="delete_group">
		</form>
	~;
}

sub edit_group {
	my $groupname = $query->param('groupname') || '';

	EDIT_GROUP_EXEC: {
		if ($groupname) {
			last EDIT_GROUP_EXEC if (&group_exists($groupname));
			last EDIT_GROUP_EXEC if ($action eq 'add_group' or $action eq 'add_group_attribute');

			my $groups = group_search($groupname);
			&print_group_options;
			print qq!<center>\n!;
			print qq!No match for group '$groupname'!;
			if (scalar(@$groups)) {
				print qq!, however, I found the following similar groups:<br><br>\n!;
				foreach my $group (@$groups) {
					print qq!<a href="$cgi?action=edit_group&groupname=$group">$group</a><br>\n!;
				}
			} else {
				print qq!.<br><br>\n!;
			}
			print qq!</center>\n!;
			return;
		} else {
			&print_group_options;
			print qq~
				<form action="$cgi" method="post">
				<table width="100%" cellspacing="2" cellpadding="0" border="0">
					<tr><td align="center">Groupname</td></tr>
					<tr><td align="center"><input type="text" name="groupname"></td></tr>
					<tr><td><br></td></tr>
					<tr><td width="100%" align="center"><input type="submit" value="Edit Group"></td></tr>
				</table>
				<input type="hidden" name="action" value="edit_group">
				</form>
			~;
			return;
		}
	}

	print_group_options($groupname, '<hr>');

	&add_group_attribute if ($action eq 'add_group_attribute');
	&edit_group_attribute if ($action eq 'edit_group_attribute');

	print qq~
		<table width="100%" cellspacing="2" cellpadding="4" border="0" cols="4">
		<tr>
			<td colspan="2">groupname: <b>$groupname</b></td>
			<td align="center"><a href="#check_items">check items</a></td>
			<td align="center"><a href="#reply_items">reply items</a></td>
		</tr>
		</table>

		<center>
		<table width="90%" border="0">
		<tr><td>
	~;

	my $reply_items = $radius->fetch_group_reply_items($groupname);
	my $check_items = $radius->fetch_group_check_items($groupname);

	###################
	### CHECK ITEMS ###
	###################
	print qq~
		<a name="check_items">
		<form action="$cgi" method="post"><table cellspacing="2" cellpadding="4" border="1" width="100%" cols="3">
		<caption>check items</caption>
		<tr>
			<th bgcolor="$highlight">attribute</th>
			<th bgcolor="$highlight">value</th>
			<th bgcolor="$highlight">remove</th>
		</tr>
	~;
	foreach my $attribute (sort keys %$check_items) {
		$check_items->{$attribute} = [$check_items->{$attribute}] unless ref $check_items->{$attribute};
		foreach my $value (@{$check_items->{$attribute}}) {
			print qq~
				<tr>
					<td align="center">
						<a href="$cgi?action=edit_group_attribute&groupname=$groupname&attribute=$attribute&ovalue=$value&type=check">$attribute</a>
					</td>
					<td align="center">$value</td>
					<td align="center"><input type="checkbox" name="attributes" value="$attribute:$value"></td>
				</tr>
			~;
		}
	}
	print qq~
		</table>
		<table cellspacing="2" cellpadding="2" border="0" width="100%" cols="2">
		<tr>
			<input type="hidden" name="action" value="remove_group_attributes">
			<input type="hidden" name="groupname" value="$groupname">
			<input type="hidden" name="type" value="check">
			<td align="left"><input type="submit" value="Remove selected items"></td>
			</form>
			<td align="right">
				<form action="$cgi" method="post">
				<input type="submit" value=" Add a check item ">
				<input type="hidden" name="action" value="add_group_attribute">
				<input type="hidden" name="type" value="check">
				<input type="hidden" name="groupname" value="$groupname">
				</form>
			</td>
		</tr>
		</table>
		</a>
		<br>
	~;

	###################
	### REPLY ITEMS ###
	###################
	print qq~
		<a name="reply_items">
		<form action="$cgi" method="post"><table cellspacing="2" cellpadding="4" border="1" width="100%" cols="3">
		<caption>reply items</caption>
		<tr>
			<th bgcolor="$highlight">attribute</th>
			<th bgcolor="$highlight">value</th>
			<th bgcolor="$highlight">remove</th>
		</tr>
	~;
	foreach my $attribute (sort keys %$reply_items) {
		$reply_items->{$attribute} = [$reply_items->{$attribute}] unless ref $reply_items->{$attribute};
		foreach my $value (@{$reply_items->{$attribute}}) {
			print qq~
				<tr>
					<td align="center">
						<a href="$cgi?action=edit_group_attribute&groupname=$groupname&attribute=$attribute&ovalue=$value&type=reply">$attribute</a>
					</td>
					<td align="center">$value</td>
					<td align="center"><input type="checkbox" name="attributes" value="$attribute:$value"></td>
				</tr>
			~;
		}
	}
	print qq~
		</table>
		<table cellspacing="2" cellpadding="2" border="0" width="100%" cols="2">
		<tr>
			<input type="hidden" name="action" value="remove_group_attributes">
			<input type="hidden" name="groupname" value="$groupname">
			<input type="hidden" name="type" value="reply">
			<td align="left"><input type="submit" value="Remove selected items"></td>
			</form>
			<td align="right">
				<form action="$cgi" method="post">
				<input type="submit" value=" Add a reply item ">
				<input type="hidden" name="action" value="add_group_attribute">
				<input type="hidden" name="type" value="reply">
				<input type="hidden" name="groupname" value="$groupname">
				</form>
			</td>
		</tr>
		</table>
		</a>
		<br>
	~;

	print qq~
		</td></tr>
		</table>
		</center>
	~;
}

sub group_summary {
	&print_group_options;

	my $groups = $radius->fetch_group_list;

	print qq~
		<center>
		<table cellspacing="2" cellpadding="2" border="0">
			<tr><td bgcolor="$highlight">Groupname</td><td bgcolor="$highlight" colspan="2">Manage</td></tr>
	~;
	foreach my $group (@$groups) {
		print qq~
			<tr bgcolor="white">
				<td>$group</td>
				<td><a href="$cgi?action=edit_group&groupname=$group">edit</a></td>
				<td><a href="$cgi?action=delete_group&groupname=$group">delete</a></td>
			</tr>
		~;
	}
	print qq~
		</table>
		</center>
	~;
}

sub group_search {
	my $groupname = shift;

	my $groups = {};
	my $groups_list = [];

	my $sql = "SELECT DISTINCT GroupName FROM radgroupcheck WHERE GroupName like '%$groupname%'";
	$radius->print_debug($sql, 'SQL');
	my $sth = $radius->{dbh}->prepare($sql);
	$sth->execute || &terminal_error;
	while (my ($group) = $sth->fetchrow_array) {
		$groups->{$group} = 1;
	}
	$sth->finish;

	$sql = "SELECT DISTINCT GroupName FROM radgroupreply WHERE GroupName like '%$groupname%'";
	$radius->print_debug($sql, 'SQL');
	$sth = $radius->{dbh}->prepare($sql);
	$sth->execute || &terminal_error;
	while (my ($group) = $sth->fetchrow_array) {
		$groups->{$group} = 1;
	}
	$sth->finish;

	foreach my $group (sort keys %$groups) {
		push @$groups_list, $group;
	}

	return $groups_list;
}

sub group_exists {
	my $groupname = shift;
	return 1 if (	$radius->select_count('radgroupreply', {GroupName => $groupname}) or
			$radius->select_count('radgroupcheck', {GroupName => $groupname}));
	return 0;
}

sub remove_group_attributes {
	my $groupname = $query->param('groupname');
	my @attributes = $query->param('attributes');
	my $type = $query->param('type') || '';

	unless ($groupname) {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Missing groupname');
		return;
	}
	unless ($type eq 'check' or $type eq 'reply') {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid type');
		&edit_group;
		return;
	}

	foreach (@attributes) {
		my ($attribute, $value) = /^(.*?):(.*)$/;
		$radius->delete_group_attribute($groupname, $type, $attribute, $value);
	}

	unless (group_exists($groupname)) {
		&print_group_options;
		print "<center>All attributes/groups for $groupname have been deleted.</center>\n";
		return;
	}

	&edit_group;
}

sub add_group_attribute {
	my $groupname = $query->param('groupname');
	my $type = $query->param('type') || '';
	my $attribute = $query->param('attribute') || '';
	my $value = $query->param('value');
	my $save = $query->param('save');

	unless ($type eq 'check' or $type eq 'reply') {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid type');
		return;
	}

	if ($save eq 'yes') {
		$radius->insert_group_attribute($groupname, $type, $attribute, $value);
		return;
	}

	my $attributes = $radius->fetch_attribute_list;

	print qq~
		<center>
		<script language="JavaScript">
		</script>
		<table width="90%" border="0" cellspacing="2" cellpadding="4" cols="2">
		<tr>
		<td align="center">
			<form action="$cgi" method="post" name="attribute_form">
			<select name="attribute" onChange="javascript: attribute_form.submit()">
	~;

	foreach my $attrib (@$attributes) {
		$attribute = $attrib->{Attribute} unless $attribute;
		print qq!<option value="$attrib->{Attribute}"!;
		print q! selected! if $attribute eq $attrib->{Attribute};
		print qq!>$attrib->{Attribute}\n!;
	}

	print qq~
			</select>
			<input type="hidden" name="action" value="add_group_attribute">
			<input type="hidden" name="groupname" value="$groupname">
			<input type="hidden" name="type" value="$type">
			<input type="submit" value="Go">
			</form>
		</td>
		<td align="center">
			<form action="$cgi" method="post" name="value_form">
	~;

	my $attrib = $radius->fetch_attribute_info($attribute);	

	unless (ref $attrib) {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid attribute');
		return;
	}

	$_ = $attrib->{Format};
	SWITCH: {
		if (/^string$/)	{ print qq!<input type="textbox" name="value">\n!; last SWITCH; };
		if (/^ipaddr$/)	{ print qq!<input type="textbox" name="value">\n!; last SWITCH; };
		if (/^date$/)	{ print qq!<input type="textbox" name="value">\n!; last SWITCH; };
		if (/^integer$/) {
			my $values = $radius->fetch_attribute_values($attrib->{Attribute});
			if ($values) {
				print qq!<select name="value">\n!;
				foreach my $value (@$values) {
					print qq!<option value="$value">$value\n!;
				}
				print qq!</select>\n!;
			} else {
				print qq!<input type="textbox" name="value">\n!;
			}
			last SWITCH;
		};
	}

	print qq~
		<input type="submit" value="Add">
		</td>
		<input type="hidden" name="action" value="add_group_attribute">
		<input type="hidden" name="groupname" value="$groupname">
		<input type="hidden" name="attribute" value="$attribute">
		<input type="hidden" name="type" value="$type">
		<input type="hidden" name="save" value="yes">
		</form></table>
		<hr>
		</center>
	~;
}

sub edit_group_attribute {
	my $groupname = $query->param('groupname');
	my $type = $query->param('type') || '';
	my $attribute = $query->param('attribute') || '';
	my $ovalue = $query->param('ovalue');
	my $nvalue = $query->param('nvalue');
	my $save = $query->param('save');

	unless ($type eq 'check' or $type eq 'reply') {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid type');
		return;
	}

	$attribute = $radius->fetch_attribute_info($attribute);	

	unless (ref $attribute) {
		$radius->set_error('MISSING_REQUIRED');
		safe_error('Recieved an invalid attribute');
		return;
	}

	if ($save eq 'yes') {
		$radius->update_group_attribute($groupname, $type, $attribute->{Attribute}, $ovalue, $nvalue);
		return;
	}

	print qq~
		<center>
		<script language="JavaScript">
		</script>
		<table width="90%" border="0" cellspacing="2" cellpadding="4" cols="2"><form action="$cgi" method="post">
		<tr>
		<td align="center">
			Attribute: $attribute->{Attribute}
		</td>
		<td>
	~;

	$_ = $attribute->{Format};
	SWITCH: {
		if (/^string$/)	{ print qq!<input type="textbox" name="nvalue" value="$ovalue">\n!; last SWITCH; };
		if (/^ipaddr$/)	{ print qq!<input type="textbox" name="nvalue" value="$ovalue">\n!; last SWITCH; };
		if (/^date$/)	{ print qq!<input type="textbox" name="nvalue" value="$ovalue">\n!; last SWITCH; };
		if (/^integer$/) {
			my $values = $radius->fetch_attribute_values($attribute->{Attribute});
			if ($values) {
				print qq!<select name="nvalue">\n!;
				foreach my $value (@$values) {
					print qq!<option value="$value"!;
					print q! selected! if ($value eq $ovalue);
					print qq!>$value\n!;
				}
				print qq!</select>\n!;
			} else {
				print qq!<input type="textbox" name="nvalue" value="$ovalue">\n!;
			}
			last SWITCH;
		};
	}

	print qq~
		<input type="submit" value="Save">
		</td>
		<input type="hidden" name="action" value="edit_group_attribute">
		<input type="hidden" name="groupname" value="$groupname">
		<input type="hidden" name="type" value="$type">
		<input type="hidden" name="attribute" value="$attribute->{Attribute}">
		<input type="hidden" name="ovalue" value="$ovalue">
		<input type="hidden" name="save" value="yes">
		</form></table>
		<hr>
		</center>
	~;
}

#################################################
#		Realm management		#
#################################################

sub realm_management {
	&print_realm_options;

	my $realms = $radius->fetch_realm_list;

	print qq~
		<center>
		<table width="90%" border="0">
		<tr><td>
		<form action="$cgi" method="post"><table cellspacing="2" cellpadding="4" border="1" cols="5">
			<tr>
				<th bgcolor="$highlight">realmname</th>
				<th bgcolor="$highlight">NAS</th>
				<th bgcolor="$highlight">auth port</th>
				<th bgcolor="$highlight">options</th>
				<th bgcolor="$highlight">remove</th>
			</tr>
	~;
	foreach my $realm (@$realms) {
		print qq~
			<tr bgcolor="white">
				<td align="center"><a href="$cgi?action=edit_realm&realmname=$realm->{realmname}">$realm->{realmname}</a></td>
				<td align="center">$realm->{nas}</td>
				<td align="center">$realm->{authport}</td>
				<td align="center">$realm->{options}</td>
				<td align="center"><input type="checkbox" name="realms" value="$realm->{realmname}"></td>
			</tr>
		~;
	}
	print qq~
		</table>
		<table cellspacing="2" cellpadding="2" border="0" width="100%" cols="2">
		<tr>
			<input type="hidden" name="action" value="delete_realm">
			<td align="left"><input type="submit" value="Remove selected realms"></td>
			</form>
			<td align="right">
				<form action="$cgi" method="post">
				<input type="submit" value=" Add a realm ">
				<input type="hidden" name="action" value="add_realm">
				</form>
			</td>
		</tr>
		</table>
		</td></tr>
		</table>
		</center>
	~;
}

sub add_realm {
	my $realmname = $query->param('realmname') || '';
	my $nas = $query->param('nas') || '';
	my $authport = $query->param('authport') || '';
	my @options = $query->param('options');

	ADD_REALM_EXEC: {
		if ($realmname) {
			if (&realm_exists($realmname)) {
				$radius->set_error('ITEM_EXISTS');
				safe_error('Realm Already Exists');
				&edit_realm;
				return;
			}
			unless ($nas) {
				$radius->set_error('MISSING_REQUIRED');
				safe_error('A required value was blank');
				last ADD_REALM_EXEC;
			}
			$radius->insert_realm($realmname, $nas, $authport, \@options);
			print qq!<center>Realm Added</center><br>\n!;
			&realm_management;
			return;
		}
	}

	&print_realm_options;

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">Realmname</td></tr>
			<tr><td align="center"><input type="text" name="realmname" value="$realmname"></td></tr>
			<tr><td><br></td></tr>
			<tr><td align="center">NAS</td></tr>
			<tr><td align="center"><select name="nas">
	~;

	print qq!<option value="LOCAL">LOCAL\n!;
	my $nas_list = $radius->fetch_nas_list;
	foreach my $nas_i (@$nas_list) {
		print qq!<option value="$nas_i->{nasname}"!;
		print q! selected! if ($nas eq $nas_i->{nasname});
		print qq!>$nas_i->{nasname}\n!;
	}

	print qq~
			</select></td></tr>
			<tr><td><br></td></tr>
			<tr><td align="center">Auth Port</td></tr>
			<tr><td align="center"><input type="text" name="authport" value="$authport"></td></tr>
			<tr><td><br></td></tr>
			<tr><td align="center">Options</td></tr>
			<tr><td align="center"><select name="options" multiple>
	~;

	my $selected_options = {};
	if (scalar(@options)) {
		foreach my $option (@options) {
			$selected_options->{$option} = 1;
		}
	}

	my $option_list = $radius->fetch_realm_options;
	foreach my $option (@$option_list) {
		print qq!<option value="$option"!;
		print q! selected! if (exists $selected_options->{$option});
		print qq!>$option\n!;
	}
	
	print qq~
			<tr><td><br></td></tr>
			<tr><td width="100%" align="center"><input type="submit" value="Add Realm"></td></tr>
		</table>
		<input type="hidden" name="action" value="add_realm">
		</form>
	~;
}

sub delete_realm {
	my $realmname = $query->param('realmname') || '';
	my $confirmation = $query->param('confirm') || '';
	my @realms = $query->param('realms');

	DELETE_MULTIPLE_REALMS: {
		if (@realms) {
			foreach my $realm (@realms) {
				next unless realm_exists($realm);
				$radius->delete_realm($realm);
			}
			&realm_management;
			return;
		}
	}

	DELETE_REALM_EXEC: {
		if ($realmname) {
			unless (&realm_exists($realmname)) {
				$radius->set_error('ITEM_NOT_FOUND');
				safe_error('Realm Not Found');
				last DELETE_REALM_EXEC;
			}
			unless ($confirmation eq 'yes') {
				print_realm_options($realmname);
				print qq~
					<form action="$cgi" method="post">
					<table width="100%" cellspacing="2" cellpadding="2" border="0">
						<tr><td align="center">Are you sure you want to delete the realm '$realmname'</td></tr>
						<tr><td><br></td></tr>
						<tr><td align="center"><input type="submit" value="Confirm Deletion">
					</table>
					<input type="hidden" name="action" value="delete_realm">
					<input type="hidden" name="confirm" value="yes">
					<input type="hidden" name="realmname" value="$realmname">
					</form>
				~;
				return;
			}
			$radius->delete_realm($realmname);
			&print_realm_options;
			print qq!<center>Realm '$realmname' has been deleted</center>\n!;
			return;
		}
	}

	print_realm_options($realmname);

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">Realmname</td></tr>
			<tr><td align="center"><input type="text" name="realmname" value="$realmname"></td></tr>
			<tr><td><br></td></tr>
			<tr><td width="100%" align="center"><input type="submit" value="Delete Realm"></td></tr>
		</table>
		<input type="hidden" name="action" value="delete_realm">
		</form>
	~;
}

sub edit_realm {
	my $realmname = $query->param('realmname') || '';
	my $save = $query->param('save') || '';

	EDIT_REALM_EXEC: {
		if ($realmname) {
			if ($save eq 'yes') {
				my $orealmname = $query->param('orealmname');
				if (($realmname ne $orealmname) && &realm_exists($realmname)) {
					$radius->set_error('EXISTS');
					safe_error('realm already exists');
					last EDIT_REALM_EXEC;
				}
				my $nas = $query->param('nas') || '';
				my $authport = $query->param('authport') || '';
				my @options = $query->param('options');
				unless ($nas) {
					$radius->set_error('MISSING_REQUIRED');
					safe_error('A required value was blank');
					last EDIT_REALM_EXEC;
				}
				$radius->update_realm($orealmname, $realmname, $nas, $authport, \@options);
				print qq!<center>Realm Saved...</center><br>\n!;
				last EDIT_REALM_EXEC;
			}

			last EDIT_REALM_EXEC if (&realm_exists($realmname));

			my $realms = realm_search($realmname);
			&print_realm_options;
			print qq!<center>\n!;
			print qq!No match for realm '$realmname'!;
			if (scalar(@$realms)) {
				print qq!, however, I found the following similar realms:<br><br>\n!;
				foreach my $realm (@$realms) {
					print qq!<a href="$cgi?action=edit_realm&realmname=$realm">$realm</a><br>\n!;
				}
			} else {
				print qq!.<br><br>\n!;
			}
			print qq!</center>\n!;
			return;
		} else {
			&print_realm_options;
			print qq~
				<form action="$cgi" method="post">
				<table width="100%" cellspacing="2" cellpadding="0" border="0">
					<tr><td align="center">Realmname</td></tr>
					<tr><td align="center"><input type="text" name="realmname"></td></tr>
					<tr><td><br></td></tr>
					<tr><td width="100%" align="center"><input type="submit" value="Edit Realm"></td></tr>
				</table>
				<input type="hidden" name="action" value="edit_realm">
				</form>
			~;
			return;
		}
	}

	print_realm_options($realmname, '<hr>');

	my $realm = $radius->fetch_realm_info($realmname);

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">Realmname</td></tr>
			<tr><td align="center"><input type="text" name="realmname" value="$realmname"></td></tr>
			<tr><td><br></td></tr>
			<tr><td align="center">NAS</td></tr>
			<tr><td align="center"><select name="nas">
	~;

	my $nas_list = $radius->fetch_nas_list;
	foreach my $nas_i (@$nas_list) {
		print qq!<option value="$nas_i->{nasname}"!;
		print q! selected! if ($realm->{nas} eq $nas_i->{nasname});
		print qq!>$nas_i->{nasname}\n!;
	}

	print qq~
			</select></td></tr>
			<tr><td><br></td></tr>
			<tr><td align="center">Auth Port</td></tr>
			<tr><td align="center"><input type="text" name="authport" value="$realm->{authport}"></td></tr>
			<tr><td><br></td></tr>
			<tr><td align="center">Options</td></tr>
			<tr><td align="center"><select name="options" multiple>
	~;

	my $selected_options = {};
	my @options = split /,/, $realm->{options};
	foreach my $option (@options) {
		$selected_options->{$option} = 1;
	}

	my $option_list = $radius->fetch_realm_options;
	foreach my $option (@$option_list) {
		print qq!<option value="$option"!;
		print q! selected! if (exists $selected_options->{$option});
		print qq!>$option\n!;
	}
	
	print qq~
			<tr><td><br></td></tr>
			<tr><td width="100%" align="center"><input type="submit" value="Save Realm"></td></tr>
		</table>
		<input type="hidden" name="action" value="edit_realm">
		<input type="hidden" name="save" value="yes">
		<input type="hidden" name="orealmname" value="$realmname">
		</form>
	~;

}

sub realm_search {
	my $realmname = shift;

	my $realms = [];

	my $sql = "SELECT realmname FROM realms WHERE realmname like '%$realmname%'";
	$radius->print_debug($sql, 'SQL');
	my $sth = $radius->{dbh}->prepare($sql);
	$sth->execute || &terminal_error;
	while (my ($realm) = $sth->fetchrow_array) {
		push @$realms, $realm;
	}
	$sth->finish;

	return $realms;
}

sub realm_exists {
	my $realmname = shift;
	return 1 if $radius->select_count('realms', {realmname => $realmname});
	return 0;
}

#################################################
#		Nas management			#
#################################################

sub nas_management {
	&print_nas_options;

	my $nas_list = $radius->fetch_nas_list;

	print qq~
		<center>
		<table width="90%" border="0">
		<tr><td>
		<form action="$cgi" method="post"><table cellspacing="2" cellpadding="4" border="1" cols="9">
			<tr>
				<th bgcolor="$highlight">nasname</th>
				<th bgcolor="$highlight">shortname</th>
				<th bgcolor="$highlight">ip address</th>
				<th bgcolor="$highlight">type</th>
				<th bgcolor="$highlight">ports</th>
				<th bgcolor="$highlight">secret</th>
				<th bgcolor="$highlight">community</th>
				<th bgcolor="$highlight">snmp</th>
				<th bgcolor="$highlight">remove</th>
			</tr>
	~;
	foreach my $nas (@$nas_list) {
		print qq~
			<tr bgcolor="white">
				<td align="center"><a href="$cgi?action=edit_nas&nasname=$nas->{nasname}">$nas->{nasname}</a></td>
				<td align="center">$nas->{shortname}</td>
				<td align="center">$nas->{ipaddr}</td>
				<td align="center">$nas->{type}</td>
				<td align="center">$nas->{ports}</td>
				<td align="center">$nas->{secret}</td>
				<td align="center">$nas->{community}</td>
				<td align="center">$nas->{snmp}</td>
				<td align="center"><input type="checkbox" name="nass" value="$nas->{nasname}"></td>
			</tr>
		~;
	}
	print qq~
		</table>
		<table cellspacing="2" cellpadding="2" border="0" width="100%" cols="2">
		<tr>
			<input type="hidden" name="action" value="delete_nas">
			<td align="left"><input type="submit" value="Remove selected nas'"></td>
			</form>
			<td align="right">
				<form action="$cgi" method="post">
				<input type="submit" value=" Add a nas ">
				<input type="hidden" name="action" value="add_nas">
				</form>
			</td>
		</tr>
		</table>
		</td></tr>
		</table>
		</center>
	~;
}

sub add_nas {
	my $nasname = $query->param('nasname') || '';
	my $shortname = $query->param('shortname') || '';
	my $ipaddr = $query->param('ipaddr') || '';
	my $type = $query->param('type') || '';
	my $ports = $query->param('ports') || '';
	my $secret = $query->param('secret') || '';
	my $community = $query->param('community') || '';
	my $snmp = $query->param('snmp') || '';

	ADD_NAS_EXEC: {
		if ($nasname) {
			if (&nas_exists($nasname)) {
				$radius->set_error('ITEM_EXISTS');
				safe_error('Nas Already Exists');
				&edit_nas;
				return;
			}
			unless ($shortname && $ipaddr && $secret) {
				$radius->set_error('MISSING_REQUIRED');
				safe_error('A required value was blank');
				last ADD_NAS_EXEC;
			}
			$radius->insert_nas($nasname, $shortname, $ipaddr, $type, $ports, $secret, $community, $snmp);
			print qq!<center>Nas Added</center><br>\n!;
			&nas_management;
			return;
		}
	}

	&print_nas_options;

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">NAS Name</td></tr>
			<tr><td align="center"><input type="text" name="nasname" value="$nasname"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">Short Name</td></tr>
			<tr><td align="center"><input type="text" name="shortname" value="$shortname"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">IP Address</td></tr>
			<tr><td align="center"><input type="text" name="ipaddr" value="$ipaddr"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">Type</td></tr>
			<tr><td align="center"><input type="text" name="type" value="$type"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">Ports</td></tr>
			<tr><td align="center"><input type="text" name="ports" value="$ports"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">Secret</td></tr>
			<tr><td align="center"><input type="text" name="secret" value="$secret"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">SNMP Community</td></tr>
			<tr><td align="center"><input type="text" name="community" value="$community"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">SNMP Port Limit Checking</td></tr>
			<tr><td align="center">
				<input type="radio" name="snmp" value="on">On
				<input type="radio" name="snmp" value="off" checked>Off
			</td></tr>
			<tr><td><br></td></tr>

			<tr><td width="100%" align="center"><input type="submit" value="Add Nas"></td></tr>
		</table>
		<input type="hidden" name="action" value="add_nas">
		</form>
	~;
}

sub delete_nas {
	my $nasname = $query->param('nasname') || '';
	my $confirmation = $query->param('confirm') || '';
	my @nas_list = $query->param('nass');

	DELETE_MULTIPLE_NASS: {
		if (@nas_list) {
			if ($confirmation eq 'yes') {
				foreach my $nas (@nas_list) {
					next unless nas_exists($nas);
					$radius->delete_nas($nas);
				}
				&nas_management;
				return;
			}
			print qq~
				<center>
				<form action="$cgi" method="post">
				<table width="100%" cellspacing="2" cellpadding="2" border="0">
					<tr><td align="center">Warning: This will also delete any realms which point to the specified nas'.</td></tr>
					<tr><td><br></td></tr>
					<tr><td align="center"><input type="submit" value="Confirm Deletion">
				</table>
				<input type="hidden" name="action" value="delete_nas">
				<input type="hidden" name="confirm" value="yes">
			~;

			foreach my $nas (@nas_list) {
				print qq!<input type="hidden" name="nass" value="$nas">\n!;
			}

			print qq~
				</center>
			~;
			return;
		}
	}

	DELETE_NAS_EXEC: {
		if ($nasname) {
			unless (&nas_exists($nasname)) {
				$radius->set_error('ITEM_NOT_FOUND');
				safe_error('Realm Not Found');
				last DELETE_NAS_EXEC;
			}
			unless ($confirmation eq 'yes') {
				print_nas_options($nasname);
				print qq~
					<form action="$cgi" method="post">
					<table width="100%" cellspacing="2" cellpadding="2" border="0">
						<tr><td align="center">Are you sure you want to delete the nas '$nasname'</td></tr>
						<tr><td><br></td></tr>
						<tr><td align="center"><input type="submit" value="Confirm Deletion">
					</table>
					<input type="hidden" name="action" value="delete_nas">
					<input type="hidden" name="confirm" value="yes">
					<input type="hidden" name="nasname" value="$nasname">
					</form>
				~;
				return;
			}
			$radius->delete_nas($nasname);
			&print_nas_options;
			print qq!<center>Nas '$nasname' has been deleted</center>\n!;
			return;
		}
	}

	print_nas_options($nasname);

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">Nasname</td></tr>
			<tr><td align="center"><input type="text" name="nasname" value="$nasname"></td></tr>
			<tr><td><br></td></tr>
			<tr><td width="100%" align="center"><input type="submit" value="Delete Nas"></td></tr>
		</table>
		<input type="hidden" name="action" value="delete_nas">
		</form>
	~;
}

sub edit_nas {
	my $nasname = $query->param('nasname') || '';
	my $save = $query->param('save') || '';

	EDIT_NAS_EXEC: {
		if ($nasname) {
			if ($save eq 'yes') {
				my $onasname = $query->param('onasname');
				if (($nasname ne $onasname) && &nas_exists($nasname)) {
					$radius->set_error('EXISTS');
					safe_error('nas already exists');
					last EDIT_NAS_EXEC;
				}
				my $shortname = $query->param('shortname') || '';
				my $ipaddr = $query->param('ipaddr') || '';
				my $type = $query->param('type') || '';
				my $ports = $query->param('ports') || '';
				my $secret = $query->param('secret') || '';
				my $community = $query->param('community') || '';
				my $snmp = $query->param('snmp') || '';
				unless ($shortname && $ipaddr && $secret) {
					$radius->set_error('MISSING_REQUIRED');
					safe_error('A required value was blank');
					last EDIT_NAS_EXEC;
				}
				$radius->update_nas($onasname, $nasname, $shortname, $ipaddr, $type, $ports, $secret, $community, $snmp);
				print qq!<center>NAS Saved...</center><br>\n!;
				last EDIT_NAS_EXEC;
			}

			last EDIT_NAS_EXEC if (&nas_exists($nasname));

			my $nas_list = nas_search($nasname);
			&print_nas_options;
			print qq!<center>\n!;
			print qq!No match for nas '$nasname'!;
			if (scalar(@$nas_list)) {
				print qq!, however, I found the following similar nas':<br><br>\n!;
				foreach my $nas (@$nas_list) {
					print qq!<a href="$cgi?action=edit_nas&nasname=$nas">$nas</a><br>\n!;
				}
			} else {
				print qq!.<br><br>\n!;
			}
			print qq!</center>\n!;
			return;
		} else {
			&print_nas_options;
			print qq~
				<form action="$cgi" method="post">
				<table width="100%" cellspacing="2" cellpadding="0" border="0">
					<tr><td align="center">NAS Name</td></tr>
					<tr><td align="center"><input type="text" name="nasname"></td></tr>
					<tr><td><br></td></tr>
					<tr><td width="100%" align="center"><input type="submit" value="Edit NAS"></td></tr>
				</table>
				<input type="hidden" name="action" value="edit_nas">
				</form>
			~;
			return;
		}
	}

	print_nas_options($nasname, '<hr>');

	my $nas = $radius->fetch_nas_info($nasname);

	print qq~
		<form action="$cgi" method="post">
		<table width="100%" cellspacing="2" cellpadding="0" border="0">
			<tr><td align="center">NAS Name</td></tr>
			<tr><td align="center"><input type="text" name="nasname" value="$nasname"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">Short Name</td></tr>
			<tr><td align="center"><input type="text" name="shortname" value="$nas->{shortname}"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">IP Address</td></tr>
			<tr><td align="center"><input type="text" name="ipaddr" value="$nas->{ipaddr}"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">Type</td></tr>
			<tr><td align="center"><input type="text" name="type" value="$nas->{type}"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">Ports</td></tr>
			<tr><td align="center"><input type="text" name="ports" value="$nas->{ports}"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">Secret</td></tr>
			<tr><td align="center"><input type="text" name="secret" value="$nas->{secret}"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">SNMP Community</td></tr>
			<tr><td align="center"><input type="text" name="community" value="$nas->{community}"></td></tr>
			<tr><td><br></td></tr>

			<tr><td align="center">SNMP Community</td></tr>
			<tr><td align="center">
	~;
	print q!<input type="radio" name="snmp" value="on"!;
	print q! checked! if ($nas->{snmp} eq 'on');
	print q!>On!;
	print q!<input type="radio" name="snmp" value="off"!;
	print q! checked! if ($nas->{snmp} eq 'off');
	print q!>Off!;
	print qq~
			</td></tr>
			<tr><td><br></td></tr>

			<tr><td width="100%" align="center"><input type="submit" value="Save Nas"></td></tr>
		</table>
		<input type="hidden" name="action" value="edit_nas">
		<input type="hidden" name="save" value="yes">
		<input type="hidden" name="onasname" value="$nasname">
		</form>
	~;
}

sub nas_search {
	my $nasname = shift;

	my $nas_list = [];

	my $sql = "SELECT nasname FROM nas WHERE nasname like '%$nasname%'";
	$radius->print_debug($sql, 'SQL');
	my $sth = $radius->{dbh}->prepare($sql);
	$sth->execute || &terminal_error;
	while (my ($nas) = $sth->fetchrow_array) {
		push @$nas_list, $nas;
	}
	$sth->finish;

	return $nas_list;
}

sub nas_exists {
	my $nasname = shift;
	return 1 if $radius->select_count('nas', {nasname => $nasname});
	return 0;
}

#################################################
#		Usage report			#
#################################################

sub usage_report {
	my $order_by_fields = {};
	$order_by_fields->{UserName} = 'UserName';
	$order_by_fields->{Hours} = 'sum';
	my $start_time = $query->param('start_time') || '';
	my $stop_time = $query->param('stop_time') || '';
	my $order_by = $query->param('order_by') || '';
	my $sort_type = $query->param('sort_type') || 'asc';
	my $submit = $query->param('submit') || '';
	my $table = $query->param('table') || 'radacct';

	my $check_asc;
	my $check_desc;

	if ($sort_type eq 'asc') {
		$check_asc = 'checked';
	} elsif ($sort_type eq 'desc') {
		$check_desc = 'checked';
	}
	
	my $check_ra;
	my $check_ras;

	if ($table eq 'radacct') {
		$check_ra = 'checked';
	} elsif ($table eq 'radacct_summary') {
		$check_ras = 'checked';
	}
	print qq~
		<script language=javascript>

		function fill_dates() {
  
			date = new Date();
			month = date.getMonth();
			month++;
			year = date.getFullYear();
			time = date.getTime();
			day = date.getDate();
			hour = date.getHours();
			minutes = date.getMinutes();
			seconds = date.getSeconds();
			time = year+"-"+month+"-"+day+" "+hour+":"+minutes+":"+seconds;
	
			document.usage_report.start_time.value = time;
			document.usage_report.stop_time.value = time;
			document.usage_report.username.focus();
		}

		</script>

		<center>
		<form action="$cgi" method="post" name="usage_report">
		<table cellspacing="2" cellpadding="2" border="0" cols="2">
		<tr><th bgcolor="$highlight" colspan="2">Usage Report</th></tr>
		<tr>
			<td align="center">
				Start Time:<br>
				<input type="text" name="start_time" value="$start_time" size="18">
			</td>
			<td align="center">
				Order By<br>
				<select name="order_by">
	~;

	foreach my $field (keys %$order_by_fields) {
		print qq!<option value="$order_by_fields->{$field}"!;
		print q! selected! if ($order_by_fields->{$field} eq $order_by);
		print qq!>$field\n!;
	}

	print qq~
				</select>
			</td>
		</tr>
		<tr>
			<td align="center">
				Stop Time:<br>
				<input type="text" name="stop_time" value="$stop_time" size="18">
			</td>
			<td align="center">
				<input type="radio" name="sort_type" value="asc" $check_asc>Asc
				<input type="radio" name="sort_type" value="desc" $check_desc>Desc
			</td>
		</tr>
		<tr>
			<td align="center" colspan="2">
				<input type="radio" name="table" value="radacct" $check_ra>radacct
				<input type="radio" name="table" value="radacct_summary" $check_ras>radacct_summary
			</td>
		</tr>
		<tr>
			<td align="center"><input type="submit" value="Execute Report"></td>
			<td align="center"><input type="button" value="Fill Current Dates" onClick="fill_dates()"></td>
		</tr>
		</table>
		<input type="hidden" name="action" value="usage_report">
		<input type="hidden" name="submit" value="yes">
		</form>
		</center>
	~;

	USAGE_REPORT_EXEC: {
		if ($submit eq 'yes') {
			my $where;
			my $sql;
			if ($table eq 'radacct') {
				$sql = 'SELECT UserName, SUM(AcctSessionTime) / 3600 AS sum FROM radacct';
				$where .= " AcctStartTime >= ".$radius->{dbh}->quote($start_time)." AND" if ($start_time);
				$where .= " AcctStopTime <= ".$radius->{dbh}->quote($stop_time)." AND AcctStopTime != 0" if ($stop_time);
				$where =~ s/AND$//;

			} elsif ($table eq 'radacct_summary') {
				$sql = 'SELECT UserName, SUM(TotalSessionTime) / 3600 AS sum FROM radacct_summary';
				$where .= " PeriodStart >= ".$radius->{dbh}->quote($start_time)." AND" if ($start_time);
				$where .= " PeriodStop <= ".$radius->{dbh}->quote($stop_time)." AND PeriodStop != 0" if ($stop_time);
				$where =~ s/AND$//;

			} else {
				return ($radius->set_error('MISSING_REQUIRED') && $radius->print_error('Invalid Table'));
			}

			$sql .= $where ? $where : '';
			$sql .= " GROUP BY UserName HAVING sum != 0 ORDER BY $order_by $sort_type";

			$radius->print_debug($sql, 'SQL');
			my $sth = $radius->{dbh}->prepare($sql);
			$sth->execute || &terminal_error;

			print qq~
				<center>
				<hr>
				<table cellspacing="2" cellpading="2" border="1">
				<tr>
					<th bgcolor="$highlight">Username</th>
					<th bgcolor="$highlight">Hours used</th>

				</tr>
			~;

			my $user_count = 0;
			my $hour_count = 0;
			while (my $row = $sth->fetchrow_hashref) {
				$user_count++;
				$hour_count += $row->{sum};

				print qq~
					<tr>
						<td align="center">$row->{UserName}</td>
						<td align="center">$row->{sum}</td>
					</tr>

				~;
			}

			$sth->finish;

			print qq~
				</table>
				<br>
				
				<table cellspacing="2" cellpading="2" border="1" cols="2">
				<caption>Totals</caption>
				<tr>
					<td align="right" bgcolor="$highlight">Users:</td>
					<td align="left">$user_count</td>

				</tr>
				<tr>
					<td align="right" bgcolor="$highlight">Hours:</td>
					<td align="left">$hour_count</td>

				</tr>
				</table>
				</center>
			~;
		}
	}

}

#################################################
#		Login report			#
#################################################

sub login_report {
	my $username		= $query->param('username')		|| '';
	my $start_time		= $query->param('start_time')		|| '';
	my $stop_time		= $query->param('stop_time')		|| '';
	my $framed_ipaddr	= $query->param('framed_ipaddr')	|| '';
	my $nas_ipaddr		= $query->param('nas_ipaddr')		|| '';
	my $nas_port		= $query->param('nas_port')		|| '';
	my $nas_port_type	= $query->param('nas_port_type')	|| '';
	my $terminate_cause	= $query->param('terminate_cause')	|| '';
	my $submit		= $query->param('submit')		|| '';

	$start_time = $stop_time = current_time() unless ($submit eq 'yes');

	print qq~
		<script language=javascript>

		function fill_dates() {
  
			date = new Date();
			month = date.getMonth();
			month++;
			year = date.getFullYear();
			time = date.getTime();
			day = date.getDate();
			hour = date.getHours();
			minutes = date.getMinutes();
			seconds = date.getSeconds();
			time = year+"-"+month+"-"+day+" "+hour+":"+minutes+":"+seconds;
	
			document.login_report.start_time.value = time;
			document.login_report.stop_time.value = time;
			document.login_report.username.focus();
		}

		function clear_form() {

			document.login_report.username.value = ''
			document.login_report.start_time.value = ''
			document.login_report.stop_time.value = ''
			document.login_report.nas_ipaddr.value = ''
			document.login_report.nas_port.value = ''
			document.login_report.nas_port_type.value = ''
			document.login_report.framed_ipaddr.value = ''
			document.login_report.terminate_cause.selectedIndex = 0;
			document.login_report.username.focus()
		}

		</script>

		<center>
		<form action="$cgi" method="post" name="login_report">
		<table cellspacing="2" cellpadding="2" border="0">
		<tr><th bgcolor="$highlight" colspan="4">Login Report</th></tr>
		<tr>
			<td align="right">Username:</td>
			<td align="left"><input type="text" name="username" value="$username" size="20"></td>
			<td align="right">NAS IP Address:</td>
			<td align="left"><input type="text" name="nas_ipaddr" value="$nas_ipaddr" size="20"></td>
		</tr>
		<tr>
			<td align="right">From Time:</td>
			<td align="left"><input type="text" name="start_time" value="$start_time" size="20"></td>
			<td align="right">NAS Port:</td>
			<td align="left"><input type="text" name="nas_port" value="$nas_port" size="20"></td>
		</tr>
		<tr>
			<td align="right">To Time:</td>
			<td align="left"><input type="text" name="stop_time" value="$stop_time" size="20"></td>
			<td align="right">NAS Port Type:</td>
			<td align="left"><input type="text" name="nas_port_type" value="$nas_port_type" size="20"></td>
		</tr>
		<tr>
			<td align="right">Framed IP Address:</td>
			<td align="left"><input type="text" name="framed_ipaddr" value="$framed_ipaddr" size="20"></td>
			<td align="right">Terminate Cause:</td>
			<td align="left">
				<select name="terminate_cause">
					<option value=''>
		~;
		my $terminate_causes = $radius->fetch_attribute_values('Acct-Terminate-Cause');
		foreach my $cause (@$terminate_causes) {
			print qq!<option value="$cause"!;
			print q! selected! if ($terminate_cause eq $cause);
			print qq!>$cause\n!;
		}
		print qq~
				</select>
			</td>
		</tr>
		<tr>
			<td align="center" colspan="2"><input type="button" value="Fill Current Dates" onClick="fill_dates()"></td>
			<td align="center"><input type="submit" value="Execute Report"></td>
			<td align="center"><input type="button" value="Clear Form" onClick="clear_form()"></td>
		</tr>
		</table>
		<input type="hidden" name="action" value="login_report">
		<input type="hidden" name="submit" value="yes">
		</form>
		</center>
	~;

	$radius->delete_acct_sessions([$query->param('radacctids')]) if $action eq 'remove_sessions';

	LOGIN_REPORT_EXEC: {
		if ($submit eq 'yes') {
			my $sql = q~
				SELECT	radacct.RadAcctId,
					radacct.UserName,
					radacct.FramedIPAddress,
					radacct.AcctStartTime,
					radacct.AcctStopTime,
					radacct.AcctSessionTime,
					radacct.RadacctID,
					radacct.nasipaddress,
					radacct.nasportid
				FROM	radius.radacct
				WHERE	
			~;

			if ($start_time && $stop_time) {
				$sql .= "radacct.AcctStopTime >= '$start_time' AND radacct.AcctStartTime <= '$stop_time'";
			} else {
				$sql .= 'radacct.AcctStopTime = 0';
			}

			$sql .= " AND radacct.UserName = '$username'" if $username;
			$sql .= " AND radacct.NASIPAddress = '$nas_ipaddr'" if $nas_ipaddr;
			$sql .= " AND radacct.NASPortId = '$nas_port'" if $nas_port;
			$sql .= " AND radacct.NASPortType = '$nas_port_type'" if $nas_port_type;
			$sql .= " AND radacct.FramedIPAddress = '$framed_ipaddr'" if $framed_ipaddr;
			$sql .= " AND radacct.AcctTerminateCause = '$terminate_cause'" if $terminate_cause;

			$sql .= " ORDER BY radacct.UserName";

			$radius->print_debug($sql, 'SQL');
			my $sth = $radius->{dbh}->prepare($sql);
			$sth->execute || &terminal_error;

			print qq~
				<center>
				<hr>
				<form action="$cgi" method="post">
				<table cellspacing="2" cellpading="2" border="1">
				<tr>
					<th bgcolor="$highlight">Username</th>
					<th bgcolor="$highlight">IP Address</th>
					<th bgcolor="$highlight">NAS IP/Port</th>
					<th bgcolor="$highlight">Start Time</th>
					<th bgcolor="$highlight">Stop Time</th>
					<th bgcolor="$highlight">Time Used</th>
					<th bgcolor="$highlight">Remove</th>
				</tr>
			~;

			my $session_count = 0;
			my $total_time = 0;
			while (my $row = $sth->fetchrow_hashref) {
				$session_count++;

				my $seconds = calc_duration($row->{AcctStartTime}, $row->{AcctStopTime});
				$total_time += $seconds;
				my $duration = format_seconds($seconds);

				print qq~
					<tr>
						<td align="center"><a href="$cgi?action=session_detail&radacctid=$row->{RadAcctId}">$row->{UserName}</a></td>
						<td align="center">$row->{FramedIPAddress}</td>
						<td align="center">$row->{nasipaddress}/S$row->{nasportid}</td>
						<td align="center">$row->{AcctStartTime}</td>
						<td align="center">$row->{AcctStopTime}</td>
						<td align="center">$duration</td>
						<td align="center"><input type="checkbox" name="radacctids" value="$row->{RadAcctId}"></td>
					</tr>
				~;
			}

			$sth->finish;

			$total_time = format_seconds($total_time);

			print qq~
				</table>
				<input type="hidden" name="action" value="remove_sessions">
				<input type="hidden" name="submit" value="yes">
				<input type="hidden" name="username" value="$username">
				<input type="hidden" name="start_time" value="$start_time">
				<input type="hidden" name="stop_time" value="$stop_time">
				<input type="hidden" name="framed_ipaddr" value="$framed_ipaddr">
				<input type="hidden" name="nas_ipaddr" value="$nas_ipaddr">
				<input type="hidden" name="nas_port" value="$nas_port">
				<input type="hidden" name="nas_port_type" value="$nas_port_type">
				<input type="hidden" name="terminate_cause" value="$terminate_cause">
				<input type="submit" value="Remove Selected Sessions">
				<input type="reset" value="Clear Selections">
				</form>
				<br>
				
				<table cellspacing="2" cellpading="2" border="1" cols="2">
				<caption>Totals</caption>
				<tr>
					<td align="right" bgcolor="$highlight">Sessions:</td>
					<td align="left">$session_count</td>

				</tr>
				<tr>
					<td align="right" bgcolor="$highlight">Total Time:</td>
					<td align="left">$total_time</td>

				</tr>
				</table>
				</center>
			~;
		}
	}
}

sub session_detail {
	my $radacctid = $query->param('radacctid') || '';

	return ($radius->set_error('MISSING_REQUIRED') && $radius->print_error('Missing RadAcctId')) unless $radacctid;

	my $session = $radius->fetch_session_info($radacctid);

	print qq~
		<center>
		<table cellspacing="4" cellpadding="2" border="1" cols="2">
		<tr><th bgcolor="$highlight">Attribute</th><th bgcolor="$highlight">Value</th></tr>
		<tr><td align="center">Radius-Id</td><td align="center">$session->{RadAcctId}</td></tr>
		<tr><td align="center">UserName</td><td align="center">$session->{UserName}</td></tr>
		<tr><td align="center">Realm</td><td align="center">$session->{Realm}</td></tr>
		<tr><td align="center">NAS-IP-Address</td><td align="center">$session->{NASIPAddress}</td></tr>
		<tr><td align="center">NAS-Port-Id</td><td align="center">$session->{NASPortId}</td></tr>
		<tr><td align="center">NAS-Port-Type</td><td align="center">$session->{NASPortType}</td></tr>
		<tr><td align="center">Acct-Start-Time</td><td align="center">$session->{AcctStartTime}</td></tr>
		<tr><td align="center">Acct-Stop-Time</td><td align="center">$session->{AcctStopTime}</td></tr>
		<tr><td align="center">Acct-Session-Time</td><td align="center">$session->{AcctSessionTime}</td></tr>
		<tr><td align="center">Acct-Authentic</td><td align="center">$session->{AcctAuthentic}</td></tr>
		<tr><td align="center">Connect-Info</td><td align="center">$session->{ConnectInfo}</td></tr>
		<tr><td align="center">Acct-Input-Octets</td><td align="center">$session->{AcctInputOctets}</td></tr>
		<tr><td align="center">Acct-Output-Octets</td><td align="center">$session->{AcctOutputOctets}</td></tr>
		<tr><td align="center">Called-Station-Id</td><td align="center">$session->{CalledStationId}</td></tr>
		<tr><td align="center">Calling-Station-Id</td><td align="center">$session->{CallingStationId}</td></tr>
		<tr><td align="center">Acct-Terminate-Cause</td><td align="center">$session->{AcctTerminateCause}</td></tr>
		<tr><td align="center">Service-Type</td><td align="center">$session->{ServiceType}</td></tr>
		<tr><td align="center">Framed-Protocol</td><td align="center">$session->{FramedProtocol}</td></tr>
		<tr><td align="center">Framed-IP-Address</td><td align="center">$session->{FramedIPAddress}</td></tr>
		<tr><td align="center">Acct-Start-Delay-Time</td><td align="center">$session->{AcctStartDelay}</td></tr>
		<tr><td align="center">Acct-Stop-Delay-Time</td><td align="center">$session->{AcctStopDelay}</td></tr>
		</table>
		</center>
	~;
}

#################################################
#		Summary report			#
#################################################

sub summary_report {
	my $username		= $query->param('username')		|| '';
	my $sessions		= $query->param('sessions')		|| '';
	my $start_time		= $query->param('start_time')		|| '';
	my $stop_time		= $query->param('stop_time')		|| '';
	my $total_session_time	= $query->param('total_session_time')	|| '';
	my $total_input_octets	= $query->param('total_input_octets')	|| '';
	my $total_output_octets	= $query->param('total_output_octets')	|| '';
	my $submit		= $query->param('submit')		|| '';

	$start_time = $stop_time = current_time() unless ($submit eq 'yes');

	print qq~
		<script language=javascript>

		function fill_dates() {
  
			date = new Date();
			month = date.getMonth();
			month++;
			year = date.getFullYear();
			time = date.getTime();
			day = date.getDate();
			hour = date.getHours();
			minutes = date.getMinutes();
			seconds = date.getSeconds();
			time = year+"-"+month+"-"+day+" "+hour+":"+minutes+":"+seconds;
	
			document.summary_report.start_time.value = time;
			document.summary_report.stop_time.value = time;
			document.summary_report.username.focus();
		}

		function clear_form() {

			document.summary_report.username.value = ''
			document.summary_report.sessions.value = ''
			document.summary_report.start_time.value = ''
			document.summary_report.stop_time.value = ''
			document.summary_report.total_session_time.value = ''
			document.summary_report.total_input_octets.value = ''
			document.summary_report.total_output_octets.value = ''
			document.summary_report.username.focus()
		}

		</script>

		<center>
		<form action="$cgi" method="post" name="summary_report">
		<table cellspacing="2" cellpadding="2" border="0">
		<tr><th bgcolor="$highlight" colspan="4">Summary Report</th></tr>
		<tr>
			<td align="right">Username:</td>
			<td align="left"><input type="text" name="username" value="$username" size="20"></td>
			<td align="right">Sessions:</td>
			<td align="left"><input type="text" name="sessions" value="$sessions" size="20"></td>
		</tr>
		<tr>
			<td align="right">From Time:</td>
			<td align="left"><input type="text" name="start_time" value="$start_time" size="20"></td>
			<td align="right">To Time:</td>
			<td align="left"><input type="text" name="stop_time" value="$stop_time" size="20"></td>
		</tr>
		<tr>
			<td align="center" colspan="2"><input type="button" value="Fill Current Dates" onClick="fill_dates()"></td>
			<td align="center"><input type="submit" value="Execute Report"></td>
			<td align="center"><input type="button" value="Clear Form" onClick="clear_form()"></td>
		</tr>
		</table>
		<input type="hidden" name="action" value="summary_report">
		<input type="hidden" name="submit" value="yes">
		</form>
		</center>
	~;

	SUMMARY_REPORT_EXEC: {
		if ($submit eq 'yes') {
			my $sql = qq~
				SELECT	radacct_summary.UserName,
					radacct_summary.Sessions,
					radacct_summary.PeriodStart,
					radacct_summary.PeriodStop,
					radacct_summary.TotalSessionTime,
					radacct_summary.TotalInputOctets,
					radacct_summary.TotalOutputOctets
				FROM	radius.radacct_summary
				WHERE	radacct_summary.PeriodStart >= '$start_time' AND
					radacct_summary.PeriodStop <= '$stop_time'
			~;

			$sql .= " AND radacct_summary.UserName = '$username'" if $username;
			$sql .= " AND radacct_summary.sessions = '$sessions'" if $sessions;

			$sql .= " ORDER BY radacct_summary.UserName";

			$radius->print_debug($sql, 'SQL');
			my $sth = $radius->{dbh}->prepare($sql);
			$sth->execute || &terminal_error;

			print qq~
				<center>
				<hr>
				<form action="$cgi" method="post">
				<table cellspacing="2" cellpading="2" border="1">
				<tr>
					<th bgcolor="$highlight">Username</th>
					<th bgcolor="$highlight">Sessions</th>
					<th bgcolor="$highlight">Start Time</th>
					<th bgcolor="$highlight">Stop Time</th>
					<th bgcolor="$highlight">Total Session Time</th>
					<th bgcolor="$highlight">Total Input Octets</th>
					<th bgcolor="$highlight">Total Output Octets</th>
				</tr>
			~;

			my $session_count = 0;
			my $total_time = 0;
			while (my $row = $sth->fetchrow_hashref) {
				$session_count++;

				$total_time += $row->{TotalSessionTime};
				$row->{TotalSessionTime} = format_seconds($row->{TotalSessionTime});

				print qq~
					<tr>
						<td align="center">$row->{UserName}</a></td>
						<td align="center">$row->{Sessions}</td>
						<td align="center">$row->{PeriodStart}</td>
						<td align="center">$row->{PeriodStop}</td>
						<td align="center">$row->{TotalSessionTime}</td>
						<td align="center">$row->{TotalInputOctets}</td>
						<td align="center">$row->{TotalOutputOctets}</td>
					</tr>
				~;
			}

			$sth->finish;

			$total_time = format_seconds($total_time);

			print qq~
				</table>
				<input type="hidden" name="action" value="remove_sessions">
				<input type="hidden" name="submit" value="yes">
				<input type="hidden" name="username" value="$username">
				<input type="hidden" name="start_time" value="$start_time">
				<input type="hidden" name="stop_time" value="$stop_time">
				<input type="hidden" name="sessions" value="$sessions">
				</form>
				<br>
				
				<table cellspacing="2" cellpading="2" border="1" cols="2">
				<caption>Totals</caption>
				<tr>
					<td align="right" bgcolor="$highlight">Sessions:</td>
					<td align="left">$session_count</td>

				</tr>
				<tr>
					<td align="right" bgcolor="$highlight">Total Time:</td>
					<td align="left">$total_time</td>

				</tr>
				</table>
				</center>
			~;
		}
	}
}

#################################################
#		Radius log			#
#################################################

sub radius_log {
	my $lines = $query->param('lines') || '';
	print qq~
		<script language="javascript">
			function logwin() {
				window.open("$cgi?action=scrolling_log", "logwin",
							"width=750,height=500,status=no,resizable=no,toolbar=no,menubar=no,scrollbars");
				}
		</script>
		<center>
		<form action="$cgi" method="post">
		<table cellspacing="2" cellpadding="2" cols="1">
		<tr><th bgcolor="$highlight">Number of lines:</th></tr>
		<tr><td align="center"><input type="text" name="lines" value="$lines"></td></tr>
		<tr><td align="center"><input type="Submit" value="Report"> <input type="button" value="Scrolling Log" onClick="logwin()"></td></tr>
		</table>
		<input type="hidden" name="action" value="radius_log">
		</form>
		</center>
	~;

	RADIUS_LOG_EXEC: {
		if ($lines =~ /^(\d+)$/) {
			$| = 1;
			delete $ENV{PATH};
			print qq~
				<hr>
				<center>
				<form action="$cgi" method="post">
				<table width="95%">
				<tr><td>
			~;
			my $output = `/usr/bin/tail -$1 $logfile`;
			print '<pre>'.($query->escapeHTML($output)).'</pre>';
			print qq~
				</td></tr>
				</table>
				<input type="hidden" name="action" value="radius_log">
				<input type="hidden" name="lines" value="$lines">
				<input type="submit" value="Refresh">
				</form>
				</center>
			~;
		}
	}
}

sub scrolling_log {
	print "Content-type: text/html\n\n" unless ($radius->{content_printed});

	$_ = $query->param('frame') || '';

	FRAME: {
		if (/^$/) {
			print qq~
				<html>
				<head>
					<title>ICRADIUS Web Management</title>
				</head>

				<frameset rows="10%,90%" frameborder="0" border="0" framespacing="2">
				<frame src="$cgi?action=scrolling_log&frame=toolbar" name="toolbar" scrolling="no">
				<frame src="$cgi?action=scrolling_log&frame=radiuslog" name="radiuslog">
				</frameset>

				</html>
			~;
			last FRAME;
		}

		if (/^toolbar$/) {
			print qq~
				<html>
				<head>
				</head>
				<body bgcolor="white">
					<center>
					<form action="$cgi" method="post">
						<input type="button" value="Toggle Scrolling" onClick="top.radiuslog.toggleScroll()">
					</form>
					</center>
				</body>
				</html>
			~;
			last FRAME;
		}

		if (/^radiuslog$/) {
			$| = 1;
			delete $ENV{PATH};

			print qq~
				<html>
				<script language="javascript">

					// toggle scrolling mode
					function toggleScroll() {
						if (scrolltimeron) {
							stopScroll();
						} else {
							scrollit(8);
						}
					}

					timerShow = null;
					var scrolltimer = null;
					var scrolltimeron = false;

					function stopScroll() {
						if (scrolltimeron) {
							clearTimeout(scrolltimer);
						}
						scrolltimeron = false;
					}

					function scrollit(speed) {
						s = speed;
						window.scrollBy(0,s);
						scrolltimer = setTimeout('scrollit(s)','50');
						scrolltimeron = true;
					}
					toggleScroll();

				</script>
				<body bgcolor="white" onScroll="stopScroll()">
			~;
			print qq!<pre>\n!;
			
			open LOG, "/usr/bin/tail -f $logfile|";

			while (<LOG>) {
				print $query->escapeHTML($_);
			}

			print qq~
				</pre>
				</body>
				</html>
			~;

			last FRAME;
		}
	}
}

#################################################
#               Date/Time Functions             #
#################################################

sub current_time {
	my @time = localtime(time());
	return sprintf("%d-%.2d-%.2d %.2d:%.2d:%.2d", ($time[5]+1900), ($time[4]+1), $time[3], $time[2], $time[1], $time[0]);
}

sub calc_duration {
	my ($days, $hours, $minutes, $seconds) = Date::Calc::Delta_DHMS(parse_datetime(shift), parse_datetime(shift));
	return (($days*86400)+($hours*3600)+($minutes*60)+($seconds)); 
}

sub format_seconds {
	my $seconds = shift;
	my $days = my $hours = my $minutes = 0;

	while ($seconds >= 86400) { $days++; $seconds -= 86400; }
	while ($seconds >= 3600) { $hours++; $seconds -= 3600; }
	while ($seconds >= 60) { $minutes++; $seconds -= 60; }

	return ($days ? "$days".'d' : '').($hours || $days ? "$hours".'h' : '').($minutes || $hours || $days ? "$minutes".'m' : '').("$seconds".'s');
}

sub parse_datetime {
	return (shift =~ /^(\d+)-(\d+)-(\d+) (\d+):(\d+):(\d+)$/);
}

#################################################
#               Misc Functions                  #
#################################################

sub print_user_options {
	my $username = shift || '';
	my $separator = shift || '<br>';
	$username = "&username=$username" if $username;
	print qq~
		<table cellspacing="2" cellpadding="0" width="100%" border="0">
		<td align="center" width="25%">&lt;<a href="$cgi?action=add_user">add user</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=delete_user$username">delete user</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=edit_user$username">edit user</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=user_summary">user summary</a>&gt;</td>
		</table>
		$separator
	~;
}

sub print_group_options {
	my $groupname = shift || '';
	my $separator = shift || '<br>';
	$groupname = "&groupname=$groupname" if $groupname;
	print qq~
		<table cellspacing="2" cellpadding="0" width="100%" border="0">
		<td align="center" width="25%">&lt;<a href="$cgi?action=add_group">add group</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=delete_group$groupname">delete group</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=edit_group$groupname">edit group</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=group_summary">group summary</a>&gt;</td>
		</table>
		$separator
	~;
}

sub print_realm_options {
	my $realmname = shift || '';
	my $separator = shift || '<br>';
	$realmname = "&realmname=$realmname" if $realmname;
	print qq~
		<table cellspacing="2" cellpadding="0" width="100%" border="0">
		<td align="center" width="25%">&lt;<a href="$cgi?action=add_realm">add realm</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=delete_realm$realmname">delete realm</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=edit_realm$realmname">edit realm</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=realm_management">realm summary</a>&gt;</td>
		</table>
		$separator
	~;
}

sub print_nas_options {
	my $nasname = shift || '';
	my $separator = shift || '<br>';
	$nasname = "&nasname=$nasname" if $nasname;
	print qq~
		<table cellspacing="2" cellpadding="0" width="100%" border="0">
		<td align="center" width="25%">&lt;<a href="$cgi?action=add_nas">add nas</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=delete_nas$nasname">delete nas</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=edit_nas$nasname">edit nas</a>&gt;</td>
		<td align="center" width="25%">&lt;<a href="$cgi?action=nas_management">nas summary</a>&gt;</td>
		</table>
		$separator
	~;
}

sub print_header {
	return if $radius->{header_printed};
	print "Content-type: text/html\n\n" unless $radius->{content_printed};
	print qq~
		<html>
		<head>
		<title>ICRADIUS Management</title>
		</head>
		<style type="text/css"><!--
			a:link {text-decoration: none}
			a:visited {text-decoration: none}
			a:active {text-decoration: none}
			body {font-family:Comic Sans Ms;}
			p,a,br,ul,ol,td,th,li,small,blockquote,h1,h2,h3 {font-family:Comic Sans Ms;}
		--></style>
		<body bgcolor="white">
		<center>
		<table border="0" cellspacing="4" cellpadding="0" width="850">
		<caption><-(--::<<< ICRADIUS Management Tool >>>::--)-></caption>
		<tr>
		<td width="15%" valign="top">
			<table border="1" cellspacing="2" cellpadding="4" width="100%">
			<tr>
			<th width="150" bgcolor="$highlight">Management</th>
			</tr>
			<tr>
			<td width="150" valign="top">
				<table cellspacing="2" cellpadding="0" border="0" width="100%">
				<tr><th>&middot;</th><td align="center"><a href="$cgi?action=user_management">user</a></td><th>&middot;</th></tr>
				<tr><th>&middot;</th><td align="center"><a href="$cgi?action=group_management">group</a></td><th>&middot;</th></tr>
				<tr><th>&middot;</th><td align="center"><a href="$cgi?action=realm_management">realm</a></td><th>&middot;</th></tr>
				<tr><th>&middot;</th><td align="center"><a href="$cgi?action=nas_management">nas</a></td><th>&middot;</th></tr>
				</table>
			</td>
			</tr>
			<tr>
			<th width="150" bgcolor="$highlight">Reports</th>
			</tr>
			<tr>
			<td width="150" valign="top">
				<table cellspacing="2" cellpadding="0" border="0" width="100%">
				<tr><th>&middot;</th><td align="center"><a href="$cgi?action=usage_report">usage</a></td><th>&middot;</th></tr>
				<tr><th>&middot;</th><td align="center"><a href="$cgi?action=login_report">login</a></td><th>&middot;</th></tr>
				<tr><th>&middot;</th><td align="center"><a href="$cgi?action=summary_report">summary</a></td><th>&middot;</th></tr>
				<tr><th>&middot;</th><td align="center"><a href="$cgi?action=radius_log">radius log</a></td><th>&middot;</th></tr>
				</table>
			</td>
			</tr>
			<tr><td width="150" bgcolor="$highlight"><br></td></tr>
			</table>
		</td>
		<td width="85%" valign="top">
			<table border="1" cellspacing="2" cellpadding="4" width="100%">
			<td width="100%">

	~;
	$radius->{header_printed} = 1;
}

sub print_footer {
	return if $radius->{footer_printed};
	print qq~
			<br>
			</td>
			</table>
		</td>
		</tr>
		</table>
		</center>
		</body>
		</html>
	~;
	$radius->{footer_printed} = 1;
}

sub safe_error {
	my $error = shift || '';
	print qq!<center><font color="red">Error:</font>!;
	print qq! [$radius->{error_string} ($radius->{error_number})]:! if exists $radius->{error_number};
	print ''.($error ? " $error" : '').qq!</center><br>\n!;
}

sub terminal_error {
	my $error = shift || '';
	safe_error($error);
	exit;
}

sub authenticate {
	my $username = $query->param('icradiusweb_username');
	my $password = $query->param('icradiusweb_password');

	return 1 if (&check_existing_session);

	$radius->print_debug('No valid session found, so proceeding to login');

	if ($username && $password) {
		return 1 if (check_new_session($username, $password));

		&print_header;
		safe_error('Invalid Login');
	}

	&print_header;
	&print_login;
	&print_footer;

	return 0;
}

sub check_existing_session {
	$radius->print_debug('Checking for an existing session');

	if ($authentication_type eq '3') {
		my $icisp_admin = new Custdb::Admin;
		my $icisp_db = new Custdb::DB;

		$icisp_db->connect();
		$icisp_admin->init($icisp_db, $query);

		require 'common.pl';

		if (checkLogin($icisp_admin, $query) eq $OK) {
			$radius->{content_printed} = 1 if $icisp_admin->{printedheader};
			return 1;
		}
		exit;
	}

	my $session = $query->cookie(-name => $cookie_name);

	open SESSIONS, "<$sessions_file" or do {
		&print_header; terminal_error("Unable to open file $sessions_file: $!\n");
	};
	flock SESSIONS, LOCK_SH;

	my $count;
	while (<SESSIONS>) {
		chomp;
		my ($time, $sessionid) = /^(\d+):(\w+)/;
		$radius->print_debug("Checking session $sessionid against $session");
		if ($sessionid eq $session) {
			if ((time() - $time) <= $session_timeout) {
				$radius->print_debug('Found valid session');
				close SESSIONS;
				return 1;
			}
			$radius->print_debug('Session found which has expired');
			$count++;
		}
	}
	$radius->print_debug('No sessions found for user') unless $count;

	flock SESSIONS, LOCK_UN;

	close SESSIONS;
	return 0;
}

sub check_new_session {
	my $username = shift;
	my $password = shift;

	$_ = $authentication_type;
	AUTH_TYPE: {
		if (/^1$/) {
			return 0 unless ($radius->user_exists($username));
			return 0 unless ($radius->select_count('radcheck', {Attribute => 'Radius-Operator', Value => 'Yes', UserName => $username}));

			my $sql = "SELECT Value FROM radcheck WHERE Attribute = 'Password' AND UserName = '$username'";
			$radius->print_debug($sql, 'SQL');
			my $sth = $radius->{dbh}->prepare($sql);
			$sth->execute || &terminal_error;
			my ($mysql_password) = $sth->fetchrow_array;
			$sth->finish;

			$password = crypt($password, $mysql_password) if ($radius->select_count('radcheck',
					[[{UserName => $username}],
					[{Attribute => 'Auth-Type', Value => 'Crypt-Local'}]]));

			return write_session($username) if ($mysql_password eq $password);

			last AUTH_TYPE;
		}
		if (/^2$/) {
			my $r = new Authen::Radius(Host => $radhost, Secret => $radsecret);
			return 0 unless $r->check_pwd($username, $password);
			for my $attr (($r->get_attributes)) {
				if ($attr->{'Name'} eq '100') {
					return write_session($username) if unpack('N', $attr->{'RawValue'});
				}
			} 
			last AUTH_TYPE;
		}
	}
	return 0;
}

sub print_login {
	&print_header;

	print qq~
		<center>
		<form action="$cgi" method="post">
			Username:<br>
			<input type="textbox" name="icradiusweb_username"><br><br>
			Password:<br>
			<input type="password" name="icradiusweb_password"><br><br>
			<input type="submit" value="Login"><br>
		</form>
		</center>
	~;

	&print_footer;
}

sub write_session {
	my $username = shift;

	&clean_sessions;

	$radius->print_debug("Writing new session for $username");

	my $md5 = new MD5;
	$md5->add(time());
	$md5->add($username);
	$md5->add($$);
	my $sessionid = $md5->hexdigest;
	
	my $cookie = $query->cookie(-name => $cookie_name, -value => $sessionid, -domain => $cookie_domain);
	print $query->header(-cookie => $cookie);
	$radius->{content_printed} = 1;

	my $time = time();

	open SESSIONS, ">>$sessions_file" or die "Unable to open file $sessions_file: $!\n";
	flock SESSIONS, LOCK_EX;
	print SESSIONS $time, ":$sessionid\n";
	flock SESSIONS, LOCK_UN;
	close SESSIONS;

	return 1;
}

sub clean_sessions {
	$radius->print_debug('Cleaning stale sessions');

	open OLDSESSIONS, "<$sessions_file" or die "Unable to open file $sessions_file: $!\n";
	open NEWSESSIONS, ">$tmp_sessions_file" or die "Unable to open file $tmp_sessions_file: $!\n";

	die "Cannot write to symlink $tmp_sessions_file\n" if (-l $tmp_sessions_file);
	flock OLDSESSIONS, LOCK_EX;

	while (my $line = <OLDSESSIONS>) {
		my ($time) = split /:/, $line;
		if ((time() - $time) <= $session_timeout) {
			$radius->print_debug('Rewriting valid session');
			print NEWSESSIONS $line;
		}
	}

	flock OLDSESSIONS, LOCK_UN;

	close OLDSESSIONS;
	close NEWSESSIONS;

	open OLDSESSIONS, ">$sessions_file" or die "Unable to open file $sessions_file: $!\n";
	open NEWSESSIONS, "<$tmp_sessions_file" or die "Unable to open file $tmp_sessions_file: $!\n";

	flock OLDSESSIONS, LOCK_EX;

	print OLDSESSIONS while (<NEWSESSIONS>);

	flock OLDSESSIONS, LOCK_UN;

	close OLDSESSIONS;
	close NEWSESSIONS;


	unlink $tmp_sessions_file;
}
