/*
 * sql_module.c	- MySQL routines for ICRADIUS and FreeRADIUS SQL module 
 *
 * Mike Machado <mike@innercite.com>
 */

#include <stdio.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include 	"radiusd.h"

/*************************************************************************
 *
 *	Function: sql_query
 *
 *	Purpose: Issue a query to the database
 *
 *************************************************************************/
SQLSOCK *sql_create_socket(void) {
	SQLSOCK *socket;

	if ((socket = malloc(sizeof(SQLSOCK))) == NULL) {
		log(L_CONS|L_ERR, "sql_create_socket: no memory");
		exit(1);
	}

	mysql_init(&(socket->conn));
	if (getpid() == acct_pid) {
		if (!(socket->sock = mysql_real_connect(&(socket->conn), sql->config->sql_acct_server, sql->config->sql_acct_login, sql->config->sql_acct_password, sql->config->sql_acct_db, 0, NULL, 0))) {
			log(L_ERR, "Init: Couldn't connect socket to MySQL server %s@%s:%s", sql->config->sql_acct_login, sql->config->sql_acct_server, sql->config->sql_acct_db);
			socket->sock = NULL;
			return NULL;
		} else {
			DEBUG2(" Connected accounting (pid %d) SQL socket to %s@%s:%s", acct_pid, sql->config->sql_acct_login,sql->config->sql_acct_server,sql->config->sql_acct_db);
		}
	} else {
		if (!(socket->sock = mysql_real_connect(&(socket->conn), sql->config->sql_server, sql->config->sql_login, sql->config->sql_password, sql->config->sql_db, 0, NULL, 0))) {
			log(L_ERR, "Init: Couldn't connect socket to MySQL server %s@%s:%s", sql->config->sql_login, sql->config->sql_server, sql->config->sql_db);
			socket->sock = NULL;
			return NULL;
		} else {
			DEBUG2(" Connected SQL socket to %s@%s:%s", sql->config->sql_login,sql->config->sql_server,sql->config->sql_db);
		}
	}
	return socket;
}

/*************************************************************************
 *
 *	Function: sql_query
 *
 *	Purpose: Issue a query to the database
 *
 *************************************************************************/
int sql_query(SQLSOCK *socket, char *querystr) {

	if (!sql_check_socket(socket))
		return 0;
	if (sql->config->sqltrace)
		DEBUG(querystr);
	if (mysql_query(socket->sock, querystr) == 0)
		return 1;
	else
		return 0;
}


/*************************************************************************
 *
 *	Function: sql_select_query
 *
 *	Purpose: Issue a select query to the database
 *
 *************************************************************************/
int sql_select_query(SQLSOCK *socket, char *querystr) {

	if (!sql_check_socket(socket))
		return 0;
	if (sql->config->sqltrace)
		DEBUG(querystr);
	mysql_query(socket->sock, querystr);
	if (sql_store_result(socket) && sql_num_fields(socket)) 
		return 1;
	else
		return 0;
}


/*************************************************************************
 *
 *	Function: sql_store_result
 *
 *	Purpose: database specific store_result function. Returns a result
 *               set for the query.
 *
 *************************************************************************/
int sql_store_result(SQLSOCK *socket) {

	if (!sql_check_socket(socket))
		return 0;
	if (!(socket->result = mysql_store_result(socket->sock))) {
		log(L_ERR,"MYSQL Error: Cannot get result");
		log(L_ERR,"MYSQL Error: %s",mysql_error(socket->sock));
		return 0;
	}
	return 1;

}


/*************************************************************************
 *
 *	Function: sql_num_fields
 *
 *	Purpose: database specific num_fields function. Returns number
 *               of columns from query
 *
 *************************************************************************/
int sql_num_fields(SQLSOCK *socket) {

	int	num = 0;

	if (!sql_check_socket(socket))
		return -1;

#if MYSQL_VERSION_ID >= 32224
	if (!(num = mysql_field_count(socket->sock))) {
#else
	if (!(num = mysql_num_fields(socket->sock))) {
#endif
		log(L_ERR,"MYSQL Error: Cannot get result");
		log(L_ERR,"MYSQL error: %s",mysql_error(socket->sock));
	}
	return num;
}


/*************************************************************************
 *
 *	Function: sql_num_rows
 *
 *	Purpose: database specific num_rows. Returns number of rows in
 *               query
 *
 *************************************************************************/
int sql_num_rows(SQLSOCK *socket) {

	if (!sql_check_socket(socket))
		return 0;
	return mysql_num_rows(socket->result);
}


/*************************************************************************
 *
 *	Function: sql_fetch_row
 *
 *	Purpose: database specific fetch_row. Returns a SQL_RES struct
 *               with all the data for the query
 *
 *************************************************************************/
SQL_ROW sql_fetch_row(SQLSOCK *socket) {

	if (!sql_check_socket(socket))
		return NULL;
	return mysql_fetch_row(socket->result);
}



/*************************************************************************
 *
 *	Function: sql_free_result
 *
 *	Purpose: database specific free_result. Frees memory allocated
 *               for a result set
 *
 *************************************************************************/
void sql_free_result(SQLSOCK *socket) {

	if (sql_check_socket(socket))
		mysql_free_result(socket->result);
}



/*************************************************************************
 *
 *	Function: sql_error
 *
 *	Purpose: database specific error. Returns error associated with
 *               connection
 *
 *************************************************************************/
char *sql_error(SQLSOCK *socket) {

	if (!sql_check_socket(socket))
		return NULL;
	return mysql_error(socket->sock);
}


/*************************************************************************
 *
 *	Function: sql_close
 *
 *	Purpose: database specific close. Closes an open database
 *               connection
 *
 *************************************************************************/
void sql_close(SQLSOCK *socket) {

	if (sql_check_socket(socket)) {
		mysql_close(socket->sock);
		socket->sock = NULL;
	}
}


/*************************************************************************
 *
 *	Function: sql_finish_query
 *
 *	Purpose: End the query, such as freeing memory
 *
 *************************************************************************/
void sql_finish_query(SQLSOCK *socket) {

}



/*************************************************************************
 *
 *	Function: sql_finish_select_query
 *
 *	Purpose: End the select query, such as freeing memory or result
 *
 *************************************************************************/
void sql_finish_select_query(SQLSOCK *socket) {

	if (sql_check_socket(socket))
		sql_free_result(socket);
}


/*************************************************************************
 *
 *	Function: sql_affected_rows
 *
 *	Purpose: End the select query, such as freeing memory or result
 *
 *************************************************************************/
int sql_affected_rows(SQLSOCK *socket) {

	if (!sql_check_socket(socket))
		return -1;
	return mysql_affected_rows(socket->sock);
}


/*************************************************************************
 *
 *      Function: sql_escape_string
 *
 *      Purpose: Esacpe "'" and any other wierd charactors
 *
 *************************************************************************/
int sql_escape_string(SQLSOCK *socket, char *to, char *from, int length) {

	if (!sql_check_socket(socket))
		return 0;
#if MYSQL_VERSION_ID >= 32315
	return mysql_real_escape_string(socket->sock, to, from, length);
#else
	return mysql_escape_string(to, from, length);
#endif
}

/*************************************************************************
 *
 *      Function: sql_check_socket 
 *
 *      Purpose: Check an SQL socket for a valid database connection
 *
 *************************************************************************/
int sql_check_socket(SQLSOCK *socket) {

	if (socket->sock == NULL) {
		log(L_CONS|L_ERR, "sql_check_socket: socket %d not connected", socket->id);
		return 0;
	} else {
		return 1;
	}
}
