/*
 *
 *	RADIUS
 *	Remote Authentication Dial In User Service
 *
 *
 *	Livingston Enterprises, Inc.
 *	6920 Koll Center Parkway
 *	Pleasanton, CA   94566
 *
 *	Copyright 1992 Livingston Enterprises, Inc.
 *
 *	Permission to use, copy, modify, and distribute this software for any
 *	purpose and without fee is hereby granted, provided that this
 *	copyright and permission notice appear on all copies and supporting
 *	documentation, the name of Livingston Enterprises, Inc. not be used
 *	in advertising or publicity pertaining to distribution of the
 *	program without specific prior permission, and notice be given
 *	in supporting documentation that copying and distribution is by
 *	permission of Livingston Enterprises, Inc.   
 *
 *	Livingston Enterprises, Inc. makes no representations about
 *	the suitability of this software for any purpose.  It is
 *	provided "as is" without express or implied warranty.
 *
 */

char dict_sccsid[] =
"@(#)dict.c	1.5 Copyright 1992 Livingston Enterprises Inc\n"
"@(#)dict.c	2.2 Copyright 1998 Cistron Internet Services B.V.";

#include	<stdio.h>
#include	<stdlib.h>
#include	<sys/types.h>
#include	<pwd.h>
#include	<ctype.h>

#include	"radiusd.h"

#ifndef USEMYSQL
static DICT_ATTR	*dictionary_attributes;
static DICT_VALUE	*dictionary_values;
static DICT_VENDOR	*dictionary_vendors;
#endif

static int		vendorno = 1;

#ifdef NOCASE
#define DICT_STRCMP strcasecmp
#else
#define DICT_STRCMP strcmp
#endif

/*
 *	Free the dictionary_attributes and dictionary_values lists.
 */
void dict_free(void)
{
	DICT_ATTR	*dattr, *anext;
	DICT_VALUE	*dval, *vnext;
	DICT_VENDOR	*dvend, *enext;

	for (dattr = dictionary_attributes; dattr; dattr = anext) {
		anext = dattr->next;
		free(dattr);
	}
	for (dval = dictionary_values; dval; dval = vnext) {
		vnext = dval->next;
		free(dval);
	}
	for (dvend = dictionary_vendors; dvend; dvend = enext) {
		enext = dvend->next;
		free(dvend);
	}
	dictionary_attributes = NULL;
	dictionary_values = NULL;
	dictionary_vendors = NULL;
	vendorno = 1;
}

/*
 *	Add vendor to the list.
 */
int addvendor(char *name, int value)
{
	DICT_VENDOR *vval;

	if ((vval =(DICT_VENDOR *)malloc(sizeof(DICT_VENDOR))) ==
	    (DICT_VENDOR *)NULL) {
		log(L_ERR|L_CONS, "dict_init: out of memory");
		return(-1);
	}
	strcpy(vval->vendorname, name);
	vval->vendorpec  = value;
	vval->vendorcode = vendorno++;

	/* Insert at front. */
	vval->next = dictionary_vendors;
	dictionary_vendors = vval;

	return 0;
}

/*
 *	Initialize the dictionary.  Read all ATTRIBUTES into
 *	 the dictionary_attributes list.  Read all VALUES into
 *	 the dictionary_values list.
 */
#ifndef USEMYSQL
int
dict_init(char *fn)
{
	FILE	*dictfd;
	char	dummystr[64];
	char	namestr[64];
	char	valstr[64];
	char	attrstr[64];
	char	typestr[64];
	char	vendorstr[64];
	int	line_no;
	DICT_ATTR	*attr;
	DICT_VALUE	*dval;
	DICT_VENDOR	*v;
	char	buffer[256];
	int	value;
	int	type;
	int	vendor;
	int	is_attrib;
#ifdef ATTRIB_NMC
	int	vendor_usr_seen = 0;
	int	is_nmc = 0;
#endif

	if (fn == NULL) dict_free();

	if (fn) {
		if (fn[0] == '/')
			strcpy(buffer, fn);
		else
			sprintf(buffer, "%s/%s", radius_dir, fn);
	} else
		sprintf(buffer, "%s/%s", radius_dir, RADIUS_DICTIONARY);

	if((dictfd = fopen(buffer, "r")) == (FILE *)NULL) {
		log(L_CONS|L_ERR, "dict_init: Couldn't open dictionary: %s",
			buffer);
		return(-1);
	}

	line_no = 0;
	while(fgets(buffer, sizeof(buffer), dictfd) != (char *)NULL) {
		line_no++;
		
		/* Skip empty space */
		if(*buffer == '#' || *buffer == '\0' || *buffer == '\n') {
			continue;
		}

		if (strncasecmp(buffer, "$INCLUDE", 8) == 0) {

			/* Read the $INCLUDE line */
			if(sscanf(buffer, "%s%s", dummystr, valstr) != 2) {
				log(L_ERR,
			"%s: Invalid filename on line %d of dictionary\n",
					progname, line_no);
				return(-1);
			}
			if (dict_init(valstr) < 0)
				return -1;
			continue;
		}

		is_attrib = 0;
		if (strncmp(buffer, "ATTRIBUTE", 9) == 0)
			is_attrib = 1;
#ifdef ATTRIB_NMC
		is_nmc = 0;
		if (strncmp(buffer, "ATTRIB_NMC", 10) == 0)
			is_attrib = is_nmc = 1;
#endif
		if (is_attrib) {
			/* Read the ATTRIBUTE line */
			vendor = 0;
			vendorstr[0] = 0;
			if(sscanf(buffer, "%s%s%s%s%s", dummystr, namestr,
					valstr, typestr, vendorstr) < 4) {
				log(L_ERR,
			"%s: Invalid attribute on line %d of dictionary\n",
					progname, line_no);
				return(-1);
			}

#ifdef ATTRIB_NMC
			/*
			 *	Convert ATTRIB_NMC into our format.
			 *	We might need to add USR to the list of
			 *	vendors first.
			 */
			if (is_nmc && vendorstr[0] == 0) {
				if (!vendor_usr_seen) {
					if (addvendor("USR", VENDORPEC_USR) < 0)
						return -1;
					vendor_usr_seen = 1;
				}
				strcpy(vendorstr, "USR");
			}
#endif

			/*
			 * Validate all entries
			 */
			if(strlen(namestr) > 31) {
				log(L_ERR|L_CONS,
		"dict_init: Invalid name length on line %d of dictionary",
					line_no);
				return(-1);
			}

			if(!isdigit(*valstr)) {
				log(L_ERR|L_CONS,
			"dict_init: Invalid value on line %d of dictionary",
					line_no);
				return(-1);
			}
			if (valstr[0] != '0')
				value = atoi(valstr);
			else
				sscanf(valstr, "%i", &value);

			if(strcmp(typestr, "string") == 0) {
				type = PW_TYPE_STRING;
			}
			else if(strcmp(typestr, "integer") == 0) {
				type = PW_TYPE_INTEGER;
			}
			else if(strcmp(typestr, "ipaddr") == 0) {
				type = PW_TYPE_IPADDR;
			}
			else if(strcmp(typestr, "date") == 0) {
				type = PW_TYPE_DATE;
			}
			else {
				log(L_ERR|L_CONS,
			"dict_init: Invalid type on line %d of dictionary",
					line_no);
				return(-1);
			}

			for (v = dictionary_vendors; v; v = v->next) {
				if (strcmp(vendorstr, v->vendorname) == 0)
					vendor = v->vendorcode;
			}
			if (vendorstr[0] && !vendor) {
				log(L_ERR|L_CONS,
			"dict_init: unknown vendor %s on line %d of dictionary",
				vendorstr, line_no);
				return -1;
			}

			/* Create a new attribute for the list */
			if((attr = (DICT_ATTR *)malloc(sizeof(DICT_ATTR))) ==
					(DICT_ATTR *)NULL) {
				log(L_ERR|L_CONS, "dict_init: out of memory");
				return(-1);
			}
			strcpy(attr->name, namestr);
			attr->value = value;
			attr->type = type;
			if (vendor)
				attr->value |= (vendor << 16);

			/*
			 *	Add to the front of the list, so that
			 *	values at the end of the file override
			 *	those in the begin.
			 */
			attr->next = dictionary_attributes;
			dictionary_attributes = attr;

		}
		else if (strncmp(buffer, "VALUE", 5) == 0) {

			/* Read the VALUE line */
			if(sscanf(buffer, "%s%s%s%s", dummystr, attrstr,
						namestr, valstr) != 4) {
				log(L_ERR|L_CONS,
		"dict_init: Invalid value entry on line %d of dictionary",
					line_no);
				return(-1);
			}

			/*
			 * Validate all entries
			 */
			if(strlen(attrstr) > 31) {
				log(L_ERR|L_CONS,
		"dict_init: Invalid attribute length on line %d of dictionary",
					line_no);
				return(-1);
			}

			if(strlen(namestr) > 31) {
				log(L_ERR|L_CONS,
		"dict_init: Invalid name length on line %d of dictionary",
					line_no);
				return(-1);
			}

			if(!isdigit(*valstr)) {
				log(L_ERR|L_CONS,
			"dict_init: Invalid value on line %d of dictionary",
					line_no);
				return(-1);
			}
			value = atoi(valstr);

			/* Create a new VALUE entry for the list */
			if((dval = (DICT_VALUE *)malloc(sizeof(DICT_VALUE))) ==
					(DICT_VALUE *)NULL) {
				log(L_ERR|L_CONS, "dict_init: out of memory");
				return(-1);
			}
			strcpy(dval->attrname, attrstr);
			strcpy(dval->name, namestr);
			dval->value = value;

			/* Insert at front. */
			dval->next = dictionary_values;
			dictionary_values = dval;
		}
		else if(strncmp(buffer, "VENDOR", 6) == 0) {

			/* Read the VENDOR line */
			if(sscanf(buffer, "%s%s%s", dummystr, attrstr,
						valstr) != 3) {
				log(L_ERR|L_CONS,
		"dict_init: Invalid vendor entry on line %d of dictionary",
					line_no);
				return(-1);
			}

			/*
			 * Validate all entries
			 */
			if(strlen(attrstr) > 31) {
				log(L_ERR|L_CONS,
		"dict_init: Invalid attribute length on line %d of dictionary",
					line_no);
				return(-1);
			}

			if(!isdigit(*valstr)) {
				log(L_ERR|L_CONS,
			"dict_init: Invalid value on line %d of dictionary",
					line_no);
				return(-1);
			}
			value = atoi(valstr);

			/* Create a new VENDOR entry for the list */
			if (addvendor(attrstr, value) < 0)
				return -1;
#ifdef ATTRIB_NMC
			if (value == VENDORPEC_USR)
				vendor_usr_seen = 1;
#endif
		}
	}
	fclose(dictfd);
	return(0);
}
#endif

/*************************************************************************
 *
 *	Function: dict_attrget
 *
 *	Purpose: Return the full attribute structure based on the
 *		 attribute id number.
 *
 *************************************************************************/

DICT_ATTR	*
dict_attrget(attribute)
int	attribute;
{
	DICT_ATTR	*attr;

	attr = dictionary_attributes;
	while(attr != (DICT_ATTR *)NULL) {
		if(attr->value == attribute) {
			return(attr);
		}
		attr = attr->next;
	}
	return((DICT_ATTR *)NULL);
}

/*************************************************************************
 *
 *	Function: dict_attrfind
 *
 *	Purpose: Return the full attribute structure based on the
 *		 attribute name.
 *
 *************************************************************************/

DICT_ATTR	*
dict_attrfind(attrname)
char	*attrname;
{
	DICT_ATTR	*attr;

	attr = dictionary_attributes;
	while(attr != (DICT_ATTR *)NULL) {
		if(DICT_STRCMP(attr->name, attrname) == 0) {
			return(attr);
		}
		attr = attr->next;
	}
	return((DICT_ATTR *)NULL);
}

/*************************************************************************
 *
 *	Function: dict_valfind
 *
 *	Purpose: Return the full value structure based on the
 *		 value name.
 *
 *************************************************************************/

DICT_VALUE	*
dict_valfind(valname)
char	*valname;
{
	DICT_VALUE	*val;

	val = dictionary_values;
	while(val != (DICT_VALUE *)NULL) {
		if(DICT_STRCMP(val->name, valname) == 0) {
			return(val);
		}
		val = val->next;
	}
	return((DICT_VALUE *)NULL);
}

/*************************************************************************
 *
 *	Function: dict_valgetname (MYSQL Patch)
 *
 *	Purpose: Return the name based on the integer lvalue
 *
 *************************************************************************/
char * dict_valgetname(value, attrname)
UINT4	value;
char	*attrname;
{
	DICT_VALUE	*val;

	val = dictionary_values;
	while(val != (DICT_VALUE *)NULL) {
		if(DICT_STRCMP(val->attrname, attrname) == 0 &&
						val->value == value) {
			return((char *)&(val->name));
		}
		val = val->next;
	}
	return((char *)NULL);
}

/*************************************************************************
 *
 *	Function: dict_valget
 *
 *	Purpose: Return the full value structure based on the
 *		 actual value and the associated attribute name.
 *
 *************************************************************************/

DICT_VALUE	*
dict_valget(value, attrname)
UINT4	value;
char	*attrname;
{
	DICT_VALUE	*val;

	val = dictionary_values;
	while(val != (DICT_VALUE *)NULL) {
		if(DICT_STRCMP(val->attrname, attrname) == 0 &&
						val->value == value) {
			return(val);
		}
		val = val->next;
	}
	return((DICT_VALUE *)NULL);
}

/*
 *	Get the PEC (Private Enterprise Code) of the vendor
 *	based on it's internal number.
 */
int dict_vendorpec(int code)
{
	DICT_VENDOR	*v;

	for (v = dictionary_vendors; v; v = v->next)
		if (v->vendorcode == code)
			break;

	return v ? v->vendorpec : 0;
}

/*
 *	Get the internal code of the vendor based on its PEC.
 */
int dict_vendorcode(int pec)
{
	DICT_VENDOR	*v;

	for (v = dictionary_vendors; v; v = v->next)
		if (v->vendorpec == pec)
			break;

	return v ? v->vendorcode : 0;
}

