/*
 * proxy.c	Proxy stuff.
 *
 * Version:	@(#)proxy.c  1.51  04-May-1999  miquels@cistron.nl
 */

char proxy_sccsid[] =
"@(#)proxy.c 	1.51 Copyright 1999 Cistron Internet Services B.V.";

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<time.h>
#include	<ctype.h>

#include	"radiusd.h"

/*
 *	Make sure send_buffer is aligned properly.
 */
static		int i_send_buffer[1024];
static		char *send_buffer = (char *)i_send_buffer;

int		proxy_id = 1;
AUTH_REQ	*proxy_requests;

static int allowed [] = {
	PW_SERVICE_TYPE,
	PW_FRAMED_PROTOCOL,
	PW_FILTER_ID,
	PW_FRAMED_MTU,
	PW_FRAMED_COMPRESSION,
	PW_LOGIN_SERVICE,
	PW_REPLY_MESSAGE,
	PW_SESSION_TIMEOUT,
	PW_IDLE_TIMEOUT,
	PW_PORT_LIMIT,
	0,
};

/*
 *	Generate a random vector.
 */
void random_vector(char *vector)
{
	int	randno;
	int	i;

	srand(time(0) + getpid());
	for(i = 0;i < AUTH_VECTOR_LEN;) {
		randno = rand();
		memcpy(vector, &randno, sizeof(int));
		vector += sizeof(int);
		i += sizeof(int);
	}
}

/*
 *	Cleanup old outstanding requests.
 */
static void proxy_cleanup(void)
{
	AUTH_REQ 		*a, *last, *next;
	time_t			now;

	last = NULL;
	now  = time(NULL);

	for (a = proxy_requests; a; a = next) {
		next = a->next;
		if (a->timestamp + MAX_REQUEST_TIME < now) {
			if (last)
				last->next = a->next;
			else
				proxy_requests = a->next;

			if (a->data) free(a->data);
			if (a->server_reply) pairfree(a->server_reply);
			free(a);

			continue;
		}
		last = a;
	}
}

/*
 *	Add a proxy-pair to the end of the request.
 */
static VALUE_PAIR *proxy_addinfo(AUTH_REQ *authreq)
{
	VALUE_PAIR		*proxy_pair, *vp;

	if  (!(proxy_pair = (VALUE_PAIR *)malloc(sizeof(VALUE_PAIR)))) {
		log(L_ERR|L_CONS, "no memory");
		exit(1);
	}
	memset(proxy_pair, 0, sizeof(VALUE_PAIR));

	strcpy(proxy_pair->name, "Proxy-State");
	proxy_pair->attribute = PW_PROXY_STATE;
	proxy_pair->type = PW_TYPE_STRING;
	sprintf(proxy_pair->strvalue, "%04x", (unsigned)authreq->server_id);
	proxy_pair->length = 4;

	for (vp = authreq->request; vp && vp->next; vp = vp->next)
		;
	vp->next = proxy_pair;
	return vp;
}

/*
 *	Add the authreq to the list.
 */
int proxy_addrequest(AUTH_REQ *authreq, int *proxy_id)
{
	char		*m;
	AUTH_REQ	*a, *last = NULL;
	int		id = -1;

	/*
	 *	See if we already have a similar outstanding request.
	 */
	for (a = proxy_requests; a; a = a->next) {
		if (a->ipaddr == authreq->ipaddr &&
		    a->id == authreq->id &&
		    memcmp(a->vector, authreq->vector, sizeof(a->vector)) == 0)
			break;
		last = a;
	}
	if (a) {
		/*
		 *	Yes, this is a retransmit so delete the
		 *	old request.
		 */
		id = a->server_id;
		if (last)
			last->next = a->next;
		else
			proxy_requests = a->next;
		if (a->data) free(a->data);
		if (a->server_reply) pairfree(a->server_reply);
		free(a);
	}
	if (id < 0) {
		id = (*proxy_id)++;
		*proxy_id &= 0xFFFF;
	}

	authreq->next = NULL;
	authreq->child_pid = -1;
	authreq->timestamp = time(NULL);

	/*
	 *	Copy the static data into malloc()ed memory.
	 */
	if ((m = malloc(authreq->data_len)) == NULL) {
		log(L_ERR|L_CONS, "no memory");
		exit(1);
	}
	memcpy(m, authreq->data, authreq->data_len);
	authreq->data = m;

	authreq->next = proxy_requests;
	proxy_requests = authreq;

	return id;
}

/*
 *	Decode a password and encode it again.
 *
 *	FIXME: Ascend gear sends short passwords, they should be
 *	AUTH_PASS_LEN. We should fix this up, instead we send a
 *	short password to next radius server too. Well, as long as it works..
 */
static void passwd_recode(char *secret_key, char *vector,
		char *pw_digest, char *pass, int passlen)
{
	char	passtmp[AUTH_PASS_LEN];
	char	passwd[AUTH_PASS_LEN];
	char	md5buf[256];
	int	i;
	int	len;

	/*
	 *	Decode. Copy "pass" to "passwd" first since
	 *	Ascend gear likes to send short passwords.
	 *	Should be a multiple of AUTH_PASS_LEN.
	 */
	memset(passwd, 0, AUTH_PASS_LEN);
	memcpy(passwd, pass, AUTH_PASS_LEN);

	for(i = 0 ;i < AUTH_PASS_LEN; i++)
		passwd[i] ^= pw_digest[i];
	passwd[AUTH_PASS_LEN] = '\0';

	/*
	 *	Encode with new secret.
	 */
	memset(passtmp, 0, sizeof(passtmp));
	len = strlen(secret_key);
	strcpy(md5buf, secret_key);
	memcpy(md5buf + len, vector, AUTH_VECTOR_LEN);
	md5_calc(passtmp, md5buf, len + AUTH_VECTOR_LEN);
	for(i = 0; passwd[i] && i < AUTH_PASS_LEN; i++)
		passtmp[i] ^= passwd[i];

	/*
	 *	Copy newly encoded password back to
	 *	where we got it from.
	 */
	memcpy(pass, passtmp, passlen);
}

/*
 *	Relay the request to a remote server.
 *	Returns:  1 success (we reply, caller returns without replying)
 *	          0 fail (caller falls through to normal processing)
 *		 -1 fail (we don't reply, caller returns without replying)
 */
int proxy_send(AUTH_REQ *authreq, int activefd)
{
	VALUE_PAIR		*namepair;
	VALUE_PAIR		*vp, *pp;
	AUTH_HDR		*auth;
	char			*secret_key;
	char			*length_ptr;
	int			len, total_length;
	int			vendorcode, vendorpec;
	UINT4			lvalue;
	char			vector[AUTH_VECTOR_LEN];
	char			pw_digest[16];
	char			saved_username[AUTH_STRING_LEN];
	char			*realmname;
	REALM			*realm;
	CLIENT			*client;
	char			*ptr;
	short			rport;
	struct sockaddr_in	saremote, *sin;
	char			*what = "unknown";

	/*
	 *	Look up name.
	 */
	namepair = pairfind(authreq->request, PW_USER_NAME);
	if (namepair == NULL)
		return 0;
	strcpy(saved_username, namepair->strvalue);

	/*
	 *	Find the realm from the _end_ so that we can
	 *	cascade realms: user@realm1@realm2.
	 *	Use the original username if available.
	 */
	if (authreq->username[0]) {
		strncpy(namepair->strvalue, authreq->username,
			sizeof(namepair->strvalue));
		namepair->strvalue[sizeof(namepair->strvalue) - 1] = 0;
	}

	/*
	 *	Now check if we know this realm!
	 *	A NULL realm is OK.
	 *	If not found, we treat it as usual).
	 */
	if ((realmname = strrchr(namepair->strvalue, '@')) != NULL)
		realmname++;
	if ((realm = realm_find(realmname ? realmname : "NULL")) == NULL) {
		strcpy(namepair->strvalue, saved_username);
		return 0;
	}

	/*
	 *	The special server LOCAL ?
	 */
	if (strcmp(realm->server, "LOCAL") == 0) {
		strcpy(namepair->strvalue, saved_username);
		if (realm->striprealm &&
		    ((realmname = strrchr(namepair->strvalue, '@')) != NULL)) {
			*realmname = 0;
			namepair->length = strlen(namepair->strvalue);
		}
		return 0;
	}

	if ((client = client_find(realm->ipaddr)) == NULL) {
		log(L_PROXY, "cannot find secret for server %s",
			realm->server);
		strcpy(namepair->strvalue, saved_username);
		return 0;
	}

	if (realmname != NULL) {
		if (realm->striprealm)
			realmname[-1] = 0;
		strncpy(authreq->realm, realmname, sizeof(authreq->realm));
		authreq->realm[sizeof(authreq->realm) - 1] = 0;
	} else
		authreq->realm[0] = 0;

	namepair->length = strlen(namepair->strvalue);

	if (authreq->code == PW_AUTHENTICATION_REQUEST)
		rport = realm->auth_port;
	else
		rport = realm->acct_port;

	secret_key = client->secret;

	authreq->server_ipaddr = realm->ipaddr;

	/*
	 *	Is this a valid & signed request ?
	 */
	switch (authreq->code) {
		case PW_AUTHENTICATION_REQUEST:
			what = "authentication";
			/*
			 *	FIXME: we have already calculated the
			 *	digest in rad_auth_init()
			 */
			if (calc_digest(pw_digest, authreq) != 0) {
				authfree(authreq);
				return -1;
			}
			break;
		case PW_ACCOUNTING_REQUEST:
			what = "accounting";
			if (calc_acctdigest(pw_digest, authreq) < 0) {
				/*
				 *	FIXME: complain
				 */
				authfree(authreq);
				return -1;
			}
			break;
	}

	/*
	 *	Now build a new request and send it to the remote radiusd.
	 *
	 *	FIXME: it could be that the id wraps around too fast if
	 *	we have a lot of requests, it might be better to keep
	 *	a seperate ID value per remote server.
	 *
	 *	OTOH the remote radius server should be smart enough to
	 *	compare _both_ ID and vector. Right ?
	 */

	/*
	 *	XXX: we re-use the vector from the original request
	 *	here, since that's easy for retransmits ...
	 */
	/* random_vector(vector); */
	memcpy(vector, authreq->vector, sizeof(vector));
	auth = (AUTH_HDR *)send_buffer;
	memset(auth, 0, sizeof(AUTH_HDR));
	auth->code = authreq->code;
	if (auth->code == PW_AUTHENTICATION_REQUEST)
		memcpy(auth->vector, vector, AUTH_VECTOR_LEN);

	/*
	 *	Add the request to the list of outstanding requests.
	 *	Note that authreq->server_id is a 16 bits value,
	 *	while auth->id contains only the 8 least significant
	 *	bits of that same value.
	 */
	authreq->server_id = proxy_addrequest(authreq, &proxy_id);
	auth->id = authreq->server_id & 0xFF;

	/*
	 *	Add PROXY_STATE attribute.
	 */
	pp = proxy_addinfo(authreq);

	/*
	 *	If there is no PW_CHAP_CHALLENGE attribute but there
	 *	is a PW_CHAP_PASSWORD we need to add it since we can't
	 *	use the request authenticator anymore - we changed it.
	 */
	if (pairfind(authreq->request, PW_CHAP_PASSWORD) &&
	    pairfind(authreq->request, PW_CHAP_CHALLENGE) == NULL) {
		if (!(vp = (VALUE_PAIR *)malloc(sizeof(VALUE_PAIR)))) {
			log(L_ERR|L_CONS, "no memory");
			exit(1);
		}
		memset(vp, 0, sizeof(VALUE_PAIR));

		strcpy(vp->name, "CHAP-Challenge");
		vp->attribute = PW_CHAP_CHALLENGE;
		vp->type = PW_TYPE_STRING;
		vp->length = AUTH_VECTOR_LEN;
		memcpy(vp->strvalue, authreq->vector, AUTH_VECTOR_LEN);
		pairadd((&authreq->request), vp);
	}

	/*
	 *	Ready to send the request.
	 */
	DEBUG("Sending %s request of id %d to %lx (server %s:%d)",
		what, auth->id, realm->ipaddr, realm->server, rport);

	total_length = AUTH_HDR_LEN;

	/*
	 *	Put all the attributes into a buffer.
	 */
	ptr = auth->data;
	for (vp = authreq->request; vp; vp = vp->next) {
		/*
		 *	Check for overflow.
		 */
		if (total_length + vp->length + 16 >= sizeof(i_send_buffer))
			break;

		debug_pair(stdout, vp);

		/*
		 *	This could be a vendor-specific attribute.
		 */
		length_ptr = NULL;
		if ((vendorcode = VENDOR(vp->attribute)) > 0 &&
		    (vendorpec  = dict_vendorpec(vendorcode)) > 0) {
			*ptr++ = PW_VENDOR_SPECIFIC;
			length_ptr = ptr;
			*ptr++ = 6;
			lvalue = htonl(vendorpec);
			memcpy(ptr, &lvalue, 4);
			ptr += 4;
			total_length += 6;
		} else if (vp->attribute > 0xff) {
			/*
			 *	Ignore attributes > 0xff
			 */
			continue;
		} else
			vendorpec = 0;

#ifdef ATTRIB_NMC
		if (vendorpec == VENDORPEC_USR) {
			lvalue = htonl(vp->attribute & 0xFFFF);
			memcpy(ptr, &lvalue, 4);
			total_length += 2;
			*length_ptr  += 2;
			ptr          += 4;
		} else
#endif
		*ptr++ = (vp->attribute & 0xFF);

		switch (vp->type) {
			case PW_TYPE_STRING:
#ifdef ATTRIB_NMC
				if (vendorpec != VENDORPEC_USR)
#endif
					*ptr++ = vp->length + 2;
				if (length_ptr) *length_ptr += vp->length + 2;
				total_length += 2 + vp->length;
				memcpy(ptr, vp->strvalue, vp->length);

				/*
				 *	Re-encode passwd on the fly.
				 */
				if (vp->attribute == PW_PASSWORD)
					passwd_recode(secret_key, vector,
						pw_digest, ptr, vp->length);

				ptr += vp->length;
				break;
			case PW_TYPE_INTEGER:
			case PW_TYPE_DATE:
			case PW_TYPE_IPADDR:
				/*
				 *	FIXME: this works for IPADDR because
				 *	it is still stored internally in
				 *	hostorder (ugh).
				 */
#ifdef ATTRIB_NMC
				if (vendorpec != VENDORPEC_USR)
#endif
					*ptr++ = sizeof(UINT4) + 2;
				if (length_ptr) *length_ptr += sizeof(UINT4)+ 2;
				lvalue = htonl(vp->lvalue);
				memcpy(ptr, &lvalue, sizeof(UINT4));
				ptr += sizeof(UINT4);
				total_length += sizeof(UINT4) + 2;
				break;
			default:
				break;
		}
	}
	auth->length = htons(total_length);

	/*
	 *	If this is not an authentication request, we
	 *	need to calculate the md5 hash over the entire packet
	 *	and put it in the vector.
	 */
	if (auth->code != PW_AUTHENTICATION_REQUEST) {
		len = strlen(secret_key);
		if (total_length + len < sizeof(i_send_buffer)) {
			strcpy(send_buffer + total_length, secret_key);
			md5_calc(auth->vector, send_buffer, total_length+len);
		}
	}

	/*
	 *	And send it to the remote radius server.
	 */
	sin = (struct sockaddr_in *) &saremote;
	memset ((char *) sin, '\0', sizeof (saremote));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = htonl(realm->ipaddr);
	sin->sin_port = htons(rport);

	sendto(activefd, auth, total_length, 0,
		(struct sockaddr *)sin, sizeof(struct sockaddr_in));

	/*
	 *	Remove proxy-state from list.
	 */
	pairfree(pp->next);
	pp->next = NULL;

	/*
	 *	And restore username.
	 */
	strcpy(namepair->strvalue, saved_username);
	namepair->length = strlen(namepair->strvalue);

	return 1;
}


/*
 *	We received a response from a remote radius server.
 *	Find the original request, then return.
 *	Returns:   0 proxy found
 *		  -1 error don't reply
 */
int proxy_receive(AUTH_REQ *authreq, int activefd)
{
	VALUE_PAIR	*vp, *last, *prev, *x;
	VALUE_PAIR	*allowed_pairs;
	AUTH_REQ	*oldreq, *lastreq;
	char		*s;
	int		pp = -1;
	int		i;

	/*
	 *	First cleanup old outstanding requests.
	 */
	proxy_cleanup();

	/*
	 *	FIXME: calculate md5 checksum!
	 */

	/*
	 *	Find the last PROXY_STATE attribute.
	 */
	oldreq  = NULL;
	lastreq = NULL;
	last    = NULL;
	x       = NULL;
	prev    = NULL;

	for (vp = authreq->request; vp; vp = vp->next) {
		if (vp->attribute == PW_PROXY_STATE) {
			prev = x;
			last = vp;
		}
		x = vp;
	}
	if (last && last->strvalue) {
		/*
		 *	Merit really rapes the Proxy-State attribute.
		 *	See if it still is a valid 4-digit hex number.
		 */
		s = last->strvalue;
		if (strlen(s) == 4 && isxdigit(s[0]) && isxdigit(s[1]) &&
		    isxdigit(s[2]) && isxdigit(s[3])) {
			pp = strtol(last->strvalue, NULL, 16);
		} else {
			log(L_PROXY, "server %s mangled Proxy-State attribute",
			client_name(authreq->ipaddr));
		}
	}

	/*
	 *	Now find it in the list of outstanding requests.
	 */

	for (oldreq = proxy_requests; oldreq; oldreq = oldreq->next) {
		/*
		 *	Some servers drop the proxy pair. So
		 *	compare in another way if needed.
		 */
		if (pp >= 0 && pp == oldreq->server_id)
			break;
		if (pp < 0 &&
		    authreq->ipaddr == oldreq->server_ipaddr &&
		    authreq->id     == (oldreq->server_id & 0xFF))
			break;
		lastreq = oldreq;
	}

	if (oldreq == NULL) {
		log(L_PROXY, "Unreckognized proxy reply from server %s - ID %d",
			client_name(authreq->ipaddr), authreq->id);
		return -1;
	}

	/*
	 *	Remove oldreq from list.
	 */
	if (lastreq)
		lastreq->next = oldreq->next;
	else
		proxy_requests = oldreq->next;

	/*
	 *	Remove proxy pair from list.
	 */
	if (last) {
		if (prev)
			prev->next = last->next;
		else
			authreq->request = last->next;
	}

#if 0 /* Not needed anymore. */
	/*
	 *	Rebuild username.
	 */
	if ((vp = pairfind(oldreq->request, PW_USER_NAME)) == NULL) {
		log(L_PROXY, "Huh? old request lost PW_USER_NAME");
		return -1;
	}
	sprintf(vp->strvalue, "%s@%s", vp->strvalue, oldreq->realm);
	vp->length = strlen(vp->strvalue);
#endif

	/*
	 *	Only allow some attributes to be propagated from
	 *	the remote server back to the NAS, for security.
	 */
	allowed_pairs = NULL;
	for(i = 0; allowed[i]; i++)
		pairmove2(&allowed_pairs, &(authreq->request), allowed[i]);
	pairfree(authreq->request);

	/*
	 *	Now rebuild the AUTHREQ struct, so that the
	 *	normal functions can process it.
	 */
	oldreq->server_reply = allowed_pairs;
	oldreq->server_code  = authreq->code;
	oldreq->validated    = 1;
	memcpy(authreq->data, oldreq->data, oldreq->data_len);
	free(oldreq->data);
	oldreq->data = authreq->data;
	memcpy(authreq, oldreq, sizeof(AUTH_REQ));

	return 0;
}

