/* HTBL - a simple hash table */

#include <evblib/str/str.h>
#include <evblib/db/list.h>


/* Flags for HTBL/HTHE_PUT */

#define HTBL_PF_FREEKEY		1
#define HTBL_PF_FREEDATA	2
#define HTBL_PF_FREEALL		3
#define HTBL_PF_TOFRONT		4


/* Types */

typedef uint32_t hashval_t;

typedef hashval_t (*htfunc_t)(void *data, ssize_t datal);

typedef struct htent {
	hashval_t hv;
	STR_T key;
	STR_T data;
	struct htent *prev, *next;
	int flags;
} HTENT_T;				/* hash table entry */

typedef struct hthent {
	HTENT_T *head, *tail;
} HTHENT_T;				/* hash table head entry */

typedef struct htbl {
	hashval_t mod;
	HTHENT_T *t;
	htfunc_t f;
} HTBL_T;				/* hash table */


/* htbl_init returns -1 if a memory allocation error occurs, 0 otherwise */

int htbl_init(HTBL_T *ht, hashval_t entries, htfunc_t f);

void htbl_done(HTBL_T *ht);

/* This returns the head entry in headent, which can be used to implement
 * LRU or MRU schemes by calling hthe_tofront or hthe_toback with headent and
 * the returned entry.
 *
 * The return variables headent and hv are also set if the hash value or key is
 * not in the table, and can be used for a subsequent hthe_put, saving a call
 * to your hash function for conditional puts. */

HTENT_T *htbl_getent(HTBL_T *ht, char *key, ssize_t keyl, 
		     HTHENT_T **headent, hashval_t *hashval);

/* These return the created hash table entry or 0 on memory allocation errors */

HTENT_T *htbl_addent(HTBL_T *ht, char *key, ssize_t keyl, 
		     char *data, ssize_t datal, int flags);

HTENT_T *hthe_addent(HTHENT_T *he, hashval_t hashval, char *key, ssize_t keyl, 
		     char *data, ssize_t datal, int flags);

/* Move an entry to front or back */

static inline void hthe_enttofront(HTHENT_T *he, HTENT_T *ent)
{
	LIST_DEL(&he->head, &he->tail, ent);
	LIST_ADDHEAD(&he->head, &he->tail, ent);
}

static inline void hthe_enttoback(HTHENT_T *he, HTENT_T *ent)
{
	LIST_DEL(&he->head, &he->tail, ent);
	LIST_ADDTAIL(&he->head, &he->tail, ent);
}

/* Delete an entry from the table */

void hthe_delent(HTHENT_T *he, HTENT_T *ent);

/* Iterate over head entries in the table and over entries under the head */

#define htbl_getfirsthe(ht)	 ((ht)->t)
#define htbl_getnexthe(ht, he)	 (((he) < (ht)->t + (ht)->mod) ? (he) + 1 : 0)
#define htbe_getfirstent(he)	 ((he)->head)
#define htbe_getnextent(he, ent) ((ent)->next)

