/*
 * STREAM - Functions to support efficient I/O streams based on fastrings
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2004/01/13 - EvB - Created
 */

char stream_id[] = "STREAM - Copyright (C) 2004 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <string.h>		/* for memchr */
#include <stdlib.h>		/* for malloc */

#include <evblib/stream/stream.h>
#include <evblib/misc/pagesize.h>
#include <evblib/misc/misc.h>	/* for safe_read */


/* 
 * FUNCTIONS
 */


STRM_T *strm_new(int fd, ssize_t bufsize)
{
    STRM_T *ret;
    size_t pagesize;
 
    pagesize = GETPAGESIZE;
    bufsize = ((bufsize + pagesize - 1) / pagesize) * pagesize;

    ret = (STRM_T *)malloc(sizeof(STRM_T)); if (!ret) return 0;
    memset(ret, 0, sizeof(STRM_T));
    ret->bufsize = bufsize;
    ret->r = ring_new(bufsize * 2); if (!ret->r) { free(ret); return 0; }
    ret->fd = fd;

    return ret;
}


void strm_del(STRM_T *strm)
{
    ring_del(strm->r);
    free(strm);
}


STR_T strm_getline(STRM_T *strm)
{
    ssize_t max, len, rd, rl;
    char *end, *rp;
    
    for(;;) {
	max = ring_maxget(strm->r);
	end = memchr(strm->r->r, '\n', max);
	if (end) { rl = end - strm->r->r + 1; break; }

	if (strm->eof) { rl = max; break; }
        rd = ring_maxput(strm->r); 
	if (!rd) { rl = max; break; }	   /* line cut in two */
	if (rd > strm->bufsize) rd = strm->bufsize;
	len = safe_read(strm->fd, strm->r->w, rd);
	ring_put(strm->r, len);
	if (!len) strm->eof++;
    }
    strm->aux++;
    rp = strm->r->r; 
    ring_get(strm->r, rl);
    return (STR_T){rp, rl};
}


/* Ensures len bytes minimum available in buffer; returns actual size */

STR_T strm_ensure(STRM_T *strm, ssize_t len)
{
    ssize_t max, hasread, toread;
    STR_T ret;

    for(;;) {
	max = ring_maxget(strm->r); 
	if (max >= len) { ret.l = len; break; }

	if (strm->eof)  { ret.l = max; break; }
	toread = ring_maxput(strm->r); 
	if (!toread) { ret.l = max; break; }
	hasread = safe_read(strm->fd, strm->r->w, toread);
	if (hasread) { ring_put(strm->r, hasread); continue; }
	strm->eof++;
    }
    ret.p = strm->r->r;
    return ret;
}

/*
 * vim:softtabstop=4:sw=4
 */

