/*
 * STRIO - Functions to format and output integers and strings
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2005/11/04 - EvB - Created
 */

char strio_id[] = "STRIO - Copyright (C) 2005 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <stdarg.h>				/* for va_* */

#include <evblib/sysdefs/sysdefs-libc.h>
#include <evblib/misc/misc.h>			/* for safe_malloc */

#include <evblib/strio/strio.h>


/*
 * FUNCTIONS
 */


/* Read an entire file, using readstrfd */

STR_T readstr(char *fname)
{
    STR_T ret;
    int fd;

    fd = open(fname, O_RDONLY);
    if (fd >= 0) { ret = readstrfd(fd); close(fd); return ret; }

    ret.p = 0;
    ret.l = -errno;
    return ret;
}


/* Read an entire open file from start, regardless of current file position; 
 * gets size beforehand using lseek. Creates a harmless race condition; 
 * requires seekable files. */

STR_T readstrfd(int fd)
{
    STR_T ret;
    off_t size;

    ret.p = 0;
    size = lseek(fd, 0, SEEK_END);
    if (size < 0 || lseek(fd, 0, SEEK_SET)) { ret.l = -errno; return ret; }

    /* test whether it fits in ssize_t; return overflow if it doesn't,
     * which is same as lseek returns if size does not fit in off_t */
    if (size > (SIZE_MAX >> 1) - 1) { ret.l = -EOVERFLOW; return ret; }

    ret.p = (char *)malloc(size);
    if (!ret.p) { ret.l = -errno; return ret; }
    
    /* if the file shrunk after calling lseek, we give what we have */
    ret.l = read(fd, ret.p, size);
    if (ret.l > 0) return ret;

    if (ret.l < 0) ret.l = -errno;
    free(ret.p); 
    ret.p = 0;
    return ret;
}


/* Read an entire open file without seeking. We start using a buffer given
 * by guesslen and if that's full, we allocate a new one, copy the old
 * one and continue reading. The size of the new one is chosen such that 
 * after two reallocations, a memory hole exists that is big enough for the
 * third to fit in, without wasting too much space. This means the increase
 * factor i must be smaller than n in the golden rule 1/n = 1+n
 * (0.618033988746...) and as close to it as possible. We use i = .615384 or
 * 8/13, so that we can calculate trunc(l*i) by calculating l*8/(l*8+l*4+l).
 * The caller must to guess the correct order of magnitude. Guessing too high
 * wastes space; guessing too small may cause many reallocations. */

#define BUF_PCTINCREASE(l)	((l<<3)/((l<<3) + (l<<2) + l))

STR_T readstreof(int fd, ssize_t guesslen)
{
    STR_T ret;
    size_t rl;
    void *p;

    ret.p = (char *)malloc(guesslen);
    if (!ret.p) { ret.l = -errno; return ret; }

    ret.l = 0;			/* keeps amount we have read so far */
    for(;;) {
	rl = read(fd, ret.p + ret.l, guesslen - ret.l);
	if (rl < 0) break;
	ret.l += rl;
	if (ret.l < guesslen) { close(fd); return ret; }

	guesslen = BUF_PCTINCREASE(guesslen);
#ifdef NO_REALLOC
	p = malloc(guesslen);
	if (!p) break;
	memcpy(p, ret.p, ret.l);
	free(ret.p);
#else
	p = realloc(ret.p, guesslen);
	if (!p) break;
#endif
	ret.p = p;
    }

    ret.l = -errno;
    free(ret.p);
    ret.p = 0;
    return ret;
}


int strtoint(STRTOPROTO)
{
    int ret;
#include <strio/strtoord.inc>
}

long strtolong(STRTOPROTO)
{
    long ret;
#include <strio/strtoord.inc>
}

unsigned int strtouint(STRTOPROTO)
{
    unsigned int ret;
#include <strio/strtoord.inc>
}

unsigned long strtoulong(STRTOPROTO)
{
    unsigned long ret;
#include <strio/strtoord.inc>
}


/* For an ASCIIz full path name based on two str_ts, and return
 * length of directory component. Part of strio.c, because more platform 
 * dependent than str.c */

char *strtopath(char *path, ssize_t pathl, char *fname, ssize_t fnamel,
		ssize_t *retpathlen)
{
    char *ret, *s;

    ret = (char *)malloc(pathl + fnamel + 2);	/* separator and NUL added */
    if (!ret) return ret;

    /* Absolute fnames cause path to be ignored */
    if (fnamel && *fname == '/') pathl = 0;

    /* Start returned fname buffer with path; apply separator if needed */
    if (pathl) { 
	memcpy(ret, path, pathl); 
	if (path[pathl - 1] - '/') ret[pathl++] = '/';
    }

    /* Append given fnamename and zero terminate */
    pathl += cptstr(ret + pathl, fnamel + 1, fname, fnamel);

    /* Return length of directory component in full pathname */
    if (retpathlen && (s = memrchr(ret, '/', pathl))) *retpathlen = s - ret + 1;

    return ret;
}


/*   A note on variadic functions
 *
 *   The C99 stdarg specification has a flaw, in my opinion, in that it allows
 *   implementations to define va_list as containing a reference to the index
 *   variable instead of requiring va_list to contain the index variable
 *   itself. That creates the situation that a va_arg call can never be undone,
 *   except by the top level ellipsis function and only by rewinding
 *   completely. You can't save state, and you can't pass the index by value.
 *   The only workaround is va_copy, which is not available in older
 *   implementations and may copy the whole argument list instead of just a
 *   pointer, if a register calling convention is used combined with a simple
 *   implementation of va_arg.
 */


ssize_t formatstr(char *p, ssize_t l, int *skip, ...)
{
    ssize_t ret;
    va_list ap;

    va_start(ap, skip);
#if 0
    ret = vformatstr(p, l, skip, ap);
#else
    ret = 0;
#endif
    va_end(ap);
    return ret;
}


/* Indexes into parameter array. The ones for floating point conversion are
 * at the front, because then we can easily pass part of the array to the
 * conversion function. */

enum {
    P_FEXP,
    P_FEXPSTR,
    P_FEXPMIN,
    P_FEXPMULT,
    P_FSIGDIG,
    P_FPOINTCHR,
    PARAM_FLOATCOUNT,
    P_BASE = PARAM_FLOATCOUNT,
    P_ALIGN,
    P_ALDIST,
    P_ALCHR,
    P_ALSRCH,
    P_XFORM,
    P_UPPERCASE,
    P_PADSTRL,
    P_PADSTRR,
    P_TRUNCSTRL,
    P_TRUNCSTRR,
    P_TRUNCEXTL,
    P_TRUNCEXTR,
    PARAM_COUNT,
    P_NONE = PARAM_COUNT,
    P_UNKNOWN
};


/* Parameter constants */

#define PV_ALIGN_RIGHT	0
#define PV_ALIGN_LEFT	1
#define PV_XFORM_NONE	0
#define PV_XFORM_PRT	1
#define PV_XFORM_HEX	2
#define PV_XFORM_BER	3	/* set bit 7 on all bytes except last */
#define PV_XFORM_REV	4	/* reverse, used for integer conversion */
#define PV_ALSRCH_FIRST	0
#define PV_ALSRCH_LAST	1


/* Parameter types */

enum {
    PT_ORD,
    PT_CHR,
    PT_STR
};


/* Used member in val union */

enum {
    VT_I,
    VT_U,
    VT_F,
    VT_S,
};


#if 0
/* Declares auto-generated table containing number of digits required to
 * represent UINTMAX_MAX in bases 2-256. */

#include <evblib/strio/basetable.gen>


ssize_t vformatstr(char *buf, ssize_t bufl, int *skip, va_list ap)
{
    STR_T p[PARAM_COUNT], pname, pval;
    unsigned int wmin, wmax, taken, mustskip, pn, pt, vt;
    ssize_t ret, got;
    char *fmt, type, *s, *v, cvtbuf[2048];
    union {
	intmax_t i;	 /* C99 stdint.h type; define yourself if unavailable */
	uintmax_t u;	 /* C99 stdint.h type; define yourself if unavailable */
	long double f;
	STR_T s;
    } val;

    ret = 0;
    taken = mustskip = 0;
    if (skip) mustskip = *skip;

    for(;;) {
	/* Get format string */

	fmt = va_arg(ap, char *); 
	if (!fmt) break;

	/* Get field width spec. This all works correctly if *fmt == 0 too */

	wmin = strtouint(fmt, -1, &got, 10);
	fmt += got;
	if (*fmt == '.') wmax = strtouint(fmt, -1, &got, 10);

	/* Initialize formatting parameters. Defaults are for integer 
	 * conversion, because we have the most of those. */

	memset(p, 0, sizeof(p));
	p[P_BASE].l = 10;
	p[P_ALIGN].l = PV_ALIGN_RIGHT;
	P[P_XFORM].l = PV_XFORM_REV;

	/* Get argument type spec. We don't pull the va_arg yet because 
	   we may have parameters that need va_args first. We do set 
	   defaults for formatting parameters, where applicable, so that they
	   can be overridden by given parameters. */

	type = *fmt;
	if (type) fmt++;
	switch(type) {
	    case 'A':
	    case 'X': p[P_UPPERCASE].l++;
	    case 'a':
	    case 'x': p[P_BASE].l = 16; 
		      break;
	    case 'D': p[P_ALDIST].l += 2;
	    case 'F': p[P_ALDIST].l += 2; 
		      p[P_ALCHR].l = '.'; 
		      p[P_XFORM].l = PV_XFORM_NONE; 
		      break;
	    case 'S': 
	    case 's': 
	    case 'm': 
	    case 'M': p[P_ALIGN].l = PV_ALIGN_LEFT; 
		      p[P_XFORM].l = PV_XFORM_NONE;
	}

	/* Get formatting parameters */

	s = strchr(fmt, ';');	    /* introduces parameters */
	while(s) {
	    fmt = s + 1;
	    s = strchr(fmt, ';');

	    /* Get one formatting A/V pair */

	    pname.p = fmt;
	    pname.l = s ? s - fmt : strlen(fmt);
	    if (!pname.l) continue;

	    /* Split pair and determine parameter source
	     * (none if pval.l == 0, from va_list if pval.l < 0, 
	     * otherwise from string at pval.p, pval.l) */

	    pval.l = 0;
	    if (pname.p[pname.l - 1] == '*') { pname.l--; pval.l = -1; }
	    else if ((v = memchr(pname.p, '=', pname.l))) {
		pval.p = v + 1;
		pval.l = pname.l - (pval.p - pname.p);
		pname.l -= pval.l + 1;
	    }

	    /* Determine parameter to set in pn and parameter type in pt
	     * (n/a if pn is P_NONE) */

	    pt = PT_ORD;
	    pn = P_UNKNOWN;
	    switch(pname.l) {		

		/* Single letter parameters */
		case 1:	
		    pn = P_NONE;	/* default if not specified */
		    switch(*pname.p) {
			case 'v': p[P_XFORM].l = PV_XFORM_NONE;	/* clears rev */
			case 'n': p[P_BASE].l = 256; break;
			case 'p': p[P_XFORM].l = PV_XFORM_PRT; break;
			case 'X': p[P_UPPERCASE].l++;
			case 'x': p[P_XFORM].l = PV_XFORM_HEX; break;
			case '<': p[P_ALIGN].l = PV_ALIGN_LEFT; 
				  pt = PT_ORD; pn = P_ALDIST; break;
			case '>': p[P_ALIGN].l = PV_ALIGN_RIGHT; 
				  pt = PT_ORD; pn = P_ALDIST; break;
			case 'l': p[P_ALSRCH].l = PV_ALSRCH_LAST; 
			case 'f': pt = PT_CHR; pn = P_ALCHR; break;
			case 'B': p[P_UPPERCASE].l++;
			case 'b': pt = PT_ORD; pn = P_BASE; break;
			case 'd': pt = PT_ORD; pn = P_FSIGDIG; break;
			case 'e': p[P_FEXP].l++;
				  pt = PT_ORD; pn = P_FEXPMIN; break;
			case 'E': p[P_FEXP].l++;
				  pt = PT_STR; pn = P_FEXPSTR; break;
			case '.': pt = P_CHR; pn = P_FPOINTCHR; break;
			default:  pn = P_UNKNOWN;
				  
		    }
		    break;

		/* Two letter parameters */
		case 2:
		    /* lp, rp, lt, rt, le, re */
		    if (*pname.p == 'l' || *pname.p == 'r') {
			/* Note: the R parameters must follow the L ones in
			 * the array */
			switch(pname.p[1]) {
			    case 'p': pt = PT_STR; pn = P_PADSTRL; break;
			    case 'e': p[P_TRUNCEXTL + (*pname.p == 'r')].l++;
			    case 't': pt = PT_STR; pn = P_TRUNCSTRL; break;
			}
			if (pn != P_UNKNOWN) { 
			    pn += *pname.p == 'r'; break; 
			}
		    }
		    /* em */
		    if (pname.p[0] == 'e' && pname.p[1] == 'm') {
			p[P_FEXP].l++;
			pt = PT_ORD; pn = P_FEXPMULT; break;
		    }
		    break;

		/* Three letter parameters */
		case 3:
		    if (!memcmp(pname.p, "ber", 3)) {
			p[P_BASE].l = 128;
			p[P_XFORM].l = PV_XFORM_BER;
			pn = P_NONE;
		    }
		    break;
	    }

	    /* Now get the parameter value according to place and type */

	    if (pval.l == -1) {	    /* get it as-is from the va_list */
		switch(pt) {
		    case PT_STR: pval.p = va_arg(ap, char *);	/* no break */
		    default:     pval.l = va_arg(ap, int); break;
		}
	    }
	    else {		    /* or convert it from the format string */
		switch(pt) {
		    case PT_CHR: pval.l = *pval.p; break;
		    case PT_ORD: pval.l = strtoint(pval.p, pval.l, 0, 10);
		}
	    }

	    /* Now set the parameter and the bitmap, if we have one */

	    if (pn < 0 || pn > PARAM_COUNT) continue;
	    p[pn] = pval;
	}

	/* Now take value from va_list and set conversion type vt according to
	 * type specifier */

	vt = VT_I;	    /* Signed integral types */
	switch(type) {
	    case 'c': 
	    case 'h': 
	    case 'i':
	    case 'd': val.i = va_arg(ap, int);			goto typedone;
	    case 'l': val.i = va_arg(ap, int32_t);		goto typedone;
#ifdef HAVE_64BITDATA
	    case 'q': val.i = va_arg(ap, int64_t);		goto typedone;
#endif
	    case 'z': val.i = (intmax_t)va_arg(ap, ssize_t);	goto typedone;
	}
	vt = VT_U;	    /* Unsigned integral types */
	switch(type) {
	    case 'X': 
	    case 'x':
	    case 'C': 
	    case 'H':
	    case 'u': val.u = va_arg(ap, unsigned int);		goto typedone;
	    case 'L': val.u = va_arg(ap, uint32_t);		goto typedone;
#ifdef HAVE_64BITDATA
	    case 'Q': val.u = va_arg(ap, uint64_t));		goto typedone;
#endif
	    case 'a':
	    case 'A': val.u = (uintmax_t)va_arg(ap, void *);	goto typedone;
	    case 'Z': val.u = (uintmax_t)va_arg(ap, size_t);	goto typedone;
	}
#ifdef HAVE_FLOAT
	vt = VT_F;	    /* Floating point types */
	switch(type) {
	    case 'F': val.f = (long double)va_arg(ap, double);	goto typedone;
	    case 'D': val.f = va_arg(ap, long double);		goto typedone;
	}
#endif
	vt = VT_S;	    /* String types */
	val.s.l = -1;
	switch(type) {
	    case 's': val.s.p = va_arg(ap, char *);		goto typedone;
	    case 'S': val.s.p = va_arg(ap, char *);	     /* fallthrough */
		      val.s.l = va_arg(ap, ssize_t);		goto typedone;
	    case 'm': val.s.p = strerror(errno);		goto typedone;
	    case 'M': val.s.p = strerror(va_arg(ap, int));	goto typedone;
	    case '.': val.s.l = 0;				goto typedone;
	}

typedone:
	/* While skipping, loop here, as we've done all to stay in sync */

	if (mustskip) { mustskip--; continue; }

	/* We now have a value type in vt, a value in val, parameters in p
	 * and an output buffer at buf, buflen (modified as we progress). */

	/* Do type-agnostic input checking, limiting and fixups here. */

	if (wmax && wmin > wmax) wmin = wmax;	/* limit wmin to wmax, if any */
	if (wmin > buflen) goto abrt;		/* punt early, saves time */
	if (p[P_BASE] < 2) p[P_BASE] = 2;
	if (p[P_BASE] > 256) p[P_BASE] = 256;

	/* Convert to output buffer according to type. When not outputting
	 * strings, the maximum is 2 kB per field; an amount we can allocate
	 * on the stack without too much worries. Note that this destroys the
	 * original value. But we are now done with that anyway. */

	switch(vt) {
	    case VT_I: 
		val.s.l = _imaxtostr(cvtbuf, sizeof(cvtbuf), val.i, p[P_BASE]);
		val.s.p = cvtbuf;
		break;
	    case VT_U: 
		val.s.l = _umaxtostr(cvtbuf, sizeof(cvtbuf), val.u, p[P_BASE]);
		val.s.p = cvtbuf;
		break;
	    case VT_F:
		val.s.l = longdoubletostr(cvtbuf, sizeof(cvtbuf), val.f, 
					  &p, PARAM_FLOATCOUNT);
		val.s.p = cvtbuf;
		break;
	}

	/* Note: imaxtostr is _imaxtostr combined with memrevcpy; and we should
	 * template the i*tostr functions the same way as the strto* ones. */

	/* Now we have a string to be output in val.s. Perform transformation,
	 * see if it fits, and align it into output buffer according to
	 * parameters. */

    }

abrt:
    if (skip) *skip = taken;
    return ret;
}
#endif

    
/*
 * vim:softtabstop=4:sw=4
 */


/* formatout(1, "s", "Hello! I am ", "2d;le", 5, "s", " years old\n"); */
