/*
 * CHANNELS - Handles common operations on channels; also contains the code
 *            to process incoming data from them. For outgoing interface
 *            messages, we just use the generic meta_avtomsg() in metaops.c.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/10/29 - EvB - Created
 */

char channels_id[] = "CHANNELS - Copyright (C) 2001 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <malloc.h>	/* For malloc() / free() */
#include <string.h>	/* For memset() */

#include <channels.h>	/* Also includes srvtypes.h */
#include <jobs.h>	/* For job_toiface(), job_tochan(), job_fromchan() */
#include <language.h>	/* For VM. Also includes metaops for META_AV. */

#define DEBUGLEVEL 1	/* Include D1 statements */
#include <debug.h>


/*
 * FUNCTIONS
 */


/* Add an ASCII pair from a ring to an A/V item list - used by
   chan_handle_read below */

static void chan_ascmsgtoav(META_AV **head, META_AV **tail, 
			    CHAN *ch, ssize_t len)
{
	static char line[C_MAX_MSGSIZE + 16];
	char *atr, *c, *o, *end;
	META_AV *av, *aclav;
	META_VAL *v;
	ssize_t rl;

	/* Init AV for easier cleanup */
	av = 0;

	/* Take the indicated amount from the ring - we need to copy for
	   getitembyspec and getvalbyspec anyway. If those worked on rings,
	   we could do everything straight on the ring, but now it doesn't 
	   seem worth it to use prttoa, atoip and atoord directly on the 
	   ring, because we still need to copy the other elements. */

	/* Note that 'len' is already bound by proc_new and proc_msgavailable */
	ring_get(ch->proc->r, line, len); end = line + len;
	D1(msg(F_PROC, L_DEBUG, "- got part of ASCII message: '%s'\n", 
	       dbg_cvtstr(line, len)));

	/* Skip initial spaces and tabs */
	for(atr = line; atr < end && (*atr == ' ' || *atr == '\t'); atr++);

	/* End the attr at the first space, tab or equals sign */
	for(c = atr; c < end && *c != ' ' && *c != '\t' && *c != '='; c++);
	if (c >= end) goto ch_amtav_invpair;
	*c = 0;

	/* Allocate new A/V item at this point */
	av = (META_AV *)malloc(sizeof(META_AV));
	if (!av) { msg(F_MISC, L_ERR, "chan_ascmsgtoav: ERROR: No memory!\n"); 
		   return; }
	memset(av, 0, sizeof(META_AV));

	/* Find the specified item in the dictionary */
	av->i = meta_getitembyspec(ch->iface->c->m, atr);
	if (!av->i) { msg(F_PROC, L_NOTICE, "chan_ascmsgtoav: ERROR: Invalid "
					    "or unknown attribute '%s' "
					    "specified by '%s' (pid %d)!\n",
			  atr, ch->proc->argv[0], ch->proc->pid); 
		      free(av); return; }

	/* Skip all following spaces, tabs and equals signs (Jon P. says: 
	   be liberal in what you accept...). The value starts after this. */
	for(c++; c < end && (*c == ' ' || *c == '\t' || *c == '='); c++);

	/* Now parse the value. First check its style */
	if (ch->iface->flags & AVMSG_HEXVALUE) {

		/* We are doing hex values for all types */
		if (MT_ISORD(av->i->val_type)) {

			/* Ordinal type; do not allow empty values */
			if (c >= end) goto ch_amtav_invpair;
			av->l = meta_atoord(c, end - c, 0, 0, &rl, 16);
			if (!rl) goto ch_amtav_invpair;
		}
		else if (c < end) {

			/* Non-empty string type. Allocate half the number of 
			   input bytes, rounding up. Is always enough. */
			av->p = (char *)malloc((end - c + 1) >> 1);
			if (!av->p) { msg(F_MISC, L_ERR, "chan_ascmsgtoav: "
							 "ERROR: No memory!\n");
				      free(av); return; }

			/* Fill the string, byte by byte */
			for(o = av->p; end - c >= 2; c++, c++, o++) {
				*o = meta_atoord(c, 2, 0, 0, &rl, 16);
				if (rl != 2) break;
			}
			
			/* Set the length and flags */
			av->l = o - av->p;
			av->flags |= AV_FREE_P;
		}

		/* Check against ACL, add to list and return */
		goto ch_amtav_done;
	}

	/* No hex values; do not allow empty values */
	if (c >= end) goto ch_amtav_invpair;

	/* Handle according to found item's data type */
	switch(av->i->val_type) {
	  case MT_INTEGER:
	  case MT_DATE:
		av->l = meta_atoord(c, end - c, 0, 0, &rl, 10);
		if (rl) break;

		/* Try to find named constant */
		*end = 0;			/* allowed, see top */
		v = meta_getvalbyname(ch->iface->c->m, av->i, c);
		if (!v) goto ch_amtav_invpair;
		av->l = v->nr;
		break;

	  case MT_IPADDR:
		av->l = meta_atoip(c, end - c, 0, 0, &rl);
		if (!rl) goto ch_amtav_invpair;
		break;

	  default:
	  	/* Note: we allocate as many bytes as there are left in the
		   input message. This is safe, because prttoa can never 
		   output more data than we feed it, only less. Guaranteed. */
		av->p = (char *)malloc(end - c);
		if (!av->p) { msg(F_MISC, L_ERR, "chan_ascmsgtoav: ERROR: No "
						 "memory!\n"); 
			      free(av); return; }
		av->l = meta_prttoa(c, end - c, 0, 0, &rl, av->p, end - c);
		av->flags |= AV_FREE_P;
	}

ch_amtav_done:
	/* Check if we have a non-empty receive ACL */
	if (ch->iface->recvacl) {

		/* We do - find attribute in ACL */
		for(aclav = ch->iface->recvacl;
		    aclav && aclav->i != av->i;
		    aclav = aclav->next);
		    
		/* Return now if not found */
		if (!aclav) return;
	}

	/* Done; add pair to list and return. */
	meta_addav(head, tail, 0, 0, av);
	return;

ch_amtav_invpair:
	msg(F_PROC, L_NOTICE, "chan_ascmsgtoav: ERROR: Invalid AV pair '%s' "
			      "received from '%s' (pid %d)!\n", 
	    dbg_cvtstr(line,len), ch->proc->argv[0], ch->proc->pid);
	if (av) free(av);
	return;
}


/* Add all binary pairs from a ring to a list of A/V items - used by
   chan_handle_read() below */

static void chan_binmsgtoav(META_AV **head, META_AV **tail, 
			    CHAN *ch, ssize_t len)
{
	U_INT32_T binmsg[(C_MAX_MSGSIZE + 16) >> 2], *i, *e;
	META_ORD spcnr, vndnr, atrnr;
	ssize_t atrlen;
	META_SPC *spc;
	META_AV *av, *aclav;
	
	/* Note that 'len' is already bound by proc_new and proc_msgavailable */
	ring_get(ch->proc->r, (char *)binmsg, len);
	D1(msg(F_PROC, L_DEBUG, "- got binary message:\n")); 
	D1(if (msg_thresh[F_PROC] >= L_DEBUG) hexdump((char *)binmsg, len));

	/* Loop through attributes, starting at (char *)msg + 8 */
	e = binmsg + ((len + 3) >> 2);
	for(i = binmsg + 2; i < e; i += (atrlen + 3) >> 2) {

		/* Get values from attribute header. Warning: netint32 will
		   evaluate its argument 4 times if little-endian ;o) */
		spcnr = netint32(*i); i++;
		vndnr = netint32(*i); i++;
		atrnr = netint32(*i); i++;
		atrlen = netint32(*i); i++;
		msg(F_PROC, L_DEBUG, "- got spcnr %ld, vndnr %ld, atrnr %ld, "
				     "atrlen %ld\n",
		    (long)spcnr, (long)vndnr, (long)atrnr, (long)atrlen);

		spc = meta_getspcbynr(ch->iface->c->m, spcnr);
		if (!spc) { msg(F_PROC, L_NOTICE, "chan_binmsgtoav: ERROR: "
						  "Unknown space %d!\n", spcnr);
			    continue; }

		/* Allocate new A/V item at this point */
		av = (META_AV *)malloc(sizeof(META_AV));
		if (!av) { msg(F_MISC, L_ERR, "chan_binmsgtoav: ERROR: No "
					      "memory!\n"); return; }
		memset(av, 0, sizeof(META_AV));

		/* Find the item in the dictionary */
		av->i = meta_getitembynr(ch->iface->c->m, spc, atrnr, vndnr);
		if (!av->i) { msg(F_PROC, L_NOTICE, "chan_binmsgtoav: ERROR: "
						    "Unknown attribute %d, "
						    "vendor %d, space %d speci"
						    "fied by '%s' (pid %d)!\n",
				  atrnr, vndnr, spcnr, ch->proc->argv[0], 
				  ch->proc->pid); free(av); continue; }

		/* See if we have a non-empty receive ACL */
		if (ch->iface->recvacl) {

			/* We do - find attribute in ACL */
			for(aclav = ch->iface->recvacl;
			    aclav && aclav->i != av->i;
			    aclav = aclav->next);
		    
			/* Skip attribute if not found */
			if (!aclav) {
				msg(F_PROC, L_DEBUG, "  not in ACL; ignored\n");
				free(av);
				continue;
			}
		}

		/* Get data according to type */
		if (MT_ISORD(av->i->val_type)) {

			/* Ordinal - get value from even-multiple-of-4 sized
			   field and store in av->l */
			av->p = 0;
			av->l = getord((char *)i, (atrlen + 3) & ~3);
		}
		else {
			/* String - copy to new buf at av->p, length in av->l */
			av->p = (char *)malloc(atrlen);
			if (!av->p) { msg(F_MISC, L_ERR, "chan_binmsgtoav: "
							 "ERROR: No memory!\n");
				      free(av); return; }
			memcpy(av->p, (char *)i, atrlen);
			av->l = atrlen;
			av->flags |= AV_FREE_P;
		}

		/* Add A/V item to list */
		meta_printav(ch->iface->c->m, av, 0);
		meta_addav(head, tail, 0, 0, av);
	}
}


/* To be called when select says a channel's subprocess' pipe has data */

void chan_handle_read(CHAN *ch, time_t t)
{
	IFACE *i;
	JOB *j;
	ssize_t len;

	/* Transfer as many bytes to the ring as posssible; reset watchdog */
	proc_handle_read(ch->proc, t);

	/* Loop while we have jobs on the recv. queue and a (partial) message */
	while(ch->recvqt && proc_msgavailable(ch->proc, t, &len)) {

		if (ch->iface->flags & C_DV_FLAGS_ASCII) {
			
			/* ASCII: if partial message: decode and resume loop */
			if (len) {
				chan_ascmsgtoav(&(ch->recvqt->vm->head[VM_REP]),
						&(ch->recvqt->vm->tail[VM_REP]),
						ch, len);
				ring_discard(ch->proc->r, 1);
				continue;
			}

			/* if full message: skip LF */
			ring_discard(ch->proc->r, 1);
			msg(F_PROC, L_DEBUG, "- end of ASCII message - "
					     "continuing job.\n");
		}
		else {
			/* Decode message */
			chan_binmsgtoav(&(ch->recvqt->vm->head[VM_REP]),
					&(ch->recvqt->vm->tail[VM_REP]),
					ch, len);
			ch->proc->expectedlen = -1;		/* ack */
			msg(F_PROC, L_DEBUG, "- end of binary message - "
					     "continuing job.\n");
		}

		/* We have a full answer for the topmost job. Take it from the
		   queue and run it until it ends or makes a new iface call. */
		j = job_fromchan(ch); 
		i = job_run(ch->iface->c, j); 
		if (i) job_toiface(i, j, t);
	}

	/* If there are no more jobs on recv. queue, clear receiving state */
	if (!ch->recvqt) {
		msg(F_PROC, L_DEBUG, "- no more jobs - cleared receiving "
				     "state\n");
		ch->proc->state &= ~PRS_RECEIVING;
		ch->proc->timer = 0;
	}

	/* No message yet or anymore */
	return;
}


/* To be called when select says a channel's subprocess' pipe has more room */

void chan_handle_write(CHAN *ch, time_t t)
{
	JOB *j;

	for(;;) {
		/* Write as many bytes as we can from the tx ring, set 
		   receiving state if we actually transfered anything, clear 
		   sending state if there's nothing more in the tx ring */
		proc_handle_write(ch->proc, t);

		/* See if we're done sending and the iface has more jobs */
		if (!(ch->proc->state & PRS_SENDING) && (j=ch->iface->sendqt)) {

			msg(F_PROC, L_DEBUG, "- channel now idle; taking job "
					     "from shared send queue.\n");

			/* It has. Take the job from the shared send queue */
			ch->iface->sendqt = j->next;
			if (ch->iface->sendqt) ch->iface->sendqt->prev = 0;
			else ch->iface->sendqh = 0;

			/* Give it to this channel and loop */
			job_tochan(ch, j, t);
			continue;
		}
		return;
	}
}
