/*
 * METATYPE - Defines value types for metadata items. Any common 
 *	      type-dependent code should be added here.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/03/17 - EvB - Created
 * 2001/05/11 - EvB - Added the IPAddr and UnixTime types
 * 2001/06/25 - EvB - Added a couple of generic conversion functions
 * 2001/10/16 - EvB - Added bare string support to prttoa: if first char
 * 		      isn't a quote, take chars up to first whitespace. This 
 * 		      is very convenient in the ASCII interface when using
 * 		      the ascfile module on legacy users files. It does not
 * 		      affect the language's syntax, because the compiler checks
 * 		      for a quote first before running prttoa at all.
 * 2001/11/21 - EvB - Fixed minus handling in meta_atoord; it used to
 * 		      support 0x-55 and not -0x55. That's now done properly.
 */

char metatype[] = "METATYPE - Copyright (C) 2001 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <stdlib.h>	/* for strtoul */
#include <string.h>	/* for memset */

#include <misc.h>	/* For MIN */
#include <metatype.h>


/*
 * GLOBALS
 */


/* Type names */

static char *typenames[MT_TYPE_CNT + 1] = {
	"String",
	"Integer",
	"IPAddr",
	"Date",
	"Unknown"
	};


/*
 * FUNCTIONS
 */


/* 
 * Type name lookup functions 
 */


char *meta_gettypebynr(int type)
{
	return typenames[type <= MT_TYPE_CNT ? type : MT_TYPE_CNT];
}


int meta_gettypebyname(char *name)
{
	int ret;

	for(ret = 0;
	    ret < MT_TYPE_CNT && strcmp(typenames[ret], name);
	    ret++);
	return ret;
}


/* 
 * Conversion functions. The goal of these is to be able to use the same
 * functions in all places where we need to deal with converting to and from
 * strings:
 *
 * - the dictionary options parser
 * - the expression compiler
 * - the virtual machine's conversion instructions
 * - the meta_avtomsg and meta_msgtoav functions (if ASCII interface is used)
 */


/* Convert an ordinal value to ASCII in a random base (2 <= base <= 36).
   Returns the number of bytes written to the output buffer (max. buflen) */

int meta_ordtoa(char *buf, ssize_t buflen, int minlen, int base, META_ORD ord)
{
	static char *cvtbl = "0123456789abcdefghijklmnopqrstuvwxyz";
	char tmp[128], *c;
	int ret;

	if (!buf || buflen <= 0 || minlen > 128 || minlen > buflen) return 0;
	if (!ord || base < 2 || base > 36) {
		if (minlen <= 0) { *buf = '0'; return 1; }
		memset(buf, '0', minlen); return minlen;
	}

	ret = 0;
	if (ord < 0) {
		/* Negatives are done like this instead of doing ord = -ord
		   first, because the largest negative can not be negated. */ 

		*buf++ = '-'; ret++; buflen--; 
		for(c = tmp; ord || minlen > 0; minlen--) 
			*c++ = cvtbl[-(ord % base)], ord /= base;
	}
	else {
		for(c = tmp; ord || minlen > 0; minlen--) 
			*c++ = cvtbl[ord % base], ord /= base;
	}

	/* Reverse resulting string */
	for( ; c > tmp && buflen; buflen--) 
		*buf++ = *--c, ret++;

	return ret;
}


/* Convert an IP address in host order to a dotted quad ASCII string. 
   Returns the number of bytes written to the output buffer (max. buflen) */

int meta_iptoa(char *buf, ssize_t buflen, META_ORD ord)
{
	int n, ret;

	n = meta_ordtoa(buf, buflen, 0, 10, (ord >> 24) & 0xff);
	buf += n; buflen -= n; ret = n; if (buflen <= 0) return ret;
	*buf++ = '.'; buflen--; ret++;

	n = meta_ordtoa(buf, buflen, 0, 10, (ord >> 16) & 0xff);
	buf += n; buflen -= n; ret += n; if (buflen <= 0) return ret;
	*buf++ = '.'; buflen--; ret++;

	n = meta_ordtoa(buf, buflen, 0, 10, (ord >> 8) & 0xff);
	buf += n; buflen -= n; ret += n; if (buflen <= 0) return ret;
	*buf++ = '.'; buflen--; ret++;

	n = meta_ordtoa(buf, buflen, 0, 10, ord & 0xff);
	buf += n; buflen -= n; ret += n; 

	return ret;
}


/*
 * Conversion functions from normal and ring-type strings to something else.
 * These are in here not because I like to reinvent the weel, but because I
 * need to be able to limit the number of bytes gotten and want to use the
 * same functions on rings as well. Eg. there's no such thing as strntol.
 *
 * Source is always given as buf, buflen (=maximum to get from buffer), pos
 * (=position relative to start of buffer), mod (=ring size of buffer). Pos
 * must already be limited to mod - 1. All of these functions can return the 
 * number of bytes gotten in *gotten.
 *
 * Destination: strings are given as buf, buflen (=maximum to write in buffer).
 * In this case, the function returns the number of bytes written. For ordinal
 * destinations the value itself is returned.
 * 
 */


/* Convenience macro for these functions */

#define ADV(n)		pos += (n); if (mod) pos %= mod; g += (n);	\
			if (g >= buflen) 

#define RET		{ if (gotten) (*gotten) = g; return ret; }

/* Convert string in random base to ordinal value */

META_ORD meta_atoord(char *buf, ssize_t buflen, ssize_t pos, ssize_t mod,
		     ssize_t *gotten, int base)
{
	META_ORD ret;
	ssize_t g;
	int neg, c;

	/* Init value and number of bytes gotten (which is updated by ADV) */
	ret = 0; g = 0; if (buflen <= 0) RET;

	/* Get first char to see if it's a negative */
	neg = 0; if (buf[pos] == '-') { neg = 1; ADV(1) RET; }

	/* Autodetect base */
	if (!base) {
		base = 10;
		if (buf[pos] == '0') {
			base = 8; ADV(1) RET;
			if (buf[pos] == 'x') { base = 16; ADV(1) RET; }
		}
	}

	/* Convert */
	for(;;) {
		c = buf[pos] & 0xff;
		c = (base <= 10 || ISDEC(c)) ? (c - '0') : (TOLWR(c) - 'a'+10);
		if (c < 0 || c >= base) break;
		ret = ret * base + c;
		ADV(1) break;
	}
	if (neg) ret = -ret;
	RET;
}


/* Convert dotted quad string to IP address in host order */

META_ORD meta_atoip(char *buf, ssize_t buflen, ssize_t pos, ssize_t mod,
		    ssize_t *gotten) 
{
	META_ORD tmp, ret;
	ssize_t g, n;

	/* The standard things */
	ret = 0; g = 0; if (buflen < 7) RET;

	tmp = meta_atoord(buf, 3, pos, mod, &n, 10); 
	if (!n || tmp < 0 || tmp > 255) RET; ret |= tmp << 24;
	ADV(n) {g=0; RET;} if (buf[pos] != '.') {g=0; RET;} ADV(1) {g=0; RET;}

	tmp = meta_atoord(buf, 3, pos, mod, &n, 10); 
	if (!n || tmp < 0 || tmp > 255) {g=0; RET;} ret |= tmp << 16;
	ADV(n) {g=0; RET;} if (buf[pos] != '.') {g=0; RET;} ADV(1) {g=0; RET;}

	tmp = meta_atoord(buf, MIN(3, buflen - g), pos, mod, &n, 10); 
	if (!n || tmp < 0 || tmp > 255) {g=0; RET;} ret |= tmp << 8;
	ADV(n) {g=0; RET;} if (buf[pos] != '.') {g=0; RET;} ADV(1) {g=0; RET;}

	tmp = meta_atoord(buf, MIN(3, buflen - g), pos, mod, &n, 10); 
	if (!n || tmp < 0 || tmp > 255) {g=0; RET;} ret |= tmp;
	ADV(n);
	RET;
}


/* Convert a string to a printable string */

ssize_t meta_atoprt(char *buf, ssize_t buflen, ssize_t pos, ssize_t mod,
		    ssize_t *gotten, int dblbkslash, char *dst, ssize_t dstlen)
{
	ssize_t ret, g, n;
	int c;

	/* The standard things */
	ret = 0; g = 0; if (buflen <= 0) RET;

	/* Write characters, encoding quotes and non-printable chars as \x1f */
	for(;;) {
		c = buf[pos];

		/* Normal char */
		if (ISPRT(c) && c != '\'' && c != '\"' && c != '\\') {
			dstlen--; if (dstlen < 0) RET;
			*dst++ = c; ret++;
			ADV(1) RET;
			continue;
		}

		/* Non-printable char */
		dstlen -= 4 + (dblbkslash != 0); if (dstlen < 0) RET;
		*dst++ = '\\'; if (dblbkslash) *dst++ = '\\', ret++;
		*dst++ = 'x'; ret += 2;
		n = meta_ordtoa(dst, 2, 2, 16, c & 0xff); dst += n; ret += n;
		ADV(1) RET;
	}
}


/* Convert quoted strings to normal ones. Supports \xff, \377, \n, \r, \\, \",
   and \' -style escapes. This is indeed a superset of atoprt's output. 
   Note that the quotes should still be present; the function stops when it
   finds a non-escaped quote or if it reaches the end of the input buffer.
   If the first character isn't a single or double quote, the function takes
   the whole rest of the input buffer. So be careful when you call this. */

ssize_t meta_prttoa(char *buf, ssize_t buflen, ssize_t pos, ssize_t mod,
		    ssize_t *gotten, char *dst, ssize_t dstlen)
{
	ssize_t ret, g, n;
	int quote, c;

	/* The standard things */
	ret = 0; g = 0; if (buflen <= 0) RET;
	
	/* Get the character at the beginning. If a quote char, stop
	   when a matching one is found, otherwise use the whole string. */
	quote = buf[pos];
	if (quote == '\'' || quote == '\"') { ADV(1) RET; }	/* Skip quote */
	else quote = 0;						/* No quote */

	for(;;) {
		/* Get character and test for quote if we're supposed to */
		c = buf[pos] & 0xff;
		if (quote && c == quote) { ADV(1); RET; } 	/* Fully done */

		/* Whatever happens, we output one char */
		dstlen--; if (dstlen < 0) RET;

		/* Normal char, so copy, skip and continue if we have more */
		if (c != '\\') { *dst++ = c; ret++; ADV(1) RET; continue; }

		/* Backslash, so skip that and see what we have */
		ADV(1) RET; c = buf[pos];

		if (c >= '0' && c <= '7') {
			*dst++ = meta_atoord(buf, MIN(buflen,3), pos,mod,&n,8);
			ret++; if (!n) RET; ADV(n) RET;
		}
		else if (c == 'x') {
			ADV(1) RET;
			*dst++ = meta_atoord(buf, MIN(buflen,2), pos,mod,&n,16);
			ret++; if (!n) RET; ADV(n) RET;
		}
		else if (c == 'n') { *dst++ = '\n'; ret++; ADV(1) RET; }
		else if (c == 'r') { *dst++ = '\r'; ret++; ADV(1) RET; }
		else { *dst++ = c; ret++; ADV(1) RET; }
	}
}


/*
 * Some derivates of the general conversion functions above
 */


/* Convert a decimal C string to an ordinal value, or META_ORD_ERR if fails */

META_ORD meta_dtoord(char *s)
{
	META_ORD ret;
	ssize_t l, n;

	ret = meta_atoord(s, l = strlen(s), 0, 0, &n, 0);
	if (n == l) return ret;

	return META_ORD_ERR;
}

