/*
 * INCLUDES & DEFINES
 */


#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <time.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <errno.h>

#include <termios.h>
#include <stdio.h>

#include <subprocs.h>
#include <debug.h>


/*
 * MAIN
 */

struct termios saved_ts;

int setraw()
{
	struct termios ts;

	if (tcgetattr(0, &ts)) { perror("tcgetattr"); return -1; }
	saved_ts = ts;
	ts.c_iflag = ICRNL;
	ts.c_cflag = CREAD+CS8+CLOCAL;
	cfsetispeed(&ts, B38400);
	cfsetospeed(&ts, B38400);
	ts.c_lflag = 0;
	ts.c_cc[VMIN] = 1;
	ts.c_cc[VTIME] = 0;
	if (tcsetattr(0, TCSANOW, &ts)) { perror("tcsetattr"); return -1; }

	return 0;
}


int setnorm()
{
	if (tcsetattr(0, TCSANOW, &saved_ts)) { perror("tcsetattr"); return -1;}

	return 0;
}


#define SP_PID	0
#define SP_RET	1
#define SP_SIG	1

int sigwfd = -1;

void sighandler(int sig)
{
	int packet[2] = {0, 0};

	if (sig == SIGCHLD) packet[SP_PID] = wait(&(packet[SP_RET])); 
	else packet[SP_SIG] = sig;

	if (sigwfd != -1) write(sigwfd, packet, sizeof(int) << 1);
	signal(sig, sighandler);
}


int main()
{
	PROC_SET *ps;
	PROC *p;
	int sigpkt[2], sigrfd, n, queue;
	fd_set rfds, wfds;
	struct timeval tv, *tvp;
	time_t t;
	char c;

	msg_setthresh(F_PROC, L_DEBUG);
	
	/* Create a procset */
	ps = (PROC_SET *)malloc(sizeof(PROC_SET));
	if (!ps) { 
		msg(F_MISC, L_ERR,"main: ERROR: Could not allocate procset!\n");
		return 1; 
	}
	memset(ps, 0, sizeof(PROC_SET));

	/* Create the unstarted subprocess and add to set */
	p = proc_new("newproccld_asc arg1 arg2 arg3", 0, 
		     C_DV_FLAGS_ASCII, 10, 0, 0, 0);
	if (!p) {
		msg(F_MISC, L_ERR, "main: ERROR: Could not define subproc!\n");
		free(ps); return 1;
	}
	p->next = ps->p; ps->p = p;

	/* Set raw terminal handling */
	setraw();

	/* Do signal handling */
	if (pipe(sigpkt) == -1 ||
	    fcntl(sigpkt[PIPE_R], F_SETFD, 1) == -1 ||
	    fcntl(sigpkt[PIPE_W], F_SETFD, 1) == -1 ||
	    fcntl(sigpkt[PIPE_W], F_SETFL, O_NONBLOCK) == -1) {

		msg(F_MISC, L_ERR, "main: ERROR: Could not create nonblocking "
				   "pipe!\n");
		free(ps); return 1;
	}
	sigwfd = sigpkt[PIPE_W]; sigrfd = sigpkt[PIPE_R];
	signal(SIGCHLD, sighandler); 
	signal(SIGHUP, sighandler); 
	signal(SIGINT, sighandler); 
	signal(SIGQUIT, SIG_IGN); 
	signal(SIGTERM, sighandler); 
	signal(SIGPIPE, SIG_IGN);

	/* Initialise initial event mask */
	FD_SET(0, &(ps->rfds));
	FD_SET(sigrfd, &(ps->rfds)); 
	ps->highestfd = sigrfd;
	ps->firsttimer = 0;
	for(p = ps->p; p; p = p->next) {
		if (p->state & PRS_SENDING) FD_SET(p->wfd, &(ps->wfds));
		if (p->state & PRS_RECEIVING) FD_SET(p->rfd, &(ps->rfds));
		ps->highestfd = MAX(ps->highestfd, MAX(p->wfd, p->rfd));

		if (p->timer && (p->timer < ps->firsttimer || !ps->firsttimer)){
			ps->firsttimer = p->timer;
		}
	}

	/* THE BIG LOOP */

	printf("Press 's' to start, 't/k' to term/kill, 'e' to end, 'w' to\n"
	       "send a message, or 'q' to quit.\n");
	t = 0;
	queue = 0;	/* simulated # of messages on channel's recv queue */
	for(;;) {

		/* Prepare fd sets and timeout for select */
		rfds = ps->rfds; wfds = ps->wfds; tvp = 0;
		if (ps->firsttimer) { 
			tv.tv_sec = ps->firsttimer - t; tv.tv_usec = 0; 
			if (tv.tv_sec < 0) tv.tv_sec = 0, tv.tv_usec = 1;
			tvp = &tv; 
		}

		/* Do the select */
		n = select(ps->highestfd + 1, &rfds, &wfds, 0, tvp);
		if (n == -1) { 
			if (errno == EINTR) continue;
			msg(F_MISC, L_ERR, "main: ERROR: select: %s!\n",
			    strerror(errno));
			break;
		}

		/* Get the time ourselves - don't rely on select updating tv */
		time(&t);

		/* First see if we got a keypress. If so, read it for later.*/
		c = 0;
		if (FD_ISSET(0, &rfds)) {
			read(0, &c, 1);
			if (c == 'q') break;
		}

		/* Next, see if we got a signal or terminated process */
		memset(sigpkt, 0, sizeof(int) << 1);
		if (FD_ISSET(sigrfd, &rfds)) {

			read(sigrfd, sigpkt, sizeof(int) << 1);
			if (!sigpkt[SP_PID]) {
				msg(F_MISC, L_NOTICE, "main: Signal %d receive"
						      "d.\n", sigpkt[SP_SIG]);
			}
		}

		/* Loop through all processes in the set, doing whatever is
		   necessary. This may seem a little wasteful, but a lot of
		   operations would be O(n) on this list anyway if we were to
		   try some things better. So, we may as well take advantage 
		   of it and use the most naive timer algorithm possible, 
		   which at least keeps (re)setting the timers cheap. We also
		   avoid tricks like indexes on FDs etc., because you then 
		   need to loop through the fd sets, and highestfd is probably 
		   even higher than the number of procs). Even using the number
		   of ready fds to skip things won't help much in this loop. */

		ps->firsttimer = 0;
		for(p = ps->p; p; p = p->next) {
			
			/* First see if the proc can send or receive some data.
			   Also clear the fds from the set, we'll set them 
			   later if we still have the same state. Do it now, 
			   because the fd's may be -1 after a handler call. */
			if (p->wfd != -1) {
				FD_CLR(p->wfd, &(ps->wfds)); 
				if (FD_ISSET(p->wfd, &wfds)) {
					proc_handle_write(p, t);

					/* If this would be for real: check
					   if there's anything on this proc's
					   channel's interface's shared queue 
					   and if we have room for it, take 
					   the job, moving it also to this
					   proc's channel's receive queue. */

					/* Before anybody puts someting on the
					   shared queue, he should check the
					   next RR proc's ring first, and the
					   others next. Put it on the ring on
					   which there's room, set that proc's
					   PRS_SENDING (as there's something in
					   the ring) and call proc_handle_write
					   to have a try before we select(). */

					/* If there's room on no ring at all,
					   put it on the shared queue instead;
					   the first proc that gets enough
					   room will take it. This does require
					   that it is checked first that the
					   message is smaller than RING_CAP. */
				}
			}
			if (p->rfd != -1) {
				FD_CLR(p->rfd, &(ps->rfds));
				if (FD_ISSET(p->rfd, &rfds)) {

					proc_handle_read(p, t);
#if 0 					/* FIXME! */
					while ((n = proc_msgavailable(p, t))) {
						printf("===Incoming msg:\n");
						ring_write(p->r, 1, 0, n); 
						ring_discard(p->r, 1);
						printf("\n");
						p->expectedlen = -1;
						queue--;
					}
#endif

					if (!queue) {
						p->state = PRS_IDLE;
						p->timer = 0;
					}
				}
			}
		
			/* Next, see if its timer expired */
			if (p->timer && t >= p->timer) proc_handle_timeout(p,t);

			/* Lastly, see if it died */
			if (p->pid && p->pid == sigpkt[SP_PID])
				proc_handle_end(p, t, sigpkt[SP_RET]);

			/* Temporarily, see if we need to do something
			   artificial to it ;o) */
			switch(c) {
			  case 's':
			  	if (!p->pid) proc_start(p, t);
				break;
			  case 't': case 'k':
			  	if (p->pid) 
					kill(p->pid, c=='t' ? SIGTERM :SIGKILL);
				break;
			  case 'e':
			  	if (p->pid) proc_stop(p, t);
				break;
			  case 'w':
				if (!p->pid) break;
				queue++;
				ring_put(p->w, "Hello!\n", 7);
				p->state = PRS_SENDING;
				p->timer = t + p->xfertimeout;
				printf("===Outgoing message put in ring.\n");
				proc_handle_write(p, t);
				break;
			}

			/* Now, update this proc's entries in the fd sets */
			if (p->wfd != -1 && (p->state & PRS_SENDING))
				FD_SET(p->wfd, &(ps->wfds));
			if (p->rfd != -1 && (p->state & PRS_RECEIVING))
				FD_SET(p->rfd, &(ps->rfds));
			ps->highestfd = MAX(ps->highestfd, MAX(p->wfd, p->rfd));

			/* And maintain the first expiring timer */
			if (p->timer && (p->timer < ps->firsttimer || 
					 !ps->firsttimer)) {
				ps->firsttimer = p->timer;
			}
		}

		printf("-state %d-time %ld-timeout %ld-frsttmr %ld-queue %d\n",
			ps->p->state, t, ps->p->timer, ps->firsttimer, queue);
	}

	/* Set terminal handling to normal again */
	setnorm();

	/* Close signalling pipe */
	n = sigwfd; sigwfd = -1; close(n); close(sigrfd);

	/* Free procs in set */
	while(ps->p) { 
		p = ps->p->next; 
		proc_del(ps->p); 
		ps->p = p; 
	}

	/* Free set itself */
	free(ps);

	return 0;
}

