/*
 * LANGDISASM - Quick & dirty stack machine disassembler
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/05/06 - EvB - Created
 */

char langdisasm_id[] = "LANGDISASM - Copyright (C) 2001 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */

#include <stdio.h>	/* For printf */

#include <language.h>

#define DEBUGLEVEL 0
#include <debug.h>	/* Just for dbg_cvtstr */


/*
 * FUNCTIONS
 */

void lang_disassemble(META *m, INSN *i, ssize_t bufsize)
{
	int n;
	INSN *e;

	e = (INSN *)((char *)i + bufsize);
	for(n = 0; i < e; n++, i++) {

		switch(i->op) {
		  case OP_HALT:
		  	printf("%6d: HALT\n", n);
			break;
		  case OP_ABORT:
		  	printf("%6d: ABORT\n", n);
			break;
		  case OP_NOP:
		  	printf("%6d: NOP\n", n); 
			break;
		  case OP_PUSHINT:
		  	printf("%6d: PUSHINT %ld (0x%lx)\n", 
				n, i->imm.ord, i->imm.ord);
			break;
		  case OP_PUSHSTR: 
		  	printf("%6d: PUSHSTR \"%s\"\n", 
				n, dbg_cvtstr((char *)(i+1), i->imm.d.str_len));
			n += i->imm.d.disp; i += i->imm.d.disp;
			break;
		  case OP_PUSHAV:
			printf("%6d: PUSHAV %s/%s/%s (%s",
				n,
				i->imm.i.item->spc->name, 
				meta_getvndbynr(m, i->imm.i.item->vnd)->name,
				i->imm.i.item->name,
				i->imm.i.flags & AV_FIRST ? "first" : "last");
			if (i->imm.i.flags & AV_USEREPVALID) {
				printf(" from %s)\n",
				    i->imm.i.flags & AV_USEREP ? "REP" : "REQ");
			}
			else {
				printf(" from %s)\n",
				    i->imm.i.flags & AV_USEREP ? "inverted default list for op" : "default list for op");
			}
			break;
		  case OP_POP:		printf("%6d: POP\n", n); break;
		  case OP_NEG:		printf("%6d: NEG\n", n); break;
		  case OP_NOT:		printf("%6d: NOT\n", n); break;
		  case OP_MUL:		printf("%6d: MUL\n", n); break;
		  case OP_DIV:		printf("%6d: DIV\n", n); break;
		  case OP_CIDRMASK:	printf("%6d: CIDRMASK\n", n); break;
		  case OP_MOD:		printf("%6d: MOD\n", n); break;
		  case OP_ADD:		printf("%6d: ADD\n", n); break;
		  case OP_SUB:		printf("%6d: SUB\n", n); break;
		  case OP_SHL:		printf("%6d: SHL\n", n); break;
		  case OP_SHR:		printf("%6d: SHR\n", n); break;
		  case OP_XOR:		printf("%6d: XOR\n", n); break;
		  case OP_AND:		printf("%6d: AND\n", n); break;
		  case OP_OR:		printf("%6d: OR\n", n); break;
		  case OP_GE:		printf("%6d: GE\n", n); break;
		  case OP_LE:		printf("%6d: LE\n", n); break;
		  case OP_GT:		printf("%6d: GT\n", n); break;
		  case OP_LT:		printf("%6d: LT\n", n); break;
		  case OP_EQ:		printf("%6d: EQ\n", n); break;
		  case OP_NE:		printf("%6d: NE\n", n); break;
		  case OP_BF:		printf("%6d: BF\n", n); break;
		  case OP_AF:		printf("%6d: AF\n", n); break;
		  case OP_BL:		printf("%6d: BL\n", n); break;
		  case OP_AL:		printf("%6d: AL\n", n); break;
		  case OP_FO:		printf("%6d: FO\n", n); break;
		  case OP_LO:		printf("%6d: LO\n", n); break;
		  case OP_MD5:		printf("%6d: MD5\n", n); break;
		  case OP_HEX:		printf("%6d: HEX\n", n); break;
		  case OP_XORSTR:	printf("%6d: XORSTR\n", n); break;
		  case OP_CONCAT:	printf("%6d: CONCAT\n", n); break;
		  case OP_GESTR:	printf("%6d: GESTR\n", n); break;
		  case OP_LESTR:	printf("%6d: LESTR\n", n); break;
		  case OP_GTSTR:	printf("%6d: GTSTR\n", n); break;
		  case OP_LTSTR:	printf("%6d: LTSTR\n", n); break;
		  case OP_EQSTR:	printf("%6d: EQSTR\n", n); break;
		  case OP_NESTR:	printf("%6d: NESTR\n", n); break;
		  case OP_NO:		printf("%6d: NO\n", n); break;
		  case OP_JMPZ:		printf("%6d: JMPZ %+d\n",
		  			       n, i->imm.d.disp); break;
		  case OP_JMPNZ:	printf("%6d: JMPNZ %+d\n",
		  			       n, i->imm.d.disp); break;
		  case OP_ADDAV:	printf("%6d: ADDAV\n", n); break;
		  case OP_REPLACEAV:	printf("%6d: REPLACEAV\n", n); break;
		  case OP_DELAV:	printf("%6d: DELAV\n", n); break;
		  case OP_DELALLAV:	printf("%6d: DELALLAV\n", n); break;
		  case OP_MOVEALLAV:	printf("%6d: MOVEALLAV\n", n); break;
		  case OP_CALLIFACE:	printf("%6d: CALLIFACE\n", n); break;
		  case OP_ORD2RAW:	printf("%6d: ORD2RAW\n", n); break;
		  case OP_ORD2DEC:	printf("%6d: ORD2DEC\n", n); break;
		  case OP_ORD2HEX:	printf("%6d: ORD2HEX\n", n); break;
		  case OP_ORD2DQ:	printf("%6d: ORD2DQ\n", n); break;
		  case OP_ORD2YMDHMS:	printf("%6d: ORD2YMDHMS\n", n); break;
		  case OP_ORD2DFMT:	printf("%6d: ORD2DFMT\n", n); break;
		  case OP_DEC2ORD:	printf("%6d: DEC2ORD\n", n); break;
		  case OP_HEX2ORD:	printf("%6d: HEX2ORD\n", n); break;
		  case OP_DQ2ORD:	printf("%6d: DQ2ORD\n", n); break;
		  case OP_YMDHMS2ORD:	printf("%6d: YMDHMS2ORD\n", n); break;
		  case OP_YMD2ORD:	printf("%6d: YMD2ORD\n", n); break;
		  case OP_HMS2ORD:	printf("%6d: HMS2ORD\n", n); break;
		  default:		printf("%6d: ???\n", n); break;
		}
	}
}

