/*
 * METADICT - Create metadata definitions from dictionary textfiles.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/03/17 - EvB - Created
 * 2001/04/17 - EvB - Added support for value definitions
 * 2001/09/13 - EvB - Added nodec and noenc options (don't decode, except for
 * 		      subspaces / don't encode). Does not affect interfaces.
 * 2004/03/23 - EvB - Made dictionary format backwards compatible with 
 *                    Livingston/Cistron/FreeRADIUS; completely rewritten.
 */

char metadict_id[] = "METADICT - Copyright (C) 2001,2004 Emile van Bergen.";

/* #define DEBUG_DICT 1 */

/*
 * INCLUDES & DEFINES
 */


#include <string.h>	/* memchr, memrchr, strerror */
#include <stdlib.h>	/* free */
#include <unistd.h>	/* open */
#include <fcntl.h>	/* O_RDONLY */
#include <errno.h>	/* errno, strerror */
#ifdef DEBUG_DICT
#include <stdio.h>
#endif

#include <metatype.h>	/* instead of metadata.h for str_atoord */
#include <stream.h>

#include <debug.h>


/*
 * CONSTANTS
 */

/* Option names - indexes must match constant values above */

static char *optnames[OPT_X_OPT_CNT] = {
	"len_adj",
	"val_size",	/* these have a integer arg */

	"atr_ofs",
	"atr_size",
	"vnd_ofs",
	"vnd_size",
	"len_ofs",
	"len_size",
	"val_ofs",	
	"encrypt",	/* these have a positive integer arg */

	"single",	
	"nodec",
	"noenc",
	"has_tag",
	"reject",
	"acctresp",	/* these have an optional integer arg, default 1 */

	"space",
	"subspace",
	"vendor",
	"val_type",	
	"attribute",	/* these have an integer or string arg */
};


/*
 * FUNCTIONA
 */

void apply_options(OPTS *dst, OPTS *src) 
{
    int n;

    for(n = 0; n < OPT_X_OPT_CNT; n++) 
	if (src->set & (1 << n)) dst->v[n] = src->v[n], dst->set |= 1 << n;
}

void print_options(OPTS *o)
{
#if DEBUG_DICT
    META_SPC *spc;
    META_ITEM *atr;
    int n;

    printf("           ");
    for(n = 0; n < OPT_X_OPT_CNT; n++) 
	if (o->set & (1 << n)) {
	    if (n < OPT_X_SPECIAL_START || n > OPT_X_SPECIAL_END) { printf("%s=%lu (%ld) ", optnames[n], (long)o->v[n], (long)o->v[n]); continue; }
	    printf("%s=", optnames[n]);
	    switch(n) {
		case OPT_SPACE:
		case OPT_SUBSPACE:
		    spc = (META_SPC *)o->v[n];
		    printf("%.*s (%ld) ", spc->namel, spc->name, spc->nr);
		    break;
		case OPT_VENDOR:
		case OPT_VAL_TYPE:
		    printf("(%ld) ", o->v[n]);
		    break;
		case OPT_ATTRIBUTE:
		    atr = (META_ITEM *)o->v[n];
		    printf("%.*s (%ld) ", atr->namel, atr->name, atr->nr);
		    break;
	    }
	}
    printf("\n");
    fflush(stdout);
#endif
}

/* search order if no = sign:
   bool optnames (cs) done, vendors(cs) cont, options (cs) done, types(lc)
   apply order:
   options last */
 
int parse_option(META *m, OPTS *opts, OPTS *o, STR str, int warn)
{
    META_ORD nr;
    META_VND *v;
    META_ITEM *a;
    OPTS *so;
    char *s;
    STR arg;
    int n;
    ssize_t nrl;

    /* printf("o=[%.*s] ", str.l, str.p); */
    s = memchr(str.p, '=', str.l);
    if (s) {
	arg.p = s + 1; 
	arg.l = str.p + str.l - s - 1;
	str.l = s - str.p;

	for(n = 0; 
	    n < OPT_X_OPT_CNT &&
	    (strlen(optnames[n]) != str.l ||
	     memcmp(optnames[n], str.p, str.l));
	    n++);
	if (n == OPT_X_OPT_CNT) { msg(F_DICT, L_NOTICE, "Warning: ignoring unknown option '%*.s'\n", str.l, str.p); return -1; }

	o->set |= 1 << n;
	nr = str_atoord(arg.p, arg.l, &nrl, 0);
	if (n < OPT_X_SPECIAL_START || n > OPT_X_SPECIAL_END) {
	    if (nrl != arg.l) { msg(F_DICT, L_ERR, "ERROR: need numeric value instead of '%.*s' for '%.*s'", arg.l, arg.p, str.l, str.p); return -1; }
	    o->v[n] = nr;
	    return 0;
	}
	switch(n) {
	    case OPT_SPACE:
	    case OPT_SUBSPACE:
		if ((o->v[n] = (META_ORD)(nrl == arg.l ? meta_getspcbynr(m, nr) : meta_getspcbynamestr(m, arg.p, arg.l)))) return 0;
		break;				/* not found */
	    case OPT_VENDOR:
		if (nrl == arg.l) { o->v[n] = nr; return 0; }
		v = meta_getvndbynamestr(m, arg.p, arg.l);
		if (v) { o->v[n] = v->nr; return 0; }
		break;				/* not found */
	    case OPT_VAL_TYPE:
		str_tolower(arg.p, arg.l);
		if ((o->v[n] = meta_gettypebynamestr(arg.p, arg.l)) != MT_TYPE_CNT) return 0;
		break;				/* not found */
	    case OPT_ATTRIBUTE:
		a = meta_getitembyspecstr(m, arg.p, arg.l);
		if (a) { o->v[n] = a->nr; return 0; }
		break;				/* not found */
	}
	/* Note that we may arrive here with the option bit set in o->set 
	   without having actually set a value in o->v if that's unavailable */
	msg(F_DICT, L_ERR, "ERROR: unknown %.*s '%.*s'!\n", str.l, str.p, arg.l, arg.p); 
	return -1;
    }

    /* No = sign. First search boolean optnames, if found, set and we're done */
    for(n = OPT_X_BOOL_START; 
	n <= OPT_X_BOOL_END &&
	(strlen(optnames[n]) != str.l ||
	 memcmp(optnames[n], str.p, str.l));
	n++);
    if (n <= OPT_X_BOOL_END) { o->v[n] = 1; o->set |= 1 << n; return 0; }

    /* Search vendors; set PEC, apply ANYVENDOR options and continue */
    v = meta_getvndbynamestr(m, str.p, str.l);
    if (v) {
	o->v[OPT_VENDOR] = v->nr;
	o->set |= 1 << OPT_VENDOR;
	for(so = opts;
	    so && (so->namel - 9 || memcmp(so->name, "anyvendor", 9));
	    so = so->next);
	if (so) apply_options(o, so);
    }

    /* Convert to lowercase */
    str_tolower(str.p, str.l);

    /* If no vendor was found, search types and continue */
    nr = MT_TYPE_CNT;
    if (!v) {
	nr = meta_gettypebynamestr(str.p, str.l);
	if (nr < MT_TYPE_CNT) {
	    o->v[OPT_VAL_TYPE] = nr; 
	    o->set = 1 << OPT_VAL_TYPE; 
	}
    }

    /* Finally, search options */
    for(so = opts;
	so && (so->namel - str.l || memcmp(so->name, str.p, str.l));
	so = so->next);
    if (so) { apply_options(o, so); return 0; }

    /* If not found anything and we were asked, then warn */
    if (warn && !v && nr == MT_TYPE_CNT) msg(F_DICT, L_ERR, "Warning: unknown option '%.*s' ignored\n", str.l, str.p);
    return 0;
}


/* statements -- bitmap for easily testing for multiple types */

#define ST_OPT		1
#define ST_SPC		2
#define ST_VND		4
#define ST_ATR		8
#define ST_VAL		16
#define ST_INC		32
#define ST_BEG		64

/* opts is a pointer to the head variable. When adding, we add 
   at the head by creating a new object, setting its next 
   pointer to the current head given by *opts, and setting the 
   head to the newly created object. This means we can only
   walk the list in an order opposite to the one in which we 
   added items. Which is fine, because we then never need to 
   replace items, only to add new ones, to make newer ones 
   override the old. */

int readdict(META *m, OPTS **opts, char *path, ssize_t pathl, 
             char *fname, ssize_t fnamel)
{
    STR line, stmt, name, vname, str;
    STRM *strm;
    OPTS *o, to, bo;
    META_ITEM *i;
    META_ORD nr;
    char pathbuf[PATH_MAX + 1], *s;
    ssize_t n;
    int fd, st;

    /* Assemble path and fname into pathbuf; set pathl to basename length */

    if (pathl && fnamel && *fname == '/') pathl = 0;
    else { pathl = cpstr(pathbuf, sizeof(pathbuf) - 2, path, pathl); 
	   if (path[pathl - 1] - '/') pathbuf[pathl++] = '/'; }
    pathl += cptstr(pathbuf + pathl, sizeof(pathbuf) - pathl, fname, fnamel);
    s = memrchr(pathbuf, '/', pathl);
    pathl = 0; if (s) pathl = s - pathbuf + 1;

    fd = open(pathbuf, O_RDONLY); 
    if (fd == -1) { msg(F_DICT, L_ERR, "Could not open '%s': %s!\n", pathbuf, strerror(errno)); return -1; }

    strm = strm_new(fd, 256); 
    if (!strm) { msg(F_DICT, L_ERR, "Could not create stream!\n"); close(fd); return -1; }

    msg(F_DICT, L_NOTICE, "Reading '%s'\n", pathbuf);
    memset(&bo, 0, sizeof(bo));
    while(line = strm_getline(strm), line.l) {
cont_line:
	msg(F_DICT, L_DEBUG, "%04d: %.*s", strm->aux, line.l, line.p);

	/* Get statement (first word) */

	stmt = str_getword(&line, "\r\n\t ", 4);
	if (!stmt.l || *stmt.p == '#') continue;
	str_toupper(stmt.p, stmt.l);

	/* There's one statement without any arguments */

	if (stmt.l > 4 && !memcmp(stmt.p, "END-", 4)) {
	    if (bo.namel + 4 - stmt.l || memcmp(bo.name, stmt.p + 4, bo.namel)) { msg(F_DICT, L_NOTICE, "Note: %.*s does not match earlier BEGIN-%.*s but ending it anyway, in %s, line %d\n", stmt.l, stmt.p, bo.namel, bo.name, pathbuf, strm->aux); }
	    memset(&bo, 0, sizeof(bo));
	    continue;
	}

	/* All following statements have arguments. We tokenize first to
	   allow us to efficiently combine argument parsing. */

	if (stmt.l == 7 && !memcmp(stmt.p, "$OPTION", stmt.l)) st = ST_OPT;
	else if (stmt.l == 5 && !memcmp(stmt.p, "SPACE", stmt.l)) st = ST_SPC;
	else if (stmt.l == 5 && !memcmp(stmt.p, "VALUE", stmt.l)) st = ST_VAL;
	else if (stmt.l == 6 && !memcmp(stmt.p, "VENDOR", stmt.l)) st = ST_VND;
	else if ((stmt.l == 9 && !memcmp(stmt.p, "ATTRIBUTE", stmt.l)) ||
	    	 (stmt.l > 7  && !memcmp(stmt.p, "ATTRIB_", 7))) st = ST_ATR;
	else if (stmt.l > 6 && !memcmp(stmt.p, "BEGIN-", 6)) st = ST_BEG;
	else if (stmt.l == 8 && !memcmp(stmt.p, "$INCLUDE", stmt.l)) st=ST_INC;
	else { msg(F_DICT, L_NOTICE, "Warning: ignoring unknown statement '%.*s' in %s, line %d\n", stmt.l, stmt.p, pathbuf, strm->aux); continue; }

	/* Except for ST_BEG, all statements have a name as first argument.
	   Only ST_VAL has another name following it. For SPC, VAL, VND and
	   ATR, a number follows. Only OPT, SPC, ATR and BEG have options. */

	if (!(st & ST_BEG)) {					/* name */
	    name = str_getword(&line, "\r\n\t ", 4);
	    if (!name.l || *name.p == '#') goto errarg;
	}
	if (st & ST_VAL) {					/* val. name */
	    vname = str_getword(&line, "\r\n\t ", 4);
	    if (!vname.l || *vname.p == '#') goto errarg;
	}
	if (st & (ST_SPC | ST_VAL | ST_VND | ST_ATR)) {		/* number */
	    str = str_getword(&line, "\r\n\t ", 4);
	    if (!str.l || *str.p == '#') goto errarg;
	    n = 0; nr = str_atoord(str.p, str.l, &n, 0);
	    if (n != str.l) goto errarg;
	}
	if (st & (ST_OPT | ST_SPC | ST_ATR | ST_BEG)) {		/* options */
	    /* options are created as follows: 1. apply statement's options,
	       2. apply BEGIN- options, 3. apply immediate options. */
	    memset(&to, 0, sizeof(to));
	    if (st & (ST_SPC | ST_ATR)) parse_option(m, *opts, &to, stmt, 0);
	    if (st & ST_ATR) apply_options(&to, &bo);
	    n = 0;
	    do {
		while(str = str_getword(&line, "\r\n\t, ", 5),
		      str.l && *str.p != '#') {
		    n |= parse_option(m, *opts, &to, str, 1);
		}
		line = strm_getline(strm);
	    }
	    while(line.l && (*line.p == ' ' || *line.p == '\t'));
	    if (n) goto errmsg;
	    if (!line.l) break;
	}

	/* We have the arguments, now handle the statement itself. Note that
	   it is vital that all statements that have options, end with a 
	   goto cont_line instead of a plain continue, as we've already read
	   the next line. */

	if (st & ST_VAL) {				/* add value */
	    msg(F_DICT, L_DEBUG, "  val: attribute=[%.*s] name=[%.*s] nr=%ld\n", name.l, name.p, vname.l, vname.p, (long)nr);
	    i = meta_getitembyspecstr(m, name.p, name.l);
	    if (!i) { msg(F_DICT, L_ERR, "ERROR: Value '%.*s' (%d) given for unknown attribute '%.*s'\n", vname.l, vname.p, (long)nr, name.l, name.p); goto errmsg; }
	    meta_addval(m, i, nr, vname.p, vname.l);
	    continue;
	}
	if (st & ST_ATR) {				/* add attribute */
	    msg(F_DICT, L_DEBUG, "  attr: name=[%.*s] nr=%ld\n", name.l, name.p, (long)nr);
	    print_options(&to);
	    if (!to.v[OPT_SPACE]) { msg(F_DICT, L_ERR, "ERROR: No space specified for attribute '%.*s'\n", name.l, name.p); goto errmsg; }
	    meta_additembyopt(m, (META_SPC *)to.v[OPT_SPACE], 
			      nr, name.p, name.l, &to);
	    goto cont_line; /* we had options so already have a line */
	}
	if (st & ST_VND) {				/* add vendor */
	    msg(F_DICT, L_DEBUG, "  vend: name=[%.*s] nr=%ld\n", name.l, name.p, (long)nr);
	    meta_addvnd(m, nr, name.p, name.l);
	    continue;
	}
	if (st & ST_SPC) {				/* add space */
	    msg(F_DICT, L_DEBUG, "  space: name=[%.*s] nr=%ld\n", name.l, name.p, (long)nr);
	    print_options(&to);
	    meta_addspc(m, nr, name.p, name.l, 
			to.v[OPT_ATR_OFS], to.v[OPT_ATR_SIZE], 
			to.v[OPT_VND_OFS], to.v[OPT_VND_SIZE], 
			to.v[OPT_SINGLE]);
	    goto cont_line; /* we had options so already have a line */
	}
	if (st & ST_INC) {				/* include file */
	    if (readdict(m, opts, pathbuf, pathl, name.p, name.l)) goto errinc;
	    msg(F_DICT, L_DEBUG, "Returning to '%s'\n", pathbuf);
	    continue;
	}
	if (st & ST_OPT) {				/* add options */
	    o = (OPTS *)safe_malloc(sizeof(OPTS));
	    memcpy(o, &to, sizeof(OPTS));
	    o->namel = cptstr(o->name, sizeof(o->name), name.p, name.l);
	    str_tolower(o->name, o->namel);
	    o->next = *opts;
	    *opts = o;
	    goto cont_line; /* we had options so already have a line */
	}
	if (st & ST_BEG) {
	    if (bo.namel) { msg(F_DICT, L_NOTICE, "Note: %.*s ends earlier BEGIN-%.*s, in %s, line %d\n", stmt.l, stmt.p, bo.namel, bo.name, pathbuf, strm->aux); }
	    bo = to;
	    bo.namel = cptstr(bo.name, sizeof(bo.name), stmt.p + 6, stmt.l - 6);
	    goto cont_line; /* we had options so already have a line */
	}
    }

    strm_del(strm);
    close(fd);
    return 0;

errarg:
    msg(F_DICT, L_ERR, "No valid name or number for %.*s!\n", stmt.l, stmt.p);
    goto errmsg;
errinc:
    msg(F_DICT, L_ERR, "  included from %s, line %d\n", pathbuf, strm->aux);
    goto err;
errmsg:
    msg(F_DICT, L_ERR, "  in %s, line %d\n", pathbuf, strm->aux);
err:
    strm_del(strm);
    close(fd);
    return -1;
}


META *meta_newfromdict(char *basepath) 
{
    META *m;
    OPTS *o, *no;
    int n;

    o = 0;
    m = meta_new(); if (!m) return 0;
    n = readdict(m, &o, basepath, strlen(basepath),
		 "dictionary.openradius", 21);
    while(o) { no = o->next; free(o); o = no; }
    if (!n) return m;

    meta_del(m);
    return 0;
}

