#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>	/* MALLOC */

#include <pagemirr.h>
#include <debug.h>

typedef struct {
    struct mirrpage mp;
    char *r, *w;
} RING;

#define ring_capacity(R) ((R)->mp.l - 1)
#define ring_isempty(R) ((R)->r == (R)->w)
#define ring_maxget(R) ((((R)->w + (R)->mp.l) - (R)->r) % (R)->mp.l)
#define ring_maxput(R) ((((R)->r + (R)->mp.l) - ((R)->w + 1)) % (R)->mp.l)

#define ring_empty(R) ((R)->r = (R)->w)
#define ring_get(R,len) \
	((R)->r = (R)->mp.p + (((R)->r - (R)->mp.p + (len)) % (R)->mp.l))
#define ring_put(R,len) \
	do { mirr_update(&(R)->mp, (R)->w - (R)->mp.p, (len)); \
	    (R)->w = (R)->mp.p + (((R)->w - (R)->mp.p + (len)) % (R)->mp.l); \
	} while(0)

RING *ring_new(size_t size)
{
    RING *ret;

    ret = (RING *)malloc(sizeof(RING)); if (!ret) return 0;
    memset(ret, 0, sizeof(RING));
    if (mirr_new(size, &ret->mp) == 0) { free(ret); return 0; }
    ret->r = ret->w = ret->mp.p;
    return ret;
}

void ring_del(RING *r)
{
    mirr_del(&r->mp);
    free(r);
}

/* Returns bytes transferred, 0 if EOF, -1 if error (errno set),
   or -2 if no ring capacity */

int ring_read(RING *r, int fd)
{
    size_t len;
    int ret;

    len = ring_maxput(r); 
    if (len) {
	ret = read(fd, r->w, len);
	if (ret > 0) ring_put(r, ret);
	return ret;
    }
    return -2;
}

/* Returns bytes transferred, -1 if error (errno set),
   or -2 if nothing in ring to write */

int ring_write(RING *r, int fd)
{
    size_t len;
    int ret;

    len = ring_maxget(r);
    if (len) {
	ret = write(fd, r->r, len);
	if (ret > 0) ring_get(r, ret);
	return ret;
    }
    return -2;
}

int main()
{
    struct mirrpage mp;
    int n;
    RING *r;

    if (mirr_new(32, &mp) == 0) return -1;

    strcpy(mp.p, "Hello World!");
    mirr_update(&mp, 0, 8);		    /* Partial update: "Hello Wo" */
    printf("%s\n", (char *)mp.p + mp.l);

    memset(mp.p, 0, mp.l << 1);
    strcpy(mp.p + mp.l - 2, "Hello World!\n");	    /* Wrap after "He" */
    mirr_update(&mp, mp.l - 2, 13);
    printf("1=%s\n2=%s\n", (char *)mp.p, (char *)mp.p + mp.l);

    mirr_del(&mp);

    r = ring_new(128);
    if (!r) { printf("ERROR: Could not allocate new style ring!\n"); return -1; }

/*    r->r = r->w = r->mp.p + r->mp.l - 13;	*/

    memcpy(r->w, "abcdefghijklmnopqrstuvwxyz", 26);
    ring_put(r, 26);

    printf("get(5): %s\n", dbg_cvtstr(r->r, 5));
    ring_get(r, 5);

    memcpy(r->w, "ABCDEFGHIJKLMNOPQRSTUVWXYZ", 26);
    ring_put(r, 26);

    printf("get(32): %s\n", dbg_cvtstr(r->r, 32));
    ring_get(r, 32);

    n = ring_maxget(r);
    printf("get(rest=%d): %s\n", n, dbg_cvtstr(r->r, n));
    ring_get(r, n);

    ring_del(r);
    return 0;
}
