/*
 * METATYPE - Defines value types for metadata items. Any common 
 *	      type-dependent code should be added here.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/03/17 - EvB - Created
 * 2001/05/11 - EvB - Added ISORD macro and the ordinal types other than INT
 * 2001/06/25 - EvB - Added a couple of generic conversion functions
 */


/*
 * INCLUDES & DEFINES
 */


#include <sys/types.h>	/* For ssize_t only */
#include <metadata.h>


/* Currently known type numbers */

#define MT_STRING	0
#define MT_INTEGER	1
#define MT_IPADDR	2
#define MT_DATE		3
#define MT_TYPE_CNT	4


/* Must be true for all types of which the data can be stored in a META_ORD */

#define MT_ISORD(mt)		((mt) >= 1 && (mt) <= 3)


/* Some character handling macros, also used by conversion functions. Warning:
 * evaluate arguments multiple times, hence explicitly shown as macros, with
 * uppercase names. */

#define ISPRT(c)	((c) >= 0x20 && (c) <= 0x7e)

#define ISUPR(c)	((c) >= 'A' && (c) <= 'Z')
#define ISLWR(c)	((c) >= 'a' && (c) <= 'z')

#define TOUPR(c)	(ISLWR(c) ? (c) & ~0x20 : (c))
#define TOLWR(c)	(ISUPR(c) ? (c) | 0x20 : (c))

#define ISDEC(c)	((c) >= '0' && (c) <= '9')

#define ISHEX(c)	(((c) >= 'a' && (c) <= 'f') ||		\
			 ((c) >= 'A' && (c) <= 'F') ||		\
			 ISDEC(c))


/*
 * PROTOTYPES
 */


/*
 * Type name lookup functions 
 */

char *meta_gettypebynr(int type);
int meta_gettypebyname(char *name);


/*
 * Conversion to ASCII
 */

/* Convert an ordinal value to ASCII in a random base (2 <= base <= 36).
   Returns the number of bytes written to the output buffer, max. buflen. */

int meta_ordtoa(char *buf, ssize_t buflen, int minlen, int base, META_ORD ord);

/* Convert an IP address in host order to a dotted quad ASCII string. 
   Returns the number of bytes written to the output buffer (max. buflen) */

int meta_iptoa(char *buf, ssize_t buflen, META_ORD ord);


/*
 * Conversion from ASCII 
 */

/* Convert string in random base to ordinal value */

META_ORD meta_atoord(char *buf, ssize_t buflen, ssize_t pos, ssize_t mod,
		     ssize_t *gotten, int base);

/* Convert dotted quad string to IP address in host order */

META_ORD meta_atoip(char *buf, ssize_t buflen, ssize_t pos, ssize_t mod,
		    ssize_t *gotten);

/* Convert a string to a printable string */

ssize_t meta_atoprt(char *buf, ssize_t buflen, ssize_t pos, ssize_t mod,
		    ssize_t *gotten, int dblbkslash, char *dst, ssize_t dstlen);

/* Convert quoted strings to normal ones. Supports \xff, \377, \n, \r, \\ and \"
   -style quotes. This is indeed a superset of atoprt's output. 
   Note that the quotes must still be present; the function stops when it
   finds a non-escaped quote or if it reaches the end of the input buffer. */

ssize_t meta_prttoa(char *buf, ssize_t buflen, ssize_t pos, ssize_t mod,
		    ssize_t *gotten, char *dst, ssize_t dstlen);


/*
 * Derived conversion functions
 */

/* Convert an ASCII number to an ordinal value, or META_ORD_ERR if it fails */

META_ORD meta_dtoord(char *s);

