/*
 * $Id: buildreq.c,v 1.6 1996/01/06 01:58:18 lf Exp $
 *
 * Copyright (C) 1995 Lars Fenneberg
 *
 * See the file COPYRIGHT for the respective terms and conditions. 
 * If the file is missing contact me at in5y050@public.uni-hamburg.de 
 * and I'll send you a copy.
 *
 */

#include <sys/file.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <time.h>
#include <syslog.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "radius.h"

unsigned char rc_get_seqnbr(void);

/*
 * Function: rc_buildreq
 *
 * Purpose: builds a skeleton RADIUS request using information from the
 * 	    config file.
 *
 */

void rc_buildreq(SEND_DATA *data, int code, char *server, unsigned short port,
		 int timeout, int retries)
{
	data->server = server;
	data->svc_port = port;
	data->seq_nbr = rc_get_seqnbr();
	data->timeout = timeout;
	data->retries = retries;
	data->code = code;
}

/*
 * Function: rc_guess_seqnbr
 *
 * Purpose: return a random sequence number
 *
 */

static unsigned char rc_guess_seqnbr(void)
{
	srandom((unsigned int)(time(NULL)+getpid()));
	return (unsigned char)(random() & UCHAR_MAX);
}

/*
 * Function: rc_get_seqnbr
 *
 * Purpose: generate a sequence number
 *
 */

unsigned char rc_get_seqnbr(void)
{
	FILE *sf;
	int tries = 1;
	int seq_nbr;
	
	if ((sf = fopen(conf_seqfile, "a+")) == NULL)
	{
		rc_log(LOG_ERR,"rc_get_seqnbr: couldn't open sequence file %s: %s", conf_seqfile, strerror(errno)); 
		fclose(sf); /* just in case */
		/* well, so guess a sequence number */
		return rc_guess_seqnbr(); 
	}	
	
	while (flock(fileno(sf), LOCK_EX|LOCK_NB) != 0)
	{
		if (errno != EWOULDBLOCK) {
			rc_log(LOG_ERR, "rc_get_seqnbr: flock failure: %s: %s", conf_seqfile, strerror(errno));
			fclose(sf);	
			return rc_guess_seqnbr();
		}
		tries++;
		if (tries <= 10) 
			usleep(700); 
		else 
			break;
	}
	
	if (tries > 10) {
		rc_log(LOG_ERR,"rc_get_seqnbr: couldn't get lock after %d tries: %s", tries-1, conf_seqfile);
		fclose(sf);
		return rc_guess_seqnbr();
	}
	
	rewind(sf);
	if (fscanf(sf, "%d", &seq_nbr) != 1) {
		rc_log(LOG_ERR,"rc_get_seqnbr: fscanf failure: %s: %s", conf_seqfile, strerror(errno));
		seq_nbr = rc_guess_seqnbr();
	}
	
	rewind(sf);
	ftruncate(fileno(sf),0);
	fprintf(sf,"%d\n", (seq_nbr+1) & UCHAR_MAX);
	
	fflush(sf); /* fflush because a process may read it between the flock and fclose */
	
	if (flock(fileno(sf), LOCK_UN) != 0)
		rc_log(LOG_ERR, "rc_get_seqnbr: couldn't release lock on %s: %s", conf_seqfile, strerror(errno));

	fclose(sf);

	return (unsigned char)seq_nbr;
}

/*
 * Function: rc_auth
 *
 * Purpose: Builds an authentication request for port id client_port
 *	    with the value_pairs send.
 *
 * Returns: received value_pairs in received, messages from the server in msg
 *	    and 0 on success, negative on failure as return value
 *
 */

int rc_auth(UINT4 client_port, VALUE_PAIR *send, VALUE_PAIR **received, 
	    char *msg)
{
	SEND_DATA       data;
	UINT4		client_id;
	int		result;
	int		i;
	
	data.send_pairs = send;
	data.receive_pairs = NULL;

	/*
	 * Fill in NAS-IP-Address
	 */
	 
	if ((client_id = rc_own_ipaddress()) == 0)
		return (ERROR_RC);
 
	rc_avpair_add(&(data.send_pairs), PW_NAS_IP_ADDRESS, &client_id); 	

	/*
	 * Fill in NAS-Port
	 */
	
	rc_avpair_add(&(data.send_pairs), PW_NAS_PORT, &client_port);

	result = ERROR_RC;
	for(i=0; (i<conf_numauth) && (result != OK_RC) && (result != BADRESP_RC)
		; i++)
	{
		if (data.receive_pairs != NULL) {
			rc_avpair_free(data.receive_pairs);
			data.receive_pairs = NULL;
		}
		rc_buildreq(&data, PW_ACCESS_REQUEST, conf_authserver[i], 
			    conf_authserver_port[i], conf_radius_timeout,
			    conf_radius_retries);
		result = rc_send_server (&data, msg);
	}

	*received = data.receive_pairs;

	return result;
}

/*
 * Function: rc_acct
 *
 * Purpose: Builds an accounting request for port id client_port
 *	    with the value_pairs send
 *
 * Remarks: NAS-IP-Address, NAS-Port and Acct-Delay-Time get filled
 *	    in by this function, the rest has to be supplied.
 */

int rc_acct(UINT4 client_port, VALUE_PAIR *send)
{
	SEND_DATA       data;
	VALUE_PAIR	*adt_vp;
	UINT4		client_id;
	int		result;
	time_t		start_time, dtime;
	char		msg[4096];
	int		i;
	
	data.send_pairs = send;
	data.receive_pairs = NULL;

	/*
	 * Fill in NAS-IP-Address
	 */
	 
	if ((client_id = rc_own_ipaddress()) == 0)
		return (ERROR_RC);
 
	rc_avpair_add(&(data.send_pairs), PW_NAS_IP_ADDRESS, &client_id); 	

        /*
         * Fill in NAS-Port
         */
                          
        rc_avpair_add(&(data.send_pairs), PW_NAS_PORT, &client_port);

	/*
	 * Fill in Acct-Delay-Time
	 */
	
	dtime = 0;
	adt_vp = rc_avpair_add(&(data.send_pairs), PW_ACCT_DELAY_TIME, &dtime);

	start_time = time(NULL); 
	result = ERROR_RC;
	for(i=0; (i<conf_numacct) && (result != OK_RC) && (result != BADRESP_RC)
		; i++)
	{
		if (data.receive_pairs != NULL) {
			rc_avpair_free(data.receive_pairs);
			data.receive_pairs = NULL;
		}
		rc_buildreq(&data, PW_ACCOUNTING_REQUEST, conf_acctserver[i], 
			    conf_acctserver_port[i], conf_radius_timeout,
			    conf_radius_retries);

		dtime = time(NULL) - start_time;
		rc_avpair_assign(adt_vp, &dtime);
		
		result = rc_send_server (&data, msg);
	}

	rc_avpair_free(data.receive_pairs);
	
	return result;
}

/*
 * Function: rc_check
 *
 * Purpose: ask the server hostname on the specified port for a
 *	    status message
 *
 */

int rc_check(char *host, unsigned short port)
{
	SEND_DATA       data;
	int		result;
	char		msg[4096];
	UINT4		client_id, service_type;
	
	data.send_pairs = data.receive_pairs = NULL;

	/*
	 * Fill in NAS-IP-Address, although it isn't neccessary
	 */
	 
	if ((client_id = rc_own_ipaddress()) == 0)
		return (ERROR_RC);
 
	rc_avpair_add(&(data.send_pairs), PW_NAS_IP_ADDRESS, &client_id); 	

	/*
	 * Fill in Service-Type
	 */
	
	service_type = PW_ADMINISTRATIVE_USER;
	rc_avpair_add(&(data.send_pairs), PW_SERVICE_TYPE, &service_type); 	

	rc_buildreq(&data, PW_STATUS_SERVER, host, port, conf_radius_timeout,
		    conf_radius_retries);
	result = rc_send_server (&data, msg);
	rc_log(LOG_INFO, msg);

	rc_avpair_free(data.receive_pairs);
	
	return result;
}

