/*
 * $Id: config.c,v 1.7 1996/01/06 01:58:18 lf Exp $
 *
 * Copyright (C) 1995 Lars Fenneberg
 *
 * See the file COPYRIGHT for the respective terms and conditions. 
 * If the file is missing contact me at in5y050@public.uni-hamburg.de 
 * and I'll send you a copy.
 *
 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <netdb.h>
#include <syslog.h>

#include "radius.h"

char *conf_authserver[SERVER_MAX];
char *conf_acctserver[SERVER_MAX];
int  conf_authserver_port[SERVER_MAX];
int  conf_acctserver_port[SERVER_MAX];
char *conf_clients;
char *conf_dictionary;
char *conf_default_realm;
int conf_auth_type;
int conf_radius_timeout;
int conf_radius_retries;
char *conf_login_local;
char *conf_login_radius;
int conf_login_tries;
char *conf_seqfile;
int conf_login_timeout;
char *conf_mapfile;

int conf_numauth;	/* number of authentication servers */
int conf_numacct;	/* number of accounting servers */

static int rc_test_config(char *);

/*
 * Function: rc_read_config
 *
 * Purpose: read the global config file
 * 
 * Returns: 0 on success, -1 when failure
 */

int rc_read_config(char *filename)
{
	FILE *configfd;
	char buffer[512], *p, *q;
	char expect[7];
	struct servent *svp;

	/* Initialize the config variables */
	conf_numauth = conf_numacct = 0;
	conf_clients = conf_dictionary = NULL;
	conf_default_realm = NULL;
	conf_authserver[0] = conf_acctserver[0] = NULL;
	conf_auth_type = AUTH_RADIUS_FST | AUTH_LOCAL_SND;
	conf_radius_timeout = 3;
	conf_radius_retries = 10;
	conf_login_local = conf_login_radius = NULL;
	conf_login_tries = 3;
	conf_seqfile = NULL;
	conf_login_timeout = 60;
	conf_mapfile = NULL;
	
	if ((configfd = fopen(filename,"r")) == NULL)
	{
		rc_log(LOG_ERR,"rc_read_config: can't open %s: %s", filename, strerror(errno));
		return (-1);
	}

#define SKIP(p)	while(*p && isspace(*p)) p++;

	while (fgets(buffer, sizeof(buffer), configfd) != NULL)
	{
		p = buffer;
		
		SKIP(p);
		
		if ((*p == '\n') || (*p == '#') || (*p == '\0'))
			continue;

		p[strlen(p)-1] = '\0';
		
		if (!strncmp(p, "clients", 7))
		{
			p += 7;
			SKIP(p);
			conf_clients = strdup(p);						
		}
		else if (!strncmp(p, "authserver", 10))
		{
			p += 10;
			SKIP(p);
			
			if ((q = strchr(p,':')) != NULL)
			{
				*q = '\0';
				q++;
				conf_authserver_port[conf_numauth] = atoi(q);
			}
			else
			{
				if ((svp = getservbyname ("radius", "udp")) == NULL)
				{
				  conf_authserver_port[conf_numacct] = PW_AUTH_UDP_PORT;
			 	}
			 	else
			 	{
				  conf_authserver_port[conf_numacct] = ntohs (svp->s_port);
				}
			}

			conf_authserver[conf_numauth++] = strdup(p);
		}
		else if (!strncmp(p, "acctserver", 10))
		{
			p += 10;
			SKIP(p);

			if ((q = strchr(p,':')) != NULL)
			{
				*q = '\0';
				q++;
				conf_acctserver_port[conf_numacct] = atoi(q);
			} 
			else
			{
				if ((svp = getservbyname ("radacct", "udp")) == NULL)
				{
				  conf_acctserver_port[conf_numacct] = PW_ACCT_UDP_PORT;
			 	}
			 	else
			 	{
				  conf_acctserver_port[conf_numacct] = ntohs (svp->s_port);
				}
			}

			conf_acctserver[conf_numacct++] = strdup(p);
		}
		else if (!strncmp(p, "dictionary", 10))
		{
			p += 10;
			SKIP(p);
			conf_dictionary = strdup(p);
		}
		else if (!strncmp(p, "default_realm", 13))
		{
			p += 13;
			SKIP(p);
			conf_default_realm = strdup(p);
		}
		else if (!strncmp(p, "auth_order", 10))
		{
			p += 10;
			SKIP(p);
			
			*expect = '\0';
			conf_auth_type = 0;
			if (!strncmp(p, "local", 5))
			{
				p += 5;
				SKIP(p);
				strcpy(expect, "radius");
				
				conf_auth_type = AUTH_LOCAL_FST;
			}
			else if (!strncmp(p, "radius", 6))
			{
				p += 6;
				SKIP(p);
				strcpy(expect, "local");
				
				conf_auth_type = AUTH_RADIUS_FST;
			} 
			else
			{
				rc_log(LOG_ERR,"%s: auth_order: unknown keyword", filename);
				return (-1); 
			}
			
			SKIP(p);
			
			if (*p != '\0')
			{
				if (*p != ',')
				{
					rc_log(LOG_ERR,"%s: auth_order: missing or misplaced comma\n", filename);
					return (-1);
				}
				else
				{
					p++;
					SKIP(p);
			
					if (strcmp(p, expect) == 0)
					{
						if (strcmp(p, "local") == 0)
							conf_auth_type = conf_auth_type | AUTH_LOCAL_SND;
						else if (strcmp(p, "radius") == 0)
							conf_auth_type = conf_auth_type | AUTH_RADIUS_SND;   
					}
					else
					{
						rc_log(LOG_ERR,"%s: auth_order: unknown keyword", filename);
						return (-1); 
					}
				}
			}
		}
		else if (!strncmp(p, "login_local", 11))
		{
			p += 11;
			SKIP(p);
			conf_login_local = strdup(p);						
		}
		else if (!strncmp(p, "login_radius", 12))
		{
			p += 12;
			SKIP(p);
			conf_login_radius = strdup(p);						
		}
		else if (!strncmp(p, "radius_timeout", 14))
		{
			p += 14;
			SKIP(p);
			conf_radius_timeout = atoi(p);
		}
		else if (!strncmp(p, "radius_retries", 14))
		{
			p += 14;
			SKIP(p);
			conf_radius_retries = atoi(p);
		}
		else if (!strncmp(p, "login_tries", 11))
		{
			p += 11;
			SKIP(p);
			conf_login_tries = atoi(p);
		}
		else if (!strncmp(p, "seqfile", 7))
		{
			p += 7;
			SKIP(p);
			conf_seqfile = strdup(p);						
		}
		else if (!strncmp(p, "login_timeout", 13))
		{
			p += 13;
			SKIP(p);
			conf_login_timeout = atoi(p);						
		}
		else if (!strncmp(p, "mapfile", 7))
		{
			p += 7;
			SKIP(p);
			conf_mapfile = strdup(p);						
		}
		else
		{
			rc_log(LOG_ERR,"%s: unknown line: %s", filename, p);
			return (-1);
		}
	}
	fclose(configfd);

#undef SKIP
	
	return rc_test_config(filename);
}

/*
 * Function: rc_test_config
 *
 * Purpose: test the configuration the user supplied
 *
 * Returns: 0 on success, -1 when failure
 */

static int rc_test_config(char *filename)
{
	struct stat st;

	if (!conf_numauth)
	{
		rc_log(LOG_ERR,"%s: no authserver specified", filename);
		return (-1);
	}
	if (!conf_numacct)
	{
		rc_log(LOG_ERR,"%s: no acctserver specified", filename);
		return (-1);
	}
	if (!conf_clients)
	{
		rc_log(LOG_ERR,"%s: no clients file specified", filename);
		return (-1);
	}                                                                         
	if (!conf_dictionary)
	{
		rc_log(LOG_ERR,"%s: no dictionary specified", filename);
		return (-1);
	}

	if (conf_radius_timeout <= 0)
	{
		rc_log(LOG_ERR,"%s: radius_timeout <= 0 is illegal", filename);
		return (-1);
	}
	if (conf_radius_retries <= 0) 
	{
		rc_log(LOG_ERR,"%s: radius_retries <= 0 is illegal", filename);
		return (-1);
	}	
	if (stat(conf_login_local, &st) == 0)
	{
		if (!S_ISREG(st.st_mode)) {
			rc_log(LOG_ERR,"%s: not a regular file: %s", filename, conf_login_local);
			return (-1);
		}
	} else {
		rc_log(LOG_ERR,"%s: file not found: %s", filename, conf_login_local);
		return (-1);
	}
	if (stat(conf_login_radius, &st) == 0)
	{
		if (!S_ISREG(st.st_mode)) {
			rc_log(LOG_ERR,"%s: not a regular file: %s", filename, conf_login_radius);
			return (-1);
		}
	} else {
		rc_log(LOG_ERR,"%s: file not found: %s", filename, conf_login_radius);
		return (-1);
	}
	if (conf_login_tries <= 0)
	{
		rc_log(LOG_ERR,"%s: login_tries <= 0 is illegal", filename);
		return (-1);
	}	
	if (conf_seqfile == NULL)
	{
		rc_log(LOG_ERR,"%s: seqfile not specified", filename);
		return (-1);
	}	
	if (conf_login_timeout <= 0)
	{
		rc_log(LOG_ERR,"%s: login_timeout <= 0 is illegal", filename);
		return (-1);
	}	
	if (conf_mapfile == NULL)
	{
		rc_log(LOG_ERR,"%s: mapfile not specified", filename);
		return (-1);
	}	

	return 0;
}

/*
 * Function: rc_find_match
 *
 * Purpose: see if ip_addr is one of the ip addresses of hostname
 *
 * Returns: 0 on success, -1 when failure
 *
 */

static int rc_find_match (UINT4 *ip_addr, char *hostname)
{
	UINT4           addr;
	char          **paddr;
	struct hostent *hp;

	if (rc_good_ipaddr (hostname) == 0)
	{
		if (*ip_addr == ntohl(inet_addr (hostname)))
		{
			return (0);
		}
	}
	else
	{
		if ((hp = gethostbyname (hostname)) == (struct hostent *) NULL)
		{
			return (-1);
		}
		for (paddr = hp->h_addr_list; *paddr; paddr++)
		{
			addr = ** (UINT4 **) paddr;
			if (ntohl(addr) == *ip_addr)
			{
				return (0);
			}
		}
	}
	return (-1);
} 

/*
 * Function: rc_find_server
 *
 * Purpose: search a server in the clients file
 *
 * Returns: 0 on success, -1 on failure
 *
 */
 
int rc_find_server (char *server_name, UINT4 *ip_addr, char *secret)
{
	UINT4    	myipaddr = 0;
	int             len;
	int             result;
	FILE           *clientfd;
	char           *h;
	char           *s;
	char           *host2;
	char            buffer[128];
	char            hostnm[AUTH_ID_LEN + 1];

	/* Get the IP address of the authentication server */
	if ((*ip_addr = rc_get_ipaddr (server_name)) == (UINT4) 0)
	{
		rc_log(LOG_ERR, "rc_find_server: no such RADIUS server: %s", server_name);
		return (-1);
	}
	if ((clientfd = fopen (conf_clients, "r")) == (FILE *) NULL)
	{
		rc_log(LOG_ERR, "rc_find_server: couldn't open file: %s: %s", strerror(errno), conf_clients);
		return (-1);
	}
	
	if ((myipaddr = rc_own_ipaddress()) == 0)
		return (-1);

	result = 0;
	while (fgets (buffer, sizeof (buffer), clientfd) != (char *) NULL)
	{
		if (*buffer == '#')
		{
			continue;
		}

		if ((h = strtok (buffer, " \t\n")) == NULL) /* first hostname */
		{
			continue;
		}

		memset (hostnm, '\0', AUTH_ID_LEN);
		len = strlen (h);
		if (len > AUTH_ID_LEN)
		{
			len = AUTH_ID_LEN;
		}
		strncpy (hostnm, h, len);
		hostnm[AUTH_ID_LEN] = '\0';

		if ((s = strtok (NULL, " \t\n")) == NULL) /* and secret field */
		{
			continue;
		}

		memset (secret, '\0', MAX_SECRET_LENGTH);
		len = strlen (s);
		if (len > MAX_SECRET_LENGTH)
		{
			len = MAX_SECRET_LENGTH;
		}
		strncpy (secret, s, len);
		secret[MAX_SECRET_LENGTH] = '\0';

		if (!strchr (hostnm, '/')) /* If single name form */
		{
			if (rc_find_match (ip_addr, hostnm) == 0)
			{
				result++;
				break;
			}
		}
		else /* <name1>/<name2> "paired" form */
		{
			strtok (hostnm, "/");
			if (rc_find_match (&myipaddr, hostnm) == 0)
			{	     /* If we're the 1st name, target is 2nd */
				host2 = strtok (NULL, " ");
				if (rc_find_match (ip_addr, host2) == 0)
				{
					result++;
					break;
				}
			}
			else	/* If we were 2nd name, target is 1st name */
			{
				if (rc_find_match (ip_addr, hostnm) == 0)
				{
					result++;
					break;
				}
			}
		}
	}
	fclose (clientfd);
	if (result == 0)
	{
		memset (buffer, '\0', sizeof (buffer));
		memset (secret, '\0', sizeof (secret));
		rc_log(LOG_ERR, "rc_find_server: couldn't find RADIUS server %s in %s",
			 server_name, conf_clients);
		return (-1);
	}
	return 0;
}
