/*
 * $Id: radlogin.c,v 1.12 1996/03/15 13:53:04 lf Exp $
 *
 * Copyright (C) 1995 Lars Fenneberg
 *
 * See the file COPYRIGHT for the respective terms and conditions. 
 * If the file is missing contact me at in5y050@public.uni-hamburg.de 
 * and I'll send you a copy.
 *
 */

static char	rcsid[] =
		"$Id: radlogin.c,v 1.12 1996/03/15 13:53:04 lf Exp $";

static char	copyright_1[] =
		"$Copyright: (C) 1995 Lars Fenneberg $";

static char	copyright_2[] =
		"$Parts: (C) 1992 Livingston Enterprises, Inc. $";

static char	copyright_3[] =
		"$Parts: (C) 1992,1993, 1994,1995 The Regents of the \
University of Michigan and Merit Network, Inc. All Rights Reserved $";

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/param.h>
#include	<netinet/in.h>
#include	<sys/time.h>
#include	<sys/signal.h>

#include	<stdio.h>
#include	<netdb.h>
#include	<unistd.h>
#include	<pwd.h>
#include	<string.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<syslog.h>

#ifdef SHADOW_PASSWORD
#include <shadow.h>
#endif

#include	"radius.h"

static char *pname = NULL;
static struct env *env = NULL;

void alarm_handler(int sn)
{
	rc_log(LOG_ERR,"\r\nlogin timed out after %d seconds. Bye.", conf_login_timeout);
	sleep(1);
	exit(1);	
}

int auth_radius(UINT4 client_port, char *username, char *passwd)
{

	VALUE_PAIR 	*send, *received, *vp;
	UINT4		service, ftype, ctype;
	char 		msg[4096], *p, username_realm[256];
	char            name[2048], value[2048]; /* more than enough */
	int 		result;

	send = NULL;

	/*
	 * Determine and fill in Service-Type
	 */
	
#ifdef SCP
	/* determine based on the username what kind of service is requested.
	   this allows you to use one password for all accounts, but the
	   Merit radiusd supplies you just with the right information you
	   need for the specified service type	-lf, 03/15/96 */
	 
	switch (*username)
	{
		case 'S':
				service = PW_FRAMED;
				ftype = PW_SLIP;
				ctype = 0;
				username++;
				break;
		case 'C':
				service = PW_FRAMED;
				ftype = PW_SLIP;
				ctype = PW_VAN_JACOBSON_TCP_IP;
				username++;
				break;
		case 'P':
				service = PW_FRAMED;
				ftype = PW_PPP;
				ctype = 0;
				username++;
				break;
		default:
				service = PW_LOGIN;
				ftype = 0;
				ctype = 0;
				break;
	}
#else
	service = PW_LOGIN;	
	ftype = 0;
	ctype = 0;
#endif

	rc_avpair_add(&send, PW_SERVICE_TYPE, &service);

	if (ftype != 0)
	{
		rc_avpair_add(&send, PW_FRAMED_PROTOCOL, &ftype);	
	} 
	 
	 
	/*
	 * Fill in User-Name
	 */

	 strncpy(username_realm, username, sizeof(username_realm));

	 /* Append default realm */
	 if ((strchr(username_realm, '@') == NULL) && conf_default_realm &&
	     (*conf_default_realm != '\0'))
	 {
		strncat(username_realm, "@", sizeof(username_realm));
		strncat(username_realm, conf_default_realm, sizeof(username_realm));
	 } 

	rc_avpair_add(&send, PW_USER_NAME, username_realm);
	
	/*
	 * Fill in User-Password
	 */
	 
	rc_avpair_add(&send, PW_USER_PASSWORD, passwd); 	  

	
	result = rc_auth(client_port, send, &received, msg);
	
	if (result == OK_RC)
	{
		rc_add_env(env, "RADIUS_USER_NAME", username);
	
		vp = received;
		while (vp)
		{
			strcpy(name, "RADIUS_");
			rc_avpair_tostr(vp, name+7, sizeof(name)-7, value, sizeof(value));
			
			/* Translate "-" => "_" and uppercase*/
			for(p = name; *p; p++) {
				*p = toupper(*p);
				if (*p == '-') *p = '_';
			}
			
			if (rc_add_env(env, name, value) < 0)
			{
				rc_log(LOG_CRIT, "rc_add_env: FATAL: not enough space for environment (increase ENV_SIZE)");
				abort();
			}
					
			vp = vp->next;
		}

#ifdef SCP
		/* the RADIUS server doesnt distinguish between compressed
		   and non-compressed slip. so we explicitly set 
		   Framed-Compression here	  -lf, 03/15/96 */

		if ((ftype = PW_SLIP) && (ctype == PW_VAN_JACOBSON_TCP_IP))
			rc_add_env(env, "RADIUS_FRAMED_COMPRESSION", "Van-Jacobson-TCP-IP");

#endif
		rc_log(LOG_NOTICE, "\"%s\" RADIUS authentication OK", username);
	}
	else
	{
		rc_log(LOG_NOTICE, "\"%s\" RADIUS authentication failed (RC=%i)", username, result);
	}
	
	return result;
}

int auth_local(char *username, char *passwd)
{
	struct passwd	*pw;
	char		*xpasswd;
#ifdef SHADOW_PASSWORD
	struct spwd	*spw;
#endif

	if ((pw = getpwnam(username)) == NULL) {
		endpwent();
		rc_log(LOG_NOTICE, "\"%s\" local authentication failed", username);
		return BADRESP_RC;
	}
	endpwent();
	
#ifdef SHADOW_PASSWORD
        if((spw = getspnam(pw->pw_name)) == NULL) {
		endspent();
		rc_log (LOG_NOTICE, "\"%s\" local authentication failed", username);
		return BADRESP_RC;
        }
        else 
        { 
        	pw->pw_passwd = spw->sp_pwdp; 
        }
        endspent();
#endif /* SHADOW_PASSWORD */

	xpasswd = crypt(passwd, pw->pw_passwd);
	
	if (*pw->pw_passwd == '\0' || strcmp(xpasswd, pw->pw_passwd)) {
		rc_log(LOG_NOTICE, "\"%s\" local authentication failed", username);
		return BADRESP_RC;		
	}

	rc_log(LOG_NOTICE, "\"%s\" local authentication OK", username);
	return OK_RC;
}

void
local_login(char *username)
{
	/* login should spot this... but who knows what old /bin/logins
	 * may be still around
	 */
	if (*username == '-') {
		rc_log(LOG_WARNING, "username can't start with a dash");
		exit(ERROR_RC);
	}
	/* the new shadow login seems to require either a -r or a -h
	 * flag for -f to work (so source code, lmain.c) so we supply
	 * it here. shouldn't hurt on other systems,	-lf, 03/13/96
	 */
	execle(conf_login_local, conf_login_local, "-h", "localhost", "-f", username, NULL, env->env);
	rc_log(LOG_ERR, "couldn't execute %s: %s", conf_login_local, strerror(errno));
	sleep(1);	/* give the user time to read */
	exit(ERROR_RC);
}

void
radius_login(char *username)
{
	execle(conf_login_radius, conf_login_radius, NULL, env->env);
	rc_log(LOG_ERR, "couldn't execute %s: %s", conf_login_radius, strerror(errno));
	sleep(1);	/* give the user time to read */
	exit(ERROR_RC);
}

void usage(void)
{
	fprintf(stderr,"Usage: %s [-Vh] [-i <client_port>]\n", pname);
	fprintf(stderr,"  -i		ttyname to send to the server\n");
	fprintf(stderr,"  -V		output version information\n");
	fprintf(stderr,"  -h		output this text\n");     
        exit(ERROR_RC);
}

void version(void)
{
	fprintf(stderr,"%s: %s\n", pname ,rcsid);
	exit(ERROR_RC);
}

int
main (int argc, char **argv)
{
	int             result;
	char		username[128];
	char            passwd[AUTH_PASS_LEN + 1];
	int 		tries, remaining, c;
	UINT4		client_port;

        extern char *optarg;
        extern int optind;

	pname = (pname = strrchr(argv[0],'/'))?pname+1:argv[0];
	
	if (rc_read_config(RADIUSCLIENT_CONF) != 0)
		exit(ERROR_RC);
	
	if (rc_read_dictionary(conf_dictionary) != 0)
		exit (ERROR_RC);

	if (rc_read_mapfile(conf_mapfile) != 0)
		exit (ERROR_RC);

	client_port = rc_map2id(ttyname(1));

	while ((c = getopt(argc,argv,"i:hV")) > 0)
	{
		switch(c) {
			case 'i':
				client_port = rc_map2id(optarg);
				break;
			case 'V':
				version();
				break;
			case 'h':
				usage();
				break;
			default:
				exit(ERROR_RC);
				break;
		}
	}
	
	if ((argc - optind) == 1)
	{
		strncpy(username,argv[optind], sizeof(username));
	}
	else
	{
		*username = '\0';
	}

	if ((env = rc_new_env(ENV_SIZE)) == NULL)
	{
		rc_log(LOG_CRIT, "rc_new_env: FATAL: out of memory");
		abort();
	}
	
	if (rc_import_env(env,environ) < 0)
	{
		rc_log(LOG_CRIT, "rc_import_env: FATAL: not enough space for environment (increase ENV_SIZE)");
		abort();
	}

	signal(SIGALRM, alarm_handler);

	remaining = conf_login_timeout;
	tries = 1;
	while (tries <= conf_login_tries)
	{
	 alarm(remaining);

	 while (!*username) {
	 		strncpy(username, rc_getstr ("login: ",1), sizeof(username));
	 }
	 strncpy (passwd, rc_getstr("Password: ",0), sizeof (passwd));		

	 remaining = alarm(0);

 	 if (conf_auth_type & AUTH_LOCAL_FST)
 	 {
 	 	result = auth_local(username, passwd);
 	 	if (result != OK_RC) {
 	 		if (conf_auth_type & AUTH_RADIUS_SND) {
 	 			result = auth_radius(client_port, username, passwd);
 	 			if (result == OK_RC) {
					memset(passwd, '\0', sizeof(passwd));
 	 				radius_login(username); /* never returns */
 	 			}
 	 		}
 	 	} else {
			memset(passwd, '\0', sizeof(passwd));
 	 		local_login(username); /* never returns */
 	 	}
 	 }
 	 else
 	 {
		result = auth_radius(client_port, username, passwd);
 	 	if (result != OK_RC) {
 	 		if (conf_auth_type & AUTH_LOCAL_SND) {
 	 			result = auth_local(username, passwd);
 	 			if (result == OK_RC)
 	 			{
 	 				memset(passwd, '\0', sizeof(passwd));
 	 				local_login(username); /* never returns */
 	 			}
 	 		}
 	 	} else {
 	 		memset(passwd, '\0', sizeof(passwd));
 	 		radius_login(username); /* never returns */
 	 	}
 	 }

	 *username = '\0';
	 
	 if ((++tries) <= conf_login_tries) {
		alarm(remaining);
	 	sleep(tries * 2);
	 	remaining = alarm(0);
	 }

	}

	rc_log(LOG_INFO, "maximum login tries exceeded. Go away!");
	sleep(1);
	
	exit (ERROR_RC);
}
