/*
 * $Id: radlogin.c,v 1.13 1996/03/18 23:20:19 lf Exp $
 *
 * Copyright (C) 1995,1996 Lars Fenneberg
 *
 * See the file COPYRIGHT for the respective terms and conditions. 
 * If the file is missing contact me at in5y050@public.uni-hamburg.de 
 * and I'll send you a copy.
 *
 */

static char	rcsid[] =
		"$Id: radlogin.c,v 1.13 1996/03/18 23:20:19 lf Exp $";

static char	copyright_1[] =
		"$Copyright: (C) 1995,1996 Lars Fenneberg $";

static char	copyright_2[] =
		"$Parts: (C) 1992 Livingston Enterprises, Inc. $";

static char	copyright_3[] =
		"$Parts: (C) 1992,1993, 1994,1995 The Regents of the \
University of Michigan and Merit Network, Inc. All Rights Reserved $";

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/param.h>
#include	<netinet/in.h>
#include	<sys/time.h>
#include	<sys/signal.h>

#include	<stdio.h>
#include	<netdb.h>
#include	<unistd.h>
#include	<pwd.h>
#include	<string.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<syslog.h>
#include	<limits.h>

#ifdef SHADOW_PASSWORD
#include <shadow.h>
#endif

#include	"radius.h"
#include	"messages.h"
#include	"pathnames.h"

static char *pname = NULL;
static struct env *env = NULL;

void alarm_handler(int sn)
{
	rc_log(LOG_ERR, SC_TIMEOUT, conf_login_timeout);
	sleep(1);
	exit(1);	
}

int auth_radius(UINT4 client_port, char *username, char *passwd)
{

	VALUE_PAIR 	*send, *received, *vp;
	UINT4		service, ftype, ctype;
	char 		msg[4096], *p, username_realm[256];
	char            name[2048], value[2048]; /* more than enough */
	int 		result;

	send = NULL;

	/*
	 * Determine and fill in Service-Type
	 */
	
#ifdef SCP
	/* determine based on the username what kind of service is requested.
	   this allows you to use one password for all accounts, but the
	   Merit radiusd supplies you just with the right information you
	   need for the specified service type	-lf, 03/15/96 */
	 
	switch (*username)
	{
		case 'S':
				service = PW_FRAMED;
				ftype = PW_SLIP;
				ctype = 0;
				username++;
				break;
		case 'C':
				service = PW_FRAMED;
				ftype = PW_SLIP;
				ctype = PW_VAN_JACOBSON_TCP_IP;
				username++;
				break;
		case 'P':
				service = PW_FRAMED;
				ftype = PW_PPP;
				ctype = 0;
				username++;
				break;
		default:
				service = PW_LOGIN;
				ftype = 0;
				ctype = 0;
				break;
	}
#else
	service = PW_LOGIN;	
	ftype = 0;
	ctype = 0;
#endif

	rc_avpair_add(&send, PW_SERVICE_TYPE, &service);

	/* Fill in Framed-Protocol, if neccessary */
	
	if (ftype != 0)
	{
		rc_avpair_add(&send, PW_FRAMED_PROTOCOL, &ftype);	
	} 

	/* Fill in Framed-Compression, if neccessary */

	if (ctype != 0)
	{
		rc_avpair_add(&send, PW_FRAMED_COMPRESSION, &ctype);	
	} 
	 
	/*
	 * Fill in User-Name
	 */

	 strncpy(username_realm, username, sizeof(username_realm));

	 /* Append default realm */
	 if ((strchr(username_realm, '@') == NULL) && conf_default_realm &&
	     (*conf_default_realm != '\0'))
	 {
		strncat(username_realm, "@", sizeof(username_realm));
		strncat(username_realm, conf_default_realm, sizeof(username_realm));
	 } 

	rc_avpair_add(&send, PW_USER_NAME, username_realm);
	
	/*
	 * Fill in User-Password
	 */
	 
	rc_avpair_add(&send, PW_USER_PASSWORD, passwd); 	  

	
	result = rc_auth(client_port, send, &received, msg);
	
	if (result == OK_RC)
	{
		rc_add_env(env, "RADIUS_USER_NAME", username);
	
		vp = received;
		while (vp)
		{
			strcpy(name, "RADIUS_");
			rc_avpair_tostr(vp, name+7, sizeof(name)-7, value, sizeof(value));
			
			/* Translate "-" => "_" and uppercase*/
			for(p = name; *p; p++) {
				*p = toupper(*p);
				if (*p == '-') *p = '_';
			}
			
			if (rc_add_env(env, name, value) < 0)
			{
				rc_log(LOG_CRIT, "rc_add_env: FATAL: not enough space for environment (increase ENV_SIZE)");
				abort();
			}
					
			vp = vp->next;
		}

		rc_log(LOG_NOTICE, SC_RADIUS_OK, username);
	}
	else
	{
		rc_log(LOG_NOTICE, SC_RADIUS_FAILED, username, result);
	}
	
	return result;
}

int auth_local(char *username, char *passwd)
{
	struct passwd	*pw;
	char		*xpasswd;
#ifdef SHADOW_PASSWORD
	struct spwd	*spw;
#endif

	if ((pw = getpwnam(username)) == NULL) {
		endpwent();
		rc_log(LOG_NOTICE, SC_LOCAL_FAILED, username);
		return BADRESP_RC;
	}
	endpwent();
	
#ifdef SHADOW_PASSWORD
        if((spw = getspnam(pw->pw_name)) == NULL) {
		endspent();
		rc_log (LOG_NOTICE, SC_LOCAL_FAILED, username);
		return BADRESP_RC;
        }
        else 
        { 
        	pw->pw_passwd = spw->sp_pwdp; 
        }
        endspent();
#endif /* SHADOW_PASSWORD */

	xpasswd = crypt(passwd, pw->pw_passwd);
	
	if (*pw->pw_passwd == '\0' || strcmp(xpasswd, pw->pw_passwd)) {
		rc_log(LOG_NOTICE, SC_LOCAL_FAILED, username);
		return BADRESP_RC;		
	}

	rc_log(LOG_NOTICE, SC_LOCAL_OK, username);
	return OK_RC;
}

void
local_login(char *username)
{
	/* login should spot this... but who knows what old /bin/logins
	 * may be still around
	 */
	if (*username == '-') {
		rc_log(LOG_WARNING, "username can't start with a dash");
		exit(ERROR_RC);
	}
	/* the new shadow login seems to require either a -r or a -h
	 * flag for -f to work (so source code, lmain.c) so we supply
	 * it here. shouldn't hurt on other systems,	-lf, 03/13/96
	 */
	execle(conf_login_local, conf_login_local, "-h", "localhost", "-f", username, NULL, env->env);
	rc_log(LOG_ERR, "couldn't execute %s: %s", conf_login_local, strerror(errno));
	sleep(1);	/* give the user time to read */
	exit(ERROR_RC);
}

void
radius_login(char *username)
{
	execle(conf_login_radius, conf_login_radius, NULL, env->env);
	rc_log(LOG_ERR, "couldn't execute %s: %s", conf_login_radius, strerror(errno));
	sleep(1);	/* give the user time to read */
	exit(ERROR_RC);
}

int
login_allowed(char *tty)
{
	FILE *fp;
#ifdef PATH_MAX
	char fname[PATH_MAX];
#else
	char fname[1024];
#endif
	int c;

	strcpy(fname, conf_nologin);
	if (access(fname, F_OK) < 0) {
		sprintf(fname, "%s.%s", conf_nologin, tty);
		if (access(fname, F_OK) < 0)
			return 1;
	}

	if ((fp = fopen(fname, "r")) != NULL)
	{
		while ((c = fgetc(fp)) != EOF)
		{
			if (c == '\n')
				fputc(c, stdout);
			fputc(c, stdout);
		}
		fflush(stdout);
		fclose(fp);
	} else {
		printf(SC_NOLOGIN);
	}
	return (0);		
}

void usage(void)
{
	fprintf(stderr,"Usage: %s [-Vh] [-i <client_port>]\n", pname);
	fprintf(stderr,"  -i		ttyname to send to the server\n");
	fprintf(stderr,"  -V		output version information\n");
	fprintf(stderr,"  -h		output this text\n");     
        exit(ERROR_RC);
}

void version(void)
{
	fprintf(stderr,"%s: %s\n", pname ,rcsid);
	exit(ERROR_RC);
}

int
main (int argc, char **argv)
{
	int             result;
	char		username[128];
	char            passwd[AUTH_PASS_LEN + 1];
	int 		tries, remaining, c;
	UINT4		client_port;
	char		*tty, *ttyn;
	void 		(*login_func)(char *);	

        extern char *optarg;
        extern int optind;

	pname = (pname = strrchr(argv[0],'/'))?pname+1:argv[0];
	
	if (rc_read_config(_PATH_ETC_RADIUSCLIENT_CONF) != 0)
		exit(ERROR_RC);
	
	if (rc_read_dictionary(conf_dictionary) != 0)
		exit (ERROR_RC);

	if (rc_read_mapfile(conf_mapfile) != 0)
		exit (ERROR_RC);

	ttyn = ttyname(0);
	if ((tty = strrchr(ttyn, '/')) == NULL)
		tty = ttyn;
	else
		tty += 1;

	client_port = rc_map2id(ttyn);

	while ((c = getopt(argc,argv,"i:hV")) > 0)
	{
		switch(c) {
			case 'i':
				client_port = rc_map2id(optarg);
				break;
			case 'V':
				version();
				break;
			case 'h':
				usage();
				break;
			default:
				exit(ERROR_RC);
				break;
		}
	}
	
	if ((argc - optind) == 1)
	{
		strncpy(username,argv[optind], sizeof(username));
	}
	else
	{
		*username = '\0';
	}

	if ((env = rc_new_env(ENV_SIZE)) == NULL)
	{
		rc_log(LOG_CRIT, "rc_new_env: FATAL: out of memory");
		abort();
	}
	
	if (rc_import_env(env,environ) < 0)
	{
		rc_log(LOG_CRIT, "rc_import_env: FATAL: not enough space for environment (increase ENV_SIZE)");
		abort();
	}

	signal(SIGALRM, alarm_handler);

	remaining = conf_login_timeout;
	tries = 1;
	while (tries <= conf_login_tries)
	{
	 alarm(remaining);

	 while (!*username) {
	 		strncpy(username, rc_getstr (SC_LOGIN, 1), sizeof(username));
	 }
	 strncpy (passwd, rc_getstr(SC_PASSWORD, 0), sizeof (passwd));		

	 remaining = alarm(0);
	 
	 login_func = NULL;

 	 if (conf_auth_type & AUTH_LOCAL_FST)
 	 {
 	 	result = auth_local(username, passwd);
 	 	if (result != OK_RC) {
 	 		if (conf_auth_type & AUTH_RADIUS_SND) {
 	 			result = auth_radius(client_port, username, passwd);
 	 			if (result == OK_RC) {
 					login_func = radius_login;
 	 			}
 	 		}
 	 	} else {
			login_func = local_login;
 	 	}
 	 }
 	 else
 	 {
		result = auth_radius(client_port, username, passwd);
 	 	if (result != OK_RC) {
 	 		if (conf_auth_type & AUTH_LOCAL_SND) {
 	 			result = auth_local(username, passwd);
 	 			if (result == OK_RC)
 	 			{
					login_func = local_login;
 	 			}
 	 		}
 	 	} else {
			login_func = radius_login;
 	 	}
 	 }

	 memset(passwd, '\0', sizeof(passwd));

	 if (login_func != NULL)
	 	if (login_allowed(tty)) {
	 		(*login_func)(username);
		} else {
			sleep(1);
			exit (ERROR_RC);
		}

	 *username = '\0';
	 
	 if ((++tries) <= conf_login_tries) {
		alarm(remaining);
	 	sleep(tries * 2);
	 	remaining = alarm(0);
	 }

	}

	rc_log(LOG_INFO, SC_EXCEEDED);
	sleep(1);
	
	exit (ERROR_RC);
}
