/*
 * $Id: term.c,v 1.1 1996/05/05 00:13:07 lf Exp $
 *
 * Copyright (C) 1996 Lars Fenneberg
 *
 * Partly inspired by mgetty+sendfax by Gert Doering. mgetty+sendfax
 * is Copyright (C) 1993,1994,1995,1996, Gert Doering.
 *
 * See the file COPYRIGHT for the respective terms and conditions. 
 * If the file is missing contact me at in5y050@public.uni-hamburg.de 
 * and I'll send you a copy.
 *
 */

#include <termios.h>

#include "config.h"
#include "radius.h"

static struct speeds_s {
    speed_t	cbaud;
    int	 speed;	
} speeds[] = {
	{ B0,	   0	},
	{ B50,	  50	},
	{ B75,	  75	},
	{ B110,	  110	},
	{ B134,	  134	},
	{ B150,	  150	},
	{ B200,	  200	},
	{ B300,	  300	},
	{ B600,	  600	},
#ifdef	B900
	{ B900,	  900	},
#endif
	{ B1200,  1200	},
	{ B1800,  1800	},
	{ B2400,  2400	},
#ifdef	B3600
	{ B3600,  3600	},
#endif
	{ B4800,  4800	},
#ifdef	B7200
	{ B7200,  7200	},
#endif
	{ B9600,  9600	},
#ifdef	B14400
	{ B14400, 14400	},
#endif
#ifdef	B19200
	{ B19200, 19200	},
#endif
#ifdef	B28800
	{ B28800, 28800	},
#endif
#ifdef	B38400
	{ B38400, 38400	},
#endif
#ifdef	EXTA
	{ EXTA,	  19200	},
#endif
#ifdef	EXTB
	{ EXTB,	  38400	},
#endif
#ifdef	B57600
	{ B57600, 57600	},
#endif
#ifdef	B115200
	{ B115200,115200},
#endif
#ifdef B230400
	{ B230400,230400},
#endif
	{ -1,	  -1,	}
};

/*
 * Function: rc_settermspeed
 *
 * Purpose: set the terminal speed
 *
 */

void rc_settermspeed(struct termios *term, int speed)
{
	int s;
	int cbaud = -1;
	
	for (s = 0; speeds[s].cbaud != -1; s++) {
		if (speeds[s].speed == speed) {
			cbaud = speeds[s].cbaud;
			break;
		}	
	}

	if (cbaud == -1)
		return;

	cfsetospeed(term, cbaud);
	cfsetispeed(term, cbaud);
}

/*
 * Function: rc_settermhardflow
 *
 * Purpose: initialize all flags to sane values
 *
 */

void rc_settermflow(struct termios *term, int type)
{
	term->c_cflag &= ~CRTSCTS;
	term->c_iflag &= ~(IXON | IXOFF | IXANY);

	if (type & FLOW_HARD)
		term->c_cflag |= CRTSCTS;
	if (type & FLOW_XON_IN)
		term->c_iflag |= IXOFF;
	if (type & FLOW_XON_OUT)
		term->c_iflag |= IXON | IXANY; 
}

/*
 * Function: rc_settermcarrier
 *
 * Purpose: initialize all flags to sane values
 *
 */

void rc_settermcarrier(struct termios *term, int on)
{
	if (on)
		term->c_cflag  &= ~CLOCAL;
	else
		term->c_cflag |= CLOCAL;
}

/*
 * Function: rc_settermsane
 *
 * Purpose: brings a terminal into a sane state
 *
 */

void rc_settermsane(struct termios *term)
{
    term->c_iflag = BRKINT | IGNPAR | IXON | IXANY;
    term->c_oflag = OPOST | TAB3;
    term->c_cflag &= ~(CSIZE | CSTOPB | PARENB | PARODD);
    term->c_cflag |= CS8 | CREAD | HUPCL | CLOCAL;
    term->c_lflag = ECHOK | ECHOE | ECHO | ISIG | ICANON;
    term->c_cc[VEOF] = 0x04;
    term->c_cc[VEOL] = 0;
}

/*
 * Function: rc_settermraw
 *
 * Purpose: switches a terminal into raw mode
 *
 */

void rc_settermraw(struct termios *term, int signals)
{
	if (signals)
		term->c_lflag = ISIG;
	else
		term->c_lflag = 0;
		 
	term->c_iflag = 0;
	term->c_oflag = 0;
		
	term->c_cc[VMIN] = 1;
	term->c_cc[VTIME] = 0;
}

/*
 * Function: rc_getterm
 *
 * Purpose: get the terminal mode
 *
 */

int rc_getterm(int fd, struct termios *term)
{
	return (tcgetattr(fd, term));
}

/*
 * Function: rc_setterm
 *
 * Purpose: set the terminal mode
 *
 */

int
rc_setterm(int fd, struct termios *term)
{
	return (tcsetattr(fd, TCSAFLUSH, term));
}

/*
 * Function: rc_termhangup
 *
 * Purpose: hang up the modem by setting baud rate to 0
 *
 */

void rc_termhangup(int fd)
{
	struct termios term, term_old;

	rc_getterm(fd, &term);
	term_old = term;
	rc_settermspeed(&term, 0);	
	rc_setterm(fd, &term);
	mdelay(500);
	rc_setterm(fd, &term_old);
}
