/*
 * $Id: outbound.c,v 1.1 1996/05/05 00:13:21 lf Exp $
 *
 * Copyright (C) 1996 Lars Fenneberg
 *
 * See the file COPYRIGHT for the respective terms and conditions. 
 * If the file is missing contact me at in5y050@public.uni-hamburg.de 
 * and I'll send you a copy.
 *
 */

#include <syslog.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include <sys/socket.h>
#include <sys/fcntl.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/termios.h>

#include "config.h"
#include "radius.h"
#include "radlogin.h"
#include "messages.h"

/* telnet options used in this file, actually there're a lot more */
#define TN_ECHO         0001		/* echo option */
#define TN_SGA          0003		/* suppress GOAHEAD option */
#define TN_LINEMODE     0042		/* linemode */
#define TN_WILL         0373		/* i offer, or ack for DO */
#define TN_WONT		0374		/* i will stop, or nack for DO */ 
#define TN_DO           0375		/* please do, or ack for WILL */
#define TN_DONT         0376		/* stop, or nack for WILL */
#define TN_IAC          0377		/* Is A Command */

struct line_info {
	char *port;
	int speed;
	int flowctl;
};

static char outbound_port[PATH_MAX];
static int state = 0;
static int telnet = 0;
static int is_term = 0;
static struct termios term_old;
static int mfd;

int keepalive = 1;

void
send_telnet(u_char cmd, u_char opt)
{
	char buf[4];

	sprintf(buf,"%c%c%c", TN_IAC, cmd, opt);	
	write(1, buf, 3);
}

int
handle_telnet(u_char *buff, int count)
{
	u_char *p;
	int valid_telnet;

	valid_telnet = 0;
	
	for(p = buff; (p - buff) < count; p++) {

		/* this whole state machine is horribly simple... so simple
		   that it relies on the other side to prevent negotiation
		   loops. */

		switch (*p) {
			case TN_IAC:
				state = TN_IAC;
				break;
			case TN_WILL:
				if (state == TN_IAC)
					state = TN_WILL;
				else
					state = 0;
				break;
			case TN_WONT:
				if (state == TN_IAC)
					state = TN_WONT;
				else
					state = 0;
				break;
			case TN_DO:
				if (state == TN_IAC)
					state = TN_DO;
				else
					state = 0;
				break;
			case TN_DONT:
				if (state == TN_IAC)
					state = TN_DONT;
				else
					state = 0;
				break;
			case TN_ECHO:
			case TN_SGA:
				switch (state) {
					case TN_WILL:
						send_telnet(TN_DONT, *p);
						valid_telnet = 1;
						state = 0;
						break;
					case TN_WONT:
						send_telnet(TN_DONT, *p);
						valid_telnet = 1;
						state = 0;
						break;
					case TN_DO:
						valid_telnet = 1;
						state = 0;
						break;
					case TN_DONT:
						send_telnet(TN_WILL, *p);
						valid_telnet = 1;
						/* fall through */
					default:
						state = 0;
						break;
				}
				break;
			case TN_LINEMODE:
			default:
				switch(state) {
					case TN_WILL:
						send_telnet(TN_DONT, *p);
						valid_telnet = 1;
						state = 0;
						break;
					case TN_DO:
						send_telnet(TN_WONT, *p);
						valid_telnet = 1;
						/* fall through */
					default:
						state = 0;
						break;
				}
				break;
		}
	}

	return valid_telnet;
}

void detect_telnet(void)
{
	fd_set readfds;
	struct timeval tv;
	u_char opt[128];
	int len, status, times;

	send_telnet(TN_WILL, TN_ECHO);
	send_telnet(TN_WILL, TN_SGA);

	FD_ZERO(&readfds);
	FD_SET(0, &readfds);

	tv.tv_sec =  0;
	tv.tv_usec = 2500;
	
	times = 0;
	
	while ((times < 20) && ((status = select(1, &readfds, (fd_set *)NULL,
		                             (fd_set *)NULL, &tv)) >= 0))
	{
		if (FD_ISSET(0, &readfds)) {	
			len = read(0, opt, sizeof(opt));
			telnet = telnet || handle_telnet(opt, len);
		}

		FD_ZERO(&readfds);
		FD_SET(0, &readfds);
		
		tv.tv_sec = 0;
		tv.tv_usec = 50000L;
		
		times++;
	}

	/* try to hide the gibberish about telnet options... */
	if (!telnet) {
		fputs("\r      \r", stdout);
		fflush(stdout);
	}		
}

static void
do_io_socket_port(int mfd)
{
	fd_set deffds;
	fd_set excpfds;
	fd_set readfds;

	u_char buff[4096];	
	int count, to_write, status;
	
	FD_ZERO(&deffds);
	FD_SET(0, &deffds);
	FD_SET(mfd, &deffds);
	
	for(;;) {
		readfds = deffds;
		excpfds = deffds;
		
		do {
			if ((status = select(mfd+1,
			                     &readfds, (fd_set *)NULL, &excpfds, NULL)) <= 0)
			                     sleep(1);
		} while (status <= 0);

	        if (FD_ISSET(0,&excpfds))
			return;

		if (FD_ISSET(mfd,&excpfds)) {
			write(1, SC_OUT_EXCEPTION, strlen(SC_OUT_EXCEPTION)); 
			return;
		}

		if (FD_ISSET(0, &readfds)) {
			count = read(0, buff, sizeof(buff));	
			if (count <= 0) {
				return;
			} else {
				if(count != 0) {
					to_write = count;
					do {
						count = write(mfd, buff, to_write);
						if (count > 0)
							to_write -= count;
					} while ((to_write > 0) && (count > 0));
					if (count <= 0) {
						write(1, SC_OUT_WRITE, strlen(SC_OUT_WRITE)); 
						return;
					}
				}
			}
		}
		if (FD_ISSET(mfd, &readfds)) {
			count = read(mfd, buff, sizeof(buff));	
			if (count <= 0) {
				write(1, SC_OUT_READ, strlen(SC_OUT_READ)); 
				return;
			} else {
				to_write = count;
				do {
					count = write(1, buff, to_write);
					if (count > 0)
						to_write -= count;
				} while ((to_write > 0) && (count > 0));
				if (count <= 0) 
					return;
			}
		}
	}
}

static int
get_lock(char *port)
{
	char tlock[PATH_MAX], lock[PATH_MAX];
	char *p, buf[10];
	int fd;
	struct stat st;
	
	if ((p = strrchr(port, '/')) == NULL) {
		rc_log(LOG_ERR, "ERROR can't determine port basename: %s", port);
		return (-1);
	} 

	sprintf(lock,"%s%s", LOCK_PREFIX, p+1);
	sprintf(tlock,"%s%s.%d", LOCK_PREFIX, p+1, getpid());
	
	/* if file is already there, don't try to create a link */
	if (!access(lock, R_OK))
		return (-1); 
	
	if ((fd = open(tlock, O_CREAT|O_WRONLY|O_EXCL, 0444)) < 0) {
		rc_log(LOG_ERR,"ERROR can't open temporary lockfile %s: %s", tlock, strerror(errno));
		return (-1);
	}
	
	sprintf(buf,"%8d\n", getpid());
	write(fd, buf, 9);		
	
	close(fd);

	if (link(tlock, lock) < 0) {
		unlink(tlock);
		return (-1);
	}
	
	if (stat(lock, &st) < 0) {
		rc_log(LOG_ERR,"ERROR can't stat %s: %s", lock, strerror(errno));
		unlink(tlock);
		unlink(lock);
		return (-1);
	}

	/* someone else was a little bit quicker (NFS) */
	if (st.st_nlink != 2) {
		unlink(tlock);
		return (-1);	
	}
	
	unlink(tlock);
	return 0;
}

static struct line_info *
get_port(void)
{
	static char port[PATH_MAX];
	static struct line_info info = { port, 0 };
	char buf[4096], *p = NULL;
	FILE *f;
	int line;
	char *port_pool = rc_conf_str("port_pool");

	if ((f = fopen(port_pool,"r")) == NULL) {
		rc_log(LOG_ERR, "ERROR can't open file %s: %s", port_pool, strerror(errno));
		return NULL;
	}
	
	line = 0;
	while (fgets(buf,sizeof(buf),f) != NULL) {
		line++;
		port[strlen(buf)-1] = '\0';
		
		p = strtok(buf," \t");
		if (!p) {
			rc_log(LOG_ERR, "ERROR syntax error in file %s, line %d", port_pool, line);
			return NULL;
		} else {
			if (*p != '/')
				sprintf(port, "/dev/%s", p);
			else
				strncpy(info.port, p, sizeof(port));
		}
		
		p = strtok(NULL," \t");
		if (!p) {
			rc_log(LOG_ERR, "ERROR syntax error in file %s, line %d", port_pool, line);
			return NULL;		
		} else {
			info.speed = atoi(p);
		}
		
		p = strtok(NULL," \t");
		if (!p) {
			rc_log(LOG_ERR, "ERROR syntax error in file %s, line %d", port_pool, line);
			return NULL;		
		} else {
			info.flowctl = 0;
			
			if (p[strlen(p)-1] == '\n')
				p[strlen(p)-1] = '\0'; 
				
			while ((p = strtok(p, ",")) != NULL) {
				if (!strcmp(p,"FLOW_HARD"))
					info.flowctl |= FLOW_HARD;
				else if (!strcmp(p,"FLOW_XON_IN"))
					info.flowctl |= FLOW_XON_IN;
				else if (!strcmp(p,"FLOW_XON_OUT"))
					info.flowctl |= FLOW_XON_OUT;
				else {
					rc_log(LOG_ERR, "ERROR syntax error in file %s, line %d", port_pool, line);
					return NULL;		
				}	
				p = NULL;
			}
		}
		
		if (!get_lock(info.port)) {
			fclose(f);
			return &info;
		}
	}
	
	fclose(f); 
	
	return NULL;
}

static void
free_port(char *port)
{
	char buf[PATH_MAX];
	char *p;
	
	if ((p = strrchr(port, '/')) == NULL) {
		rc_log(LOG_CRIT, "ERROR can't free port %s", port);
		return;
	} 

	sprintf(buf,"%s%s", LOCK_PREFIX, p+1);

	unlink(buf);
}

static void
sig_exit(int sn)
{
	if (is_term)
		rc_setterm(0, &term_old);
		
	rc_termhangup(mfd);
	close(mfd);

	free_port(outbound_port);
		
	exit(ERROR_RC);
}

void
outbound_login(char *username)
{
	int on = 1;
	struct termios term;
	struct line_info *info;

	is_term = isatty(0);

	/* activate keepalive, normally it's a good idea, but
	   not over dial on demand lines, etc.. */
	if (keepalive && !is_term)
	 	setsockopt(0, SOL_SOCKET, SO_KEEPALIVE, &on, sizeof(on));

	signal(SIGPIPE, sig_exit);
	signal(SIGHUP, sig_exit);
	signal(SIGINT, sig_exit);
	signal(SIGQUIT, sig_exit);
	signal(SIGTERM, sig_exit);

	if (!(info = get_port())) {
		rc_log(LOG_CRIT, "ERROR no free ports available");
		exit(ERROR_RC);
	}
	strcpy(outbound_port, info->port);
	
	if ((mfd = open(outbound_port, O_RDWR|O_NDELAY)) < 0) {
		rc_log(LOG_CRIT, "ERROR open on %s failed: %s", outbound_port, strerror(errno));
		exit(ERROR_RC);
	} 
	
	(void) rc_getterm(mfd, &term);
	rc_settermsane(&term);
	rc_settermraw(&term, 0);
	rc_settermspeed(&term, info->speed);
	rc_settermflow(&term, info->flowctl);
	rc_settermcarrier(&term, 0);
	(void) rc_setterm(mfd, &term);
	
	if (is_term) {
		(void) rc_getterm(0, &term);
		term_old = term;
		rc_settermraw(&term, 1);
		(void) rc_setterm(0, &term);
	}

	printf("Connected to %s, speed %d, flow%s%s%s.\r\nLine is %s8bit clean%s.\r\n", 
		outbound_port, 
		info->speed, 
		(info->flowctl & FLOW_HARD)?" RTS/CTS":"",
		(info->flowctl & FLOW_XON_IN)?", XON (in)":"",
		(info->flowctl & FLOW_XON_OUT)?", XON (out)":"",
	        telnet?"not ":"",telnet?" (TELNET)":"");
	        
	fflush(stdout);

	do_io_socket_port(mfd);		

	if (is_term)
		(void) rc_setterm(0, &term_old);
	
	rc_termhangup(mfd);
	close(mfd);	

	free_port(outbound_port);
	
	exit(OK_RC);
}
