/*
 * $Id: rport.c,v 1.1 1996/05/05 00:13:21 lf Exp $
 *
 * Copyright (C) 1996 Lars Fenneberg
 * 
 * Based on rmodem.c by Bjorn Ekwall, Juergen Weigert/Michael Schroeder,
 * Oliver Laumann, Copyright (C) 1994,1993,1987 under GPL. 
 *
 * See the file COPYRIGHT for the respective terms and conditions. 
 * If the file is missing contact me at in5y050@public.uni-hamburg.de 
 * and I'll send you a copy.
 *
 */

static char     rcsid[] =
                "$Id: rport.c,v 1.1 1996/05/05 00:13:21 lf Exp $";
                
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <limits.h>
#include <errno.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <termios.h>
#include <ctype.h>
#include <netdb.h>
#include <sys/signal.h>
#include <sys/fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define MAX(a, b)     ((a) > (b) ? (a) : (b))

/*
 * you shouldn't have to change this unless you are using the new pty
 * code (128 ptys) 
 */

#define PTY_FILE_SIZE  11
#define MASTER_PREFIX "/dev/pty"
#define SLAVE_PREFIX  "/dev/tty"
#define PTY_RANGE_0   "pqrs"
#define PTY_RANGE_1   "0123456789abcdef"

static char *pname;
static char slink[PATH_MAX];

void settermraw(struct termios *term, int signals)
{
	if (signals)
		term->c_lflag = ISIG;
	else
		term->c_lflag = 0;
		 
	term->c_iflag = 0;
	term->c_oflag = 0;
		
	term->c_cc[VMIN] = 1;
	term->c_cc[VTIME] = 0;
}

int getterm(int fd, struct termios *term)
{
	return (tcgetattr(fd, term));
}

int setterm(int fd, struct termios *term)
{
	return (tcsetattr(fd, TCSAFLUSH, term));
}

int good_ipaddr (char *addr)
{
	int             dot_count;
	int             digit_count;

	if (addr == NULL)
		return (-1);

	dot_count = 0;
	digit_count = 0;
	while (*addr != '\0' && *addr != ' ')
	{
		if (*addr == '.')
		{
			dot_count++;
			digit_count = 0;
		}
		else if (!isdigit (*addr))
		{
			dot_count = 5;
		}
		else
		{
			digit_count++;
			if (digit_count > 3)
			{
				dot_count = 5;
			}
		}
		addr++;
	}
	if (dot_count != 3)
	{
		return (-1);
	}
	else
	{
		return (0);
	}
}

unsigned long get_ipaddr (char *host)
{
	struct hostent *hp;

	if (good_ipaddr (host) == 0)
	{
		return ntohl(inet_addr (host));
	}
	else if ((hp = gethostbyname (host)) == (struct hostent *) NULL)
	{
		fprintf(stderr, "couldn't resolve hostname: %s", host);
		return (0UL);
	}
	return ntohl((*(unsigned long *) hp->h_addr));
} 

void
sig_exit (int sig)
{
	/* close the master side */
	close(0);
	close(1);
	
	unlink(slink);
	exit(1);	
}

int
getpty(char **sname)
{

	char mfile[PTY_FILE_SIZE];
	static char sfile[PTY_FILE_SIZE];
	char *m, *s, *q, *p;
	int mfd;

	strcpy(mfile, MASTER_PREFIX);
	strcpy(sfile, SLAVE_PREFIX);
	
	m = mfile + PTY_FILE_SIZE - 3; 
	s = sfile + PTY_FILE_SIZE - 3;

	*sname = NULL;
	
	for (q = PTY_RANGE_0; (m[0] = *q) != '\0'; q++) {
		for (p = PTY_RANGE_1; (m[1] = *p) != '\0'; p++) {

			if ((mfd = open(mfile, O_RDWR)) == -1)
				continue;
				
			s[0] = *q;
			s[1] = *p;
			
			if (open(sfile, O_RDWR) < 0) {
				close(mfd);
				continue;
			}
		
			*sname = sfile;
			
			return mfd;
		}
	}


	return (-1);	
}

int
net_open(char *server, int port)
{
	struct sockaddr_in sin;
	int nfd;

	memset(&sin, 0, sizeof(sin));
	
	sin.sin_family = AF_INET;
	if ((sin.sin_addr.s_addr = htonl(get_ipaddr(server))) == 0) {
		fprintf(stderr, "ERROR unknown server (%s)\r\n", server);
		return (-1);
	}
	
	sin.sin_port = htons(port);
	
	nfd = socket(AF_INET, SOCK_STREAM, 0);
	
	if (connect(nfd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		close(nfd);
		return (-1);
	}
	
	return nfd;
}

void
do_io(char *server, int port)
{
	fd_set deffds;
	fd_set readfds;
	fd_set excpfds;
	u_char buff[4096];	
	int nfd = -1;
	int count, to_write, status;
	struct timeval tv;

	FD_ZERO(&deffds);
	FD_SET(0, &deffds);

	/* loop as long as there is no one connected to the slave */
	for (;;) {
		readfds = deffds;
		excpfds = deffds;
		
		tv.tv_sec = 0;
		tv.tv_usec = 100000L;

		status = select(1,&readfds, (fd_set *)NULL, &excpfds, &tv);
				
		if ((status > 0) && (FD_ISSET(0, &readfds) && !FD_ISSET(0, &excpfds)))
			break;

		sleep(1);
	}

	if ((nfd = net_open(server, port)) < 0) {
		fprintf(stderr,"ERROR network open failed (%s)\r\n", server);
		return;
	}

	FD_SET(nfd, &deffds);

	for(;;) {
		readfds = deffds;
		excpfds = deffds;

		do {
			if ((status = select(MAX(nfd+1,1),&readfds, 
			                    (fd_set *)NULL, &excpfds, NULL)) == 0)
				sleep(1);
		} while (status == 0);
		
		if (status < 0) {
			fprintf(stderr, "ERROR select error: %s\r\n", strerror(errno));
			close(nfd);
			return;
		}

		if (FD_ISSET(0,&excpfds))  {
			close(nfd);
			return;
		}

		if (FD_ISSET(nfd,&excpfds)) {
			fprintf(stderr, "ERROR network error (%s)\r\n", server);
			close(nfd);
			return;
		}

		if (FD_ISSET(0, &readfds)) {
			count = read(0, buff, sizeof(buff));	
			if (count <= 0) {
				fprintf(stderr,"ERROR read failed\r\n");
				close(nfd);
				return;
			} else {
				to_write = count;
				do {
					count = write(nfd, buff, to_write);
					if (count > 0)
						to_write -= count;
				} while ((to_write > 0) && (count > 0));
				if (count <= 0) {
					fprintf(stderr,"ERROR network write failed\r\n");
					close(nfd);
					return;
				}
			}
		}
		if (FD_ISSET(nfd, &readfds)) {
			count = read(nfd, buff, sizeof(buff));	
			if (count <= 0) {
				fprintf(stderr,"ERROR network read failed (%s)\r\n", server);
				close(nfd);
				return;
			} else {
				to_write = count;
				do {
					count = write(1, buff, to_write);
					if (count > 0)
						to_write -= count;
				} while ((to_write > 0) && (count > 0));
				if (count <= 0) {
					close(nfd);
					return;
				}
			}
		}
	}
}

void usage(void)
{
        fprintf(stderr,"Usage: %s [-Vh] server:port symlink-path\n\n", pname);
        fprintf(stderr,"  -V            output version information\n");
        fprintf(stderr,"  -h            output this text\n");
	exit(1);
}        

void version(void)
{
        fprintf(stderr,"%s: %s\n", pname ,rcsid);
        exit(1);
}

int
main(int argc, char **argv)
{
	
	int mfd, sfd, c, fd;
	int port = 0;
	char *slave, *p;
	char *server = NULL;
	struct termios term;
        
        extern char *optarg;
        extern int optind;

	pname = (pname = strrchr(argv[0],'/'))?pname+1:argv[0];

	while ((c = getopt(argc,argv,"hV")) > 0)
	{
		switch(c) {
			case 'V':
				version();
				break;
			case 'h':
				usage();
				break;
			default:
				return(1);
				break;
		}
	}

        argc -= optind;
        argv += optind;
	
	if (argc == 2) {
		if ((p = strchr(argv[0], ':')) != NULL) {
			*p = '\0';
			server = argv[0];
			port = atoi(p+1);
		} else {
			fprintf(stderr, "%s: no port specfied", pname);
			return(1);
		}
		
		if (*argv[1] == '/')
			strncpy(slink, argv[1], sizeof(slink));
		else
			sprintf(slink, "/dev/%s", argv[1]);
	} else {
		usage();
	}

	if ((mfd = getpty(&slave)) < 0)
	{
		fprintf(stderr, "%s: can't find a free pty", pname);
		return(1);
	}

	if ((sfd = open(slave, O_RDWR)) < 0)
	{
		close(mfd);
		fprintf(stderr, "%s: can't open slave: %s", pname, slave);
		return(1);
	}
	close(sfd);

	if (symlink(slave, slink) < 0)
	{
		fprintf(stderr, "%s: symlink from %s to %s failed: %s", pname, slave, slink, strerror(errno));
		return(1);
	}

	signal(SIGPIPE, SIG_IGN);
	signal(SIGINT, SIG_IGN);
	signal(SIGTSTP, SIG_IGN);
	
	signal(SIGHUP, sig_exit);
	signal(SIGTERM, sig_exit);
	
	switch (fork()) {
		case -1:
			fprintf(stderr, "%s: fork failed: %s", pname, strerror(errno));
			return(1);
			break;
		case 0:
			close(0);
			close(1);
			close(2);
			
			/* release our controlling terminal */
			if ((fd = open( "/dev/tty", O_RDWR )) >= 0 ) {
				ioctl( fd, TIOCNOTTY, NULL );
				close(fd);
			}
			      
			/* abandon our process group */
			setsid();                                      

			dup(mfd);
			dup(mfd);
			dup(mfd);

			for(;;) {
				if ((sfd = open(slave, O_RDWR)) < 0)
					return(1);
				
				(void) getterm(sfd, &term);
				settermraw(&term, 0);
				(void) setterm(sfd, &term);

				close(sfd);

				do_io(server, port);

				/* hangup the reading process on slave */
				signal(SIGHUP, SIG_IGN);
				vhangup();
				signal(SIGHUP, sig_exit);
			}
			/* never reached */
			return(0);
			break;
		default:
			close(mfd);
			break;
	}
	
	exit(0);	
}