/*
 *
 *	RADIUS
 *	Remote Authentication Dial In User Service
 *
 *
 *	Livingston Enterprises, Inc.
 *	6920 Koll Center Parkway
 *	Pleasanton, CA   94566
 *
 *	Copyright 1992 Livingston Enterprises, Inc.
 *	Copyright 1996-1999 Cistron Internet Services B.V.
 *
 *	Permission to use, copy, modify, and distribute this software for any
 *	purpose and without fee is hereby granted, provided that this
 *	copyright and permission notice appear on all copies and supporting
 *	documentation, the name of Livingston Enterprises, Inc. not be used
 *	in advertising or publicity pertaining to distribution of the
 *	program without specific prior permission, and notice be given
 *	in supporting documentation that copying and distribution is by
 *	permission of Livingston Enterprises, Inc.   
 *
 *	Livingston Enterprises, Inc. makes no representations about
 *	the suitability of this software for any purpose.  It is
 *	provided "as is" without express or implied warranty.
 *
 */

char files_sccsid[] =
"@(#)users.c	1.12 Copyright 1992 Livingston Enterprises Inc\n"
"@(#)files.c	2.43 Copyright 1999 Cistron Internet Services B.V.";

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/stat.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<grp.h>
#include	<time.h>
#include	<ctype.h>
#include	<fcntl.h>

#ifdef DBM
#  include	<dbm.h>
#endif
#ifdef NDBM
#  include	<ndbm.h>
#endif

#include	"radiusd.h"

static	int  huntgroup_match(VALUE_PAIR *, char *);
#ifdef NDBM
static	DBM *dbmfile;
#endif

#include "cache.h"

struct pair_list {
	char *name;
	VALUE_PAIR *check;
	VALUE_PAIR *reply;
	int lineno;
	struct pair_list *next;
};
typedef struct pair_list PAIR_LIST;

static char *months[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

PAIR_LIST	*users;
PAIR_LIST	*huntgroups;
PAIR_LIST	*hints;
CLIENT		*clients;
NAS		*naslist;
REALM		*realms;

#define USERPARSE_EOS	(0)
#define USERPARSE_COMMA (1)

/*
 *	Returns a copy of a pair list.
 */
static VALUE_PAIR *paircopy(VALUE_PAIR *from)
{
	VALUE_PAIR *vp = NULL;
	VALUE_PAIR *last = NULL;
	VALUE_PAIR *i, *t;

	for(i = from; i; i = i->next) {
		if ((t = malloc(sizeof(VALUE_PAIR))) == NULL)
			continue;
		memcpy(t, i, sizeof(VALUE_PAIR));
		t->next = NULL;
		if (last)
			last->next = t;
		else
			vp = t;
		last = t;
	}

	return vp;
}


/*
 *	Move attributes from one list to the other
 *	if not already present.
 */
void pairmove(VALUE_PAIR **to, VALUE_PAIR **from)
{
	VALUE_PAIR *tailto, *i, *next;
	VALUE_PAIR *tailfrom = NULL;
	int has_password = 0;

	if (*to == NULL) {
		*to = *from;
		*from = NULL;
		return;
	}

	/*
	 *	First, see if there are any passwords here, and
	 *	point "tailto" to the end of the "to" list.
	 */
	tailto = *to;
	for(i = *to; i; i = i->next) {
		if (i->attribute == PW_PASSWORD ||
		/*
		 *	FIXME: this seems to be needed with PAM support
		 *	to keep it around the Auth-Type = Pam stuff.
		 *	Perhaps we should only do this if Auth-Type = Pam?
		 */
#ifdef PAM
		    i->attribute == PAM_AUTH_ATTR ||
#endif
		    i->attribute == PW_CRYPT_PASSWORD)
			has_password = 1;
		tailto = i;
	}

	/*
	 *	Loop over the "from" list.
	 */
	for(i = *from; i; i = next) {
		next = i->next;
		/*
		 *	If there was a password in the "to" list,
		 *	do not move any other password from the
		 *	"from" to the "to" list.
		 */
		if (has_password &&
		    (i->attribute == PW_PASSWORD ||
#ifdef PAM
		     i->attribute == PAM_AUTH_ATTR ||
#endif
		     i->attribute == PW_CRYPT_PASSWORD)) {
			tailfrom = i;
			continue;
		}

		/*
		 *	We never move "Fall-Through" to the "to" list.
		 */
		if (i->attribute == PW_FALL_THROUGH) {
			tailfrom = i;
			continue;
		}

		/*
		 *	We always move "Hint" and "Framed-Route" to the
		 *	"to" list.
		 *
		 *	If the attribute is already present in "to",
		 *	do not move it from "from" to "to".
		 *
		 *	NOTE: THIS MAY CAUSE PROBLEMS FOR YOU.
		 *	Comment out this section of code if you don't
		 *	like it.
		 */
		if ((i->attribute != PW_HINT) &&
		    (i->attribute != PW_FRAMED_ROUTE) &&
		    (pairfind(*to, i->attribute) != NULL)) {
			DEBUG2("WARNING: Duplicate attribute %s is being ignored!", i->name);
			tailfrom = i;
			continue;
		}
		if (tailfrom)
			tailfrom->next = next;
		else
			*from = next;
		tailto->next = i;
		i->next = NULL;
		tailto = i;
	}
}

/*
 *	Move one kind of attributes from one list to the other
 */
void pairmove2(VALUE_PAIR **to, VALUE_PAIR **from, int attr)
{
	VALUE_PAIR *to_tail, *i, *next;
	VALUE_PAIR *iprev = NULL;

	/*
	 *	Find the last pair in the "to" list and put it in "to_tail".
	 */
	if (*to != NULL) {
		to_tail = *to;
		for(i = *to; i; i = i->next)
			to_tail = i;
	} else
		to_tail = NULL;

	for(i = *from; i; i = next) {
		next = i->next;

		if (i->attribute != attr) {
			iprev = i;
			continue;
		}

		/*
		 *	Remove the attribute from the "from" list.
		 */
		if (iprev)
			iprev->next = next;
		else
			*from = next;

		/*
		 *	Add the attribute to the "to" list.
		 */
		if (to_tail)
			to_tail->next = i;
		else
			*to = i;
		to_tail = i;
		i->next = NULL;
	}
}



/*
 *	Turn printable string into correct tm struct entries
 */
static void user_gettime(char *valstr, struct tm *tm)
{
	int	i;

	/* Get the month */
	for(i = 0; i < 12; i++) {
		if(strncmp(months[i], valstr, 3) == 0) {
			tm->tm_mon = i;
			i = 13;
		}
	}

	/* Get the Day */
	tm->tm_mday = atoi(&valstr[4]);

	/* Now the year */
	tm->tm_year = atoi(&valstr[7]) - 1900;
}


/*
 *	Strip a username, based on Prefix/Suffix from the "users" file.
 *	Not 100% safe, since we don't compare attributes.
 */
void presuf_setup(VALUE_PAIR *request_pairs)
{
	PAIR_LIST	*pl;
	VALUE_PAIR	*presuf_pair;
	VALUE_PAIR	*name_pair;
	VALUE_PAIR	*tmp;
	char		name[32];

	if ((name_pair = pairfind(request_pairs, PW_USER_NAME)) == NULL)
		return;

	for (pl = users; pl; pl = pl->next) {
		/*
		 *	Matching name?
		 */
		if (strncmp(pl->name, "DEFAULT", 7) != 0 &&
		    strcmp(pl->name, name_pair->strvalue) != 0)
			continue;
		/*
		 *	Find Prefix / Suffix.
		 */
		if ((presuf_pair = pairfind(pl->check, PW_PREFIX)) == NULL &&
		    (presuf_pair = pairfind(pl->check, PW_SUFFIX)) == NULL)
			continue;
		if (presufcmp(presuf_pair, name_pair->strvalue, name, sizeof(name)) != 0)
			continue;
		/*
		 *	See if username must be stripped.
		 */
		if ((tmp = pairfind(pl->check, PW_STRIP_USERNAME)) != NULL &&
		    tmp->lvalue == 0)
			continue;
		/* sizeof(name_pair->strvalue) > sizeof(name) */
		strcpy(name_pair->strvalue, name);
		name_pair->length = strlen(name_pair->strvalue);
		break;
	}
}



/*
 *	Compare a portno with a range.
 */
static int portcmp(VALUE_PAIR *check, VALUE_PAIR *request)
{
	char buf[AUTH_STRING_LEN];
	char *s, *p;
	int lo, hi;
	int port = request->lvalue;

	/* Same size */
	strcpy(buf, check->strvalue);
	s = strtok(buf, ",");
	while(s) {
		if ((p = strchr(s, '-')) != NULL)
			p++;
		else
			p = s;
		lo = atoi(s);
		hi = atoi(p);
		if (lo <= port && port <= hi) {
			return 0;
		}
		s = strtok(NULL, ",");
	}

	return -1;
}


/*
 *	See if user is member of a group.
 *	We also handle additional groups.
 */
static int groupcmp(VALUE_PAIR *check, char *username)
{
	struct passwd *pwd;
	struct group *grp;
	char **member;
	int retval;

	if (cache_passwd && (retval = H_groupcmp(check, username)) != -2)
		return retval;

	if ((pwd = rad_getpwnam(username)) == NULL)
		return -1;

	if ((grp = getgrnam(check->strvalue)) == NULL)
		return -1;

	retval = (pwd->pw_gid == grp->gr_gid) ? 0 : -1;
	if (retval < 0) {
		for (member = grp->gr_mem; *member && retval; member++) {
			if (strcmp(*member, pwd->pw_name) == 0)
				retval = 0;
		}
	}
	return retval;
}

/*
 *	Compare prefix/suffix.
 */
int presufcmp(VALUE_PAIR *check, char *name, char *rest, int restlen)
{
	char *realm;
	int len, namelen;
	int ret = -1;

#if 0 /* DEBUG */
	printf("Comparing %s and %s, check->attr is %d\n",
		name, check->strvalue, check->attribute);
#endif

	/*
	 *	Temporarily strip off the realm, unless ofcourse
	 *	we explicitly want to compare against a realm.
	 */
	realm = NULL;
	if (strchr(check->strvalue, '@') == NULL) {
		realm = strchr(name, '@');
		if (realm) *realm = 0;
	}

	len = strlen(check->strvalue);
	switch (check->attribute) {
		case PW_PREFIX:
			ret = strncmp(name, check->strvalue, len);
			if (ret == 0 && rest)
				strNcpy(rest, name + len, restlen);
			break;
		case PW_SUFFIX:
			namelen = strlen(name);
			if (namelen < len)
				break;
			ret = strcmp(name + namelen - len, check->strvalue);
			if (ret == 0 && rest) {
				strncpy(rest, name, namelen - len);
				rest[namelen - len] = 0;
			}
			break;
	}

	if (realm) *realm = '@';
	if (ret == 0 && rest && realm) {
		len = strlen(rest);
		strNcpy(rest + len, realm, restlen -len);
	}

	return ret;
}



/*
 *	Compare two pair lists except for the password information.
 *	Return 0 on match.
 */
static int paircmp(VALUE_PAIR *request, VALUE_PAIR *check)
{
	VALUE_PAIR *check_item = check;
	VALUE_PAIR *auth_item;
	int result = 0;
	char username[AUTH_STRING_LEN];
	int compare;

	username[0] = '\0';
	while (result == 0 && check_item != NULL) {
		switch (check_item->attribute) {
			/*
			 *	Attributes we skip during comparison.
			 *	These are "server" check items.
			 */
			case PW_EXPIRATION:
			case PW_LOGIN_TIME:
			case PW_PASSWORD:
			case PW_CRYPT_PASSWORD:
			case PW_AUTHTYPE:
#ifdef PAM /* cjd 19980706 */
                        case PAM_AUTH_ATTR:
#endif
			case PW_SIMULTANEOUS_USE:
			case PW_STRIP_USERNAME:
				check_item = check_item->next;
				continue;
		}
		/*
		 *	See if this item is present in the request.
		 */
		auth_item = request;
		for (; auth_item != NULL; auth_item = auth_item->next) {
			switch (check_item->attribute) {
				case PW_PREFIX:
				case PW_SUFFIX:
				case PW_GROUP_NAME:
				case PW_GROUP:
					if (auth_item->attribute  !=
					    PW_USER_NAME)
						continue;
					/* Sizes are the same */
					strcpy(username, auth_item->strvalue);
				case PW_HUNTGROUP_NAME:
					break;
				case PW_HINT:
					if (auth_item->attribute !=
					    check_item->attribute)
						continue;
					if (strcmp(check_item->strvalue,
					    auth_item->strvalue) != 0)
						continue;
					break;
				default:
					if (auth_item->attribute !=
					    check_item->attribute)
						continue;
			}
			break;
		}
		if (auth_item == NULL) {
			result = -1;
			continue;
		}

		/*
		 *	OK it is present now compare them.
		 */
		
		compare = 0;	/* default result */
		switch(check_item->type) {
			case PW_TYPE_STRING:
				if (check_item->attribute == PW_PREFIX ||
				    check_item->attribute == PW_SUFFIX) {
					if (presufcmp(check_item,
					    auth_item->strvalue, username,
					    sizeof(username)) != 0)
						return -1;
				}
				else
				if (check_item->attribute == PW_GROUP_NAME ||
				    check_item->attribute == PW_GROUP) {
					compare = groupcmp(check_item, username);
				}
				else
				if (check_item->attribute == PW_HUNTGROUP_NAME){
					compare = (huntgroup_match(request,
						check_item->strvalue) == 0);
					DEBUG("COMPARE = %d", compare);
					break;
				}
				else
				compare = strcmp(auth_item->strvalue,
						 check_item->strvalue);
				break;

			case PW_TYPE_INTEGER:
				if (check_item->attribute == PW_NAS_PORT_ID) {
					compare = portcmp(check_item,auth_item);
					break;
				}
				/*FALLTHRU*/
			case PW_TYPE_IPADDR:
				compare = auth_item->lvalue - check_item->lvalue;
				break;

			default:
				return -1;
				break;
		}

		switch (check_item->operator)
		  {
		  default:
		  case PW_OPERATOR_EQUAL:
		    if (compare != 0) return -1;
		    break;

		  case PW_OPERATOR_NOT_EQUAL:
		    if (compare == 0) return -1;
		    break;

		  case PW_OPERATOR_LESS_THAN:
		    if (compare >= 0) return -1;
		    break;

		  case PW_OPERATOR_GREATER_THAN:
		    if (compare <= 0) return -1;
		    break;
		    
		  case PW_OPERATOR_LESS_EQUAL:
		    if (compare > 0) return -1;
		    break;

		  case PW_OPERATOR_GREATER_EQUAL:
		    if (compare < 0) return -1;
		    break;
		  }
		


		if (result == 0)
			check_item = check_item->next;
	}

	return result;

}

/*
 *	Compare two pair lists. At least one of the check pairs
 *	has to be present in the request.
 */
static int hunt_paircmp(VALUE_PAIR *request, VALUE_PAIR *check)
{
	VALUE_PAIR *check_item = check;
	VALUE_PAIR *auth_item;
	int result = -1;

	if (check == NULL) return 0;

	while (result != 0 && check_item != (VALUE_PAIR *)NULL) {
		switch (check_item->attribute) {
			/*
			 *	Attributes we skip during comparison.
			 *	These are "server" check items.
			 */
			case PW_EXPIRATION:
			case PW_PASSWORD:
			case PW_AUTHTYPE:
/* cjd 19980706
 */
#ifdef PAM
                        case PAM_AUTH_ATTR:
#endif
			case PW_SIMULTANEOUS_USE:
				check_item = check_item->next;
				continue;
		}
		/*
		 *	See if this item is present in the request.
		 */
		auth_item = request;
		while(auth_item != (VALUE_PAIR *)NULL) {
			if (check_item->attribute == auth_item->attribute ||
			    ((check_item->attribute == PW_GROUP_NAME ||
			      check_item->attribute == PW_GROUP) &&
			     auth_item->attribute  == PW_USER_NAME))
				break;
			auth_item = auth_item->next;
		}
		if (auth_item == (VALUE_PAIR *)NULL) {
			check_item = check_item->next;
			continue;
		}

		/*
		 *	OK it is present now compare them.
		 */
		
		switch(check_item->type) {

			case PW_TYPE_STRING:
				if (check_item->attribute == PW_GROUP_NAME ||
				    check_item->attribute == PW_GROUP) {
					if (groupcmp(check_item, auth_item->strvalue)==0)
							result = 0;
				}
				else
				if (check_item->attribute == PW_HUNTGROUP_NAME){
					if (huntgroup_match(request,
						check_item->strvalue))
							result = 0;
				}
				else
				if (strcmp(check_item->strvalue,
						auth_item->strvalue) == 0) {
					result = 0;
				}
				break;

			case PW_TYPE_INTEGER:
				if (check_item->attribute == PW_NAS_PORT_ID) {
					if (portcmp(check_item, auth_item) == 0)
						result = 0;
					break;
				}
				/*FALLTHRU*/
			case PW_TYPE_IPADDR:
				if(check_item->lvalue == auth_item->lvalue) {
					result = 0;
				}
				break;

			default:
				break;
		}
		check_item = check_item->next;
	}

	return result;

}

/*
 *	Copy username from the beginning of the buffer and return
 *	a pointer to the first non-space charachter after that.
 *
 *	We allow "" and \ to include spaces in the username.
 */
static char *getusername(char *buffer, char *entry)
{
	char *ptr;
	char *to;
	int spc_seen = 0;

	ptr = buffer;
	to = entry;

	while (*ptr && !spc_seen) {
		switch (*ptr) {
			case '"':
				ptr++;
				while (*ptr && *ptr != '"')
					*to++ = *ptr++;
				if (*ptr) ptr++;
				break;
			case '\\':
				ptr++;
				*to++ = *ptr;
				if (*ptr) ptr++;
				break;
			case ' ':
			case '\t':
			case '\n':
				spc_seen = 1;
				break;
			default:
				*to++ = *ptr++;
				break;
		}
	}
	*to = 0;

	while (*ptr == ' ' || *ptr == '\t' || *ptr == '\n')
		ptr++;

	return ptr;
}


/*
 *	Free a PAIR_LIST
 */
static void pairlist_free(PAIR_LIST **pl)
{
	PAIR_LIST *p, *next;

	for (p = *pl; p; p = next) {
		if (p->name) free(p->name);
		if (p->check) pairfree(p->check);
		if (p->reply) pairfree(p->reply);
		next = p->next;
		free(p);
	}
	*pl = NULL;
}

/*
 *	Fixup a check line.
 *	If Password or Crypt-Password is set, but there is no
 *	Auth-Type, add one (kludge!).
 */
static void auth_type_fixup(VALUE_PAIR *check)
{
	VALUE_PAIR	*vp;
	VALUE_PAIR	*c = NULL;
	int		n;

	/*
	 *	See if a password is present. Return right away
	 *	if we see Auth-Type.
	 */
	for (vp = check; vp; vp = vp->next) {
		if (vp->attribute == PW_AUTHTYPE)
			return;
		if (vp->attribute == PW_PASSWORD) {
			c = vp;
			n = PW_AUTHTYPE_LOCAL;
		}
		if (vp->attribute == PW_CRYPT_PASSWORD) {
			c = vp;
			n = PW_AUTHTYPE_CRYPT;
		}
	}

	if (c == NULL)
		return;

	/*
	 *	Add an Auth-Type attribute.
	 *	FIXME: put Auth-Type _first_ (doesn't matter now,
	 *	might matter some day).
	 *	
	 */
	if ((vp = malloc(sizeof(VALUE_PAIR))) == NULL) {
		log(L_CONS|L_ERR, "no memory");
		exit(1);
	}
	memset(vp, 0, sizeof(VALUE_PAIR));
	strcpy(vp->name, "Auth-Type");
	vp->attribute = PW_AUTHTYPE;
	vp->type = PW_TYPE_INTEGER;
	vp->lvalue = n;

	vp->next = c->next;
	c->next = vp;

}

static void pair_sanity(char *fn, PAIR_LIST *entry)
{
	VALUE_PAIR *pair;

	for (entry = users; entry; entry = entry->next) {
		for (pair = entry->reply; pair; pair = pair->next) {
			if ((pair->attribute == PW_PASSWORD) ||
			    (pair->attribute == PW_CHAP_PASSWORD) ||
			    (pair->attribute >= 1000 &&
			     pair->attribute <= 65536)) {

		log(L_ERR|L_CONS, "\n[%s:%d] WARNING: Check item \"%s\"\n"
		    "\tfound in reply item list for user \"%s\".\n"
		    "\tThis attribute MUST go on the first line"
		    " with the other check items.\n", fn, entry->lineno,
		    pair->name, entry->name);

			}
		}
	}
}


#define FIND_MODE_NAME	0
#define FIND_MODE_REPLY	1

/*
 *	Read the users, huntgroups or hints file.
 *	Return a PAIR_LIST.
 */
static PAIR_LIST *file_read(char *file, int complain)
{
	FILE		*fp;
	int		mode = FIND_MODE_NAME;
	char		entry[256];
	char		buffer[256];
	char		*ptr, *s;
	VALUE_PAIR	*check_tmp;
	VALUE_PAIR	*reply_tmp;
	PAIR_LIST	*pl = NULL, *last = NULL, *t;
	int		lineno = 0;
	int		old_lineno;
	int		parsecode;

	/*
	 *	Open the table
	 */
	if ((fp = fopen(file, "r")) == NULL) {
		if (!complain) return NULL;
		log(L_CONS|L_ERR, "Couldn't open %s for reading", file);
		return NULL;
	}

	parsecode = USERPARSE_EOS;
	/*
	 *	Read the entire file into memory for speed.
	 */
	while(fgets(buffer, sizeof(buffer), fp) != (char *)NULL) {
		lineno++;

		if (buffer[0] == '#' || buffer[0] == '\n') continue;

parse_again:
		if(mode == FIND_MODE_NAME) {
			/*
			 *	Find the entry starting with the users name,
			 *	ignoring whitespace.
			 */
			if (isspace(buffer[0])) {
				if (parsecode != USERPARSE_EOS) {
					log(L_ERR|L_CONS,
					    "%s[%d]: Unexpected trailing comma for entry %s",
					    file, lineno, entry);
					fclose(fp);
					return NULL;
				}
				continue;
			}

			/*
			 *	FIXME:	support `\ ' space in usernames.
			 *		And "user name" ofcourse.
			 */
#if 1
			ptr = getusername(buffer, entry);
#else
			for (ptr = buffer; !isspace(*ptr) && *ptr; ptr++)
				;
			strncpy(entry, buffer, ptr - buffer);
			entry[ptr - buffer] = 0;
#endif

			/*
			 *	Include another file if we see
			 *	$INCLUDE filename
			 */
			if (strcasecmp(entry, "$include") == 0) {
				while(isspace(*ptr))
					ptr++;
				s = ptr;
				while (!isspace(*ptr))
					ptr++;
				*ptr = 0;
				if ((t = file_read(s, 1)) == NULL)
					continue;
				if (last)
					last->next = t;
				else
					pl = t;
				last = t;
				while (last && last->next)
					last = last->next;
				continue;
			}

			/*
			 *	Parse the check values
			 */
			check_tmp = NULL;
			reply_tmp = NULL;
			old_lineno = lineno;
			parsecode = userparse(ptr, &check_tmp);
			if(parsecode < 0) {
				log(L_ERR|L_CONS,
				"%s[%d]: Parse error (check) for entry %s",
					file, lineno, entry);
				fclose(fp);
				return NULL;
			} else if (parsecode == USERPARSE_COMMA) {
				log(L_ERR|L_CONS,
				    "%s[%d]: Unexpected trailing comma in check item list for entry %s",
				    file, lineno, entry);
				fclose(fp);
				return NULL;
			}
			mode = FIND_MODE_REPLY;
			parsecode = USERPARSE_COMMA;
		} /* mode == FIND_MODE_NAME */
		else {
			if(*buffer == ' ' || *buffer == '\t') {
				if (parsecode != USERPARSE_COMMA) {
					log(L_ERR|L_CONS,
				"%s[%d]: Syntax error: Previous line is missing a trailing comma for entry %s",
						file, lineno, entry);
					fclose(fp);
					return NULL;
				}

				/*
				 *	Parse the reply values
				 */
				parsecode = userparse(buffer, &reply_tmp);
				if (parsecode < 0) {
					log(L_ERR|L_CONS,
				"%s[%d]: Parse error (reply) for entry %s",
						file, lineno, entry);
					fclose(fp);
					return NULL;
				}
			}
			else {
				/*
				 *	Done with this entry...
				 */
				if ((t = malloc(sizeof(PAIR_LIST))) == NULL) {
					perror(progname);
					exit(1);
				}
				auth_type_fixup(check_tmp);
				memset(t, 0, sizeof(*t));
				t->name = strdup(entry);
				t->check = check_tmp;
				t->reply = reply_tmp;
				t->lineno = old_lineno;
				check_tmp = NULL;
				reply_tmp = NULL;
				if (last)
					last->next = t;
				else
					pl = t;
				last = t;

				pair_sanity(file, t);

				mode = FIND_MODE_NAME;
				if (buffer[0] != 0)
					goto parse_again;
			}
		}
	}
	/*
	 *	Make sure that we also read the last line of the file!
	 */
	if (mode == FIND_MODE_REPLY) {
		buffer[0] = 0;
		goto parse_again;
	}
	fclose(fp);

	return pl;
}


/*
 *	Find the named user in the DBM user database.
 *	Returns: -1 not found
 *	          0 found but doesn't match.
 *	          1 found and matches.
 */
#ifdef USE_DBM
static int dbm_find(char *name, VALUE_PAIR *request_pairs,
		VALUE_PAIR **check_pairs, VALUE_PAIR **reply_pairs)
{
	datum		named;
	datum		contentd;
	char		*ptr;
	VALUE_PAIR	*check_tmp;
	VALUE_PAIR	*reply_tmp;
	int		ret = 0;

	named.dptr = name;
	named.dsize = strlen(name);
#ifdef DBM
	contentd = fetch(named);
#endif
#ifdef NDBM
	contentd = dbm_fetch(dbmfile, named);
#endif
	if(contentd.dptr == NULL)
		return -1;

	check_tmp = NULL;
	reply_tmp = NULL;

	/*
	 *	Parse the check values
	 */
	ptr = contentd.dptr;
	contentd.dptr[contentd.dsize] = '\0';

	if (*ptr != '\n' && userparse(ptr, &check_tmp) != 0) {
		log(L_ERR|L_CONS, "Parse error (check) for user %s", name);
		pairfree(check_tmp);
		return -1;
	}
	while(*ptr != '\n' && *ptr != '\0') {
		ptr++;
	}
	if(*ptr != '\n') {
		log(L_ERR|L_CONS, "Parse error (no reply pairs) for user %s",
			name);
		pairfree(check_tmp);
		return -1;
	}
	ptr++;

	/*
	 *	Parse the reply values
	 */
	if (userparse(ptr, &reply_tmp) != 0) {
		log(L_ERR|L_CONS, "Parse error (reply) for user %s", name);
		pairfree(check_tmp);
		pairfree(reply_tmp);
		return -1;
	}

	/*
	 *	See if the check_pairs match.
	 */
	auth_type_fixup(check_tmp);
	if (paircmp(request_pairs, check_tmp) == 0) {
		ret = 1;
		pairmove(reply_pairs, &reply_tmp);
		pairmove2(reply_pairs, &reply_tmp, PW_FALL_THROUGH);
		pairmove(check_pairs, &check_tmp);
	}
	pairfree(reply_tmp);
	pairfree(check_tmp);

	return ret;
}
#endif /* DBM */

/*
 *	See if a VALUE_PAIR list contains Fall-Through = Yes
 */
static int fallthrough(VALUE_PAIR *vp)
{
	VALUE_PAIR *tmp;

	tmp = pairfind(vp, PW_FALL_THROUGH);

	return tmp ? tmp->lvalue : 0;
}


#ifdef ASCEND_PORT_HACK
/*
 *	dgreer --
 *	This hack changes Ascend's wierd port numberings
 *      to standard 0-??? port numbers so that the "+" works
 *      for IP address assignments.
 */
static int ascend_port_number(int nas_port)
{
	int service;
	int line;
	int channel;

	if (nas_port > 9999) {
		service = nas_port/10000; /* 1=digital 2=analog */
		line = (nas_port - (10000 * service)) / 100;
		channel = nas_port-((10000 * service)+(100 * line));
		nas_port =
			(channel - 1) + (line - 1) * ASCEND_CHANNELS_PER_LINE;
	}
	return nas_port;
}
#endif

/*
 *	Find the named user in the database.  Create the
 *	set of attribute-value pairs to check and reply with
 *	for this user from the database. The main code only
 *	needs to check the password, the rest is done here.
 */
int user_find(char *name, VALUE_PAIR *request_pairs,
		VALUE_PAIR **check_pairs, VALUE_PAIR **reply_pairs)
{
	int		nas_port = 0;
	VALUE_PAIR	*check_tmp;
	VALUE_PAIR	*reply_tmp;
	VALUE_PAIR	*tmp;
	PAIR_LIST	*pl;
	int		found = 0;
#ifdef USE_DBM
	int		i, r;
	char		buffer[256];
#endif

	/* 
	 *	Check for valid input, zero length names not permitted 
	 */
	if (name[0] == 0) {
		log(L_ERR, "zero length username not permitted\n");
		return -1;
	}

	/*
	 *	Find the NAS port ID.
	 */
	if ((tmp = pairfind(request_pairs, PW_NAS_PORT_ID)) != NULL)
		nas_port = tmp->lvalue;

	/*
	 *	Find the entry for the user.
	 */
#ifdef USE_DBM
	if (use_dbm) {
		/*
		 *	FIXME: No Prefix / Suffix support for DBM.
		 */
		sprintf(buffer, "%.200s/%.50s", radius_dir, RADIUS_USERS);
#ifdef DBM
		if (dbminit(buffer) != 0)
#endif
#ifdef NDBM
		if ((dbmfile = dbm_open(buffer, O_RDONLY, 0)) == NULL)
#endif
		{
			log(L_ERR|L_CONS, "cannot open dbm file %s",
				buffer);
			return 0;
		}

		r = dbm_find(name, request_pairs, check_pairs, reply_pairs);
		if (r > 0) found = 1;
		if (r <= 0 || fallthrough(*reply_pairs)) {

			pairdelete(reply_pairs, PW_FALL_THROUGH);

			sprintf(buffer, "DEFAULT");
			i = 0;
			while ((r = dbm_find(buffer, request_pairs,
			       check_pairs, reply_pairs)) >= 0 || i < 2) {
				if (r > 0) {
					found = 1;
					if (!fallthrough(*reply_pairs))
						break;
					pairdelete(reply_pairs,PW_FALL_THROUGH);
				}
				sprintf(buffer, "DEFAULT%d", i++);
			}
		}
#ifdef DBM
		dbmclose();
#endif
#ifdef NDBM
		dbm_close(dbmfile);
#endif
	} else
	/*
	 *	Note the fallthrough through the #endif.
	 */
#endif

	for(pl = users; pl; pl = pl->next) {

		if (strcmp(name, pl->name) && strcmp(pl->name, "DEFAULT"))
			continue;

		if (paircmp(request_pairs, pl->check) == 0) {
			DEBUG2("  users: Matched %s at %d",
			       pl->name, pl->lineno);
			found = 1;
			check_tmp = paircopy(pl->check);
			reply_tmp = paircopy(pl->reply);
			pairmove(reply_pairs, &reply_tmp);
			pairmove(check_pairs, &check_tmp);
			pairfree(reply_tmp);
			pairfree(check_tmp);

			/*
			 *	Fallthrough?
			 */
			if (!fallthrough(pl->reply))
				break;
		}
	}

	/*
	 *	See if we succeeded.
	 */
	if (!found)
		return -1;

	/*
	 *	Fix dynamic IP address if needed.
	 */
	if ((tmp = pairfind(*reply_pairs, PW_ADD_PORT_TO_IP_ADDRESS)) != NULL){
		if (tmp->lvalue != 0) {
			tmp = pairfind(*reply_pairs, PW_FRAMED_IP_ADDRESS);
			if (tmp) {
				/*
			 	 *	FIXME: This only works because IP
				 *	numbers are stored in host order
				 *	everywhere in this program.
				 */
#ifdef ASCEND_PORT_HACK
				nas_port = ascend_port_number(nas_port);
#endif
				tmp->lvalue += nas_port;
			}
		}
		pairdelete(reply_pairs, PW_ADD_PORT_TO_IP_ADDRESS);
	}

	/*
	 *	Remove server internal parameters.
	 */
	pairdelete(reply_pairs, PW_FALL_THROUGH);

	return 0;
}

/*
 *	Copy a data field from the buffer.  Advance the buffer
 *	past the data field.
 */
static	void fieldcpy(char *string, int len, char **uptr)
{
	char	*ptr;

	ptr = *uptr;
	if(*ptr == '"') {
		ptr++;
		while(*ptr != '"' && *ptr != '\0' && *ptr != '\n') {
			if (len-- > 0)
				*string++ = *ptr++;
			else
				ptr++;
		}
		*string = '\0';
		if(*ptr == '"') {
			ptr++;
		}
		*uptr = ptr;
		return;
	}

	while(*ptr != ' ' && *ptr != '\t' && *ptr != '\0' && *ptr != '\n' &&
						*ptr != '=' && *ptr != ',') {
			if (len-- > 0)
				*string++ = *ptr++;
			else
				ptr++;
	}
	*string = '\0';
	*uptr = ptr;
	return;
}

#define PARSE_MODE_NAME		0
#define PARSE_MODE_EQUAL	1
#define PARSE_MODE_VALUE	2
#define PARSE_MODE_INVALID	3

/*
 *	Parse the buffer to extract the attribute-value pairs.
 */
int userparse(char *buffer, VALUE_PAIR **first_pair)
{
	int		mode;
	int		x;
	char		attrstr[256];
	char		valstr[256];
	char		*s;
	DICT_ATTR	*attr = NULL;
	DICT_VALUE	*dval;
	VALUE_PAIR	*pair, *pair2;
	struct tm	*tm;
	time_t		timeval;
	int		operator;
	int		rcode;

	rcode = USERPARSE_EOS;
	mode = PARSE_MODE_NAME;
	while(*buffer != '\n' && *buffer != '\0') {

		if (*buffer == ',') {
			rcode = USERPARSE_COMMA;
		}

		if(*buffer == ' ' || *buffer == '\t' || *buffer == ',') {
			buffer++;
			continue;
		}

		rcode = USERPARSE_EOS;
		switch(mode) {

		case PARSE_MODE_NAME:
			/* Attribute Name */
			fieldcpy(attrstr, sizeof(attrstr), &buffer);
			if((attr = dict_attrfind(attrstr)) ==
						(DICT_ATTR *)NULL) {
#if 0 /* Be quiet. */
				log(L_ERR|L_CONS, "unknown attribute %s",
					attrstr);
#endif
				return(-1);
			}
			mode = PARSE_MODE_EQUAL;
			break;

		case PARSE_MODE_EQUAL:
			mode = PARSE_MODE_VALUE;
			/* '=' Equal sign */
			if(*buffer == '=') {
				buffer++;
				operator = PW_OPERATOR_EQUAL;
			} else
			/* '<' */
			if(*buffer == '<') {
				buffer++;
				operator = PW_OPERATOR_LESS_THAN;
			} else
			/* '>' */
			if(*buffer == '>') {
				buffer++;
				operator = PW_OPERATOR_GREATER_THAN;
			} else
			/* '!=' */
			if(memcmp(buffer, "!=", 2) == 0) {
				buffer += 2;
				operator = PW_OPERATOR_NOT_EQUAL;
			} else
			/* '<=' */
			if(memcmp(buffer, "<=", 2) == 0) {
				buffer += 2;
				operator = PW_OPERATOR_LESS_EQUAL;
			} else
			/* '>=' */
			if(memcmp(buffer, ">=", 2) == 0) {
				buffer += 2;
				operator = PW_OPERATOR_GREATER_EQUAL;
			}
			else {
				log(L_ERR|L_CONS, "expected `='");
				return(-1);
			}
			break;

		case PARSE_MODE_VALUE:
			/* Value */
			fieldcpy(valstr, sizeof(valstr), &buffer);
			if((pair = (VALUE_PAIR *)malloc(sizeof(VALUE_PAIR))) ==
						(VALUE_PAIR *)NULL) {
				log(L_CONS|L_ERR, "no memory");
				exit(1);
			}
			/* No need for strncpy - same size */
			strcpy(pair->name, attr->name);
			pair->attribute = attr->value;
			pair->type = attr->type;
			pair->operator = operator;

			/* _Always_ copy value as string. */
			strNcpy(pair->strvalue, valstr, sizeof(pair->strvalue));

			switch(pair->type) {

			case PW_TYPE_STRING:
				/* Pair->strvalue has been set already */
				pair->length = strlen(pair->strvalue);
				break;

			case PW_TYPE_INTEGER:
				/*
				 *	If it starts with a digit, it must
				 *	be a number (or a range).
				 */
				if (isdigit(*valstr)) {
					pair->lvalue = atoi(valstr);
					pair->length = 4;
				}
				else if((dval = dict_valfind(valstr)) ==
							(DICT_VALUE *)NULL) {
					free(pair);
					log(L_ERR|L_CONS, "unknown value %s",
						valstr);
					return(-1);
				}
				else {
					pair->lvalue = dval->value;
					pair->length = 4;
				}
				break;

			case PW_TYPE_IPADDR:
				if (pair->attribute != PW_FRAMED_IP_ADDRESS) {
					pair->lvalue = get_ipaddr(valstr);
					break;
				}

				/*
				 *	We allow a "+" at the end to
				 *	indicate that we should add the
				 *	portno. to the IP address.
				 */
				x = 0;
				if (valstr[0]) {
					for(s = valstr; s[1]; s++)
						;
					if (*s == '+') {
						*s = 0;
						x = 1;
					}
				}
				pair->lvalue = get_ipaddr(valstr);
				pair->length = 4;

				/*
				 *	Add an extra (hidden) attribute.
				 */
				if((pair2 = malloc(sizeof(VALUE_PAIR))) ==
							NULL) {
					log(L_CONS|L_ERR, "no memory");
					exit(1);
				}
				strcpy(pair2->name, "Add-Port-To-IP-Address");
				pair2->attribute = PW_ADD_PORT_TO_IP_ADDRESS;
				pair2->type = PW_TYPE_INTEGER;
				pair2->lvalue = x;
				pair2->length = 4;
				pairadd(first_pair, pair2);
				break;

			case PW_TYPE_DATE:
				timeval = time(0);
				tm = localtime(&timeval);
				user_gettime(valstr, tm);
#ifdef TIMELOCAL
				pair->lvalue = (UINT4)timelocal(tm);
#else /* TIMELOCAL */
				pair->lvalue = (UINT4)mktime(tm);
#endif /* TIMELOCAL */
				pair->length = 4;
				break;

			default:
				free(pair);
#if 0 /* Yeah yeah */
				log(L_ERR|L_CONS, "unknown attr. type %d",
					pair->type);
#endif
				return(-1);
			}
			pairadd(first_pair, pair);
			mode = PARSE_MODE_NAME;
			break;

		default:
			mode = PARSE_MODE_NAME;
			break;
		}
	}

	/*
	 *	Double-check that we've parsed a complete Attribute=value
	 */
	if (mode != PARSE_MODE_NAME) {
		log(L_ERR|L_CONS, "Got end of line while still parsing last attribute");
		return(-1);
	}

	return rcode;
}

/*
 *	Match a username with a wildcard expression.
 *	Is very limited for now.
 */
static int matches(char *name, PAIR_LIST *pl, char *matchpart, int matchlen)
{
	int len, wlen;
	int ret = 0;
	char *wild = pl->name;
	VALUE_PAIR *tmp;

	/*
	 *	We now support both:
	 *
	 *		DEFAULT	Prefix = "P"
	 *
	 *	and
	 *		P*
	 */
	if ((tmp = pairfind(pl->check, PW_PREFIX)) != NULL ||
	    (tmp = pairfind(pl->check, PW_SUFFIX)) != NULL) {

		if (strncmp(pl->name, "DEFAULT", 7) == 0 ||
		    strcmp(pl->name, name) == 0)
			return !presufcmp(tmp, name, matchpart, matchlen);
	}

	/*
	 *	Shortcut if there's no '*' in pl->name.
	 */
	if (strchr(pl->name, '*') == NULL &&
	    (strncmp(pl->name, "DEFAULT", 7) == 0 ||
	     strcmp(pl->name, name) == 0)) {
		strNcpy(matchpart, name, matchlen);
		return 1;
	}

	/*
	 *	Normally, we should return 0 here, but we
	 *	support the old * stuff.
	 *	FIXME: this doesn't support realsm yet, while
	 *	presufcmp does!
	 */
	len = strlen(name);
	wlen = strlen(wild);

	if (len == 0 || wlen == 0) return 0;

	if (wild[0] == '*') {
		wild++;
		wlen--;
		if (wlen <= len && strcmp(name + (len - wlen), wild) == 0) {
			strNcpy(matchpart, name, matchlen);
			matchpart[len - wlen] = 0;
			ret = 1;
		}
	} else if (wild[wlen - 1] == '*') {
		if (wlen <= len && strncmp(name, wild, wlen - 1) == 0) {
			strNcpy(matchpart, name + wlen - 1, matchlen);
			ret = 1;
		}
	}

	return ret;
}




/*
 *	Add hints to the info sent by the terminal server
 *	based on the pattern of the username.
 */
int hints_setup(VALUE_PAIR *request_pairs)
{
	char		newname[AUTH_STRING_LEN];
	char		*name;
	VALUE_PAIR	*add;
	VALUE_PAIR	*last;
	VALUE_PAIR	*tmp;
	PAIR_LIST	*i;
	int		do_strip;
#if defined(NT_DOMAIN_HACK) || defined(SPECIALIX_JETSTREAM_HACK)
	char		*ptr;
#endif

	if (hints == NULL)
		return 0;

	/* 
	 *	Check for valid input, zero length names not permitted 
	 */
	if ((tmp = pairfind(request_pairs, PW_USER_NAME)) == NULL)
		name = NULL;
	else
		name = tmp->strvalue;

	if (name == NULL || name[0] == 0)
		/*
		 *	Will be complained about later.
		 */
		return 0;

#ifdef NT_DOMAIN_HACK
	/*
	 *	Windows NT machines often authenticate themselves as
	 *	NT_DOMAIN\username. Try to be smart about this.
	 *	Note that sizeof(newname) == sizeof(name)
	 *
	 *	FIXME: should we handle this as a REALM ?
	 */
	if ((ptr = strchr(name, '\\')) != NULL) {
		strncpy(newname, ptr + 1, sizeof(newname));
		newname[sizeof(newname) - 1] = 0;
		strcpy(name, newname);
	}
#endif /* NT_DOMAIN_HACK */

#ifdef SPECIALIX_JETSTREAM_HACK
	/*
	 *	Specialix Jetstream 8500 24 port access server.
	 *	If the user name is 10 characters or longer, a "/"
	 *	and the excess characters after the 10th are
	 *	appended to the user name.
	 *
	 *	Reported by Lucas Heise <root@laonet.net>
	 */
	if (strlen(name) > 10 && name[10] == '/') {
		for (ptr = name + 11; *ptr; ptr++)
			*(ptr - 1) = *ptr;
		*(ptr - 1) = 0;
	}
#endif

	/*
	 *	Small check: if Framed-Protocol present but Service-Type
	 *	is missing, add Service-Type = Framed-User.
	 */
	if (pairfind(request_pairs, PW_FRAMED_PROTOCOL) != NULL &&
	    pairfind(request_pairs, PW_SERVICE_TYPE) == NULL) {
		if ((tmp = malloc(sizeof(VALUE_PAIR))) != NULL) {
			memset(tmp, 0, sizeof(VALUE_PAIR));
			strcpy(tmp->name, "Service-Type");
			tmp->attribute = PW_SERVICE_TYPE;
			tmp->type = PW_TYPE_INTEGER;
			tmp->lvalue = PW_FRAMED_USER;
			pairmove(&request_pairs, &tmp);
		}
	}

	for (i = hints; i; i = i->next) {
		if (matches(name, i, newname, sizeof(newname))) {
			DEBUG2("  hints: Matched %s at %d",
			       i->name, i->lineno);
			break;
		}
	}

	if (i == NULL) return 0;

	add = paircopy(i->reply);

#if 0 /* DEBUG */
	printf("In hints_setup, newname is %s\n", newname);
#endif

	/*
	 *	See if we need to adjust the name.
	 */
	do_strip = 1;
	if ((tmp = pairfind(i->reply, PW_STRIP_USERNAME)) != NULL
	     && tmp->lvalue == 0)
		do_strip = 0;
	if ((tmp = pairfind(i->check, PW_STRIP_USERNAME)) != NULL
	     && tmp->lvalue == 0)
		do_strip = 0;

	if (do_strip) {
		tmp = pairfind(request_pairs, PW_USER_NAME);
		if (tmp) {
			/* Same size */
			strcpy(tmp->strvalue, newname);
			tmp->length = strlen(tmp->strvalue);
		}
	}

	/*
	 *	Now add all attributes to the request list,
	 *	except the PW_STRIP_USERNAME one.
	 */
	pairdelete(&add, PW_STRIP_USERNAME);
	for(last = request_pairs; last && last->next; last = last->next)
		;
	if (last) last->next = add;

	return 0;
}

/*
 *	See if the huntgroup matches.
 */
static int huntgroup_match(VALUE_PAIR *request_pairs, char *huntgroup)
{
	PAIR_LIST	*i;

	for (i = huntgroups; i; i = i->next) {
		if (strcmp(i->name, huntgroup) != 0)
			continue;
		if (paircmp(request_pairs, i->check) == 0) {
			DEBUG2("  huntgroups: Matched %s at %d",
			       i->name, i->lineno);
			break;
		}
	}

	return (i != NULL);
}


/*
 *	See if we have access to the huntgroup.
 *	Returns  0 if we don't have access.
 *	         1 if we do have access.
 *	        -1 on error.
 */
int huntgroup_access(VALUE_PAIR *request_pairs)
{
	PAIR_LIST	*i;
	int		r = 1;

	if (huntgroups == NULL)
		return 1;

	for(i = huntgroups; i; i = i->next) {
		/*
		 *	See if this entry matches.
		 */
		if (paircmp(request_pairs, i->check) != 0)
			continue;

		r = 0;
		if (hunt_paircmp(request_pairs, i->reply) == 0) {
			r = 1;
		}
		break;
	}

	return r;
}


/*
 *	Debug code.
 */
#if 0
static void debug_pair_list(PAIR_LIST *pl)
{
	VALUE_PAIR *vp;

	while(pl) {
		printf("Pair list: %s\n", pl->name);
		printf("** Check:\n");
		for(vp = pl->check; vp; vp = vp->next) {
			printf("    ");
			fprint_attr_val(stdout, vp);
			printf("\n");
		}
		printf("** Reply:\n");
		for(vp = pl->reply; vp; vp = vp->next) {
			printf("    ");
			fprint_attr_val(stdout, vp);
			printf("\n");
		}
		pl = pl->next;
	}
}
#endif

/*
 *	Free a CLIENT list.
 */
static void clients_free(CLIENT *cl)
{
	CLIENT *next;

	while(cl) {
		next = cl->next;
		free(cl);
		cl = next;
	}
}


/*
 *	Read the clients file.
 */
int read_clients_file(char *file)
{
	FILE	*fp;
	char	buffer[256];
	char	hostnm[128];
	char	secret[32];
	char	shortnm[32];
	int	lineno = 0;
	CLIENT	*c;

	clients_free(clients);
	clients = NULL;

	if ((fp = fopen(file, "r")) == NULL) {
		log(L_CONS|L_ERR, "cannot open %s", file);
		return -1;
	}
	while(fgets(buffer, 256, fp) != NULL) {
		lineno++;
		if (buffer[0] == '#' || buffer[0] == '\n')
			continue;

		shortnm[0] = 0;
		if (sscanf(buffer, "%127s%31s%31s", hostnm, secret, shortnm) < 2) {
			log(L_ERR, "%s[%d]: syntax error", file, lineno);
			continue;
		}
		if (shortnm[0] == '#') shortnm[0] = 0;

		if ((c = malloc(sizeof(CLIENT))) == NULL) {
			log(L_CONS|L_ERR, "%s[%d]: out of memory",
				file, lineno);
			return -1;
		}

		c->ipaddr = get_ipaddr(hostnm);
		strNcpy(c->secret, secret, sizeof(c->secret));
		strNcpy(c->shortname, shortnm, sizeof(c->shortname));
		strNcpy(c->longname, ip_hostname(c->ipaddr),
			sizeof(c->longname));

		c->next = clients;
		clients = c;
	}
	fclose(fp);

	return 0;
}


/*
 *	Find a client in the CLIENTS list.
 */
CLIENT *client_find(UINT4 ipaddr)
{
	CLIENT *cl;

	for(cl = clients; cl; cl = cl->next)
		if (ipaddr == cl->ipaddr)
			break;

	return cl;
}


/*
 *	Find the name of a client (prefer short name).
 */
char *client_name(UINT4 ipaddr)
{
	CLIENT *cl;

	if ((cl = client_find(ipaddr)) != NULL) {
		if (cl->shortname[0])
			return cl->shortname;
		else
			return cl->longname;
	}
	return ip_hostname(ipaddr);
}

/*
 *	Free a NAS list.
 */
static void nas_free(NAS *cl)
{
	NAS *next;

	while(cl) {
		next = cl->next;
		free(cl);
		cl = next;
	}
}

/*
 *	Free a REALM list.
 */
static void realm_free(REALM *cl)
{
	REALM *next;

	while(cl) {
		next = cl->next;
		free(cl);
		cl = next;
	}
}

/*
 *	Read the nas file.
 */
int read_naslist_file(char *file)
{
	FILE	*fp;
	char	buffer[256];
	char	hostnm[128];
	char	shortnm[32];
	char	nastype[32];
	int	lineno = 0;
	NAS	*c;

	nas_free(naslist);
	naslist = NULL;

	if ((fp = fopen(file, "r")) == NULL) {
		log(L_CONS|L_ERR, "cannot open %s", file);
		return -1;
	}
	while(fgets(buffer, 256, fp) != NULL) {
		lineno++;
		if (buffer[0] == '#' || buffer[0] == '\n')
			continue;
		nastype[0] = 0;
		if (sscanf(buffer, "%127s%31s%31s", hostnm, shortnm, nastype) < 2) {
			log(L_ERR, "%s[%d]: syntax error", file, lineno);
			continue;
		}
		if ((c = malloc(sizeof(NAS))) == NULL) {
			log(L_CONS|L_ERR, "%s[%d]: out of memory",
				file, lineno);
			return -1;
		}

		c->ipaddr = get_ipaddr(hostnm);
		strNcpy(c->nastype, nastype, sizeof(c->nastype));
		strNcpy(c->shortname, shortnm, sizeof(c->shortname));
		strNcpy(c->longname, ip_hostname(c->ipaddr),
			sizeof(c->longname));

		c->next = naslist;
		naslist = c;
	}
	fclose(fp);

	return 0;
}

/*
 *	Read the realms file.
 */
int read_realms_file(char *file)
{
	FILE	*fp;
	char	buffer[256];
	char	realm[32];
	char	hostnm[128];
	char	opts1[32], opts2[32];
	char	*p;
	int	lineno = 0;
	REALM	*c;

	realm_free(realms);
	realms = NULL;

	if ((fp = fopen(file, "r")) == NULL) {
#if 1 /* For now - realms file is not obligatory */
		return 0;
#else
		log(L_CONS|L_ERR, "cannot open %s", file);
		return -1;
#endif
	}
	while(fgets(buffer, 256, fp) != NULL) {
		lineno++;
		if (buffer[0] == '#' || buffer[0] == '\n')
			continue;
		opts1[0] = opts2[0] = 0;
		if (sscanf(buffer, "%31s%127s%31s%31s",
		    realm, hostnm, opts1, opts2) < 2) {
			log(L_ERR, "%s[%d]: syntax error", file, lineno);
			continue;
		}
		if ((c = malloc(sizeof(REALM))) == NULL) {
			log(L_CONS|L_ERR, "%s[%d]: out of memory",
				file, lineno);
			return -1;
		}

		if ((p = strchr(hostnm, ':')) != NULL) {
			*p++ = 0;
			c->auth_port = atoi(p);
			c->acct_port = c->auth_port + 1;
		} else {
			c->auth_port = auth_port;
			c->acct_port = acct_port;
		}
		if (strcmp(hostnm, "LOCAL") != 0)
			c->ipaddr = get_ipaddr(hostnm);
		strNcpy(c->realm, realm, sizeof(c->realm));
		strNcpy(c->server, hostnm, sizeof(c->server));
		/* Yes we could do this more intelligently */
		c->striprealm = strcmp(opts1, "nostrip") != 0 &&
				strcmp(opts2, "nostrip") != 0;
		c->dohints    = strcmp(opts1, "hints") == 0 ||
				strcmp(opts2, "hints") == 0;
		c->next = realms;
		realms = c;
	}
	fclose(fp);

	return 0;
}

/*
 *	Find a realm in the REALM list.
 */
REALM *realm_find(char *realm)
{
	REALM *cl;

	for(cl = realms; cl; cl = cl->next)
		if (strcmp(cl->realm, realm) == 0)
			break;
	if (cl) return cl;
	for(cl = realms; cl; cl = cl->next)
		if (strcmp(cl->realm, "DEFAULT") == 0)
			break;
	return cl;
}



/*
 *	Find a nas in the NAS list.
 */
NAS *nas_find(UINT4 ipaddr)
{
	NAS *cl;

	for(cl = naslist; cl; cl = cl->next)
		if (ipaddr == cl->ipaddr)
			break;

	return cl;
}


/*
 *	Find the name of a nas (prefer short name).
 */
char *nas_name(UINT4 ipaddr)
{
	NAS *cl;

	if ((cl = nas_find(ipaddr)) != NULL) {
		if (cl->shortname[0])
			return cl->shortname;
		else
			return cl->longname;
	}
	return ip_hostname(ipaddr);
}

/*
 *	Find the name of a nas (prefer short name) based on the request.
 *	FIXME: the implementation in acct.c:rad_accounting_orig() is better.
 */
char *nas_name2(AUTH_REQ *authreq)
{
	UINT4	ipaddr;
	NAS	*cl;
	VALUE_PAIR	*pair;

	if ((pair = pairfind(authreq->request, PW_NAS_IP_ADDRESS)) != NULL)
		ipaddr = pair->lvalue;
	else
		ipaddr = authreq->ipaddr;

	if ((cl = nas_find(ipaddr)) != NULL) {
		if (cl->shortname[0])
			return cl->shortname;
		else
			return cl->longname;
	}
	return ip_hostname(ipaddr);
}

/*
 *	Return a short string showing the terminal server, port
 *	and calling station ID.
 */
char *auth_name(AUTH_REQ *authreq, int do_cli)
{
	static char	buf[300];
	VALUE_PAIR	*cli;
	VALUE_PAIR	*pair;
	int		port = 0;

	if ((cli = pairfind(authreq->request, PW_CALLING_STATION_ID)) == NULL)
		do_cli = 0;
	if ((pair = pairfind(authreq->request, PW_NAS_PORT_ID)) != NULL)
		port = pair->lvalue;

	sprintf(buf, "from nas %.128s/S%d%s%.128s", nas_name2(authreq), port,
		do_cli ? " cli " : "", do_cli ? cli->strvalue : "");

	return buf;
}

#if USE_DBM
/*
 *	See if a potential DBM file is present.
 */
static int checkdbm(char *users, char *ext)
{
	char buffer[256];
	struct stat st;

	sprintf(buffer, "%.200s%.50s", users, ext);

	return stat(buffer, &st);
}
#endif


/*
 *	(Re-) read the configuration files.
 */
int read_config_files()
{
	char buffer[256];

	pairlist_free(&users);
	pairlist_free(&huntgroups);
	pairlist_free(&hints);

	sprintf(buffer, "%.200s/%.50s", radius_dir, RADIUS_USERS);
#if USE_DBM
	if (!use_dbm && users &&
		(checkdbm(users->name, ".dir") == 0 ||
		 checkdbm(users->name, ".db") == 0))
			log(L_INFO|L_CONS, "DBM files found but no -b flag given - NOT using DBM");
#endif
	if (!use_dbm && (users = file_read(buffer, 1)) == NULL)
		return -1;

	sprintf(buffer, "%.200s/%.50s", radius_dir, RADIUS_HUNTGROUPS);
	huntgroups = file_read(buffer, 0);
	sprintf(buffer, "%.200s/%.50s", radius_dir, RADIUS_HINTS);
	hints      = file_read(buffer, 0);
	sprintf(buffer, "%.200s/%.50s", radius_dir, RADIUS_CLIENTS);
	if (read_clients_file(buffer) < 0)
		return -1;
	sprintf(buffer, "%.200s/%.50s", radius_dir, RADIUS_NASLIST);
	if (read_naslist_file(buffer) < 0)
		return -1;
	sprintf(buffer, "%.200s/%.50s", radius_dir, RADIUS_REALMS);
	if (read_realms_file(buffer) < 0)
		return -1;

#if XTRADIUS
	sprintf(buffer, "%s/%s", radius_dir, XTRADIUS_PARAMS);
	read_execparams_file(buffer);
#endif

	if (cache_passwd) {
		log(L_INFO, "HASH:  Reinitializing hash structures "
			"and lists for caching...");
		if(buildHashTable() < 0) {
			log(L_ERR, "HASH:  unable to create user "
				"hash table.  disable caching and run debugs");
			return -1;
		}
		if (buildGrpList() < 0) {
			log(L_ERR, "HASH:  unable to cache groups file.  "
				"disable caching and run debugs");
			return -1;
		}
	}

	return 0;
}

#if XTRADIUS
/*
 *	Read the execparams file.
 */
int read_execparams_file(char *file)
{
	FILE	*fp;
	char	buffer[256];
	char	lparamchar[128];
	char	lparamcode[128];
	int	lineno = 0, counter = 0;
	DICT_ATTR	*attr = NULL;

	if ((fp = fopen(file, "r")) == NULL) {
		log(L_CONS|L_ERR, "cannot open %s", file);
		return -1;
	}
	while(fgets(buffer, 256, fp) != NULL) {
//		log(L_INFO, "%s[%d]: %s", file, lineno,buffer);
		lineno++;
		if (buffer[0] == '#' || buffer[0] == '\n')
			continue;

		if (sscanf(buffer, "%s%s", lparamcode,lparamchar) < 1) {
			log(L_ERR, "%s[%d]: syntax error", file, lineno);
			continue;
		}

		if((attr = dict_attrfind(lparamcode)) == (DICT_ATTR *)NULL) {
			log(L_ERR|L_CONS, "unknown attribute %s", lparamcode);
		}
		else {
			paramchar[counter] = lparamchar[0];
			paramcode[counter] = attr->value;
//			log(L_INFO, "%c - %i",paramchar[counter],paramcode[counter]);
			counter++;
		}
	}
	paramchar[counter]='\0';
	fclose(fp);

	return 0;
}
#endif
