/* 
 * ncrypt.c - Simple Nomad <thegnome@nmrc.org>
 *
 * NMRC file encryptor/decryptor - based off of AES code for Rijndael and
 * Serpent found on www.farm9.com. The original farm9 work done by Joh 
 * Johannsen <jojo@farm9.com>, which was based off of work done by Gary 
 * Rancier <mephis5@softhome.net>. The Twofish code was taken from Doug 
 * Whiting's C implementation from www.counterpane.com.
 * 
 * $Id: ncrypt.c,v 1.2 2002/12/31 20:15:27 s-nomad Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <unistd.h>
#include "ncrypt.h"
#include "wipe_file.h"
#include "rand_gen.h"
#include "sha1.h"
#include "mem.h"


/**********
 * globals
 **********/
keyInstanceT keyInstT;
keyInstanceS keyInstS;
keyInstanceR keyInstR;
cipherInstance cipherInst;
cipherInstanceT cipherInstT;

/**********
 * routines
 **********/

/*
 * prog usage
 */
static void usage(int exit_code, char *prompt) 
{
  fprintf(stderr,"ncrypt v%s\n",VER);
  if (exit_code != 0) fprintf(stderr,"** ERROR: %s\n",prompt);
  fprintf(stderr,"Usage:\n");
  fprintf(stderr,"ncrypt -e|-d -i <infile> -o <outfile> -a <algo> (options)\n");
  fprintf(stderr,"  -e is encryption, -d is decryption\n"); 
  fprintf(stderr,"  -a algorithms are 1-AES (Rijndael), 2-Serpent, 3-Twofish (default is AES)\n");
  fprintf(stderr,"Options include:\n");
  fprintf(stderr,"  -w wipe input file without encrypting (be careful!)\n");
  fprintf(stderr,"  -z wipe infile when encrypting (be careful!)\n");
  fprintf(stderr,"  -V print version info and exit\n\n");
  exit(exit_code);
}

/*
 * encryption routine
 */
void encrypt_file(int algo, int fsize, FILE *fpin, FILE *fpout)
{ 
  BYTE block[4 * 4];
  int size = 0; 

  while (1) 
  {
    guaranteed_memset(block, 0, 16);
    size = fread(block, 1, 16, fpin);
        
    if (size > 0) 
    { 
      switch(algo)
      {
        case 1: /* rijndeal */
          blockEncryptR(&cipherInst, &keyInstR, block, 128, block); break;
        case 2: /* serpent */
          blockEncryptS(&cipherInst, &keyInstS, block, 128, block); break;
        case 3: /* twofish */
          blockEncryptT(&cipherInstT, &keyInstT, block, 128, block); break;
      }

      /* note that on the last block, we are padding to 16 bytes */
      if (fwrite(block, 16, 1, fpout) != 1) 
      {
        fprintf(stderr,"** ERROR writing data\n");
        exit(-1);
      }
    }
    else break;
  } 
}

/*
 * decryption routine
 */
void decrypt_file(int algo, int fsize, FILE *fpin, FILE *fpout)
{
  BYTE block[4 * 4];
  int i, writesize = 0, size = 0; 
  
  while (fsize > 0) 
  {
    guaranteed_memset(block, 0, 16);
    size = fread(block, 1, 16, fpin);
    if (size > 0)
    { 
      switch(algo)
      {
        case 1: /* rijndeal */
          blockDecryptR(&cipherInst, &keyInstR, block, 128, block); break;
        case 2: /* serpent */
          blockDecryptS(&cipherInst, &keyInstS, block, 128, block); break;
        case 3: /* twofish */
          blockDecryptT(&cipherInstT, &keyInstT, block, 128, block); break;
      } 

      /* on last block, eliminate the padding if it exists */
      if(fsize == 16) 
      {
        for(i=0; i<16; i++) 
          if(block[i] != 0) writesize = i+1; 
      }
      else writesize = 16;
      if (fwrite(block, writesize, 1, fpout) != 1)
      {
        fprintf(stderr,"** ERROR writing data\n");
        exit(-1);
      } 
      if (writesize != 16) break;
      fsize -= size;
    }
    else break;
  } 
}

int main(int argc, char **argv) 
{
  int i,r=0,c,lock=0,mode=0,input=0,output=0,algo=1,wipe=0,wiped=0,len=0; 
  char *pass, *pass2, *ifile=0, *ofile=0;
  BYTE direction=0;
  BYTE keyMaterial[320]; 
  struct stat statbuf;
  FILE* fpin = NULL;
  FILE* fpout = NULL;

  if(getuid()==0)
  {
    lock = mlockall(MCL_CURRENT|MCL_FUTURE);
    if(lock==-1)
    {
      fprintf(stderr,"Running as root but unable to lock memory to prevent paging to swap\n");
      exit(-1);
    }
    lock=1;
  }
  
  /* process args */
  while((c = getopt(argc, argv, "a:edi:o:h?zwV")) != EOF)
  {
    switch (c)
    {
      case 'a': algo = atoi(optarg); break;
      case 'e': direction = DIR_ENCRYPT; mode++; break;
      case 'd': direction = DIR_DECRYPT; mode++; break;
      case 'i': ifile = optarg; input++; break;
      case 'o': ofile = optarg; output++; break;
      case 'z': wipe = 1; break;
      case 'w': direction = WIPEONLY; break; 
      case 'V': fprintf(stdout,"\n\tThis is Ncrypt version %s\n\n",VER); return 0;
      case 'h':
      case '?': usage(0,NULL); 
    }
  }

  /* begin post arg processing, checking for errors */
  if ((mode != 1) && (direction != WIPEONLY))
    usage(-1,"You have to specify either -e or -d"); 

  if ((direction == DIR_DECRYPT) && (wipe))
    usage(-1,"You cannot wipe during decryption mode");

  if ((algo < 1) || (algo > 3)) 
    usage(-1,"Unsupported algorithm"); 

  if ((wipe) && (direction == WIPEONLY))
    usage(-1,"You cannot use both -z and -w");

  /* this implementation of twofish only uses 128 len keys */
  if (algo == 3) len = 128;
  else len = 256;

  if (!input)
    usage(-1,"You have to specify an input file");

  if (stat(ifile,&statbuf) == -1)
  {
    fprintf(stderr,"** ERROR: Input file %s not found\n",ifile);
    exit(-1);
  } 
  fpin = fopen(ifile, "rb");
  if (fpin == NULL)
  { 
    fprintf(stderr,"Problem with %s\n",ifile);
    usage(-1,"Cannot read input file");
  }
  if (direction != WIPEONLY)
  {
    if (!output) usage(-1,"You have to specify an output file");

    fpout = fopen(ofile, "wb");
    if (fpout == NULL) 
    {
      fprintf(stderr,"Problem with %s\n",ofile);
      usage(-1,"Cannot open output file");
    }
  }
  /* end post arg processing */

  if(lock) fprintf(stderr,"NOTE: Running as root and memory is locked from paging to disk\n");
  else fprintf(stderr,"WARNING: Running unprivileged and memory is not locked from paging to disk\n");

  if (direction != WIPEONLY)
  {
    /* if we are encrypting, ask for password twice */
    if (direction == DIR_ENCRYPT)
    {
      i=0;
      while(1)
      {
        if(i>2)
        {
          fprintf(stderr,"The password must be entered in correctly.\n");
          exit(-1);
        } 
        pass = read_pass("Enter encryption password: ");
        if(pass == NULL) exit(-1);
        pass2 = read_pass("Re-enter encryption password: ");
        if(pass2 == NULL) exit(-1);
        if(strcmp(pass,pass2))
        {
          fprintf(stderr,"Passwords do not match!\n");
          i++;
          continue;
        }
        /* erase one of the plaintext password instances from memory
	 * as soon as we check the two instances against each other
	 */
        // while(*pass2) *pass2++ = 0;
	guaranteed_memset(pass2,0,strlen(pass));
        break;
      }
    }

    /* else just ask once */
    else 
      pass = read_pass("Enter decryption password: "); 

    /* make plaintext password into a SHA1 hash */
    extend_mat(hash_string_with_sha1(pass), &keyMaterial[0], len);

    /* after hashing, erase the plaintext password instance from memory */
    // while(*pass) *pass++ = 0;
    guaranteed_memset(pass,0,strlen(pass));

    switch(algo)
    {
      case 1: /* rijndael */
        r = makeKeyR(&keyInstR, direction, len, keyMaterial); break;
      case 2: /* serpent */
        r = makeKeyS(&keyInstS, direction, len, keyMaterial); break;
      case 3: /* twofish */
        r = makeKeyT(&keyInstT, direction, len, NULL);
        if (r != TRUE) break;
        for(i=0; i<(len/32); i++) keyInstT.key32[i] = keyMaterial[i];
        reKey(&keyInstT); 
        break;
    } 

    /* we've made our key from the hash, so remove the hash from memory */
    guaranteed_memset(keyMaterial,0,len);

    if (r != TRUE) 
    {
      fprintf(stderr,"** ERROR: On makeKey: %d\n", r);
      exit(-1);
    }
  
    /* init the algo we are using */
    switch(algo)
    {
      case 1: /* rijndael */
        r = cipherInitR(&cipherInst, MODE_CBC, NULL); break;
      case 2: /* serpent */
        r = cipherInitS(&cipherInst, MODE_ECB, NULL); break;
      case 3: /* twofish */
        r = cipherInitT(&cipherInstT, MODE_CBC, NULL); break;
    }

    if (r != TRUE) 
    {
      fprintf(stderr,"** ERROR: On cipherInit: %d\n", r);
      exit(-1);
    }
  }

  switch(direction)
  {
    case DIR_ENCRYPT: /* encryption */
      encrypt_file(algo, (int)statbuf.st_size, fpin, fpout);
      fclose(fpout);
      fclose(fpin);
      /* if we are wiping, wipe the file */
      if(wipe) wiped = wipe_file(ifile); 
      break; 
    case DIR_DECRYPT: /* decryption */
      decrypt_file(algo, (int)statbuf.st_size, fpin, fpout);
      fclose(fpin);
      fclose(fpout);
      break;
    case WIPEONLY: /* no crypto, just wiping a file */
      wiped = wipe_file(ifile); 
      break;
  } 

  if((wipe) || (direction == WIPEONLY))
  {
    switch(wiped)
    {
      case FILE_ERR:
	fprintf(stderr,"** ERROR: During file wipe, %s not erased\n",ifile);
	return(-1);
      case FILE_NOT_FOUND:
	fprintf(stderr,"** ERROR: File %s not found\n",ifile);
	return(-1);
      case FILE_NOT_WRITEABLE:
	fprintf(stderr,"** ERROR: File %s is read only, cannot wipe it\n",ifile);
	return(-1);
      default: exit(0);
    }
  } 

  exit(0);
}
