/* 
 * ncrypt.c - Simple Nomad <thegnome@nmrc.org>
 *          - Inertia <inertia@nmrc.org>
 *
 * NMRC file encryptor/decryptor - based off of AES code for Rijndael and
 * Serpent found on www.farm9.com. The original farm9 work done by Joh 
 * Johannsen <jojo@farm9.com>, which was based off of work done by Gary 
 * Rancier <mephis5@softhome.net>. The Twofish code was taken from Doug 
 * Whiting's C implementation from www.counterpane.com.
 * 
 * $Id: ncrypt.c,v 1.6 2003/08/08 20:28:38 s-nomad Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <unistd.h>
#include "ncrypt.h"
#include "cmdline.h"
#include "encrypt_file.h"
#include "decrypt_file.h"
#include "wipe_file.h"
#include "rand_gen.h"
#include "sha1.h"
#include "mem.h"
#include "get_encryption_password.h"
#include "get_decryption_password.h"

#ifdef __FREEBSD__
#define LOCKING 0
#else
#define LOCKING 1
#endif

/**********
 * globals
 **********/
keyInstanceT keyInstT;
keyInstanceS keyInstS;
keyInstanceR keyInstR;
cipherInstance cipherInst;
cipherInstanceT cipherInstT;
struct gengetopt_args_info ARGS;
const char* PROGNAME;

/**********
 * routines
 **********/
void initialize_crypto(char* pass)
{
    BYTE keyMaterial[320];
    BYTE direction = (ARGS.encrypt_given) ? DIR_ENCRYPT : DIR_DECRYPT;
    int r = 0;
    int i = 0;
    int len = (ARGS.twofish_given) ? 128 : 256; /* this version of twofish only uses 128 len keys */

    /* make plaintext password into a SHA1 hash */
    extend_mat(hash_string_with_sha1(pass), &keyMaterial[0], len);

    /* after hashing, erase the plaintext password instance from memory */
    guaranteed_memset(pass,0,strlen(pass));

    if (ARGS.aes_given || ARGS.rijndael_given) r = makeKeyR(&keyInstR, direction, len, keyMaterial);
    if (ARGS.serpent_given)  r = makeKeyS(&keyInstS, direction, len, keyMaterial);
    if (ARGS.twofish_given)
    {
        r = makeKeyT(&keyInstT, direction, len, NULL);
        if (r == TRUE)
        {
            for(i=0; i<(len/32); i++) keyInstT.key32[i] = keyMaterial[i];
            reKey(&keyInstT);
        }
    }

    /* we've made our key from the hash, so remove the hash from memory */
    guaranteed_memset(keyMaterial,0,len);

    if (r != TRUE)
    {
        fprintf(stderr,"** ERROR: On makeKey: %d\n", r);
        exit(-1);
    }

    /* init the algo we are using */
    if (ARGS.aes_given || ARGS.rijndael_given) r = cipherInitR(&cipherInst,  MODE_CBC, NULL);
    if (ARGS.serpent_given)  r = cipherInitS(&cipherInst,  MODE_ECB, NULL);
    if (ARGS.twofish_given)  r = cipherInitT(&cipherInstT, MODE_CBC, NULL);
    if (r != TRUE)
    {
        fprintf(stderr,"** ERROR: On cipherInit: %d\n", r);
        exit(-1);
    }
}


#ifndef WIN32
void check_for_lockable_memory()
{
    int lock = 0;
    if (getuid() == 0)
    {
        lock = mlockall(MCL_CURRENT|MCL_FUTURE);
        if (lock == -1)
        {
            fprintf(stderr,"Running as root but unable to lock memory to prevent paging to swap\n");
            exit(-1);
        }
        lock = 1;
    }
    if (ARGS.verbose_given)
    {
	if (lock)
            fprintf(stderr,"NOTE: Running as root and memory is locked from paging to disk\n");
        else
            fprintf(stderr,"WARNING: Running unprivileged and memory is not locked from paging to disk\n");
    }
}
#endif


int wipe(int wipe_mode)
{
    int i = 0;
    
    if(ARGS.input_given)
	    wipe_file(ARGS.input_arg,ARGS.verbose_given,wipe_mode);
    else for (i = 0 ; i < ARGS.inputs_num ; i++)
	wipe_file(ARGS.inputs[i],ARGS.verbose_given,wipe_mode);
    return 0;
}

int encrypt(FILE* ifp, FILE* ofp)
{
    char* pass = get_encryption_password();
    initialize_crypto(pass);
    free(pass);
    encrypt_file(ifp,ofp);
    return 0;
}



int decrypt(FILE* ifp, FILE* ofp)
{
    struct stat statbuf;
    int fsize = 0;
    char* pass = get_decryption_password();
    initialize_crypto(pass);
    free(pass);
    fstat(fileno(ifp),&statbuf);
    fsize = (int)statbuf.st_size;
    decrypt_file(ifp,fsize,ofp);
    return 0;
}



FILE* determine_input_file()
{
    FILE* ifp;
    ifp = fopen(ARGS.input_arg,"rb");
    if (ifp == NULL)
    {
      fprintf(stderr,"ERROR: cannot open file for reading: %s\n",ARGS.input_arg);
      exit(-1); 
    }
    return ifp;
}



FILE* determine_output_file()
{
    FILE* ofp;
    ofp = fopen(ARGS.output_arg,"wb");
    if (ofp == NULL)
    {
        fprintf(stderr,"ERROR: cannot open file for writing: %s\n",ARGS.output_arg);
        exit(-1); 
    }
    return ofp;
}



int main(int argc, char** argv)
{
    FILE* ifp = NULL;
    FILE* ofp = NULL;
    int rc = 0, wipe_mode = 0;

    PROGNAME = argv[0];
    if (cmdline_parser(argc,argv,&ARGS) != 0) exit(1);

    if (!ARGS.aes_given && !ARGS.rijndael_given && !ARGS.serpent_given && !ARGS.twofish_given)
	    ARGS.rijndael_given = 1;
    if (ARGS.verbose_given)
    {
        if ((ARGS.rijndael_given || ARGS.aes_given ) && ARGS.output_given)  printf("NOTE: aes (rijndael) given\n");
        if (ARGS.serpent_given  && ARGS.output_given)  printf("NOTE: serpent given\n");
        if (ARGS.twofish_given  && ARGS.output_given)  printf("NOTE: twofish given\n");
    }

#ifndef WIN32
    check_for_lockable_memory();
#endif
    
    if (ARGS.input_given) ifp = determine_input_file();
    if (ARGS.output_given) ofp = determine_output_file();
    
    if (ARGS.wipe_given_gutmann) wipe_mode = WIPE_GUTMANN;
    if (ARGS.wipe_given_military) wipe_mode = WIPE_MILITARY;
    if ((ARGS.wipe_given_gutmann) && (ARGS.wipe_given_military)) wipe_mode = WIPE_BOTH;
    if (ARGS.wipe_only)
    {
	    if(!wipe_mode) wipe_mode = WIPE_MILITARY;
	    return wipe(wipe_mode);
    }

    if (ARGS.encrypt_given) 
    {
	    rc = encrypt(ifp,ofp);
	    if (ARGS.wipe_on_encrypt && (rc == 0))
	    {
		    if(!wipe_mode) wipe_mode = WIPE_MILITARY;
		    rc = wipe(wipe_mode);
	    }
    }
    if (ARGS.decrypt_given) rc = decrypt(ifp,ofp);

    fclose(ifp);
    fclose(ofp);
    return rc;
}

