/* Overwrite - a secure deletion software
 * Copyright (C) 2000, Salvatore Sanfilippo <antirez@linuxcare.com>
 * This software is under GPL version 2 of license
 */

#include <stdio.h>
#include <string.h> /* FreeBSD bzero() (used in FD_ZERO) anti-waring */
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef __linux__
#include <linux/fs.h>
#include <linux/hdreg.h>
#endif /* __linux__ */
#include <unistd.h>
#include <sys/time.h>
#include <stdlib.h>
#include <errno.h>
#include "sha1.h"

/*** tunable defines ***/
#define DEV_RANDOM			"/dev/random"
/* /dev/random timeout, when set this consider that in the worst case
 * the user may wait ~ DEV_RANDOM_TIMEOUT*20 seconds. */
#define DEV_RANDOM_TIMEOUT		3
#define DEV_URANDOM			"/dev/urandom"
#define START_RANDOM_PASS		4
#define END_RANDOM_PASS			4
#define PATTERN_EXTENSION_FACTOR	1024
/*** --------------- ***/

#define PRNG_RESEED \
		do { \
			long timestamp; \
			timestamp = get_usec(); \
			prng_seed((char*)&timestamp, sizeof(timestamp)); \
		} while(0)

#define SWAP(x,y) do { int t; t = x; x = y; y = t; } while(0)

char prng_state[20];	/* the pseudorandom number generator state */

/* Pattern from Peter Gutmann's "Secure Deletion of Data from Magnetic
 * and Solid-State Memory".
 */

#define OVERWRITE_PATTERNS_N	27
/* WARNING: the first byte is the size of the pattern */
char *overwrite_data[] = {
	"\001\125",			/* 1  0x55 */
	"\001\252",			/* 2  0xAA */
	"\003\222\111\044",		/* 3  0x92 0x49 0x24 */
	"\003\111\044\222",		/* 4  0x49 0x24 0x92 */
	"\003\044\222\111",		/* 5  0x24 0x92 0x49 */
	"\001\000",			/* 6  0x00 */
	"\001\021",			/* 7  0x11 */
	"\001\042",			/* 8  0x22 */
	"\001\063",			/* 9  0x33 */
	"\001\104",			/* 10 0x44 */
	"\001\125",			/* 11 0x55 */
	"\001\146",			/* 12 0x66 */
	"\001\167",			/* 13 0x77 */
	"\001\210",			/* 14 0x88 */
	"\001\231",			/* 15 0x99 */
	"\001\252",			/* 16 0xAA */
	"\001\273",			/* 17 0xBB */
	"\001\314",			/* 18 0xCC */
	"\001\335",			/* 19 0xDD */
	"\001\356",			/* 20 0xEE */
	"\001\377",			/* 21 0xFF */
	"\003\222\111\044",		/* 22 0x92 0x49 0x24 */
	"\003\111\044\222",		/* 23 0x49 0x24 0x92 */
	"\003\044\222\111",		/* 24 0x24 0x92 0x49 */
	"\003\155\266\333",		/* 25 0x6D 0xB6 0xDB */
	"\003\266\333\155",		/* 26 0xB6 0xDB 0x6D */
	"\003\333\155\266",		/* 27 0xDB 0x6D 0xB6 */
	NULL
};

int opt_leave = 0;
int opt_verbose = 0;
int opt_force = 0;
int opt_device = 0;

/* prototypes */
int prng_first_reseed(void);
void prng_seed(char *sample, int size);
void prng_get(char *buf, int size);
long get_usec(void);
int physical_sync(unsigned int device);
int overwrite_random(int fd, long size);
int overwrite_pattern(int fd, long size, char *pattern);
int device_rewind(int fd, char *name);
void show_help(char *pname);

int main(int argc, char **argv)
{
	char *filename;
	int pass_sequence[OVERWRITE_PATTERNS_N];
	int k, fd, argindex = 1;
	int minusminus = 0;
	unsigned int r;
	struct stat buf;

	if (argc < 2) {
		fprintf(stderr,
			"Usage: %s [-lvfh] <filename> ... [filenameN]\n"
			"Try %s -h for more information\n",
			argv[0], argv[0]);
		exit(1);
	}

	/* here I'm very near to ascii art... */
	for (k = 1; k < argc; k++) {
		if (argv[k][0] == '-') {
			char *p = argv[k];

			if (argv[k][1] == '-') {
				k = argc;
				continue;
			}
			while(*++p) {
				switch (*p) {
				case 'l': /* doesn't unlink the file */
					printf("(l)eave option enabled\n");
					opt_leave = 1;
					break;
				case 'v': /* verbose */
					printf("(v)erbose mode enabled\n");
					opt_verbose = 1;
					break;
				case 'f': /* force -- loop mode */
					printf("(f)force option enabled\n");
					opt_force = 1;
					opt_leave = 1;
					break;
				case 'd': /* specified file is a device */
					printf("(d)evice mode enabled\n");
					opt_device = 1;
					opt_leave = 1;
					break;
				case 'h':
					show_help(argv[0]);
					exit(0);
				default:
					fprintf(stderr, "Unknown option '%c'\n",
						*p);
					show_help(argv[0]);
					exit(1);
				}
			}
		}
	}

	prng_first_reseed();
	PRNG_RESEED; /* the starting instant is a source of entropy */
	physical_sync(0); /* maybe this will increase the entropy */

	while (argv[argindex]) {
		if (!minusminus && argv[argindex][0] == '-') {
			if (argv[argindex][1] == '-')
				minusminus = 1;
			argindex++;
			if (opt_force && !argv[argindex])
				argindex = 1;
			continue;
		}

		filename = argv[argindex];
		argindex++;

		if (opt_force && !argv[argindex])
			argindex = 1;

		if ((fd = open(filename, O_WRONLY)) == -1) {
			perror("open");
			exit(1);
		}

		if (fstat(fd, &buf) == -1) {
			perror("stat");
			exit(1);
		}

		/* printf("[(%ld) %s]: ", buf.st_ino, filename); */
		printf("[%s %d bytes]: ", filename, (int)buf.st_size);
		fflush(stdout);

		/* set pass_sequence[k] = k */
		for (k = 0; k < OVERWRITE_PATTERNS_N; k++)
			pass_sequence[k] = k;

		/* shuffling */
		for (k = 0; k < OVERWRITE_PATTERNS_N; k++) {
			prng_get((char*)&r, sizeof(r));
			r %= OVERWRITE_PATTERNS_N;
			SWAP(pass_sequence[k], pass_sequence[r]);
		}

		/* overwrite with pseudorandom bytes */
		for (k = 0; k < START_RANDOM_PASS; k++) {
			if (overwrite_random(fd, buf.st_size) != 0)
				exit(1);
			physical_sync(buf.st_dev);
			if (opt_device)
				fd = device_rewind(fd, filename);
		}

		/* overwrite with pattern in pseudorandom order */
		for (k = 0; k < OVERWRITE_PATTERNS_N; k++) {
			if (overwrite_pattern(fd, buf.st_size,
			overwrite_data[k]) != 0)
				exit(1);
			physical_sync(buf.st_dev);
			if (opt_device)
				fd = device_rewind(fd, filename);
		}

		/* overwrite with pseudorandom bytes */
		for (k = 0; k < START_RANDOM_PASS; k++) {
			if (overwrite_random(fd, buf.st_size) != 0)
				exit(1);
			physical_sync(buf.st_dev);
			if (opt_device)
				fd = device_rewind(fd, filename);
		}

		printf("\n");
		close(fd);
		if (!opt_leave && unlink(filename) == -1) {
			perror("unlink");
			/* isn't a fatal error */
		}
	}
	return 0;
}

/* this function is used to rewind a device that does not support
 * the [l]seek(2)ing.
 */
int device_rewind(int fd, char *name)
{
	int newfd;

	close(fd);
	if ((newfd = open(name, O_WRONLY)) == -1) {
		perror("open in divice_rewind()");
		exit(1);
	}
	return newfd;
}

#define MIN(x,y) x < y ? x : y
int overwrite_random(int fd, long size)
{
	char randomdata[1024];

	if (opt_device)
		size = 1024;

	if (!opt_device && lseek(fd, 0, SEEK_SET) == -1) {
		perror("lseek");
		return -1;
	}

	while (size) {
		prng_get(randomdata, 1024);
		if (write(fd, randomdata, MIN(1024, size)) == -1) {
			if (opt_device && errno == ENOSPC)
				break;
			perror("write");
			return -1;
		}
		if (!opt_device)
			size -= MIN(1024, size);
	}

	printf("r");
	fflush(stdout);

	return 0;
}

int overwrite_pattern(int fd, long size, char *pattern)
{
	int pattern_size = (unsigned char) pattern[0];
	char *p;
	int k;

	if (opt_device)
		size = 1024;

	if (!opt_device && lseek(fd, 0, SEEK_SET) == -1) {
		perror("lseek");
		return -1;
	}

	p = malloc(pattern_size*PATTERN_EXTENSION_FACTOR);
	if (!p) {
		perror("malloc");
		return -1;
	}
	for (k = 0; k < PATTERN_EXTENSION_FACTOR; k++)
		memcpy(p+(k*pattern_size), pattern+1, pattern_size);

	pattern_size *= PATTERN_EXTENSION_FACTOR;

	while (size) {
		if (write(fd, p, MIN(pattern_size, size)) == -1) {
			if (opt_device && errno == ENOSPC)
				break;
			perror("write");
			free(p);
			return -1;
		}
		if (!opt_device)
			size -= MIN(pattern_size, size);
	}

	free(p);

	printf("P");
	fflush(stdout);

	return 0;
}

long get_usec(void)
{
	struct timeval tmptv;

	gettimeofday(&tmptv, NULL);
	return tmptv.tv_usec;
}

/*
 * How to flush the HD cache under Linux (strace of hdparm -f /dev/hda).
 * This works only for IDE (AFAIK)
 *
 * open("/dev/hda", O_RDONLY)              = 3
 * fsync(3)                                = 0
 * ioctl(3, BLKFLSBUF, 0)                  = 0
 * ioctl(3, HDIO_DRIVE_CMD, 0)             = 0
 * close(3)                                = 0
 */
int physical_sync(unsigned int device)
{
/* Under linux sync(2) waits that writing is done, this is a good
 * entropy source.
 */

/* It appears only Linux has non-void return on sync(2). */
#ifdef __linux__
	if (sync() == -1) {
		perror("sync");
		return -1;
	}
#else
	sync();
#endif

#ifdef __linux__
	/* only root can flush the HD cache */
	if (getuid() == 0 && device != 0 && !opt_device) {
		int fd;
		int major = device >> 8;
		int minor = device & 0xff;
		char *devp = NULL;

		if (major == 3 && minor >= 0 && minor <= 16)
			devp = "/dev/hda";
		else if (major == 3 && minor >= 64 && minor <= 80)
			devp = "/dev/hdb";
		else if (major == 22 && minor >= 0 && minor <= 16)
			devp = "/dev/hdc";
		else if (major == 22 && minor >= 64 && minor <= 80)
			devp = "/dev/hdd";
		else if (major == 33 && minor >= 0 && minor <= 16)
			devp = "/dev/hde";
		else if (major == 33 && minor >= 64 && minor <= 80)
			devp = "/dev/hdf";
		else if (major == 34 && minor >= 0 && minor <= 16)
			devp = "/dev/hdg";
		else if (major == 34 && minor >= 64 && minor <= 80)
			devp = "/dev/hdh";

		if (devp != NULL) {
#ifdef DEBUG
			fprintf(stderr, "try to flush the cache of %s\n", devp);
#endif
			if ((fd = open(devp, O_RDONLY)) == -1) {
				perror("open");
				return -1;
			} else if (ioctl(fd, BLKFLSBUF, NULL) == -1 ||
			   	ioctl(fd, HDIO_DRIVE_CMD, NULL) == -1) {
				perror("ioctl");
				return -1;
			}
			close(fd);
		}
	}
#endif /* __linux__ */

	PRNG_RESEED;
	return 0;
}

/* The random number generator is just SHA1 in counter mode.
 * The reseed of the generator is performed using the HD sync delay.
 * It's silly to try an entropy estimation here, we just try to reseed
 * the generator with all the entropy we have.
 * prng_seed() seeds the internal state of the generator with new entropy.
 * prng_get() is used to obtain pseudorandom bytes.
 */
void prng_seed(char *sample, int size)
{
	SHA1_CTX ctx;

	SHA1_Init(&ctx);
	SHA1_Update(&ctx, prng_state, 20);
	SHA1_Update(&ctx, sample, size);
	SHA1_Final(prng_state, &ctx);
}

void prng_get(char *buf, int size)
{
	SHA1_CTX ctx;
	static unsigned long long counter; /* not initialized */
	static char left[20];
	static unsigned int n_left = 0;

	while (size) {
		if (n_left >= size) {
			memcpy(buf, left+n_left-size, size);
			n_left -= size;
			return;
		}

		memcpy(buf, left, n_left);
		buf+=n_left;
		size-=n_left;

		SHA1_Init(&ctx);	
		SHA1_Update(&ctx, prng_state, 20);
		SHA1_Update(&ctx, (char*) &counter, sizeof(counter));
		SHA1_Final(left, &ctx);
		counter++;
		n_left = 20;
	}
}

int prng_first_reseed(void)
{
	fd_set rfds;
	struct timeval tv;
	int retval, timeout = DEV_RANDOM_TIMEOUT, n_read;
	int bytes_left = 20, fd;
	char *statep = prng_state;

	fd = open(DEV_RANDOM, O_RDONLY);

	/* We wait for /dev/random, if it's too slow (i.e. there are a lot
	 * of accesses or too less entropy-supplier events) we try with
	 * the faster /dev/urandom.
	 * The timeout for /dev/random is set to DEV_URANDOM_TIMEOUT seconds,
	 * for every successful read(2) that doesn't supply all the bytes
	 * needed the timeout is set to DEV_URANDOM_TIMEOUT-
	 */
	while(fd != -1) {
		FD_ZERO(&rfds);
		FD_SET(fd, &rfds);
		tv.tv_sec = timeout--; /* every time less patient */
		tv.tv_usec = 0;

		if (tv.tv_sec == 0)
			break;

		retval = select(fd+1, &rfds, NULL, NULL, &tv);
		if (retval) {
			n_read = read(fd, statep, bytes_left);
			if (n_read == -1) {
				close(fd);
				break; /* try urandom */
			}
			if (opt_verbose)
				printf("[prng first reseed] %d bytes readed "
					"from /dev/random\n", n_read);
			statep += n_read;
			bytes_left -= n_read;
			if (bytes_left == 0)
				return 0; /* 20 bytes readed */
		} else {
			/* timeout reached, try urandom */
			close(fd);
			break;
		}
	}

	/* try with /dev/urandom */
	fd = open(DEV_URANDOM, O_RDONLY);
	if (fd == -1)
		return -1;
	n_read = read(fd, statep, bytes_left);
	if (opt_verbose)
		printf("[prng first reseed] %d bytes readed "
			"from /dev/urandom\n", n_read);
	return n_read;
}

void show_help(char *pname)
{
	printf(
	"Usage: %s [-lvfdh] <filename> ... [filenameN]\n"
	"  -l      don't unlink(2) the file after overwritting\n"
	"  -v      verbose mode\n"
	"  -f      force overwrite in loop mode, implies -l\n"
	"  -d      specify that the target is a block or char device\n"
	"  -h      show this help\n",
		pname);
}
