/*****************************************************************************
 * Secure Locate v1.6                                                        *
 * May 28, 1999                                                              *
 * Programmed by: Kevin Lindsay                                              *
 * Copyright (c) 1998 Kevin Lindsay & Netnation Communications Inc.          *
 *                                                                           *
 * NetNation Communications:  http://www.netnation.com/                      *
 * Secure Locate: ftp://ftp.mkintraweb.com/pub/linux/slocate/                *
 *                                                                           *
 * Patches:  Sean Mcnulty <lazy@ccf.rutgers.edu>                             *
 *                * Fixed a bug which caused the decode function to fail.    *
 *           Ulf Betlehem <flu@iki.fi>                                       *
 *                * multiple databases                                       *
 *                * LOCATE_PATH environment variable support                 *
 *                * -o, --output options                                     *
 *           Jim Dougharty  <james.dougharty@sabre.com>                      *
 *                * Recursive directory walk will not exit if a directory    *
 *                * cannot be read.  May happen on some NFS directories.     *
 *           Glen Maynard <glennm@mediaone.net>                              *
 *                * Multiple search strings are now possible                 *
 *                * Uses access() instead of opendir() to see if a directory *
 *                  is readable.                                             *
 *           R.G. Wood <rgwood@stormix.com>                                  *
 *                * Made a Debian Package for Secure Locate                  *
 *           Alexander V. Lukyanov" <lav@long.yar.ru>                        *
 *                * Fixed some performance issues that I over looked.  Thanx *
 *                  To Alex, slocate -u is much faster!                      *
 *           Matt Heckaman <matt@MLINK.NET>                                  *
 *                * Created a patch to make Secure Locate work with FreeBSD. *
 *                                                                           *
 * Report any Bugs to: klindsay@mkintraweb.com                               *
 *                                                                           *
 * Even though Secure Locate was written from scratch, some recognition goes *
 * to James A. Woods <jwoods@adobe.com> for writing the original GNU locate. *
 *****************************************************************************/

/*****************************************************************************
 *                                                                            
 * Secure Locate -- search database for filenames that match patterns without 
 *                  showing files that the user using slocate does not have   
 *                  access to.                                                
 *                                                                            
 *    Copyright (C) 1994 Free Software Foundation, Inc.                       
 *                                                                            
 *    This program is free software; you can redistribute it and/or modify    
 *    it under the terms of the GNU General Public License as published by    
 *    the Free Software Foundation; either version 2, or (at your option)     
 *    any later version.                                                      
 *                                                                            
 *    This program is distributed in the hope that it will be useful,         
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of          
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           
 *    GNU General Public License for more details.                            
 *                                                                            
 *    You should have received a copy of the GNU General Public License       
 *    along with this program; if not, write to the Free Software             
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               
 *                                                                            
 *****************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <grp.h>
#include <ctype.h>
#include <fnmatch.h>
#include <regex.h>

#include "link.h"

/* GLOBALS */

#define VERSION "Secure Locate v1.6 - Released May 28, 1999\n"
#define GRPFILE "/etc/group"
#define SLOC_ESC 0x80
#define SLOC_GRP "slocate"
#define SLOC_UID 0
#define MIN_BLK 64
/* Warn if a database is older than this.  8 days allows for a weekly
 *    update that takes up to a day to perform.  */
#define WARN_SECONDS (60 * 60 * 24 * 8)

/* Printable version of WARN_SECONDS.  */
#define WARN_MESSAGE "8 days"

char **SLOCATE_PATH = NULL;

/* More fitting paths for FreeBSD -matt */
#if defined(__FreeBSD__)
char *SLOCATEDB = "/var/db/slocate/slocate.db";
char *TMPSLOCATEDB = "/var/db/slocate/slocate.tmp";
char *SLOCATEDB_DIR = "/var/db/slocate/";
#else
/* These paths are good for Linux machines */
char *SLOCATEDB = "/var/lib/slocate/slocate.db";
char *TMPSLOCATEDB = "/var/lib/slocate/slocate.tmp";
char *SLOCATEDB_DIR = "/var/lib/slocate/";
#endif
char *EXCLUDE_DIR;
int EXCLUDE=0;
int VERBOSE=0;
int QUIET=0;
int REGEXP=0;
char *regexp;
char *progname;
char *tmp_path=NULL;
char *cur_dir=NULL;
char prog_CWD[4096];
short fr_num=0;
int t_num=0;
uid_t UID;
gid_t GID;
int first=1;
char slevel='1';
unsigned short SLOC_GID=65534; /* Seems to be the default nobody gid */
int max_queries=0;

int decode_db(char *database, char *str);

/* Usage */

void
usage() 
{
   printf("%s\n"
          "Copyright (c) 1999 Kevin Lindsay & Netnation Communications Inc.\n\n"
          "search usage:   %s [-q] [-d <path>] [--database=<path>] <search string>...\n"
          "                %s [-r <regexp>] [--regexp=<regexp>]\n"
          "database usage: %s [-qv] [-o <file>] [--output=<file>]\n"
          "                %s [-e <dir1,dir2,...>] [-l <level>] <[-U <path>] [-u]>\n"
          "general usage:  %s [-Vh] [--version] [--help]\n\n"
          "   Options:\n"
          "   -u                 - Create slocate database starting at path /.\n"
          "   -U <dir>           - Create slocate database starting at path <dir>.\n"          
          "   -e <dir1,dir2,...> - Exclude directories from the slocate database.\n"
          "   -l <level>         - Security level. \n"
          "                           0 turns security checks off. This will make\n"
          "                             searchs faster.\n"
          "                           1 turns security checks on. This is the default.\n"
          "   -q                 - Quiet mode.  Error messages are suppressed.\n"
          "   -n <num>            - Limit the amount of results shown to <num>.\n"
          "   -r <regexp>\n"
          "   --regexp=<regexp>  - Search the database using a basic POSIX regular\n"
          "                        expression.\n"
          "   -o <file>\n"
          "   --output=<file>    - Specfies the database to create.\n"
          "   -d <path>\n"
          "   --database=<path>  - Specfies the path of databases to search in.\n"
          "   -h\n"
          "   --help             - Display this help.\n"
          "   -v\n"
          "   --verbose          - Verbose mode. Display files when creating database.\n"
          "   -V\n"
          "   --version          - Display version.\n"
          "\n"
          "Author: Kevin Lindsay\n"
          "Bugs:   klindsay@mkintraweb.com\n"
          "FTP:    ftp.mkintraweb.com/pub/linux/slocate/\n"
          "\n",VERSION,progname,progname,progname,progname,progname);
   
   exit(0);
}

/* My fchdir() so that I can do cleaner error checking */

int
MYfchdir(int fd, char *filepath, char *func)
{   
   if (fchdir(fd) == -1) {
      if (!QUIET)
          fprintf(stderr,"%s: %s: fchdir: %s: %s\n",progname,func,filepath,strerror(errno));
      return(0);
   }
   return(1);
}

/* My chdir() so that I can do cleaner error checking */
int
MYchdir(char *filepath, char *func)
{
   if (chdir(filepath) == -1) {
      if (!QUIET)      
          fprintf(stderr,"%s: %s: chdir: %s: %s\n",progname,func,filepath,strerror(errno));
      return(0);
   }
   return(1);
}

static void
put_short (int c, FILE *fp)
{
   putc (c >> 8, fp);
   putc (c, fp);
}

static int
get_short (FILE *fp)
{
   register short x;
   
   x = fgetc (fp);
   return (x << 8) | (fgetc (fp) & 0xff);
}

/* Parse Output Path */

void
parse_create_path(char *path)
{
   struct stat statres;

   if (!path || strlen(path) == 0) return;

   stat(path, &statres);
   if (S_ISDIR(statres.st_mode)) {
   if (!QUIET)
        fprintf(stderr,"%s: %s is a directory\n", progname,path);
   exit(1);
   }
   SLOCATEDB = malloc(strlen(path)+1);
   SLOCATEDB[0] = '\0';
   strcat(SLOCATEDB, path);

   TMPSLOCATEDB = malloc(strlen(path)+5);
   TMPSLOCATEDB[0] = '\0';
   strcat(TMPSLOCATEDB, path);
   strcat(TMPSLOCATEDB, ".tmp");
}

/* Parse Path */

void
parse_decode_path(char *path)
{
   char *pathcopy;
   char *part;
   int i;

   if (!path || strlen(path) == 0) return;

   i = 1;
   part = path;
   while ((part = strchr(part+1, ':'))) i++;
   SLOCATE_PATH = malloc(i * sizeof(char *));

   i = 0;
   pathcopy = malloc(strlen(path)+1);
   pathcopy[0] = '\0';
   strcat(pathcopy, path);
   part = strtok(pathcopy, ":");
   while(part) {
   SLOCATE_PATH[i++] = part;
   part = strtok(NULL, ":");
   }
   SLOCATE_PATH[i] = NULL;
}

/* Parse Dash */

void
parse_dash(char *option)
{
   char *ptr;
   int i=0;
   char *database;
   int res=1;
   
   for (ptr = option;*ptr != 0 && *ptr != '=' ; ptr++)
       *ptr = toupper(*ptr);

   if (!strcmp(option,"HELP"))
       usage();
       
   if (!strcmp(option,"VERSION")) {
      printf("%s",VERSION);
      exit(0);
   }
   
   if (!strcmp(option,"VERBOSE"))
       VERBOSE=1;
   
   *ptr = '\0';
   ptr++;

   if (!strcmp(option,"OUTPUT")) {
      parse_create_path(ptr);
   }

   if (!strcmp(option,"DATABASE")) {
      parse_decode_path(ptr);
   }
   
   if (!strcmp(option,"REGEXP")) {
      REGEXP = 1;
      regexp = malloc(strlen(ptr)+1);
      *regexp = 0;
      strcat(regexp,ptr);
      while ((database = SLOCATE_PATH[i++])) res |= decode_db(database, ptr);
      exit(!res);
   }

}


/* Make sure that the DB Directory exists */

int
check_dir()
{
   int res=1;
   struct stat tmpstat;
   
   if (lstat(SLOCATEDB_DIR,&tmpstat) == -1) {
      if (!QUIET)
          fprintf(stderr,"%s: error accessing DB Directory: %s : %s\n",progname,SLOCATEDB_DIR,strerror(errno));
      res=0;
   }
   
   return(res);
}

/* Get the GID for group slocate */

unsigned short
get_gid()
{
   unsigned short GGID=SLOC_GID;
   struct group *grpres;
   
   if ((grpres = getgrnam(SLOC_GRP)) == NULL) {
      if (!QUIET) {
         fprintf(stderr,"%s: Could not find the group: %s in the /etc/group file.\n",progname,SLOC_GRP);
         fprintf(stderr,"This is a result of the group missing or a corrupted group file.\n");
      }
      exit(1);
   }

   GGID = grpres->gr_gid;
   
   return GGID;
}

/* Parse Exclude Command */

char *
parse_exclude(char *estr)
{
   char *excludestr=NULL;
   char *ptr1;
   char *ptr2=NULL;
   int grow=1;
   int elen;
   
   EXCLUDE=1;

   excludestr = (char *)malloc(1);
   excludestr[0] = '\0';
   
   elen = strlen(estr);

   ptr1 = estr;
   ptr2 = ptr1;

   while (ptr2[0] != '\0')  {

      if ((ptr2 = strchr(ptr1,',')) == NULL) {
         ptr2 = strchr(ptr1,'\0');
      }

      grow += (ptr2-ptr1)+2;
      if (*(ptr2-1) == '/') {
         grow--;
         ptr2--;
      }
      
      excludestr = realloc(excludestr,grow);
      strcat(excludestr,"*");
      strncat(excludestr,ptr1,ptr2-ptr1);
      strcat(excludestr,"*");
      
      if (*ptr2 == '/')
          ptr2++;

      if (ptr2[0] == ',')
          ptr1 = ptr2+1;
   }
   
   return (excludestr);
}

int
match_exclude(char *path,char *filedir) {
   int res=0;
   char *newstr;
   
   newstr = malloc(strlen(path)+strlen(filedir)+3);
   newstr[0] = '\0';
   strcat(newstr,"*");
   strcat(newstr,path);
   strcat(newstr,filedir);   
   strcat(newstr,"*");
   
   if (strstr(EXCLUDE_DIR,newstr) != NULL)       
       res=1;
   
   free(newstr);
   
   return res;
}

/* Load Dir Contents into Linked List */

dir_item *
load_dir(int dir_fd, dir_item *listptr)
{
   DIR *recdir;
   struct dirent *filename = NULL;
   struct stat statres;
   int top=0;

   if (MYfchdir(dir_fd,listptr->name,"LD_1") == -1) return(NULL);
   
   if ((recdir = opendir(".")) == NULL) {
      if (!QUIET)
          fprintf(stderr,"%s: Opendir: %s: %s\n",progname,listptr->name,strerror(errno));
      return(NULL);
   }

   /* Open Current Directory 
   if ((filename = readdir(recdir)) == NULL) {
      if (!QUIET)
          fprintf(stderr,"%s: Readdir Error: Load List: %s: %s\n",progname,listptr->name,strerror(errno));
      closedir(recdir);
      return(NULL);
   }

   seekdir(recdir,telldir(recdir)); */

   while ((filename = readdir(recdir)) != NULL) {

      while (!strcmp(filename->d_name,".") || !strcmp(filename->d_name,"..")) {
         top=1;
         if ((filename = readdir(recdir)) == NULL)
             break;
      }
      
      if (top) {

         listptr = add_right(listptr);
         listptr = listptr->right;
         
         /* If directory is emtpy then add a space to listptr */         
         if (filename == NULL) {
            listptr->name = malloc(1);
            listptr->name[0] = '\0';
            listptr->empty = 1;
            listptr->type = 0;
            listptr->root = 0;
            listptr->name_len = 1;
            closedir(recdir);
            return(listptr);            
         }
         
         top=0;
      } else {
         listptr = add_down(listptr);
         listptr = listptr->down;
      }
      
      /* Add information from filenames->d_name to listptr */
      listptr->name = malloc(strlen(filename->d_name)+1);
      listptr->name[0] = '\0';
      strcat(listptr->name,filename->d_name);
      listptr->name_len = strlen(listptr->name);      

      if (lstat(filename->d_name,&statres) == -1) {
         if (!QUIET)
             fprintf(stderr,"%s: lstat: %s: %s\n",progname,filename->d_name,strerror(errno));
      }
      
      if (S_ISDIR(statres.st_mode))
          listptr->type=1;
      else
          listptr->type=0;
      
      listptr->root = 0;

   }
   
   closedir(recdir);
   
   return(listptr);
}

/* Get Current Directory */
char *
get_cur_dir(dir_item *listptr, char *dirstr)
{
   
   dir_item *dir_ptr;
   char **dirarray;
   int items=0;
   int i;
   int grow;
   

   if (cur_dir != NULL) {
      free(cur_dir);
   }

   dir_ptr = listptr;
   
   while (!dir_ptr->root) {
      dir_ptr=dir_ptr->left;
      items++;
   }

   dirarray = calloc(items,sizeof(char *));
   
   dir_ptr=listptr->left;
   
   for (i = items-1; i != -1; i--) {
      dirarray[i] = malloc(strlen(dir_ptr->name)+1);
      dirarray[i][0] = '\0';
      strcat(dirarray[i],dir_ptr->name);

      dir_ptr=dir_ptr->left;
   }

   cur_dir = malloc(1);
   cur_dir[0] = '\0';
   grow=1;
   
   for (i = 0; i != items; i++) {
      grow+=strlen(dirarray[i])+1;
      cur_dir = realloc(cur_dir,grow);
      strcat(cur_dir,dirarray[i]);
      if ((strcmp(dirstr,"/") != 0 && i == 0) || i > 0)
          strcat(cur_dir,"/");
   }

   for (i = 0; i != items; i++) {
      free(dirarray[i]);
   }
   
   free(dirarray);

   
   return(cur_dir);
}

/* FRCODE - Incremental Encoding algorithm */

int
frcode(FILE *fd,char *dir_path, char *filename)
{
   int res=0;
   char *cur_path;
   char *new_path;
   char *ptr1;
   char *ptr2;

   int i;
   
   cur_path = malloc(strlen(dir_path)+strlen(filename)+1);
   cur_path[0]='\0';
   strcat(cur_path,dir_path);
   strcat(cur_path,filename);
   
   if (VERBOSE)
       fprintf(stdout,"%s\n",cur_path);
   
   if (tmp_path != NULL) {
      ptr1=cur_path;
      ptr2=tmp_path;
      
      for (i=0; ptr1[0] == ptr2[0] && (ptr1[0] != '\0' && ptr2[0] != '\0') ; i++) {
         ptr1++;
         ptr2++;
      }

      fr_num = i - t_num;
         
      t_num = t_num + fr_num;            
      
      ptr2 = strchr(ptr1,'\0');
      new_path=malloc((ptr2-ptr1)+1);
      new_path[0] = '\0';
      strncat(new_path,ptr1,ptr2-ptr1);         

   } else {
      new_path = malloc(strlen(cur_path)+1);
      new_path[0] = '\0';
      strcat(new_path,cur_path);
      t_num = strlen(new_path);
   }
   
   if (tmp_path != NULL)
       free(tmp_path);
   
   tmp_path = malloc(strlen(cur_path)+1);
   tmp_path[0] = '\0';
   strcat(tmp_path,cur_path);
   
   if (fr_num > 127 || fr_num < -127) {
      putc (SLOC_ESC, fd);
      put_short (fr_num, fd);
   } else
       putc(fr_num,fd);
   
   fputs(new_path,fd);
   putc('\0',fd);

   if (first) {
      chmod(TMPSLOCATEDB,00640);
      chown(TMPSLOCATEDB,UID,SLOC_GID);
      first=0;
   }
   
   free(new_path);
   free(cur_path);
   
   return res;
}

/* Create Database */

int
create_db(char *dirstr)
{
   int res=0;
   struct stat statres;
   int dir_fd;
   dir_item *listptr=NULL;
   dir_item *tmpptr=NULL;
   int load;
   int jump=0;
   int list_down=0;
   FILE *fd;

   if (UID != SLOC_UID) {
      if (!QUIET)
          fprintf(stderr,"%s: You are not authorized to create an slocate database!\n",progname);
      return(0);
   }
   
   fd = fopen(TMPSLOCATEDB,"w");
   putc(slevel,fd);
   
   if (dirstr) {
      /* Make sure there are no problems accessing the source directory */
      if (lstat(dirstr,&statres) == -1) {
         if (!QUIET)
             fprintf(stderr,"%s: lstat: d%s/: %s\n",progname,dirstr,strerror(errno));
    fclose(fd);
         return(-1);
      }
   }
   
   /* Initialize 2D Linked List */
   listptr = init_2D_list();

   if (dirstr) {
       if (dirstr[strlen(dirstr)-1] == '/') dirstr[strlen(dirstr)-1] = '\0';
   } else {
      dirstr = malloc(2);
      dirstr[0]='/';
      dirstr[1]='\0';
   }

   
   listptr->name = malloc(strlen(dirstr)+1);
   listptr->name[0] = '\0';
   strcat(listptr->name,dirstr);
   listptr->st_uid = statres.st_uid;
   listptr->st_gid = statres.st_gid;
   listptr->st_mode = statres.st_mode;
   listptr->root = 1;
   listptr->empty=0;
   
   
   /* Open Source Directory */
   if ((dir_fd = open(dirstr,O_RDONLY)) == -1) {
      if (!QUIET)
          fprintf(stderr,"%s: open: %s: %s\n",progname,dirstr,strerror(errno));
      fclose(fd);
      return(-1);
   }
   
   /* Change CWD to source dir */
   if (!MYfchdir(dir_fd,dirstr,"CDB_1")) {
      close(dir_fd);
      fclose(fd);
      return(0);
   }
   
   if (EXCLUDE && match_exclude(dirstr,""))
       load = 0;
   else
       load = 1;

   while (!jump) {
      if (load) {
         /* Load directory contents into linked list */
         listptr = load_dir(dir_fd,listptr);
         load=0;
         listptr=listptr->left->right;
      }
      
      if (!listptr->empty) {

         tmpptr = listptr;
         if (list_down) {
            listptr=listptr->down;
         }
         list_down=0;         
         while (listptr != NULL) {
            if (!EXCLUDE || (EXCLUDE && !match_exclude(get_cur_dir(listptr,dirstr),listptr->name))) {               
               frcode(fd,get_cur_dir(listptr,dirstr),listptr->name);               
            }
            if (!listptr->type) {
               listptr=listptr->down;
            } else {
               load=1;
               break;
            }
         }
      } else {
         if (!MYchdir("..","create_db")) {
            close(dir_fd);
       fclose(fd);
            return(1);
         }         
         listptr=listptr->left;
         free_right(listptr);
         tmpptr=listptr;
         listptr=listptr->down;
         if (listptr != NULL)
             load = listptr->type;
         list_down=0;
      }
      
      if (listptr == NULL) {
         
         while (listptr == NULL) {
            listptr=tmpptr;
            if (listptr->root) {
               jump=1;
               break;
            }
            listptr=listptr->left;
            
            if (!MYchdir("..","CDB_2")) {
               close(dir_fd);
          fclose(fd);
               return(1);
            }
            
            close(dir_fd);
            
            if ((dir_fd = open(".",O_RDONLY)) == -1) {
               if (!QUIET)
                   fprintf(stderr,"%s: open: CDB_3: %s\n",progname,strerror(errno));
          fclose(fd);
               return(1);
            }
            
            tmpptr=listptr;
            free_right(listptr);
            listptr=listptr->down;            
         }
         
         load=0;

      } else if (listptr->type) {
         
         /* Change to new directory */
         if ((EXCLUDE && match_exclude(get_cur_dir(listptr,dirstr),listptr->name) == 1) || (chdir(listptr->name) == -1)) {
            load=0;
            list_down=1;
         } else {
            
            close(dir_fd);
            
            if ((dir_fd = open(".",O_RDONLY)) == -1) {
               if (!QUIET)
                 fprintf(stderr,"%s: open: CDB_5: %s: %s%s\n",progname,strerror(errno),
                         get_cur_dir(listptr,dirstr),listptr->name);
               //   return(1);  Don't bail just because we can't read a directory.
               // This might happen even as root on some NFS partitions.
               load=0;
               list_down=0;          
            }
         }
                  
      }

   }   
   
   fclose(fd);
   chdir(prog_CWD);
   rename(TMPSLOCATEDB,SLOCATEDB);
   chmod(SLOCATEDB,00640);
   chown(SLOCATEDB,UID,SLOC_GID);   
   return res;
}

/* Decode Database */

int
decode_db(char *database, char *str)
{
   int res = 1;
   FILE *fd;
   short code_num;
   int pathlen=0;
   register char ch;
   int grow=0;
   int jump=0;
   int first=1;
   char *codedpath=NULL;
   char *tmpcodedpath;    /* to fix around realloc() problem */
   char *ptr;
   int printit=0;
   int globflag=0;
   char *globptr1;
   struct stat statres;
   time_t now;
   char *chk1;
   char tmpch;
   regex_t *preg=NULL;
   char *errbuf=NULL;
   int nmatch=32;
   regmatch_t pmatch[32];
   int cur_queries = 0;
   
   if ((fd = fopen(database,"r")) == NULL) {
      if (!QUIET)
          fprintf(stderr,"%s: decode_db(): %s: %s\n",progname,database,strerror(errno));
      return(0);
   }

   lstat(database,&statres);

   if (S_ISDIR(statres.st_mode)) {
   if (!QUIET)
        fprintf(stderr,"%s: decode_db(): %s is a directory\n",progname,database);
   return(0);
   }

   time(&now);
   if (now - statres.st_mtime > WARN_SECONDS)
   {
      fprintf(stdout,"%s: warning: database %s' is more than %s old\n",progname,database,WARN_MESSAGE);
   }   
   
   slevel = getc(fd);
   
   codedpath = malloc(MIN_BLK);
   codedpath[0] = '\0';
   ptr=codedpath;
   
   if ((globptr1 = strchr(str,'*')) != NULL ||
       (globptr1 = strchr(str,'?')) != NULL ||
       ((globptr1 = strchr(str,'[')) != NULL && strchr(str,']') != NULL))
       globflag = 1;
   
   while ((code_num = getc(fd)) != EOF) {
      
      if (code_num == SLOC_ESC) {
         code_num = get_short(fd);
      } else if (code_num > 127)
             code_num = code_num - 256;

      jump=0;
      
      if (code_num < 0)
         grow += code_num;
      
      ptr+=code_num;
      
      pathlen = ptr - codedpath;

      while(!jump) {
         ch = getc(fd);
         grow++;
         pathlen++;
         if (grow == 64) {

            tmpcodedpath = realloc(codedpath,pathlen+MIN_BLK);
            ptr += (tmpcodedpath - codedpath);
            codedpath = tmpcodedpath;
            
            grow=0;
         }
         codedpath[pathlen-1] = ch;
   
         if (ch == '\0')
            jump=1;
      }
      
      if (first) {
         ptr=ptr+strlen(codedpath);
         first=0;
      }
      
      pathlen--;            
      
      printit=0;
      
      if (REGEXP) {
         globflag=0;
         preg = malloc(sizeof(regex_t));
         if ((res = regcomp(preg,regexp,0)) != 0) {
            regerror(res, preg, errbuf,1024);
            if (!QUIET)
              fprintf(stdout,"error: %s: regular expression: %s\n",progname,errbuf);
            exit(1);
         }         
         
      }

      if ((globflag && !fnmatch(str,codedpath,0)) || (!REGEXP && strstr(codedpath,str) != NULL) || (REGEXP && regexec(preg,codedpath,nmatch,pmatch,0) == 0)) {
         if (slevel == '1') {
            if (UID == 0 || lstat(codedpath,&statres) == 0) {
               printit=1;
               chk1 = codedpath+1;
               while ((chk1 = strchr(chk1,'/')) != NULL) {
                  chk1++;
                  tmpch=*chk1;
                  *chk1='\0';

                  if (access(codedpath, R_OK) == -1)
                    printit=0;
                  *chk1=tmpch;
               }
            }
         } else
             printit=1;
      }
      
      if (REGEXP)
        regfree(preg);
      
      if (printit) {
         cur_queries += 1;
         printf("%s\n",codedpath);
         if (max_queries > 0 && cur_queries == max_queries)
           exit(0);
      }
   }      
   
   fclose(fd);
   
   return(1);
}

/* Main Function */

int
main(int args, char **argv)
{
   int res=0;
   int ch;
   extern char *optarg;
   extern int optind, opterr, optopt;
   char *p;
   int SPECDIR=0;
   int ROOTDIR=0;   
   char *spec_dir=NULL;
   char *database;
   int i=0;


   progname = ((p = strrchr(argv[0],'/')) ? p+1 : *argv);     
   
   if (!strcmp(progname,"updatedb"))
     ROOTDIR=1;
   
   /* Get Current Working Directory */
   
   getcwd(prog_CWD, 4095);
   
   if (strlen(prog_CWD) == 4095) {
      fprintf(stderr,"%s: Current Working Directory is too large!",progname);
      exit(1);
   }

   
   if (args < 2 && !ROOTDIR)
     usage();
   
   if (!check_dir())
     return(1);
   
   UID = getuid();        
   GID = getgid();

   parse_create_path(SLOCATEDB);
   parse_decode_path(SLOCATEDB);
   parse_decode_path(getenv("LOCATE_PATH"));

   while ((ch = getopt(args,argv,"VvuhqU:r:o:e:l:d:-:n:")) != EOF) {
      switch(ch) {
      case 'h':
         usage();
         break;
      case 'q':
         QUIET=1;
         break;
      case 'V':
         printf("%s",VERSION);
         exit(1);
         break;
      case 'v':
         VERBOSE=1;
         break;
      case 'e':
         EXCLUDE_DIR = parse_exclude(optarg);
         break;
       case 'l':
         slevel = optarg[0];
         if (slevel != '0' && slevel != '1') {
            if (!QUIET)
              fprintf(stderr,"%s: ERROR: Security level must be 0 or 1.\n",progname);
            exit(1);
         }
         break;
      case 'u':
         ROOTDIR=1;
         break;
      case 'U':
         ROOTDIR=0;
         SPECDIR=1;
         spec_dir = malloc(strlen(optarg)+1);
         *spec_dir = 0;         
         strcat(spec_dir,optarg);
         break;
       case 'r':
         REGEXP = 1;
         regexp = malloc(strlen(optarg)+1);
         *regexp = 0;
         strcat(regexp,optarg);
         while ((database = SLOCATE_PATH[i++])) res |= decode_db(database, optarg);
         return(!res);
         break;
      case 'o':
         parse_create_path(optarg);
         break;
      case 'd':
         parse_decode_path(optarg);
         break;
      case '-':
         parse_dash(optarg);
         break;
       case 'n':
         for (i=0; i < strlen(optarg); i+=1) {
            if (!isdigit(optarg[i])) {
               fprintf(stderr,"%s: ERROR: Invalid argument for option -n\n",progname);
               exit(1);
            }
         }
         max_queries = atoi(optarg);
         break;
      default:
         return(1);
         break;
         
      }
   }
   
   SLOC_GID = get_gid(GRPFILE);
   
   if (SPECDIR) {
      res = create_db(spec_dir);
   } else if (ROOTDIR)
      res = create_db((char *)NULL);
   else if (argv[optind]) {
      int j = optind;
      int i = 0;
      
      while (j < args) {
         while ((database = SLOCATE_PATH[i++])) res |= decode_db(database, argv[j++]);
         i = 0;
      }
   } else
       usage();

   return(!res);
}
