/*****************************************************************************
 * Secure Locate v2.1                                                        *
 * October 19, 1999                                                          *
 * Programmed by: Kevin Lindsay                                              *
 * Copyright (c) 1998 Kevin Lindsay & Netnation Communications Inc.          *
 *                                                                           *
 * NetNation Communications   http://www.netnation.com/                      *
 * Secure Locate: ftp://ftp.mkintraweb.com/pub/linux/slocate/                *
 *                http://pep.stormix.com/slocate/                            *
 *                                                                           *
 * Patches:  Sean Mcnulty <lazy@ccf.rutgers.edu>                             *
 *                * Fixed a bug which caused the decode function to fail.    *
 *           Ulf Betlehem <flu@iki.fi>                                       *
 *                * multiple databases                                       *
 *                * LOCATE_PATH environment variable support                 *
 *                * -o, --output options                                     *
 *           Jim Dougharty  <james.dougharty@sabre.com>                      *
 *                * Recursive directory walk will not exit if a directory    *
 *                * cannot be read.  May happen on some NFS directories.     *
 *           Glen Maynard <glennm@mediaone.net>                              *
 *                * Multiple search strings are now possible                 *
 *                * Uses access() instead of opendir() to see if a directory *
 *                  is readable.                                             *
 *           R.G. Wood <rgwood@stormix.com>                                  *
 *                * Made a Debian Package for Secure Locate                  *
 *           Alexander V. Lukyanov" <lav@long.yar.ru>                        *
 *                * Fixed some performance issues that I over looked.  Thanx *
 *                  To Alex, slocate -u is much faster!                      *
 *           Matt Heckaman <matt@MLINK.NET>                                  *
 *                * Created a patch to make Secure Locate work with FreeBSD. *
 *           Luca Berra <bluca@vodka.it>                                     *
 *                * Added case insensitive option and optimized code to make *
 *                  searching faster.                                        *
 *           Hans-Juergen Godau <godau@wi-inf.uni-essen.de>                  *
 *                * Fixed a segfault when searching through more than one    *
 *                  database.                                                *
 *                                                                           *
 * Report any Bugs to: klindsay@mkintraweb.com                               *
 *                                                                           *
 * Even though Secure Locate was written from scratch, some recognition goes *
 * to James A. Woods <jwoods@adobe.com> for writing the original GNU locate. *
 *                                                                           *
 * This product includes software developed by the University of             *
 * California, Berkeley and its contributors.                                *
 *                                                                           *
 *****************************************************************************/

/*****************************************************************************
 *                                                                            
 * Secure Locate -- search database for filenames that match patterns without 
 *                  showing files that the user using slocate does not have   
 *                  access to.                                                
 *                                                                            
 *    Copyright (C) 1994 Free Software Foundation, Inc.                       
 *                                                                            
 *    This program is free software; you can redistribute it and/or modify    
 *    it under the terms of the GNU General Public License as published by    
 *    the Free Software Foundation; either version 2, or (at your option)     
 *    any later version.                                                      
 *                                                                            
 *    This program is distributed in the hope that it will be useful,         
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of          
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           
 *    GNU General Public License for more details.                            
 *                                                                            
 *    You should have received a copy of the GNU General Public License       
 *    along with this program; if not, write to the Free Software             
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               
 *                                                                            
 *****************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <grp.h>
#include <ctype.h>
#include <fnmatch.h>
#include <regex.h>
#include "sl_fts.h"

#include "link.h"
#include "misc.h"

#ifdef __FreeBSD__
# include <sys/param.h>
# include <sys/ucred.h>
# include <sys/mount.h>
# undef FNM_CASEFOLD
#endif

/* GLOBALS */

#define VERSION "Secure Locate v2.1 - Released October 19, 1999\n"
#define GRPFILE "/etc/group"
#define SLOC_ESC 0x80
#define SLOC_GRP "slocate"
#define SLOC_UID 0
#define MIN_BLK 64
#define MTAB_FILE "/etc/mtab"

/* Warn if a database is older than this.  8 days allows for a weekly
 *    update that takes up to a day to perform.  */
#define WARN_SECONDS (60 * 60 * 24 * 8)

/* Printable version of WARN_SECONDS.  */
#define WARN_MESSAGE "8 days"

char **SLOCATE_PATH = NULL;

/* More fitting paths for FreeBSD -matt */
#if defined(__FreeBSD__)
char *SLOCATEDB = "/var/db/slocate/slocate.db";
char *TMPSLOCATEDB = "/var/db/slocate/slocate.db.tmp";
char *SLOCATEDB_DIR = "/var/db/slocate/";
#else
/* These paths are good for Linux machines */
char *SLOCATEDB = "/var/lib/slocate/slocate.db";
char *TMPSLOCATEDB = "/var/lib/slocate/slocate.db.tmp";
char *SLOCATEDB_DIR = "/var/lib/slocate/";
#define UPDATEDB_CONF "/etc/updatedb.conf"
#endif
char *EXCLUDE_DIR=NULL;
int EXCLUDE=0;
int VERBOSE=0;
int QUIET=0;
int NOCASE=0;
int REGEXP=0;
int NEWOUTPUT=0;
char *regexp;
char *progname;
char *tmp_path=NULL;
char *cur_dir=NULL;
char prog_CWD[4096];
short fr_num=0;
int t_num=0;
uid_t UID;
gid_t GID;
int first=1;
char slevel='1';
unsigned short SLOC_GID=65534; /* Seems to be the default nobody gid */
int max_queries=0;

int decode_db(char *database, char *str);

/* Usage */

void
usage() 
{
   int i;
   printf("%s\n"
          "Copyright (c) 1999 Kevin Lindsay & Netnation Communications Inc.\n\n"
          "search usage:   %s [-qi] [-d <path>] [--database=<path>] <search string>...\n"
          "                %s [-r <regexp>] [--regexp=<regexp>]\n"
          "database usage: %s [-qv] [-o <file>] [--output=<file>]\n"
          "                %s [-e <dir1,dir2,...>] [-f <fs_type1,...> ] [-l <level>]\n"
          ,VERSION,progname,progname,progname,progname);
   
   for (i = 0; i < strlen(progname)-1; i+=1)
     printf(" ");
   
   printf(
#ifndef __FreeBSD__
          "                  [-c] <[-U <path>] [-u]>\n"
#else
          "                  <[-U <path>] [-u]>\n"
#endif
          "general usage:  %s [-Vh] [--version] [--help]\n\n"
          "   Options:\n"
          "   -u                 - Create slocate database starting at path /.\n"
          "   -U <dir>           - Create slocate database starting at path <dir>.\n",progname);
#ifndef __FreeBSD__
   printf("   -c                 - Parse original GNU Locate's '/etc/updatedb.conf'\n"
          "                        when using the -u or -U options.  If 'updatedb' is\n"
          "                        symbolically linked to the '%s' binary, the\n"
          "                        original configuration file will automatically be\n"
          "                        used.\n",progname);
#endif
   printf("   -e <dir1,dir2,...> - Exclude directories from the slocate database when\n"
          "                        using the -u or -U options.\n"
          "   -f <fs_type1,...>  - Exclude file system types from the slocate database\n"
          "                        when using the -u or -U options. (ie. NFS, etc).\n"                                  
          "   -l <level>         - Security level. \n"
          "                           0 turns security checks off. This will make\n"
          "                             searchs faster.\n"
          "                           1 turns security checks on. This is the default.\n"
          "   -q                 - Quiet mode.  Error messages are suppressed.\n"
          "   -n <num>           - Limit the amount of results shown to <num>.\n"
          "   -i                 - Does a case insensitive search.\n"
          "   -r <regexp>\n"
          "   --regexp=<regexp>  - Search the database using a basic POSIX regular\n"
          "                        expression.\n"
          "   -o <file>\n"
          "   --output=<file>    - Specfies the database to create.\n"
          "   -d <path>\n"
          "   --database=<path>  - Specfies the path of databases to search in.\n"
          "   -h\n"
          "   --help             - Display this help.\n"
          "   -v\n"
          "   --verbose          - Verbose mode. Display files when creating database.\n"
          "   -V\n"
          "   --version          - Display version.\n"
          "\n"
          "Author: Kevin Lindsay\n"
          "Bugs:   klindsay@mkintraweb.com\n"
          "FTP:    ftp://ftp.mkintraweb.com/pub/linux/slocate/\n"
          "HTTP:   http://pep.stormix.com/slocate/\n"
          "\n");
   
   exit(0);
}

static void
put_short (int c, FILE *fp)
{
   putc (c >> 8, fp);
   putc (c, fp);
}

static int
get_short (FILE *fp)
{
   register short x;
   
   x = fgetc (fp);
   return (x << 8) | (fgetc (fp) & 0xff);
}

/* Parse Output Path */

void
parse_create_path(char *path)
{
   int ret_val;
   
   if (!path || strlen(path) == 0) return;
   
   ret_val = creat(path, S_IRWXU|S_IRGRP);
   if(ret_val==-1)
   {
      if (!QUIET)
        fprintf(stderr,"%s: could not create database: %s: %s\n", progname,path,strerror(errno));
      exit(1);
   }
      
   SLOCATEDB = malloc(strlen(path)+1);
   SLOCATEDB[0] = '\0';
   strcat(SLOCATEDB, path);

   TMPSLOCATEDB = malloc(strlen(path)+5);
   TMPSLOCATEDB[0] = '\0';
   strcat(TMPSLOCATEDB, path);
   strcat(TMPSLOCATEDB, ".tmp");
}

/* Parse Path */

void
parse_decode_path(char *path)
{
   char *pathcopy;
   char *part;
   int i;
   
   if (!path || strlen(path) == 0) return;
   
   i = 1;
   part = path;
   while ((part = strchr(part+1, ':'))) i++;
   SLOCATE_PATH = malloc(i * sizeof(char *));
   
   i = 0;
   pathcopy = malloc(strlen(path)+1);
   pathcopy[0] = '\0';
   strcat(pathcopy, path);
   part = strtok(pathcopy, ":");
   while(part) {
      SLOCATE_PATH[i++] = part;
      part = strtok(NULL, ":");
   }
   SLOCATE_PATH[i] = NULL;
}

/* Parse Dash */

void
parse_dash(char *option)
{
   char *ptr;
   int i=0;
   char *database;
   int res=1;
   
   for (ptr = option;*ptr != 0 && *ptr != '=' ; ptr++)
       *ptr = toupper(*ptr);

   if (!strcmp(option,"HELP"))
       usage();
       
   if (!strcmp(option,"VERSION")) {
      printf("%s",VERSION);
      exit(0);
   }
   
   if (!strcmp(option,"VERBOSE"))
       VERBOSE=1;
   
   *ptr = '\0';
   ptr++;

   if (!strcmp(option,"OUTPUT")) {
      parse_create_path(ptr);
      NEWOUTPUT=1;
   }

   if (!strcmp(option,"DATABASE")) {
      parse_decode_path(ptr);
   }
   
   if (!strcmp(option,"REGEXP")) {
      REGEXP = 1;
      regexp = malloc(strlen(ptr)+1);
      *regexp = 0;
      strcat(regexp,ptr);
      while ((database = SLOCATE_PATH[i++])) res |= decode_db(database, ptr);
      exit(!res);
   }

}


/* Make sure that the DB Directory exists */

int
check_dir()
{
   int res=1;
   struct stat tmpstat;
   
   if (lstat(SLOCATEDB_DIR,&tmpstat) == -1) {
      if (!QUIET)
          fprintf(stderr,"%s: error accessing DB Directory: %s : %s\n",progname,SLOCATEDB_DIR,strerror(errno));
      res=0;
   }
   
   return(res);
}

/* Get the GID for group slocate */

unsigned short
get_gid()
{
   unsigned short GGID=SLOC_GID;
   struct group *grpres;
   
   if ((grpres = getgrnam(SLOC_GRP)) == NULL) {
      if (!QUIET) {
         fprintf(stderr,"%s: Could not find the group: %s in the /etc/group file.\n",progname,SLOC_GRP);
         fprintf(stderr,"This is a result of the group missing or a corrupted group file.\n");
      }
      exit(1);
   }

   GGID = grpres->gr_gid;
   
   return GGID;
}

/* Parse Exclude Command */

int
parse_exclude(char *estr)
{
   char *excludestr=NULL;
   char *ptr1;
   char *ptr2=NULL;
   int grow=1;
   int elen;   

   EXCLUDE=1;   
   
   excludestr = (char *)malloc(1);
   excludestr[0] = '\0';
   
   elen = strlen(estr);

   ptr1 = estr;
   ptr2 = ptr1;

   while (ptr2[0] != '\0')  {

      if ((ptr2 = strchr(ptr1,',')) == NULL) {
         ptr2 = strchr(ptr1,'\0');
      }

      grow += (ptr2-ptr1)+2;
      if (*(ptr2-1) == '/' && (ptr2-1) != estr && *(ptr2-2) != ',' ) {
         grow--;
         ptr2--;
      }
      
      excludestr = realloc(excludestr,grow);
      strcat(excludestr,"*");
      strncat(excludestr,ptr1,ptr2-ptr1);
      strcat(excludestr,"*");
      
      if (*ptr2 == '/')
          ptr2++;

      if (ptr2[0] == ',')
          ptr1 = ptr2+1;
   }
   
   if (!EXCLUDE_DIR) {
      EXCLUDE_DIR = malloc(strlen(excludestr)+1);
      *EXCLUDE_DIR = 0;      
   } else {
      EXCLUDE_DIR = realloc(EXCLUDE_DIR,strlen(EXCLUDE_DIR)+strlen(excludestr)+1);
   }
   
   strcat(EXCLUDE_DIR,excludestr);
   free(excludestr);

#ifdef DEBUG
   printf("E: %s\n",EXCLUDE_DIR);
#endif

   return 1;
}


#ifdef __FreeBSD__
/* Get File System type in the form of a string. "*fstype*" */

char *
get_fs_type(int fs_type)
{
   if (fs_type == MOUNT_UFS)
     return("*UFS*");
   else if (fs_type == MOUNT_NFS)
     return("*NFS*");
   else if (fs_type == MOUNT_MFS)
     return("*MFS*");
   else if (fs_type == MOUNT_MSDOS)
     return("*MSDOS*");
   else if (fs_type == MOUNT_LFS)
     return("*LFS*");
   else if (fs_type == MOUNT_LOFS)
     return("*LOFS*");   
   else if (fs_type == MOUNT_FDESC)
     return("*FDESC*");
   else if (fs_type == MOUNT_PORTAL)
     return("*PORTAL*");
   else if (fs_type == MOUNT_NULL)
     return("*NULL*");
   else if (fs_type == MOUNT_UMAP)
     return("*UMAP*");
   else if (fs_type == MOUNT_KERNFS)
     return("*KERNFS*");
   else if (fs_type == MOUNT_PROCFS)
     return("*PROCFS*");
   else if (fs_type == MOUNT_AFS)
     return("*AFS*");
   else if (fs_type == MOUNT_CD9660)
     return("*CD9660*");
   else if (fs_type == MOUNT_UNION)
     return("*UNION*");
   else if (fs_type == MOUNT_DEVFS)
     return("*DEVFS*");
   else if (fs_type == MOUNT_EXT2FS)
     return("*EXT2FS*");
   else if (fs_type == MOUNT_TFS)
     return("*TFS*");
   else
     return("*NONE*");
}
#endif

/* Parse File System Type Exclusion */
int
parse_fs_exclude(char *estr)
{
   char *ptr;
   char *newestr=NULL;
   int estr_size=0;
   
   /* Uppercase all characters in estr to make parsing easier.
    * Also add '*'s around each fs type */
   if (estr) {
      estr_size = strlen(estr)+2;
      newestr = malloc(estr_size+1);
      *newestr = 0;
      strcat(newestr,"*");
      for (ptr = estr; *ptr != 0; ptr += 1) {
         *ptr = toupper(*ptr);
         if (*ptr == ',') {
            estr_size += 1;
            newestr = realloc(newestr,estr_size+1);
            strcat(newestr,"*");
         } else {
            strncat(newestr,ptr,1);
         }
      }
      strcat(newestr,"*");
      estr = newestr;
   }
   
   /* FreeBSD File System Status */

#ifdef __FreeBSD__
     {
        struct statfs *fs_stat;
        long bufsize=4096;
        int num_mounts=0;
        int i;
        char *exclude_str=NULL;
        
        fs_stat = malloc(sizeof(struct statfs));
        
        num_mounts = getfsstat(fs_stat,bufsize,MNT_WAIT);
        
        for (i = 0; i < num_mounts; i+=1) {
           if (strstr(estr,get_fs_type(fs_stat[i].f_type))) {
              if (!exclude_str) {
                 exclude_str = malloc(strlen(fs_stat[i].f_mntonname)+1);
                 *exclude_str = 0;
              } else {
                 exclude_str = realloc(exclude_str,strlen(exclude_str)+strlen(fs_stat[i].f_mntonname)+2);
                 strcat(exclude_str,",");
              }
              
              strcat(exclude_str,fs_stat[i].f_mntonname);
              
           }
        }
        
        if (exclude_str)
          parse_exclude(exclude_str);
     }
#else
   /* Linux File System Status */
     {
      char *fbuf=NULL;
      char *head_ptr;
      char *tail_ptr;
      char *exclude_str=NULL;
      char *match_str=NULL;
   
      fbuf = load_file(MTAB_FILE);
      
      if (*fbuf == 0) {
         fbuf += 1;
         if (!QUIET)
           fprintf(stderr,"%s: File System Exclude: Could not open file %s: %s\n",progname,MTAB_FILE,fbuf);
         exit(1);
      }
      
      head_ptr = fbuf;
      
      while (head_ptr) {
         /* find filesystem type */
         if ((head_ptr = strchr(head_ptr,' '))) {
            head_ptr += 1;
            head_ptr = strchr(head_ptr,' ');
         }

         if (!head_ptr)
           continue;
         
         head_ptr += 1;
         
         tail_ptr = strchr(head_ptr,' ');
         if (!tail_ptr) {
            head_ptr = NULL;
            continue;
         }
         
         *tail_ptr = 0;
         
         /* Check if file sytem type exists in exclude string */

         match_str = realloc(match_str,strlen(head_ptr)+3);
         match_str[0] = '*';
         match_str[1] = 0;
         strcat(match_str,head_ptr);
         strcat(match_str,"*");
         
         for (ptr = match_str; *ptr != 0; ptr += 1)
           *ptr = toupper(*ptr);
         *tail_ptr = ' ';
         
         /* If a match is found, move back a few bytes to locate the path */
         if (strstr(estr,match_str)) {
            
            head_ptr -= 2;
            
            while (*head_ptr != ' ' && head_ptr != fbuf)
              head_ptr -= 1;
                                    
            if (head_ptr == fbuf) {
               if (!QUIET)
                 fprintf(stderr,"%s: File System Exclude: (1) corrupt mtab file: %s\n",progname,MTAB_FILE);
               exit(1);                  
            }
            
            head_ptr += 1;
            
            if (!(tail_ptr = strchr(head_ptr,' '))) {
               if (!QUIET)
                 fprintf(stderr,"%s: File System Exclude: (2) corrupt mtab file: %s\n",progname,MTAB_FILE);
               exit(1);
            }
            
            *tail_ptr = 0;
            
            /* Add the path to the exclude string */
            if (!exclude_str) {
               exclude_str = malloc(strlen(head_ptr)+1);
               *exclude_str = 0;
               strcat(exclude_str,head_ptr);
            } else {
               exclude_str = realloc(exclude_str,strlen(exclude_str)+strlen(head_ptr)+2);
               strcat(exclude_str,",");
               strcat(exclude_str,head_ptr);
            }
            
            *tail_ptr = ' ';
         }
         
         head_ptr = strchr(head_ptr,'\n');
                           
      }

# ifdef DEBUG
        printf("=-%s-=\n",exclude_str);
# endif
      
      if (exclude_str)
        parse_exclude(exclude_str);

   }
#endif   

   return 1;
}

#ifndef __FreeBSD__

/* Parse updatedb.conf file */

void
parse_updatedb_conf()
{
   char *fbuf;
   char *head_ptr;
   char *tail_ptr;
   char *ptr;
   char tmp_ch;
   char *parse_str=NULL;
   int times=0;
   char var_type[11];
   
   fbuf = load_file(UPDATEDB_CONF);
   
   if (*fbuf == 0) {
      fbuf += 1;
/*      if (!QUIET) {
         fprintf(stderr,"%s: could not access %s: %s\n",progname,UPDATEDB_CONF,fbuf);
         fflush(stderr);
      } */
      return;
   }
   
   for (times = 0; times < 2; times += 1) {
   
      head_ptr = fbuf;
      
      var_type[0] = 0;
      
      if (times == 0)
        strcat(var_type,"PRUNEFS");
      else
        strcat(var_type,"PRUNEPATHS");        
      
      if ((head_ptr = strstr(fbuf,var_type))) {
         ptr = head_ptr;
         while (ptr != fbuf && *ptr != '\n' && *ptr != '#')
           ptr -= 1;
         
         if (*ptr != '#') {
            while (*head_ptr != '"' && *head_ptr != '\n' && *head_ptr != '\0')
              head_ptr += 1;
            
            if (*head_ptr == '"') {
               head_ptr += 1;
               
               /* Get each file system from PRUNEFS variable */
               
               while (*head_ptr != '"' && *head_ptr != 0) {
                  
                  while (*head_ptr != 0 && isspace(*head_ptr))
                    head_ptr += 1;            
                  
                  if (*head_ptr == 0 || *head_ptr == '"')
                    break;
                  
                  tail_ptr = head_ptr;
                  while (*tail_ptr != 0 && !isspace(*tail_ptr) && *tail_ptr != '"')
                    tail_ptr += 1;
                  
                  if (*tail_ptr == 0)
                    break;
                  
                  tmp_ch = *tail_ptr;
                  
                  *tail_ptr = 0;
                  
                  if (!parse_str) {
                     parse_str = malloc(strlen(head_ptr)+1);
                     *parse_str = 0;
                  } else {
                     parse_str = realloc(parse_str,strlen(parse_str)+strlen(head_ptr)+2);
                     strcat(parse_str,",");
                  }
                  
                  strcat(parse_str,head_ptr);
                  
                  *tail_ptr = tmp_ch;
                  head_ptr = tail_ptr;
                  
               }
               
            }
         }
      }
      
      if (parse_str) {
         if (times == 0)     
           parse_fs_exclude(parse_str);
         else
           parse_exclude(parse_str);
      }      
      
      if (parse_str)
        free(parse_str);
      
      parse_str = NULL;
   }
   
# ifdef DEBUG
   printf("Final Exclude: %s\n",EXCLUDE_DIR);
   exit(0);   
# endif
}

#endif


/* Check to see if a path matches an excluded one. */

int
match_exclude(char *path,char *filedir) {
   int res=0;
   char *newstr;
   
   newstr = malloc(strlen(path)+strlen(filedir)+3);
   newstr[0] = '\0';
   strcat(newstr,"*");
   strcat(newstr,path);
   strcat(newstr,filedir);   
   strcat(newstr,"*");
   
   if (strstr(EXCLUDE_DIR,newstr) != NULL)       
       res=1;
   
   free(newstr);
   
   return res;
}

/* FRCODE - Incremental Encoding algorithm */

int
frcode(FILE *fd,char *dir_path, char *filename)
{
   int res=0;
   char *cur_path;
   char *new_path;
   char *ptr1;
   char *ptr2;

   int i;
   
   cur_path = malloc(strlen(dir_path)+strlen(filename)+1);
   cur_path[0]='\0';
   strcat(cur_path,dir_path);
   strcat(cur_path,filename);
   
   if (VERBOSE)
       fprintf(stdout,"%s\n",cur_path);
   
   if (tmp_path != NULL) {
      ptr1=cur_path;
      ptr2=tmp_path;
      
      for (i=0; ptr1[0] == ptr2[0] && (ptr1[0] != '\0' && ptr2[0] != '\0') ; i++) {
         ptr1++;
         ptr2++;
      }

      fr_num = i - t_num;
         
      t_num = t_num + fr_num;            
      
      ptr2 = strchr(ptr1,'\0');
      new_path=malloc((ptr2-ptr1)+1);
      new_path[0] = '\0';
      strncat(new_path,ptr1,ptr2-ptr1);         

   } else {
      new_path = malloc(strlen(cur_path)+1);
      new_path[0] = '\0';
      strcat(new_path,cur_path);
      t_num = strlen(new_path);
   }
   
   if (tmp_path != NULL)
       free(tmp_path);
   
   tmp_path = malloc(strlen(cur_path)+1);
   tmp_path[0] = '\0';
   strcat(tmp_path,cur_path);
   
   if (fr_num > 127 || fr_num < -127) {
      putc (SLOC_ESC, fd);
      put_short (fr_num, fd);
   } else
       putc(fr_num,fd);
   
   fputs(new_path,fd);
   putc('\0',fd);

   if (first) {
      chmod(TMPSLOCATEDB,00640);
      chown(TMPSLOCATEDB,UID,SLOC_GID);
      first=0;
   }
   
   free(new_path);
   free(cur_path);
   
   return res;
}

/* Create Database */

int
create_db(char *dirstr)
{
   int res=0;
   struct stat statres;
   FILE *fd;
   FTS *dir;
   FTSENT *file;
   int i;
   char **dirchk;
   

   if (!NEWOUTPUT && UID != SLOC_UID) {
      if (!QUIET)
          fprintf(stderr,"%s: You are not authorized to create a default slocate database!\n",progname);
      return(1);
   } 

   fd = fopen(TMPSLOCATEDB,"w");
   putc(slevel,fd);
   
   if (dirstr) {
      /* Make sure there are no problems accessing the source directory */
      if (lstat(dirstr,&statres) == -1) {
         if (!QUIET)
      fprintf(stderr,"%s: lstat: d%s/: %s\n",progname,dirstr,strerror(errno));
    fclose(fd);
         return(1);
      }
      
      if (strlen(dirstr) > 1) {
    if (dirstr[strlen(dirstr)-1] == '/')
      dirstr[strlen(dirstr)-1] = 0;        
      }
   }
   else {
      dirstr = malloc(2);
      *dirstr = 0;
      strcat(dirstr,"/");
   }
   
   dirchk = malloc(sizeof(char **)*2);
   
   *dirchk = dirstr;
   dirchk[1] = NULL;
   
   dir = fts_open(dirchk,FTS_PHYSICAL,NULL);   
   
   /* The new FTS() funtionality */
   
   for (i = 0; i > -1; i+=1) {
      file = fts_read(dir);
      
      if (!file)
   break;
      
      if (file->fts_info != FTS_DP && file->fts_info != FTS_NS) {
    
    if ((EXCLUDE && !match_exclude(file->fts_path,"")) || !EXCLUDE)
      frcode(fd,file->fts_path,"");
    else {
       fts_set(dir,file,FTS_SKIP);
    }
      }
      
   }
   
   fts_close(dir);
   
   fclose(fd);
   chdir(prog_CWD);
   rename(TMPSLOCATEDB,SLOCATEDB);
   chmod(SLOCATEDB,00640);
   chown(SLOCATEDB,UID,SLOC_GID);   
   return res;
}

/* Decode Database */

int
decode_db(char *database, char *str)
{
   int res = 1;
   FILE *fd;
   short code_num;
   int pathlen=0;
   register char ch;
   int grow=0;
   int jump=0;
   int first=1;
   char *codedpath=NULL;
   char *tmpcodedpath;    /* to fix around realloc() problem */
   char *ptr;
   int printit=0;
   int globflag=0;
   char *globptr1;
   struct stat statres;
   time_t now;
   char *chk1;
   char tmpch;
   regex_t *preg=NULL;
   char *errbuf=NULL;
   int nmatch=32;
   regmatch_t pmatch[32];
   int cur_queries = 0;
   int reg_res;
   int foundit = 0;
#ifndef FNM_CASEFOLD
   char *casestr=NULL;
   char *casecodedpath=NULL;
   char *cp=NULL;
#endif
     
   
   if ((fd = fopen(database,"r")) == NULL) {
      if (!QUIET)
          fprintf(stderr,"%s: decode_db(): %s: %s\n",progname,database,strerror(errno));
      return(0);
   }

   lstat(database,&statres);

   if (S_ISDIR(statres.st_mode)) {
   if (!QUIET)
        fprintf(stderr,"%s: decode_db(): %s is a directory\n",progname,database);
   return(0);
   }

   time(&now);
   if (now - statres.st_mtime > WARN_SECONDS)
   {
      fprintf(stderr,"%s: warning: database %s' is more than %s old\n",progname,database,WARN_MESSAGE);
   }   
   
   slevel = getc(fd);
   
   codedpath = malloc(MIN_BLK);
   codedpath[0] = '\0';
   ptr=codedpath;
   
   if ((globptr1 = strchr(str,'*')) != NULL ||
       (globptr1 = strchr(str,'?')) != NULL ||
       ((globptr1 = strchr(str,'[')) != NULL && strchr(str,']') != NULL))
       globflag = 1;
   
   if (REGEXP) {   
      preg = malloc(sizeof(regex_t));
      if ((reg_res = regcomp(preg,regexp,NOCASE?REG_ICASE:0)) != 0) {
         regerror(reg_res, preg, errbuf,1024);
         if (!QUIET)
           fprintf(stderr,"error: %s: regular expression: %s\n",progname,errbuf);
         exit(1);
      }
#ifndef FNM_CASEFOLD
      
   } else if (NOCASE) {
      casestr=strdup(str);
      for (cp = casestr; *cp; cp++)
        *cp = tolower(*cp);
#endif /* FNM_CASEFOLD */
      
   }
   
   while ((code_num = getc(fd)) != EOF) {
      
      if (code_num == SLOC_ESC) {
         code_num = get_short(fd);
      } else if (code_num > 127)
             code_num = code_num - 256;

      jump=0;
      
      if (code_num < 0)
         grow += code_num;
      
      ptr+=code_num;
      
      pathlen = ptr - codedpath;

      while(!jump) {
         ch = getc(fd);
         grow++;
         pathlen++;
         if (grow == 64) {

            tmpcodedpath = realloc(codedpath,pathlen+MIN_BLK);
            ptr += (tmpcodedpath - codedpath);
            codedpath = tmpcodedpath;
            
            grow=0;
         }
         codedpath[pathlen-1] = ch;
   
         if (ch == '\0')
            jump=1;
      }
      
      if (first) {
         ptr=ptr+strlen(codedpath);
         first=0;
      }
      
      pathlen--;            
      
      printit=0;
      
      if (REGEXP) {
         foundit = !regexec(preg,codedpath,nmatch,pmatch,0);         
      } else if (NOCASE) {
#ifdef FNM_CASEFOLD /* i suppose i also have strcasestr */
         if (globflag) {
            foundit=!fnmatch(str,codedpath,FNM_CASEFOLD);
            
         } else {
            foundit=(strcasestr(codedpath,str) != NULL);
         }
#else /* FNM_CASEFOLD */
         casecodedpath=strdup(codedpath);
         for (cp = casecodedpath; *cp; cp++)
           *cp = tolower(*cp);
         if (globflag) {
            foundit=!fnmatch(casestr,casecodedpath,0);
         } else {
            foundit=(strstr(casecodedpath,casestr) != NULL);            
         }
         free(casecodedpath);
#endif /* FNM_CASEFOLD */

      } else {
         if (globflag) {
            foundit=!fnmatch(str,codedpath,0);
         } else {
            foundit=(strstr(codedpath,str) != NULL);
         }
      }
      
      if (foundit) {
         if (slevel == '1') {
            if (UID == 0 || lstat(codedpath,&statres) == 0) {
               printit=1;
               chk1 = codedpath+1;
               while ((chk1 = strchr(chk1,'/')) != NULL) {
                  chk1++;
                  tmpch=*chk1;
                  *chk1='\0';

                  if (access(codedpath, R_OK) == -1)
                    printit=0;
                  *chk1=tmpch;
               }
            }
         } else
             printit=1;
      }
      
      if (printit) {
         res = 0;
         cur_queries += 1;
         printf("%s\n",codedpath);
         if (max_queries > 0 && cur_queries == max_queries)
           exit(0);
      }
   }
   
   if (REGEXP)
     regfree(preg);
#ifndef FNM_CASEFOLD
   else if (NOCASE)
     free(casestr);
#endif /* FNM_CASEFOLD */
     
   
   fclose(fd);
   
   return(res);
}

/* Main Function */

int
main(int args, char **argv)
{
   int res=0;
   int ch;
   extern char *optarg;
   extern int optind, opterr, optopt;
   char *p;
   int SPECDIR=0;
   int ROOTDIR=0;   
   char *spec_dir=NULL;
   char *database;
   int i=0;


   progname = ((p = strrchr(argv[0],'/')) ? p+1 : *argv);     
   
   if (!strcmp(progname,"updatedb")) {
      ROOTDIR=1;
      #ifndef __FreeBSD__
        parse_updatedb_conf();
      #endif
   }
   
   /* Get Current Working Directory */
   
   getcwd(prog_CWD, 4095);
   
   if (strlen(prog_CWD) == 4095) {
      fprintf(stderr,"%s: Current Working Directory is too large!",progname);
      exit(1);
   }

   
   if (args < 2 && !ROOTDIR)
     usage();
   
   if (!check_dir())
     return(1);
   
   UID = getuid();        
   GID = getgid();

   parse_decode_path(SLOCATEDB);
   parse_decode_path(getenv("LOCATE_PATH"));

   while ((ch = getopt(args,argv,"VvuhqU:r:o:e:l:d:-:n:f:ci")) != EOF) {         
      switch(ch) {
       case 'h':
         usage();
         break;
       case 'q':
         QUIET=1;
         break;
       case 'V':
         printf("%s",VERSION);
         exit(0);
         break;
       case 'v':
         VERBOSE=1;
         break;
       case 'e':
         parse_exclude(optarg);
         break;
       case 'f':
         parse_fs_exclude(optarg);
         break;
       case 'c':
#ifndef __FreeBSD__
         parse_updatedb_conf();
#endif
         break;
       case 'l':
         slevel = optarg[0];
         if (slevel != '0' && slevel != '1') {
            if (!QUIET)
              fprintf(stderr,"%s: ERROR: Security level must be 0 or 1.\n",progname);
            exit(1);
         }
         break;
       case 'u':
         ROOTDIR=1;
         break;
       case 'U':
         ROOTDIR=0;
         SPECDIR=1;
         spec_dir = malloc(strlen(optarg)+1);
         *spec_dir = 0;         
         strcat(spec_dir,optarg);
         break;
       case 'r':
         REGEXP = 1;
         regexp = malloc(strlen(optarg)+1);
         *regexp = 0;
         strcat(regexp,optarg);
         while ((database = SLOCATE_PATH[i++])) res |= decode_db(database, optarg);
         return(res);
         break;
       case 'o':
         parse_create_path(optarg);
         NEWOUTPUT=1;
         break;
       case 'd':
         parse_decode_path(optarg);
         break;
       case '-':
         parse_dash(optarg);
         break;
       case 'n':
         for (i=0; i < strlen(optarg); i+=1) {
            if (!isdigit(optarg[i])) {
               fprintf(stderr,"%s: ERROR: Invalid argument for option -n\n",progname);
               exit(1);
            }
         }
         max_queries = atoi(optarg);
         break;
       case 'i':
         NOCASE=1;         
         break;
       default:
         return(1);
         break;
         
      }
   }
   
   SLOC_GID = get_gid(GRPFILE);
   
   if (SPECDIR) {
      res = create_db(spec_dir);
   } else if (ROOTDIR)
      res = create_db((char *)NULL);
   else if (argv[optind]) {
      int j = optind;
      int i = 0;
      
      while (j < args) {
         /* while ((database = SLOCATE_PATH[i++])) res |= decode_db(database, argv[j++]);
            i = 0;
          * A Bug fix by Hans-Juergen Godau <godau@wi-inf.uni-essen.de>
          * Prevents segfault when using multiple databases */
         while ((database = SLOCATE_PATH[i++]))
           res |= decode_db(database, argv[j]);
         i = 0; j += 1;
         
      }
   } else
       usage();

   return(res);
}
