// Epilog - Generic text file analysis and forwarding
// Copyright 2001-2007 InterSect Alliance Pty Ltd
// http://www.intersectalliance.com/
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// See Readme.txt file for more information.
//
// NOTE: Use the \data\64bit.bat file to compile in 64 bit mode!!!!
#include <stdio.h>
#include <stdlib.h>
#include <winsock2.h>
#include <windows.h>
#include <time.h>
#include <tchar.h>
#include <sys/stat.h>

#include "NTServApp.h"
#include "Epilog.h"
#include "webserver.h"
#include "MD5.h"
#include "Version.h"

// memory testing only
//#define MEMDEBUG 1

#ifdef MEMDEBUG
// MEMORY LEAK DETECTION ROUTINES
	// #define _CRTDBG_MAP_ALLOC
	#define CRTDBG_MAP_ALLOC
	#include <stdlib.h>
	#include <crtdbg.h>
#endif

// #define DEBUG_TO_FILE 1

// Quick function proto
void DebugMsg(const char* pszFormat, ...);


// Pull this from registry
DWORD			WEBSERVER_ACTIVE = 0;

DWORD			g_dwLastError			= 0L;
char			Hostname[100];
ThreadStruct	g_Info;
//SOCKET *		g_hSockets;						// Array of socket pointers - one for each destination.
//char **			SocketNames;					// HostNames for each of the sockets.
//int				SocketCount=0;
//SOCKET			g_hSocket = INVALID_SOCKET;     // client/server socket, used in for/next loops.
int				nStopListening=0;
TCHAR			DELIM[2]="	";					// TAB

int				SNAREDEBUG=0;
int LCCount=0;
MsgCache *LCHead=NULL;
MsgCache *LCTail=NULL;
MsgCache *LCCurrent=NULL;

LogNode *lhead=NULL;
LogNode *ltail=NULL;
LogNode *lcurrent=NULL;

int MCCount=0;
EventCache *MCHead=NULL;
EventCache *MCTail=NULL;
EventCache *MCCurrent=NULL;

// Locker
HANDLE hMutex;


static Node *head, *currentnode;

// Host target list
static HostNode *hosthead, *hostcurrentnode;

CEpilogService::CEpilogService()
:CNTService("Epilog")
{
	//TODO: Initialize your class members here
}

BOOL CEpilogService::OnInit()
{
	char szError[MAX_STRING];
	//TODO: Perform any initialization that needs to be done before intering the main loop

	SNAREDEBUG=this->DEBUGSET;

	if( !InitWinsock( szError, _countof(szError) ) )
	{
		if(SNAREDEBUG) { DebugMsg(szError); }
		return FALSE;
	}

	// send_logs thread
	m_hEventList[0] = CreateEvent(NULL, TRUE, FALSE, NULL);

	// Web server
	m_hEventList[1] = CreateEvent(NULL, TRUE, FALSE, NULL);

	// Web server reset
	m_hEventList[2] = CreateEvent(NULL, TRUE, FALSE, NULL);

	
	if(m_hEventList[0] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() send_logs thread failed"); }	return FALSE;	}
	if(m_hEventList[1] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() Web Server failed"); }	return FALSE;	}
	if(m_hEventList[2] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() Web Server Reset failed"); }	return FALSE;	}

	hMutex = CreateMutex(NULL,FALSE,"EpilogAgentLock");
	if(hMutex == NULL) {
		if(SNAREDEBUG) { DebugMsg("I cannot create the Epilog Agent 'Mutex' lock. This probably means that you already have another instance of the Epilog Agent running.\nPlease stop the other incarnation of the Epilog Agent (eg: net stop epilog) before continuing."); }
		return FALSE;
	}

	if(SNAREDEBUG) { DebugMsg("Epilog Initialisation complete"); }
	// return FALSE here if initialization failed & the service shouldn't start
	return TRUE;
}

void CEpilogService::Run()
{
	DWORD dwSyslogHeader=0; // Send the Syslog header?
	DWORD dwPortNumber=6162;
	DWORD dwRestrictIP=0;
	DWORD dwUsePassword=0;

	// Define a log buffer of 8k.
	// Should be enough for an overwhelming majority of circumstances.
	TCHAR logbuffer[MAX_EVENT]="";

  	TCHAR szError[MAX_STRING]="";

	TCHAR lpszIPAddress[16];
	TCHAR lpszPassword[256];
	errno_t err;

	short nEventCount=3; // including the web server event

	// Destination for log events. Default it to something safe.
	TCHAR lpszDestination[SIZE_OF_DESTINATION]="127.0.0.1";

	DWORD SocketType=SOCKETTYPE_UDP;	// 0 for UDP
	DWORD dwWaitRes=0;
	DWORD dwDestPort=6161;
	DWORD dwSyslog=13;
	DWORD dwObjectiveCount=0;
	Node **MatchList;
	BOOL MatchFound=0;

	// Syslog, and output log time variables.
	time_t currenttime;
	struct tm *newtime;

	if(SNAREDEBUG) { DebugMsg("Epilog is Running"); }

	// READ in our data
	GetHostname(Hostname,_countof(Hostname));
	GetDestPort(&dwDestPort);
	GetSocketType(&SocketType);
	GetSyslog(&dwSyslog);
	GetSyslogHeader(&dwSyslogHeader);
	GetWEBSERVER_ACTIVE(&WEBSERVER_ACTIVE);
	GetPortNumber(&dwPortNumber);
	GetDestination(lpszDestination,_countof(lpszDestination));
	ReadLogs();
	if(dwSyslogHeader) {
		GetDelim(DELIM,_countof(DELIM));
	}
	GetPassword(lpszPassword,_countof(lpszPassword));
	GetIPAddress(lpszIPAddress,_countof(lpszIPAddress));

	// Load the objective data here.
	dwObjectiveCount=ReadObjectives();
	if(dwObjectiveCount) {
		// Malloc a array for our FastCheckObjective process
		MatchList = (Node **)malloc(dwObjectiveCount * sizeof(Node *));
		if(!MatchList) {
			if(SNAREDEBUG) DebugMsg("Cannot allocate memory for our internal Objective match list");
			dwObjectiveCount=0;
		}
	}


	hosthead=OpenSockets(lpszDestination,dwDestPort,SocketType);

	
	if(SNAREDEBUG) DebugMsg("Sockets opened/connected");
	////////////////// REMEMBER TO FREE THESE SOCKETS LATER!!!!   REDREDRED

	// Open the socket (UDP or TCP) to the indicated port
	//g_hSocket = ConnectToServer( lpszDestination, (UINT)dwDestPort, szError,_countof(szError), SocketType );

	//if( g_hSocket == INVALID_SOCKET ) {
	//	DebugMsg(szError);
	//}

	// Ok, we have finished our general configuration reads.
	if(WEBSERVER_ACTIVE) {
		if(SNAREDEBUG) { DebugMsg("Starting web thread.\n"); }
		if(InitWebServer((unsigned short)dwPortNumber,lpszPassword,lpszIPAddress) >0) {
			StartThread(m_hEventList[1]);
		} else {
			WEBSERVER_ACTIVE = 0;
		}
	}

	// Set the terminate flag to zero.
	// setting this value to TRUE will terminate the service,
	// and ask it to save it's current status.
	g_Info.bTerminate = FALSE;

	if(SNAREDEBUG) { DebugMsg("Entering main loop."); }
	// This is the service's main run loop.
	while (m_bIsRunning) {
		// TODO: Add code to perform processing here  
		// If we have been asked to terminate, do so.
		if(g_Info.bTerminate) {
			m_bIsRunning=0;
			break;
		}	
		
		// The service performs one check per 5 seconds. This should not be
		// a significant drain on resources.
		dwWaitRes=WaitForMultipleObjects(nEventCount,m_hEventList,FALSE,1000);

		// if(dwWaitRes != WAIT_FAILED && dwWaitRes != WAIT_TIMEOUT)
		if(dwWaitRes != WAIT_FAILED) {
			if (dwWaitRes == WAIT_OBJECT_0) {
				ResetEvent(m_hEventList[0]);
				if(SNAREDEBUG) { DebugMsg("Sending Logs"); }
				MsgCache *mymsg;
				errno_t err;

				mymsg = LCHead;
				//while (mymsg && !caught_kill) {
				while (mymsg) {
					char header[256];
					char CurrentDate[16]="";
					static char szSendString[MAX_OUTPUT_STRING]=""; // Nice big memory buffer - just in case.
					
					// Check objectives
					// NOTE: If there are no objectives, send all?
					if(dwObjectiveCount) {
						if(g_Info.bTerminate) {
							m_bIsRunning=0;
							break;
						}
					}
					
					if(dwSyslogHeader) {
						time(&currenttime);
						newtime = (struct tm *)malloc(sizeof(struct tm));
						err=localtime_s(newtime,&currenttime);
						syslogdate(CurrentDate,newtime);
						_snprintf_s(header,_countof(header),_TRUNCATE,"<%ld>%s %s %s%s0%s",dwSyslog,CurrentDate,Hostname,mymsg->type,DELIM,DELIM);
					} else {
						if (strncmp("GenericLog",mymsg->type,10))
							_snprintf_s(header,_countof(header),_TRUNCATE,"%s%s%s%s0%s",Hostname,DELIM,mymsg->type,DELIM,DELIM);
						else
							_snprintf_s(header,_countof(header),_TRUNCATE,"%s%sGenericLog%s0%s0%s0%s0%s",Hostname,DELIM,DELIM,DELIM,DELIM,DELIM,DELIM);
					}
					
					_snprintf_s(szSendString,_countof(szSendString),_TRUNCATE,"%s%s\0",header,mymsg->msg);
					
					if(SNAREDEBUG && szSendString) { DebugMsg("DEBUG: Sending the following string to the server: %s",szSendString); }
					
					BOOL DataSent=0;

					hostcurrentnode=hosthead;
					while(hostcurrentnode) {
						if(hostcurrentnode->Socket == INVALID_SOCKET) {
							// Try to reestablish here.
							// Since socket connects use a fair bit of system resources, try and do it nicely.
							if(SNAREDEBUG) { DebugMsg("Socket is toast for %s. Trying to reestablish.",hostcurrentnode->HostName); }
							
							hostcurrentnode->Socket = ConnectToServer( hostcurrentnode, (UINT)dwDestPort, szError, _countof(szError), SocketType );
							
							if(hostcurrentnode->Socket == INVALID_SOCKET) {
								// Hmm. Try again later.
								// Jump to the next socket
								hostcurrentnode=hostcurrentnode->next;
								continue;
							}
						}

						
						if( !SendToSocket(hostcurrentnode, szSendString, (int)strlen(szSendString), szError, _countof(szError)) )
						{
							if(SNAREDEBUG) { if(szError) { DebugMsg(szError); } }
							if(SNAREDEBUG) { DebugMsg("Socket for %s is toast. Breaking out - will reestablish next time.",hostcurrentnode->HostName); }
							// Close the socket. Restablish it on the next cycle, if we can.
							closesocket(hostcurrentnode->Socket);
							hostcurrentnode->Socket=INVALID_SOCKET;
						} else {
							DataSent=1;
						}

						hostcurrentnode=hostcurrentnode->next;
					}


					// Did we push out at least one record?
					if(!DataSent) {
						// Break out of the for/next loop.
						if(SNAREDEBUG) { DebugMsg("FAILED TO SEND, breaking out..."); }
						break;
						// need to implement cache here DMM
					}
					MCCurrent = (EventCache *)malloc(sizeof(EventCache));
					if (MCCurrent) {
						strncpy_s(MCCurrent->Hostname,_countof(Hostname),Hostname,_TRUNCATE);
						time_t ttime=time(NULL);
						struct tm *ptmTime;
						errno_t err;
						ptmTime = (struct tm *)malloc(sizeof(struct tm));
						err = localtime_s(ptmTime,&ttime);
						strftime(MCCurrent->SubmitTime, _countof(MCCurrent->SubmitTime),"%a %b %d %H:%M:%S %Y", ptmTime);
						strncpy_s(MCCurrent->msg, MAX_OUTPUT_STRING, mymsg->msg,_TRUNCATE);
						strncpy_s(MCCurrent->type, MAX_TYPE, mymsg->type,_TRUNCATE);
						MCCurrent->seenflag=0;
						MCCurrent->next = NULL;
						MCCurrent->prev = NULL;
						dwWaitRes = WaitForSingleObject(hMutex,500);
						if(dwWaitRes == WAIT_OBJECT_0) {
							if (MCCount >= WEB_CACHE_SIZE) {
								//Lock Mutex and drop the oldest record
								EventCache *temp;
								temp = MCTail;
								MCTail = MCTail->prev;
								MCTail->next = NULL;
								free(temp);
								MCCount--;
							}
						}
						ReleaseMutex(hMutex);
						if (MCHead) {
							MCHead->prev = MCCurrent;
							MCCurrent->next = MCHead;
						} 
						MCHead = MCCurrent;
						if (!MCTail) MCTail = MCCurrent;
						MCCount++;
					} else {
						DebugMsg("Unable to allocate cache\n");
					}
					LCHead = mymsg->next;
					free(mymsg);
					LCCount--;
					mymsg = LCHead;
					if (!mymsg) {
						LCTail=NULL;
					}
				}
			} else if (dwWaitRes == WAIT_OBJECT_0+1) {
				// Web server has data to read.
				ResetEvent(m_hEventList[1]);
				if(SNAREDEBUG) { DebugMsg("WEB Server Connect."); }

				if(WEBSERVER_ACTIVE) {
					// Let handleconnect have the eventlist pointer, in case it needs
					// to signal a restart
					if(SNAREDEBUG) { DebugMsg("About to HandleConnect"); }
					HandleConnect(m_hEventList[2]);

					if(SNAREDEBUG) { DebugMsg("Running thread again."); }
					StartThread(m_hEventList[1]);
				}
				
				continue;
			} else if (dwWaitRes == WAIT_OBJECT_0+2) {

				// We need to re-read our configuration file.
				ResetEvent(m_hEventList[2]);
			    
				DestroyList();

				if(WEBSERVER_ACTIVE) {
					CloseWebServer();
				}

				// cancel blocking calls, if any
				WSACancelBlockingCall();
	
				// Close all active sockets.
				HostNode * temphostnode;

				hostcurrentnode=hosthead;
				while(hostcurrentnode) {
					if(hostcurrentnode->Socket != INVALID_SOCKET) {
						closesocket(hostcurrentnode->Socket);
						hostcurrentnode->Socket=INVALID_SOCKET;
					}
					temphostnode=hostcurrentnode->next;

					// Free the RAM associated with this node. We don't need it any more.
					free(hostcurrentnode);
					hostcurrentnode=temphostnode;
					// Just in case
					hosthead=hostcurrentnode;
				}
				hosthead=NULL;


				// READ in our data
				GetHostname(Hostname,_countof(Hostname));
				GetDestPort(&dwDestPort);
				GetSocketType(&SocketType);
				GetSyslog(&dwSyslog);
				GetSyslogHeader(&dwSyslogHeader);
				GetWEBSERVER_ACTIVE(&WEBSERVER_ACTIVE);
				GetPortNumber(&dwPortNumber);
				GetDestination(lpszDestination,_countof(lpszDestination));
				ReadLogs();
				if(dwSyslogHeader) {
					GetDelim(DELIM,_countof(DELIM));
				}
				GetPassword(lpszPassword,_countof(lpszPassword));
				GetIPAddress(lpszIPAddress,_countof(lpszIPAddress));
	
				if(MatchList) {
					free(MatchList);
				}
				// Load the objective data here.
				dwObjectiveCount=ReadObjectives();
				if(dwObjectiveCount) {
					// Malloc a array for our FastCheckObjective process
					MatchList = (Node **)malloc(dwObjectiveCount * sizeof(Node *));
					if(!MatchList) {
						if(SNAREDEBUG) DebugMsg("Cannot allocate memory for our internal Objective match list");
						dwObjectiveCount=0;
					}
				}
				ResetCurrentNode();

				// Open our outgoing sockets.
				hosthead=OpenSockets(lpszDestination,dwDestPort,SocketType);

				// Ok, we have finished our general configuration reads.

				if(WEBSERVER_ACTIVE) {
					if(SNAREDEBUG) { DebugMsg("Starting web thread."); }
					if(InitWebServer((unsigned short)dwPortNumber,lpszPassword,lpszIPAddress) >0) {
						StartThread(m_hEventList[1]);
					} else {
						WEBSERVER_ACTIVE = 0;
					}
				}

				continue;
			} else if (dwWaitRes == WAIT_TIMEOUT) {
 				if(SNAREDEBUG) { DebugMsg("Timeout hit"); }
			} else {
				if(SNAREDEBUG) { DebugMsg("Warning: An event occured that I am not programmed to deal with. Continuing"); }
				continue;
			}
			lcurrent = lhead;
			while (lcurrent) {
				if(SNAREDEBUG) DebugMsg("Checking for changes in: %s", lcurrent->name);
				int ret,length;
				int big_msg=0;
				int grab_count=0;
				ret = file_has_changed(lcurrent);
				if (ret > 0) {
					if(SNAREDEBUG) DebugMsg("Found a change, grabbing and sending");
					err = fopen_s(&lcurrent->fs,lcurrent->name, "rb");
					if (err) {
						if(SNAREDEBUG) DebugMsg("Failed to get file pointer");
						continue; //If this happens, something is very wrong
					}
					fseek(lcurrent->fs, lcurrent->size, SEEK_SET);
					lcurrent->old_size = lcurrent->size;
					//lcurrent->old_size = ftell(lcurrent->fs);
					while ((length = GetLine(lcurrent->fs, logbuffer, MAX_EVENT)) > 0) {
						if (!strncmp(logbuffer,"          ",10)) break;
						lcurrent->size=lcurrent->size + length;
						if (logbuffer[0] == '#') continue;
						if (logbuffer[0] == '\n') continue;
						if (big_msg) {
							if (logbuffer[strlen(logbuffer) - 1] == '\n') {
								big_msg = 0;
							}
							continue;
						}
						if (lcurrent->pmsg && strlen(lcurrent->pmsg)) {
							strncat_s(lcurrent->pmsg, MAX_EVENT - strlen(lcurrent->pmsg) - 1, logbuffer,_TRUNCATE);
							strncpy_s(logbuffer, MAX_EVENT, lcurrent->pmsg,_TRUNCATE);
							lcurrent->pmsg[0]='\0';
						}
						if (logbuffer[strlen(logbuffer) - 1] != '\n') {
							if (strlen(logbuffer) >= MAX_EVENT -1) {
								logbuffer[MAX_EVENT]='\0';
								big_msg = 1;
							} else {
								strncpy_s(lcurrent->pmsg, MAX_EVENT, logbuffer,_TRUNCATE);
								continue;
							}
						}
						if (logbuffer[strlen(logbuffer) - 1] == '\n') logbuffer[strlen(logbuffer) - 1]='\0';
						if (logbuffer && CheckObjective(logbuffer)) {
							//#######################################
							if (LCCount > DEFAULT_CACHE) {
								if(SNAREDEBUG) DebugMsg("Log Cache FULL - deleting oldest message\n");
								LCCurrent = LCHead;
								LCHead = LCCurrent->next;
								free(LCCurrent);
							}
							LCCurrent = (MsgCache *)malloc(sizeof(MsgCache));
							if (LCCurrent) {
								LCCurrent->msglen=strlen(logbuffer);
								strncpy_s(LCCurrent->msg, _countof(LCCurrent->msg), logbuffer,_TRUNCATE);
								LCCurrent->msg[LCCurrent->msglen]='\0';
								LCCurrent->next=NULL;
								strncpy_s(LCCurrent->type, MAX_TYPE, lcurrent->type,_TRUNCATE);
							} else {
								if(SNAREDEBUG) DebugMsg("Unable to allocate cache\n");
							}
							LCCount++;
							if (LCTail) {
								LCTail->next=LCCurrent;
							}
							LCTail=LCCurrent;
							if (!LCHead) {
								LCHead=LCCurrent;
							}
							if(SNAREDEBUG) DebugMsg("About to set event\n");
							::SetEvent(m_hEventList[0]);
							//#######################################
						}
						grab_count++;
						if (grab_count >= MAX_LINE_GRAB) {
						//if (grab_count >= 1) {
							if(SNAREDEBUG) DebugMsg("Breaking out to allow events to send\n");
							//lcurrent->size=ftell(lcurrent->fs);
							break;
						}
					}
					//lcurrent->size = ftell(lcurrent->fs);
					fclose(lcurrent->fs);
				} else {
					// The file hasn't changed OR
					// there was an error that we can't do anything about
					if(SNAREDEBUG) DebugMsg("No changes found");
				}
				lcurrent = lcurrent->next;
			}
		}
	}

	if(SNAREDEBUG) { DebugMsg("Epilog Closing"); }
	free(MatchList);

	CloseWebServer();

	HostNode * temphostnode;
	// Free our host linked list.
	hostcurrentnode=hosthead;
	while(hostcurrentnode) {
		if(hostcurrentnode->Socket != INVALID_SOCKET) {
			TerminateWinsock(hostcurrentnode->Socket);
		}
		temphostnode=hostcurrentnode;
		hostcurrentnode=hostcurrentnode->next;
		free(temphostnode);
	}

	if( m_hEventList[1] ) ::CloseHandle(m_hEventList[1]);

	// Free memory used by the objectives lists
	DestroyList();
}

void GetFQDN(char *string, const int length)
{
	struct hostent *phostent;
	
	if(!string) return;

	// Now, grab fully qualified hostname here.
	// Get the normal name.
	if(gethostname(string, length)) {		
		strncpy_s(string,length,"localhost.unknown",_TRUNCATE);
		return;
	}
	
	// Now perform a lookup on that name.
	phostent=gethostbyname(string);
	if(phostent) {
		while(phostent->h_aliases && *phostent->h_aliases) {
			if(strlen(*(phostent->h_aliases)) > strlen(string) && !strncmp(string,*(phostent->h_aliases),strlen(string))) {
				strncpy_s(string,length,*(phostent->h_aliases),_TRUNCATE);
			}
			phostent->h_aliases++;
		}
		if(strlen(phostent->h_name) > strlen(string) && !strncmp(string,phostent->h_name,strlen(string))) {
			strncpy_s(string,length,phostent->h_name,_TRUNCATE);
		}
	}
}

// Match a DOS wildcard against a string.
// eg: wildmatch("c:\blah\abc???.*","c:\blah\abc123.txt");
int wildmatch(char *pattern, char *source)
{
	if(!pattern || !source) {
		return(0);
	}
    if(*pattern == 0)
        return (*source == 0);

    // special case
    if(strspn(pattern, "*") == strlen(pattern))
        return 1;

    if(*source == 0)
        return 0;

    switch(*pattern){
    case '*':
        return wildmatch(pattern, source+1) || wildmatch(pattern+1, source) || wildmatch(pattern+1, source+1);
    case '?':
        return wildmatch(pattern+1, source+1);
    default:
        return (*pattern == *source) && wildmatch(pattern+1, source+1);
    }
}

// Match a DOS wildcard against a string.
// eg: wildmatch("C:\blah\abc???.*","c:\blah\abc123.txt");
int wildmatchi(char *pattern, char *source)
{
	if(!pattern || !source) {
		return(0);
	}
    if(*pattern == 0)
        return (*source == 0);

    // special case
    if(strspn(pattern, "*") == strlen(pattern))
        return 1;

    if(*source == 0)
        return 0;

    switch(*pattern){
    case '*':
        return wildmatchi(pattern, source+1) || wildmatchi(pattern+1, source) || wildmatchi(pattern+1, source+1);
    case '?':
        return wildmatchi(pattern+1, source+1);
    default:
		char lpattern,lsource;
		lpattern = tolower(*pattern);
		lsource = tolower(*source);
		
        return (lpattern == lsource) && wildmatchi(pattern+1, source+1);
    }
}



void syslogdate(char *sdate, struct tm *cdate)
{
	char Month[4];
	char Date[3];
	char Hour[3];
	char Min[3];
	char Sec[3];

	if(!sdate || !cdate) return;

	switch (cdate->tm_mon) {
		case 0: strcpy_s(Month,_countof(Month),"Jan"); break;
		case 1: strcpy_s(Month,_countof(Month),"Feb"); break;
		case 2: strcpy_s(Month,_countof(Month),"Mar"); break;
		case 3: strcpy_s(Month,_countof(Month),"Apr"); break;
		case 4: strcpy_s(Month,_countof(Month),"May"); break;
		case 5: strcpy_s(Month,_countof(Month),"Jun"); break;
		case 6: strcpy_s(Month,_countof(Month),"Jul"); break;
		case 7: strcpy_s(Month,_countof(Month),"Aug"); break;
		case 8: strcpy_s(Month,_countof(Month),"Sep"); break;
		case 9: strcpy_s(Month,_countof(Month),"Oct"); break;
		case 10: strcpy_s(Month,_countof(Month),"Nov"); break;
		default: strcpy_s(Month,_countof(Month),"Dec"); break;
	}

	if(cdate->tm_mday<10) {
		_snprintf_s(Date,3,_TRUNCATE," %d\0",cdate->tm_mday);
	} else {
		_snprintf_s(Date,3,_TRUNCATE,"%d\0",cdate->tm_mday);
	}

	if(cdate->tm_hour<10) {
		_snprintf_s(Hour,3,_TRUNCATE,"0%d\0",cdate->tm_hour);
	} else {
		_snprintf_s(Hour,3,_TRUNCATE,"%d\0",cdate->tm_hour);
	}

	if(cdate->tm_min<10) {
		_snprintf_s(Min,3,_TRUNCATE,"0%d\0",cdate->tm_min);
	} else {
		_snprintf_s(Min,3,_TRUNCATE,"%d\0",cdate->tm_min);
	}

	if(cdate->tm_sec<10) {
		_snprintf_s(Sec,3,_TRUNCATE,"0%d\0",cdate->tm_sec);
	} else {
		_snprintf_s(Sec,3,_TRUNCATE,"%d\0",cdate->tm_sec);
	}

	_snprintf_s(sdate,16,_TRUNCATE,"%s %s %s:%s:%s\0",Month,Date,Hour,Min,Sec);
}


//////////////////////////////////////////////////////////////////
// SendToSocket
//              sends a buffer (buf) of size size to the specified socket
//              returns TRUE or FALSE.
BOOL  SendToSocket(HostNode *hcn, char *buf, int nSize, char *szError, int eSize)
{
	// int   rv;
	int bytessent=0;

	if(!buf || !szError || !eSize || !hcn || hcn->Socket == INVALID_SOCKET) return(0);

	do {
		bytessent = sendto(hcn->Socket,buf,nSize,0,(SOCKADDR *)&hcn->server,sizeof(hcn->server));
		if(bytessent==-1) {
			if(SNAREDEBUG) { DebugMsg("error sending to server. WSA ERROR: %d",WSAGetLastError()); }
			return FALSE;
		}
		buf+= bytessent;
		nSize -= bytessent;
	} while(nSize > 0);

	return TRUE;
}


///////////////////////////////////////////////////////////////////
// InitWinsock
//              starts up winsock.dll or wsock32.dll
BOOL InitWinsock( char *szError, int size )
{
	WSAData wsData;

	if(!szError) return(FALSE);
	
	WORD wVersionRequested = WINSOCK_VERSION;
	
	if(WSAStartup(wVersionRequested, &wsData) != 0)
	{
		// :( error
		if( szError )
		{
			sprintf_s(szError,size,"WSAStartup failed: WSA ERROR: %d\r\n",
				WSAGetLastError());
			if(SNAREDEBUG) { if(szError) { DebugMsg(szError); } }
		}
		return FALSE;
	}
	
	// all is well
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////
// TerminateWinsock
//      call this function with the current socket or INVALID_SOCKET
void TerminateWinsock( SOCKET hSocket )
{
	// cancel blocking calls, if any
	WSACancelBlockingCall();
	
	// close socket
	if( hSocket != INVALID_SOCKET ) {
		closesocket(hSocket);
		hSocket=INVALID_SOCKET;
	}
		
	// unload winsock
	WSACleanup();
}

// Note that hosthead is a global.
HostNode * OpenSockets(char *lpszDestination, DWORD dwDestPort, DWORD SocketType)
{
	HostNode * headnode=NULL;
	char szError[MAX_STRING];

	headnode=(HostNode *)malloc(sizeof(HostNode));
	if(!headnode) {
		// Oh dear.. out of RAM?
		return(headnode);
	}

	headnode->next=NULL;
	strncpy_s(headnode->HostName,511,lpszDestination,_TRUNCATE);
	headnode->Socket=ConnectToServer( headnode, (UINT)dwDestPort, szError,_countof(szError), SocketType );
	if(headnode->Socket == INVALID_SOCKET) {
		DebugMsg("Problem opening Socket to %s: %s",lpszDestination,szError);
	}
		
	return(headnode);
}



//////////////////////////////////////////////////////////////
// ConnectToServer:
//    connects to a server on a specified port number
//    returns the connected socket
SOCKET ConnectToServer(HostNode *hcn, UINT nPort, char *szError, int size, DWORD SocketType)
{
	SOCKET hSocket;
	struct hostent far *hp;
	
	
	if(!hcn->HostName) return INVALID_SOCKET;
	if(!szError || !size) return INVALID_SOCKET;
	if(!*hcn->HostName) return INVALID_SOCKET;
	
	hcn->SocketType = SocketType;
	// Should use something nicer to check for IP addresses...
	if( isdigit(hcn->HostName[0])) {
		ZeroMemory((char *) &hcn->server, sizeof(hcn->server));
		hcn->server.sin_family      = AF_INET;
		hcn->server.sin_addr.s_addr = inet_addr(hcn->HostName);
		hcn->server.sin_port        = htons(nPort);
	} else {
		if ( (hp = (struct hostent far *) gethostbyname(hcn->HostName)) == NULL)	{
			_snprintf_s(szError,size,_TRUNCATE,"Error: gethostbyname failed: %s.",hcn->HostName);
			return INVALID_SOCKET;
		}
		
		ZeroMemory((char *)&hcn->server, sizeof(hcn->server));
		CopyMemory((char *) &hcn->server.sin_addr,hp->h_addr,hp->h_length);
		hcn->server.sin_family = hp->h_addrtype;
		hcn->server.sin_port = htons(nPort);
	}

	// create socket
	if((hSocket = socket(AF_INET, SOCK_DGRAM, 0)) == INVALID_SOCKET) {
		_snprintf_s(szError,size,_TRUNCATE,"socket failed to create datagram socket: %d\n",WSAGetLastError());
		return INVALID_SOCKET;
	}
	return hSocket;
}

int CheckObjective(char *searchterm)
{
	if(!searchterm) {
		if(SNAREDEBUG) { DebugMsg("CheckObjective: No Search Term supplied"); }
		return(-1);
	}

	ResetCurrentNode();
	//If there are no filter terms, match everything
	if (!IsValidItem()) {
		return(1);
	}

	while (IsValidItem()) {
		if (wildmatchi(currentnode->match,searchterm)) {
			if (currentnode->excludematchflag) {
				return(0);
			} else {
				return(1);
			}
		}
		NextItemInList();
	}

	ResetCurrentNode();
	return(1);
}

int ReadObjectives()
{
	Reg_Objective reg_objective;
	DWORD dw_objective_error;
	int i_objective_count=0;
	// HERE: Turn off all auditing, unless there are NO objectives to read.

	while((dw_objective_error = Read_Objective_Registry(i_objective_count,&reg_objective))==0) {	
		AddToList(reg_objective.str_match, reg_objective.dw_match_type);
		i_objective_count++;
	}

	return(i_objective_count);
}

// Linked List Functions

void CreateLinkedList(void)
{
    head = currentnode = NULL;
}

static Node * AddToList(char *match, int excludematchflag)
{
    static Node *newNode=NULL;

	if(!match) {
		return((Node *)NULL);
	}

    newNode = (Node *) malloc(sizeof(Node));

    if (newNode == NULL) {
        if(SNAREDEBUG) DebugMsg("AddToList(): error in dynamic memory allocation\nCould not add a new objective into our linked list. You may be low on memory.\n");
        return((Node *)NULL);
    }

	newNode->excludematchflag=excludematchflag;
	strncpy_s(newNode->match,_countof(newNode->match),match,_TRUNCATE);
	
    if (head == NULL) {
        head = newNode;
        newNode->next = NULL;
    } else {
        newNode->next = head;
        head = newNode;
    }

    return newNode;
}

void RemoveFromListHead(void)
{
    Node *tempPtr;

    if (NULL == head)
        return;

    tempPtr = head;
    head = head->next;

    free(tempPtr);
}

void RemoveFromList(Node* node)
{
    Node *tempPtr, *previousPtr;

    if (NULL == node) {
        return;
	}

    if (head == node) {
        RemoveFromListHead();
        return;
    }

    tempPtr = head;

    while (NULL != tempPtr) {
        previousPtr = tempPtr;
        tempPtr = tempPtr ->next;

        if (tempPtr == node) {
            previousPtr->next = tempPtr->next;
            free(tempPtr);
            break;
        }
    }
}

void ResetCurrentNode(void)
{
    currentnode = head;
}

int IsValidItem(void)
{
    return ((NULL == currentnode) ? 0 : 1);
}

Node * GetCurrentItem()
{
    return (currentnode);
}

void NextItemInList(void)
{
	if (NULL == currentnode) {
        return;
	}
    currentnode = currentnode->next;
}

void DestroyList(void)
{
	while (NULL != head) {
		Node *tempPtr = head;
		head = head->next;

		free(tempPtr);
	}

	LCCurrent=LCHead;
	while(LCCurrent) {
		LCHead=LCCurrent->next;
		free(LCCurrent);
		LCCurrent=LCHead;
	}
	LCTail=NULL;
	LCHead=NULL;
	MCCurrent=MCHead;
	while(MCCurrent) {
		MCHead=MCCurrent->next;
		free(MCCurrent);
		MCCurrent=MCHead;
	}
	MCTail=NULL;
	MCHead=NULL;
	MCTail=NULL;
	MCCount=0;
	lcurrent=lhead;
	while(lcurrent) {
		lhead=lcurrent->next;
		free(lcurrent);
		lcurrent=lhead;
	}
	ltail=NULL;
	lhead=NULL;
}


void CEpilogService::OnShutdown() {
	// Audit: Set that global variable to TRUE so that the threads
	// receive the terminate message

	g_Info.bTerminate=TRUE;
	
	if(SNAREDEBUG) { DebugMsg("Epilog Shutdown request received"); }

	// Call a fake event so that the subroutine	gets the shutdown message
	// through the setting of g_Info.bTerminate
	if(m_hEventList[1])
		::SetEvent(m_hEventList[0]);
}

void CEpilogService::OnStop() {
	g_Info.bTerminate=TRUE;
	
	if(SNAREDEBUG) { DebugMsg("Epilog Stop request received"); }
	
	// Call a fake event so that the subroutine	gets the shutdown message
	// through the setting of g_Info.bTerminate
	if(m_hEventList[1])
		::SetEvent(m_hEventList[0]);
}

void CEpilogService::OnSignal() {
}

// Process user control requests
BOOL CEpilogService::OnUserControl(DWORD dwOpcode)
{
    switch (dwOpcode) {
    case SERVICE_CONTROL_USER + 0:

        // Save the current status in the registry
        SaveStatus();
        return TRUE;

    default:
        break;
    }
    return FALSE; // say not handled
}

// Save the current status in the registry
void CEpilogService::SaveStatus()
{
	// TODO: add code to save the sate of the service in the registry
}


// Configuration reading routines

void GetHostname(char * Hostname,int size)
{
	// Grab the fully qualified hostname
	// Note: if the user has explicitly set a hostname in the registry, use that instead.

	if(!Hostname) return;
	if(!size) return;

	Hostname[0]='\0';
	MyGetProfileString("Config","Clientname",Hostname,size);
	if(Hostname[0]=='\0') {
		GetFQDN(Hostname,size);
	}
}

void GetDestPort(DWORD * dwDestPort)
{
	if(!dwDestPort) return;
	// Where should we be sending this stuff (syslog ID)
	*dwDestPort=MyGetProfileDWORD("Network","DestPort",6161);
}

void GetSyslog(DWORD * dwSyslog)
{
	if(!dwSyslog) return;
	*dwSyslog=MyGetProfileDWORD("Network","SyslogDest",13);
}

void GetSyslogHeader(DWORD * dwSyslogHeader)
{
	if(!dwSyslogHeader) return;
	*dwSyslogHeader=MyGetProfileDWORD("Network","Syslog",0);
}

void GetWEBSERVER_ACTIVE(DWORD * WEBSERVER_ACTIVE)
{
	if(!WEBSERVER_ACTIVE) return;
	*WEBSERVER_ACTIVE=MyGetProfileDWORD("Remote","Allow",0);
}

void GetPortNumber(DWORD * dwPortNumber)
{
	DWORD dwUsePort;

	if(!dwPortNumber) return;
	*dwPortNumber=6162;
	dwUsePort=MyGetProfileDWORD("Remote","WebPortChange",0);
	if(dwUsePort) {
		*dwPortNumber=MyGetProfileDWORD("Remote","WebPort",6162);
	}
}

void GetDestination(char * lpszDestination,int size)
{
	if(!lpszDestination) return;
	if(!size) return;
	
	strncpy_s(lpszDestination,size,"127.0.0.1",_TRUNCATE);
	if(!MyGetProfileString("Network","Destination",lpszDestination,size))
	{
		// Problem. Couldn't retrieve the destination from the registry.
		// Default it to something harmless.
		MyWriteProfileString("Network","Destination",lpszDestination);
	}
}

int ReadLogs()
{
	Reg_Log reg_log;
	DWORD dw_log_error;
	int i_log_count=0;

	if (SNAREDEBUG) DebugMsg("Reading Log entries");
	while((dw_log_error = Read_Log_Registry(i_log_count,&reg_log))==0) {	
		AddLogWatch(&reg_log);
		i_log_count++;
	}
	if (SNAREDEBUG) DebugMsg("Found %d logs [return code: %d]", i_log_count, dw_log_error);
	return(i_log_count);
}

void GetDelim(char * DELIM,int size)
{
	if(!DELIM) return;
	if(size < 2) return;

	DELIM[0]=9;	// TAB character
	DELIM[1]='\0';
	// Only use a different character for delimiters if we're sending data via syslog.
	MyGetProfileString("Config","Delimiter",DELIM,size);
}

void GetPassword(char * lpszPassword,int size)
{
	DWORD dwUsePassword;

	if(!lpszPassword) return;
	if(!size) return;

	strncpy_s(lpszPassword,size,"",_TRUNCATE);
	dwUsePassword=MyGetProfileDWORD("Remote","AccessKey",0);
	if(dwUsePassword) {
		if(!MyGetProfileString("Remote","AccessKeySet",lpszPassword,size))
		{
			// Problem. Couldn't retrieve the destination from the registry.
			// Default it to something harmless.
			strncpy_s(lpszPassword,size,"",_TRUNCATE);
		}
	}
}

void GetIPAddress(char * lpszIPAddress,int size)
{
	DWORD dwRestrictIP;
	struct hostent *phostent;
	struct in_addr in;

	if(!lpszIPAddress) return;
	if(!size) return;

	strcpy_s(lpszIPAddress,size,"");
	dwRestrictIP=MyGetProfileDWORD("Remote","Restrict",0);

	if(dwRestrictIP) {
		if(!MyGetProfileString("Remote","RestrictIP",lpszIPAddress,size))
		{
			// Problem. Couldn't retrieve the destination from the registry.
			// Default it to something harmless.
			strncpy_s(lpszIPAddress,size,"127.0.0.1",_TRUNCATE);
		} else {
			// HERE Check to see if it is an IP address.. if not, then do this:
			phostent=gethostbyname(lpszIPAddress);
			if(phostent) {
				// strncpy_s(lpszIPAddress,size,phostent->h_addr_list[0],_TRUNCATE);
				memcpy(&in.s_addr, *(phostent->h_addr_list), sizeof (in.s_addr));
				strncpy_s(lpszIPAddress,size,inet_ntoa(in),_TRUNCATE);
			}
		}
	}
	//DMM possible memory problem
}

void GetSocketType(DWORD * SocketType)
{
	if(!SocketType) return;
	*SocketType=MyGetProfileDWORD("Network","SocketType",0);
}


// END Configuration Reading Routines



// Duplicate of the CNTService DebugMsg
void DebugMsg(const char* pszFormat, ...)
{
	char buf[8192];
	char date[50];
	char time[50];

	if(!pszFormat) return;
	
	SYSTEMTIME st;

	GetLocalTime(&st);
	GetDateFormat(LOCALE_SYSTEM_DEFAULT,0,&st,"dd'/'MM'/'yyyy",date,_countof(date));
	GetTimeFormat(LOCALE_SYSTEM_DEFAULT,0,&st,"HH':'mm':'ss",time,_countof(time));

	_snprintf_s(buf, 8192,_TRUNCATE, "[Epilog %s](%lu - %s %s): ",EPILOG_VERSION,GetCurrentThreadId(),date,time);
	va_list arglist;
	va_start(arglist, pszFormat);
    _vsnprintf_s(&buf[strlen(buf)],8192-strlen(buf),_TRUNCATE,pszFormat,arglist);
	va_end(arglist);
    _snprintf_s(buf,8192,_TRUNCATE,"%s\n",buf);

	if(buf) { printf("%s",buf); fflush(stdout); }

#ifdef DEBUG_TO_FILE
	FILE *fp;

	if(!strlen(pszFormat)) {
		// Send a "" to truncate the file back to zero
		err=fopen_s(&fp, "C:\\SNAREDebug.log","wt");
	} else {
		err=fopen_s(&fp,"C:\\SNAREDebug.log","a");
	}
	if(!err) {
		fprintf(fp,"%s",buf);
		fflush(fp);
		fclose(fp);
	}
#endif
}

// Dump the current eventlog record to a file.

int AddLogWatch(Reg_Log *rl)
{
	struct stat stats;
	int length;
	errno_t err;
	char logbuffer[MAX_EVENT], c;
	lcurrent = (LogNode *)malloc(sizeof(LogNode));
	if (lcurrent) {
		lcurrent->next=NULL;
		strncpy_s(lcurrent->type, MAX_TYPE, rl->type,_TRUNCATE);
		strncpy_s(lcurrent->format, SIZE_OF_LOGNAME, rl->format,_TRUNCATE);
		// Firstly check if the entry is a directory
		if (validate_file_or_directory(rl->name) == 2) {
			if(SNAREDEBUG) DebugMsg("FOUND A DIRECTORY: %s", rl->name);
			strncpy_s(lcurrent->dir_name, SIZE_OF_LOGNAME, rl->name,_TRUNCATE);
			if (rl->name[strlen(rl->name)-1] != '\\') strncat_s(lcurrent->dir_name, SIZE_OF_LOGNAME, "\\",_TRUNCATE);
			lcurrent->dir=1;
			lcurrent->dir_check = 0;
			lcurrent->name[0]='\0';
			// this function will create all the necessary log watchers
			FindDirFile(lcurrent);
		} else {
			strncpy_s(lcurrent->name, SIZE_OF_LOGNAME, rl->name,_TRUNCATE);
			lcurrent->dir = 0;
			lcurrent->dir_check = 0;
			lcurrent->dir_name[0]='\0';
		}
		if(SNAREDEBUG) DebugMsg("Watching '%s' at location: %s", lcurrent->type, lcurrent->name);
		// Grab and record the current stats
		if (stat(lcurrent->name, &stats) < 0) {
			if(SNAREDEBUG) DebugMsg("Failed to find stats for %s\n", lcurrent->name);
			lcurrent->last_error=time(&lcurrent->last_error);
		} else {
			lcurrent->mtime = stats.st_mtime;
			lcurrent->old_size = stats.st_size;
			//Before we set the size, we need to find the true end of the file
			err = fopen_s(&lcurrent->fs,lcurrent->name, "rb");
			if (!err) {
				fseek(lcurrent->fs, -1,SEEK_END);
				c=fgetc(lcurrent->fs);
				if (c == '\0' || c == ' ') {
					//Possibly SMTP or old IIS format, start searching for the true end of the file
					fseek(lcurrent->fs, 0,SEEK_SET);
					lcurrent->size = 0;
					while ((length = GetLine(lcurrent->fs, logbuffer, MAX_EVENT)) > 0) {
						if (!strncmp(logbuffer,"          ",10)) break;
						lcurrent->size=lcurrent->size + length;
					}
				} else {
					//looks like we are ok to watch the file from the end.
					lcurrent->size = stats.st_size;
				}
				fclose(lcurrent->fs);
			} else {
				// This shouldn't happen at this point, but if it does, just use the file size
				lcurrent->size = stats.st_size;
			}
			lcurrent->dev = stats.st_dev;
			lcurrent->ino = stats.st_ino;
			lcurrent->mode = stats.st_mode;
			lcurrent->last_error=0;
		}
		lcurrent->next=NULL;
		lcurrent->pmsg[0]='\0';
		// Now that the file is ready, seek to the end of the file
		// and prepare to capture further output

		if (ltail) {
			ltail->next=lcurrent;
			ltail=lcurrent;
		}
		if (!lhead) {
			lhead=lcurrent;
			ltail=lcurrent;
		}
	} else {
		if(SNAREDEBUG) DebugMsg("Failed to allocate memory for log file\n");
		return(0);
	}
	return(0);
}

int file_has_changed (LogNode *f) {
	// Grab the new set of stats
	struct _stat stats;
	if (f->dir) f->dir_check++;
	if (f->dir && f->last_error) {
		if (f->dir_check < 20) return(0);
		if (FindDirFile(f)) {
			f->last_error = 0;
		}
		f->dir_check = 0;
	}
	if (_stat(f->name, &stats) < 0) {
		if (!f->last_error) DebugMsg("Failed to find log file: %s\n", f->name);
		f->last_error=time(&f->last_error);
		return(-1);
	}
	// Firstly, check if the file has been rotated
	if (f->dev != stats.st_dev || f->ino != stats.st_ino || f->last_error) {
		// change detected, reset the stats and start from the begining
		if(SNAREDEBUG) DebugMsg("rotation detected\n");
		f->size = 0;
		f->mtime = stats.st_mtime;
		f->dev = stats.st_dev;
		f->ino = stats.st_ino;
		f->mode = stats.st_mode;
		f->last_error = 0;
		// seek to the begining of the file to grab anything we might have missed
		// fseek(fs, 0, SEEK_SET);
		return(1);
	}

	if (f->size == stats.st_size &&
	    f->mtime == stats.st_mtime) {
		// Everything else is exactly the same
		// so do nothing for now
		if(SNAREDEBUG) DebugMsg("Size and mtime are the same");
		// return straight away if not a directory, or if under 20
		// seconds has passed, this should minimise any resource
		// impacts
		if (!f->dir) return(0);
		if (f->dir_check < 20) return(0);
		f->dir_check = 0;
		//just make sure that there is no new file to watch yet
		if (FindDirFile(f) == DIR_NEW_FILE) return(1);
		else return(0);
	} else {
		int change=0;
		// Something has changed, find out what and take action
		if (stats.st_size < f->size) {
			if(SNAREDEBUG) DebugMsg("%s: File truncated\n", f->name);
			change = 1;
		}
		if (f->mtime > stats.st_mtime) {
			if(SNAREDEBUG) DebugMsg("%s: Modified in the past\n");
			change = 1;
		}

		// Reset the descriptors
		//f->size = stats.st_size;
		f->mtime = stats.st_mtime;
		f->dev = stats.st_dev;
		f->ino = stats.st_ino;
		f->mode = stats.st_mode;
		if (change) {
			// Something went wrong, so ditch the old descriptor and
			// keep the new one.  However, since it was not a rotation,
			// i.e. we still have the same dev and inode numbers, then
			// seek to the beginning of the file and only return a change
			// if the file is larger than zero.
			//fseek(fs, f->size, SEEK_SET);
			f->size = 0;
			if (stats.st_size == (off_t) 0)
				return(0);
			else
				return(1);
		}
		return(1);
	}
}

char * getdate(char *date, BOOL ytday)
{
   time_t t;
   struct tm *tm;
   errno_t err;

   char year[3],month[3],day[3];
   int tempyear,tempmonth,tempday;

   t = time(NULL);
   if (ytday) t = t - (24 * 60 * 60);
   tm = (struct tm *)malloc(sizeof(struct tm));
   err=gmtime_s(tm,&t);

   tempyear = tm->tm_year;
   if(tempyear > 100) {
	tempyear -= 100;
   }

   if(tempyear >=10) {
	_snprintf_s(year,3,_TRUNCATE,"%d",tempyear);
   } else {
	_snprintf_s(year,3,_TRUNCATE,"0%d",tempyear);
   }

   tempmonth=tm->tm_mon+1;
   if(tempmonth >=10) {
	_snprintf_s(month,3,_TRUNCATE,"%d",tempmonth);
   } else {
	_snprintf_s(month,3,_TRUNCATE,"0%d",tempmonth);
   }

   tempday=tm->tm_mday;
   if(tempday >=10) {
	_snprintf_s(day,3,_TRUNCATE,"%d",tempday);
   } else {
	_snprintf_s(day,3,_TRUNCATE,"0%d",tempday);
   }

   _snprintf_s(date,7,_TRUNCATE,"%s%s%s",year,month,day);

   return(date);
}

int FindDirFile(LogNode *f)
{
	HANDLE handle=INVALID_HANDLE_VALUE;
	WIN32_FIND_DATA firstData, findData;
	char currentdate[7]; // YYMMDD
	char filespec[132];
	char fmt[SIZE_OF_LOGNAME];
	char *pos, *pos2;
	getdate(currentdate);
	if (strlen(f->format)) {
		strncpy_s(fmt,SIZE_OF_LOGNAME,f->format,_TRUNCATE);
		pos = strstr(fmt,"%");
		if (pos) {
			pos2 = pos + 1;
			*pos = '\0';
			_snprintf_s(filespec,_countof(filespec),_TRUNCATE,"%s%s%s%s",f->dir_name,fmt,currentdate,pos2);
		} else {
			_snprintf_s(filespec,_countof(filespec),_TRUNCATE,"%s%s",f->dir_name,fmt);
		}
	} else {
		_snprintf_s(filespec,_countof(filespec),_TRUNCATE,"%s*%s*.*",f->dir_name,currentdate);
	}
	if(SNAREDEBUG) DebugMsg("+++++++++++++++ checking today: %s", filespec);
	handle = FindFirstFile(filespec, &firstData);
	if (handle != INVALID_HANDLE_VALUE) {
		//Make sure we have the last file instead
		while (FindNextFile(handle,&firstData)) continue;
		if(SNAREDEBUG) DebugMsg("+++++++++++++++ Found file: %s", firstData.cFileName);
		//if(SNAREDEBUG) DebugMsg("Checking for other files");
		//while (FindNextFile(handle, &findData)) {
		//	if(SNAREDEBUG) DebugMsg("+++++++++++++++ Found file: %s", findData.cFileName);
		//	_snprintf_s(f->name,SIZE_OF_LOGNAME,_TRUNCATE,"%s%s",f->dir_name,findData.cFileName);
		//	//AddLogWatch(name,type);
		//}
	} else {
		//check to see if we are still logging to yesterday
		getdate(currentdate, true);
		if (strlen(f->format)) {
			pos = strstr(fmt,"%");
			if (pos) {
				_snprintf_s(filespec,_countof(filespec),_TRUNCATE,"%s%s%s%s",f->dir_name,fmt,currentdate,pos2);
			} else {
				_snprintf_s(filespec,_countof(filespec),_TRUNCATE,"%s%s",f->dir_name,fmt);
			}
		} else {
			_snprintf_s(filespec,_countof(filespec),_TRUNCATE,"%s*%s*.*",f->dir_name,currentdate);
		}
		if(SNAREDEBUG) DebugMsg("Could not find a file for today, checking yesterday: %s", filespec);
		handle = FindFirstFile(filespec, &firstData);
		if (handle == INVALID_HANDLE_VALUE) {
			if (strlen(f->name)) {
				if(SNAREDEBUG) DebugMsg("#Could not find a file for yesterday either, continuing to watch %s.", f->name);
				return(0);
			} else {
				if(SNAREDEBUG) DebugMsg("Could not find a file for yesterday either, watching ex%s.log.", currentdate);
				_snprintf_s(f->name,SIZE_OF_LOGNAME,_TRUNCATE,"%sex%s.log",f->dir_name,currentdate);
				FindClose(handle);
				return(0);
			}
		}
		if(SNAREDEBUG) DebugMsg("+++++++++++++++ Found file: %s", firstData.cFileName);
		//while (FindNextFile(handle, &findData)) {
		//	if(SNAREDEBUG) DebugMsg("+++++++++++++++ Found file: %s", findData.cFileName);
		//	_snprintf_s(f->name,SIZE_OF_LOGNAME,_TRUNCATE,"%s%s",f->dir_name,findData.cFileName);
		//	//AddLogWatch(f->name,f->type);
		//}
	}
	_snprintf_s(f->name,SIZE_OF_LOGNAME,_TRUNCATE,"%s%s",f->dir_name,firstData.cFileName);
	FindClose(handle);
	if (strstr(firstData.cFileName,f->name)) return(1);
	else return(DIR_NEW_FILE);
}
