/**********************************************************
 *
 * Snare for AIX version 1.1
 *
 * Author: InterSect Alliance Pty Ltd
 *
 * Copyright 2001-2005 InterSect Alliance Pty Ltd
 *
 * Last Modified: 27/10/2005
 *
 * Available under the terms of the GNU Public Licence.
 * - See www.gnu.org
 *
 **********************************************************
 *
 * Snare for AIX is a user-space program
 * that interacts with the audit facility within the
 * AIX operating system as a streamcmd filter.
 *
 * Snare takes data from the auditpr command, and sends
 * the resulting text data over the network to a remote server
 * using UDP.
 *
 **********************************************************
 *
 * History:
 *       7/10/2004  Test version - based on Snare for AIX
 *       27/10/2005 Version 1.1 released 
 *
 **********************************************************
 *
 * Compilation Instructions:
 *    See makefile.
 *    cc -o snarecore snarecore.c webserver.c WebPages.c -lsocket -lnsl -lintl
 *
 **********************************************************/

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/audit.h>

#include <netinet/in.h>


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <signal.h>
#include <limits.h>
#include <dirent.h>
#include <strings.h>
#include <values.h>
#include <time.h>

#include <regex.h>

// Manually define snprintf etc..
#define SILLY_AIX 1

#include "shared.h"
#include "snarecore.h"
#include "webserver.h"


////// DEBUG
#include <stdarg.h>
void DebugMsg(const char *pszFormat, ...);
////// END DEBUG

// Linked List Functions
static Node *head, *currentnode;
static int AuditDestination = AUDIT_TO_STDOUT;	// Send to STDOUT unless otherwise stated.
static int SyslogDestination=13;			// user.notice

char hostid[MAX_HOSTID] = "";
int remote_allow = 0;
int remote_webport = 6161;
char remote_restrictip[16] = "";
char remote_password[256] = "";

FILE *AuditFile = (FILE *) NULL;
int AuditSocket = 0;
struct sockaddr_in AuditSocketName;

/* Define an error variable so we can communicate with auditsvc */
extern int errno;

int main(int argc, char *argv[])
{
	/* General program related variables */
	char logbuffer[LOGBUFFERSIZE];	/* Holds the data from auditpr */

	/* Variables that relate to the network socket */
	// struct hostent *hp;	/* Host Pointer */
	// struct sockaddr_in socketname;
	// struct in_addr ia;

	/* Audit related variables */
	int webpid = 0;

	// NODE that matches the objective search term.
	Node *nodematch;

	char username[MAX_USERNAME];
	char searchterm[MAX_PATH];
	char eventname[MAX_EVENTNAME];
	char options[MAX_OPTIONS];
	char *startrecord;
	int returncode;

	/* The only person who can run auditsvc is root, so lets
	   make sure that we discourage other users */
	if (getuid() != 0) {
		fprintf(stderr, "This program can only be run by root.\n");
		exit(1);
	}

	// Read our configuration file, and set up audit.
	returncode=read_config_file(0);

	if (!hostid) {
		getfqdn(hostid);
	} else {
		if (strlen(hostid) == 0) {
			getfqdn(hostid);
		}
	}

	if (remote_allow) {
		webpid = fork();
		if (webpid == 0) {
			// Child process - Run our web server
			while (InitWebServer
			       (remote_webport, remote_restrictip,
				remote_password) == -1) {
				// Problem binding. Sleep for a moment.
				sleep(5);
			}
			while ((returncode = StartThread()) == 1) {
				// Problem binding. Keep trying.
				sleep(5);
			}
			exit(0);
		}
	}

	
	while(fgets(logbuffer,sizeof(logbuffer),stdin)) {
		startrecord=logbuffer;
		username[0]='\0';
		searchterm[0]='\0';
		eventname[0]='\0';
		options[0]='\0';
		returncode=0;

		/* This is where we implement the SNARE event checker process */
		returncode =
		    GetAIXDetails(logbuffer,
				  username,
				  searchterm,
				  eventname,
				  options);

		while ((nodematch =
			CheckObjective(username,
				       searchterm,
				       eventname,
				       options,
				       returncode))
		       != (Node *) NULL) {

			if (sendevent
			    (startrecord,
			     nodematch->
			     criticality) != 0) {
				// Report an error, but continue
				fprintf(stderr,
					"error sending log entry to destination\n");
			}
		}
	}

	if (webpid) {
		kill(webpid, SIGTERM);
		waitpid(webpid,NULL,0);
	}

	if (AuditSocket) {
		close(AuditSocket);
	}

	DestroyList();

	/* All done */
	return 0;
}


// Shortread: don't re-read all the objectives - just the general configuration items.
int read_config_file(int shortread)
{
	FILE *configfile = (FILE *) NULL;
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];	// Should be enough for most config lines.
	// Would love to use gchar here instead, but need to keep this simple.
	// Config file header.
	int headertype = 0;

	// Clear some default variables.
	remote_allow = 0;
	remote_webport = 6161;
	strncpy(remote_restrictip, "", sizeof(remote_restrictip));
	strncpy(remote_password, "", sizeof(remote_password));

	// Clear the audit destination
	AuditDestination = 0;

	configfile = fopen(CONFIG_FILENAME, "r");

	if (configfile == (FILE *) NULL) {
		DebugMsg("Cannot open audit configuration file");
		return (0);
	}

	while (fgets(inputbuffer, MAX_AUDIT_CONFIG_LINE, configfile)) {

		// Kill whitespace from start and end of line.
		trim(inputbuffer);

		if(strlen(inputbuffer) < 3) {
			continue;
		}

		if (!iscomment(inputbuffer)) {
			// Is this line a header?
			if (isheader(inputbuffer)) {
				headertype = getheader(inputbuffer);
			} else {
				if (headertype == CONFIG_OBJECTIVES) {
					// Save off enough space to store the data we need.
					char event[MAX_AUDIT_CONFIG_LINE],
					    user[MAX_AUDIT_CONFIG_LINE],
					    path[MAX_AUDIT_CONFIG_LINE];
					char event2[MAX_AUDIT_CONFIG_LINE],
					    options[MAX_OPTIONS] = "";
					char *eventpointer = NULL;
					char *eventpointer2 = NULL;
					int criticality;
					int returncode;
					int excludeflag = 0;
					int excludematchflag=0;

					// Do we need to skip over the objectives?
					if (shortread == 1) {
						continue;
					}


					if ((criticality =
					     splitobjective(inputbuffer,
							    event, user,
							    path,
							    &excludeflag,
							    &excludematchflag,
							    &returncode))
					    > -1) {
						// add the objective to the linked list.
						trimallwhitespace(event);

						// HERE: Split event into comma separated list of events                                
						eventpointer = event;
						eventpointer2 = event;

						// While there are no more commas
						while (eventpointer2) {
							eventpointer2 =
							    strstr
							    (eventpointer,
							     ",");
							if (eventpointer2
							    ==
							    (char *) NULL)
							{
								// No commas left. Just copy to the end of the line.
								strncpy
								    (event2,
								     eventpointer,
								     MAX_AUDIT_CONFIG_LINE);
							} else {
								strncpy
								    (event2,
								     eventpointer,
								     (eventpointer2
								      -
								      eventpointer));
								// Make sure we have a null on the end of the line.
								event2
								    [eventpointer2
								     -
								     eventpointer]
								    = '\0';
							}
							if (eventpointer2) {
								// Skip the comma
								eventpointer
								    =
								    eventpointer2
								    + 1;
							}
							// Clear out any options strings
							strcpy(options,
							       "");

							if (regmatch
							    (event2,
							     "\\(.*\\)$"))
							{
								char *position;

								// Bonus, we have something. Grab it, add to options.
								// First, trim off the last bracket.
								event2
								    [strlen
								     (event2)
								     - 1] =
								    '\0';
								position =
								    strstr
								    (event2,
								     "(");

								strncpy
								    (options,
								     position
								     + 1,
								     MAX_OPTIONS);
								// truncate the event string at the bracket.
								*position =
								    '\0';
							}

							if (!strcmp
							    (event,
							     "*"))
							{
								AddToList
								    ("AUDIT_ALL",
								     user,
								     path,
								     criticality,
								     "",
								     excludeflag,
								     excludematchflag,
								     returncode);
							} else {
								AddToList
								    (event2,
								     user,
								     path,
								     criticality,
								     options,
								     excludeflag,
								     excludematchflag,
								     returncode);
							}
						}
					} else {
						fprintf(stderr,
							"WARNING: Cannot process objective - please ensure configuration line\n\tcontains valid criticality, event, criticality, return, user and match elements\n\t[%s]\n",
							inputbuffer);
					}
				} else if (headertype == CONFIG_OUTPUT) {
					if (isfilename(inputbuffer)) {
						if (!open_audit_output
						    (inputbuffer)) {
							fprintf(stderr,
								"WARNING: Could not open the file specified in the audit configuration: %s - Auditing to STDOUT\n",
								inputbuffer);
						}
					} else if (isnetwork(inputbuffer)) {
						if (!open_audit_network
						    (inputbuffer)) {
							fprintf(stderr,
								"WARNING: Could not open the network specified in the audit configuration: %s - Auditing to STDOUT\n",
								inputbuffer);
						}
					} else if(issyslog(inputbuffer)) {
						SyslogDestination=get_syslog_dest(inputbuffer);
						if(!SyslogDestination) {
							SyslogDestination=13;
							fprintf(stderr,"WARNING: Could not establish the correct syslog destination from the audit configuration: %s - Sending to user.notice\n",inputbuffer);
						}
					}

				} else if (headertype == CONFIG_HOSTID) {
					gethostident(inputbuffer, hostid,
						     MAX_HOSTID);
				} else if (headertype == CONFIG_REMOTE) {
					// Grab the remote control stuff here
					if (regmatchi
					    (inputbuffer, "^allow=")) {
						if (regmatchi
						    (inputbuffer, "=1")) {
							remote_allow = 1;
						}
					} else
					    if (regmatchi
						(inputbuffer,
						 "^listen_port=")) {
						remote_webport =
						    getport(inputbuffer);
					} else
					    if (regmatchi
						(inputbuffer,
						 "^restrict_ip=")) {
						if (!getconfstring
						    (inputbuffer,
						     remote_restrictip,
						     sizeof
						     (remote_restrictip)))
						{
							strncpy
							    (remote_restrictip,
							     "",
							     sizeof
							     (remote_restrictip));
						}
					} else
					    if (regmatchi
						(inputbuffer,
						 "^accesskey=")) {
						if (!getconfstring
						    (inputbuffer,
						     remote_password,
						     sizeof
						     (remote_password))) {
							strncpy
							    (remote_password,
							     "",
							     sizeof
							     (remote_password));
						}
					}
				} else {
					fprintf(stderr,
						"WARNING: Configuration file line does not fit in any recognised header.\n\t%s\n",
						inputbuffer);
				}
			}
		}
	}

	fclose((FILE *) configfile);
	return (1);
}

int iscomment(char *line)
{
	// Verify that there is something to check.
	if (line == (char *) NULL)
		return (1);

	// And that there is some data within.
	if (strlen(line) == 0 || strlen(line) > MAX_AUDIT_CONFIG_LINE)
		return (1);

	// If the first non-whitespace character is a hash, this is a comment line.
	while (*line) {
		// Space or tab or newline / formfeed
		if (*line == ' ' || *line == 9 || *line == 10
		    || *line == 12) {
			line++;
		} else if (*line == '#') {
			return (1);
		} else {
			// Ahh. A non-whitespace, non hash character.
			return (0);
		}
	}

	// If we are here, then the whole line must have been whitespace
	return (1);
}

int isheader(char *string)
{
	if (string[0] == '[' && string[strlen(string) - 1] == ']') {
		return (1);
	}
	return (0);
}

int isfilename(char *string)
{
	if (regmatch(string, "^file=")) {
		return (1);
	} else {
		return (0);
	}
}

// Network output
int isnetwork(char *string)
{
	if (regmatch(string, "^network=")) {
		return (1);
	} else {
		return (0);
	}
}

int issyslog(char *string)
{
	if(regmatch(string,"^syslog=")) {
		return(1);
	}
	return(0);
}

int open_audit_output(char *string)
{
	string += strlen("file=");
	if (strlen(string)) {
		if (regmatchi(string, "stdout")) {
			AuditDestination |= AUDIT_TO_STDOUT;
			return (1);
		}
		AuditFile = fopen(string, "a");
		if (AuditFile != (FILE *) NULL) {
			AuditDestination |= AUDIT_TO_FILE;
			return (1);
		} else {
			AuditDestination |= AUDIT_TO_STDOUT;
			return (0);
		}
	}

	return (0);
}

// For the moment, require the web server to do the work of converting
// destinations to numerics.
int get_syslog_dest(char *string)
{
	int destination;
	string+=strlen("syslog=");
	destination=atoi(string);
	if(destination > 0) {
		return(destination);
	}
	return(0);
}

// Open the network - destination host and port
int open_audit_network(char *string)
{
	char desthost[MAX_HOSTID];
	int port = 6161;	// Default port
	char *pos;
	struct hostent *hp;

	// Initialise desthost.
	strncpy(desthost, "localhost", MAX_HOSTID);

	string += strlen("network=");
	if (strlen(string)) {
		if (regmatchi(string, "stdout")) {
			AuditDestination |= AUDIT_TO_STDOUT;
			return (1);
		}

		AuditSocket = socket(AF_INET, SOCK_DGRAM, 0);
		if (AuditSocket < 0) {
			fprintf(stderr,
				"Cannot open network socket - auditing to STDOUT\n");
			AuditDestination |= AUDIT_TO_STDOUT;
			return (0);
		}

		pos = strstr(string, ":");
		if (pos) {
			if ((pos - string) < MAX_HOSTID) {
				strncpy(desthost, string, (pos - string));
				desthost[pos - string] = '\0';
			} else {
				strncpy(desthost, string, MAX_HOSTID);
				desthost[MAX_HOSTID - 1] = '\0';
			}
			pos++;
			// As long as the user has a number there.
			if (strlen(pos)) {
				port = atoi(pos);
			}
		} else {
			// No colon. Assume the default port to be 6161
			strncpy(desthost, string, MAX_HOSTID);
		}

		// Just in case the user has entered a blank field
		if (strlen(desthost) == 0) {
			strncpy(desthost, "localhost", MAX_HOSTID);
		}

		hp = gethostbyname(desthost);
		if (hp == 0) {
			fprintf(stderr,
				"Cannot resolve host %s - auditing to STDOUT\n",
				hostid);
			close(AuditSocket);
			AuditDestination |= AUDIT_TO_STDOUT;
			return (0);
		}
		bcopy(hp->h_addr, &AuditSocketName.sin_addr, hp->h_length);
		AuditSocketName.sin_family = AF_INET;
		AuditSocketName.sin_port = htons(port);

		AuditDestination |= AUDIT_TO_NETWORK;
		if(port == 514) {
			AuditDestination |= AUDIT_TO_SYSLOG;
		}
		return (1);
	}

	AuditDestination |= AUDIT_TO_STDOUT;
	return (0);
}

int getheader(char *string)
{
	char temp[256];
	char *stringp;
	strncpy(temp, string, 256);

	stringp = temp;

	// Remove the first and last bracket.
	stringp++;
	stringp[strlen(stringp) - 1] = '\0';

	if (regmatchi(stringp, "^objectives$")) {
		return (CONFIG_OBJECTIVES);
	} else if (regmatchi(stringp, "^events$")) {
		return (CONFIG_EVENTS);
	} else if (regmatchi(stringp, "^file$")
		   || regmatchi(stringp, "^output$")) {
		return (CONFIG_OUTPUT);
	} else if (regmatchi(stringp, "^hostid$")) {
		return (CONFIG_HOSTID);
	} else if (regmatchi(stringp, "^remote$")) {
		return (CONFIG_REMOTE);
	}

	fprintf(stderr, "Unknown header in configuration file: %s\n",
		stringp);
	return (0);
}

// Return the host identifier
char *getconfstring(char *string, char *file, int length)
{
	char *stringp = string;

	stringp = strstr(string, "=");
	if (stringp != (char *) NULL) {
		stringp++;
		if (strlen(stringp)) {
			strncpy(file, stringp, length - 1);
		} else {
			return ((char *) NULL);
		}
	} else {
		return ((char *) NULL);
	}
	return ((char *) file);
}

int getport(char *string)
{
	char *stringp = string;
	char strPort[10];

	stringp = strstr(string, "=");

	if (stringp != (char *) NULL) {
		stringp++;
		if (strlen(stringp)) {
			strncpy(strPort, stringp, 10);
			return (atoi((char *) strPort));
		} else {
			return (0);
		}
	} else {
		return (0);
	}
}

// Return the host identifier
char *gethostident(char *string, char *host, int length)
{
	char *stringp = string;

	stringp = strstr(string, "=");
	if (stringp != (char *) NULL) {
		stringp++;
		if (strlen(stringp)) {
			strncpy(host, stringp, length - 1);
		} else {
			return ((char *) NULL);
		}
	} else {
		return ((char *) NULL);
	}
	return ((char *) host);
}

// Return a string that contains the criticality
// Note that string is assumed to be MAX_AUDITREC in size.
char *criticalitystring(int criticality, char *string)
{
	switch (criticality) {
	case CRITICALITY_CRITICAL:
		return (strncpy(string, "critical", MAX_AUDITREC));
	case CRITICALITY_PRIORITY:
		return (strncpy(string, "priority", MAX_AUDITREC));
	case CRITICALITY_WARNING:
		return (strncpy(string, "warning", MAX_AUDITREC));
	case CRITICALITY_INFO:
		return (strncpy(string, "information", MAX_AUDITREC));
	default:
		return (strncpy(string, "clear", MAX_AUDITREC));
	}
}


// Remove start / end whitespace, including the newline
void trim(char *string)
{
	char *pointer;
	char *pointer2;

	// Verify that there is something to check.
	if (string == (char *) NULL)
		return;

	// And that there is some data within.
	if (strlen(string) == 0)
		return;

	// Start from the end, work backwards.
	pointer = &string[strlen(string) - 1];

	while ((*pointer == ' ' || *pointer == 9 || *pointer == 10
		|| *pointer == 12) && pointer >= string) {
		*pointer = '\0';
		pointer--;
	}

	// Are we back at the start of the string? If so, this line must be null.
	if (pointer == string)
		return;
	// Pointer is now at the last non-whitespace character of the string.

	pointer2 = string;
	while ((*pointer2 == ' ' || *pointer2 == 9 || *pointer2 == 10
		|| *pointer2 == 12) && pointer2 < pointer) {
		pointer2++;
	}
	// pointer2 will now point to the start of the first non-null character
	// Copy the truncated string back to the original.
	strcpy(string, pointer2);
}

void trimallwhitespace(char *string)
{
	char *readpointer;
	char *writepointer;

	// Verify that there is something to check.
	if (string == (char *) NULL)
		return;

	// And that there is some data within.
	if (strlen(string) == 0)
		return;

	// Start from the end, work backwards.
	readpointer = string;
	writepointer = string;
	while (readpointer < (string + strlen(string))) {
		if (*readpointer == ' ' || *readpointer == 9
		    || *readpointer == 10 || *readpointer == 12) {
			readpointer++;
		} else {
			if (writepointer != readpointer) {
				*writepointer = *readpointer;
			}
			readpointer++;
			writepointer++;
		}

	}
}

// Strip tabs and newlines out of a string.
char *strip(char *string)
{
	char *stringp = string;

	while (*stringp != '\0') {
		if (*stringp == '\t' || *stringp == '\n') {
			*stringp = ' ';
		}
		stringp++;
	}
	return (string);
}

// Take an identified objective line
// NOTE: string, event, user and match must all be the same buffer size!
int splitobjective(char *string, char *event, char *user, char *match,
		   int *excludeflag, int *excludematchflag, int *returncode)
{
	char *startevent, *startuser, *startmatch, *startcrit,
	    *startreturn;
	char *stringpointer, *endstring;
	char *eventp, *userp, *matchp, *critp;
	char criticality[MAX_AUDIT_CONFIG_LINE];
	int crit = 0;

	*returncode = RETURNCODE_ANY;

	// Do some basic sanity checks.
	if (string == (char *) NULL || event == (char *) NULL
	    || user == (char *) NULL || match == (char *) NULL) {
		return (-1);
	}
	if (!strlen(string)) {
		return (-1);
	}

	startcrit = strstr(string, "criticality=");
	startevent = strstr(string, "event=");
	startreturn = strstr(string, "return=");
	startuser = strstr(string, "user=");
	if (startuser == (char *) NULL) {
		startuser = strstr(string, "user!=");
		if (startuser != (char *) NULL) {
			// EXCLUDE USERS rather than include.
			*excludeflag = 1;
		}
	}
	startmatch = strstr(string, "match=");
	if (startmatch == (char *) NULL)
	{
		startmatch = strstr (string, "match!=");
		if (startmatch != (char *) NULL)
		{
			// EXCLUDE rather than include.
			*excludematchflag = 1;
		}
	}

	// string pointers for iteration.
	stringpointer = string;
	eventp = event;
	userp = user;
	matchp = match;
	critp = criticality;

	// Pointer to the last character in the string.
	endstring = &string[strlen(string)];

	if (startevent == (char *) NULL || startuser == (char *) NULL ||
	    startmatch == (char *) NULL || startcrit == (char *) NULL
	    || startreturn == (char *) NULL) {
		// Problem, this line is malformed. We really cannot proceed with this line.
		fprintf(stderr,
			"The following line does not contain the criticality, event, return, user and match elements: [%s]",
			string);
		return (-1);
	}
	// Start with the event
	stringpointer = startcrit + strlen("criticality=");
	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startuser
		   && stringpointer != startmatch
		   && stringpointer != endstring)) {
		*critp = *stringpointer;
		critp++;
		stringpointer++;
	}
	*critp = '\0';
	trim(criticality);
	crit = atoi(criticality);
	if (crit < CRITICALITY_CLEAR)
		crit = CRITICALITY_CLEAR;
	if (crit > CRITICALITY_CRITICAL)
		crit = CRITICALITY_CRITICAL;

	// Start with the event
	stringpointer = startevent + strlen("event=");
	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startuser
		   && stringpointer != startmatch
		   && stringpointer != endstring)) {
		*eventp = *stringpointer;
		eventp++;
		stringpointer++;
	}
	*eventp = '\0';
	// Remove extra whitespace at start and end.
	trim(event);

	stringpointer = startreturn + strlen("return=");
	if (regmatchi(stringpointer, "^success")) {
		*returncode = RETURNCODE_SUCCESS;
	} else if (regmatchi(stringpointer, "^failure")) {
		*returncode = RETURNCODE_FAILURE;
	} else {
		// Any thing else is either success or failure.
		*returncode = RETURNCODE_ANY;
	}

	if (*excludeflag) {
		stringpointer = startuser + strlen("user!=");
	} else {
		stringpointer = startuser + strlen("user=");
	}

	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startmatch
		   && stringpointer != startevent
		   && stringpointer != endstring)) {
		*userp = *stringpointer;
		stringpointer++;
		userp++;
	}
	*userp = '\0';
	// Remove extra whitespace at start and end.
	trim(user);

	if (*excludematchflag) {
		stringpointer = startmatch + strlen ("match!=");
	} else {
		stringpointer = startmatch + strlen ("match=");
	}

	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startevent
		   && stringpointer != startuser
		   && stringpointer != endstring)) {
		*matchp = *stringpointer;
		stringpointer++;
		matchp++;
	}
	*matchp = '\0';
	// Remove extra whitespace at start and end.
	trim(match);

	// Return the criticality value.
	return (crit);
}


// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
int regmatch(const char *string, const char *pattern)
{
	int status;
	regex_t re;

	if (regcomp(&re, pattern, REG_EXTENDED | REG_NOSUB) != 0) {
		return (0);
	}
	status = regexec(&re, string, (size_t) 0, NULL, 0);
	regfree(&re);
	if (status != 0) {
		return (0);
	}
	return (1);
}

// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
// COMPILED REGULAR EXPRESSION version
int regmatchC(const char *string, regex_t re)
{
	int status;

	status = regexec(&re, string, (size_t) 0, NULL, 0);
	if (status != 0) {
		return (0);
	}
	return (1);
}

// Match string against an extended regular expression. Case insensitive.
// Return 1 for match, 0 for no-match or error.
int regmatchi(const char *string, const char *pattern)
{
	int status;
	regex_t re;

	if (regcomp(&re, pattern, REG_EXTENDED | REG_NOSUB | REG_ICASE) !=
	    0) {
		return (0);
	}

	status = regexec(&re, string, (size_t) 0, NULL, 0);
	regfree(&re);
	if (status != 0) {
		return (0);
	}
	return (1);
}

// Linked List Functions

void CreateLinkedList(void)
{
	head = currentnode = NULL;
}

int IsListEmpty(void)
{
	if (NULL == head)
		return 1;
	else
		return 0;
}

Node *AddToList(char *eventname, char *username, char *path,
		int criticality, char *options, int excludeflag, int excludematchflag,
		int returncode)
{
	Node *newNode = NULL;

	newNode = (Node *) malloc(sizeof(Node));

	if (newNode == NULL) {
		fprintf(stderr,
			"AddToList(): error in dynamic memory allocation\nCould not add a new objective into our linked list. You may be low on memory.\n");
		return ((Node *) NULL);
	}

	strncpy(newNode->event_name,eventname,MAX_EVENTNAME);
	newNode->criticality = criticality;
	newNode->returncode = returncode;
	newNode->excludeflag = excludeflag;
	newNode->excludematchflag = excludematchflag;

	strncpy(newNode->username, username, MAX_USERREG);
	// REDRED: NOTE: Must free regexp in objective list at end!
	regcomp(&newNode->usernameRE, username, REG_EXTENDED | REG_NOSUB);

	strncpy(newNode->path, path, PATH_MAX);
	regcomp(&newNode->pathRE, path, REG_EXTENDED | REG_NOSUB);

	strncpy(newNode->options, options, MAX_OPTIONS);
	regcomp(&newNode->optionsRE, options,
		REG_EXTENDED | REG_NOSUB | REG_ICASE);

	if (head == NULL) {
		head = newNode;
		newNode->next = NULL;
	} else {
		newNode->next = head;
		head = newNode;
	}

	return newNode;
}

void RemoveFromListHead(void)
{
	Node *tempPtr;

	if (NULL == head)
		return;

	tempPtr = head;
	head = head->next;

	// Free regular expressions.
	regfree(&tempPtr->optionsRE);
	regfree(&tempPtr->usernameRE);
	regfree(&tempPtr->pathRE);

	free(tempPtr);
}

void RemoveFromList(Node * node)
{
	Node *tempPtr, *previousPtr;

	if (NULL == node)
		return;

	if (head == node) {
		RemoveFromListHead();
		return;
	}

	tempPtr = head;

	while (NULL != tempPtr) {
		previousPtr = tempPtr;
		tempPtr = tempPtr->next;

		if (tempPtr == node) {
			previousPtr->next = tempPtr->next;

			// Free regular expressions.
			regfree(&tempPtr->optionsRE);
			regfree(&tempPtr->usernameRE);
			regfree(&tempPtr->pathRE);

			free(tempPtr);
			break;
		}
	}
}

void ResetCurrentNode(void)
{
	currentnode = head;
}

int IsValidItem(void)
{
	return (NULL == currentnode) ? 0 : 1;
}

Node *GetCurrentItem()
{
	return (currentnode);
}

void NextItemInList(void)
{
	if (NULL == currentnode)
		return;

	currentnode = currentnode->next;
}

void DestroyList(void)
{
	if (NULL == head) {
		return;
	}

	while (NULL != head) {
		Node *tempPtr = head;
		head = head->next;

		// Free regular expressions.
		regfree(&tempPtr->optionsRE);
		regfree(&tempPtr->usernameRE);
		regfree(&tempPtr->pathRE);

		free(tempPtr);
	}
}


// Pull out the fully qualified domain name, if possible.
char *getfqdn(char *FQDN)
{
	char hname[MAX_HOSTID];
	struct hostent *hp;

	if (gethostname(hname, MAX_HOSTID)) {
		strncpy(FQDN, "localhost.unknown", MAX_HOSTID);
		return (FQDN);
	}

	strncpy(FQDN, hname, MAX_HOSTID);

	hp = gethostbyname(hname);
	if (hp) {
		while (hp->h_aliases && *hp->h_aliases) {
			if (strlen(*(hp->h_aliases)) > strlen(hname)
			    && !strncmp(hname, *(hp->h_aliases),
					strlen(hname))) {
				strncpy(FQDN, *(hp->h_aliases),
					MAX_HOSTID);
			}
			hp->h_aliases++;
		}
		if (strlen(hp->h_name) > strlen(hname)
		    && !strncmp(hname, hp->h_name, strlen(hname))) {
			strncpy(FQDN, hp->h_name, MAX_HOSTID);
		}
	}

	return (FQDN);
}

int SendToSocket(char *message)
{
	if (AuditSocket) {
		if (sendto(AuditSocket, message,
			   strlen(message), 0,
			   (const struct sockaddr *) &AuditSocketName,
			   sizeof(struct sockaddr))
		    < 0) {
			puts(message);
			return (1);
		}
	} else {
		puts(message);
		return (1);
	}
	return (0);
}


////////////////////////////////////////////////////////////////////////////////
// This routine will send an event to the user-selected output device.
////////////////////////////////////////////////////////////////////////////////
int sendevent(char *string, int criticality)
{
	char stringout[MAX_AUDITREC] = "";
	static unsigned int EventCounter=1;
	char blank[2]="";
	char *position=string;
	int size=0;
	char *start=string;
	char *startdate=blank;
	char *startevent=blank;
	char *startcommand=blank;
	char *startuser=blank;
	char *startrealuser=blank;
	char *startpid=blank;
	char *startppid=blank;
	char *startreturn=blank;
	char *starttrail=blank;


	if(!string) {
		return(0);
	}

	if(EventCounter >= UINT_MAX) {
		EventCounter=1;
	}

	size=strlen(string);

	// Work through our string, and convert it into tab-delimited fields.
	// end-fields: 24, 40, 72, 81, 90, 99, 108, 129
	if(size > 24) {
		start=string; startdate=start;
		position=string+24;
		while(position > start && *position==' ') {
			*position='\0';
			position--;
		}
	}
	if(size > 40) {
		start=string+25; startevent=start;
		position=string+40;
		while(position > start && *position==' ') {
			*position='\0';
			position--;
		}
	}
	if(size > 72) {
		start=string+41; startcommand=start;
		position=string+72;
		while(position > start && *position==' ') {
			*position='\0';
			position--;
		}
	}
	if(size > 81) {
		start=string+73; startuser=start;
		position=string+81;
		while(position > start && *position==' ') {
			*position='\0';
			position--;
		}
	}
	if(size > 90) {
		start=string+82; startrealuser=start;
		position=string+90;
		while(position > start && *position==' ') {
			*position='\0';
			position--;
		}
	}
	if(size > 99) {
		start=string+91; startpid=start;
		position=string+99;
		while(position > start && *position==' ') {
			*position='\0';
			position--;
		}
	}
	if(size > 108) {
		start=string+100; startppid=start;
		position=string+108;
		while(position > start && *position==' ') {
			*position='\0';
			position--;
		}
	}
	if(size > 129) {
		start=string+109; startreturn=start;
		position=string+129;
		// NOTE tab:
		while(position > start && (*position==' ' || *position=='	')) {
			*position='\0';
			position--;
		}
	}
	if(size > 130) {
		starttrail=string+130;
		// Kill off tabs/spaces at the start of this field.
		while(*starttrail == ' ' || *starttrail == '\t')  {
			starttrail++;
		}
	}
	// Ok, string is now broken up with null terminators. Go through and collect our items of interest.
	

	// NEW audit record format, including host and AIXAudit identifier string:
	snprintf(stringout, MAX_AUDITREC, "%s\tAIXAudit\t%d\t%d\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s", hostid,
		 criticality, EventCounter, startdate,startevent,startcommand,startuser,startrealuser,startpid,startppid,startreturn,starttrail);

	EventCounter++;

	if (AuditDestination & AUDIT_TO_STDOUT) {
		// Send to STDOUT.
		printf("%s\n", stringout);
	}

	if (AuditDestination & AUDIT_TO_FILE) {
		// Write to file
		if (AuditFile != (FILE *) NULL) {
			fprintf(AuditFile, "%s\n", stringout);
			fflush(AuditFile);
		}
	}

	if (AuditDestination & AUDIT_TO_NETWORK) {
		if(AuditDestination & AUDIT_TO_SYSLOG) {
			// Do Syslog stuff here.
			time_t currenttime;
			struct tm *newtime;
			char CurrentDate[16]="";

			time(&currenttime);
			newtime=localtime(&currenttime);
			syslogdate(CurrentDate,newtime);

			snprintf(stringout,MAX_AUDITREC,"<%d> %s %s AIXAudit\t%d\t%s",SyslogDestination,CurrentDate,hostid,criticality,string);
		}
		if (AuditSocket) {
			SendToSocket(stringout);
		}
	}

	return (0);
}

Node *CheckObjective(char *username, char *searchterm, char *eventname,
		     char *options, int returncode)
{
	static int firstcall = 0;
	static Node *currentnode;

	if (firstcall == 0) {
		ResetCurrentNode();
		firstcall = 1;
	}

	while (IsValidItem()) {
		currentnode = GetCurrentItem();
		if ((!strcmp(eventname, currentnode->event_name)
		     || !strcmp(currentnode->event_name, "AUDIT_ALL"))
		    && (returncode == currentnode->returncode
			|| currentnode->returncode == RETURNCODE_ANY)) {
			// Are we including users, or excluding.
			if (currentnode->excludeflag ^ regmatchC (username, currentnode->usernameRE)) {
				if (currentnode->excludematchflag ^ regmatchC (searchterm, currentnode->pathRE)) {
					if (regmatchC
					    (options,
					     currentnode->optionsRE)) {
						NextItemInList();
						return (currentnode);
					}
				}
			} else {
				if (regmatchC
				    (username, currentnode->usernameRE)
				    && regmatchC(searchterm,
						 currentnode->pathRE)) {
					if (regmatchC
					    (options,
					     currentnode->optionsRE)) {
						NextItemInList();
						return (currentnode);
					}
				}
			}
		}
		NextItemInList();
	}

	ResetCurrentNode();
	return ((Node *) NULL);
}

// Pick out the core details from an audit event, and set the username/searchterm/eventname
int GetAIXDetails(char *logbuffer, char *username, char *searchterm,
		  char *eventname, char *options)
{
	char *position = logbuffer;
	char tempreturn[20]="";
	int eventsize=MAX_EVENTNAME;
	int usersize=MAX_USERNAME;
	int searchsize=MAX_PATH;
	int optionssize=MAX_OPTIONS;
	int detailedsearch=0;

	if(!logbuffer || !username || !searchterm || !eventname || !options) {
		return(0);
	}

	// Kill the newline
	position=logbuffer+strlen(logbuffer)-1;
	if(*position == '\n') {
		*position='\0';
	}

	searchterm[0] = '\0';
	eventname[0] = '\0';
	username[0] = '\0';
	options[0] = '\0';

	position=logbuffer+25;
	// Grab the date first.
	if(eventsize > 16) { eventsize=16; }
	strncpy(eventname,position,eventsize);

	// Kill whitespace
	trim(eventname);
	position=logbuffer+73;
	if(usersize>9) { usersize=9; }
	strncpy(username,position,usersize);
	trim(username);

	position=logbuffer+109;
	strncpy(tempreturn,position,20);
	trim(tempreturn);

	if(!strcmp(eventname,"PROC_Execute") || !strcmp(eventname,"PROC_LPExecute")) {
		// Pull the command out of the extra details.
		if(strlen(logbuffer) > 130)  {
			position=logbuffer+130;
			position = FindToken("name ",position);
			if(position) {
				strncpy(searchterm,position,searchsize);
				detailedsearch=1;
			}
		}
	} else if(!strcmp(eventname,"USER_SU")) {
		position=logbuffer+130;
		strncpy(searchterm,position,searchsize);
	} else if(!strncmp(eventname,"FILE_",5)) {
		position=logbuffer+130;
		position=FindToken("path: ",position);
		if(position) {
			strncpy(searchterm,position,searchsize);
			detailedsearch=1;
		} else {
			position=FindToken("filename ",logbuffer+130);
			if(position) {
				strncpy(searchterm,position,searchsize);
				detailedsearch=1;
			} else {
				position=FindToken("topath: ",logbuffer+130);
				if(position) {
					strncpy(searchterm,position,searchsize);
					detailedsearch=1;
				}
			}
		}
		position=logbuffer+130;
		position=FindToken("mode: ",position);
		if(position) {
			char *position2;
			position2=strstr(position," ");
			if(position2) {
				if(position2-position < optionssize) {
					optionssize=position2-position;
				}
				// Indicates read vs write.
				// Copy until the next space.
				strncpy(options,position,optionssize);
				options[optionssize]='\0';
			}
		}
	}

	// Do we have significant details?
	if(!detailedsearch) {
		// No? Use the entire extra details field as a search term.
		if(strlen(logbuffer) > 130) {
			strncpy(searchterm,logbuffer+130,searchsize);
		} else {
			// Oh dear.. fall back to the entire record.
			strncpy(searchterm,logbuffer,searchsize);
		}
	}

	if (!strcmp(tempreturn, "OK")) {
		return (RETURNCODE_SUCCESS);
	}
	return (RETURNCODE_FAILURE);
}

char *FindToken(char *token, char *buffer)
{
	char *position;

	if(!token || !buffer) {
		return((char *)NULL);
	}
	position=strstr(buffer, token);
	if(position) {
		position +=strlen(token);
	}
	return(position);
}

void syslogdate(char *sdate, struct tm *cdate)
{
	char Month[4];
	char Date[3];
	char Hour[3];
	char Min[3];
	char Sec[3];

	if(!sdate || !cdate) return;

	switch (cdate->tm_mon) {
		case 0: strcpy(Month,"Jan"); break;
		case 1: strcpy(Month,"Feb"); break;
		case 2: strcpy(Month,"Mar"); break;
		case 3: strcpy(Month,"Apr"); break;
		case 4: strcpy(Month,"May"); break;
		case 5: strcpy(Month,"Jun"); break;
		case 6: strcpy(Month,"Jul"); break;
		case 7: strcpy(Month,"Aug"); break;
		case 8: strcpy(Month,"Sep"); break;
		case 9: strcpy(Month,"Oct"); break;
		case 10: strcpy(Month,"Nov"); break;
		default: strcpy(Month,"Dec"); break;
	}

	if(cdate->tm_mday<10) {
		snprintf(Date,3," %d%c",cdate->tm_mday,0);
	} else {
		snprintf(Date,3,"%d%c",cdate->tm_mday,0);
	}

	if(cdate->tm_hour<10) {
		snprintf(Hour,3,"0%d%c",cdate->tm_hour,0);
	} else {
		snprintf(Hour,3,"%d%c",cdate->tm_hour,0);
	}

	if(cdate->tm_min<10) {
		snprintf(Min,3,"0%d%c",cdate->tm_min,0);
	} else {
		snprintf(Min,3,"%d%c",cdate->tm_min,0);
	}

	if(cdate->tm_sec<10) {
		snprintf(Sec,3,"0%d%c",cdate->tm_sec,0);
	} else {
		snprintf(Sec,3,"%d%c",cdate->tm_sec,0);
	}

	snprintf(sdate,16,"%s %s %s:%s:%s%c",Month,Date,Hour,Min,Sec,0);
}


void DebugMsg(const char *pszFormat, ...)
{
	char buf[8192] = "";
	// FILE *fp;

	va_list arglist;
	va_start(arglist, pszFormat);
	vsnprintf(&buf[strlen(buf)], 8192 - strlen(buf) - 1, pszFormat,
		  arglist);
	va_end(arglist);

	printf("%s\n",buf);
	fflush(stdout);

	//fp = fopen("/tmp/SNARE-OUT", "a");
	//fprintf(fp,"%s\n",buf);
	//fclose(fp);
}
