// SNARE - Audit / EventLog analysis and forwarding
// Copyright 2001-2003 InterSect Alliance Pty Ltd
// http://www.intersectalliance.com/
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// See Readme.txt file for more information.
//

#include <stdio.h>
#include <stdlib.h>
#include <winsock2.h>
#include <windows.h>
#include <time.h>
#include <tchar.h>
#include <sys/stat.h>

#include "NTServApp.h"
#include "SnareCore.h"
#include "webserver.h"

// #define DEBUG_TO_FILE 1

// Quick function proto
void DebugMsg(const char* pszFormat, ...);


// Pull this from registry
DWORD			WEBSERVER_ACTIVE = 0;

DWORD			g_dwLastError			= 0L;
char			Hostname[100];
ThreadStruct	g_Info;
SOCKET			g_hSocket = INVALID_SOCKET;      // global client/server socket
int				nStopListening=0;
DWORD			dwEventIDRead[6];				// maintain entries for those events that we have already read.
TCHAR			DELIM[2]="	";	// TAB

int				SNAREDEBUG=0;

static Node *head, *currentnode;

CSnarecoreService::CSnarecoreService()
:CNTService("SNARE")
{
	//TODO: Initialize your class members here
}

BOOL CSnarecoreService::OnInit()
{
	char szError[MAX_STRING];
	//TODO: Perform any initialization that needs to be done before intering the main loop

	SNAREDEBUG=this->DEBUGSET;

	if( !InitWinsock( szError ) )
	{
		if(SNAREDEBUG) { DebugMsg(szError); }
		return FALSE;
	}

	m_hEventList[0] = CreateEvent(NULL, TRUE, FALSE, NULL);
	m_hEventList[1] = CreateEvent(NULL, TRUE, FALSE, NULL);
	m_hEventList[2] = CreateEvent(NULL, TRUE, FALSE, NULL);
	m_hEventList[3] = CreateEvent(NULL, TRUE, FALSE, NULL);
	m_hEventList[4] = CreateEvent(NULL, TRUE, FALSE, NULL);
	m_hEventList[5] = CreateEvent(NULL, TRUE, FALSE, NULL);
	
	// Web server
	m_hEventList[6] = CreateEvent(NULL, TRUE, FALSE, NULL);

	// PIPE server
	m_hEventList[7] = CreateEvent(NULL, TRUE, FALSE, NULL);

	// Web server reset
	m_hEventList[8] = CreateEvent(NULL, TRUE, FALSE, NULL);

	
	if(m_hEventList[0] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() 0 failed"); }	return FALSE;	}
	if(m_hEventList[1] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() 1 failed"); }	return FALSE;	}
	if(m_hEventList[2] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() 2 failed"); }	return FALSE;	}
	if(m_hEventList[3] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() 3 failed"); }	return FALSE;	}
	if(m_hEventList[4] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() 4 failed"); }	return FALSE;	}
	if(m_hEventList[5] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() 5 failed"); }	return FALSE;	}
	
	if(m_hEventList[6] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() Web Server failed"); }	return FALSE;	}
	if(m_hEventList[7] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() PIPE failed"); }	return FALSE;	}
	if(m_hEventList[8] == NULL)
	{	if(SNAREDEBUG) { DebugMsg("CreateEvent() Web Server Reset failed"); }	return FALSE;	}


	
	if(SNAREDEBUG) { DebugMsg("SNARE Initialisation complete"); }
	// return FALSE here if initialization failed & the service shouldn't start
	return TRUE;
}

void CSnarecoreService::Run()
{
	DWORD dwEventLogRecords = 0, dwOldestEventLogRecord = 0, dwNewestEventLogRecord = 0,
		dwEvLogStart = 0, dwEvLogCounter = 0, dwNumberOfBytesToRead = 0, 
		dwBytesRead = 0, dwMinNumberOfBytesNeeded = 0, dwCancel = 0, dwClose = 0;

	DWORD ClearTabs=0;
	DWORD dwSyslogHeader=0; // Send the Syslog header?
	DWORD dwPortNumber=80;
	DWORD dwRestrictIP=0;
	DWORD dwUsePassword=0;

	DWORD dwCatchUpCount=0;

	long Category=0;

	// Define an eventlogrecord buffer of 8k.
	// Should be enough for an overwhelming majority of circumstances.
	TCHAR EventLogRecordBuffer[MAX_EVENT];
	TCHAR SourceName[100];	// Eg: "Security" or "Active Directory"
	TCHAR SIDType[100];		// User or System
	TCHAR EventLogType[50]=""; // Warning / Information / success / failure
	TCHAR ExpandedString[MAX_STRING];
	TCHAR DataString[MAX_STRING];
	TCHAR ComputerName[256];
	TCHAR UserName[256];
	TCHAR szCategoryString[256]; // "Detailed Tracking"

  	TCHAR szError[MAX_STRING];

	TCHAR SubmitTime[26]="None Yet";
	TCHAR WriteTime[26];

	TCHAR lpszIPAddress[16];
	TCHAR lpszPassword[256];

	OVERLAPPED Overlapped;	// PIPE variable

	short nEventCount=9; // including the web server event and pipe events

	short TimeoutCounter=0;
	short LogCounter=0;
	short SnareTimeout=0;

	// Destination for log events. Default it to something safe.
	TCHAR lpszDestination[512]="127.0.0.1";

	PEVENTLOGRECORD pELR = 0;
	PSID UserSID = 0;
	unsigned long EventID=0;
	
	BOOL bRetVal = FALSE;
	BOOL fExit = FALSE;
	UINT uStep = 0, uStepAt = 0, uPos = 0, uOffset = 0;

	int nRet;
	
	SOCKET g_hSocket;

	DWORD dwWaitRes=0;
	DWORD Offset=0;
	DWORD dwDestPort=6161;
	DWORD dwSyslog=13;
	
	TCHAR EventLogSourceName[6][_MAX_PATH + 1];
	strcpy(EventLogSourceName[0],"Security");
	strcpy(EventLogSourceName[1],"System");
	strcpy(EventLogSourceName[2],"Application");
	strcpy(EventLogSourceName[3],"Directory Service");
	strcpy(EventLogSourceName[4],"DNS Server");
	strcpy(EventLogSourceName[5],"File Replication Service");

	// This is the internal Snare eventlog counter.
	DWORD EventLogCounter[6];
	EventLogCounter[0]=0;
	EventLogCounter[1]=0;
	EventLogCounter[2]=0;
	EventLogCounter[3]=0;
	EventLogCounter[4]=0;
	EventLogCounter[5]=0;

	dwEventIDRead[0]=0; dwEventIDRead[1]=0; dwEventIDRead[2]=0;
	dwEventIDRead[3]=0; dwEventIDRead[4]=0; dwEventIDRead[5]=0;
	dwEventIDRead[6]=0; dwEventIDRead[7]=0;

	hWritePipe=INVALID_HANDLE_VALUE;


	if(SNAREDEBUG) { DebugMsg("SNARE is Running"); }


	// READ in our data
	GetHostname(Hostname,sizeof(Hostname));
	GetDestPort(&dwDestPort);
	GetSyslog(&dwSyslog);
	GetSyslogHeader(&dwSyslogHeader);
	GetWEBSERVER_ACTIVE(&WEBSERVER_ACTIVE);
	GetPortNumber(&dwPortNumber);
	GetDestination(lpszDestination,sizeof(lpszDestination));
	if(dwSyslogHeader) {
		GetDelim(DELIM,sizeof(DELIM));
	}
	GetPassword(lpszPassword,sizeof(lpszPassword));
	GetIPAddress(lpszIPAddress,sizeof(lpszIPAddress));
	GetClearTabs(&ClearTabs);

	// Load the objective data here.
	ReadObjectives();

	// Open the socket (UDP) to the indicated port
	g_hSocket = ConnectToServer( lpszDestination, (UINT)dwDestPort, szError );

	if( g_hSocket == INVALID_SOCKET )
	{
		DebugMsg(szError);
		DebugMsg("Sending Data to localhost");
		// The best we can do is point it at localhost.
		g_hSocket = ConnectToServer( "127.0.0.1", (UINT)dwDestPort, szError );
		if( g_hSocket == INVALID_SOCKET ) {
			DebugMsg(szError);
			// Give up.
			return;
		}
	}

	// Ok, we have finished our general configuration reads.

	if(SNAREDEBUG) { DebugMsg("Opening event log sources\n"); }

	// Open each event log, and if the log exists, bind the event notifier.
	hEventLog[0] = OpenEventLog( NULL, EventLogSourceName[0] );
	if(hEventLog[0])
	{
		if(CheckLogExists(EventLogSourceName[0]))
		{
			nRet = NotifyChangeEventLog( hEventLog[0], m_hEventList[0] );
			if(!nRet) { if(SNAREDEBUG) { DebugMsg("Event Bind 0 failed"); } return; }
		
			// Work out the latest audit log record
			GetOldestEventLogRecord(hEventLog[0], &dwOldestEventLogRecord);
			GetNumberOfEventLogRecords(hEventLog[0], &dwEventLogRecords);
			if(dwEventLogRecords) {
				dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;
			} else {
				dwNewestEventLogRecord = 0;
			}

			dwEventIDRead[0]=dwNewestEventLogRecord;
			if(SNAREDEBUG) { DebugMsg("Opened %s\n",EventLogSourceName[0]); }

			// Set log retention to zero.


		} else {
			CloseEventLog(hEventLog[0]);
			hEventLog[0]=NULL;
		}

	}
	hEventLog[1] = OpenEventLog( NULL, EventLogSourceName[1] );
	if(hEventLog[1])
	{
		if(CheckLogExists(EventLogSourceName[1]))
		{
			nRet = NotifyChangeEventLog( hEventLog[1], m_hEventList[1] );
			if(!nRet) { if(SNAREDEBUG) { DebugMsg("Event Bind 1 failed"); } return; }

			// Work out the latest audit log record
			GetOldestEventLogRecord(hEventLog[1], &dwOldestEventLogRecord);
			GetNumberOfEventLogRecords(hEventLog[1], &dwEventLogRecords);
			if(dwEventLogRecords) {
				dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;
			} else {
				dwNewestEventLogRecord = 0;
			}
			dwEventIDRead[1]=dwNewestEventLogRecord;
			if(SNAREDEBUG) { DebugMsg("Opened %s\n",EventLogSourceName[1]); }
		} else {
			CloseEventLog(hEventLog[1]);
			hEventLog[1]=NULL;
		}
	}
	
	hEventLog[2] = OpenEventLog( NULL, EventLogSourceName[2] );
	if(hEventLog[2])
	{
		if(CheckLogExists(EventLogSourceName[2]))
		{
			nRet = NotifyChangeEventLog( hEventLog[2], m_hEventList[2] );
			if(!nRet) { if(SNAREDEBUG) { DebugMsg("Event Bind 2 failed"); } return; }

			// Work out the latest audit log record
			GetOldestEventLogRecord(hEventLog[2], &dwOldestEventLogRecord);
			GetNumberOfEventLogRecords(hEventLog[2], &dwEventLogRecords);
			if(dwEventLogRecords) {
				dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;
			} else {
				dwNewestEventLogRecord = 0;
			}
			dwEventIDRead[2]=dwNewestEventLogRecord;
			if(SNAREDEBUG) { DebugMsg("Opened %s\n",EventLogSourceName[2]); }
		} else {
			CloseEventLog(hEventLog[2]);
			hEventLog[2]=NULL;
		}
	}
	
	// Optional event logs - ie: not present on NT
	// NOTE: If this eventlog does not exist, Windows returns
	//       a handle to the Application log.
	hEventLog[3] = OpenEventLog( NULL, EventLogSourceName[3] );
	if(hEventLog[3])
	{
		if(CheckLogExists(EventLogSourceName[3]))
		{
			nRet = NotifyChangeEventLog( hEventLog[3], m_hEventList[3] );
			if(!nRet) { if(SNAREDEBUG) { DebugMsg("Event Bind 3 failed"); } return; }

			// Work out the latest audit log record
			GetOldestEventLogRecord(hEventLog[3], &dwOldestEventLogRecord);
			GetNumberOfEventLogRecords(hEventLog[3], &dwEventLogRecords);
			if(dwEventLogRecords) {
				dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;
			} else {
				dwNewestEventLogRecord=0;
			}
			dwEventIDRead[3]=dwNewestEventLogRecord;
			if(SNAREDEBUG) { DebugMsg("Opened %s\n",EventLogSourceName[3]); }

		} else {
			CloseEventLog(hEventLog[3]);
			hEventLog[3]=NULL;
		}
	}

	// Optional event logs - ie: not present on NT
	hEventLog[4] = OpenEventLog( NULL, EventLogSourceName[4] );
	if(hEventLog[4])
	{
		if(CheckLogExists(EventLogSourceName[4]))
		{
			nRet = NotifyChangeEventLog( hEventLog[4], m_hEventList[4] );
			if(!nRet) { if(SNAREDEBUG) { DebugMsg("Event Bind 4 failed"); } return; }

			// Work out the latest audit log record
			GetOldestEventLogRecord(hEventLog[4], &dwOldestEventLogRecord);
			GetNumberOfEventLogRecords(hEventLog[4], &dwEventLogRecords);
			if(dwEventLogRecords) {
				dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;
			} else {
				dwNewestEventLogRecord = 0;
			}
			dwEventIDRead[4]=dwNewestEventLogRecord;
			if(SNAREDEBUG) { DebugMsg("Opened %s\n",EventLogSourceName[4]); }
		} else {
			CloseEventLog(hEventLog[4]);
			hEventLog[4]=NULL;
		}
	}

	// Optional event logs - ie: not present on NT
	hEventLog[5] = OpenEventLog( NULL, EventLogSourceName[5] );
	if(hEventLog[5])
	{
		if(CheckLogExists(EventLogSourceName[5]))
		{
			nRet = NotifyChangeEventLog( hEventLog[5], m_hEventList[5] );
			if(!nRet) { if(SNAREDEBUG) { DebugMsg("Event Bind 5 failed"); } return; }

			// Work out the latest audit log record
			GetOldestEventLogRecord(hEventLog[5], &dwOldestEventLogRecord);
			GetNumberOfEventLogRecords(hEventLog[5], &dwEventLogRecords);
			if(dwEventLogRecords) {
				dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;
			} else {
				dwNewestEventLogRecord = 0;
			}
			dwEventIDRead[5]=dwNewestEventLogRecord;
			if(SNAREDEBUG) { DebugMsg("Opened %s\n",EventLogSourceName[5]); }
		} else {
			CloseEventLog(hEventLog[5]);
			hEventLog[5]=NULL;
		}
	}

	if(WEBSERVER_ACTIVE) {
		if(SNAREDEBUG) { DebugMsg("Starting web thread.\n"); }
		if(InitWebServer((unsigned short)dwPortNumber,lpszPassword,lpszIPAddress) >0) {
			StartThread(m_hEventList[6]);
		} else {
			WEBSERVER_ACTIVE = 0;
		}
	}

	if(SNAREDEBUG) { DebugMsg("Creating named pipe.\n"); }

	// Create the pipe here. If we create it earlier, then there's a chance that
	// The GUI could lock onto the pipe before we have an event handler in place.
	hPipe = CreateNamedPipe("\\\\.\\pipe\\snare",PIPE_ACCESS_OUTBOUND|FILE_FLAG_OVERLAPPED,
							PIPE_WAIT,1,5000,5000,0,NULL);
    
	if (hPipe == INVALID_HANDLE_VALUE)
	{
		if(SNAREDEBUG) { DebugMsg("Error in CreateNamedPipe = %d\n",GetLastError()); }
		// return FALSE; // This is not a terminal problem.
	}

	// Monitor the pipe
	if (hPipe != INVALID_HANDLE_VALUE) {
		int piperc;
		Overlapped.hEvent = m_hEventList[7];
		//Despite what is written in the MS API, these values **_MUST_** be set
		Overlapped.Offset = 0;
		Overlapped.OffsetHigh = 0;

		if (ConnectNamedPipe(hPipe,&Overlapped) == 0) {
			piperc = GetLastError();
		}
	
		if ((piperc != ERROR_IO_PENDING) && 
			(piperc != ERROR_PIPE_CONNECTED))
		{
			// Not going to be able to do any pipe transactions.
			hPipe=INVALID_HANDLE_VALUE;

		}
		// OK pipe connected.
		if(SNAREDEBUG) { DebugMsg("Pipe connected.\n"); }
	}

	// Set the terminate flag to zero.
	// setting this value to TRUE will terminate the service,
	// and ask it to save it's current status.
	g_Info.bTerminate = FALSE;

	
	if(SNAREDEBUG) { DebugMsg("Entering main loop.\n"); }
	// This is the service's main run loop.
    while (m_bIsRunning) 
	{
		//TODO: Add code to perform processing here  
		// If we have been asked to terminate, do so.
		if(g_Info.bTerminate)
		{
			m_bIsRunning=0;
			break;
		}	
		
		// The service performs one check per second. This should not be
		// a significant drain on resources.
		dwWaitRes=WaitForMultipleObjects(nEventCount,m_hEventList,FALSE,1000);

		// if(dwWaitRes != WAIT_FAILED && dwWaitRes != WAIT_TIMEOUT)
		if(dwWaitRes != WAIT_FAILED)
		{
			UINT EventTriggered=0;

			if(dwWaitRes == WAIT_OBJECT_0) {
				EventTriggered=0;
				ResetEvent(m_hEventList[0]);
				TimeoutCounter=0;
				if(SNAREDEBUG) { DebugMsg("WAIT_OBJECT 0 Triggered.\n"); }
			} else if (dwWaitRes == WAIT_OBJECT_0+1) {
				EventTriggered=1;
				ResetEvent(m_hEventList[1]);
				TimeoutCounter=0;
				if(SNAREDEBUG) { DebugMsg("WAIT_OBJECT 1 Triggered.\n"); }
			} else if (dwWaitRes == WAIT_OBJECT_0+2) {	
				EventTriggered=2;
				ResetEvent(m_hEventList[2]);
				TimeoutCounter=0;
				if(SNAREDEBUG) { DebugMsg("WAIT_OBJECT 2 Triggered.\n"); }
			} else if (dwWaitRes == WAIT_OBJECT_0+3) {	
				EventTriggered=3;
				ResetEvent(m_hEventList[3]);
				TimeoutCounter=0;
				if(SNAREDEBUG) { DebugMsg("WAIT_OBJECT 3 Triggered.\n"); }
			} else if (dwWaitRes == WAIT_OBJECT_0+4) {	
				EventTriggered=4;
				ResetEvent(m_hEventList[4]);
				TimeoutCounter=0;
				if(SNAREDEBUG) { DebugMsg("WAIT_OBJECT 4 Triggered.\n"); }
			} else if (dwWaitRes == WAIT_OBJECT_0+5) {	
				EventTriggered=5;
				TimeoutCounter=0;
				ResetEvent(m_hEventList[5]);
				if(SNAREDEBUG) { DebugMsg("WAIT_OBJECT 5 Triggered.\n"); }
			} else if (dwWaitRes == WAIT_OBJECT_0+6) {
				// Web server has data to read.
				ResetEvent(m_hEventList[6]);
				if(SNAREDEBUG) { DebugMsg("WEB Server Connect.\n"); }

				if(WEBSERVER_ACTIVE) {
					// Let handleconnect have the eventlist pointer, in case it needs
					// to signal a restart

					HandleConnect(m_hEventList[8]);

					if(SNAREDEBUG) { DebugMsg("Running thread again.\n"); }
					StartThread(m_hEventList[6]);
				}
				
				continue;
			} else if (dwWaitRes == WAIT_OBJECT_0+7) {
				// PIPE event.
				ResetEvent(m_hEventList[7]);
				int piperc;
				
				if (hPipe != INVALID_HANDLE_VALUE) {
					int pipecount=0;

					char str_temp[500] = "";
					char str_temp2[500];
					DWORD dwWritten;

					if(!dwEvLogCounter) {
						_snprintf(str_temp2,500,"");
					} else {
						_snprintf(str_temp2,500,"Internal Windows Eventlog counter is %d\n",dwEvLogCounter);
					}
					
					//We can now write to the pipe, since we have a connected client
					_snprintf(str_temp,sizeof(str_temp),"SNARE v2 Status:\n" \
							"SNARE is currently ACTIVE\n\n" \
							"Sending data to remote system %s on port %d\n" \
							"%s\n" \
							"Last event was generated at %s",lpszDestination,dwDestPort,str_temp2,SubmitTime);


					// If lpOverlapped is not NULL, lpNumberOfBytesWritten can be NULL.
					if (WriteFile(hPipe,str_temp,strlen(str_temp),NULL,&Overlapped) == 0)	{
						if(SNAREDEBUG) { DebugMsg("WriteFile error = %d\n",GetLastError()); }

						if (GetLastError() == ERROR_IO_PENDING) {
							if (GetOverlappedResult(hPipe,&Overlapped,&dwWritten,TRUE) == 0) {
								if(SNAREDEBUG) { DebugMsg("Error in GetOverlappedResult = %d\n",GetLastError()); }
							} else {
								if(SNAREDEBUG) { DebugMsg("GetOverlappedResult bytes written = %d\n",dwWritten); }
							}
						}
					} else {
						if(SNAREDEBUG) { DebugMsg("Successfull WriteFile\n"); }
					}

				
					// Done. Close down the pipe, and reconnect it.
					DisconnectNamedPipe(hPipe);

					if (ConnectNamedPipe(hPipe,&Overlapped) == 0) {
						piperc = GetLastError();
					}
	
					if ((piperc != ERROR_IO_PENDING) && 
						(piperc != ERROR_PIPE_CONNECTED))
					{
						// Not going to be able to do any pipe transactions.
						hPipe=INVALID_HANDLE_VALUE;
					}


					// NOW: If user connects to us, then it's time to attempt a connect to the GUI PIPE.
					if(hWritePipe == INVALID_HANDLE_VALUE) {
						// Pipe not open - time to try to open a path to the GUI
						hWritePipe = CreateFile("\\\\.\\pipe\\snaregui",GENERIC_WRITE,0,NULL,
							OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
						
						if(hWritePipe == INVALID_HANDLE_VALUE) {
							if(SNAREDEBUG) { DebugMsg("Info: Could not grab SnareGUI pipe handle\n"); }
						}
					}
				}
				continue;
			} else if (dwWaitRes == WAIT_OBJECT_0+8) {

				// We need to re-read our configuration file.
				ResetEvent(m_hEventList[8]);
			    
				DestroyList();

				if(WEBSERVER_ACTIVE) {
					CloseWebServer();
				}

				// cancel blocking calls, if any
				WSACancelBlockingCall();
	
				// close socket
				if( g_hSocket != INVALID_SOCKET ) {
					closesocket(g_hSocket);
				}

				// READ in our data
				GetHostname(Hostname,sizeof(Hostname));
				GetDestPort(&dwDestPort);
				GetSyslog(&dwSyslog);
				GetSyslogHeader(&dwSyslogHeader);
				GetWEBSERVER_ACTIVE(&WEBSERVER_ACTIVE);
				GetPortNumber(&dwPortNumber);
				GetDestination(lpszDestination,sizeof(lpszDestination));
				if(dwSyslogHeader) {
					GetDelim(DELIM,sizeof(DELIM));
				}
				GetPassword(lpszPassword,sizeof(lpszPassword));
				GetIPAddress(lpszIPAddress,sizeof(lpszIPAddress));
				GetClearTabs(&ClearTabs);

				// Load the objective data here.
				ReadObjectives();

				// Open the socket (UDP) to the indicated port
				g_hSocket = ConnectToServer( lpszDestination, (UINT)dwDestPort, szError );

				if( g_hSocket == INVALID_SOCKET )
				{
					DebugMsg(szError);
					DebugMsg("Sending Data to localhost");
					// The best we can do is point it at localhost.
					g_hSocket = ConnectToServer( "127.0.0.1", (UINT)dwDestPort, szError );
					if( g_hSocket == INVALID_SOCKET ) {
						DebugMsg(szError);
						// Give up.
						return;
					}
				}

				// Ok, we have finished our general configuration reads.

				if(WEBSERVER_ACTIVE) {
					if(SNAREDEBUG) { DebugMsg("Starting web thread.\n"); }
					if(InitWebServer((unsigned short)dwPortNumber,lpszPassword,lpszIPAddress) >0) {
						StartThread(m_hEventList[6]);
					} else {
						WEBSERVER_ACTIVE = 0;
					}
				}

			} else if (dwWaitRes == WAIT_TIMEOUT) {
				if(SNAREDEBUG) { DebugMsg("Timeout hit"); }
				// Every 20 seconds, check one of the open log files for new events.
				// This is just in case notifychangeeventlog does not let us know
				// about new events
				if(TimeoutCounter < 20) {
					TimeoutCounter++;
					continue;
				}
				TimeoutCounter=0;
				
				short tcounter=1;
				// Skip eventlogs that we do not have.
				while(hEventLog[LogCounter] == NULL && tcounter < 6) {
					tcounter++;
					LogCounter++;
					if(LogCounter > 5) {
						LogCounter=0;
					}
				}

				if(SNAREDEBUG) { DebugMsg("20 seconds have elapsed. Checking a log file: %d",LogCounter); }

				if(hEventLog[LogCounter]!=NULL) {
					EventTriggered=LogCounter;
				}
				
				LogCounter++;
				if(LogCounter > 5) {
					LogCounter=0;
				}

				SnareTimeout=1;
			} else {
				if(SNAREDEBUG) { DebugMsg("Warning: An event occured that I am not programmed to deal with. Continuing\n"); }
				continue;
			}

			// The first eventlog record in the file - absolute record number.
			GetOldestEventLogRecord(hEventLog[EventTriggered], &dwOldestEventLogRecord);
			// The total number of eventlog records.
			GetNumberOfEventLogRecords(hEventLog[EventTriggered], &dwEventLogRecords);
			// The last eventlog record number
			dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;

			// Have we been signaled without any new events being added?
			if(((dwNewestEventLogRecord == dwEventIDRead[EventTriggered]) && dwNewestEventLogRecord != 0 && SnareTimeout==0))
			{
				// This should not be happening.. unless the newest event log has a id of 0
				// for some reason, or perhaps we have rolled around to the exact same
				// audit record number while the audit service was not running.
				// Either way, ignore the event, and do nothing.
				if(SNAREDEBUG) { DebugMsg("Log rotation problem. Continuing at next event."); }

				// Sleep for a moment, just to give the system time to rotate appropriately.
				Sleep(1000);
				continue;
			} else if((dwNewestEventLogRecord == dwEventIDRead[EventTriggered]) && dwNewestEventLogRecord != 0 && SnareTimeout==1) {
				// SnareTimeout has asked us to check the eventlog for new events as a result
				// of a regular timeout. If there are no new events, continue on.
				SnareTimeout=0;
				continue;
			} else {
				SnareTimeout=0;

				dwEvLogStart=dwEventIDRead[EventTriggered]+1;
				// Has the eventlog somehow overtaken us, or is this the first time we
				// have been run?

				if(dwOldestEventLogRecord > dwEventIDRead[EventTriggered])
				{
					// Yes. Set our start counter to the first record.
					// This probably means that windows NT is overwriting old events,
					// and we have been overtaken. VERY doubtful that this will happen
					// within the one second timeout.
					dwEventIDRead[EventTriggered]=dwOldestEventLogRecord;
					dwEvLogStart = dwOldestEventLogRecord;
					
					if(SNAREDEBUG) { DebugMsg("Acceleration problem: I have lost my old event pointer. Recalculating"); }
				}
				if(dwNewestEventLogRecord < dwEventIDRead[EventTriggered])
				{
					// Oh dear, the event id has recycled!
					// This means that someone has cleared the eventlog.
					// Set our data pointer back to the start of the event log.
					CloseEventLog(hEventLog[EventTriggered]);
					hEventLog[EventTriggered] = OpenEventLog( NULL, EventLogSourceName[EventTriggered] );
					nRet = NotifyChangeEventLog( hEventLog[EventTriggered], m_hEventList[EventTriggered] );
					if(!nRet) { if(SNAREDEBUG) { DebugMsg("Event Bind failed"); } }

					// Grab these details again just in case.
					// They seem to persist, but read them again anyway.
					GetOldestEventLogRecord(hEventLog[EventTriggered], &dwOldestEventLogRecord);
					// The total number of eventlog records.
					GetNumberOfEventLogRecords(hEventLog[EventTriggered], &dwEventLogRecords);
					// The last eventlog record number
					dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;

					// Jump to the newest event log
					dwEventIDRead[EventTriggered]=dwNewestEventLogRecord;
					dwEvLogStart = dwNewestEventLogRecord;
					if(SNAREDEBUG) { DebugMsg("Event log seems to have wrapped!. Continuing."); }
				}

				for(dwEvLogCounter=dwEvLogStart; dwEvLogCounter <= dwNewestEventLogRecord; dwEvLogCounter++)
				{
					// Has the user requested that we exit here?
					// This may be called if the user has multiple megabytes of
					// data to send and they didn't realise it.
					if(g_Info.bTerminate)
					{
						m_bIsRunning=0;
						break;
					}						
					
					// NOTE: THIS WILL POTENTIALLY READ MULTIPLE RECORDS INTO THE BUFFER!
					// Ignore any after the first, and iterate through the file.
					// Read the Event Log records we have not yet seen
					// NOTE 2: These events can be BIG - 78k in one case. (DrWatson dump)
					bRetVal = ReadEventLog( hEventLog[EventTriggered], EVENTLOG_BACKWARDS_READ|EVENTLOG_SEEK_READ, dwEvLogCounter, EventLogRecordBuffer,
							  MAX_EVENT,&dwBytesRead,&dwMinNumberOfBytesNeeded );
					// I would like to filter out events that have been caused by Snare here,
					// but there seems to be no way to correalate the process ID reported in
					// the event with real process ID of Snare

					pELR=(PEVENTLOGRECORD)EventLogRecordBuffer;

					if(!bRetVal) {
						// Problem encountered. Exit cleanly.
						g_dwLastError = GetLastError();
						if(g_dwLastError == ERROR_INSUFFICIENT_BUFFER) {
							if(SNAREDEBUG) { DebugMsg("Not enough buffer available for a event log record. Dropped event."); }
							// Buffer is not large enough? An event greater than 8k? You're joking.
							// For the moment, drop this event.
							// We may wish to consider malloc'ing some extra, RAM
							// to retrieve the event (Like we did in the old backlog)
							// but I don't think it's worthwhile, due to the risk of memory leaks.

							// NOTE: User report of a 78k!!! log message (drwatson dump)

							continue; // to the next event.
						}  else if (g_dwLastError == ERROR_EVENTLOG_FILE_CHANGED || g_dwLastError == ERROR_EVENTLOG_FILE_CORRUPT) {
							// Someone cleared the file, close and reread!
							CloseEventLog(hEventLog[EventTriggered]);
							hEventLog[EventTriggered] = OpenEventLog( NULL, EventLogSourceName[EventTriggered] );
							nRet = NotifyChangeEventLog( hEventLog[EventTriggered], m_hEventList[EventTriggered] );
							if(!nRet) { if(SNAREDEBUG) { DebugMsg("Event Bind failed"); } }

							// Grab these details just in case.
							// They should persist, but read them again anyway.
							GetOldestEventLogRecord(hEventLog[EventTriggered], &dwOldestEventLogRecord);
							// The total number of eventlog records.
							GetNumberOfEventLogRecords(hEventLog[EventTriggered], &dwEventLogRecords);
							// The last eventlog record number
							dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;

							dwEventIDRead[EventTriggered]=dwOldestEventLogRecord;
							dwEvLogStart = dwOldestEventLogRecord;

							if(SNAREDEBUG) { DebugMsg("Eventlog has been cleared. Re-created pointers to log"); }

						} else if (g_dwLastError == ERROR_INVALID_PARAMETER || g_dwLastError == ERROR_HANDLE_EOF || g_dwLastError == RPC_S_UNKNOWN_IF) {
							// This error generally means that the user has a small
							// audit buffer set to overwrite old audit events.
							// We have not been quick enough to catch the old event before
							// it is overwritten, so Continue on to the next one and try to
							// catch up again.
							m_bIsRunning=1;

							// Add one to our catchup variable - this should help us if we get into an endless loop here.
							dwCatchUpCount++;

							if(dwCatchUpCount > 100) {
								// Hmm.. we have problems, we can't seem to catch up to the audit record
								// pointer. Lets try and sleep for a moment in the hope that the system
								// stabilises. Note: This will mean that we lose a few records (potentially)
								// but at least we won't grind the system to a halt..
								dwCatchUpCount=0;
								if(SNAREDEBUG) { DebugMsg("Acceleration problem: Small audit buffer means that I cannot catch up to the most recent audit event. Sleeping for a moment."); }
								
								Sleep(1000);
								
								GetOldestEventLogRecord(hEventLog[EventTriggered], &dwOldestEventLogRecord);
								// The total number of eventlog records.
								GetNumberOfEventLogRecords(hEventLog[EventTriggered], &dwEventLogRecords);
								// The last eventlog record number
								dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;
							}

							break; // Break out of the for loop.
						} else {
							// Unknown error. Windows NT is  doing something very strange.
							if(SNAREDEBUG) { DebugMsg("An unknown error occurred - g_dwLastError is %d. Continuing.",g_dwLastError); }
							// m_bIsRunning=0;
							m_bIsRunning=1;
							
							// Sleep for a second, since we do NOT want to get into an endless loop
							// here, and eat CPU.
							Sleep(1000);

							GetOldestEventLogRecord(hEventLog[EventTriggered], &dwOldestEventLogRecord);
							// The total number of eventlog records.
							GetNumberOfEventLogRecords(hEventLog[EventTriggered], &dwEventLogRecords);
							// The last eventlog record number
							dwNewestEventLogRecord = (dwEventLogRecords + dwOldestEventLogRecord) -1;
							
							break; // Break out of the for loop.
						}
					}
					
					// We received a good event. Clear our catch-up variable.
					dwCatchUpCount=0;
					
					if(bRetVal)
					{
						// No error. Proceed.
						Offset = sizeof(EVENTLOGRECORD);
						// Grab the source name, the start of which is eventlogrecordbuffer + sizeof(eventlogrecord)
						strncpy(SourceName, (LPTSTR)((LPBYTE)pELR + Offset),sizeof(SourceName));
						// Jump to the next element of the eventlog record
						Offset += strlen(SourceName) +1; // length + \0 character
						// Note that there is a chance that the sourcename is longer than 1024.
						// If this is the case, then this event is probably corrupted.

						strncpy(ComputerName, (LPTSTR)((LPBYTE)pELR + Offset),sizeof(ComputerName));
						if(SNAREDEBUG) { if(ComputerName) { DebugMsg("Received event from computer %s.\n",ComputerName); } }

						if(pELR->UserSidLength > 0)
						{
							char RefDomainName[MAX_STRING];
							SID_NAME_USE SidNameUse = (SID_NAME_USE)(SidTypeUser - 1);

							// Malloc some space for the usersid.
							UserSID = (SID *)GlobalAlloc(GPTR,pELR->UserSidLength);
							// Malloc failed. Dump this event, try the next.
							if(UserSID == NULL) {
								if(SNAREDEBUG) { DebugMsg("Memory allocation problems: Could not allocate UserSID"); }
								continue;
							}
							memcpy(UserSID, (PSID)((LPBYTE)pELR + pELR->UserSidOffset), pELR->UserSidLength);
							
							*RefDomainName = *UserName = '\0';
							
							// Note that I don't care about the RefDomainName
							// or SidNameUse here.
							unsigned long tempsize=sizeof(UserName);
							unsigned long tempsize2=sizeof(RefDomainName);
							bRetVal = LookupAccountSid(NULL, UserSID, 
								UserName, &tempsize, 
								RefDomainName, &tempsize2,								
								&SidNameUse);
							if(bRetVal) {
								GetSIDType(SidNameUse,SIDType,sizeof(SIDType));
							} else {
								UserName[0]='\0';
								SIDType[0]='\0';
							}
							GlobalFree(UserSID);
						} else {
							UserName[0]='\0';
						}
						if(SNAREDEBUG) { if(UserName) { DebugMsg("User Name grabbed: %s.\n",UserName); } }

						GetEventLogType(EventLogType, pELR->EventType, sizeof(EventLogType));

						if(pELR->TimeGenerated && pELR->TimeWritten) {
							struct tm *ptmTime;

							ptmTime = localtime((time_t *)&(pELR->TimeGenerated));
							if(ptmTime) {
								strftime(SubmitTime, sizeof(SubmitTime),"%a %b %d %H:%M:%S %Y", ptmTime);
							} else {
								// Could not pull back date/time from the event. Use current time.
								time_t ltime;
								ltime=time(NULL);
								ptmTime = localtime(&ltime);
								strftime(SubmitTime, sizeof(SubmitTime),"%a %b %d %H:%M:%S %Y", ptmTime);
							}
							ptmTime = localtime((time_t *)&(pELR->TimeWritten));
							if(ptmTime) {
								strftime(WriteTime, sizeof(WriteTime),"%a %b %d %H:%M:%S %Y", ptmTime);
							} else {
								time_t ltime;
								ltime=time(NULL);
								ptmTime = localtime(&ltime);
								strftime(WriteTime, sizeof(WriteTime),"%a %b %d %H:%M:%S %Y", ptmTime);
							}
							
							//lstrcpyn(SubmitTime, asctime(localtime((time_t *)&(pELR->TimeGenerated))), sizeof(SubmitTime));
							// lstrcpyn(WriteTime, asctime(localtime((time_t *)&(pELR->TimeWritten))), sizeof(WriteTime));
							// May need to chomp off the last character - asctime return a newline!
							// SubmitTime[strlen(SubmitTime)-1]='\0';
							// WriteTime[strlen(WriteTime)-1]='\0';
						} else {
							SubmitTime[0]='\0';
							WriteTime[0]='\0';
						}

						if(SNAREDEBUG) { if(SubmitTime) { DebugMsg("Date and Time grabbed: %s.\n",SubmitTime); } }

						EventID = pELR->EventID;
						ExpandStrings(pELR,EventLogSourceName[EventTriggered],ExpandedString,sizeof(ExpandedString));

						if(SNAREDEBUG) { DebugMsg("Strings Expanded\n"); }

						GetDataString(pELR,DataString,sizeof(DataString));

						if(SNAREDEBUG) { DebugMsg("DataStrings Grabbed\n"); }

						Category = pELR->EventCategory;

						GetCategoryString(pELR,EventLogSourceName[EventTriggered],szCategoryString,sizeof(szCategoryString));

						if(SNAREDEBUG) { if(szCategoryString) { DebugMsg("Category String grabbed: %s.\n",szCategoryString); } }

						// Send out to network
						// First set the separator in the strings area to use tabs instead.
						char szSendString[MAX_OUTPUT_STRING]=""; // Nice big memory buffer - just in case.
						char szTempString[MAX_STRING]=""; // Approximately the maximmum we could reasonably expect to transfer over UDP
						UINT counter=0;

						// Chop down the ExpandedString into nice easy bits.
						if(strlen(ExpandedString))
						{
							UINT stringsize=0;
							BOOL bDelim=0;
							BOOL bNewLine=0;

							if(SNAREDEBUG) { DebugMsg("Expanding Strings...\n"); }

							if(strlen(ExpandedString) >= MAX_STRING) {
								stringsize=MAX_STRING;
							} else {
								stringsize=strlen(ExpandedString);
							}

							// Within the "Extra strings" section of a windows log,
							// get rid of newlines (cr / lf), and change any multiple-tabs to a single tab.
							
							while(counter < stringsize)
							{
								// Replace any delimiter characters (usually tab)
								// with spaces.
								// Also, if the ClearTabs registry key is set, kill off the tabs.
								if(ExpandedString[counter]==DELIM[0] || (ExpandedString[counter]==9 && ClearTabs==1))
								{
									bNewLine=0;
									if(!bDelim) {
										if((strlen(szTempString) + 1) < MAX_STRING) {
											strcat(szTempString," ");
										}
										bDelim=1;
									}
								}
								else if(ExpandedString[counter]==10 || ExpandedString[counter]==13)
								{
									// CR/LF - substitute a space instead.
									if((strlen(szTempString) + 1) < MAX_STRING) {
										strcat(szTempString," ");
									}
									
									bDelim=0;
									bNewLine=1;
								} else if(ExpandedString[counter]==' ' && bNewLine==1) {
									// Lots of spaces after newlines. Trim them down.
									1;
								} else {
									char szTempString2[2]="";

									bNewLine=0;
									bDelim=0;
									szTempString2[0]=(char)ExpandedString[counter];
									szTempString2[1]='\0';
									if((strlen(szTempString) + strlen(szTempString2)) < MAX_STRING) {
										strcat(szTempString,szTempString2);
									}
								}
								counter++;
							}
						} else {
							// Could not expand the strings.
							strcpy(szTempString,"");
						}
						
						char CurrentDate[16]="";
						// Check that we have all the data that we need.
						// Note: DataString may not exist, but everything else should.
						// Removed this bit: && strlen(UserName) && strlen(SIDType)

						// Sanitise some of the data if it's not available.
						if(!strlen(EventLogSourceName[EventTriggered]))
							strncpy(EventLogSourceName[EventTriggered],"Unknown",sizeof(EventLogSourceName[EventTriggered]));
						if(!strlen(SubmitTime))
							strncpy(SubmitTime,"Unknown",sizeof(SubmitTime));
						if(!strlen(SourceName))
							strncpy(SourceName,"Unknown",sizeof(SourceName));
						if(!strlen(EventLogType))
							strncpy(EventLogType,"Unknown",sizeof(EventLogType));
						if(!strlen(ComputerName))
							strncpy(ComputerName,"Unknown",sizeof(ComputerName));
						if(!strlen(szTempString))
							strncpy(szTempString,"Unknown",sizeof(szTempString));
						if(!strlen(szCategoryString))
							strncpy(szCategoryString,"Unknown",sizeof(szCategoryString));

						if(strlen(EventLogSourceName[EventTriggered]) && strlen(SubmitTime) && strlen(SourceName) && strlen(EventLogType) && strlen(ComputerName) && DataString && strlen(szTempString) && strlen(szCategoryString))
						{
							char *stringp;
							if(!strlen(UserName)) {
								strncpy(UserName,"Unknown User",sizeof(UserName));
								strncpy(SIDType,"N/A",sizeof(SIDType));
							}

							// This is the point at which we integrate our regular expression handling
							// and event filtering facilities.
							// NOTE:
							// Was going to implement a regular expression matching capability,
							// - but based on the limitations of the NT Audit subsystem, it's probably
							//   better that we use dos wildcards instead.
							//

							DWORD ShortEventID=0;
							// Cut off the severity, flags, and  facility data
							// Just leave the real event ID.
							ShortEventID = EventID & 65535;
							TCHAR header[256];

							Node * checknode;
							BOOL nodematch=0;
							int criticality=0;

							if(SNAREDEBUG) { DebugMsg("Attempting to match this event against our objectives...\n"); }
							
							// Check objectives
							// NOTE: If there are no objectives, send all?
							if(!head) {
								nodematch=1;
							} else {
								int etype=0,stype=0;

								if(!strcmp("Success Audit",EventLogType)) {
									etype = TYPE_SUCCESS;
								} else if(!strcmp("Failure Audit",EventLogType)) {
									etype = TYPE_FAILURE;
								} else if(!strcmp("Information",EventLogType)) {
									etype = TYPE_INFO;
								} else if(!strcmp("Warning",EventLogType)) {
									etype = TYPE_WARN;
								} else if(!strcmp("Error",EventLogType)) {
									etype = TYPE_ERROR;
								} else { etype = TYPE_INFO; }	// Default to INFO if no valid logtype provided.
	

								if(!strcmp("Security",EventLogSourceName[EventTriggered])) {
									stype = LOG_SEC;
								} else if(!strcmp("System",EventLogSourceName[EventTriggered]))	{
									stype = LOG_SYS;
								} else if(!strcmp("Application",EventLogSourceName[EventTriggered])) {
									stype = LOG_APP;
								} else if(!strcmp("Directory Service",EventLogSourceName[EventTriggered])) {
									stype = LOG_DIR;
								} else if(!strcmp("DNS Service",EventLogSourceName[EventTriggered])) {
									stype = LOG_DNS;
								} else if(!strcmp("File Replication Service",EventLogSourceName[EventTriggered])) {
									stype = LOG_REP;
								} else { stype = LOG_APP; } // Assume application log if no valid source provided.


								while(checknode=CheckObjective(ShortEventID,UserName,szTempString,
									etype,stype)) {
									nodematch=1;
									if(criticality < checknode->criticality) {
										criticality = checknode->criticality;
									}
									// break;
								}
							}


							if(SNAREDEBUG) { DebugMsg("Node Matched: %d.\n",nodematch); }

							// END
							if(nodematch) {
								if(dwSyslogHeader) {
									time_t currenttime;
									struct tm *newtime;
									time(&currenttime);                
									newtime=localtime(&currenttime);
									
									syslogdate(CurrentDate,newtime);


									_snprintf(header,255,"<%ld>%s %s MSWinEventLog%s%d%s",dwSyslog,CurrentDate,Hostname,DELIM,criticality,DELIM);
								} else {
									_snprintf(header,255,"%s%sMSWinEventLog%s%d%s",Hostname,DELIM,DELIM,criticality,DELIM);
								}
								
								_snprintf(szSendString,MAX_OUTPUT_STRING-1,"%s%s%s%d%s%s%s%ld%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%d\0",header,EventLogSourceName[EventTriggered],DELIM,dwEvLogCounter,DELIM,SubmitTime,DELIM,ShortEventID,DELIM,SourceName,DELIM,UserName,DELIM,SIDType,DELIM,EventLogType,DELIM,ComputerName,DELIM,szCategoryString,DELIM,DataString,DELIM,szTempString,DELIM,EventLogCounter[EventTriggered]);


								// Jump through szSendString, and remove any newline characters.
								stringp=szSendString;
								while(*stringp) {
									// CR or LF
									if(*stringp==10 || *stringp==13) {
										*stringp=' ';
									}
									stringp++;
								}
								// Ok, now add a newline.
								strcat(szSendString,"\n");

								if(SNAREDEBUG) {
									if(SNAREDEBUG && szSendString) { DebugMsg("DEBUG: Sending the following string to the server: %s\n",szSendString); }
								}
								
								if( !SendToSocket(g_hSocket, szSendString, strlen(szSendString), szError) )
								{
									if(SNAREDEBUG) { if(szError) { DebugMsg(szError); } }
									// return;
								} else {
									// Increment the Snare internal event counter
									// Note: Maxdword is 4294967295
									EventLogCounter[EventTriggered]++;
									if(EventLogCounter[EventTriggered] >= MAXDWORD) {
										EventLogCounter[EventTriggered]=1;
									}
								}

								// Send data to the GUI, if the write-pipe is open.
								// Make sure we include the maximum criticality.
								if (hWritePipe != INVALID_HANDLE_VALUE)	{
									DWORD dwWriteSize=0;

									// if(SNAREDEBUG) {  DebugMsg("Sending data to the open GUI Pipe....\n"); }
									if(WriteFile(hWritePipe,szSendString,strlen(szSendString),&dwWriteSize,NULL)==0) {
										if(SNAREDEBUG) { DebugMsg("Warning: Could not write to GUI pipe\n"); }
										CloseHandle(hWritePipe);
										hWritePipe = INVALID_HANDLE_VALUE;
									}
								}
							}
						} else {
							if(SNAREDEBUG) {
								DebugMsg("DEBUG: I received an event that I could not process! Details are as follows:\n");
								DebugMsg("dwSyslog: ");
								if(dwSyslog) { DebugMsg("%d\n",dwSyslog);} else { DebugMsg("NO DATA\n");}
								DebugMsg("CurrentDate: ");
								if(CurrentDate) { DebugMsg("%s\n",CurrentDate);} else { DebugMsg("NO DATA\n");}
								DebugMsg("Hostname: ");
								if(Hostname) {DebugMsg("%s\n",Hostname);} else { DebugMsg("NO DATA\n");}
								DebugMsg("EventTriggered: ");
								if(EventTriggered) { DebugMsg("%d\n",EventTriggered);} else { DebugMsg("NO DATA\n");}
								DebugMsg("EventLogSourceName[EventTriggered]: ");
								if(EventLogSourceName[EventTriggered]) {DebugMsg("%s\n",EventLogSourceName[EventTriggered]);} else {DebugMsg("NO DATA\n");}
								DebugMsg("dwEvLogCounter: ");
								if(dwEvLogCounter) { DebugMsg("%d\n",dwEvLogCounter);} else { DebugMsg("NO DATA\n");}
								DebugMsg("SubmitTime: ");
								if(SubmitTime) { DebugMsg("%s\n",SubmitTime);} else { DebugMsg("NO DATA\n");}
								DebugMsg("EventID: ");
								if(EventID) { DebugMsg("%d\n",EventID);} else { DebugMsg("NO DATA\n");}
								DebugMsg("SourceName: ");
								if(SourceName) { DebugMsg("%s\n",SourceName);} else { DebugMsg("NO DATA\n");}
								DebugMsg("UserName: ");
								if(UserName) { DebugMsg("%s\n",UserName);} else { DebugMsg("NO DATA\n");}
								DebugMsg("SIDType: ");
								if(SIDType) { DebugMsg("%s\n",SIDType);} else { DebugMsg("NO DATA\n");}
								DebugMsg("EventLogType: ");
								if(EventLogType) { DebugMsg("%s\n",EventLogType);} else { DebugMsg("NO DATA\n");}
								DebugMsg("ComputerName: ");
								if(ComputerName) { DebugMsg("%s\n",ComputerName);} else { DebugMsg("NO DATA\n");}
								DebugMsg("DataString: ");
								if(DataString) { DebugMsg("%s\n",DataString);} else { DebugMsg("NO DATA\n");}
								DebugMsg("szTempString: ");
								if(szTempString) { DebugMsg("%s\n",szTempString);} else { DebugMsg("NO DATA\n");}
							}
						}
					}
				} // End for loop
				dwEventIDRead[EventTriggered]=dwNewestEventLogRecord;
			}
		}
    }

	if(SNAREDEBUG) { DebugMsg("SNARE Closing"); }

	CloseWebServer();
	
	if(hEventLog[0]) CloseEventLog(hEventLog[0]);
	if(hEventLog[1]) CloseEventLog(hEventLog[1]);
	if(hEventLog[2]) CloseEventLog(hEventLog[2]);
	if(hEventLog[3]) CloseEventLog(hEventLog[3]);
	if(hEventLog[4]) CloseEventLog(hEventLog[4]);
	if(hEventLog[5]) CloseEventLog(hEventLog[5]);

	if(hPipe != INVALID_HANDLE_VALUE) {
		CloseHandle(hPipe);
	}
	if(hWritePipe != INVALID_HANDLE_VALUE) {
		CloseHandle(hWritePipe);
	}
	
	TerminateWinsock(g_hSocket);
	if( m_hEventList[0] ) ::CloseHandle(m_hEventList[0]);
	if( m_hEventList[1] ) ::CloseHandle(m_hEventList[1]);
	if( m_hEventList[2] ) ::CloseHandle(m_hEventList[2]);
	if( m_hEventList[3] ) ::CloseHandle(m_hEventList[3]);
	if( m_hEventList[4] ) ::CloseHandle(m_hEventList[4]);
	if( m_hEventList[5] ) ::CloseHandle(m_hEventList[5]);
	
	if( m_hEventList[6] ) ::CloseHandle(m_hEventList[6]);
	if( m_hEventList[6] ) ::CloseHandle(m_hEventList[7]);

	// Free memory used by the objectives lists
	DestroyList();
}

void GetFQDN(char *string, int length)
{
	struct hostent *phostent;
	
	// Now, grab fully qualified hostname here.
	// Get the normal name.
	if(gethostname(string, length)) {		
		strncpy(string,"localhost.unknown",length);
		return;
	}
	
	
	// Now perform a lookup on that name.
	phostent=gethostbyname(string);
	if(phostent) {
		while(phostent->h_aliases && *phostent->h_aliases) {
			if(strlen(*(phostent->h_aliases)) > strlen(string) && !strncmp(string,*(phostent->h_aliases),strlen(string))) {
				strncpy(string,*(phostent->h_aliases),length);
			}
			phostent->h_aliases++;
		}
		if(strlen(phostent->h_name) > strlen(string) && !strncmp(string,phostent->h_name,strlen(string))) {
			strncpy(string,phostent->h_name,length);
		}
	}
}


BOOL GetCategoryString(PEVENTLOGRECORD pELR, char *Trigger, char *StringBuffer, DWORD length)
{
	TCHAR szKeyName[MAX_STRING]="";
	HKEY   hk = (HKEY)0;
	long Category;

	DWORD dwMaxPath;			
	DWORD dwType; // Temporary variable.
	TCHAR szExeFile[_MAX_PATH + 1]="", szExeFilePath[_MAX_PATH+1]="";
	HMODULE hModule = 0;

	StringBuffer[0]='\0';
	
	Category=pELR->EventCategory;

	if(!Category) {
		strncpy(StringBuffer,"None",length);
		StringBuffer[length-1]='\0';
		if(SNAREDEBUG) { DebugMsg("The current event has no category string."); }
		return(FALSE);
	}

	// wsprintf(szKeyName, _T("SYSTEM\\CurrentControlSet\\Services\\EventLog\\%s\\%s"), Trigger, Trigger);
	_snprintf(szKeyName, sizeof(szKeyName),_T("SYSTEM\\CurrentControlSet\\Services\\EventLog\\%s\\%s"), Trigger, Trigger);


	if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, szKeyName, 0L, KEY_READ, &hk) != NOERROR)
	{
		strncpy(StringBuffer,"None",length);
		StringBuffer[length-1]='\0';
		if(SNAREDEBUG) { DebugMsg("Could not open eventlog category string registry information"); }
		return(FALSE);
	}

	dwMaxPath = _MAX_PATH + 1;
	if(RegQueryValueEx(hk, _T("CategoryMessageFile"), 0, &dwType, (LPBYTE)szExeFile, &dwMaxPath) != NOERROR)
	{
		strncpy(StringBuffer,"None",length);
		StringBuffer[length-1]='\0';
		RegCloseKey(hk);
		if(SNAREDEBUG) { DebugMsg("Could not query categorymessagefile."); }
		return(FALSE);
	}

	// Trim off any DLLs other than the first.
	// This is to fix a problem introduced by win2k SP2, where it
	// adds a second DLL onto the end of the normal msaudite.dll.
	// There's probably a better way of doing this, as the following problems
	// COULD potentially occur:
	//	1) Someone reverses the order of msaudite.dll and the sp2.dll in the registry
	//  2) the audit description is really in the seond DLL!
	//
	// Best way is probably to try and get info from both/all in a loop, until we
	// find a real value.

	char *blankout;
	blankout = strstr(szExeFile,";");
	if(blankout) {
		*blankout = '\0';
	}

	if(ExpandEnvironmentStrings(szExeFile, szExeFilePath, _MAX_PATH + 1) == 0)
	{
		strncpy(szExeFilePath,szExeFile,_MAX_PATH+1);
		szExeFilePath[_MAX_PATH]='\0';

	}
	
	// NOTE: Windows 2000 SP2 introduces a bug here, because it
	// adds another DLL after the normal msaudite.dll (after a semicolon)
	hModule = LoadLibraryEx(szExeFilePath, 0, DONT_RESOLVE_DLL_REFERENCES);
	if(!hModule)
	{
		strncpy(StringBuffer,"None",length);
		StringBuffer[length-1]='\0';
		RegCloseKey(hk);
		if(SNAREDEBUG) { DebugMsg("LoadLibraryEx failed"); }
		return(FALSE);
	}

	if(!FormatMessage(
		FORMAT_MESSAGE_FROM_HMODULE | 
		FORMAT_MESSAGE_FROM_SYSTEM |
		FORMAT_MESSAGE_IGNORE_INSERTS,
		hModule, Category,
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
		(LPTSTR)StringBuffer, length,
		NULL))
	{		
		// No characters stored in buffer.
		strncpy(StringBuffer,"None",length);
		StringBuffer[length-1]='\0';
		RegCloseKey(hk);
		FreeLibrary(hModule);
		if(SNAREDEBUG) { DebugMsg("Could not format message"); }
		return(FALSE);
	}

	if(!StringBuffer)
	{
		strncpy(StringBuffer,"None",length);
		StringBuffer[length-1]='\0';
		RegCloseKey(hk);
		FreeLibrary(hModule);
		if(SNAREDEBUG) { DebugMsg("No string buffer!"); }
		return(FALSE);
	}

	FreeLibrary(hModule);
	RegCloseKey(hk);

	return(TRUE);
}


// Match a DOS wildcard against a string.
// eg: wildmatch("c:\blah\abc???.*","c:\blah\abc123.txt");
int wildmatch(char *pattern, char *source)
{
    if(*pattern == 0)
        return (*source == 0);

    // special case
    if(strspn(pattern, "*") == strlen(pattern))
        return 1;

    if(*source == 0)
        return 0;

    switch(*pattern){
    case '*':
        return wildmatch(pattern, source+1) || wildmatch(pattern+1, source) || wildmatch(pattern+1, source+1);
    case '?':
        return wildmatch(pattern+1, source+1);
    default:
        return (*pattern == *source) && wildmatch(pattern+1, source+1);
    }
}

// Match a DOS wildcard against a string.
// eg: wildmatch("C:\blah\abc???.*","c:\blah\abc123.txt");
int wildmatchi(char *pattern, char *source)
{
    if(*pattern == 0)
        return (*source == 0);

    // special case
    if(strspn(pattern, "*") == strlen(pattern))
        return 1;

    if(*source == 0)
        return 0;

    switch(*pattern){
    case '*':
        return wildmatchi(pattern, source+1) || wildmatchi(pattern+1, source) || wildmatchi(pattern+1, source+1);
    case '?':
        return wildmatchi(pattern+1, source+1);
    default:
		char lpattern,lsource;
		lpattern = tolower(*pattern);
		lsource = tolower(*source);
		
        return (lpattern == lsource) && wildmatchi(pattern+1, source+1);
    }
}



// Find the nth entry in the tab-delimited string 'search'.
// NOTE: may need to take into account the user delimiter
void splitstrings(char *store, int field, char *search, int size)
{
	int count;
	char *start=search;
	char *end=search;

	strcpy(store,"");

	for(count=1;count<=field;count++) {
		end=strstr(start,"	");
		if(!end) {
			if(count < field) {
				return;
			} else {
				strncpy(store,start,size);
				store[size]='\0';
				return;
			}
		} else {
			int nsize;

			if((end-start) > size) {
				nsize=size;
			} else {
				nsize=(end-start);
			}

			if(count == field) {
				strncpy(store,start,nsize);
				store[nsize]='\0';
				return;
			}

			start=end+1;
			end=start;
		}
	}
}


void syslogdate(char *sdate, struct tm *cdate)
{
	char Month[4];
	char Date[3];
	char Hour[3];
	char Min[3];
	char Sec[3];

	switch (cdate->tm_mon) {
		case 0: strcpy(Month,"Jan"); break;
		case 1: strcpy(Month,"Feb"); break;
		case 2: strcpy(Month,"Mar"); break;
		case 3: strcpy(Month,"Apr"); break;
		case 4: strcpy(Month,"May"); break;
		case 5: strcpy(Month,"Jun"); break;
		case 6: strcpy(Month,"Jul"); break;
		case 7: strcpy(Month,"Aug"); break;
		case 8: strcpy(Month,"Sep"); break;
		case 9: strcpy(Month,"Oct"); break;
		case 10: strcpy(Month,"Nov"); break;
		default: strcpy(Month,"Dec"); break;
	}

	if(cdate->tm_mday<10) {
		_snprintf(Date,3," %d\0",cdate->tm_mday);
	} else {
		_snprintf(Date,3,"%d\0",cdate->tm_mday);
	}

	if(cdate->tm_hour<10) {
		_snprintf(Hour,3,"0%d\0",cdate->tm_hour);
	} else {
		_snprintf(Hour,3,"%d\0",cdate->tm_hour);
	}

	if(cdate->tm_min<10) {
		_snprintf(Min,3,"0%d\0",cdate->tm_min);
	} else {
		_snprintf(Min,3,"%d\0",cdate->tm_min);
	}

	if(cdate->tm_sec<10) {
		_snprintf(Sec,3,"0%d\0",cdate->tm_sec);
	} else {
		_snprintf(Sec,3,"%d\0",cdate->tm_sec);
	}

	_snprintf(sdate,16,"%s %s %s:%s:%s\0",Month,Date,Hour,Min,Sec);
}


//////////////////////////////////////////////////////////////////
// SendToSocket
//              sends a buffer (buf) of size size to the specified socket
//              returns TRUE or FALSE.
BOOL  SendToSocket(SOCKET hSocket, char *buf, int nSize, char *szError)
{
	int   rv;
	
	// write it all
	while ((rv = send(hSocket, buf, nSize,0)) != nSize)
	{
		if (rv == -1)
		{
			sprintf(szError,"error sending to server. WSA ERROR: %d\r\n",
				WSAGetLastError());
			if(SNAREDEBUG) { if(szError) { DebugMsg(szError); } }
			return FALSE;
		}
		
		if (rv == nSize)
			break;
		buf += rv;
		nSize -= rv;
	}
	return TRUE;
}


BOOL GetDataString(PEVENTLOGRECORD pELR, char *DataString, DWORD length)
{
	register UINT x;
	LPBYTE			pData = 0;

	DataString[0]='\0';
	if(pELR->DataLength <= 0)
		return(FALSE);

	pData = (LPBYTE)GlobalAlloc(GPTR, pELR->DataLength * sizeof(BYTE));
	if(pData == NULL) {
		if(SNAREDEBUG) { DebugMsg("Could not allocate memory in GetDataString"); }
		return(FALSE);
	}

	memcpy(pData, (LPBYTE)((LPBYTE)pELR + pELR->DataOffset), pELR->DataLength);

	for(x = 0; x < pELR->DataLength; x += 8)
	{
		TCHAR DataStringAux[MAX_STRING];
		register UINT y;

		// wsprintf(DataStringAux, "%.4x: ", x);
		_snprintf(DataStringAux,sizeof(DataStringAux),"%.4x: ", x);
		if((strlen(DataString) + strlen(DataStringAux)) < length) {
			_tcscat(DataString, DataStringAux);
		} 
									
		for(y = x; y < x + 8; y++)
		{
			// wsprintf(DataStringAux, "%.2x ", pData[y]);
			_snprintf(DataStringAux, sizeof(DataStringAux), "%.2x ", pData[y]);
			if((strlen(DataString) + strlen(DataStringAux)) < length) {
				_tcscat(DataString, DataStringAux);
			}
		}
		if((strlen(DataString) + 2) < length) {
			_tcscat(DataString, _T("  "));
		}
									
		for(y = x; y < x + 8; y++)
		{
			if(!isprint((int)pData[y]))
				if((strlen(DataString) + 1) < length) {
					_tcscat(DataString, _T("."));
				}
			else
			{
				TCHAR s[2];
				s[0] = (TCHAR)pData[y];
				s[1] = '\0';
				if((strlen(DataString) + 1) < length) {
					_tcscat(DataString, s);
				}
			}
		}
		if((strlen(DataString) + 2) < length) {
			_tcscat(DataString, _T("\r\n"));
		}
	}
						
	if(pData)
	{
		GlobalFree(pData);
	}

	return(TRUE);
}

BOOL ExpandStrings(PEVENTLOGRECORD pELR, char *Trigger, char *StringBuffer, DWORD length)
{
	TCHAR szKeyName[MAX_STRING]="";
	LPVOID lpszBuffer = 0;
	LPBYTE pSourceName=0;
	HKEY   hk = (HKEY)0;
	LPBYTE pStrings = 0;

	DWORD dwMaxPath;			
	DWORD dwType; // Temporary variable.
	TCHAR szExeFile[_MAX_PATH + 1]="", szExeFilePath[_MAX_PATH+1]="";
	HMODULE hModule = 0;
	LPTSTR *Args = NULL;
	BOOL returncode=FALSE;
	
	// Shrink down the eventID to something valid
	DWORD ShortEventID=0;

	if(SNAREDEBUG) { DebugMsg("Inside ExpandStrings\n"); }
	
	StringBuffer[0]='\0';
	
	if(!pELR) { return(FALSE); }
	if(!pELR->NumStrings) {
		DWORD size=0;
		size=pELR->DataOffset - pELR->StringOffset;
		// Something strange with this audit event. Try and just send back the raw string
		// if it is available.
		if(size) {
			if(size > length) {
					size=length;
			}
			memcpy(StringBuffer, (LPBYTE)pELR + pELR->StringOffset, size);
			return(TRUE);
		}
		
		// Note: There are some events (eg: the one generated by Snare!
		// That do not seem to fit this (StringOffset = DataOffset) - where do we get strings
		// from????
		// return(FALSE);
		// pELR->NumStrings =1;
		ShortEventID=pELR->EventID;
		pSourceName = (LPBYTE) pELR + sizeof(EVENTLOGRECORD);

		if(!FormatMessage(
			FORMAT_MESSAGE_ALLOCATE_BUFFER | 
			FORMAT_MESSAGE_FROM_HMODULE | 
			FORMAT_MESSAGE_FROM_SYSTEM | 
			FORMAT_MESSAGE_IGNORE_INSERTS,
			hModule, ShortEventID,
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
			(LPTSTR)&lpszBuffer, 0,
			(va_list *)NULL))	{
				if(SNAREDEBUG) { DebugMsg("FormatMessage failed"); }
				return(FALSE);
		} else {
			strncpy(StringBuffer,(char *)lpszBuffer,length);
			return(TRUE);
		}
	}

	// Grab the shortened event ID to deal with buggy software that
	// does not fill out the whole DWORD.

	// HERE: Some buggy software does not write out a full dword eventid.
	//       However: By 'shortening' the eventid, it kills many legitimate
	//                event logs. *sigh*
	// ShortEventID=pELR->EventID & 65535;	// NOW REMOVED.
	ShortEventID=pELR->EventID;

	pSourceName = (LPBYTE) pELR + sizeof(EVENTLOGRECORD);

	pStrings = (LPBYTE)GlobalAlloc(GPTR,pELR->DataOffset - pELR->StringOffset);

	// Malloc failed. Dump this event, try the next.
	if(pStrings == NULL) 
	{
		if(SNAREDEBUG) { DebugMsg("Could not globalalloc in ExpandStrings"); }
		return(FALSE);
	}

	if(SNAREDEBUG) { DebugMsg("ExpandStrings: Copying data to pStrings\n"); }

	// Grab the strings (located between stringoffset and dataoffset.
	memcpy(pStrings, (LPBYTE)pELR + pELR->StringOffset, pELR->DataOffset - pELR->StringOffset);

	// wsprintf(szKeyName, _T("SYSTEM\\CurrentControlSet\\Services\\EventLog\\%s\\%s"), Trigger, pSourceName);
	_snprintf(szKeyName, sizeof(szKeyName), _T("SYSTEM\\CurrentControlSet\\Services\\EventLog\\%s\\%s"), Trigger, pSourceName);

	if(SNAREDEBUG) { if(szKeyName) { DebugMsg("ExpandStrings: Opening registry key: %s\n",szKeyName); } }

	if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, szKeyName, 0L, KEY_READ, &hk) != NOERROR)
	{
		strncpy(StringBuffer,(char *)pStrings,length);
		GlobalFree(pStrings);
		if(SNAREDEBUG) { DebugMsg("Could not open registry data in expandstrings"); }
		return(FALSE);
	}

	if(SNAREDEBUG) { DebugMsg("ExpandStrings: Querying key\n"); }

	dwMaxPath = _MAX_PATH + 1;
	if(RegQueryValueEx(hk, _T("EventMessageFile"), 0, &dwType, (LPBYTE)szExeFile, &dwMaxPath) != NOERROR)
	{
		strncpy(StringBuffer,(char *)pStrings,length);
		GlobalFree(pStrings);
		RegCloseKey(hk);
		if(SNAREDEBUG) { DebugMsg("Could not queryvalue in expandstrings"); }
		return(FALSE);
	}


	// Establish a loop for those eventlog entries that use more than one
	// DLL to retrieve audit data from. Loop through each until we find a
	// valid bit of info.

	TCHAR szExeFile2[_MAX_PATH + 1];
	char * pszExeFile2;
	char * semipos;
	
	strncpy(szExeFile2,szExeFile,_MAX_PATH);
	pszExeFile2=szExeFile2;
	
	if(SNAREDEBUG) { DebugMsg("ExpandStrings: Duplicated szExeFile: %s\n",pszExeFile2); }
	
	do {
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: checking strstr..\n"); }
		semipos=strstr(pszExeFile2,";");
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: done strstr..\n"); }
		if(semipos) {
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: found a semicolon in %s\n",pszExeFile2); }
			if(semipos-pszExeFile2 < _MAX_PATH) {
				if(SNAREDEBUG) { DebugMsg("ExpandStrings: copying\n"); }
				strncpy(szExeFile,pszExeFile2,(semipos-pszExeFile2));
				szExeFile[semipos-pszExeFile2]='\0';
				if(SNAREDEBUG) { DebugMsg("ExpandStrings: copyied: %s\n",szExeFile); }
			} else {
				if(SNAREDEBUG) { DebugMsg("ExpandStrings: copying (else)\n"); }
				strncpy(szExeFile,pszExeFile2,_MAX_PATH);
				if(SNAREDEBUG) { DebugMsg("ExpandStrings: copyied (else): %s\n",szExeFile); }
			}
			pszExeFile2=semipos+1;
		} else {
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: copying (else2)\n"); }
			strncpy(szExeFile,pszExeFile2,_MAX_PATH);
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: copyied (else2): %s\n",szExeFile); }
			pszExeFile2=(char *)NULL;
		}
		
		// If someone has added a semi-colon without actually putting info after it, break out.
		if(!strlen(szExeFile)) {
			break;
		}
		
		// zap up our pointer past the semi colon
		
		
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Expanding environment strings for %s\n",szExeFile); }
		
		if(ExpandEnvironmentStrings(szExeFile, szExeFilePath, _MAX_PATH + 1) == 0)
		{
			strncpy(szExeFilePath,szExeFile,_MAX_PATH+1);
			szExeFilePath[_MAX_PATH]='\0';
			
			// strncpy(StringBuffer,(char *)pStrings,length);
		}
		
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Loading library\n"); }
		
		// NOTE: Windows 2000 SP2 introduces a bug here, because it
		// adds another DLL after the normal msaudite.dll (after a semicolon)
		// Have changed this routine to cover this problem. We are now in a do..while loop.
		hModule=NULL;
		hModule = LoadLibraryEx(szExeFilePath, 0, DONT_RESOLVE_DLL_REFERENCES);
		if(!hModule)
		{
			if(SNAREDEBUG) { DebugMsg("LoadLibraryEx failed in expandstrings"); }
			continue;
		}
		
		if(!pELR->NumStrings)
		{
			FreeLibrary(hModule);
			if(SNAREDEBUG) { DebugMsg("No NumStrings for this event?"); }
			continue;
		}
		
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Allocating Args\n"); }
		
		if(pELR->NumStrings) {
			// Allocate memory and split up string arguments
			// Note: We are only allocating RAM here for the pointer list. The actual string
			// RAM is allocated from within GetParameterMsg
			Args = (LPTSTR *)GlobalAlloc(GPTR, (pELR->NumStrings) * sizeof(TCHAR *));
			if(Args == NULL)
			{
				FreeLibrary(hModule);
				DebugMsg("Could not allocate memory for the appropriate number of strings.");
				continue;
			}
		}
		
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Getting arguments\n"); }
		
		// No need to free anything after this call. We are just constructing a pointer list to
		// data within pELR
		if(pELR->NumStrings) {
			GetArgs(pELR,(char **)Args);
		}
		
		// Now, jump through the parametermessage stuff.
		// Basically: jump through each of the strings,
		// if it contains a %%, then send it through
		DWORD ArgCount;
		DWORD NewStringsCount=0;
		// These two variables are required for freeing the ram malloced
		// by GetParameterMsg
		char **AllocStr=NULL;
		DWORD AllocCount=0;
		
		if(pELR->NumStrings) {
			
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: Allocating space for numstrings\n"); }
			
			AllocStr=(char **)GlobalAlloc(GPTR,pELR->NumStrings);
			if(!AllocStr) {
				FreeLibrary(hModule);
				GlobalFree(Args);
				if(SNAREDEBUG) { DebugMsg("FormatMessage failed - AllocStr no go"); }			
				continue;
			}
			
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: Getting parameter messages"); }
			
			char * tArg;
			for(ArgCount=0; ArgCount<pELR->NumStrings; ArgCount++) {
				if(SNAREDEBUG && Args[ArgCount]) { DebugMsg("ExpandStrings: String is %s",Args[ArgCount]); }
				if(strstr(Args[ArgCount],"%%")) {
					if(SNAREDEBUG && Args[ArgCount]) { DebugMsg("ExpandStrings: Parameter string found: %s",Args[ArgCount]); }
					NewStringsCount++;
					
					// Args[ArgCount]=GetParameterMsg(Args[ArgCount],szKeyName);
					tArg=GetParameterMsg(Args[ArgCount],szKeyName);
					// Save off the location of the allocated RAM
					// So we can free it later.
					// if(Args[ArgCount]) {
					if(tArg) {
						Args[ArgCount]=tArg;
						AllocStr[AllocCount]=Args[ArgCount];
						AllocCount++;
					}
				}
			}
			
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: Formatting Message"); }
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: AllocCount is %d",AllocCount); }
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: Args is %ld",Args); }
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: hModule is %ld",(long)hModule); }
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: pELR is %ld",pELR); }
			if(SNAREDEBUG) { DebugMsg("ExpandStrings: pELR EventID is %d",pELR->EventID); }
			if(SNAREDEBUG && pStrings) { DebugMsg("ExpandStrings: pStrings %s",pStrings); }
			if(SNAREDEBUG && lpszBuffer) { DebugMsg("ExpandStrings: lpszbuffer is %ld",lpszBuffer); }
			
			
			if(Args) {
				// This FORMATMESSAGE call is causing a DrWATSON in some strange circumstances
				// principally on exchange boxes... (eg: eventid 287, and 1292)
				// Not sure why yet.
				// I'll encase this in a try/catch block.
				
				try {
					if(!FormatMessage(
						FORMAT_MESSAGE_ALLOCATE_BUFFER | 
						FORMAT_MESSAGE_FROM_HMODULE | 
						FORMAT_MESSAGE_FROM_SYSTEM | 
						FORMAT_MESSAGE_ARGUMENT_ARRAY,
						hModule, ShortEventID,
						MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
						(LPTSTR)&lpszBuffer, 0,
						(va_list *)Args))	{
						if(SNAREDEBUG) { DebugMsg("FormatMessage failed"); }
						
						// No characters stored in buffer.
						if(AllocCount) {
							for(ArgCount=0;ArgCount<AllocCount;ArgCount++) {
								if(AllocStr[ArgCount]) {
									GlobalFree(AllocStr[ArgCount]);
								}
							}
						}
						if(AllocStr) {
							GlobalFree(AllocStr);
						}
						
						if(hModule) {	FreeLibrary(hModule);	}
						if(Args)	{	GlobalFree(Args);		}
						if(SNAREDEBUG) { DebugMsg("FormatMessage failure cleanup complete"); }
						continue;
					}
				} catch(...) {
					if(SNAREDEBUG) { DebugMsg("FormatMessage Died - handling catch"); }
					
					// No characters stored in buffer.
					if(AllocCount) {
						for(ArgCount=0;ArgCount<AllocCount;ArgCount++) {
							if(AllocStr[ArgCount]) {
								GlobalFree(AllocStr[ArgCount]);
							}
						}
					}
					if(AllocStr) {
						GlobalFree(AllocStr);
					}
					
					if(hModule) {	FreeLibrary(hModule);	}
					if(Args)	{	GlobalFree(Args);		}
					if(SNAREDEBUG) { DebugMsg("FormatMessage failure cleanup complete"); }
					continue;
				}
			} else {
				// ARGS - null pointer.
				if(SNAREDEBUG) { DebugMsg("Args has mysteriously disappeared"); }
				
				// No characters stored in buffer.
				if(AllocCount) {
					for(ArgCount=0;ArgCount<AllocCount;ArgCount++) {
						if(AllocStr[ArgCount]) {
							GlobalFree(AllocStr[ArgCount]);
						}
					}
				}
				if(AllocStr) {
					GlobalFree(AllocStr);
				}
				
				if(hModule) {	FreeLibrary(hModule);	}
				if(Args)	{	GlobalFree(Args);		}
				if(SNAREDEBUG) { DebugMsg("Cleaning up after args dissappearance"); }
				continue;
			}
		}
	
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Freeing ram 1\n"); }
	
		if(AllocCount) {
			// And any RAM allocated within getparametermsg
			for(ArgCount=0;ArgCount<AllocCount;ArgCount++) {
				if(AllocStr[ArgCount]) {
					GlobalFree(AllocStr[ArgCount]);
				}
			}
		}
	
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Freeing ram 2\n"); }
		if(AllocStr) {
			GlobalFree(AllocStr);
		}
	
		if(!lpszBuffer)
		{
			if(hModule) { FreeLibrary(hModule); }
			if(pELR->NumStrings && Args) {
				GlobalFree(Args);
			}
			if(SNAREDEBUG) { DebugMsg("No lpszBuffer"); }
			continue;
		}
	
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Stringbuffer filling\n"); }
	
		strncpy(StringBuffer,(const char *)lpszBuffer,length);
	
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Freeing ram 3\n"); }
	
		LocalFree(lpszBuffer);
	
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Freeing ram 4\n"); }
	
		// Free the memory.
		if(pELR->NumStrings && Args) {
			GlobalFree(Args);
		}
	
		if(SNAREDEBUG) { DebugMsg("ExpandStrings: Freeing ram 5\n"); }
	
		if(hModule) { FreeLibrary(hModule); }
	
		returncode=TRUE;
		// We have some data! Break out of our loop. No point in investigating the other
		// locations for data.
		break;

	} while(pszExeFile2);

	if(SNAREDEBUG) { DebugMsg("ExpandStrings: Closing key\n"); }

	if(hk) { RegCloseKey(hk); }

	if(SNAREDEBUG) { DebugMsg("ExpandStrings: Freeing ram 6\n"); }

	if(pStrings) {
		// If we have been unable to push anything into the string buffer.
		if(!strlen(StringBuffer)) {
			strncpy(StringBuffer,(char *)pStrings,length);
			returncode=FALSE;
		}
		GlobalFree(pStrings);
	} else if(!strlen(StringBuffer)) {
		strcpy(StringBuffer,"Unknown");
		returncode=FALSE;
	}

	return(returncode);
}

char * GetParameterMsg(char *message, char *tmp)
{
	int		I, StringId, FileNameModuleSize = 100;
	char    *EndPtr;
	HKEY	nKeyHandle=0;
	BYTE	FileNameModule[MAX_STRING],expbuffer[MAX_STRING];
	HMODULE hModule = 0;

	char	lpBuffer[MAX_STRING], tmpStr[MAX_STRING], *DescrStr,*StartPtr;

	if(!strlen(tmp)) {
		return((char *)NULL);
	}

	if(RegOpenKey(HKEY_LOCAL_MACHINE,
				(LPTSTR) tmp,
				&nKeyHandle) != ERROR_SUCCESS) {
		return((char *)NULL);
		if(SNAREDEBUG) { DebugMsg("Could not open key in getparametermsg"); }
	}

	if (RegQueryValueEx(nKeyHandle,
				"ParameterMessageFile",
				NULL,
				NULL,
				FileNameModule,
				(unsigned long *)&FileNameModuleSize) != ERROR_SUCCESS) {
		if(SNAREDEBUG) { DebugMsg("Could not regqueryvalueex in getparametermsg"); }
		return((char *)NULL);
	}

	ExpandEnvironmentStrings(
				(const char *)FileNameModule,	// pointer to string with environment variables
				(char *)expbuffer,      		// pointer to string with expanded environment variables
				MAX_STRING);					// maximum characters in expanded string

	// RegCloseKey(HKEY_LOCAL_MACHINE);
	if(nKeyHandle) { RegCloseKey(nKeyHandle); }

	hModule=LoadLibraryEx((const char *)expbuffer,NULL,LOAD_LIBRARY_AS_DATAFILE);
	if(!hModule) {
		// return(message);
		return((char *)NULL);
	}

	EndPtr = message;
	StartPtr = message;
	*tmpStr = '\0';
	StringId = strtol(StartPtr + 2 ,&EndPtr,10);

	int csize=0;	// current size
	int ssize=0;	// target string size
	int copysize=0;	// amount to copy
	int overflow=0;	// did we overflow?

	while (StringId != 0)
	{
		I= FormatMessage(
				FORMAT_MESSAGE_FROM_HMODULE|
				FORMAT_MESSAGE_IGNORE_INSERTS,
				hModule,
				StringId,
				0,  // Default language
				(LPTSTR) &lpBuffer,
				MAX_STRING,
				(LPTSTR *) NULL
			);

		if (I== 0)
		{
			FreeLibrary(hModule);
			// return (message);
			return((char *)NULL);

		}
		
		csize=strlen(tmpStr);
		ssize=strlen((char *)lpBuffer);
		if((csize + ssize) >= sizeof(tmpStr)) {
			// Overflow. Copy what we can.
			copysize=sizeof(tmpStr) - csize;
			overflow=1;
		} else {
			copysize=csize+ssize;
		}

		strncat(tmpStr,(char *)lpBuffer,copysize);
		if(overflow) {
			StringId=0;
		} else {
			strcat(tmpStr, " ");		
			if (strstr(EndPtr, "%%") != NULL)
				StringId = strtol(strstr(EndPtr, "%%") + 2,&EndPtr,10);
			else
				StringId =0;
		}
	}

	FreeLibrary(hModule);

	// NOTE: the calling routine REALLY needs to free this string once it's grabbed. REDRED
	DescrStr = (char *)GlobalAlloc(GPTR,strlen(tmpStr) + 1);
	strcpy(DescrStr, tmpStr);

	return DescrStr;
}




void GetArgs(const EVENTLOGRECORD *pELR, char **Args)
{
	DWORD ArgCount;
	char * cpointer;

	if(!pELR || !Args) return;
	if(pELR->NumStrings == 0) return;

	cpointer = (char *)pELR + (pELR->StringOffset);

	for(ArgCount=0; ArgCount<pELR->NumStrings; ArgCount++) {
		Args[ArgCount] = cpointer;
		cpointer += strlen(cpointer) + 1;
	}
}

BOOL GetEventLogType(TCHAR *sz, unsigned short uEventType, DWORD length)
{
	if(!sz || !length)
		return FALSE;

	switch(uEventType)
	{
		case EVENTLOG_SUCCESS:
			strncpy(sz, _T("Success"), length);
			break;
		case EVENTLOG_ERROR_TYPE:
			strncpy(sz, _T("Error"), length);
			break;
		case EVENTLOG_WARNING_TYPE:
			strncpy(sz, _T("Warning"), length);
			break;
		case EVENTLOG_INFORMATION_TYPE:
			strncpy(sz, _T("Information"), length);
			break;
		case EVENTLOG_AUDIT_SUCCESS:
			strncpy(sz, _T("Success Audit"), length);
			break;
		case EVENTLOG_AUDIT_FAILURE:
			strncpy(sz, _T("Failure Audit"), length);
			break;
		default:
			strncpy(sz, _T("Unknown"), length);
			break;
	}

	return TRUE;
}

BOOL CheckLogExists(TCHAR *LogName)
{
	TCHAR szKeyName[MAX_STRING]="";
	//LPVOID lpszBuffer = 0;
	LPBYTE pSourceName=0;
	HKEY   hk = (HKEY)0;
	LPBYTE pStrings = 0;

	DWORD dwMaxString;			
	DWORD dwType; // Temporary variable.
	TCHAR szKeyValue[MAX_STRING]="";
	TCHAR szFileName[MAX_STRING]="";
			
	wsprintf(szKeyName, _T("SYSTEM\\CurrentControlSet\\Services\\EventLog\\%s"), LogName);

	if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, szKeyName, 0L, KEY_READ|KEY_SET_VALUE, &hk) != ERROR_SUCCESS) {
		if(SNAREDEBUG && LogName) { if(LogName) { DebugMsg("Cannot determine if log %s exists - openkey failed",LogName); } }
		return(FALSE);
	}

	dwMaxString=MAX_STRING-1;
	if(RegQueryValueEx(hk, _T("File"), 0, &dwType, (LPBYTE)szKeyValue, &dwMaxString) != ERROR_SUCCESS)
	{
		if(SNAREDEBUG && LogName) { DebugMsg("Cannot determine if log %s exists - queryvalueex failed",LogName); }
		RegCloseKey(hk);
		return(FALSE);
	}

	ExpandEnvironmentStrings(szKeyValue,szFileName,MAX_STRING-1);
	if(!strlen(szFileName)) {
		if(SNAREDEBUG && LogName) { DebugMsg("String just wont expand in checklogexists. Log %s probably doesnt exist",LogName); }
		RegCloseKey(hk);
		return(FALSE);
	}

	struct _stat buf;

	if(_stat(szFileName, &buf)) {
		RegCloseKey(hk);
		return(FALSE);
	}

	// Verify that log retention settings are set to 'overwrite as needed'.
	// Not sure if we need to restart the audit service if this succeeds..

	DWORD RetentionValue=0;
	DWORD RetentionSize=sizeof(RetentionValue);

	// Verify that the user is overwriting as required
	if(RegQueryValueEx(hk, _T("Retention"), 0, &dwType, (unsigned char *)&RetentionValue, &RetentionSize) != ERROR_SUCCESS)
	{
		if(SNAREDEBUG && LogName) {
			DebugMsg("Cannot check log retention settings - queryvalueex failed for log %s",LogName);
		}	
	}
	if(SNAREDEBUG) { DebugMsg("Log retention settings are set to %d for log %s",RetentionValue,LogName); }
	
	if(RetentionValue != 0) {
		RetentionValue=0;
		if(RegSetValueEx(hk, _T("Retention"), 0, REG_DWORD, (unsigned char *)&RetentionValue, sizeof(DWORD)) != ERROR_SUCCESS)
		{
			if(SNAREDEBUG && LogName) { DebugMsg("Cannot set log retention settings - regsetvalueex failed for log %s",LogName); }
		}
	}
	

	RegCloseKey(hk);
	return(TRUE);
}



BOOL GetSIDType(SID_NAME_USE _SidNameUse, TCHAR *szSIDType, DWORD length)
{
	if(!szSIDType || !length)
		return FALSE;

	szSIDType[0] = '\0';
	switch(_SidNameUse)
	{
		case SidTypeUser:
			strncpy(szSIDType,"User",length);
			break;
		case SidTypeGroup:
			strncpy(szSIDType,"Group",length);
			break;
		case SidTypeDomain:
			strncpy(szSIDType,"Domain",length);
			break;
		case SidTypeAlias:
			strncpy(szSIDType,"Alias",length);
			break;
		case SidTypeWellKnownGroup:
			strncpy(szSIDType,"Well Known Group",length);
			break;
		case SidTypeDeletedAccount:
			strncpy(szSIDType,"Deleted Account",length);
			break;
		case SidTypeInvalid:
			strncpy(szSIDType,"Invalid SID",length);
			break;
		case SidTypeUnknown:
			strncpy(szSIDType,"Unknown",length);
			break;
		case SidTypeComputer:
			strncpy(szSIDType,"Computer",length);
			break;
		default:
			strncpy(szSIDType,"Out of Type",length);
			break;
	}						

	return TRUE;
}


///////////////////////////////////////////////////////////////////
// InitWinsock
//              starts up winsock.dll or wsock32.dll
BOOL InitWinsock( char *szError )
{
	WSAData wsData;
	
	WORD wVersionRequested = WINSOCK_VERSION;
	
	if(WSAStartup(wVersionRequested, &wsData) != 0)
	{
		// :( error
		if( szError )
		{
			sprintf(szError,"WSAStartup failed: WSA ERROR: %d\r\n",
				WSAGetLastError());
			if(SNAREDEBUG) { if(szError) { DebugMsg(szError); } }
		}
		return FALSE;
	}
	
	// all is well
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////
// TerminateWinsock
//      call this function with the current socket or INVALID_SOCKET
void TerminateWinsock( SOCKET hSocket )
{
	// cancel blocking calls, if any
	WSACancelBlockingCall();
	
	// close socket
	if( hSocket != INVALID_SOCKET )
		closesocket(hSocket);
	
//	g_Info.bTerminate = TRUE;
	// allow threads to terminate, if any.
//	Sleep(5000);
	
	// unload winsock
	WSACleanup();
}


//////////////////////////////////////////////////////////////
// ConnectToServer:
//    connects to a server on a specified port number
//    returns the connected socket
SOCKET ConnectToServer(char *name, UINT nPort, char *szError)
{
	SOCKET hSocket;
	struct sockaddr_in server;
	struct hostent far *hp;
	
	
	if( !name || !*name )
		return INVALID_SOCKET;
	
	
	// Should use something nicer to check for IP addresses...
	if( isdigit(name[0]))
	{
		ZeroMemory((char *) &server, sizeof(server));
		server.sin_family      = AF_INET;
		server.sin_addr.s_addr = inet_addr(name);
		server.sin_port        = htons(nPort);
	}
	else
	{
		if ( (hp = (struct hostent far *) gethostbyname(name)) == NULL)
		{
			_snprintf(szError,MAX_STRING,"Error: gethostbyname failed: %s.",name);
			return INVALID_SOCKET;
		}
		
		ZeroMemory((char *)&server, sizeof(server));
		CopyMemory((char *) &server.sin_addr,hp->h_addr,hp->h_length);
		server.sin_family = hp->h_addrtype;
		server.sin_port = htons(nPort);
	} // else
	
	// create socket
	if( (hSocket = socket(AF_INET, SOCK_DGRAM, 0)) == INVALID_SOCKET)
	{
		_snprintf(szError,MAX_STRING,"socket failed to create datagram socket: %d\n",WSAGetLastError());
		return INVALID_SOCKET;
	}
	
	// connect to server.
	if (connect(hSocket,(struct sockaddr *)&server, sizeof(server)) == SOCKET_ERROR)
	{
		_snprintf(szError,MAX_STRING,"connect failed to connect to requested address: %d",WSAGetLastError());
		return INVALID_SOCKET;
	}
	
	return hSocket;
}


// Linked List Functions

void CreateLinkedList(void)
{
    head = currentnode = NULL;
}

int IsListEmpty(void)
{
    if (NULL == head)
        return 1;
    else
        return 0;
}



// Node * CheckObjective(int eventnumber, char *username, char *match,
// 				   char *eventlogtype, char *sourcename)
Node * CheckObjective(int eventnumber, char *username, char *match,
 				   int etype, int stype)
{
	static int firstcall=0;
	Node *currentnode;
	//int etype=0;
	//int stype=0;
	
	if(firstcall==0) {
		ResetCurrentNode();
		firstcall=1;
	}

	int usermatch=0;
	char * spoint;
	char tuser[256];
	
	if(SNAREDEBUG) { DebugMsg("CheckObjective: Inside CheckObjective"); }
	while(IsValidItem()) {
		currentnode=GetCurrentItem();
		if((eventnumber == currentnode->event_number || currentnode->event_number == AUDIT_ALL)
			&& (etype & currentnode->eventlogtype) && (stype & currentnode->sourcename)) {
			// Are we including users, or excluding.
			
			if(SNAREDEBUG) { DebugMsg("CheckObjective: Event match.. Checking user [%s] for a comma..",currentnode->username); }

			if(strstr(currentnode->username,",")) {
				if(SNAREDEBUG) { DebugMsg("CheckObjective: User list contains a comma.."); }
				spoint=currentnode->username;
				usermatch=0;
				do {
					spoint=string_split(',',spoint,tuser,sizeof(tuser));
					usermatch+=wildmatchi(tuser,username);
					if(SNAREDEBUG) { DebugMsg("CheckObjective: Split a user [%s] out of the string %s - username is %s - match is %d",tuser,currentnode->username,username,usermatch); }

				} while(spoint);
			} else {
				usermatch=wildmatchi(currentnode->username,username);
			}

			if(IS_OBJECT_ACCESS(eventnumber) && strlen(currentnode->match)>2) {
				// Windows 2000 does wierd things with filename auditing - eg: c:\temp\goo
				// becomes \Device\HarddiskDmVolumes\PhysicalDmVolumes\BlockVolume2\temp\goo
				// As such, if this is an object-event, FIRST, try stripping off the
				// VOLUME: (eg: C:) part of the match term, and add on a general raw device
				// term. If nothing comes up, drop back to standard tests.

				char tempmatch[SIZE_OF_GENERALMATCH];
				_snprintf(tempmatch,SIZE_OF_GENERALMATCH,"*\\Device\\HarddiskDmVolumes\\PhysicalDmVolumes\\BlockVolume*\\%s",&currentnode->match[4]);
				
				
				// Allowing multiple users (comma separated) may slow things down a little..


				if(currentnode->excludeflag) {
					if(!usermatch && wildmatchi(tempmatch,match)) {
						NextItemInList();
						return(currentnode);
					}
				} else {
					if(usermatch && wildmatchi(tempmatch,match)) {
						NextItemInList();
						return(currentnode);
					}
				}
			}
			
			if(currentnode->excludeflag) {
				if(!usermatch && wildmatchi(currentnode->match,match)) {
					NextItemInList();
					return(currentnode);
				}
			} else {
				if(usermatch && wildmatchi(currentnode->match,match)) {
					NextItemInList();
					return(currentnode);
				}
			}
		}
		NextItemInList();
	}

	ResetCurrentNode();
	return((Node *) NULL);
}

char * string_split(char divider,char *string,char *destination,int destlength)
{
	int destsize=0;

	while(*string && *string != divider) {
		if(destsize < destlength) {
			*destination=*string;
			destination++;
			destsize++;
		}

		string++;
	}
	*destination='\0';
	if(*string == divider) {
		string++;
	} else {
		return((char *)NULL);
	}
	return(string);
}


int ReadObjectives()
{
	Reg_Objective reg_objective;
	DWORD dw_objective_error;
	int i_objective_count=0;
	char *eventpointer,*eventpointer2;
	char eventnumber[11];
	int eventid;
	int criticality;
	int etype=0;
	int stype=0;
	int userflag=0; // include by default
	DWORD SetAudit=0;

	if(SNAREDEBUG) { DebugMsg("ReadObjectives: Inside read objectives\n"); }

	SetAudit=MyGetProfileDWORD("Config","Audit",0);

	// HERE: Turn off all auditing, unless there are NO objectives to read.
	if(SetAudit) {
		ClearAllAuditCategories();
	}
	
	while((dw_objective_error = Read_Objective_Registry(i_objective_count,&reg_objective))==0) {
		// For each event number defined.
		if(SNAREDEBUG) { DebugMsg("ReadObjectives: Read an objective\n"); }
		eventpointer=reg_objective.str_unformatted_eventid_match;
		eventpointer2=reg_objective.str_unformatted_eventid_match;

		userflag=reg_objective.dw_user_match_type;
		
		// While there are no more commas
		while(eventpointer2) {
			eventpointer2=strstr(eventpointer,",");
			if(eventpointer2 == (char *)NULL) {
				// No commas left. Just copy to the end of the line.
				strncpy(eventnumber,eventpointer,10);
				eventnumber[10]='\0'; // just in case
			} else {
				int size=9;
				if(eventpointer2-eventpointer < 10) {
					size=eventpointer2-eventpointer;
				}
				strncpy(eventnumber,eventpointer,size);
				// Make sure we have a null on the end of the line.
				eventnumber[size]='\0';
			}
			if(eventpointer2) {
				// Skip the comma
				eventpointer=eventpointer2+1;
			}
			
			if(!strcmp(eventnumber,"*")) {
				eventid=AUDIT_ALL;
			} else {
				eventid=atoi(eventnumber);
			}
			if((eventid >=0 && eventid <= 65535) || eventid==AUDIT_ALL) {
				// valid event ID. Continue.

				// HERE: Turn on the appropriate audit.

				if(!strcmp(reg_objective.str_critic,CRITICAL_TOKEN)) {
					criticality=EVENT_CRITICAL;
				} else if(!strcmp(reg_objective.str_critic,PRIORITY_TOKEN)) {
					criticality=EVENT_PRIORITY;
				} else if(!strcmp(reg_objective.str_critic,WARNING_TOKEN)) {
					criticality=EVENT_WARNING;
				} else if(!strcmp(reg_objective.str_critic,INFORMATION_TOKEN)) {
					criticality=EVENT_INFORMATION;
				} else if(!strcmp(reg_objective.str_critic,CLEAR_TOKEN)) {
					criticality=EVENT_CLEAR;
				}

				etype=reg_objective.dw_event_type;
				if(!etype) etype=TYPE_SUCCESS|TYPE_FAILURE|TYPE_INFO|TYPE_WARN|TYPE_ERROR;
				
				// Does the user want us to change the audit settings?
				// Only do this for SECURITY log events
				if(SetAudit && (reg_objective.dw_eventlog_type & LOG_SEC)) {
					if((etype & TYPE_SUCCESS) || (etype & TYPE_FAILURE)) {
						TurnOnEvent(eventid,etype);
					} else {
						// The user didn't specify whether to audit success or failures.
						// TURN ON SUCCESS AUDITING AND FAILURE AUDITING
						TurnOnEvent(eventid,TYPE_SUCCESS|TYPE_FAILURE);
					}
				}

				stype=reg_objective.dw_eventlog_type;
				if(!stype) stype=LOG_SEC|LOG_SYS|LOG_APP|LOG_DIR|LOG_DNS|LOG_REP;

				// Just in case the general match is empty.
				char tempmatch[SIZE_OF_GENERALMATCH];

				if(!strlen(reg_objective.str_general_match)) {
					// NOTE: general_match is > 1, so strcpy is safe here.
					strncpy(tempmatch,"*",sizeof(tempmatch));
				} else {
					_snprintf(tempmatch,sizeof(tempmatch),"*%s*",reg_objective.str_general_match);
				}

				AddToList(eventid, reg_objective.str_user_match,
						  tempmatch,
						  criticality, userflag, etype,
						  stype);
			}
		}
		
		i_objective_count++;
	}
	return(1);
}

BOOL ClearAllAuditCategories()
{
	SwitchAudit(AuditCategorySystem,0,0);
	SwitchAudit(AuditCategoryLogon,0,0);
	SwitchAudit(AuditCategoryObjectAccess,0,0);
	SwitchAudit(AuditCategoryPrivilegeUse,0,0);
	SwitchAudit(AuditCategoryDetailedTracking,0,0);
	SwitchAudit(AuditCategoryPolicyChange,0,0);
	SwitchAudit(AuditCategoryAccountManagement,0,0);
	SwitchAudit(AuditCategoryDirectoryServiceAccess,0,0);
	SwitchAudit(AuditCategoryAccountLogon,0,0);
	
	return(1);
}

BOOL TurnOnEvent(DWORD EventID,DWORD SuccessFailure)
{
	if(IS_PRIVILEGE_USE(EventID))
		SwitchAudit(AuditCategoryPrivilegeUse,SuccessFailure,1);
	if(IS_PROCESS_TRACKING(EventID))
		SwitchAudit(AuditCategoryDetailedTracking,SuccessFailure,1);
	if(IS_SYSTEM_EVENTS(EventID))
		SwitchAudit(AuditCategorySystem,SuccessFailure,1);
	if(IS_LOGON_EVENTS(EventID))
		SwitchAudit(AuditCategoryLogon,SuccessFailure,1);
	if(IS_ACCOUNT_LOGON_EVENTS(EventID))
		SwitchAudit(AuditCategoryAccountLogon,SuccessFailure,1);
	if(IS_ACCOUNT_MANAGEMENT_EVENTS(EventID))
		SwitchAudit(AuditCategoryAccountManagement,SuccessFailure,1);
	if(IS_OBJECT_ACCESS(EventID))
		SwitchAudit(AuditCategoryObjectAccess,SuccessFailure,1);
	if(IS_POLICY_CHANGE(EventID))
		SwitchAudit(AuditCategoryPolicyChange,SuccessFailure,1);
	if(IS_DIRECTORY_SERVICE_ACCESS(EventID))
		SwitchAudit(AuditCategoryDirectoryServiceAccess,SuccessFailure,1);

	return(1);
}

// Turn an audit category on or off
BOOL SwitchAudit(POLICY_AUDIT_EVENT_TYPE AuditCategory, DWORD SuccessFailure, BOOL Switch)
{
	// AuditCategorySystem, AuditCategoryLogon, AuditCategoryObjectAccess,
	// AuditCategoryPrivilegeUse, AuditCategoryDetailedTracking,
	// AuditCategoryPolicyChange, AuditCategoryAccountManagement,
	// AuditCategoryDirectoryServiceAccess, AuditCategoryAccountLogon

	LPWSTR wComputerName = NULL; 
    LSA_HANDLE PolicyHandle; 
    NTSTATUS Status; 
	PPOLICY_AUDIT_EVENTS_INFO AuditEvents;
	LSA_OBJECT_ATTRIBUTES ObjectAttributes; 
	DWORD Flag=0;

	ZeroMemory(&ObjectAttributes, sizeof(ObjectAttributes));

	Status = LsaOpenPolicy(NULL, &ObjectAttributes, POLICY_VIEW_AUDIT_INFORMATION | POLICY_SET_AUDIT_REQUIREMENTS, &PolicyHandle);

	if(Status == 0) { 
		Status = LsaQueryInformationPolicy(PolicyHandle, PolicyAuditEventsInformation, 
                (void **) &AuditEvents); 
 
	    if(Status != 0) return 0; 
 
	    // 
	    // successfully obtained AuditEventsInformation. 
	    // 
		// If audit is not turned on
	    if(AuditEvents->AuditingMode == 0) {
			// In theory, we need to turn on auditing.
			AuditEvents->AuditingMode = 1;
			Status = LsaSetInformationPolicy(PolicyHandle, PolicyAuditEventsInformation, 
                (PVOID) AuditEvents); 
		}

		if(SuccessFailure & TYPE_SUCCESS) {
			if(Switch==1) {
				Flag |= POLICY_AUDIT_EVENT_SUCCESS;
			}
		}

		if(SuccessFailure & TYPE_FAILURE) {
			if(Switch==1) {
				Flag |= POLICY_AUDIT_EVENT_FAILURE;
			}
		}

		// If the current settings mirror what we want, don't change anything.
		if((Flag == 0 && (AuditEvents->EventAuditingOptions[AuditCategory] &
			(POLICY_AUDIT_EVENT_SUCCESS | POLICY_AUDIT_EVENT_FAILURE)) == 0) ||
			Flag && AuditEvents->EventAuditingOptions[AuditCategory]) {
			Flag=POLICY_AUDIT_EVENT_UNCHANGED;
		} else if(Flag==0) {
			Flag=POLICY_AUDIT_EVENT_NONE;
		}
		
		if(Flag != POLICY_AUDIT_EVENT_UNCHANGED) {
			Status = SetAuditEvent(PolicyHandle,AuditCategory,Flag);
 			// 
			// enable audits 
			// 
			if( Status == 0 ) {
				Status = SetAuditMode(PolicyHandle, TRUE);
			}
		}

		LsaClose(PolicyHandle);
	}
 
	return(1);
}



int SetAuditEvent(LSA_HANDLE PolicyHandle, POLICY_AUDIT_EVENT_TYPE EventType,
					   POLICY_AUDIT_EVENT_OPTIONS EventOption)
{ 
    PPOLICY_AUDIT_EVENTS_INFO pae; 
    NTSTATUS Status; 
    DWORD i; // index into EventAuditingOptions 
 
    // 
    // obtain AuditEvents 
    // 
    Status = LsaQueryInformationPolicy( 
                PolicyHandle, 
                PolicyAuditEventsInformation, 
                (void **)&pae 
                ); 
 
    if(Status != 0) return Status; 
 
    // 
    // ensure we were passed a valid EventType and EventOption 
    // 
    if((ULONG)EventType > pae->MaximumAuditEventCount) {
		LsaFreeMemory(pae);
		if(SNAREDEBUG) { DebugMsg("Invalid eventtype."); }
        return -1; 
	}
		
	if(!(EventOption & POLICY_AUDIT_EVENT_MASK)) { 
        LsaFreeMemory(pae); 
		if(SNAREDEBUG) { DebugMsg("Invalid eventoption"); }
        return -1; 
    } 
 
    // 
    // set all auditevents to the unchanged status... 
    // 
    for(i = 0 ; i < pae->MaximumAuditEventCount ; i++) { 
        pae->EventAuditingOptions[i] = POLICY_AUDIT_EVENT_UNCHANGED; 
    } 
 
    // 
    // ...and update only the specified EventType 
    // 
    pae->EventAuditingOptions[EventType] = EventOption; 
 
    // 
    // set the new AuditEvents 
    // 
    Status = LsaSetInformationPolicy( 
                PolicyHandle, 
                PolicyAuditEventsInformation, 
                pae 
                ); 
 
    // 
    // free allocated memory 
    // 
    LsaFreeMemory(pae); 
 
    return Status; 
} 
 
int SetAuditMode(LSA_HANDLE PolicyHandle, BOOL bEnable)
{ 
    PPOLICY_AUDIT_EVENTS_INFO AuditEvents; 
    NTSTATUS Status; 
    DWORD i; 
 
    // 
    // obtain current AuditEvents 
    // 
    Status = LsaQueryInformationPolicy( 
                PolicyHandle, 
                PolicyAuditEventsInformation, 
                (void **)&AuditEvents 
                ); 
 
    if(Status != 0) return Status; 
 
    // 
    // update the relevant member 
    // 
    AuditEvents->AuditingMode = bEnable; 
 
    // 
    // set all auditevents to the unchanged status... 
    // 
    for(i = 0 ; i < AuditEvents->MaximumAuditEventCount ; i++) { 
        AuditEvents->EventAuditingOptions[i] = POLICY_AUDIT_EVENT_UNCHANGED; 
    } 
 
    // 
    // set the new auditing mode (enabled or disabled) 
    // 
    Status = LsaSetInformationPolicy( 
                PolicyHandle, 
                PolicyAuditEventsInformation, 
                AuditEvents 
                ); 
 
    LsaFreeMemory(AuditEvents); 
 
    return Status; 
} 


Node * AddToList(int eventnumber, char *username, char *match, int criticality, int excludeflag,
				 int eventlogtype, int sourcename)
{
    Node *newNode=NULL;

    newNode = (Node *) malloc(sizeof(Node));

    if (newNode == NULL)
    {
        fprintf(stderr, "AddToList(): error in dynamic memory allocation\nCould not add a new objective into our linked list. You may be low on memory.\n");
        return((Node *)NULL);
    }

	newNode->event_number=eventnumber;
	newNode->criticality=criticality;
	newNode->excludeflag=excludeflag;
	newNode->eventlogtype=eventlogtype;
	newNode->sourcename=sourcename;

	strncpy(newNode->username,username,sizeof(newNode->username));
	strncpy(newNode->match,match,sizeof(newNode->username));
	
    if (head == NULL)
    {
        head = newNode;
        newNode->next = NULL;
    }
    else
    {
        newNode->next = head;
        head = newNode;
    }

    return newNode;
}

void RemoveFromListHead(void)
{
    Node *tempPtr;

    if (NULL == head)
        return;

    tempPtr = head;
    head = head->next;

    free(tempPtr);
}

void RemoveFromList(Node* node)
{
    Node *tempPtr, *previousPtr;

    if (NULL == node)
        return;

    if (head == node)
    {
        RemoveFromListHead();
        return;
    }

    tempPtr = head;

    while (NULL != tempPtr)
    {
        previousPtr = tempPtr;
        tempPtr = tempPtr ->next;

        if (tempPtr == node)
        {
            previousPtr->next = tempPtr->next;
            free(tempPtr);
            break;
        }
    }
}

void ResetCurrentNode(void)
{
    currentnode = head;
}

int IsValidItem(void)
{
    return (NULL == currentnode) ? 0 : 1;
}

Node * GetCurrentItem()
{
    return (currentnode);
}

void NextItemInList(void)
{
    if (NULL == currentnode)
        return;

    currentnode = currentnode->next;
}

void DestroyList(void)
{
    if (NULL == head)
    {
        return;
    }

    while (NULL != head)
    {
        Node *tempPtr = head;
        head = head->next;

        free(tempPtr);
    }
}


void CSnarecoreService::OnShutdown() {
	// Audit: Set that global variable to TRUE so that the threads
	// receive the terminate message
	g_Info.bTerminate=TRUE;
	
	if(SNAREDEBUG) { DebugMsg("SNARE Shutdown request received"); }

	// Call a fake event so that the subroutine	gets the shutdown message
	// through the setting of g_Info.bTerminate
	if(m_hEventList[0])
		::SetEvent(m_hEventList[0]);
}


// Process user control requests
BOOL CSnarecoreService::OnUserControl(DWORD dwOpcode)
{
    switch (dwOpcode) {
    case SERVICE_CONTROL_USER + 0:

        // Save the current status in the registry
        SaveStatus();
        return TRUE;

    default:
        break;
    }
    return FALSE; // say not handled
}

// Save the current status in the registry
void CSnarecoreService::SaveStatus()
{
	// TODO: add code to save the sate of the service in the registry
}


// Configuration reading routines

void GetHostname(char * Hostname,int size)
{
	// Grab the fully qualified hostname
	// Note: if the user has explicitly set a hostname in the registry, use that instead.

	if(!Hostname) return;
	if(!size) return;

	Hostname[0]='\0';
	MyGetProfileString("Config","Clientname",Hostname,size);
	if(Hostname[0]=='\0') {
		GetFQDN(Hostname,size);
	}
}

void GetDestPort(DWORD * dwDestPort)
{
	if(!dwDestPort) return;
	// Where should we be sending this stuff (syslog ID)
	*dwDestPort=MyGetProfileDWORD("Network","DestPort",6161);
}

void GetSyslog(DWORD * dwSyslog)
{
	if(!dwSyslog) return;
	*dwSyslog=MyGetProfileDWORD("Network","SyslogDest",13);
}

void GetSyslogHeader(DWORD * dwSyslogHeader)
{
	if(!dwSyslogHeader) return;
	*dwSyslogHeader=MyGetProfileDWORD("Network","Syslog",0);
}

void GetWEBSERVER_ACTIVE(DWORD * WEBSERVER_ACTIVE)
{
	if(!WEBSERVER_ACTIVE) return;
	*WEBSERVER_ACTIVE=MyGetProfileDWORD("Remote","Allow",0);
}

void GetPortNumber(DWORD * dwPortNumber)
{
	if(!dwPortNumber) return;
	*dwPortNumber=MyGetProfileDWORD("Remote","WebPort",80);
}

void GetDestination(char * lpszDestination,int size)
{
	if(!lpszDestination) return;
	if(!size) return;
	
	strncpy(lpszDestination,"127.0.0.1",size);
	if(!MyGetProfileString("Network","Destination",lpszDestination,size))
	{
		// Problem. Couldn't retrieve the destination from the registry.
		// Default it to something harmless.
		MyWriteProfileString("Network","Destination",lpszDestination);
	}
}

void GetDelim(char * DELIM,int size)
{
	if(!DELIM) return;
	if(size < 2) return;

	DELIM[0]=9;	// TAB character
	DELIM[1]='\0';
	// Only use a different character for delimiters if we're sending data via syslog.
	MyGetProfileString("Config","Delimiter",DELIM,size);
}

void GetPassword(char * lpszPassword,int size)
{
	DWORD dwUsePassword;

	if(!lpszPassword) return;
	if(!size) return;

	strncpy(lpszPassword,"",size);
	dwUsePassword=MyGetProfileDWORD("Remote","AccessKey",0);
	if(dwUsePassword) {
		if(!MyGetProfileString("Remote","AccessKeySet",lpszPassword,size))
		{
			// Problem. Couldn't retrieve the destination from the registry.
			// Default it to something harmless.
			strncpy(lpszPassword,"",size);
		}
	}
}

void GetIPAddress(char * lpszIPAddress,int size)
{
	DWORD dwRestrictIP;
	if(!lpszIPAddress) return;
	if(!size) return;

	strcpy(lpszIPAddress,"");
	dwRestrictIP=MyGetProfileDWORD("Remote","Restrict",0);

	if(dwRestrictIP) {
		if(!MyGetProfileString("Remote","RestrictIP",lpszIPAddress,size))
		{
			// Problem. Couldn't retrieve the destination from the registry.
			// Default it to something harmless.
			strncpy(lpszIPAddress,"127.0.0.1",size);
		}
	}

}

void GetClearTabs(DWORD * ClearTabs)
{
	if(!ClearTabs) return;
	// If the user SPECIFICALLY does not want TABS in the output,
	// even IF the delimiter is not a TAB:
	*ClearTabs=MyGetProfileDWORD("Config","ClearTabs",0);
}

// END Configuration Reading Routines



// Duplicate of the CNTService DebugMsg
void DebugMsg(const char* pszFormat, ...)
{
	char buf[8192];
	char date[50];
	char time[50];
	
	SYSTEMTIME st;

	GetLocalTime(&st);
	GetDateFormat(LOCALE_SYSTEM_DEFAULT,0,&st,"dd'/'MM'/'yyyy",date,sizeof(date));
	GetTimeFormat(LOCALE_SYSTEM_DEFAULT,0,&st,"HH':'mm':'ss",time,sizeof(time));

	_snprintf(buf, 8192, "[SNARE](%lu - %s %s): ", GetCurrentThreadId(),date,time);
	va_list arglist;
	va_start(arglist, pszFormat);
    _vsnprintf(&buf[strlen(buf)],8192-strlen(buf)-1,pszFormat,arglist);
	va_end(arglist);
    _snprintf(buf,8192,"%s\n",buf);
    //OutputDebugString(buf);

	if(buf) { printf("%s",buf); fflush(stdout); }

#ifdef DEBUG_TO_FILE
	FILE *fp;
	if(!strlen(pszFormat)) {
		// Send a "" to truncate the file back to zero
		fp=fopen("C:\\SNAREDebug.log","wt");
	} else {
		fp=fopen("C:\\SNAREDebug.log","a");
	}
	if(fp) {
		fprintf(fp,"%s",buf);
		fclose(fp);
	}
#endif
}


