/**********************************************************
 *
 * Snare for Irix version 0.9
 *
 * Author: InterSect Alliance Pty Ltd
 *
 * Copyright 2001-2004 InterSect Alliance Pty Ltd
 *
 * Last Modified: 14/7/2004
 *
 * Available under the terms of the GNU Public Licence.
 * - See www.gnu.org
 *
 **********************************************************
 *
 * Snare for Irix is a user-space program
 * that interacts with the audit facility within the
 * Irix operating system as a satd filter.
 *
 * Snare takes data from the sat_interpret software, and sends
 * the resulting text data over the network to a remote server
 * using UDP.
 *
 **********************************************************
 *
 * History:
 *       7/07/2004  Test version - very basic capabilities.
 *       8/07/2004  Initial working version, based on
 *                  Snare for Solaris.
 *
 **********************************************************
 *
 * Compilation Instructions:
 *    cc -o snarecore snarecore.c webserver.c WebPages.c -lsocket -lnsl -lintl
 *
 **********************************************************/


#include <sys/types.h>
#include <sys/wait.h>
#include <sys/errno.h>
#include <sys/fcntl.h>
#include <sys/termios.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/sat.h>

#include <netinet/in.h>

#include <sat.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <limits.h>
#include <dirent.h>
#include <strings.h>
#include <values.h>

#include <regex.h>

#include "shared.h"
#include "snarecore.h"
#include "webserver.h"

////// DEBUG
#include <stdarg.h>
void DebugMsg(const char *pszFormat, ...);
////// END DEBUG

// Linked List Functions
static Node *head, *currentnode;
static int AuditDestination = AUDIT_TO_STDOUT;	// Send to STDOUT unless otherwise stated.

char hostid[MAX_HOSTID] = "";
int remote_allow = 0;
int remote_webport = 6161;
char remote_restrictip[16] = "";
char remote_password[256] = "";

FILE *AuditFile = (FILE *) NULL;
int AuditSocket = 0;
struct sockaddr_in AuditSocketName;

/* Define an error variable so we can communicate with auditsvc */
extern int errno;

int audit_off()
{
	// Turn IRIX auditing off here.
	flush_audit_events();
	return(0);
}

int audit_on()
{
	// Turn IRIX auditing on here.
	return (1);
}

int main(int argc, char *argv[])
{
	/* General program related variables */
	char logbuffer[LOGBUFFERSIZE];	/* Holds the data from praudit */

	/* Variables that relate to the network socket */
	// struct hostent *hp;	/* Host Pointer */
	// struct sockaddr_in socketname;
	// struct in_addr ia;

	/* Audit related variables */
	int webpid = 0;

	// NODE that matches the objective search term.
	Node *nodematch;

	char username[MAX_USERNAME];
	char searchterm[MAX_PATH];
	char eventname[MAX_EVENTNAME];
	char options[MAX_OPTIONS];
	char *startrecord;
	int returncode;

	/* The only person who can run auditsvc is root, so lets
	   make sure that we discourage other users */
	if (getuid() != 0) {
		fprintf(stderr, "This program can only be run by root.\n");
		exit(1);
	}

	// Is there any way to turn off auditing for THIS process
	// in IRIX? If so, do it here. TODO

	// Detect if auditing is actually running here.
	//if(0) {
	//	fprintf(stderr,
	//		"Auditing has not been enabled on this machine!\n");
	//	exit(1);
	//}

	// Wait for a few seconds, until the Irix setconfig() process
	// has completed.
	sleep(5);

	// Flush our audit events. Only turn on what we need to.
	flush_audit_events();

	// Read our configuration file, and set up audit.
	returncode=read_config_file(0);

	if (!hostid) {
		getfqdn(hostid);
	} else {
		if (strlen(hostid) == 0) {
			getfqdn(hostid);
		}
	}


	if (remote_allow) {
		webpid = fork();
		if (webpid == 0) {
			// Child process - Run our web server
			while (InitWebServer
			       (remote_webport, remote_restrictip,
				remote_password) == -1) {
				// Problem binding. Sleep for a moment.
				sleep(5);
			}
			while ((returncode = StartThread()) == 1) {
				// Problem binding. Keep trying.
				sleep(5);
			}
			exit(0);
		}
	}

	
	while(fgets(logbuffer,sizeof(logbuffer),stdin)) {
		startrecord=logbuffer;
		username[0]='\0';
		searchterm[0]='\0';
		eventname[0]='\0';
		options[0]='\0';
		returncode=0;

		/* This is where we implement the SNARE event checker process */
		returncode =
		    GetIrixDetails(logbuffer,
				  username,
				  searchterm,
				  eventname,
				  options);

		while ((nodematch =
			CheckObjective(username,
				       searchterm,
				       eventname,
				       options,
				       returncode))
		       != (Node *) NULL) {

			if (sendevent
			    (startrecord,
			     nodematch->
			     criticality) != 0) {
				// Report an error, but continue
				fprintf(stderr,
					"error sending log entry to destination\n");
			}
		}
	}

	DebugMsg("Snare Agent for Irix - Exiting!");

	audit_off();

	if (webpid) {
		kill(webpid, SIGTERM);
		waitpid(webpid,NULL,0);
	}

	if (AuditSocket) {
		close(AuditSocket);
	}

	DestroyList();

	/* All done */
	return 0;
}


int turn_event_on(int eventnumber)
{
	// Turn irix audit event on here.
	if(satstate(eventnumber) == 0) {
		return(saton(eventnumber));
	}
	return(0);
}

int turn_event_off(int eventnumber)
{
	// Turn irix audit event on here.
	if(satstate(eventnumber) == 1) {
		return(satoff(eventnumber));
	}
	return(0);
}


// Turn off all audit events in preparation for updating our audit policy.
int flush_audit_events()
{
	int count;

	for (count = 0; count < SAT_NTYPES; count++) {
		turn_event_off(count);
	}
	return(0);
}


// Shortread: don't re-read all the objectives - just the general configuration items.
int read_config_file(int shortread)
{
	FILE *configfile = (FILE *) NULL;
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];	// Should be enough for most config lines.
	// Would love to use gchar here instead, but need to keep this simple.
	// Config file header.
	int headertype = 0;

	// Clear some default variables.
	remote_allow = 0;
	remote_webport = 6161;
	strncpy(remote_restrictip, "", sizeof(remote_restrictip));
	strncpy(remote_password, "", sizeof(remote_password));

	// Clear the audit destination
	AuditDestination = 0;

	configfile = fopen(CONFIG_FILENAME, "r");

	if (configfile == (FILE *) NULL) {
		DebugMsg("Cannot open audit configuration file");
		return (0);
	}

	while (fgets(inputbuffer, MAX_AUDIT_CONFIG_LINE, configfile)) {

		// Kill whitespace from start and end of line.
		trim(inputbuffer);

		if(strlen(inputbuffer) < 3) {
			continue;
		}

		if (!iscomment(inputbuffer)) {
			// Is this line a header?
			if (isheader(inputbuffer)) {
				headertype = getheader(inputbuffer);
			} else {
				if (headertype == CONFIG_OBJECTIVES) {
					// Save off enough space to store the data we need.
					char event[MAX_AUDIT_CONFIG_LINE],
					    user[MAX_AUDIT_CONFIG_LINE],
					    path[MAX_AUDIT_CONFIG_LINE];
					char event2[MAX_AUDIT_CONFIG_LINE],
					    options[MAX_OPTIONS] = "";
					char *eventpointer = NULL;
					char *eventpointer2 = NULL;
					int criticality;
					int eventnumber;
					int returncode;
					int excludeflag = 0;

					// Do we need to skip over the objectives?
					if (shortread == 1) {
						continue;
					}


					if ((criticality =
					     splitobjective(inputbuffer,
							    event, user,
							    path,
							    &returncode,
							    &excludeflag))
					    > -1) {
						// add the objective to the linked list.
						trimallwhitespace(event);

						// HERE: Split event into comma separated list of events                                
						eventpointer = event;
						eventpointer2 = event;

						// While there are no more commas
						while (eventpointer2) {
							eventpointer2 =
							    strstr
							    (eventpointer,
							     ",");
							if (eventpointer2
							    ==
							    (char *) NULL)
							{
								// No commas left. Just copy to the end of the line.
								strncpy
								    (event2,
								     eventpointer,
								     MAX_AUDIT_CONFIG_LINE);
							} else {
								strncpy
								    (event2,
								     eventpointer,
								     (eventpointer2
								      -
								      eventpointer));
								// Make sure we have a null on the end of the line.
								event2
								    [eventpointer2
								     -
								     eventpointer]
								    = '\0';
							}
							if (eventpointer2) {
								// Skip the comma
								eventpointer
								    =
								    eventpointer2
								    + 1;
							}
							// Clear out any options strings
							strcpy(options,
							       "");

							if (regmatch
							    (event2,
							     "\\(.*\\)$"))
							{
								char *position;

								// Bonus, we have something. Grab it, add to options.
								// First, trim off the last bracket.
								event2
								    [strlen
								     (event2)
								     - 1] =
								    '\0';
								position =
								    strstr
								    (event2,
								     "(");

								strncpy
								    (options,
								     position
								     + 1,
								     MAX_OPTIONS);
								// truncate the event string at the bracket.
								*position =
								    '\0';
							}

							if ((eventnumber =
							     sat_strtoevent
							     (event2)) >
							    -1) {

								AddToList
								    (event2,
								     user,
								     path,
								     criticality,
								     options,
								     excludeflag,
								     returncode);

								// Also, ensure this event is turned on
								turn_event_on
								    (eventnumber);

							} else {
								if (!strcmp
								    (event,
								     "*"))
								{
									AddToList
									    ("AUDIT_ALL",
									     user,
									     path,
									     criticality,
									     "",
									     excludeflag,
									     returncode);
								}
							}
						}
					} else {
						fprintf(stderr,
							"WARNING: Cannot process objective - please ensure configuration line\n\tcontains valid criticality, event, criticality, return, user and match elements\n\t[%s]\n",
							inputbuffer);
					}
				} else if (headertype == CONFIG_OUTPUT) {
					if (isfilename(inputbuffer)) {
						if (!open_audit_output
						    (inputbuffer)) {
							fprintf(stderr,
								"WARNING: Could not open the file specified in the audit configuration: %s - Auditing to STDOUT\n",
								inputbuffer);
						}
					} else if (isnetwork(inputbuffer)) {
						if (!open_audit_network
						    (inputbuffer)) {
							fprintf(stderr,
								"WARNING: Could not open the network specified in the audit configuration: %s - Auditing to STDOUT\n",
								inputbuffer);
						}
					}
				} else if (headertype == CONFIG_HOSTID) {
					gethostident(inputbuffer, hostid,
						     MAX_HOSTID);
				} else if (headertype == CONFIG_REMOTE) {
					// Grab the remote control stuff here
					if (regmatchi
					    (inputbuffer, "^allow=")) {
						if (regmatchi
						    (inputbuffer, "=1")) {
							remote_allow = 1;
						}
					} else
					    if (regmatchi
						(inputbuffer,
						 "^listen_port=")) {
						remote_webport =
						    getport(inputbuffer);
					} else
					    if (regmatchi
						(inputbuffer,
						 "^restrict_ip=")) {
						if (!getconfstring
						    (inputbuffer,
						     remote_restrictip,
						     sizeof
						     (remote_restrictip)))
						{
							strncpy
							    (remote_restrictip,
							     "",
							     sizeof
							     (remote_restrictip));
						}
					} else
					    if (regmatchi
						(inputbuffer,
						 "^accesskey=")) {
						if (!getconfstring
						    (inputbuffer,
						     remote_password,
						     sizeof
						     (remote_password))) {
							strncpy
							    (remote_password,
							     "",
							     sizeof
							     (remote_password));
						}
					}
				} else {
					fprintf(stderr,
						"WARNING: Configuration file line does not fit in any recognised header.\n\t%s\n",
						inputbuffer);
				}
			}
		}
	}

	fclose((FILE *) configfile);
	return (1);
}

int iscomment(char *line)
{
	// Verify that there is something to check.
	if (line == (char *) NULL)
		return (1);

	// And that there is some data within.
	if (strlen(line) == 0 || strlen(line) > MAX_AUDIT_CONFIG_LINE)
		return (1);

	// If the first non-whitespace character is a hash, this is a comment line.
	while (*line) {
		// Space or tab or newline / formfeed
		if (*line == ' ' || *line == 9 || *line == 10
		    || *line == 12) {
			line++;
		} else if (*line == '#') {
			return (1);
		} else {
			// Ahh. A non-whitespace, non hash character.
			return (0);
		}
	}

	// If we are here, then the whole line must have been whitespace
	return (1);
}

int isheader(char *string)
{
	if (string[0] == '[' && string[strlen(string) - 1] == ']') {
		return (1);
	}
	return (0);
}

int isfilename(char *string)
{
	if (regmatch(string, "^file=")) {
		return (1);
	} else {
		return (0);
	}
}

// Network output
int isnetwork(char *string)
{
	if (regmatch(string, "^network=")) {
		return (1);
	} else {
		return (0);
	}
}

int open_audit_output(char *string)
{
	string += strlen("file=");
	if (strlen(string)) {
		if (regmatchi(string, "stdout")) {
			AuditDestination |= AUDIT_TO_STDOUT;
			return (1);
		}
		AuditFile = fopen(string, "a");
		if (AuditFile != (FILE *) NULL) {
			AuditDestination |= AUDIT_TO_FILE;
			return (1);
		} else {
			AuditDestination |= AUDIT_TO_STDOUT;
			return (0);
		}
	}

	return (0);
}


// Open the network - destination host and port
int open_audit_network(char *string)
{
	char desthost[MAX_HOSTID];
	int port = 6161;	// Default port
	char *pos;
	struct hostent *hp;

	// Initialise desthost.
	strncpy(desthost, "localhost", MAX_HOSTID);

	string += strlen("network=");
	if (strlen(string)) {
		if (regmatchi(string, "stdout")) {
			AuditDestination |= AUDIT_TO_STDOUT;
			return (1);
		}

		AuditSocket = socket(AF_INET, SOCK_DGRAM, 0);
		if (AuditSocket < 0) {
			fprintf(stderr,
				"Cannot open network socket - auditing to STDOUT\n");
			AuditDestination |= AUDIT_TO_STDOUT;
			return (0);
		}

		pos = strstr(string, ":");
		if (pos) {
			if ((pos - string) < MAX_HOSTID) {
				strncpy(desthost, string, (pos - string));
				desthost[pos - string] = '\0';
			} else {
				strncpy(desthost, string, MAX_HOSTID);
				desthost[MAX_HOSTID - 1] = '\0';
			}
			pos++;
			// As long as the user has a number there.
			if (strlen(pos)) {
				port = atoi(pos);
			}
		} else {
			// No colon. Assume the default port to be 6161
			strncpy(desthost, string, MAX_HOSTID);
		}

		// Just in case the user has entered a blank field
		if (strlen(desthost) == 0) {
			strncpy(desthost, "localhost", MAX_HOSTID);
		}

		hp = gethostbyname(desthost);
		if (hp == 0) {
			fprintf(stderr,
				"Cannot resolve host %s - auditing to STDOUT\n",
				hostid);
			close(AuditSocket);
			AuditDestination |= AUDIT_TO_STDOUT;
			return (0);
		}
		bcopy(hp->h_addr, &AuditSocketName.sin_addr, hp->h_length);
		AuditSocketName.sin_family = AF_INET;
		AuditSocketName.sin_port = htons(port);

		AuditDestination |= AUDIT_TO_NETWORK;
		return (1);
	}

	AuditDestination |= AUDIT_TO_STDOUT;
	return (0);
}

int getheader(char *string)
{
	char temp[256];
	char *stringp;
	strncpy(temp, string, 256);

	stringp = temp;

	// Remove the first and last bracket.
	stringp++;
	stringp[strlen(stringp) - 1] = '\0';

	if (regmatchi(stringp, "^objectives$")) {
		return (CONFIG_OBJECTIVES);
	} else if (regmatchi(stringp, "^events$")) {
		return (CONFIG_EVENTS);
	} else if (regmatchi(stringp, "^file$")
		   || regmatchi(stringp, "^output$")) {
		return (CONFIG_OUTPUT);
	} else if (regmatchi(stringp, "^hostid$")) {
		return (CONFIG_HOSTID);
	} else if (regmatchi(stringp, "^remote$")) {
		return (CONFIG_REMOTE);
	}

	fprintf(stderr, "Unknown header in configuration file: %s\n",
		stringp);
	return (0);
}

// Return the host identifier
char *getconfstring(char *string, char *file, int length)
{
	char *stringp = string;

	stringp = strstr(string, "=");
	if (stringp != (char *) NULL) {
		stringp++;
		if (strlen(stringp)) {
			strncpy(file, stringp, length - 1);
		} else {
			return ((char *) NULL);
		}
	} else {
		return ((char *) NULL);
	}
	return ((char *) file);
}

int getport(char *string)
{
	char *stringp = string;
	char strPort[10];

	stringp = strstr(string, "=");

	if (stringp != (char *) NULL) {
		stringp++;
		if (strlen(stringp)) {
			strncpy(strPort, stringp, 10);
			return (atoi((char *) strPort));
		} else {
			return (0);
		}
	} else {
		return (0);
	}
}

int geteventswitch(char *string)
{
	char *stringp = string;
	char *endname = string;

	stringp = strstr(string, "=");
	if (stringp != (char *) NULL) {
		endname = stringp;
		stringp++;
		if (!strcmp(stringp, "1")) {
			// Set this event.
			// Grab the event name. Insert a null into the string in place of the = character.
			*endname = '\0';
			return (sat_strtoevent(string));
		}
	} else {
		return (-1);
	}
	return (-1);
}

// Return the host identifier
char *gethostident(char *string, char *host, int length)
{
	char *stringp = string;

	stringp = strstr(string, "=");
	if (stringp != (char *) NULL) {
		stringp++;
		if (strlen(stringp)) {
			strncpy(host, stringp, length - 1);
		} else {
			return ((char *) NULL);
		}
	} else {
		return ((char *) NULL);
	}
	return ((char *) host);
}

// Return a string that contains the criticality
// Note that string is assumed to be MAX_AUDITREC in size.
char *criticalitystring(int criticality, char *string)
{
	switch (criticality) {
	case CRITICALITY_CRITICAL:
		return (strncpy(string, "critical", MAX_AUDITREC));
	case CRITICALITY_PRIORITY:
		return (strncpy(string, "priority", MAX_AUDITREC));
	case CRITICALITY_WARNING:
		return (strncpy(string, "warning", MAX_AUDITREC));
	case CRITICALITY_INFO:
		return (strncpy(string, "information", MAX_AUDITREC));
	default:
		return (strncpy(string, "clear", MAX_AUDITREC));
	}
}


// Remove start / end whitespace, including the newline
void trim(char *string)
{
	char *pointer;
	char *pointer2;

	// Verify that there is something to check.
	if (string == (char *) NULL)
		return;

	// And that there is some data within.
	if (strlen(string) == 0)
		return;

	// Start from the end, work backwards.
	pointer = &string[strlen(string) - 1];

	while ((*pointer == ' ' || *pointer == 9 || *pointer == 10
		|| *pointer == 12) && pointer >= string) {
		*pointer = '\0';
		pointer--;
	}

	// Are we back at the start of the string? If so, this line must be null.
	if (pointer == string)
		return;
	// Pointer is now at the last non-whitespace character of the string.

	pointer2 = string;
	while ((*pointer2 == ' ' || *pointer2 == 9 || *pointer2 == 10
		|| *pointer2 == 12) && pointer2 < pointer) {
		pointer2++;
	}
	// pointer2 will now point to the start of the first non-null character
	// Copy the truncated string back to the original.
	strcpy(string, pointer2);
}

void trimallwhitespace(char *string)
{
	char *readpointer;
	char *writepointer;

	// Verify that there is something to check.
	if (string == (char *) NULL)
		return;

	// And that there is some data within.
	if (strlen(string) == 0)
		return;

	// Start from the end, work backwards.
	readpointer = string;
	writepointer = string;
	while (readpointer < (string + strlen(string))) {
		if (*readpointer == ' ' || *readpointer == 9
		    || *readpointer == 10 || *readpointer == 12) {
			readpointer++;
		} else {
			if (writepointer != readpointer) {
				*writepointer = *readpointer;
			}
			readpointer++;
			writepointer++;
		}

	}
}

// Strip tabs and newlines out of a string.
char *strip(char *string)
{
	char *stringp = string;

	while (*stringp != '\0') {
		if (*stringp == '\t' || *stringp == '\n') {
			*stringp = ' ';
		}
		stringp++;
	}
	return (string);
}

// Take an identified objective line
// NOTE: string, event, user and match must all be the same buffer size!
int splitobjective(char *string, char *event, char *user, char *match,
		   int *returncode, int *excludeflag)
{
	char *startevent, *startuser, *startmatch, *startcrit,
	    *startreturn;
	char *stringpointer, *endstring;
	char *eventp, *userp, *matchp, *critp;
	char criticality[MAX_AUDIT_CONFIG_LINE];
	int crit = 0;

	*returncode = RETURNCODE_ANY;

	// Do some basic sanity checks.
	if (string == (char *) NULL || event == (char *) NULL
	    || user == (char *) NULL || match == (char *) NULL) {
		return (-1);
	}
	if (!strlen(string)) {
		return (-1);
	}

	startcrit = strstr(string, "criticality=");
	startevent = strstr(string, "event=");
	startreturn = strstr(string, "return=");
	startuser = strstr(string, "user=");
	if (startuser == (char *) NULL) {
		startuser = strstr(string, "user!=");
		if (startuser != (char *) NULL) {
			// EXCLUDE USERS rather than include.
			*excludeflag = 1;
		}
	}
	startmatch = strstr(string, "match=");

	// string pointers for iteration.
	stringpointer = string;
	eventp = event;
	userp = user;
	matchp = match;
	critp = criticality;

	// Pointer to the last character in the string.
	endstring = &string[strlen(string)];

	if (startevent == (char *) NULL || startuser == (char *) NULL ||
	    startmatch == (char *) NULL || startcrit == (char *) NULL
	    || startreturn == (char *) NULL) {
		// Problem, this line is malformed. We really cannot proceed with this line.
		fprintf(stderr,
			"The following line does not contain the criticality, event, return, user and match elements: [%s]",
			string);
		return (-1);
	}
	// Start with the event
	stringpointer = startcrit + strlen("criticality=");
	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startuser
		   && stringpointer != startmatch
		   && stringpointer != endstring)) {
		*critp = *stringpointer;
		critp++;
		stringpointer++;
	}
	*critp = '\0';
	trim(criticality);
	crit = atoi(criticality);
	if (crit < CRITICALITY_CLEAR)
		crit = CRITICALITY_CLEAR;
	if (crit > CRITICALITY_CRITICAL)
		crit = CRITICALITY_CRITICAL;

	// Start with the event
	stringpointer = startevent + strlen("event=");
	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startuser
		   && stringpointer != startmatch
		   && stringpointer != endstring)) {
		*eventp = *stringpointer;
		eventp++;
		stringpointer++;
	}
	*eventp = '\0';
	// Remove extra whitespace at start and end.
	trim(event);

	stringpointer = startreturn + strlen("return=");
	if (regmatchi(stringpointer, "^success")) {
		*returncode = RETURNCODE_SUCCESS;
	} else if (regmatchi(stringpointer, "^failure")) {
		*returncode = RETURNCODE_FAILURE;
	} else {
		// Any thing else is either success or failure.
		*returncode = RETURNCODE_ANY;
	}

	if (*excludeflag) {
		stringpointer = startuser + strlen("user!=");
	} else {
		stringpointer = startuser + strlen("user=");
	}

	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startmatch
		   && stringpointer != startevent
		   && stringpointer != endstring)) {
		*userp = *stringpointer;
		stringpointer++;
		userp++;
	}
	*userp = '\0';
	// Remove extra whitespace at start and end.
	trim(user);

	stringpointer = startmatch + strlen("match=");
	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startevent
		   && stringpointer != startuser
		   && stringpointer != endstring)) {
		*matchp = *stringpointer;
		stringpointer++;
		matchp++;
	}
	*matchp = '\0';
	// Remove extra whitespace at start and end.
	trim(match);

	// Return the criticality value.
	return (crit);
}


// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
int regmatch(const char *string, const char *pattern)
{
	int status;
	regex_t re;

	if (regcomp(&re, pattern, REG_EXTENDED | REG_NOSUB) != 0) {
		return (0);
	}
	status = regexec(&re, string, (size_t) 0, NULL, 0);
	regfree(&re);
	if (status != 0) {
		return (0);
	}
	return (1);
}

// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
// COMPILED REGULAR EXPRESSION version
int regmatchC(const char *string, regex_t re)
{
	int status;

	status = regexec(&re, string, (size_t) 0, NULL, 0);
	if (status != 0) {
		return (0);
	}
	return (1);
}

// Match string against an extended regular expression. Case insensitive.
// Return 1 for match, 0 for no-match or error.
int regmatchi(const char *string, const char *pattern)
{
	int status;
	regex_t re;

	if (regcomp(&re, pattern, REG_EXTENDED | REG_NOSUB | REG_ICASE) !=
	    0) {
		return (0);
	}

	status = regexec(&re, string, (size_t) 0, NULL, 0);
	regfree(&re);
	if (status != 0) {
		return (0);
	}
	return (1);
}

// Linked List Functions

void CreateLinkedList(void)
{
	head = currentnode = NULL;
}

int IsListEmpty(void)
{
	if (NULL == head)
		return 1;
	else
		return 0;
}

Node *AddToList(char *eventname, char *username, char *path,
		int criticality, char *options, int excludeflag,
		int returncode)
{
	Node *newNode = NULL;

	newNode = (Node *) malloc(sizeof(Node));

	if (newNode == NULL) {
		fprintf(stderr,
			"AddToList(): error in dynamic memory allocation\nCould not add a new objective into our linked list. You may be low on memory.\n");
		return ((Node *) NULL);
	}
	//newNode->event_number=eventnumber;
	strncpy(newNode->event_name,eventname,MAX_EVENTNAME);
	newNode->criticality = criticality;
	newNode->returncode = returncode;
	newNode->excludeflag = excludeflag;

	strncpy(newNode->username, username, MAX_USERREG);
	// REDRED: NOTE: Must free regexp in objective list at end!
	regcomp(&newNode->usernameRE, username, REG_EXTENDED | REG_NOSUB);

	strncpy(newNode->path, path, PATH_MAX);
	regcomp(&newNode->pathRE, path, REG_EXTENDED | REG_NOSUB);

	strncpy(newNode->options, options, MAX_OPTIONS);
	regcomp(&newNode->optionsRE, options,
		REG_EXTENDED | REG_NOSUB | REG_ICASE);

	if (head == NULL) {
		head = newNode;
		newNode->next = NULL;
	} else {
		newNode->next = head;
		head = newNode;
	}

	return newNode;
}

void RemoveFromListHead(void)
{
	Node *tempPtr;

	if (NULL == head)
		return;

	tempPtr = head;
	head = head->next;

	// Free regular expressions.
	regfree(&tempPtr->optionsRE);
	regfree(&tempPtr->usernameRE);
	regfree(&tempPtr->pathRE);

	free(tempPtr);
}

void RemoveFromList(Node * node)
{
	Node *tempPtr, *previousPtr;

	if (NULL == node)
		return;

	if (head == node) {
		RemoveFromListHead();
		return;
	}

	tempPtr = head;

	while (NULL != tempPtr) {
		previousPtr = tempPtr;
		tempPtr = tempPtr->next;

		if (tempPtr == node) {
			previousPtr->next = tempPtr->next;

			// Free regular expressions.
			regfree(&tempPtr->optionsRE);
			regfree(&tempPtr->usernameRE);
			regfree(&tempPtr->pathRE);

			free(tempPtr);
			break;
		}
	}
}

void ResetCurrentNode(void)
{
	currentnode = head;
}

int IsValidItem(void)
{
	return (NULL == currentnode) ? 0 : 1;
}

Node *GetCurrentItem()
{
	return (currentnode);
}

void NextItemInList(void)
{
	if (NULL == currentnode)
		return;

	currentnode = currentnode->next;
}

void DestroyList(void)
{
	if (NULL == head) {
		return;
	}

	while (NULL != head) {
		Node *tempPtr = head;
		head = head->next;

		// Free regular expressions.
		regfree(&tempPtr->optionsRE);
		regfree(&tempPtr->usernameRE);
		regfree(&tempPtr->pathRE);

		free(tempPtr);
	}
}


// Pull out the fully qualified domain name, if possible.
char *getfqdn(char *FQDN)
{
	char hname[MAX_HOSTID];
	struct hostent *hp;

	if (gethostname(hname, MAX_HOSTID)) {
		strncpy(FQDN, "localhost.unknown", MAX_HOSTID);
		return (FQDN);
	}

	strncpy(FQDN, hname, MAX_HOSTID);

	hp = gethostbyname(hname);
	if (hp) {
		while (hp->h_aliases && *hp->h_aliases) {
			if (strlen(*(hp->h_aliases)) > strlen(hname)
			    && !strncmp(hname, *(hp->h_aliases),
					strlen(hname))) {
				strncpy(FQDN, *(hp->h_aliases),
					MAX_HOSTID);
			}
			hp->h_aliases++;
		}
		if (strlen(hp->h_name) > strlen(hname)
		    && !strncmp(hname, hp->h_name, strlen(hname))) {
			strncpy(FQDN, hp->h_name, MAX_HOSTID);
		}
	}

	return (FQDN);
}

int SendToSocket(char *message)
{
	if (AuditSocket) {
		if (sendto(AuditSocket, message,
			   strlen(message), 0,
			   (const struct sockaddr *) &AuditSocketName,
			   sizeof(struct sockaddr))
		    < 0) {
			puts(message);
			return (1);
		}
	} else {
		puts(message);
		return (1);
	}
	return (0);
}


////////////////////////////////////////////////////////////////////////////////
// This routine will send an event to the user-selected output device.
////////////////////////////////////////////////////////////////////////////////
int sendevent(char *string, int criticality)
{
	char stringout[MAX_AUDITREC] = "";
	static int EventCounter=1;

	if(EventCounter >= MAXINT) {
		EventCounter=1;
	}

	// NEW audit record format, including host and IrixSAT identifier string:
	snprintf(stringout, MAX_AUDITREC, "%s\tIrixSAT\t%d\t%d\t%s", hostid,
		 criticality, EventCounter, string);

	EventCounter++;

	if (AuditDestination & AUDIT_TO_STDOUT) {
		// Send to STDOUT.
		printf("%s\n", stringout);
	}

	if (AuditDestination & AUDIT_TO_FILE) {
		// Write to file
		if (AuditFile != (FILE *) NULL) {
			fprintf(AuditFile, "%s\n", stringout);
			fflush(AuditFile);
		}
	}

	if (AuditDestination & AUDIT_TO_NETWORK) {
		if (AuditSocket) {
			SendToSocket(stringout);
		}
	}

	return (0);
}

Node *CheckObjective(char *username, char *searchterm, char *eventname,
		     char *options, int returncode)
{
	static int firstcall = 0;
	Node *currentnode;

	if (firstcall == 0) {
		ResetCurrentNode();
		firstcall = 1;
	}

	while (IsValidItem()) {
		currentnode = GetCurrentItem();

		if ((!strcmp(eventname, currentnode->event_name)
		     || !strcmp(currentnode->event_name, "AUDIT_ALL"))
		    && (returncode == currentnode->returncode
			|| currentnode->returncode == RETURNCODE_ANY)) {

			// Are we including users, or excluding.
			if (currentnode->excludeflag) {
				// REGMATCH REPLACEMENT CANDIDATE
				// if(!(regmatch(username,currentnode->username)) && regmatch(searchterm,currentnode->path)) {
				if (!
				    (regmatchC
				     (username, currentnode->usernameRE))
				    && regmatchC(searchterm,
						 currentnode->pathRE)) {
					// REGMATCH REPLACEMENT CANDIDATE
					//if(regmatchi(options,currentnode->options)) {
					if (regmatchC
					    (options,
					     currentnode->optionsRE)) {
						NextItemInList();
						return (currentnode);
					}
				}
			} else {
				// REGMATCH REPLACEMENT CANDIDATE
				// if(regmatch(username,currentnode->username) && regmatch(searchterm,currentnode->path)) {
				if (regmatchC
				    (username, currentnode->usernameRE)
				    && regmatchC(searchterm,
						 currentnode->pathRE)) {
					// REGMATCH REPLACEMENT CANDIDATE
					//if(regmatchi(options,currentnode->options)) {
					if (regmatchC
					    (options,
					     currentnode->optionsRE)) {
						NextItemInList();
						return (currentnode);
					}
				}
			}
		}
		NextItemInList();
	}

	ResetCurrentNode();
	return ((Node *) NULL);
}

// Pick out the core details from an audit event, and set the username/searchterm/eventname
// Note: Expects event to be in tabbed praudit format, not snare modified format.
int GetIrixDetails(char *logbuffer, char *username, char *searchterm,
		  char *eventname, char *options)
{
	char *position = logbuffer;
	char tempreturn[20]="";

	if(!logbuffer || !username || !searchterm || !eventname || !options) {
		return(0);
	}

	searchterm[0] = '\0';
	eventname[0] = '\0';
	username[0] = '\0';
	options[0] = '\0';

	position = logbuffer;
	if (position) {
		GetElement(position, 0, eventname, MAX_EVENTNAME);
	}

	if (!strlen(eventname)) {
		strncpy(eventname, "Unknown", MAX_EVENTNAME);
	}

	position = FindToken("UGID=",logbuffer);
	if (position) {
		GetElement(position, 0, username, MAX_USERNAME);
	}

	if (!strncmp(eventname, "sat_ae_identity",15)) {
		position = FindToken("TEXT=",logbuffer);
		if (position) {
			char *p=position;
			char *p2=position;
			int psize=0;
			// Pull the username out of the TEXT field
			p=strstr(p,"|");
			if(p) {
				p++;
				p=strstr(p,"|");
				if(p) {
					p++;
					p2=strstr(p,"|");
					if(p2) {
						psize=p2-p;
						if(psize >= MAX_USERNAME) {
							psize=MAX_USERNAME-1;
						}
						// If the username is unknown, fallback to UGID.
						if(strcmp(username,"?")!=0) {
							strncpy(username,p,psize);
							username[psize]='\0';
						}
					}
				}
			}
		}
	}

	position = FindToken("PATHNAME=",logbuffer);
	if(position) {
		char EVENTpath[MAX_PATH]="";
		char CWDpath[MAX_PATH]="";


		GetElement(position, 0, EVENTpath, sizeof(EVENTpath));
		if(*position != '/') {
			// FIND CWD, mangle them
			position = FindToken("CWD=",logbuffer);
			if(position) {
				GetElement(position,0,CWDpath,sizeof(CWDpath));
			}
		}
		resolved_path(CWDpath,EVENTpath,searchterm);

	} else {
		if (!strncmp(eventname, "sat_exit", 8) || !strncmp(eventname,"sat_exec",8)) {
			position = FindToken("COMMAND=",logbuffer);
			if(position) {
				GetElement(position,0,searchterm,MAX_PATH);
			} else {
				strncpy(searchterm,"Unknown",MAX_PATH);
			}
		} else {
			// MAYBE ADD *UID* searching here..



			// Not much else we can do - allow the entire buffer to be a search term.
			strncpy(searchterm,logbuffer,MAX_PATH);
		}
	}


	position = logbuffer;
	GetElement(position, 1, tempreturn, sizeof(tempreturn));

	if (!strcmp(tempreturn, "Success")) {
		return (RETURNCODE_SUCCESS);
	}
	return (RETURNCODE_FAILURE);
}

char *FindToken(char *token, char *buffer)
{
	char *position;

	if(!token || !buffer) {
		return((char *)NULL);
	}
	position=strstr(buffer, token);
	if(position) {
		position +=strlen(token);
	}
	return(position);
}

int GetElement(char *source, int count, char *dest, int buffersize)
{
	char *currentpos;
	char *position;
	char *end;
	char tempstring[MAX_PATH];
	int currentelement = 0;
	int copysize = 0;

	if (!source)
		return (0);
	if (!buffersize)
		return (0);

	dest[0] = '\0';

	strncpy(tempstring,source,sizeof(tempstring));
	currentpos=tempstring;

	end=strstr(currentpos," ");
	if(end) {
		*end='\0';
	}

	while (currentelement <= count) {
		position = strstr(currentpos, ",");
		if (!position) {
			if (currentelement == count) {
				// No problem, we're at the last element.
				strncpy(dest, currentpos, buffersize);
				dest[buffersize] = '\0';
				return (1);
			} else {
				// Not enough COMMAS in our input
				return (0);
			}
		} else {
			if (currentelement == count) {
				copysize = position - currentpos;
				if (copysize > (buffersize - 1)) {
					copysize = buffersize - 1;
				}
				strncpy(dest, currentpos, copysize);
				dest[copysize] = '\0';
				return (1);
			}
			currentpos = position + 1;
		}

		currentelement++;
	}
	return(0);
}


char *resolved_path(char *pwd, char *path, char *resolved)
{
        char temppath[PATH_MAX + PATH_MAX + 1];
        char *tresolved;

	if(!path || !pwd) {
		return((char *)NULL);
	}

	if(!strlen(pwd)) {
		strncpy(resolved,path,MAX_PATH);
		return(resolved);
	}
	if(!strlen(path)) {
		strncpy(resolved,pwd,MAX_PATH);
		return(resolved);
	}
                                                                                
        if (path[0] != '/') {
                snprintf(temppath, sizeof(temppath), "%s/%s", pwd, path);
                if ((tresolved =
                     realpath(temppath, resolved)) == (char *) NULL) {
                        // Cannot resolve the path - do what we can to get rid of
                        // . and .. out of the path.
                        resolved_path2(temppath, resolved);
                        return (resolved);
                } else {
                        return (tresolved);
                }
        } else {
                // Path is already absolute. return it.
                if (realpath(path, resolved) == (char *) NULL) {
                        // Could not resolve for some reason.
                        resolved_path2(path, resolved);
                        return (resolved);
                } else {
                        return (resolved);
                }
        }
}
                                                                                
                                                                                
int resolved_path2(char *path, char *resolved_path)
{
        char *new_path = resolved_path;
        char *max_path = resolved_path + MAX_PATH - 1;
                                                                                
        if (!path || !resolved_path) {
                return (0);
        }
        // Keep the first slash
        if (*path == '/') {
                *new_path++ = *path++;
        }
                                                                           
        while (*path != '\0' && (new_path < max_path)) {
                // Ignore stray "/".
                if (*path == '/') {
                        path++;
                        continue;
                }
                if (*path == '.') {
                        // Ignore ".".
                        if (path[1] == '\0' || path[1] == '/') {
                                path++;
                                continue;
                        }
                        if (path[1] == '.') {
                                if (path[2] == '\0' || path[2] == '/') {
                                        path += 2;
                                        // Ignore ".." at root.
                                        if (new_path == resolved_path + 1)
                                                continue;
                                        // Handle ".." by backing up.
                                        while ((--new_path)[-1] != '/');
                                        continue;
                                }
                        }
                }
                // Safely copy the next pathname component.
                while (*path != '\0' && *path != '/') {
                        if (new_path >= max_path) {
                                return (0);
                        }
                        *new_path++ = *path++;
                }
                if (*path == '/') {
                        *new_path++ = '/';
                }
        }
        return (0);
}











void DebugMsg(const char *pszFormat, ...)
{
	char buf[8192] = "";
	FILE *fp;

	va_list arglist;
	va_start(arglist, pszFormat);
	vsnprintf(&buf[strlen(buf)], 8192 - strlen(buf) - 1, pszFormat,
		  arglist);
	va_end(arglist);

	printf("%s\n",buf);
	fflush(stdout);

	fp = fopen("/tmp/SNARE-OUT", "a");
	fprintf(fp,"%s\n",buf);
	fclose(fp);
}
