#!/usr/bin/perl
#
# Snare Audit Dispatcher for Linux - Web Server
# (c) Copyright 2006 InterSect Alliance Pty Ltd
#

use IO::Socket;
use Net::hostent;
use Digest::MD5 qw(md5 md5_hex md5_base64);

sub WebServerSetup() {
	$port=shift;
	$accesskey=shift;
	$restrictip=shift;

	if(!$port) {
		$port=6161;
	}

	# Save off our parent process ID
	$ppid=getppid();

	$WEBCONTINUE=1;

	$SIG{QUIT}='WebTerminate';

	$OPEN_EVENTS="open,creat,link,symlink,truncate,ftruncate,mknod,rename,truncate64,ftruncate64";
	$REMOVE_EVENTS="unlink,rmdir";
	$PROCESS_EVENTS="execve";
	$ATTRIBUTE_EVENTS="chmod,chown,lchown,fchmod,fchown,fchown32,lchown32,chown32";
	$ADMIN_EVENTS="mount,umount,umount2,settimeofday,swapon,swapoff,reboot,setdomainname,create_module,delete_module,quotactl";
	$NETWORK_EVENTS="socketcall";
	
	
	$server = IO::Socket::INET->new(Proto=>'tcp',
					LocalPort=>$port,
					Listen=>SOMAXCONN,
					Reuse=>1);
	
	if(!$server) {
		LogMsg("Cannot start server on port $port\n");
		return(0);
	}

	# Select related variables
	$wrin = '';
	vec($wrin,fileno(STDIN),1) = 1;

	while($client=$server->accept()) {
		if(!$WEBCONTINUE) { last; }
		$client->autoflush(1);
		$clientclosed=0;
		$request="";
		while($trequest=<$client>) {
			if($trequest =~ /^[\r]*\n$/) {
				last;
			}
			chomp($trequest);
			$request.=$trequest . "\n";
		}

		if($restrictip) {
			$sockaddr=$client->peername();
			($port,$iaddr)=unpack_sockaddr_in($sockaddr);
			$straddr=inet_ntoa($iaddr);
			if($restrictip ne $straddr) {
				# Not an authorised source.
				HTMLHeader($client,"html");
				print $client "<html><body>";
				print $client "<center><div style=\"border: 1px solid black; background-color: #ffffee;\"><center><h2>Authentication Failed</h2>";
				print $client "Only authorised systems can connect to the Snare Server.<br>";
				print $client "</center><br></div></center>";
				print $client "</body></html>";
				Tail($client);
				next;
			}
		}
		if($accesskey) {
			ReadConfig();
			# Authentication required.
			if($request =~ /Authorization: Basic /) {
				# Get the supplied password.
				# md5_hex("blah");
				($null,$suppliedauth)=split(/Authorization: Basic /,$request);
				$suppliedauth =~ s/,.*//;
				$suppliedauth = base64decode($suppliedauth);
				($susername,$spassword)=split(/:/,$suppliedauth);
				$md5password=md5_hex($spassword);
				if(uc($susername) ne "SNARE" || $md5password ne $Config{"REMOTE"}{"ACCESSKEY"}) {
					# Nope!
					print $client "HTTP/1.0 401 Unauthorized\r\n" .
						"Connection: close\r\n" .
						"Content-Type: text/html\r\n" .
						"Server: SNARE\r\n" .
						"WWW-Authenticate: Basic realm=\"SNARE\"\r\n\r\n" .
					 	"<html><body><center><div style=\"border: 1px solid black; background: #ffeeee; width: 80%;\"><font color=\"red\">Unauthorised</font><br><br>The username and/or password you have supplied does not match the one specified in the Snare configuration file.</div></center></body></html>\r\n";
					close($client);
					$clientclosed=1;
				}
				
			} else {
				print $client "HTTP/1.0 401 Unauthorized\r\n" .
					"Connection: close\r\n" .
					"Content-Type: text/html\r\n" .
					"Server: SNARE\r\n" .
					"WWW-Authenticate: Basic realm=\"SNARE\"\r\n\r\n" .
					 "<html><body><center><div style=\"border: 1px solid black; background: #ffeeee; width: 80%;\"><font color=\"red\">Unauthorised</font><br><br>The username and/or password you have supplied does not match the one specified in the Snare configuration file.</div></center></body></html>\r\n";
				close($client);
				$clientclosed=1;
				next;
			}
		}
		
		if(!$WEBCONTINUE) { last; }
		if($request =~ /^GET/i) {
			($proto,$url,$version)=split(/ /,$request);
			$rc=HandleWebPages($url,$client);
			if($rc == -1) {
				$WEBCONTINUE=0;
			}

			close($client);
			$clientclosed=1;
			next;
		}
	}

	if(!$clientclosed) {
		close($client);
	}
	
	close($server);
}

sub HandleWebPages() {
	my($url,$theclient)=@_;
	if($url =~ /^\/Version$/) {
		HTMLHeader($theclient,"text");
		print $theclient "SNARE for Linux";
	} elsif($url =~ /^\/intersect\.gif$/) {
		InterSectImage($theclient);
	} elsif($url =~ /^\/Clear\.gif$/) {
		ClearImage($theclient);
	} elsif($url =~ /^\/Green\.gif$/) {
		GreenImage($theclient);
	} elsif($url =~ /^\/Yellow\.gif$/) {
		YellowImage($theclient);
	} elsif($url =~ /^\/Orange\.gif$/) {
		OrangeImage($theclient);
	} elsif($url =~ /^\/Red\.gif$/) {
		RedImage($theclient);
	} elsif($url =~ /^\/restart/) {
		HTMLHeader($theclient,"html");
		Head($theclient,"Snare Agent for Linux");
		SnareHead($theclient);
		print $theclient "<center><h1>Restarting Agent</h1></center>";
		print $theclient "<script type=\"text/javascript\">\n";
                print $theclient "function Reload() {\n";
                print $theclient "    window.location.href='/';\n";
                print $theclient "}\n";
                # 10 seconds
                print $theclient "mytimeout=setTimeout('Reload()',10000);\n";
                print $theclient "</script>\n";
		print $theclient "<p><br><center><font color=\"green\">Restarting Snare....</font></center>";
		SnareTail($theclient);
		Tail($theclient);

		# Send sigusr1.
		$scount=kill(10,$ppid);
		return(-1);
	} elsif($url =~ /^\/(set)*general/) {
		ReadConfig();

		HTMLHeader($theclient,"html");
		Head($theclient,"Snare Agent for Linux");
		SnareHead($theclient);
		print $theclient "<center><h1>General Configuration</h1></center>";

		if($url =~ /^\/setgeneral/) {
			# Reconstruct our general config
			delete($Config{"OUTPUT"}{"NETWORK"});
			%Arguments=GetArguments($url);
			$count=0;
			while(1) {
				if(length($Arguments{"NetworkDestination-$count"}) != 0) {
					$port=$Arguments{"NetworkDestPort-$count"};
					if(!$port) { $port=6161; }
					$Config{"OUTPUT"}{"NETWORK"}{$count}=$Arguments{"NetworkDestination-$count"} . ":" . $port;
				} else {
					last;
				}
				$count++;
			}
			delete($Config{"CONFIG"});

			$Config{"CONFIG"}{"USE_CRITICALITY"}=$Arguments{"NetworkUseCrit"};
			$Config{"CONFIG"}{"SET_AUDIT"}=$Arguments{"NetworkSetAudit"};
			$Config{"CONFIG"}{"SYSLOG_FACILITY"}=$Arguments{"NetworkSyslogFacility"};
			$Config{"CONFIG"}{"SYSLOG_PRIORITY"}=$Arguments{"NetworkSyslogPriority"};

			# Save the config here.
			$rc=WriteConfig();
			if($rc) {
				print $theclient "<div align=center><div style=\"border: 1px solid black; background: #ccffcc; padding: 5px;\">General configuration has been saved.</div></div><br><br>\n";
			} else {
				 print $theclient "<div align=center><div style=\"border: 1px solid black; background: #ffcccc; padding: 5px;\">General configuration has not been saved - I could not write to the snare configuration file.</div></div><br><br>\n";
			}
		}

		print $theclient "<form action=setgeneral>\n";

		print $theclient "<center><table style=\"border: 1px solid black;\" width=\"70%\" border=0>\n";
		$count=0;
		foreach $element (keys(%{$Config{"OUTPUT"}{"NETWORK"}})) {
			($dest,$port)=split(/:/,$Config{"OUTPUT"}{"NETWORK"}{$element});
			if(!$port) { $port=6161; }
			print $theclient "<tr style=\"background: #ffffaa;\"><td>" . ($count?"":"Destination Server Address and Port") . "</td><td><div align=center><input type=text name=NetworkDestination-$count size=25 value=\"$dest\"></div></td><td>";
			print $theclient "<select name=\"NetworkDestPort-$count\" size=1>";
			print $theclient "<option value=6161" . ($port!=514?" selected":"") . ">Snare - Port 6161</option>";
			print $theclient "<option value=514 " . ($port==514?" selected ":"") . "style=\"color: green\">Syslog - Port 514</option>";
			print $theclient "</select></div></td></tr>\n";
			$count++;
		}
		print $theclient "<tr style=\"background: #ffffaa;\"><td>" . ($count?"":"Destination Server Address and Port") . "</td><td><div align=center><input type=text name=NetworkDestination-$count size=25 value=\"\"></div></td><td>";
		print $theclient "<div align=center><select name=\"NetworkDestPort-$count\" size=1>";
		print $theclient "<option value=6161" . ($port!=514?" selected":"") . ">Snare - Port 6161</option>";
		print $theclient "<option value=514 " . ($port==514?" selected ":"") . "style=\"color: green\">Syslog - Port 514</option>";
		print $theclient "</select></div></td></tr>\n";
		print $theclient "<tr style=\"background: #ffffcc;\"><td>Perform a scan of ALL objectives, and display the maximum criticality?<br><i>Slower event checks</i></td><td colspan=2><input type=checkbox name=NetworkUseCrit value=1" . ($Config{"CONFIG"}{"USE_CRITICALITY"}==1?" checked":"") . "></input></td></tr>\n";
		print $theclient "<tr style=\"background: #ffffaa;\"><td>Allow SNARE to automatically set audit configuration</i></td><td colspan=2><input type=checkbox name=NetworkSetAudit value=1" . ($Config{"CONFIG"}{"SET_AUDIT"}==1?" checked":"") . "></input></td></tr>\n";
		print $theclient "<tr style=\"background: #ffffcc;\"><td>Syslog destination (if enabled)</td><td>";

		$facility=$Config{"CONFIG"}{"SYSLOG_FACILITY"};
		$priority=$Config{"CONFIG"}{"SYSLOG_PRIORITY"};

		print $theclient "<div align=center><select name=\"NetworkSyslogFacility\" size=1>";
		print $theclient "<option value=\"kernel\"" . ($facility eq "kernel"?" selected":"") . ">Kernel</option>";
		print $theclient "<option value=\"user\"" . ($facility eq "user"?" selected":"") . ">User</option>";
		print $theclient "<option value=\"mail\"" . ($facility eq "mail"?" selected":"") . ">Mail</option>";
		print $theclient "<option value=\"daemon\"" . ($facility eq "daemon"?" selected":"") . ">Daemon</option>";
		print $theclient "<option value=\"auth\"" . ($facility eq "auth"?" selected":"") . ">Auth</option>";
		print $theclient "<option value=\"syslog\"" . ($facility eq "syslog"?" selected":"") . ">Syslog</option>";
		print $theclient "<option value=\"lpr\"" . ($facility eq "lpr"?" selected":"") . ">Lpr</option>";
		print $theclient "<option value=\"news\"" . ($facility eq "news"?" selected":"") . ">News</option>";
		print $theclient "<option value=\"uucp\"" . ($facility eq "uucp"?" selected":"") . ">UUCP</option>";
		print $theclient "<option value=\"cron\"" . ($facility eq "cron"?" selected":"") . ">Cron</option>";
		print $theclient "<option value=\"authpriv\"" . ($facility eq "authpriv"?" selected":"") . ">Authpriv</option>";
		print $theclient "<option value=\"ftp\"" . ($facility eq "ftp"?" selected":"") . ">Ftp</option>";
		print $theclient "<option value=\"local0\"" . ($facility eq "local0" || !$facility?" selected":"") . ">Local0</option>";
		print $theclient "<option value=\"local1\"" . ($facility eq "local1"?" selected":"") . ">Local1</option>";
		print $theclient "<option value=\"local2\"" . ($facility eq "local2"?" selected":"") . ">Local2</option>";
		print $theclient "<option value=\"local3\"" . ($facility eq "local3"?" selected":"") . ">Local3</option>";
		print $theclient "<option value=\"local4\"" . ($facility eq "local4"?" selected":"") . ">Local4</option>";
		print $theclient "<option value=\"local5\"" . ($facility eq "local5"?" selected":"") . ">Local5</option>";
		print $theclient "<option value=\"local6\"" . ($facility eq "local6"?" selected":"") . ">Local6</option>";
		print $theclient "<option value=\"local7\"" . ($facility eq "local7"?" selected":"") . ">Local7</option>";
		print $theclient "</select></div>\n";
		print $theclient "</td><td>";
		print $theclient "<div align=center><select name=\"NetworkSyslogPriority\" size=1>";
		print $theclient "<option value=\"emergency\"" . ($priority eq "emergency"?" selected":"") . ">Emergency</option>";
		print $theclient "<option value=\"alert\"" . ($priority eq "alert"?" selected":"") . ">Alert</option>";
		print $theclient "<option value=\"critical\"" . ($priority eq "critical"?" selected":"") . ">Critical</option>";
		print $theclient "<option value=\"error\"" . ($priority eq "error"?" selected":"") . ">Error</option>";
		print $theclient "<option value=\"warning\"" . ($priority eq "warning"?" selected":"") . ">Warning</option>";
		print $theclient "<option value=\"notice\"" . ($priority eq "notice"?" selected":"") . ">Notice</option>";
		print $theclient "<option value=\"information\"" . ($priority eq "information"||!$priority?" selected":"") . ">Information</option>";
		print $theclient "<option value=\"debug\"" . ($priority eq "debug"?" selected":"") . ">Debug</option>";
		print $theclient "</select></div>\n";
		print $theclient "</td></tr>\n";
		print $theclient "</table></center><br>\n";
		print $theclient "<div align=center><input type=submit value=\"Change Configuration\"></div>\n";
		print $theclient "</form>\n";

		SnareTail($theclient);
		Tail($theclient);
	} elsif($url =~ /^\/(set)*remote/) {
		ReadConfig();
		HTMLHeader($theclient,"html");
		Head($theclient,"Snare Agent for Linux");
		SnareHead($theclient);
		print $theclient "<center><h1>Remote Control</h1></center>";

		if($url =~ /^\/setremote/) {
			# Reconstruct our general config
			$oldPassword=$Config{"REMOTE"}{"ACCESSKEY"};

			delete($Config{"REMOTE"});
			%Arguments=GetArguments($url);
			if($Arguments{"RemoteActive"} =~ /^[0-9]$/) {
				$Config{"REMOTE"}{"ALLOW"}=$Arguments{"RemoteActive"};
			} elsif(!$Arguments{"RemoteActive"}) {
				$Config{"REMOTE"}{"ALLOW"}="0";
			} else {
				$Config{"REMOTE"}{"ALLOW"}="1";
			}
			if($Arguments{"RemotePort"} =~ /^[0-9]+$/) {
				$Config{"REMOTE"}{"LISTEN_PORT"}=$Arguments{"RemotePort"};
			} else {
				$Config{"REMOTE"}{"LISTEN_PORT"}="6161";
			}
			if($Arguments{"RemoteRestrictIP"} =~ /^[0-9a-zA-Z\.]+$/) {
				$Config{"REMOTE"}{"RESTRICTIP"}=$Arguments{"RemoteRestrictIP"};
			}
			if(length($Arguments{"RemotePassword"})) {
				if($Arguments{"RemotePassword"} ne $oldPassword) {
					$Config{"REMOTE"}{"ACCESSKEY"}=md5_hex($Arguments{"RemotePassword"});
				} else {
					$Config{"REMOTE"}{"ACCESSKEY"}=$Arguments{"RemotePassword"};
				}
			}
			# Save the config here.
			$rc=WriteConfig();
			if($rc) {
				print $theclient "<div align=center><div style=\"border: 1px solid black; background: #ccffcc; padding: 5px;\">Remote configuration has been saved.</div></div><br><br>\n";
			} else {
				 print $theclient "<div align=center><div style=\"border: 1px solid black; background: #ffcccc; padding: 5px;\">Remote configuration has not been saved - I could not write to the snare configuration file.</div></div><br><br>\n";
			}
		}

		print $theclient "<form action=setremote>\n";
		print $theclient "<center><table style=\"border: 1px solid black;\">";
		print $theclient "<tr style=\"background: #ffffcc;\"><td>Allow the agent to be controlled remotely?</td>";
		print $theclient "<td><input type=checkbox name=RemoteActive value=1" . ($Config{"REMOTE"}{"ALLOW"}?" checked ":"") . "></td></tr>";
		print $theclient "<tr style=\"background: #ffffaa;\"><td>Listen on port:</td>";
		print $theclient "<td><input type=text name=RemotePort size=5 value=\"" . $Config{"REMOTE"}{"LISTEN_PORT"} . "\"></td></tr>";
		print $theclient "<tr style=\"background: #ffffcc;\"><td>Only respond to requests from the IP address:</td>";
		print $theclient "<td><input type=text name=RemoteRestrictIP value=\"" . $Config{"REMOTE"}{"RESTRICTIP"} . "\"></td></tr>";
		print $theclient "<tr style=\"background: #ffffaa;\"><td>Require a password to connect:</td>";
		print $theclient "<td><input type=password name=RemotePassword value=\"" . $Config{"REMOTE"}{"ACCESSKEY"} . "\"></td></tr>";
		print $theclient "</table></center>\n";
		print $theclient "<div align=center><input type=submit value=\"Change Configuration\"></div>\n";
		print $theclient "</form>\n";

		SnareTail($theclient);
		Tail($theclient);
	} elsif($url =~ /^\/(set)*objective/) {
		ReadConfig();
		HTMLHeader($theclient,"html");
		Head($theclient,"Snare Agent for Linux");
		SnareHead($theclient);
		print $theclient "<center><h1>Objective Control</h1></center>";
		# Maybe some javascript here to 'suggest' appropriate
		# settings based on the users match token?
		# eg: if process, then match = *,ls
		if($url =~ /^\/setobjective/) {
			# Save off our configuration.
			%Arguments=GetArguments($url);
			if(defined($Arguments{"RemoveObjective"})) {
				$count=$Arguments{"RemoveObjective"};
				delete $Config{"OBJECTIVES"}{$count};
			} else {
				delete $Config{"OBJECTIVES"};
				$count=0;
				$continue=1;
				while($continue) {
					$ObjectiveLine="";
					if($Arguments{"ObjCriticality-$count"} eq "") {
						$continue=0;
						last;
					}
					$criticality=$Arguments{"ObjCriticality-$count"};
					$eventselect=$Arguments{"ObjEventSelect-$count"};
					$events=$Arguments{"ObjEvents-$count"};
					$return=$Arguments{"ObjReturn-$count"};
					$userselect=$Arguments{"ObjUserSelect-$count"};
					$users=$Arguments{"ObjUser-$count"};
	
					if($criticality < 0 || $criticality > 4) {
						$criticality=0;
					}
	
					$ObjectiveLine="	criticality=$criticality";
	
					if($eventselect =~ /^OPEN_EVENTS$/) {
						$events=$OPEN_EVENTS;
					} elsif($eventselect =~ /^PROCESS_EVENTS$/) {
						$events=$PROCESS_EVENTS;
					} elsif($eventselect =~ /^REMOVE_EVENTS$/) {
						$events=$REMOVE_EVENTS;
					} elsif($eventselect =~ /^ATTRIBUTE_EVENTS$/) {
						$events=$ATTRIBUTE_EVENTS;
					} elsif($eventselect =~ /^ADMIN_EVENTS$/) {
						$events=$ADMIN_EVENTS;
					} elsif($eventselect =~ /^NETWORK_EVENTS$/) {
						$events=$NETWORK_EVENTS;
					} elsif($eventselect =~ /^OTHER_EVENTS$/) {
						# Does the events information look to be valid?
						if($events !~ /^[a-z0-9,]+$/) {
							# No. Fall back to process events.
							print $theclient "<div align=center><div style=\"border: 1px solid black; background: #ccffcc; padding: 5px;\">Event list in objective $count was not valid. Using Process Events instead.</div></div><br><br>\n";
							$events=$PROCESS_EVENTS;
						}
					}
	
					if($events =~ /,/) {
						$ObjectiveLine .= "	event=(" . $events . ")";
					} else {
						$ObjectiveLine .= "	event=$events";
					}
	
					if($return =~ /^SUCCESS$/) {
						$ObjectiveLine .= "	return=*,yes";
					} elsif($return =~ /^FAILURE$/) {
						$ObjectiveLine .= "	return=*,no";
					}
	
					$cmp="";
					if($userselect =~ /^INCLUDE$/) {
						$cmp="=";
					} elsif($userselect =~ /^EXCLUDE$/) {
						$cmp="!=";
					}
					if($cmp) {
						if($users =~ /,/) {
							$ObjectiveLine .= "	uid" . $cmp . "*,($users)";
						} else {
							$ObjectiveLine .= "	uid" . $cmp . "*,$users";
						}
					}
	
					# Now, for any of the other matches.
					$mcount=0;
					while(($ObjMatchToken=$Arguments{"ObjMatchToken:$mcount-$count"})) {
						$ObjMatchCMP=$Arguments{"ObjMatchCMP:$mcount-$count"};
						$ObjMatch=$Arguments{"ObjMatch:$mcount-$count"};
	
	
						if($ObjMatchToken && $ObjMatch && $ObjMatchCMP) {
							$ObjectiveLine .= "	$ObjMatchToken" . $ObjMatchCMP . "$ObjMatch";
						}
						$mcount++;
					}
	
					$Config{"OBJECTIVES"}{$count}=$ObjectiveLine;
					$count++;
				}
			}
			$rc=WriteConfig();
			if($rc) {
				print $theclient "<div align=center><div style=\"border: 1px solid black; background: #ccffcc; padding: 5px;\">Objective configuration has been saved.</div></div><br><br>\n";
			} else {
				 print $theclient "<div align=center><div style=\"border: 1px solid black; background: #ffcccc; padding: 5px;\">Objective configuration has not been saved - I could not write to the snare configuration file.</div></div><br><br>\n";
			}
		}
		if($url =~ /\?NISPOM=1$/) {
			delete($Config{"OBJECTIVES"});
			$Config{"OBJECTIVES"}{0}="	criticality=3	event=$PROCESS_EVENTS	uid=*,root";
			$Config{"OBJECTIVES"}{1}="	criticality=3	event=($ADMIN_EVENTS)	uid=*,root";
			$Config{"OBJECTIVES"}{2}="	criticality=0	event=($OPEN_EVENTS)	name=/etc/*";
			$Config{"OBJECTIVES"}{3}="	criticality=0	event=$PROCESS_EVENTS	exe=*passwd*";
			$rc=WriteConfig();
		} elsif($url =~ /\?SOX=1$/) {
			delete($Config{"OBJECTIVES"});
			$Config{"OBJECTIVES"}{0}="	criticality=0	event=$PROCESS_EVENTS	uid=*,root";
			$Config{"OBJECTIVES"}{1}="	criticality=3	event=($ADMIN_EVENTS)	uid=*,root";
			$Config{"OBJECTIVES"}{2}="	criticality=0	event=($OPEN_EVENTS)	name=/etc/*";
			$Config{"OBJECTIVES"}{3}="	criticality=0	event=$PROCESS_EVENTS	exe=*passwd*";
			$Config{"OBJECTIVES"}{4}="	criticality=0	event=$NETWORK_EVENTS";
			$rc=WriteConfig();
		} elsif($url  =~ /\?PCI=1$/) {
			delete($Config{"OBJECTIVES"});
			$Config{"OBJECTIVES"}{0}="	criticality=3	event=$PROCESS_EVENTS	uid=*,root";
			$Config{"OBJECTIVES"}{1}="	criticality=3	event=($ADMIN_EVENTS)	uid=*,root";
			$Config{"OBJECTIVES"}{2}="	criticality=0	event=($OPEN_EVENTS)	name=/etc/*";
			$Config{"OBJECTIVES"}{3}="	criticality=0	event=$PROCESS_EVENTS	exe=*passwd*";
			$Config{"OBJECTIVES"}{4}="	criticality=0	event=$NETWORK_EVENTS";
			$rc=WriteConfig();
		}
		print $theclient "<form action=setobjective>\n";

		print $theclient "<center><table style=\"border: 1px solid black;\">";
		print $theclient "<tr style=\"background: #ffffcc;\"><th>Criticality</th><th>Events</th><th>Return</th><th>User</th><th>Match</th><th>Add / Remove</th></tr>\n";
		$toggle=0;
		$count=0;
		@arraycheck=keys(%{$Config{"OBJECTIVES"}});
		$arraysize=@arraycheck;
		if($url =~ /\WAddObjective=1\b/) {
			# Add a new element to the array.
			$Config{"OBJECTIVES"}{$arraysize}="	criticality=0	event=$PROCESS_EVENTS";
			$AddObjective=1;
		} else {
			$AddObjective=0;
		}

		foreach $element (sort(keys(%{$Config{"OBJECTIVES"}}))) {
			@elements=split(/[ \t]*([a-z0-9:]+)(\!*=)([^ \t]+)[ \t]*/,$Config{"OBJECTIVES"}{$element});
			$type=0;
			%ObjectiveMatch=();
			foreach $component (@elements) {
				if($component =~ /^[ \t]*$/ || $component eq "") {
					next;
				}
				if($type==0) {
					$token=$component;
					$type++;
				} elsif($type == 1) {
					$comp=$component;
					$type++;
				} elsif($type == 2) {
					$match=$component;
					$ObjectiveMatch{$token}=$match;
					$ObjectiveCompare{$token}=$comp;
					$type=0;
				}
			}
			# Ok, we now have a match term, and comparison (=/!=) for each token.

			$crit=$ObjectiveMatch{"criticality"};
			delete $ObjectiveMatch{"criticality"};
			delete $ObjectiveCompare{"criticality"};

			$events=$ObjectiveMatch{"event"};
			$events =~ s/[\(\)]+//g;
			delete $ObjectiveMatch{"event"};
			delete $ObjectiveCompare{"event"};

			if($count) {
				print $theclient "<tr style=\"height: 5px;\"><td colspan=5></td></tr>\n";
			}
			if($toggle) {
				$toggle=0;
				print $theclient "<tr style=\"background: #ffffcc\" valign=top>";
			} else {
				$toggle=1;
				print $theclient "<tr style=\"background: #ffffaa\" valign=top>";
			}

			print $theclient "<td><select name=\"ObjCriticality-$count\" size=1 style=\"background: #eeeeff;\">";
			print $theclient "<option value=0" . ($crit lt 1 || $crit gt 4?" selected ":"") . ">0 - clear</option>";
			print $theclient "<option value=1 style=\"background: #aaffaa;\"" . ($crit eq 1?" selected ":"") . ">1 - green</option>";
			print $theclient "<option value=2 style=\"background: #ffffaa;\"" . ($crit eq 2?" selected ":"") . ">2 - yellow</option>";
			print $theclient "<option value=3 style=\"background: #ffaa66;\"" . ($crit eq 3?" selected ":"") . ">3 - orange</option>";
			print $theclient "<option value=4 style=\"background: #ffaaaa;\"" . ($crit eq 4?" selected ":"") . ">4 - red</option>";
			print $theclient "</select></td>";
			print $theclient "<td>";
			print $theclient "<select name=\"ObjEventSelect-$count\" size=1 style=\"background: #eeeeff;\">";
			$selected=0;
			if($events eq $OPEN_EVENTS) { $selected=1; }
			print $theclient "<option value=\"OPEN_EVENTS\"" . ($selected==1?" selected ":"") . " onclick='document.getElementById(\"ObjEvents-$count\").value=\"$OPEN_EVENTS\";'>Read/Write a File/Directory</option>";
			if($events eq $REMOVE_EVENTS) { $selected=2; }
			print $theclient "<option value=\"REMOVE_EVENTS\"" . ($selected==2?" selected ":"") . " onclick='document.getElementById(\"ObjEvents-$count\").value=\"$REMOVE_EVENTS\";'>Remove a File/Directory</option>";
			if($events eq $ATTRIBUTE_EVENTS) { $selected=3; }
			print $theclient "<option value=\"ATTRIBUTE_EVENTS\"" . ($selected==3?" selected ":"") . " onclick='document.getElementById(\"ObjEvents-$count\").value=\"$ATTRIBUTE_EVENTS\";'>Modify File/Directory Attributes</option>";
			if($events eq $PROCESS_EVENTS) { $selected=4; }
			print $theclient "<option value=\"PROCESS_EVENTS\"" . ($selected==4?" selected ":"") . " onclick='document.getElementById(\"ObjEvents-$count\").value=\"$PROCESS_EVENTS\";'>Start/Stop Program</option>";
			if($events eq $ADMIN_EVENTS) { $selected=5; }
			print $theclient "<option value=\"ADMIN_EVENTS\"" . ($selected==5?" selected ":"") . " onclick='document.getElementById(\"ObjEvents-$count\").value=\"$ADMIN_EVENTS\";'>Administrative Events</option>";
			if($events eq $NETWORK_EVENTS) { $selected=6; }
			print $theclient "<option value=\"NETWORK_EVENTS\"" . ($selected==6?" selected ":"") . " onclick='document.getElementById(\"ObjEvents-$count\").value=\"$NETWORK_EVENTS\";'>Network Socketcall Events</option>";

			if(!$selected) {
				print $theclient "<option value=\"OTHER_EVENTS\" selected>Other Events (described below)</option>";
			} else {
				print $theclient "<option value=\"OTHER_EVENTS\">Other Events (described below)</option>";
			}
			print $theclient "</select><br>\n";
			print $theclient "<input type=text id=\"ObjEvents-$count\" name=\"ObjEvents-$count\" size=29 value=\"$events\" style=\"background: #eeeeff;\">";
			print $theclient "</td>";


			print $theclient "<td>\n<select name=\"ObjReturn-$count\" size=1 style=\"background: #eeeeff;\">";
			print $theclient "<option value=ANY style=\"background: #ffffaa;\"" . ($ObjectiveMatch{"return"} !~ /^\*,(yes|no)$/?" selected":"") . ">Any</option>\n";
			print $theclient "<option value=SUCCESS style=\"background: #aaffaa;\"" . ($ObjectiveMatch{"return"} =~ /^\*,yes$/?" selected":"")  . ">Success</option>\n";
			print $theclient "<option value=FAILURE style=\"background: #ffaaaa;\"" . ($ObjectiveMatch{"return"} =~ /^\*,no/?" selected":"") . ">Failure</option>\n";
			print $theclient "</select>\n</td>";

			print $theclient "<td>";
			delete $ObjectiveMatch{"return"};
			delete $ObjectiveCompare{"return"};


			$user=$ObjectiveMatch{"uid"};
			$userincexc=$ObjectiveCompare{"uid"};
			$user=~s/[\(\)]+//g;
			$user =~ s/^\*,//;
			delete $ObjectiveMatch{"uid"};
			delete $ObjectiveCompare{"uid"};

			print $theclient "<select name=\"ObjUserSelect-$count\" size=1 style=\"background: #eeeeff;\">";
			print $theclient "<option value=\"ANY\" onclick='document.getElementById(\"ObjUser-$count\").value=\"\";'" . ($userincexc !~ /^!*=$/?" selected ":"") . ">Any User</option>";
			print $theclient "<option value=\"INCLUDE\"" . ($userincexc eq "="?" selected ":"") . ">Include Users</option>";
			print $theclient "<option value=\"EXCLUDE\"" . ($userincexc eq "!="?" selected ":"") . ">Exclude Users</option>";
			print $theclient "</select><br>\n";
			print $theclient "<input type=text id=\"ObjUser-$count\" name=\"ObjUser-$count\" size=14 value=\"$user\" style=\"background: #eeeeff;\">";
			print $theclient "</td>";
			print $theclient "<td>";

			# For each of the match terms specified by the user.
			# Token/value
			$mcount=0;
			foreach $matchtype (keys(%ObjectiveMatch)) {
				$token=$matchtype;
				$value=$ObjectiveMatch{$matchtype};
				$cmp=$ObjectiveCompare{$matchtype};
				print $theclient "<table border=0><tr><td>";
				print $theclient "<input type=text id=\"ObjMatchToken:$mcount-$count\" name=\"ObjMatchToken:$mcount-$count\" size=13 style=\"background: #eeeeff;\" value=\"$token\">\n";
				print $theclient "</td><td>";
				print $theclient "<select name=\"ObjMatchCMP:$mcount-$count\" size=1 style=\"background: #eeeeff;\">";
				print $theclient "<option value=\"=\"" . ($cmp ne "!="?" selected":"")  . ">=</option>";
				print $theclient "<option value=\"!=\"" . ($cmp eq "!="?" selected":"") . ">!=</option>";
				print $theclient "</select></td>\n";
				print $theclient "</tr><tr><td colspan=2>";
				print $theclient "<input type=text id=\"ObjMatch:$mcount-$count\" name=\"ObjMatch:$mcount-$count\" size=20 value=\"$value\" style=\"background: #eeeeff;\">";
				print $theclient "</td></tr></table>";
				$mcount++;
			}

			$match="AddMatch=$count";

			if($url =~ /\W$match\b/ || $mcount==0) {
				# one extra for the user to add more stuff if they need it
				print $theclient "<table border=0><tr><td>";
				print $theclient "<input type=text id=\"ObjMatchToken:$mcount-$count\" name=\"ObjMatchToken:$mcount-$count\" size=13 style=\"background: #eeeeff;\">\n";
				print $theclient "</td><td>";
				print $theclient "<select name=\"ObjMatchCMP:$mcount-$count\" size=1 style=\"background: #eeeeff;\">";
				print $theclient "<option value=\"=\" selected>=</option>";
				print $theclient "<option value=\"!=\">!=</option>";
				print $theclient "</select></td>\n";
				print $theclient "</tr><tr><td colspan=2>";
				print $theclient "<input type=text id=\"ObjMatch:$mcount-$count\" name=\"ObjMatch:$mcount-$count\" size=20 style=\"background: #eeeeff;\">";
				print $theclient "</td></tr></table>";
			}
			print $theclient "</td>";
			print $theclient "<td>";

			print $theclient "<div align=center style=\"margin-bottom: 5px; border-left: 2px solid #eeeeee; border-top: 2px solid #eeeeee; border-right: 2px solid #aaaaaa; border-bottom: 2px solid #aaaaaa; background-color: #dddddd\"><a href=\"/setobjective?RemoveObjective=$count\" style=\"font-size: 9px; color: #ff5555; text-decoration: none; display: block;\">Remove this Objective</a></div>";
			if($url !~ /\W$match\b/ && $mcount !=0) {
				print $theclient "<div align=center style=\"border-left: 2px solid #eeeeee; border-top: 2px solid #eeeeee; border-right: 2px solid #aaaaaa; border-bottom: 2px solid #aaaaaa; background-color: #dddddd\"><a href=\"/objective?AddMatch=$count\" style=\"font-size: 9px; color: #33aa33; text-decoration: none; display: block;\">Add a new Match term</a></div>";
			}
			
			print $theclient "</td>";
			print $theclient "</tr>\n";
			$count++;
		}
		if(!$AddObjective) {
			print $theclient "<tr style=\"background: #ffeeee;\"><td colspan=6>";
			print $theclient "<div align=center><div align=center style=\"border-left: 2px solid #eeffee; border-top: 2px solid #eeffee; border-right: 2px solid #aabbaa; border-bottom: 2px solid #aabbaa; background-color: #ddeedd\"><a href=\"/objective?AddObjective=1\" style=\"font-size: 9px; color: #33aa33; text-decoration: none; display: block;\">Add a new Objective</a></div></div>";
			print $theclient "</td></tr>\n";
		}
		print $theclient "</table></center>\n";
		print $theclient "<div align=center><input type=submit value=\"Change Configuration\"></div>\n";
		print $theclient "</form>";

		print $theclient "<center><table border=0 cellpadding=5><tr>";
		print $theclient "<td><a href=\"objective?NISPOM=1\" onclick=\"return(confirm('This action will overwrite your existing objectives configuration. Are you sure?'));\">Generate NISPOM Configuration</a></td>";
		print $theclient "<td><a href=\"objective?SOX=1\" onclick=\"return(confirm('This action will overwrite your existing objectives configuration. Are you sure?'));\">Generate Sarbanes/Oxley Configuration</a></td>";
		print $theclient "<td><a href=\"objective?PCI=1\" onclick=\"return(confirm('This action will overwrite your existing objectives configuration. Are you sure?'));\">Generate Payment Card Industry Configuration</a></td>";
		print $theclient "</tr></table></center>";

		SnareTail($theclient);
		Tail($theclient);
	} elsif($url =~ /^\/Users$/) {
		HTMLHeader($theclient,"text");
		ShowUsers($theclient);
	} elsif($url =~ /^\/Groups$/) {
		HTMLHeader($theclient,"text");
		ShowGroups($theclient);
	} elsif($url =~ /^\/GroupMembers$/) {
		HTMLHeader($theclient,"text");
		ShowGroupMembers($theclient);
	} elsif($url =~ /^\/LastLogins$/) {
		HTMLHeader($theclient,"text");
		ShowLastLogins($theclient);
	} elsif($url =~ /^\/events/) {
		HTMLHeader($theclient,"html");
		Head($theclient,"Snare Agent for Linux");
		SnareHead($theclient);

		print $theclient "<div align=center><table border=1 cellpadding=1 cellspacing=0>";
		print $theclient "<tr style=\"background: #ffffdd;\"><th>Criticality</th><th>Date</th><th>Time</th><th>Event</th><th>Status</th><th>Details</th></tr>\n";

		$scount=kill(12,$ppid);

		$lines="";
		$itemsfound=select($wrout=$wrin,undef,undef,1);
		while($itemsfound) {
			do {
				$bytes=0;
				eval {
					local $SIG{ALRM} = sub { die "alarm\n" };
					alarm 1;
					$bytes=sysread(STDIN,$data,1024);
					alarm 0;
				};
				if($bytes) {
					$lines .= $data;
				}
			} while($bytes);
			$itemsfound=select($wrout=$wrin,undef,undef,1);
		}
		@eventlines=split(/\n/,$lines);
		$EventCount=0;
		foreach $line (@eventlines) {
			chomp($line);
			if($line =~ /END/) {
				last;
			}

			@lineparts=split(/	/,$line);
			$system=shift(@lineparts);
			$type=shift(@lineparts);
			$critstring=shift(@lineparts);
			$critstring=~s/^criticality,//;
			$eventstring=shift(@lineparts);
			($null,$event,$date,$time)=split(/[, ]/,$eventstring);
			$remainder=join(" &nbsp;&nbsp;",@lineparts);
			if($toggle==0) { $toggle=1; } else { $toggle=0; }

			# Work out timestamp.
			$year=substr($date,0,4); $month=substr($date,4,2); $day=substr($date,6,2);
			($hour,$min,$sec)=split(/:/,$time);
			# Approximate timestamp (we dont need to be exact, so the month value is off)
			$atimestamp=$sec+($min*60)+($hour*3600)+($day * 86400) + ($month * 2678400)+ ($year * 31536000);
			if($atimestamp > $lastatimestamp) {
				$foreground="#00aa00";
				if($EventCount==0) {
					# Save off a temporary time stamp.
					if($atimestamp > $tatimestamp) {
						$tatimestamp=$atimestamp;
					}
				}
			} else {
				$foreground="#000000";
			}

			if($critstring eq "0") {
				$sendimage="/Clear.gif";
				if($toggle) { $colour="#ffffff"; } else { $colour="#eeeeee"; }
			} elsif($critstring eq "1") {
				$sendimage="/Green.gif";
				if($toggle) { $colour="#ccffcc"; } else { $colour="#bbffbb"; }
			} elsif($critstring eq "2") {
				$sendimage="/Yellow.gif";
				if($toggle) { $colour="#ffffcc"; } else { $colour="#ffffbb"; }
			} elsif($critstring eq "3") {
				$sendimage="/Orange.gif";
				if($toggle) { $colour="#ffddcc"; } else { $colour="#ffccbb"; }
			} else {
				$sendimage="/Red.gif";
				if($toggle) { $colour="#ffcccc"; } else { $colour="#ffbbbb"; }
			}

			if($remainder =~ /return,[0-9]*,yes/) {
				if($toggle) { $rcolour="#99ff99"; } else { $rcolour="#88ff88"; }
				$status="<td style=\"background: $rcolour;\">Success</td>";
			} else {
				if($toggle) { $rcolour="#ff9999"; } else { $rcolour="#ff8888"; }
				$status="<td style=\"background: $rcolour;\">Failure</td>";
			}

			print $theclient "<tr style=\"background: $colour\"><td><div align=center style=\"color: $foreground;\"><img src=\"$sendimage\">&nbsp;$critstring</div></td><td><div align=center style=\"color: $foreground;\">$date</div></td><td><div align=center style=\"color: $foreground;\">$time</div></td><td><div align=center style=\"color: $foreground;\">$event</div></td>$status<td><div align=left style=\"color: $foreground;\">$remainder</div></td></tr>\n";
			$EventCount++;
		}

		$lastatimestamp=$tatimestamp;
		if(!$EventCount) {
			print $theclient "<tr><td colspan=6><div align=center>The Snare Agent has not yet reported any events</div></td></tr>\n";
		}
		print $theclient "</table>\n</div><br>\n";

		print $theclient "<div align=center><i>The last 100 events will be displayed. The page will automatically refresh every 30 seconds</i></div>";

		# Set a refresh in 30 seconds
		print $theclient "<script type=\"text/javascript\">\n";
                print $theclient "function Reload() {\n";
                print $theclient "    window.location.href='/events';\n";
                print $theclient "}\n";
                # 30 seconds
                print $theclient "mytimeout=setTimeout('Reload()',30000);\n";
                print $theclient "</script>\n";

		SnareTail($theclient);
		Tail($theclient);
	} elsif($url =~ /^\/ConfigDump/) {
		# REDRED
		HTMLHeader($theclient,"text");
		ReadConfig();
		print $theclient "[Version]\n	Snare for Linux $VERSION\n\n";

		print $theclient "[Remote]\n";
		print $theclient "	Allow: " . $Config{"REMOTE"}{"ALLOW"} . "\n";
		print $theclient "	WebPort: " . $Config{"REMOTE"}{"LISTEN_PORT"} . "\n";
		print $theclient "	Restrict: " . $Config{"REMOTE"}{"RESTRICTIP"} . "\n";
		print $theclient "	RestrictIP: " . $Config{"REMOTE"}{"RESTRICTIP"} . "\n";
		print $theclient "	Password: " . ($Config{"REMOTE"}{"ACCESSKEY"}?"1":"0") . "\n\n";

		print $theclient "[Output]\n";
		$count=0;
		foreach $element (keys(%{$Config{"OUTPUT"}{"NETWORK"}})) {
			($dest,$port)=split(/:/,$Config{"OUTPUT"}{"NETWORK"}{$element});
			if(!$port) { $port=6161; }
			print $theclient "	Network $count: $dest:$port\n";
			$count++;
		}
		print $theclient "\n";

		print $theclient "[Config]\n";
		print $theclient "	CritAudit: " . ($Config{"CONFIG"}{"USE_CRITICALITY"}?"1":"0") . "\n";
		print $theclient "	Audit: " . ($Config{"CONFIG"}{"SET_AUDIT"}?"1":"0") . "\n";
		if(!$Config{"CONFIG"}{"SYSLOG_FACILITY"}) { $Config{"CONFIG"}{"SYSLOG_FACILITY"}="info"; }
		if(!$Config{"CONFIG"}{"SYSLOG_PRIORITY"}) { $Config{"CONFIG"}{"SYSLOG_PRIORITY"}="local1"; }
		print $theclient "	SyslogFacility: " . $Config{"CONFIG"}{"SYSLOG_FACILITY"} . "\n";
		print $theclient "	SyslogPriority: " . $Config{"CONFIG"}{"SYSLOG_PRIORITY"} . "\n";
		print $theclient "\n";

		foreach $element (sort(keys(%{$Config{"OBJECTIVES"}}))) {
			print $theclient "[Objective $element]\n";
			$tconf=$Config{"OBJECTIVES"}{$element};
			$tconf =~ s/[ \t]+/\n        /g;
			$tconf =~ s/=/: /g;
			#$tconf=str_replace("=",": ",$tconf);
			#$tconf=str_replace(" +","\n      ",$tconf);
			print $theclient "	" . $tconf . "\n\n";
		}
	} else {
		HTMLHeader($theclient,"html");
		Head($theclient,"Snare Agent for Linux");
		SnareHead($theclient);
		print $theclient "<center><h1>Snare Agent for Linux</h1></center>\n<p>";
		print $theclient "<center>Version $VERSION</center>\n<p>";
		print $theclient "<center><img src=\"/intersect.gif\"></center>";
		SnareTail($theclient);
		Tail($theclient);
	}
}

sub WriteConfig() {
	# Do nothing yet.
	open(CONFIG,">$CONFIGFILE") || return(0);
	$dtime=localtime();
	print CONFIG "# Configuration file written by Snare for Linux on $dtime\n\n";

	print CONFIG "[Remote]\n";
	print CONFIG "        allow=" . $Config{"REMOTE"}{"ALLOW"} . "\n";
	print CONFIG "        listen_port=" . $Config{"REMOTE"}{"LISTEN_PORT"} . "\n";
	if($Config{"REMOTE"}{"RESTRICTIP"}) {
		print CONFIG "        restrict_ip=" . $Config{"REMOTE"}{"RESTRICTIP"} . "\n";
	}
	if($Config{"REMOTE"}{"ACCESSKEY"}) {
		print CONFIG "        accesskey=" . $Config{"REMOTE"}{"ACCESSKEY"} . "\n";
	}

	print CONFIG "\n";

	print CONFIG "[Output]\n";
	foreach $element (keys(%{$Config{"OUTPUT"}{"NETWORK"}})) {
		($dest,$port)=split(/:/,$Config{"OUTPUT"}{"NETWORK"}{$element});
		if(!$port) { $port=6161; }
		print CONFIG "        network=$dest:$port\n";
		$count++;
	}

	print CONFIG "\n";
	print CONFIG "[Config]\n";
	print CONFIG "        use_criticality=" . ($Config{"CONFIG"}{"USE_CRITICALITY"}?"1":"0") . "\n";
	print CONFIG "        set_audit=" . ($Config{"CONFIG"}{"SET_AUDIT"}?"1":"0") . "\n";
	if(!$Config{"CONFIG"}{"SYSLOG_FACILITY"}) { $Config{"CONFIG"}{"SYSLOG_FACILITY"}="info"; }
	if(!$Config{"CONFIG"}{"SYSLOG_PRIORITY"}) { $Config{"CONFIG"}{"SYSLOG_PRIORITY"}="local1"; }
	print CONFIG "        syslog_facility=" . $Config{"CONFIG"}{"SYSLOG_FACILITY"} . "\n";
	print CONFIG "        syslog_priority=" . $Config{"CONFIG"}{"SYSLOG_PRIORITY"} . "\n";

	print CONFIG "\n";
	print CONFIG "[Objectives]\n";
	foreach $element (sort(keys(%{$Config{"OBJECTIVES"}}))) {
		print CONFIG "        " . $Config{"OBJECTIVES"}{$element} . "\n";
	}
	close(CONFIG);
}

sub ReadConfig() {
	%Config=();
	open(CONFIG,$CONFIGFILE) || return(0);
	while($line = <CONFIG>) {
		chomp($line);
		if($line =~ /^[ \t]*#/ || $line =~ /^[ \t#]*$/) {
                        # Ignore
                        next;
                } elsif ($line =~ /^[ \t]*\[[a-zA-Z]+\]/) {
                        ($null,$section)=split(/[\[\]]/,$line);
                        $section =~ tr/a-z/A-Z/;
                } else {
                        $line =~ s/^[ \t]+//;
                        $line =~ s/[ \t]+/\t/g;
                        $line =~ s/[ \t]+$//;

                        if($section eq "REMOTE") {
                                ($key,$val)=split(/=/,$line);
                                $key =~ tr/a-z/A-Z/;
                                $Config{$section}{$key}=$val;
                        } elsif($section eq "OUTPUT") {
                                # Add this into our output file array
				($key,$val)=split(/=/,$line);
                                $key =~ tr/a-z/A-Z/;
				$elements=keys(%{$Config{$section}{$key}});
                                $Config{$section}{$key}{$elements}=$val;
                        } elsif($section eq "CONFIG") {
                                ($key,$val)=split(/=/,$line);
                                $key =~ tr/a-z/A-Z/;
                                $Config{$section}{$key}=$val;
                        } elsif($section eq "OBJECTIVES") {
				$elements=keys(%{$Config{$section}});
                                $Config{$section}{$elements}=$line;
                        }
		}
	}			
	
	close(CONFIG);
}

sub GetArguments() {
	my($url)=shift;
	my(@elements)=split(/[\?\&]/,$url);
	my(%Arguments)=();

	# Get rid of the URL
	shift(@elements);

	foreach $element (@elements) {
		($key,$val)=split(/=/,$element);

		# Unescape the data
		$key=~s/\+/ /g;
		$val=~s/\+/ /g;
		$key=~s/%([0-9a-f]{2})/pack("c",hex($1))/gie;
		$val=~s/%([0-9a-f]{2})/pack("c",hex($1))/gie;

		$Arguments{$key}=$val;
	}
	return(%Arguments);
}

sub ShowUsers() {
	my($tclient)=@_;
	setpwent();
	do {
		($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$dir,$shell,$expire) = getpwent();
		if($name) {
			print $tclient "$name\t$uid\t$gcos\t$expire\n";
		}
	} while($name);
	endpwent();
}

sub ShowGroups() {
	my($tclient)=@_;
	setgrent();
	do {
		($name,$passwd,$gid,$members) = getgrent();
		if($name) {
			print $tclient "$name\t$gid\n";
		}
	} while($name);
	endgrent();
}

sub ShowGroupMembers() {
	my($tclient)=@_;

	setgrent();
	do {
		($name,$passwd,$gid,$members) = getgrent();
		if($name) {
			$GroupName{$gid}=$name;
			$GroupID{$name}=$gid;

			foreach $member (split(/ /,$members)) {
				$Group{$name}{$member}=1;
			}
		}
	} while($name);
	endgrent();

	setpwent();
	do {
		($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$dir,$shell,$expire) = getpwent();
		if($name) {
			$grname=$GroupName{$gid};
			if(!$grname) {
				$grname=$gid;
			}

			$Group{$grname}{$name}=1;
		}
	} while($name);
	endgrent();

	foreach $group (sort(keys(%Group))) {
		$members=join(",",sort(keys(%{$Group{$group}})));
		$gid=$GroupID{$group};
		if(!$gid) {
			$gid=$group;
		}
		print $tclient "$group	$gid	$members\n";
	}
}

sub ShowLastLogins() {
	my($tclient)=@_;
	$rc=open(LAST,"/usr/bin/last -a -x |");
	if(!$rc) {
		print $tclient "Cannot open last login information.";
		return(0);
	}
	while($line = <LAST>) {
		chomp($line);
		if($line !~ /^[ \t]*$/ && $line !~ /^wtmp begins/ && $line) {
			$field1=substr($line,0,9); chomp($field1);
			$field2=substr($line,9,13); chomp($field2);
			$field3=substr($line,22,17); chomp($field3);
			$field4=substr($line,41,17); chomp($field4);
			$field5=substr($line,60); chomp($field5);
			if($field1 =~ /^(runlevel|reboot)/) {
				$info="boot";
			} else {
				$info="login";
			}
			print $tclient "$info\t$field1\t$field2\t$field3\t$field4\t$field5\n";
		}
	}
	close(LAST);
	
	$rc=open(LAST,"/usr/bin/lastb -a -x |");
	if(!$rc) {
		print $tclient "Cannot open last login information.";
		return(0);
	}
	while($line = <LAST>) {
		chomp($line);
		if($line !~ /^[ \t]*$/ && $line !~ /^btmp begins/ && $line) {
			$field1=substr($line,0,9); chomp($field1);
			$field2=substr($line,9,13); chomp($field2);
			$field3=substr($line,22,17); chomp($field3);
			$field4=substr($line,41,17); chomp($field4);
			$field5=substr($line,60); chomp($field5);
			
			print $tclient "bad login\t$field1\t$field2\t$field3\t$field4\t$field5\n";
		}
	}
	close(LAST);
	return(1);
}

sub HTMLHeader() {
	my($tclient,$type)=@_;

	if($type =~ /^html$/) {
		$mytype = "html";
	} else {
		$mytype = "plain";
	}

	print $tclient "HTTP/1.0 200 OK\r\n";
	print $tclient "Server: Snare/1.0\r\n";
	print $tclient "MIME-Version: 1.0\r\n";
	print $tclient "Content-Type: text/$mytype\r\n\r\n";
}

sub Head() {
	my($tclient,$title)=@_;
	if($title eq "") {
		$title="InterSect Alliance - Information Technology Security";
	}
	print $tclient "<html><head>" .
		"<title>$title</title>" .
		"<meta name=\"TITLE\" content=\"InterSect Alliance - Information Technology Security\">" .
		"<style type=\"text/css\">\n" .
		"body {\n" .
		" font-family: Verdana,Helvetica,sans-serif;\n" .
		" font-size: 10px; font-weight: normal;\n" .
		" margin: 0px;\n" .
		"}\n" .
		"font {\n" .
		" font-family: Verdana,Helvetica,sans-serif;\n" .
		" text-decoration: none; font-size: 10px;\n" .
		" font-weight: normal;\n" .
		"}\n" .
		"table {\n" .
		" margin: 0px; padding: 0px;\n" .
		"}\n" .
		"td {\n" .
		" font-size: 75%;\n" .
		"}\n" .
		"</style>\n" .
		"<meta http-equiv=\"Pragma\" content=\"no-cache\">" .
		"<meta http-equiv=\"Expires\" content=\"-1\">" .
		"</head>\n" .
		"<body text=black bgcolor=white link=#000066 vlink=#000044 alink=#000055>" .
		"<table border=0 cellspacing=0 cellpadding=0 width=100%>\n" .
		"<tbody>\n" .
		" <tr>\n" .
		"  <td height=70><img src=\"/intersect.gif\" alt=\"InterSect\" width=205 height=70 hspace=20 vspace=0 border=0 align=Right>" .
		"</td>\n" .
		"   <td height=70 width=1% bgcolor=#eeeeee><br></td>" .
		"   <td height=70 width=1% bgcolor=#dddddd><br></td>" .
		"   <td height=70 width=1% bgcolor=#cccccc><br></td>" .
		"   <td height=70 width=1% bgcolor=#ccbbbb><br></td>" .
		"   <td height=70 width=1% bgcolor=#ccaaaa><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc9999><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc8888><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc7777><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc6666><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc5555><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc4444><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc3333><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc2222><br></td>" .
		"   <td height=70 width=1% bgcolor=#cc1111><br></td>\n" .
		"   <td height=70 width=86% bgcolor=#cc0000><font color=white><center><h1>SNARE for Linux</h1></center></font></td>\n" .
		" </tr>\n" .
		"</tbody>\n" .
		"</table>\n" .
		"<table border=0 cellspacing=0 cellpadding=0 width=100%>\n" .
		" <tr><td height=3 bgcolor=white width=100%></td></tr>\n" .
		" <tr><td height=3 bgcolor=black width=100%></td></tr>\n" .
		" <tr><td height=2 bgcolor=#AAAAAA width=100%></td></tr>\n" .
		" <tr><td height=2 bgcolor=white width=100%></td></tr>\n" .
		"</table>\n" .
		"<table border=0 cellspacing=0 cellpadding=5 width=100% height=100%>\n" .
		"<tbody>\n" .
		" <tr>\n" .
		"  <td valign=Top width=20% bgcolor=#cc0000>\n" .
		"   <div align=Center><font color=#ffffff>\n" .
		"   <br>\n" .
		"   <font face=\"Helvetica,Arial,sans-serif\" size=-1><B>\n" .
		"   <br><A HREF=\"/general\" style=\"color:FFFFFF;text-decoration:none\">General Configuration</A><br>\n" .
		"   <br><A HREF=\"/remote\" style=\"color:FFFFFF;text-decoration:none\">Remote Control Configuration</A><br>\n" .
		"   <br><A HREF=\"/objective\" style=\"color:FFFFFF;text-decoration:none\">Objectives Configuration</A><br>\n" .
		"   <br><A HREF=\"/events\" style=\"color:FFFFFF;text-decoration:none\">Display Recent Events</A><br>\n" .
		"   <br><A HREF=\"/restart\" style=\"color:FFFFFF;text-decoration:none\">Apply the Latest Audit Configuration</A><br>\n" .
		"   <br><br><font size=-2><A HREF=\"/Users\" style=\"color:FFFFFF;text-decoration:none\">Display a list of Users</A><br>\n" .
		"   <br><A HREF=\"/Groups\" style=\"color:FFFFFF;text-decoration:none\">Display a list of Groups</A><br>\n" .
		"   <br><A HREF=\"/GroupMembers\" style=\"color:FFFFFF;text-decoration:none\">Display a list of Group Members</A><br>\n" .
		"   <br><A HREF=\"/LastLogins\" style=\"color:FFFFFF;text-decoration:none\">Display a list of Logins and Reboots</A><br>\n" .
		"   </b></font>\n" .
		"   <br>\n" .
		"   </div>\n" .
		"  </td>\n" .
		"  <td width=100% valign=Top>\n" .
		"   <table cellpadding=0 cellspacing=10 border=0 width=100%>\n" .
		"    <tbody>\n" .
		"    <tr>\n" .
		"     <td valign=Top align=Justify>\n";
}

sub Tail() {
	my($tclient)=@_;
	print $tclient "     </td>\n" .
		"    </tr>\n" .
		"    </tbody>\n" .
		"   </table>\n" .
		"<center>\n" .
		"<br><br><font size=-1>(c) <a href=\"http://www.intersectalliance.com\">Intersect Alliance</a> Pty Ltd 1999-2006. " .
		"This site is powered by <a href=\"http://www.intersectalliance.com/projects/index.html\">SNARE for Linux.</a></font>" .
		"</center>\n" .
		"</td></tr></tbody></table>\n" .
		"</body></html>";
}

# Pretty page surrounds
sub SnareHead() {
	my($tclient)=@_;

}

sub SnareTail() {
	my($tclient)=@_;
	print $tclient "\n";
}

sub InterSectImage {
	my($tclient)=@_;
	my $InterSectGIF="R0lGODlhzQBGALMAAB8aF01JR2toZtwrGYF+fONVRo+Mi5eWleuCePKzrLOysszLy/jRzunl" .
	   "5f////j29SwAAAAAzQBGAEAI/gAdCBxIsKDBgw4eIFxIUCHDhxAjSkT4AMGAAQgmatw40CHH" .
	   "jxs9ghQJsuTEBxdTpkxgsqXLlzBjyvyoMuXMmzAB6NQp0MDOnzoDkOz504ADn0CTAniAVOlO" .
	   "oQqcSgVqdKpVAARrXiyA0+XQkkCJ7jRakMDPAA3MjmUYtikAsgfV6lTw9WFYgwKACj34gIFW" .
	   "jF0DCx5MeGPUsw08WtTKMqFcAGgZNghQ9ChQAQ1Kuq36s7DnzxL7/kXw4DFah6gFpk4o8ADV" .
	   "Bwpjs45d1/Lau6Bz6y5YQCvXmbV3Cx++MUHvvxcZHFSgoOGB5g4WGFhAfUFmAwqqL2i94EGD" .
	   "AwSZ/nNXreDA9gXmq5vvmb165sDBIcYnfjAB8gLz6ev3up+ifa0i5VeYgCEtROBIsyUIm0Gy" .
	   "seZgbatFqNqEBB2nUkb9BXbXZgJN9hN0YukE10FtVYbQY1clxVmKSjW0WE0HZijRhiYetIBS" .
	   "AdjVGYcUPZAXi0Bth9tBHv4EnkEWqjQUU5SxKMB2HTEFpJBXrTiVjFhO9F9NGKIIpE7g8ahZ" .
	   "ZUNmaeZGL6rUkQJNfgllQmyyCKJBHJZ55p0LNZBkSr8VhN4D66mGnkGBEjRdRwcUetQB7zmw" .
	   "HnraBYqddo3ieecDe2IUo6WcdtrRlkp6KuqoqmH6F34ZbkrqpZkOoOqq/7DqZmpNjcVqK55/" .
	   "VSrjqyR2BmdSkSHEoVtS2TbVXkV+CQB0yho5EKgp6XorQmFJudZjBJAkZq87GfvWQo/NKWyN" .
	   "8jnQwJDQXlTrtNT6KuYDN/6UmLcjGlTiWic2K68D+s5VEEq+Ycmrve6S21GbKrK1o8Edndsv" .
	   "AN3ZqRrCAAiAUJopKcfuTQ0oYMChGwsU709vCuRXwAMl2+x7FF9FpVVWFhsyfMRaPBByDj0m" .
	   "wMc898wzwgZYK2JLdfo6s4ytKqSzz0z3zGiIbw1cdLdH75rpbz/qC2XWci40NU9Vy4jxRWGX" .
	   "/dnYA6zLWgPW8fUvTEtS5KDZoD2QLp/LpYceiP6DFqRoT1B+pxBzDk33HV2PHkloon/TLRhy" .
	   "KzkuuWB9JYnh5C8N7OndgOmnOeaUN3DfgqCb9DmenDd6eulho315qpfSLRpjq7NOd6av244g" .
	   "f2f+RXpH+9UON+h/6W78RpmqPXfIuMHbZAAEzMvXsCkKLdVesBnQslRPDtSv4g6MjWqWtd/l" .
	   "MLDSQm0UsU5565RQD2zPYmYPL9XhXwl4JHx/NJ71k83A2xbBuiVAgpgmfSkDmmGGhLb9Yal/" .
	   "Q3MAwsBHL4URkGEDWVrTeqbADW6QYvo7FfliAsFvdShIhcOg9xaGr7jUDzwPyxFB7pY/s5Ww" .
	   "XocJynVUKJB7RZBBGv70oNPeQxWfcU1E3TlIAx24nxsy6IhUsSBPCphBxDyIQqvpodEKckQZ" .
	   "HkSEsMtJwVpYkPMlpWQDnCIPHQDFKfFri2WkWL1MNpoeXfGOEIINbRS0PBKO8YfL0QsCV3hB" .
	   "MiIkTsqKGBwNYkZ/FQRtDBCJyoAEovi56QEpiln7jheRHO4EjUkbiCebZZSRTQmTVXqjVTg5" .
	   "EdecxSMnq0mfwgUR6j1FamRaJCs5ogAC9OxNaGuMl760vqLg8ja63CXcTqW0+u3QkBz5GlaU" .
	   "2RVIluZDH3GLAAg0FGlSsysA45K56tedRqaILuNCJtW+eRPxqaaNKUqiJSlZF2+ys53FA6ne" .
	   "PY3nzgLtM2TeqeM/jxdLGDHIY+Jim0GsQ5K2pQyNDhVZZuClnTdVlKED1UhsGNCqAWjMOdsp" .
	   "j0ekQyc0OupNzBHcQJ5zKECdZ0QONQ8TOVkRyA1AV94xAOMSZYD3DEokMvVbEj2203mt5wHM" .
	   "OVTfBJId7ugvoxJBm0qUZ66IdqhtbKNUdCoq0ehUCl6ZsWrbKKodh1x0kFAliFQL8NG0lm5L" .
	   "CJAe3QICADs=";

	print $tclient base64decode($InterSectGIF);
}

sub ClearImage {
	my($tclient)=@_;
        my $image="R0lGODlhDQAPAOcAAP////Dw8GNjYyYmJg4ODi8vL2FhYW5ubldXV5qams/Pz9fX18zMzLOzs0ND" .
                "Q35+fnJycmBgYKenp+Pj4+Xl5fPz89XV1bS0tF1dXXBwcN7e3kVFRXp6erKystnZ2djY2NLS0nZ2" .
                "dlNTU7u7u1RUVGlpabGxsbCwsLy8vLq6uqysrKqqqnR0dE1NTUdHR66urri4uMPDw8HBwbW1taOj" .
                "o42NjUpKSq+vr7+/v9PT09bW1tDQ0K2trb6+vsnJyeHh4ebm5ufn58vLy8LCwmJiYm9vb9ra2uvr" .
                "6/f39/b29vT09O3t7d/f33t7e6mpqfj4+O/v74qKio2Ni8jIxvn59/39+/r6+Pv7+fj49v///cvL" .
                "yYqKiImJh2NjYZWVk9ra2Pz8+uHh35GRj3JycOLi4X5+fF5eXG1ta6CgntTU0u/v7dPT0aWlo3V1" .
                "c11dW3t7eY2PjmZoZ1NVVD9BQD5AP0RGRUlLSl9fX7u8u4WHhnh6eXt9fLS2te7u7sz/Zv8AzMwA" .
                "/wCZmZkzmZkAmcwAmQAAmTMzmWYAmcwzmf8AmQBmmTNmmWYzmZlmmcxmmf8zmTOZmWaZmZmZmcyZ" .
                "mf+ZmQDMmTPMmWbMZpnMmczMmf/MmQD/mTP/mWbMmZn/mcz/mf//mQAAzDMAmWYAzJkAzMwAzAAz" .
                "mTMzzGYzzJkzzMwzzP8zzABmzDNmzGZmmZlmzMxmzP9mmQCZzDOZzGaZzJmZzMyZzP+ZzADMzDPM" .
                "zGbMzJnMzMzMzP/MzAD/zDP/zGb/mZn/zMz/zP//zDMAzGYA/5kA/wAzzDMz/2Yz/5kz/8wz//8z" .
                "/wBm/zNm/2ZmzJlm/8xm//9mzACZ/zOZ/2aZ/5mZ/8yZ//+Z/wDM/zPM/2bM/5nM/8zM///M/zP/" .
                "/2b/zJn//8z///9mZmb/Zv//ZmZm//9m/2b//6UAIV9fX3d3d4aGhpaWlsvLy7KystfX193d3ePj" .
                "4+rq6vHx8fj4+P/78KCgpICAgP8AAAD/AP//AAAA//8A/wD//////yH5BAEKAP8ALAAAAAANAA8A" .
                "AAioAP8J/CdgAIECAgYOPEMiAQMLDC442DOQhQAaEyYoUWLhAgYWAideeBgvXkcWJP6ROEPjwgUU" .
                "MFGoUMGihYMEM1HE2OmSRo0WJHJaGEp0JgYBKBiUzJgRYgwBZy7Em3Bko5IjE+Jd2LNHxVQlVcJe" .
                "PUJjT40aWqyEraK2CoMaWwh6ibc2bBgxIP/twXAGjYUjFmiwwEBRYI04JObQcdACAw2FA9lxhRwQ" .
                "ADs=";
	print $tclient base64decode($image);
}

sub GreenImage {
	my($tclient)=@_;

	my $image="R0lGODlhDQAPAOd+AGyFTICucOLi4WV6T9n/j8HUknV0YqOjo3a5SI7PT+P127zvhmSzNipWJWGk" .
		"PV6dRN/8pXSoU9H+l2WZToGBgXasWJriWladP5rIbqvvZEZ0NomjZJTHXg8vAB48CB5AH11dXc/8" .
		"h1p4PE14Q1VVVVqUP5XaW1FRUXizWW29TjE/Lp/FcsH+fGquQWF8SfDw8LDZne7u7k2JP4OgXVNu" .
		"T6nXlojOUavvXN7e3mukRV54SZS7ak5gNm+QSSUlJd/z2ImJh9j/kl9uTXS0VI2PjmaSQ3t7eer/" .
		"vkR+Mru8u0RiRoG4UIGnWqTpW6HrWLS0tExbMBw9Dq/xaYeWgZ3WiXi+SonRVInWSpvgX1V6Na34" .
		"YJnWiNj/h7z/aGeSTZ7gZY6OjklXM4qKilGQN6/BgzRVHilWHW+vUWu3OXR0dGCDP3Jycrz5bnBw" .
		"cMj6h6fVlcv9gGeDSe796Hd8ZnSoZn6oXCJFD3e5S3uNd4TRR53aWdrp1L7+bDhGNf///1F9Sl1u" .
		"TDZSH6XIeHS4QUtRLUV2NGF5SdD4mLS2tZawaZ3LaShHHuHh4b7+cnTBUd3d3V51SVt3RigoKOf0" .
		"4LXrc35/b1idMsH/biNDHGanQ3m4T43LXMnwl35+fJrTbE5WLxc3EA4ODr+/v7u7u+L/rLTxeND/" .
		"g2uJSwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEKAP8ALAAAAAANAA8A" .
		"AAjFAP8J/HfCRyhJJAYOXAOCAiIcooiQSLOwz5Q9k+RM+oFHxRqBmEYEeAOjBhUqdP58+EfDDJIH" .
		"Z1AMcZTiggwzShYVGuOgxR0EaBhYKlEIlIcJmapY+WLiSp5BObx0sFMBQQILUppoaZJgSR0PZSJs" .
		"spChER8+N/RwYFImUhFPpVhw4XKJDyVFPQyJyaJoQQhTBEwFCbJCBJB/LtTs4HQEAikJgk5BEmhE" .
		"R5wNZAokmgHAkBGFgMIQIgQFCiCFCisZmINIYUAAOw==";
	print $tclient base64decode($image);
}

sub OrangeImage {
	my($tclient)=@_;

	my $image="R0lGODlhDQAPAOeEAAAAAP+ZXP+xc3pYOYNPOODg3/+hc9RzK0MtGqB3R2BNPaOjo//Kl/WxgF5H" .
		"J/+3jYM7EP+1jYpkO4+DeMxUGeOXboeHh/m5in8/Jv+CRIdoUn9hVJVqOP+7f79ZHP/Yt+h1N+yQ" .
		"c//VrWdnZ/vf1KpNG/+fa6qUgf+RQGMgAI5tVdZaL9p+Yf+oeMVfKI9jQv+7hf+2e/zp20lJSf+y" .
		"e3x8e0VFRfaRZf+aUPDw8O7u7uJgLf+GUGxLM/awidtfHJhgR3lsUL9dKN7e3m01DPBrN/+td+h4" .
		"Jf/RpdZrMRkZGatxQs5mM8bGxoyNjZVmSv/XqP/TqP/Ch8yHVY5AAbm6uXM+Dr+elNx1Pfh2J+dv" .
		"NP+uc79OEHdOQP/x7a9aK+ambsONZsFdNqBnPf/CjdNTIP+OS2IiAI6Ojv+bWIh4caFJKoqKiv+T" .
		"WOicX89lNvuHVLiIa3Jycv/LoMVMIvV6MG5ubnMzEP/Xt9mBMv/474d7bP+ma6VpTlxcXHxGKd5o" .
		"GVhYWGBNK//Bj/////+yhY+GcMRWE+h7M1YeAHl5eP2WWI9PJP+6iKNLEMqVa2ckALO0tNGRXv/p" .
		"12hKL69pNoZAEeHh4cViGrmLWngyF2I2FdXV1ex3KRwcHKlOEv2/gtOPZsPDw6BnQbu7u/+5h9p+" .
		"OwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEKAP8ALAAAAAANAA8A" .
		"AAjIAP8J/DdCCQAlMwYOtGPDSRNOTSwE8jNQDoIrJLzI0DPphIIJAq1UYhHCx4MGYKYs2fSPAxUP" .
		"dFZgMZXnAKZPll4QccSlzI46dTodAXSoxJ1EQigUydAmAA4UWX64SJGCiRYeJmjQ2MLHTJ0kZzSJ" .
		"4WGgEAwYMWJsCcAEApA1N1o8IENGCowti76MYoOhwoM5IkQgYSDADaM4/1T8CfUAzwcRoCSNeSKw" .
		"hgYCfcI8yrREwoAaCjd06UHJgaAgCgeSUrPHUCSFAQEAOw==";
	print $tclient base64decode($image);
}
	
sub YellowImage {
	my($tclient)=@_;

	my $image="R0lGODlhDQAPAOeGAHpEIuLi4eqoUSsWHfjVb+STKv/qb+WnLoZ2dv/35uCWM0hNJe3S2+6rOmtn" .
		"Tv/o3P/ocv/+if/kcv/8ieOVQ+ajXf/7f/OmTHNwT717WZl2PP/9gpCUSv/1gvbCVXVuQNG6wv3O" .
		"irNoMdPbkvmzOow/OXgtDv/9iP/k0dHKcnk+HIBrcpR2VFNUNc2JNk5UMP/6gfa4M4Z6MvDw8O7u" .
		"7nVtP//XYHppb97e3pCJd5qJbWtEP5hIR8jDZY2Gav/QWXtgaXY6IvKoSfHujfHCivvFS9uWL/zt" .
		"evXAeva5Ro2PjqJ7ULu8u5aHRPOnL//2jfnGWTgnLYJEN0VLKY9SHMh1Kf/xb/7QYf/ETvDDiP3/" .
		"m+rP2P/VW//rcpR8Pq6NfP//iYNXNKuDed3Ca//yf52hV35+fuGrZcCbVei0ToV2ceawTKeMboU/" .
		"Dv//o//fYaOOlYJ1ZJFGJ3JycnBwcG5ubv/obqVGKv7LTM/ajo5SUnpuPspuLfOvKuyWQeKkN+Ov" .
		"WdiPS//16PnFiVxNUv/IRv/////aXfi0U//LZ/X8leGiOeixTZZjLrS2tc5/RK1oPjAdI4lzTP/J" .
		"VlBSK4htduHh4eSdIWtJP/zTauatOuCjSM+TX+SyWbW9cMaWTN3GaY2MTP/p3YOFd+apaH18PLu7" .
		"u//Yaf//of/UVQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEKAP8ALAAAAAANAA8A" .
		"AAjIAP8J/LdiQKQohAYOrAOkEoMtIODcMDNwzg4xgkSJSoDiC6Y5AqVk4DSICJEQSEhBKvFPj5xH" .
		"LjYB6pRmkYIqd3gEEUHByJ80aTQduFSAjwkVgYQ0KOIBSpFCfZz4aQOgAqIkXK6cupIqBokLVMKc" .
		"ScTljQQyXQ7hmSSgEQs0mQxY2TBhg4EumT4t0eFlzBELEwJ3eDJGA5t/e2T0cKPIjZYhKZpIEujj" .
		"QykOI/J4KhOqRpyBShy0mPJiASUMORQORIBglCOFAQEAOw==";
	print $tclient base64decode($image);
}

sub RedImage {
	my($tclient)=@_;

	my $image="R0lGODlhDQAPAOeFAAAAAItbV+Li4cNhVp1US9SCd/+6q6Ojo4dkYPaDfoZ2dv/FuPd8dT4mImoo" .
		"KfGina49QfafmOuml7dRTf+Ddv+vpIeHh6U0Lu94cGVCPlg8MeNmZMBGO4VWUHMqJLyGfLI7N14Q" .
		"DGYlIdSQg2dnZ8RcWXpcXP+9rZM3LOmmnr1MUv+GdfByY/+rma5GQ6WNiUlJSch2ekVFRa1NT/+Z" .
		"hfDw8P+PhcxUU+7u7qlGQf+Me5peXb1RTt7e3v+Qfv++rFYQEOF/fv+UgehpWhkZGcbGxv+hjo2P" .
		"juafnXt7ecF9cnFKS7u8u9hTSpU4M8pURv+jkYBUSbhRUuhsanJNRf/t7YdXSf+Lffri3qdAN659" .
		"eXlGRdKEgrqdn+51bLtXR46OjlkOC4BhXPbc3YqKipNWUf97bMBfT7p0aqtFN4V2cc1aVfBnXXJy" .
		"co6AffB4am5ubsdMT+JwcP+dif+ZiVlBNbZ9dr5DPLk+QfOMh4A9N1xcXJxHQnknGVhYWP/28l8m" .
		"HZA+QNdYUU8OCs1dUf///4x/d9qJj2FAOVgpI/+Yhc5aS3UsJbS2tXE8OGUrH8R9eeSgleHh4eR9" .
		"dKx4a1xHRNXV1f+Yi9RWV6o5Mf/EuXRiWJJWThwcHPnk4X5+fP+4pcPDw44xKXM1No1VVru7u+6V" .
		"h/mLfpRkVpxdVAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA" .
		"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEKAP8ALAAAAAANAA8A" .
		"AAjFAP8J/EeiE4BOMAYOhCMDTChLoSzs2TOwTYMuY6pU+YPlRSU3AgERiHEIyYNII9CkSvSvQ58s" .
		"EFSUGHDmSxoUjDoAEnUBT5w1axY94ZDJiYhBLkAI2uDlDQs2Te64CAOEx40pDHTooGCGjSAeYRzM" .
		"kJPAxhwaNHxQwDCBEalAQfLMgWLkrI5JfMqAGcUlQoUTJ0C1MFVAj5Z/CBxBiqBJ0wkJSjgFEPgJ" .
		"wZYddj5QQtWByieFJpZkyKChziaFA5mEWt1IYUAAOw==";
	print $tclient base64decode($image);
}

sub base64decode {
	local($str)=@_;

	my $str = shift;
	my $res = "";

	$str =~ tr|A-Za-z0-9+/||cd;		 # remove non-base64 chars (padding)
	$str =~ tr|A-Za-z0-9+/| -_|;		# convert to uuencoded format
	while ($str =~ /(.{1,60})/gs) {
		my $len = chr(32 + length($1)*3/4); # compute length byte
		$res .= unpack("u", $len . $1 );	# uudecode
	}
	return($res);
}

sub WebTerminate {
	local($sig)=@_;
	$WEBCONTINUE=0;
	if(!$clientclosed) {
		close($client);
		$clientclosed=1;
	}
	close($server);
}

1;
