#define _GNU_SOURCE

#include <stdio.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <pthread.h>
#include <string.h>
#include <locale.h>
#include "libaudit.h"

#define DEFAULT_NODES 320000
#define NEW_REC_TIMEOUT 2

struct _node {
	void * line;
	struct audit_dispatcher_header hdr;
	unsigned long serial;
	time_t created;
	int complete;
	struct _node *next;
};

typedef struct _node Node;

// Local data
static volatile int signaled = 0;
static int pipe_fd;
static const char *pgm = "SnareDispatchHelper";
Node * phead=NULL;
Node * ptail=NULL;
Node * pcurrent=NULL;
Node * head=NULL;
Node * tail=NULL;
Node * current=NULL;
int PrimaryNodeCounter=0;
int SecondaryNodeCounter=0;
int NODECOUNT = DEFAULT_NODES;
pthread_mutex_t list_mutex = PTHREAD_MUTEX_INITIALIZER; 
pthread_mutex_t	data_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t  list_data_available = PTHREAD_COND_INITIALIZER;

// Local functions
int merge_records(Node *dest, Node *src);
unsigned long line2serial(char *s);
void SnareDispatcher(int *);

// SIGTERM handler
static void term_handler( int sig )
{
	signaled = 1;
}


/*
 * main is started by auditd. See dispatcher in auditd.conf
 */
int main(int argc, char *argv[])
{
	struct sigaction sa;
	int pid,PREV_ERROR=0;
	int daemon2dispatch[2];	// Pipe from daemon to SnareDispatcher
	char exepath[1024];
	char exename[64];
	void* data;
	struct iovec vec_hdr[1],vec_data[1];
	struct audit_dispatcher_header hdr;

	setlocale (LC_ALL, "");
	openlog(pgm, LOG_PID, LOG_DAEMON);
	//syslog(LOG_NOTICE, "starting...");

#ifndef DEBUG
	// Make sure we are root
	if (getuid() != 0) {
		syslog(LOG_ERR, "You must be root to run this program.");
		return 4;
	}
#endif
	// register sighandlers
	sa.sa_flags = 0 ;
	sa.sa_handler = term_handler;
	sigemptyset( &sa.sa_mask ) ;
	sigaction( SIGTERM, &sa, NULL );
	sa.sa_handler = term_handler;
	sigemptyset( &sa.sa_mask ) ;
	sigaction( SIGCHLD, &sa, NULL );
	sa.sa_handler = SIG_IGN;
	sigaction( SIGHUP, &sa, NULL );
	(void)chdir("/");

	// change over to pipe_fd
	pipe_fd = dup(0);
	close(0);
	open("/dev/null", O_RDONLY);
	fcntl(pipe_fd, F_SETFD, FD_CLOEXEC);

	strncpy(exepath,"/usr/sbin/SnareDispatcher",sizeof(exepath));
	strncpy(exename,"SnareDispatcher",sizeof(exename));

        if (pipe(daemon2dispatch) == -1) {
                syslog(LOG_NOTICE,"Cannot open pipe for audit information");
                exit(1);
        }

	pid=fork();
	if(pid == -1) {
		syslog(LOG_NOTICE,"Cannot fork to execute the SnareDispatcher process");
		exit(3);
	} else if(pid == 0) {
		// Child process
		if (dup2(daemon2dispatch[0], STDIN_FILENO) ==-1) {
                	syslog(LOG_NOTICE,"Could not reroute stdin for SnareDispatcher");
                        exit(4);
		}
		// Close unused files.
		close(daemon2dispatch[0]);
                close(daemon2dispatch[1]);

		execlp(exepath,exename,"-",(char *)0);
		syslog(LOG_NOTICE,"Could not execute the SnareDispatcher process");
		exit(5);
	}
	//parent

	close(daemon2dispatch[0]);
	pthread_t thread[1]; 

	// This is required to make sure the dispatcher can keep up with auditd
	// and prevent any dispatch errors
	nice(-1);

	pthread_create(&thread[0], NULL, (void *)SnareDispatcher, &daemon2dispatch[1]);

	// allocate data structure
	data = malloc(MAX_AUDIT_MESSAGE_LENGTH);
	if (data == NULL) {
		syslog(LOG_ERR, "Cannot allocate buffer");
		return 1;
	}

	//clear the memory
	memset(data, 0, MAX_AUDIT_MESSAGE_LENGTH);
	memset(&hdr, 0, sizeof(hdr));

	/* Get header first. it is fixed size */
	vec_hdr[0].iov_base = (void*)&hdr;
	vec_hdr[0].iov_len = sizeof(hdr);

	// Next payload, but we need to set the size according to the header
	vec_data[0].iov_base = data;


	do {
		int rc;
		struct timeval tv;
		fd_set fd;
		Node *new_rec=NULL;

		tv.tv_sec = 1;
		tv.tv_usec = 0;
		FD_ZERO(&fd);
		FD_SET(pipe_fd, &fd);

		rc = select(pipe_fd+1, &fd, NULL, NULL, &tv);
		if (rc == 0) continue;
		else if (rc == -1) break;

		//Grab the header
		rc = readv(pipe_fd, vec_hdr, 1);
		if (rc == 0 || rc == -1) {
			syslog(LOG_ERR, "rc == %d(%s). Exitting", rc, strerror(errno));
			break;
		}
		if (hdr.size > MAX_AUDIT_MESSAGE_LENGTH) {
			// Error, attempt to resync
			if (PREV_ERROR) {
				lseek(pipe_fd,0,SEEK_END);
			} else {
				char garbage[MAX_AUDIT_MESSAGE_LENGTH];
				read(pipe_fd,garbage,MAX_AUDIT_MESSAGE_LENGTH);
			}
			continue;
		}
		//Using the size defined in the header, grab the data section
		vec_data[0].iov_len = hdr.size; 
		rc = readv(pipe_fd, vec_data, 1);
		if (rc == 0 || rc == -1) {
			syslog(LOG_ERR, "rc2 == %d(%s). Exitting", rc, strerror(errno));
			break;
		}

		//Create the record and add it to the list (if there is room)
		if(PrimaryNodeCounter >= NODECOUNT) {
			syslog(LOG_ERR,"* * * * * * * Memory top-stop hit - primary nodecounter is %d, max is %d.",PrimaryNodeCounter,NODECOUNT);
			continue;
		}

		new_rec=(Node *)malloc(sizeof(Node));
		if (new_rec != NULL) {
			new_rec->hdr.size = hdr.size;
			new_rec->hdr.hlen = hdr.hlen;
			new_rec->hdr.type = hdr.type;
			new_rec->hdr.ver = hdr.ver;
			new_rec->line=malloc(MAX_AUDIT_MESSAGE_LENGTH);
			strncpy(new_rec->line,(char *)data,hdr.size);
			new_rec->next= NULL;
			pthread_mutex_lock(&list_mutex);
			if (ptail) {
				ptail->next = new_rec;
				ptail = new_rec;
			} else {
				ptail = phead = new_rec;
			}
			PrimaryNodeCounter++;
			pthread_mutex_unlock(&list_mutex);
			pthread_cond_signal(&list_data_available);
		} else {
			syslog(LOG_ERR,"* * * * * * * Cannot allocate RAM for new event, event lost.");
			// Drop this event and sleep for a few seconds.
			sleep(5);
		}

	} while(!signaled);

	return 0;
}

// Given a line from auditd, it will return the serial number or zero on error
unsigned long line2serial(char *s)
{
	unsigned long serial;
	char *ptr,*tmp;

	if (strlen(s) < 20) return 0;
	errno = 0;
	tmp = strndupa(s, MAX_AUDIT_MESSAGE_LENGTH);
	if (!tmp) return 0;
	ptr = strchr(tmp+17, ':');
	if (ptr) {
		serial = strtoul(ptr+1, NULL, 10);
		*ptr = 0;
		if (errno)
			return 0;
	} else {
		serial = 0;
	}
	if (errno)
		return 0;
	return serial;
}

// Given two Nodes, it will remove the audit(...) section from the source record
// and add it to the destination. It will also modify the headers accordingly
int merge_records(Node *dest,Node *src)
{
	char *ptr=NULL;
	char newstring[MAX_AUDIT_MESSAGE_LENGTH];
	int skip=0;

	if (strlen((char *)src->line) < 20) return 0;
	ptr = (char *)src->line;
	while (ptr) {
		skip++;
		if (*ptr == ' ') break;
		else ptr++;
	}
	if (!ptr) return 0;
	ptr++;
	src->hdr.size -= skip;
	snprintf(newstring,MAX_AUDIT_MESSAGE_LENGTH,"%.*s %.*s",dest->hdr.size, (char *)dest->line, src->hdr.size, ptr);
	strncpy(dest->line,newstring,MAX_AUDIT_MESSAGE_LENGTH);
	dest->hdr.size = dest->hdr.size + src->hdr.size + 1;
	return 1;
}

// Spawned as a pthread, this function is used to pass the complete messages onto SnareDispatcher
void SnareDispatcher(int *WRITEPOINTER) {
	Node * mycurrent;
	int continueloop=1;
	int length,leftover=0;
	struct timespec timeout;
	time_t now;

	pthread_mutex_lock(&data_mutex);
	while(continueloop && !signaled) {
		Node *myprev=NULL;
		// Waiting for head to be valid
		time(&now);
		// Run this thread every second, or when signaled.
		timeout.tv_sec = now + 1;
		timeout.tv_nsec = 0;
		pthread_cond_timedwait(&list_data_available,&data_mutex,&timeout);
		// Ok, check for more data
//#############################################
		mycurrent = phead;
		while(mycurrent && !signaled) {
			int match=0;
			mycurrent->serial = line2serial((char *)mycurrent->line);
			if (mycurrent->serial == 0) {
				syslog(LOG_ERR,"BAD SERIAL, skipping [%.*s]", mycurrent->hdr.size, (char *)mycurrent->line);
				pthread_mutex_lock(&list_mutex);
				phead = mycurrent->next;
				free(mycurrent->line);
				free(mycurrent);
				if (!phead) ptail = NULL;
				mycurrent = phead;
				pthread_mutex_unlock(&list_mutex);
				continue;
			}
			time(&mycurrent->created);
			mycurrent->complete = 0;
			//syslog(LOG_NOTICE, "Node created, now processing");
			current=head;
			//current=tail;
			now=mycurrent->created;
			while (current != NULL) {
				//syslog(LOG_NOTICE,"serial compare %lu", current->serial);
				if (current->serial == mycurrent->serial) {
					//matching records, just add the lines together
					match++;

					if (merge_records(current,mycurrent)) {
						pthread_mutex_lock(&list_mutex);
						phead = mycurrent->next;
						free(mycurrent->line);
						free(mycurrent);
						PrimaryNodeCounter--;
						if (!phead) ptail = NULL;
						mycurrent = phead;
						pthread_mutex_unlock(&list_mutex);
					} else {
						if (mycurrent) {
							pthread_mutex_lock(&list_mutex);
							phead = mycurrent->next;
							free(mycurrent->line);
							free(mycurrent);
							PrimaryNodeCounter--;
							if (!phead) ptail = NULL;
							mycurrent = phead;
							pthread_mutex_unlock(&list_mutex);
						}
					}
					break;
				}
				current = current->next;
			}
			if (!match) {
				//syslog(LOG_NOTICE, "no match, adding new record");
				if(SecondaryNodeCounter >= NODECOUNT) {
					syslog(LOG_ERR,"* * * * * * * Memory top-stop hit - secondary nodecounter is %d, max is %d.",SecondaryNodeCounter,NODECOUNT);
					// For the moment, just log, and sleep for a few seconds.
					sleep(5);
					continue;
				}

				if(!mycurrent->hdr.size) {
					//No data, drop the record and continue straight away
					syslog(LOG_ERR,"ERROR: No Data");
					pthread_mutex_lock(&list_mutex);
					phead = mycurrent->next;
					free(mycurrent->line);
					free(mycurrent);
					if (!phead) ptail = NULL;
					mycurrent = phead;
					pthread_mutex_unlock(&list_mutex);
					continue;
				}

				pthread_mutex_lock(&list_mutex);
				phead = mycurrent->next;
				PrimaryNodeCounter--;
				if (!phead) {
					ptail = NULL;
				}
				pthread_mutex_unlock(&list_mutex);

				SecondaryNodeCounter++;
				if(tail) {
					tail->next=mycurrent;
				}
				tail=mycurrent;
				if(!head) {
					head=tail;
				}

				tail->next=NULL;
				mycurrent = phead;
			}
			if (SecondaryNodeCounter > leftover + 100) break;
		}
//#############################################
		mycurrent=head;
		while(mycurrent && !signaled) {
			struct iovec vec[2];

			// Check the timeout values
			if ((now - mycurrent->created) < NEW_REC_TIMEOUT) {
				myprev = mycurrent;
				mycurrent = mycurrent->next;
				continue;
			}

			vec[0].iov_base = (void*)&mycurrent->hdr;
			vec[0].iov_len = sizeof(mycurrent->hdr);

			// Next payload 
			vec[1].iov_base = mycurrent->line;
			vec[1].iov_len = mycurrent->hdr.size; 

			length=writev(*WRITEPOINTER,vec,2);
			if(length == -1) {
				// Oh dear - write error. Check that WRITEPOINTER is still ok
				if (!WRITEPOINTER) {
					syslog(LOG_ERR,"PTHREAD: * * * * * * * Write error %d in SnareDispatcher. WRITEPOINTER has failed, restarting.",errno);
					execlp("/etc/init.d/auditd","auditd","restart","&",(char *)0);
				} else {
					//Sleep and try again
					syslog(LOG_ERR,"PTHREAD: * * * * * * * Write error %d in SnareDispatcher thread within helper. Continuing after a 5 second delay.",errno);
					sleep(5);
				}
			} else if(length == 0) {
				syslog(LOG_ERR,"PTHREAD: * * * * * * * Write failure - 0 bytes sent. Continuing after a 5 second delay.");
				sleep(5);
			} else {
				// Just in case the variable has gone away
				if(!mycurrent) {
					syslog(LOG_NOTICE,"PTHREAD: Bad continue");
					continue;
				}
				Node *temp;
				temp=mycurrent->next;
				if (!myprev) {
					//must be the head
					head=temp;
				} else {
					myprev->next = temp;
				}
				free(mycurrent->line);
				free(mycurrent);
				SecondaryNodeCounter--;
				mycurrent=temp;
				if(!mycurrent) {
					tail=myprev;
				}
				if (!tail) {
					head=NULL;
				}
			}
		}
		leftover=SecondaryNodeCounter;
//#############################################
	}
}
