// SNARE - Audit / EventLog analysis and forwarding
// Copyright 2001-2003 InterSect Alliance Pty Ltd
// http://www.intersectalliance.com/
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// See Readme.txt file for more information.
//

#include <stdio.h>
#include <stdlib.h>
#include <winsock2.h>
#include <windows.h>
#include <time.h>
#include <tchar.h>
#include <limits.h>
#include <sys/stat.h>

#include "NTServApp.h"
#include "SnareSvr.h"


// Global Variables.
ThreadStruct	g_Info;
int				SNAREDEBUG=0;


CSnareSvrService::CSnareSvrService()
:CNTService("SNARESVR")
{
	// Initialize class members here
}

BOOL CSnareSvrService::OnInit()
{
	//Perform any initialization that needs to be done before intering the main loop
	char szError[MAX_EVENT];
	
	SNAREDEBUG=this->DEBUGSET;
	
	// Save this off so we can use it to write stuff to the registry.

	if( !InitWinsock( szError ) )
	{
		if(SNAREDEBUG) { DebugMsg(szError); }
		return FALSE;
	}

	m_hEventList[0] = CreateEvent(NULL, FALSE, FALSE, NULL); // 6161
	m_hEventList[1] = CreateEvent(NULL, FALSE, FALSE, NULL); // 514
	m_hEventList[2] = CreateEvent(NULL, FALSE, FALSE, NULL); // PIPE server
	m_hEventList[3] = CreateEvent(NULL, FALSE, FALSE, NULL); // PIPE writer

	if(m_hEventList[0] == NULL || m_hEventList[1] == NULL || m_hEventList[2] == NULL || m_hEventList[3] == NULL) {
		if(SNAREDEBUG) { DebugMsg("CreateEvent() failed"); }
		return FALSE;
	}

	
	if(SNAREDEBUG) { DebugMsg("SNARE Micro Server - Initialisation complete"); }
	// return FALSE here if initialization failed & the service shouldn't start
	return TRUE;
}

void CSnareSvrService::Run()
{
	// Define an eventlogrecord buffer of 8k.
	// Should be enough for an overwhelming majority of circumstances.
	TCHAR EventLogRecordBuffer[MAX_EVENT]="";
	TCHAR EventLogRecordBuffer2[MAX_EVENT]="";
	char *pEvent=NULL;

	OVERLAPPED Overlapped;	// PIPE variable
	OVERLAPPED OverlappedWrite;	// PIPE variable

	// Destination for log events. Default it to something safe.
	// TCHAR lpszDestination[512]="127.0.0.1";

	int retval;

	SOCKET snare_socket;	// 6161
	SOCKET syslog_socket;	// 514
	SOCKADDR_IN snare_in;
	SOCKADDR_IN syslog_in;

	SOCKADDR_IN recv_socket;
	int recv_socketlen=sizeof(recv_socket);
	
	char *ipbuff;
	char ipbuffer[16];

	char SourceName[256]="";
	char LastSourceName[256]="";
	char LogType[256]="";
	char LastLogType[256]="";
	int criticality=0;
	char Date[256]="";
	char StartDate[256]="";
	char CurrentDateTime[256]="";

	char Path[4096]=""; // 4k path
	char Filename[4096]="";

	struct tm *LocalTime1=NULL;
	int tyear=0,tday=0;

	time_t CurrentTime=0;
	time_t LastTime=0;
	
	int DateChanged=0;
	int LogTypeChanged=0;
	int SourceNameChanged=0;

	unsigned long SnarePackets=0;
	unsigned long SyslogPackets=0;


	DWORD dwWaitRes=0;
	BOOL WritePipeConnected=FALSE;
	
	FILE * OutputFile=(FILE *)NULL;

	hWritePipe=INVALID_HANDLE_VALUE;

	DebugMsg("SNARE Micro Server is now active");

	// READ in our configuration
	DWORD SnarePort=MyGetProfileDWORD("Config","SnarePort",6161);
	// OUTPUT_TYPE_DATE=1  OUTPUT_TYPE_DATE_SVR=2 
	// OUTPUT_TYPE_DATE_LOG=3  OUTPUT_TYPE_DATE_LOG_SVR=4
	DWORD OutputType=MyGetProfileDWORD("Config","OutputType",OUTPUT_TYPE_DATE);	
	
	if(MyGetProfileString("Config","Path",Path,sizeof(Path)) != ERROR_SUCCESS) {
		// Default to a reasonably sane value
		DebugMsg("No Path defined. Using SystemRoot\\System32\\LogFiles");
		ExpandEnvironmentStrings("%SystemRoot%\\system32\\LogFiles",Path,sizeof(Path));
	}
	if(strlen(Path) < 3) {
		DebugMsg("Path shorter than 3 characters. Using SystemRoot\\System32\\LogFiles");
		ExpandEnvironmentStrings("%SystemRoot%\\system32\\LogFiles",Path,sizeof(Path));
	}

	if(DirExists(Path) != TRUE) {
		DebugMsg("Directory specified in the confiruation registry (%s) does not exist! Cannot continue.",Path);
		return;
	}

	CurrentTime=time(NULL);
	LocalTime1 = localtime(&CurrentTime);
	strftime(StartDate, sizeof(StartDate),"%Y%m%d %H:%M:%S", LocalTime1);


	// Set up our network ports.
	snare_socket = WSASocket(AF_INET, SOCK_DGRAM, IPPROTO_UDP, NULL, 0, WSA_FLAG_OVERLAPPED);
	syslog_socket = WSASocket(AF_INET, SOCK_DGRAM, IPPROTO_UDP, NULL, 0, WSA_FLAG_OVERLAPPED);

	if(!snare_socket || !syslog_socket) {
		DebugMsg("Cannot open sockets - Exiting.");
		return;
	}

	snare_in.sin_family=AF_INET;
	snare_in.sin_port = htons((u_short)SnarePort);
	snare_in.sin_addr.s_addr=htonl(INADDR_ANY);

	if(bind(snare_socket,(SOCKADDR *)&snare_in, sizeof(snare_in)) != 0) {
		DebugMsg("Bind Failed - snare socket. Do you already have something running on the snare UDP port?");
		return;
	}

	syslog_in.sin_family=AF_INET;
	syslog_in.sin_port = htons((u_short)514);
	syslog_in.sin_addr.s_addr=htonl(INADDR_ANY);

	if(bind(syslog_socket,(SOCKADDR *)&syslog_in, sizeof(syslog_in)) != 0) {
		DebugMsg("Bind Failed - syslog socket. Do you already have something running on the syslog UDP port?");
		return;
	}

	if(WSAEventSelect(snare_socket,m_hEventList[0],FD_READ) != 0) {
		DebugMsg("WSAEventSelect Failed - I cannot initialise the events that I need in order to monitor the network sockets");
		return;
	}
	if(WSAEventSelect(syslog_socket,m_hEventList[1],FD_READ) != 0) {	
		DebugMsg("WSAEventSelect Failed - I cannot initialise the events that I need in order to monitor the network sockets");
		return;
	}

	if(SNAREDEBUG) { DebugMsg("Creating named pipe."); }

	// Create the pipe here. If we create it earlier, then there's a chance that
	// The GUI could lock onto the pipe before we have an event handler in place.
	hPipe = CreateNamedPipe("\\\\.\\PIPE\\SnareSvr",PIPE_ACCESS_OUTBOUND|FILE_FLAG_OVERLAPPED,
							PIPE_WAIT,1,5000,5000,0,NULL);

	hWritePipe = CreateNamedPipe("\\\\.\\PIPE\\SnareSvrGUI",PIPE_ACCESS_OUTBOUND|FILE_FLAG_OVERLAPPED,
							PIPE_WAIT,1,5000,5000,0,NULL);

	
	if (hPipe == INVALID_HANDLE_VALUE || hWritePipe == INVALID_HANDLE_VALUE) {
		DebugMsg("Error in CreateNamedPipe = %d - This is not a critical problem.",GetLastError());
		// return FALSE; // This is not a terminal problem.
	}


	// Monitor the pipe
	if (hPipe != INVALID_HANDLE_VALUE) {
		int piperc;
		Overlapped.hEvent = m_hEventList[2];
		// Despite what is written in the MS API, these values **_MUST_** be set
		Overlapped.Offset = 0;
		Overlapped.OffsetHigh = 0;

		if (ConnectNamedPipe(hPipe,&Overlapped) == 0) {
			piperc = GetLastError();
		}
	
		if ((piperc != ERROR_IO_PENDING) && 
			(piperc != ERROR_PIPE_CONNECTED))
		{
			// Not going to be able to do any pipe transactions.
			hPipe=INVALID_HANDLE_VALUE;

		}
		// OK pipe connected.
		if(SNAREDEBUG) { DebugMsg("Pipe connected."); }
	}

	if (hWritePipe != INVALID_HANDLE_VALUE) {
		int piperc;
		OverlappedWrite.hEvent = m_hEventList[3];
		// Despite what is written in the MS API, these values **_MUST_** be set
		OverlappedWrite.Offset = 0;
		OverlappedWrite.OffsetHigh = 0;

		if (ConnectNamedPipe(hPipe,&OverlappedWrite) == 0) {
			piperc = GetLastError();
		}
	
		if ((piperc != ERROR_IO_PENDING) && 
			(piperc != ERROR_PIPE_CONNECTED))
		{
			// Not going to be able to do any pipe transactions.
			hWritePipe=INVALID_HANDLE_VALUE;

		}
		// OK pipe connected.
		if(SNAREDEBUG) { DebugMsg("Write Pipe connected."); }
	}


	// Set the terminate flag to zero.
	// setting this value to TRUE will terminate the service,
	// and ask it to save it's current status.
	g_Info.bTerminate = FALSE;

	
	if(SNAREDEBUG) { DebugMsg("Entering main loop."); }
	// This is the service's main run loop.
    while (m_bIsRunning) 
	{
		// If we have been asked to terminate, do so.
		if(g_Info.bTerminate)
		{
			m_bIsRunning=0;
			break;
		}	
		dwWaitRes=WaitForMultipleObjects(4,m_hEventList,FALSE,5000);

		if(dwWaitRes != WAIT_FAILED && dwWaitRes != WAIT_TIMEOUT && g_Info.bTerminate == FALSE) {
			UINT EventTriggered=-1;

			if (dwWaitRes == WAIT_OBJECT_0) {
				EventTriggered=0;
				retval = recvfrom(snare_socket,EventLogRecordBuffer,sizeof(EventLogRecordBuffer),0,
					(struct sockaddr *)&recv_socket,&recv_socketlen);

				if(retval > 0) {
					EventLogRecordBuffer[retval]='\0';
				}

				ipbuff=inet_ntoa(recv_socket.sin_addr);
				if(ipbuff) {
					strncpy(ipbuffer,ipbuff,sizeof(ipbuffer));
				} else {
					strncpy(ipbuffer,"\0",sizeof(ipbuffer));
				}
				
				SnarePackets++;
				if(SnarePackets == ULONG_MAX) {
					SnarePackets = 1;
				}
			} else if (dwWaitRes == WAIT_OBJECT_0 + 1) {
				EventTriggered=1;
				retval = recvfrom(syslog_socket,EventLogRecordBuffer,sizeof(EventLogRecordBuffer),0,
					(struct sockaddr *)&recv_socket,&recv_socketlen);

				if(retval > 0) {
					EventLogRecordBuffer[retval]='\0';
				}

				ipbuff=inet_ntoa(recv_socket.sin_addr);
				if(ipbuff) {
					strncpy(ipbuffer,ipbuff,sizeof(ipbuffer));
				} else {
					strncpy(ipbuffer,"\0",sizeof(ipbuffer));
				}

				
				SyslogPackets++;
				if(SyslogPackets == ULONG_MAX) {
					SyslogPackets = 1;
				}
			} else if (dwWaitRes == WAIT_OBJECT_0 + 2) {
				// PIPE event.
				EventTriggered=2;

				int piperc;
				if(SNAREDEBUG) { DebugMsg("Pipe Event received"); }

				if (hPipe != INVALID_HANDLE_VALUE) {
					int pipecount=0;

					char str_temp[500] = "";
					DWORD dwWritten;

					// We can now write to the pipe, since we have a connected client
					_snprintf(str_temp,sizeof(str_temp),"SNARE Status:\n" \
							"SNARE is currently ACTIVE\nStarted at %s\nReceived on Snare Port: %ld\nReceived on Syslog Port: %ld\n\n",
							StartDate,SnarePackets,SyslogPackets);

					// If lpOverlapped is not NULL, lpNumberOfBytesWritten can be NULL.
					if (WriteFile(hPipe,str_temp,strlen(str_temp),NULL,&Overlapped) == 0)	{
						int GLError;
						GLError=GetLastError();
						if(SNAREDEBUG) { DebugMsg("WriteFile error = %d",GLError); }
						if (GLError == ERROR_IO_PENDING) {
							if (GetOverlappedResult(hPipe,&Overlapped,&dwWritten,TRUE) == 0) {
								if(SNAREDEBUG) { DebugMsg("Error in GetOverlappedResult = %d",GetLastError()); }
							} else {
								if(SNAREDEBUG) { DebugMsg("GetOverlappedResult bytes written = %d",dwWritten); }
							}
						}
					} else {
						if(SNAREDEBUG) { DebugMsg("Successfull WriteFile"); }
					}


					// Done. Close down the pipe, and reconnect it.
					DisconnectNamedPipe(hPipe);

					if (ConnectNamedPipe(hPipe,&Overlapped) == 0) {
						piperc = GetLastError();
					}
	
					if ((piperc != ERROR_IO_PENDING) && 
						(piperc != ERROR_PIPE_CONNECTED))
					{
						// Not going to be able to do any pipe transactions.
						hPipe=INVALID_HANDLE_VALUE;
					}
				}
				
				continue;
			} else if (dwWaitRes == WAIT_OBJECT_0 + 3) {
				EventTriggered=3;
				//ResetEvent(m_hEventList[3]);
				WritePipeConnected=TRUE;
			}

			DateChanged=0;
			if(EventTriggered == 0 || EventTriggered == 1 && EventLogRecordBuffer) {
				CurrentTime=time(NULL);
				if(LastTime) {
					LocalTime1 = localtime(&LastTime);
					if(LocalTime1) {
						tyear=LocalTime1->tm_year;
						tday=LocalTime1->tm_yday;
					} else {
						LastTime=0;
					}
				}
				LocalTime1 = localtime(&CurrentTime);

				if(LastTime) {
					if(LocalTime1) {
						// if(LocalTime1->tm_yday != LocalTime2->tm_yday || LocalTime1->tm_year != LocalTime2->tm_year) {
						if(tday != LocalTime1->tm_yday || tyear != LocalTime1->tm_year) {
							DateChanged=1;
							strftime(Date, sizeof(Date),"%Y%m%d", LocalTime1);
						}
					} else {
						DateChanged=1;
						strftime(Date, sizeof(Date),"%Y%m%d", LocalTime1);
					}
				} else {
					DateChanged=1;
					strftime(Date, sizeof(Date),"%Y%m%d", LocalTime1);
				}
				
				LastTime=CurrentTime;
				strftime(CurrentDateTime, sizeof(CurrentDateTime),"%d %b %Y %H:%M:%S", LocalTime1);
				

				int OutputLength=strlen(EventLogRecordBuffer);
				pEvent=EventLogRecordBuffer;
				if(OutputLength) {
					// Kill off any final newlines
					char *stringp=&EventLogRecordBuffer[OutputLength-1];
					while(stringp > EventLogRecordBuffer) {
						if(*stringp=='\n') {
							*stringp='\0';
							OutputLength--;
						} else {
							break;
						}
					}
					
					strncpy(SourceName,ipbuffer,sizeof(SourceName));
					
					criticality=0;

					if(EventTriggered == 0) {
						char *tabpos;
						int copysize=0;

						strncpy(LogType,"Generic",sizeof(LogType));
						// 6161. There should be source DNS, type (criticality)
						stringp=EventLogRecordBuffer;
						tabpos=strstr(EventLogRecordBuffer,"	");
						if(tabpos) {
							pEvent=tabpos+1;
							copysize=tabpos-stringp;
							if(copysize > sizeof(SourceName)) {
								copysize=sizeof(SourceName);
							}
							strncpy(SourceName,stringp,copysize);

							stringp=tabpos+1;
							tabpos=strstr(stringp,"	");
							if(tabpos) {
								pEvent=tabpos+1;
								copysize=tabpos-stringp;
								if(copysize > sizeof(LogType)) {
									copysize=sizeof(LogType);
								}
								strncpy(LogType,stringp,copysize);
								
								stringp=tabpos+1;
								if((*stringp>='0' && *stringp <= '9') && *(stringp+1) == '	') {
									char critstr[2];
									critstr[0]=*stringp;
									critstr[1]='\0';
									criticality=atoi(critstr);
									pEvent=stringp+2;
								}
							}
						}
					} else {
						pEvent=EventLogRecordBuffer;
						strncpy(LogType,"Syslog",sizeof(LogType));
						criticality=5;	// Syslog
					}

					if(pEvent > EventLogRecordBuffer+strlen(EventLogRecordBuffer)) {
						// Oh dear - past the end. Jump back.
						pEvent=EventLogRecordBuffer+strlen(EventLogRecordBuffer);
					}
					
					if(OutputType == OUTPUT_TYPE_DATE_SVR || OutputType == OUTPUT_TYPE_DATE_LOG_SVR) {
						if(strcmp(SourceName,LastSourceName)) {
							SourceNameChanged=1;
							strncpy(LastSourceName,SourceName,sizeof(LastSourceName));
						} else {
							SourceNameChanged=0;
						}
					} else {
						SourceNameChanged=0;
					}

					if(OutputType == OUTPUT_TYPE_DATE_LOG || OutputType == OUTPUT_TYPE_DATE_LOG_SVR) {
						if(strcmp(LogType,LastLogType)) {
							LogTypeChanged=1;
						strncpy(LastLogType,LogType,sizeof(LastLogType));
						} else {
							LogTypeChanged=0;
						}
					} else {
						LogTypeChanged=0;
					}

					if(DateChanged || SourceNameChanged || LogTypeChanged) {
						// Close the old filename.
						if(OutputFile) {
							fclose(OutputFile);
						}
						// OUTPUT_TYPE_DATE=1  OUTPUT_TYPE_DATE_SVR=2
						// OUTPUT_TYPE_DATE_LOG=3  OUTPUT_TYPE_DATE_LOG_SVR=4
						if(OutputType == OUTPUT_TYPE_DATE) {
							_snprintf(Filename,sizeof(Filename),"%s\\%s.log",Path,Date);
						} else if(OutputType == OUTPUT_TYPE_DATE_SVR) {
							_snprintf(Filename,sizeof(Filename),"%s\\%s-%s.log",Path,Date,SourceName);
						} else if(OutputType == OUTPUT_TYPE_DATE_LOG) {
							_snprintf(Filename,sizeof(Filename),"%s\\%s-%s.log",Path,Date,LogType);
						} else if(OutputType == OUTPUT_TYPE_DATE_LOG_SVR) {
							_snprintf(Filename,sizeof(Filename),"%s\\%s-%s-%s.log",Path,Date,LogType,SourceName);
						}


						if(SNAREDEBUG) { DebugMsg("Opening file %s\n",Filename); }
						OutputFile=fopen(Filename,"a");
						if(!OutputFile) {
							DebugMsg("Snare Cannot open output file %s",Filename);
							strcpy(LastSourceName,"");
							strcpy(LastLogType,"");
						}
					}

					if(SNAREDEBUG) { DebugMsg("Sending log to file: %s",EventLogRecordBuffer); }
					// Write log to file here.
					if(OutputFile) {
						
						fprintf(OutputFile,"%s\n",EventLogRecordBuffer);
						// fflush(OutputFile);
					}
					if(hWritePipe != INVALID_HANDLE_VALUE && WritePipeConnected==TRUE) {
						// Send data to the GUI, if the write-pipe is open.
						// Make sure we include the maximum criticality.
						DWORD dwWriteSize=0;

						if(SNAREDEBUG) {  DebugMsg("Sending data to the open GUI Pipe...."); }
						_snprintf(EventLogRecordBuffer2,sizeof(EventLogRecordBuffer2),"%s	%s	%s	%d	%s\0",CurrentDateTime,SourceName,LogType,criticality,pEvent);
						if(SNAREDEBUG) {
							int tsize=strlen(EventLogRecordBuffer2);
							if(SNAREDEBUG) { DebugMsg("Sending: [%s] - %d bytes",EventLogRecordBuffer2,tsize); }
						}
						if(WriteFile(hWritePipe,EventLogRecordBuffer2,strlen(EventLogRecordBuffer2)+1,&dwWriteSize,NULL)==0) {
							int piperc=0;
							if(SNAREDEBUG) {
								DebugMsg("Warning: Could not write to GUI pipe");
							}
							
							CancelIo(hWritePipe);
							CancelIo(hPipe);

							DisconnectNamedPipe(hWritePipe);

							if (ConnectNamedPipe(hWritePipe,&OverlappedWrite) == 0) {
								piperc = GetLastError();
							}
	
							if ((piperc != ERROR_IO_PENDING) && 
								(piperc != ERROR_PIPE_CONNECTED)) {
								// Not going to be able to do any pipe transactions.
								hWritePipe=INVALID_HANDLE_VALUE;
							}
							
							WritePipeConnected=FALSE;
						}
					}
				}
			}
		}
    }

	if(SNAREDEBUG) { DebugMsg("SNARE Closing"); }


	if(OutputFile) {
		fclose(OutputFile);
	}
	

	if(hPipe != INVALID_HANDLE_VALUE) {
		CloseHandle(hPipe);
	}
	if(hWritePipe != INVALID_HANDLE_VALUE) {
		CloseHandle(hWritePipe);
	}
	
	TerminateWinsock();

	if( m_hEventList[0] ) ::CloseHandle(m_hEventList[0]);
	if( m_hEventList[1] ) ::CloseHandle(m_hEventList[1]);
	if( m_hEventList[2] ) ::CloseHandle(m_hEventList[2]);
	if( m_hEventList[3] ) ::CloseHandle(m_hEventList[3]);
}


///////////////////////////////////////////////////////////////////
// InitWinsock
//              starts up winsock.dll or wsock32.dll
BOOL CSnareSvrService::InitWinsock( char *szError )
{
	WSAData wsData;

	if(!szError) return(FALSE);
	
	WORD wVersionRequested = WINSOCK_VERSION;
	
	if(WSAStartup(wVersionRequested, &wsData) != 0)
	{
		// :( error
		if( szError )
		{
			sprintf(szError,"WSAStartup failed: WSA ERROR: %d\r\n",
				WSAGetLastError());
			if(SNAREDEBUG) { if(szError) { DebugMsg(szError); } }
		}
		return FALSE;
	}
	
	// all is well
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////
// TerminateWinsock
//      call this function with the current socket or INVALID_SOCKET
void CSnareSvrService::TerminateWinsock()
{
	// cancel blocking calls, if any
	WSACancelBlockingCall();
		
	// unload winsock
	WSACleanup();
}


void CSnareSvrService::OnShutdown() {
	// Audit: Set that global variable to TRUE so that the threads
	// receive the terminate message
	g_Info.bTerminate=TRUE;
	
	if(SNAREDEBUG) { DebugMsg("SNARE Shutdown request received"); }

	// Call a fake event so that the subroutine	gets the shutdown message
	// through the setting of g_Info.bTerminate
	if(m_hEventList[2]) {
		::SetEvent(m_hEventList[2]);
	}
}


// Process user control requests
BOOL CSnareSvrService::OnUserControl(DWORD dwOpcode)
{
    switch (dwOpcode) {
    case SERVICE_CONTROL_USER + 0:

        // Save the current status in the registry
        SaveStatus();
        return TRUE;

    default:
        break;
    }
    return FALSE; // say not handled
}

// Save the current status in the registry
void CSnareSvrService::SaveStatus()
{
	// TODO: add code to save the sate of the service in the registry
}

int CSnareSvrService::DirExists(char * dir)
{
    WIN32_FIND_DATA data;
    HANDLE hFile = FindFirstFile(dir, &data);

	if(!dir) {
		return(-1);
	}

    if (hFile == INVALID_HANDLE_VALUE) { // directory doesn't exist
        return FALSE;
    } else {
        // is it folder or file?
        FindClose(hFile);
        if (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
            return TRUE;
		}
        return -1;
    }
}

