/* libaudit.c -- 
 * Copyright 2004-2006 Red Hat Inc., Durham, North Carolina.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *      Steve Grubb <sgrubb@redhat.com>
 *      Rickard E. (Rik) Faith <faith@redhat.com>
 */

#include "config.h"
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include <sys/poll.h>
#include <sys/utsname.h>
#include <netinet/in.h> // inet6 addrlen
#include <netdb.h>	// gethostbyname
#include <arpa/inet.h>	// inet_ntop
#include <fcntl.h>	/* O_NOFOLLOW needs gnu defined */

#include "libaudit.h"
#include "private.h"


int audit_archadded = 0;
int audit_syscalladded = 0;
unsigned int audit_elf = 0U;
static int audit_name_to_uid(const char *name, uid_t *uid);
static int audit_name_to_gid(const char *name, gid_t *gid);


int audit_priority(int xerrno)
{
	/* If they've compiled their own kernel and did not include
	 * the audit susbsystem, they will get ECONNREFUSED. We'll
	 * demote the message to debug so its not lost entirely. */
	if (xerrno == ECONNREFUSED)
		return LOG_DEBUG;
	else
		return LOG_WARNING;
}

int audit_request_status(int fd)
{
	int rc = audit_send(fd, AUDIT_GET, NULL, 0);
	if (rc < 0) 
		audit_msg(audit_priority(errno),
			"Error sending status request (%s)", strerror(-rc));
	return rc;
}

int audit_set_enabled(int fd, uint32_t enabled)
{
	int rc;
	struct audit_status s;

	memset(&s, 0, sizeof(s));
	s.mask    = AUDIT_STATUS_ENABLED;
	s.enabled = enabled;
	rc = audit_send(fd, AUDIT_SET, &s, sizeof(s));
	if (rc < 0)
		audit_msg(audit_priority(errno),
			"Error sending enable request (%s)", strerror(-rc));
	return rc;
}

/* 
 * This function will return 0 if auditing is NOT enabled and
 * 1 if enabled, and -1 on error.
 */
int audit_is_enabled(int fd)
{
	int rc;

	if (fd < 0)
		return 0;

	if ((rc = audit_request_status(fd)) > 0) {
		struct audit_reply rep;
		int i;
		int timeout = 40; /* tenths of seconds */
		struct pollfd pfd[1];

		pfd[0].fd = fd;
		pfd[0].events = POLLIN;

	        for (i = 0; i < timeout; i++) {
			do {
				rc = poll(pfd, 1, 100);
			} while (rc < 0 && errno == EINTR);

			rc = audit_get_reply(fd, &rep, GET_REPLY_NONBLOCKING,0);
			if (rc > 0) {
                	        /* If we get done or error, break out */
                        	if (rep.type == NLMSG_DONE || 
					rep.type == NLMSG_ERROR)
	                                break;

        	                /* If its not status, keep looping */
	                        if (rep.type != AUDIT_GET)
        	                        continue;

				/* Found it... */
				return rep.status->enabled;
			}
		}
	}
	if (rc == -ECONNREFUSED) {
		/* This is here to let people that build their own kernel
		   and disable the audit system get in. ECONNREFUSED is
		   issued by the kernel when there is "no on listening". */
		return 0;
	} else if (rc == -EPERM && getuid() != 0) {
		/* If we get this, then the kernel supports auditing
		 * but we don't have enough privilege to write to the
		 * socket. Therefore, we have already been authenticated
		 * and we are a common user. Just act as though auditing
		 * is not enabled. Any other error we take seriously.
		 * This is here basically to satisfy Xscreensaver. */
		return 0;
	}
	return -1;
}

int audit_set_failure(int fd, uint32_t failure)
{
	int rc;
	struct audit_status s;

	memset(&s, 0, sizeof(s));
	s.mask    = AUDIT_STATUS_FAILURE;
	s.failure = failure;
	rc = audit_send(fd, AUDIT_SET, &s, sizeof(s));
	if (rc < 0)
		audit_msg(audit_priority(errno), 
			"Error sending failure mode request (%s)", 
			strerror(-rc));
	return rc;
}

/*
 * This function returns -1 on error and 1 on success.
 */
int audit_set_pid(int fd, uint32_t pid, rep_wait_t wmode)
{
	struct audit_status s;
	struct audit_reply rep;
	struct pollfd pfd[1];
	int rc;

	memset(&s, 0, sizeof(s));
	s.mask    = AUDIT_STATUS_PID;
	s.pid     = pid;
	rc = audit_send(fd, AUDIT_SET, &s, sizeof(s));
	if (rc < 0) {
		audit_msg(audit_priority(errno), 
			"Error setting audit daemon pid (%s)", 
			strerror(-rc));
		return rc;
	}
	if (wmode == WAIT_NO)
		return 1;

	/* Now we'll see if there's any reply message. This only
           happens on error. It is not fatal if there is no message.
	   As a matter of fact, we don't do anything with the message
	   besides gobble it. */
	pfd[0].fd = fd;
	pfd[0].events = POLLIN;
	do {
		rc = poll(pfd, 1, 100);	/* .1 second */
	} while (rc < 0 && errno == EINTR);

	(void)audit_get_reply(fd, &rep, GET_REPLY_NONBLOCKING, 0);
	return 1;
}

int audit_set_rate_limit(int fd, uint32_t limit)
{
	int rc;
	struct audit_status s;

	memset(&s, 0, sizeof(s));
	s.mask       = AUDIT_STATUS_RATE_LIMIT;
	s.rate_limit = limit;
	rc = audit_send(fd, AUDIT_SET, &s, sizeof(s));
	if (rc < 0)
		audit_msg(audit_priority(errno),
			"Error sending rate limit request (%s)",
			strerror(-rc));
	return rc;
}

int audit_set_backlog_limit(int fd, uint32_t limit)
{
	int rc;
	struct audit_status s;

	memset(&s, 0, sizeof(s));
	s.mask          = AUDIT_STATUS_BACKLOG_LIMIT;
	s.backlog_limit = limit;
	rc = audit_send(fd, AUDIT_SET, &s, sizeof(s));
	if (rc < 0)
		audit_msg(audit_priority(errno),
			"Error sending backlog limit request (%s)", 
			strerror(-rc));
	return rc;
}

int audit_request_rules_list(int fd)
{
	int rc = audit_send(fd, AUDIT_LIST, NULL, 0);
	if (rc < 0)
		audit_msg(audit_priority(errno),
			"Error sending rule list request (%s)", 
			strerror(-rc));
	return rc;
}

int audit_request_signal_info(int fd)
{
	int rc = audit_send(fd, AUDIT_SIGNAL_INFO, NULL, 0);
	if (rc < 0)
		audit_msg(LOG_WARNING,
			"Error sending signal_info request (%s)",
			strerror(-rc));
	return rc;
}

/* This function takes a watch and allocates a transport memblk
 * suitable for sending to the kernel. The caller must free it.
 * Returns NULL on failure. */
static void *watch_to_transport(struct audit_watch *req, 
		unsigned int size)
{
	struct watch_transport r;

	void *p;
	void *memblk = malloc(size);
	if (!memblk)
		return NULL;

	r.dev_major = req->dev_major;
	r.dev_minor = req->dev_minor;
	r.perms = req->perms; 
	r.valid = 0;
	r.pathlen = req->namelen;
	r.fklen =  req->fklen;

	p = mempcpy(memblk, &r, sizeof(r));
	p = mempcpy(p, req->filterkey, req->fklen);
	memcpy(p, req->name, req->namelen);
	return memblk;
}

int audit_insert_watch(int fd, struct audit_watch *req)
{
	unsigned int size = sizeof(struct watch_transport) + req->namelen +
			req->fklen;
	void *memblk = watch_to_transport(req, size);

	int rc = audit_send(fd, AUDIT_WATCH_INS, memblk, size);
	if (rc < 0)
		audit_msg(audit_priority(errno),
			"Error sending watch insert request (%s)",
			strerror(-rc));
	free(memblk);
	return rc;
}

int audit_remove_watch(int fd, struct audit_watch *req)
{
	unsigned int size = sizeof(struct watch_transport) + req->namelen +
			req->fklen;
	void *memblk = watch_to_transport(req, size);

	int rc = audit_send(fd, AUDIT_WATCH_REM, memblk, size);
	if (rc < 0)
		audit_msg(audit_priority(errno),
			"Error sending watch remove request (%s)", 
			strerror(-rc));
	free(memblk);
	return rc;
}

int audit_request_watch_list(int fd)
{
	int rc = audit_send(fd, AUDIT_WATCH_LIST, NULL, 0);
	if (rc < 0 && rc != -EINVAL) {
		audit_msg(audit_priority(errno),
			"Error sending watch list request (%s)",
			strerror(-rc));
	}
	return rc;
}

int audit_add_rule(int fd, struct audit_rule *rule, int flags, int action)
{
	int rc;

	rule->flags  = flags;
	rule->action = action;
	rc = audit_send(fd, AUDIT_ADD, rule, sizeof(*rule));
	if (rc < 0)
		audit_msg(audit_priority(errno),
			"Error sending add rule request (%s)", 
			strerror(-rc));
	return rc;
}

int audit_delete_rule(int fd, struct audit_rule *rule, int flags, int action)
{
	int rc;

	rule->flags  = flags;
	rule->action = action;
	rc = audit_send(fd, AUDIT_DEL, rule, sizeof(*rule));
	if (rc < 0) {
		if (rc == -ENOENT)
			audit_msg(audit_priority(errno),
			"Error sending delete rule request (No rule matches)");
		else
			audit_msg(LOG_WARNING,
				"Error sending delete rule request (%s)",
				strerror(-rc));
	}
	return rc;
}


/*
 * This function will retreive the loginuid or -1 if there
 * is an error.
 */
uid_t audit_getloginuid(void)
{
	uid_t uid;
	int len, in;
	char buf[16];

	errno = 0;
	in = open("/proc/self/loginuid", O_NOFOLLOW|O_RDONLY);
	if (in < 0)
		return -1;
	do {
		len = read(in, buf, sizeof(buf));
	} while (len < 0 && errno == EINTR);
	close(in);
	if (len < 0 || len >= sizeof(buf))
		return -1;
	buf[len] = 0;
	errno = 0;
	uid = strtol(buf, 0, 10);
	if (errno)
		return -1;
	else
		return uid;
}

/*
 * This function returns 0 on success and 1 on failure
 */
int audit_setloginuid(uid_t uid)
{
	char loginuid[16];
	int o, count, rc = 0;

	errno = 0;
	count = snprintf(loginuid, sizeof(loginuid), "%u", uid);
	o = open("/proc/self/loginuid", O_NOFOLLOW|O_WRONLY|O_TRUNC);
	if (o >= 0) {
		int block, offset = 0;

		while (count > 0) {
			block = write(o, &loginuid[offset], (unsigned)count);

			if (block < 0) {
				if (errno == EINTR)
					continue;
				audit_msg(LOG_ERR, "Error writing loginuid");
				close(o);
				return 1;
			}
			offset += block;
			count -= block;
		}
		close(o);
	} else {
		audit_msg(LOG_ERR, "Error opening /proc/self/loginuid");
		rc = 1;
	}
	return rc;
}

// FIXME: make internal
int audit_send_message(int fd, int type, const char *message)
{
	if (fd >= 0) {
		int rc = audit_send(fd, type, message, strlen(message)+1);
		if (rc < 0)
			audit_msg(LOG_WARNING, 
				"Error sending user message request (%s)", 
				strerror(-rc));
		return rc;
	}
	else { /* If for some reason fd < 0, send to syslog */
		// This could be made configurable. Either syslog it or exit.
		syslog(LOG_ERR, "\
Can't send to audit system: %s pid=%d uid=%d loginuid=%d message=%s",
				audit_msg_type_to_name(type),
				getpid(), getuid(), audit_getloginuid(),
				message);
		return 0;
	}
}

/*
 * This function will send a user space message to the kernel.
 * It returns the sequence number which is > 0 on success  
 * or <= 0 on error. (pam uses this) This is the main audit sending
 * function now.
 */
int audit_send_user_message(int fd, int type, const char *message)
{
	int retry_cnt = 0;
	int rc;
retry:
	rc = audit_send(fd, type, message, strlen(message)+1);
	if (rc == -ECONNREFUSED) {
		/* This is here to let people that build their own kernel
		   and disable the audit system get in. ECONNREFUSED is
		   issued by the kernel when there is "no on listening". */
		return 0;
	} else if (rc == -EPERM && getuid() != 0) {
		/* If we get this, then the kernel supports auditing
		 * but we don't have enough privilege to write to the
		 * socket. Therefore, we have already been authenticated
		 * and we are a common user. Just act as though auditing
		 * is not enabled. Any other error we take seriously.
		 * This is here basically to satisfy Xscreensaver. */
		return 0;
	} else if (rc == -EINVAL) {
		/* If we get this, the kernel doesn't understand the
		 * netlink message type. This is most likely due to
		 * being an old kernel. Use the old message type. */
		if (type >= AUDIT_FIRST_USER_MSG && 
				type <= AUDIT_LAST_USER_MSG && !retry_cnt) {

			/* do retry */
			type = AUDIT_USER;
			retry_cnt++;
			goto retry;
		} 
	}
	return rc;
}

int audit_send_message_if_enabled(int fd, const char *message)
{	// Deprecated - do not change
	if (audit_is_enabled(fd) > 0)
		return audit_send(fd, AUDIT_USER, message, strlen(message)+1);
	else
		return 0;
}

/*
 * This function is used to send avc denial messages from user
 * space back into the kernel. (dbus, nscd, shadow uses this)
 * FIXME: deprecate
 */
int audit_log_avc(int fd, int type, const char *fmt, va_list ap)
{
	int rc, retry_cnt = 0;
	char buffer[1024];

	vsnprintf(buffer, sizeof(buffer), fmt, ap);
retry:
	rc = audit_send_message(fd, type, buffer);
	if (rc >= 0)
		return rc;
	if (rc == -ECONNREFUSED) {
		/* This is here to let people that build their own kernel
		   and disable the audit system get in. ECONNREFUSED is
		   issued by the kernel when there is "no on listening". */
		return 0;
	} else if (rc == -EPERM && getuid() != 0) {
		/* If we get this, then the kernel supports auditing
		 * but we don't have enough privilege to write to the
		 * socket. Therefore, we have already been authenticated
		 * and we are a common user. Just act as though auditing
		 * is not enabled. Any other error we take seriously. */
		syslog(LOG_ERR, "\
Can't send to audit system: %s pid=%d uid=%d loginuid=%d message=%s",
				audit_msg_type_to_name(type),
				getpid(), getuid(), audit_getloginuid(),
				buffer);
		return 0;
	} else if (rc == -EINVAL) {
		/* If we get this, the kernel doesn't understand the
		 * netlink message type. This is most likely due to
		 * being an old kernel. Use the old message type. */
		if ( !retry_cnt ) {
			/* do retry */
			type = AUDIT_USER;
			retry_cnt++;
			goto retry;
		} 
	}
	return rc;
}

int audit_vlog(int fd, const char *fmt, va_list ap)
{	// Deprecated - do not change
	char buffer[1024];

	vsnprintf(buffer, sizeof(buffer), fmt, ap);
	return audit_send_message(fd, AUDIT_USER, buffer);
}

// FIXME: deprecate
int audit_log(int fd, int type, const char *fmt, ...)
{
	int rc;
	va_list ap;

	va_start(ap, fmt);
	rc = audit_log_avc(fd, type, fmt, ap);
	va_end(ap);

	return rc;
}

/*
 * This function will log a user space message to the kernel
 * It returns the sequence number which is > 0 on success and
 * 0 or less on error. 
 * FIXME: deprecate
 */
int audit_log_if_enabled(int fd, int type, const char *fmt, ...)
{
	int rc;
	va_list ap;

	va_start(ap, fmt);
	rc = audit_log_avc(fd, type, fmt, ap);
	va_end(ap);

	return rc;
}

int audit_rule_syscall(struct audit_rule *rule, int scall)
{
	int word = AUDIT_WORD(scall);
	int bit  = AUDIT_BIT(scall);

	if (word >= (AUDIT_BITMASK_SIZE-1)) 
		return -1;
	rule->mask[word] |= bit;
	return 0;
}

int audit_rule_syscallbyname(struct audit_rule *rule,
                             const char *scall)
{
	int nr, i;
	int machine;

	if (!strcmp(scall, "all")) {
		for (i = 0; i < (AUDIT_BITMASK_SIZE-1); i++) 
			rule->mask[i] = ~0;
		return 0;
	}
	if (!audit_elf)
		machine = audit_detect_machine();
	else
		machine = audit_elf_to_machine(audit_elf);
	if (machine < 0)
		return -2;
	nr = audit_name_to_syscall(scall, machine);
	if (nr < 0) {
		if (isdigit(scall[0]))
			nr = strtol(scall, NULL, 0);
	}
	if (nr >= 0) 
		return audit_rule_syscall(rule, nr);
	return -1;
}

int audit_rule_field(struct audit_rule *rule, int field,
                     unsigned int value)
{
	if (rule->field_count >= AUDIT_MAX_FIELDS) 
		return -1;
	rule->fields[rule->field_count] = field;
	rule->values[rule->field_count] = value;
	++rule->field_count;
	return 0;
}

int audit_rule_fieldpair(struct audit_rule *rule, const char *pair, int flags)
{
	char       buf[128];
	const char *f = pair;
	const char *v = strchr(pair, '=');
	int        field;
	int        negated = 0;
	int        vlen;
    
	if (!f || !v || f == v)
		return -1;

	if (v > pair && v[-1] == '!')
        	negated = 1;

	snprintf(buf, sizeof(buf), "%*.*s", (int)(v-f-negated), 
			(int)(v-f-negated), f);
	audit_msg(LOG_DEBUG,"buf=%s\n", buf);
	if ((field = audit_name_to_field(buf)) < 0) 
		return -2;
	v++;
	audit_msg(LOG_DEBUG,"f%d%s%s\n", field, negated ? "!=" : "=", v);
	rule->fields[rule->field_count] = field | (negated ? AUDIT_NEGATE : 0);
	switch (field)
	{
		case AUDIT_UID:
		case AUDIT_EUID:
		case AUDIT_SUID:
		case AUDIT_FSUID:
		case AUDIT_LOGINUID:
			vlen = strlen(v);
			if (isdigit((char)*(v))) 
				rule->values[rule->field_count] = 
					strtol(v, NULL, 0);
			else if (vlen >= 2 && *(v)=='-' && 
						(isdigit((char)*(v+1)))) 
				rule->values[rule->field_count] = 
					strtol(v, NULL, 0);
			else {
				if (audit_name_to_uid(v, 
					&rule->values[rule->field_count])) {
					audit_msg(LOG_ERR, "Unknown user: %s",
						pair);
					return -2;
				}
			}
			break;
		case AUDIT_GID:
		case AUDIT_EGID:
		case AUDIT_SGID:
		case AUDIT_FSGID:
			if (isdigit((char)*(v))) 
				rule->values[rule->field_count] = 
					strtol(v, NULL, 0);
			else {
				if (audit_name_to_gid(v, 
					&rule->values[rule->field_count])) {
					audit_msg(LOG_ERR, "Unknown group: %s",
						pair);
					return -2;
				}
			}
			break;
		case AUDIT_ARCH:
			if (audit_syscalladded) 
				return -3;
			if (isdigit((char)*(v))) {
				int machine;

				errno = 0;
				audit_elf = strtoul(v, NULL, 0);
				if (errno) 
					return -5;

				// Make sure we have a valid mapping
				machine = audit_elf_to_machine(audit_elf);
				if (machine < 0)
					return -5;
			}
			else {
				// what do we want? i686, x86_64, ia64
				// or b64, b32
				int machine;
				unsigned int bits=0, elf;
				const char *arch=v;
				if (strcasecmp("b64", arch) == 0) {
					bits = __AUDIT_ARCH_64BIT;
					machine = audit_detect_machine();
				} else if (strcasecmp("b32", arch) == 0) {
					bits = ~__AUDIT_ARCH_64BIT;
					machine = audit_detect_machine();
				} 
				else 
					machine = audit_name_to_machine(arch);

				if (machine < 0) 
					return -4;

				/* Here's where we fixup the machine.
				 * for example, they give x86_64 & want 32 bits.
				 * we translate that to i686. */
				if (bits == ~__AUDIT_ARCH_64BIT &&
					machine == MACH_86_64)
						machine = MACH_X86;
				else if (bits == ~__AUDIT_ARCH_64BIT &&
					machine == MACH_PPC64)
						machine = MACH_PPC;
				else if (bits == ~__AUDIT_ARCH_64BIT &&
					machine == MACH_S390X)
						machine = MACH_S390;

				/* Check for errors - return -6 
				 * We don't allow 32 bit machines to specify 
				 * 64 bit. */
				switch (machine)
				{
					case MACH_X86:
						if (bits == __AUDIT_ARCH_64BIT)
							return -6;
						break;
					case MACH_IA64:
						if (bits == ~__AUDIT_ARCH_64BIT)
							return -6;
						break;
					case MACH_PPC:
						if (bits == __AUDIT_ARCH_64BIT)
							return -6;
						break;
					case MACH_S390:
						if (bits == __AUDIT_ARCH_64BIT)
							return -6;
						break;
					case MACH_86_64: /* fallthrough */
					case MACH_PPC64: /* fallthrough */
					case MACH_S390X: /* fallthrough */
						break;
					default:
						return -6;
				}

				/* OK, we have the machine type, now convert
				   to elf. */
				elf = audit_machine_to_elf(machine);
				if (elf == 0)
					return -5;

				audit_elf = elf;
			}
			rule->values[rule->field_count] = audit_elf; 
			audit_archadded = 1;
			break;
		case AUDIT_DEVMAJOR...AUDIT_SUCCESS:
			if (flags == AUDIT_FILTER_ENTRY)
				return -7;
			/* fallthrough */
		default:
			rule->values[rule->field_count] = strtol(v, NULL, 0);
			break;
	}
	++rule->field_count;
	return 0;
}

void audit_rule_free(struct audit_rule *rule)
{
	free(rule);
}

static int audit_name_to_uid(const char *name, uid_t *uid)
{
	struct passwd *pw;

	pw = getpwnam(name);
	if (pw == NULL) 
		return 1;

	memset(pw->pw_passwd, ' ', strlen(pw->pw_passwd));
	*uid = pw->pw_uid;
	return 0;
}

static int audit_name_to_gid(const char *name, gid_t *gid)
{
	struct group *gr;

	gr = getgrnam(name);
	if (gr == NULL) 
		return 1;
 
	*gid = gr->gr_gid;
	return 0;
}

int audit_detect_machine(void)
{
	struct utsname uts;
	if (uname(&uts) == 0)
//		strcpy(uts.machine, "x86_64");
		return audit_name_to_machine(uts.machine);
	return -1;
}

