/* auditctl.c -- 
 * Copyright 2004, 2005 Red Hat Inc., Durham, North Carolina.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *     Steve Grubb <sgrubb@redhat.com>
 *     Rickard E. (Rik) Faith <faith@redhat.com>
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>	/* strdup needs xopen define */
#include <getopt.h>
#include <time.h>
#include <sys/stat.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/utsname.h>
#include <fcntl.h>
#include <errno.h>
#include <libgen.h>	/* For basename */
#include "libaudit.h"
#include "auditctl-llist.h"
#include "private.h"

/* This define controls how many rule options we will allow when
 * reading a rule from a file. 64 fields are allowed by the kernel, so I
 * want to allow that plus a few entries for lists and other such items */
#define NUM_OPTIONS 72

/* This define controls the size of the line that we will request when
 * reading in rules from a file. We need to allow 64 fields. 25 bytes is 
 * the largest syscall name, so lets allow 1600 per line. 
 * Unrealistic - I know. 
 */
#define LINE_SIZE 1600

/* These are all related to the fs watch code */
#define WATCH_MAY_EXEC		1
#define WATCH_MAY_WRITE		2
#define WATCH_MAY_READ		4
#define WATCH_MAY_APPEND	8


/* Global functions */
static int handle_request(int status);
static void get_reply(void);
static int audit_print_reply(struct audit_reply *rep);
static int delete_all_rules(void);
static int delete_all_watches(void);

/* Global vars */
static int fd = -1;
static int list_requested = 0;
static int add = AUDIT_FILTER_UNSET, del = AUDIT_FILTER_UNSET, action = 0;
static int ins = 0, rem = 0, ignore = 0;
static struct audit_rule  rule;
static struct audit_watch watch;

extern int audit_archadded;
extern int audit_syscalladded;
extern unsigned int audit_elf;

/*
 * This function will reset everything used for each loop when loading 
 * a ruleset from a file.
 */
static int reset_vars(void)
{
	list_requested = 0;
	audit_syscalladded = 0;
	audit_archadded = 0;
	audit_elf = 0;
	add = AUDIT_FILTER_UNSET;
	del = AUDIT_FILTER_UNSET;
	action = 0;
	ins = 0;
	rem = 0;

	memset(&rule, 0, sizeof(rule));
	memset(&watch, 0, sizeof(watch));
	if ((fd = audit_open()) < 0) {
		fprintf(stderr, "Cannot open netlink audit socket\n");
		return 1;
	}
	return 0;
}

static void usage(void)
{
    printf("usage: auditctl [options]\n"
     "      -a <l,a>     Append rule to end of <l>ist with <a>ction\n"
     "      -A <l,a>     Add rule at beginning of <l>ist with <a>ction\n"
     "      -b <backlog> Set max number of outstanding audit buffers allowed\n"
     "                   Default=64\n"
     "      -d <l,a>     Delete rule from <l>ist with <a>ction\n"
     "                   l=task,entry,exit,user,watch a=never,possible,always\n"
     "      -D           Delete all rules and watches\n"
     "      -e [0|1]     Set enabled flag\n"
     "      -f [0..2]    Set failure flag\n"
     "                   0=silent 1=printk 2=panic\n"
     "      -F f=v       Build rule: field name, value\n"
     "      -h           Help\n"
     "      -i           Ignore errors when reading rules from file\n"
     "      -k <key>     Set filterkey on watch\n"
     "      -l           List rules\n"
     "      -m text      Send a user-space message\n"
     "      -p [r|w|x|a] Set permissions filter on watch\n"
     "                   r=read, w=write, x=execute, a=append\n"
     "      -r <rate>    Set limit in messages/sec (0=none)\n"
     "      -R <file>    read rules from file\n"
     "      -s           Report status\n"
     "      -S syscall   Build rule: syscall name or number\n"
     "      -t <syscall> Translate syscall number to syscall name\n"
     "      -v           Version\n"
     "      -w <path>    Insert watch at <path>\n"
     "      -W <path>    Remove watch at <path>\n"
     );
}

static int audit_rule_setup(const char *opt, int *flags, int *act)
{
	if (strstr(opt, "task")) 
		*flags = AUDIT_FILTER_TASK;
	else if (strstr(opt, "entry"))
		*flags = AUDIT_FILTER_ENTRY;
	else if (strstr(opt, "exit"))
		*flags = AUDIT_FILTER_EXIT;
	else if (strstr(opt, "user"))
		*flags = AUDIT_FILTER_USER;
	else if (strstr(opt, "watch"))
		*flags = AUDIT_FILTER_WATCH;
	else
		return 1;
	if (strstr(opt, "never"))
		*act = AUDIT_NEVER;
	else if (strstr(opt, "possible"))
		*act = AUDIT_POSSIBLE;
	else if (strstr(opt, "always"))
		*act = AUDIT_ALWAYS;
	else
		return 1;
	return 0;
}

/*
 * This function will check the path before accepting it. It returns
 * 1 on error and 0 on success.
 */
static int check_path(const char *path)
{
	char *ptr, *base;
	size_t nlen;
	size_t plen = strlen(path);
	if (plen >= PATH_MAX) {
		fprintf(stderr, "The path passed for the watch is too big\n");
		return 1;
	}
	ptr = strdup(path);
	base = basename(ptr);
	nlen = strlen(base);
	free(ptr);
	if (nlen > NAME_MAX) {
		fprintf(stderr, "The base name of the path is too big\n");
		return 1;
	}

	/* These are warnings, not errors */
	if (strstr(path, ".."))
		fprintf(stderr, 
			"Warning - relative path notation is not supported\n");
	if (strchr(path, '*') || strchr(path, '?'))
		fprintf(stderr, 
			"Warning - wildcard notation is not supported\n");

	return 0;
}

/*
 * Setup a watch.  The "name" of the watch in userspace will be the <path> to
 * the watch.  When this potential watch reaches the kernel, it will resolve
 * down to <name> (of terminating file or directory). 
 * Returns a 1 on success & -1 on failure.
 */
static int audit_setup_watch_name(struct audit_watch *req, const char *opt, 
	int *act)
{
	if (!req->name) {
		if (check_path(opt))
			return -1;
		req->name = strdup(opt);
		if (!req->name) {
			fprintf(stderr, "Out of memory\n");
			return -1;
		}

		// Trim trailing '/' should they exist
		req->namelen = strlen(req->name);
		if (req->namelen > 2 && req->name[req->namelen-1] == '/') {
			while (req->name[req->namelen-1] == '/' &&
					 req->namelen > 1) {
				req->name[req->namelen-1] = 0;
				req->namelen--;
			}
		}

		*act = 1;
		return 1;
	}
	fprintf(stderr, "There's already a watch given\n");
	return -1;
}

/*
 * Setup a filterkey for the watch.  
 * Returns a 1 on success & -1 on failure.
 */
static int audit_setup_filterkey(struct audit_watch *req, const char *opt)
{
	if (!req->filterkey) {
		req->filterkey = strdup(opt);
		if (!req->filterkey) {
			fprintf(stderr, "Out of memory\n");
			return -1;
		}
		req->fklen = strlen(opt);
		if (req->fklen > AUDIT_FILTERKEY_MAX) {
			fprintf(stderr, "The filterkey is too big\n");
			free(req->filterkey);
			return -1;
		}
		return 1;
	}
	fprintf(stderr, "There's already a filterkey given\n");
	return -1;
}

/*
 * Setup a watch permissions.
 * Returns a 1 on success & -1 on failure.
 */
static int audit_setup_perms(struct audit_watch *req, const char *opt)
{
	unsigned int i, len;

	len = strlen(opt);
	if (len > 4)
		return -1;

	for (i = 0; i < len; i++) {
		switch (tolower(opt[i])) {
			case 'r':
				if (!(req->perms & WATCH_MAY_READ))
					req->perms |= WATCH_MAY_READ;
				else
					goto audit_watch_setup_exit;
				break;
			case 'w':
				if (!(req->perms & WATCH_MAY_WRITE))
					req->perms |= WATCH_MAY_WRITE;
				else
					goto audit_watch_setup_exit;
				break;
			case 'x':
				if (!(req->perms & WATCH_MAY_EXEC))
					req->perms |= WATCH_MAY_EXEC;
				else
					goto audit_watch_setup_exit;
				break;
			case 'a':
				if (!(req->perms & WATCH_MAY_APPEND))
					req->perms |= WATCH_MAY_APPEND;
				else
					goto audit_watch_setup_exit;
				break;
			default:
				fprintf(stderr, 
					"Permission %c isn't supported\n", 
					opt[i]);
				return -1;
		}
	}
	return 1;

audit_watch_setup_exit:
	fprintf(stderr, "Permission %c was already given\n", opt[i]);
	return -1;
}

void audit_request_both_lists(int fd)
{
	int rc;
	if (audit_request_rules_list(fd) > 0) {
		list_requested = 1;
		get_reply();
	}
 
	if ((rc = audit_request_watch_list(fd)) > 0) {
		list_requested = 2;
		get_reply();
	} else if (rc == -EINVAL)
		printf("File system watches not supported\n");
}

/*
 * returns: -2 success - no reply, -1 error - noreply, 0 success - reply,
 *  > 0 success - rule
 */
static int setopt(int count, char *vars[])
{
    int c;
    int retval = 0;

    optind = 0;
    opterr = 0;
    while ((retval >= 0) && (c = getopt(count, vars,
			"hislDve:f:r:b:a:A:d:S:F:m:t:R:w:W:k:p:")) != EOF) {
	int flags = AUDIT_FILTER_UNSET;
        switch (c) {
        case 'h':
		usage();
		retval = -1;
		break;
	case 'i':
		ignore = 1;
		break;
        case 's':
		retval = audit_request_status(fd);
		if (retval <= 0)
			retval = -1;
		else
			retval = 0; /* success - just get the reply */
		break;
        case 'e':
		if (optarg && ((strcmp(optarg, "0") == 0) ||
				(strcmp(optarg, "1") == 0))) {
			if (audit_set_enabled(fd, strtoul(optarg,NULL,0)) > 0)
				audit_request_status(fd);
			else
				retval = -1;
		} else {
			fprintf(stderr, "Enable must be 0 or 1 was %s\n", 
				optarg);
			retval = -1;
		}
		break;
        case 'f':
		if (optarg && ((strcmp(optarg, "0") == 0) ||
				(strcmp(optarg, "1") == 0) ||
				(strcmp(optarg, "2") == 0))) {
			if (audit_set_failure(fd, strtoul(optarg,NULL,0)) > 0)
				audit_request_status(fd);
			else
				return -1;
		} else {
			fprintf(stderr, "Failure must be 0, 1, or 2 was %s\n", 
				optarg);
			retval = -1;
		}
		break;
        case 'r':
		if (optarg && isdigit(optarg[0])) { 
			uint32_t rate;
			errno = 0;
			rate = strtoul(optarg,NULL,0);
			if (errno) {
				fprintf(stderr, "Error converting rate\n");
				return -1;
			}
			if (audit_set_rate_limit(fd, rate) > 0)
				audit_request_status(fd);
			else
				return -1;
		} else {
			fprintf(stderr, "Rate must be a numeric value was %s\n",
				optarg);
			retval = -1;
		}
		break;
        case 'b':
		if (optarg && isdigit(optarg[0])) {
			uint32_t limit;
			errno = 0;
			limit = strtoul(optarg,NULL,0);
			if (errno) {
				fprintf(stderr, "Error converting backlog\n");
				return -1;
			}
			if (audit_set_backlog_limit(fd, limit) > 0)
				audit_request_status(fd);
			else
				return -1;
		} else {
			fprintf(stderr, 
				"Backlog must be a numeric value was %s\n", 
				optarg);
			retval = -1;
		}
		break;
        case 'l':
		if (count != 2) {
			fprintf(stderr,
				"List request should be given by itself\n");
			retval = -1;
			break;
		}
		audit_request_both_lists(fd);
		retval = -2;
		break;
        case 'a':
		if (strstr(optarg, "task") && audit_syscalladded) {
			fprintf(stderr, 
				"Syscall auditing requested for task list\n");
			retval = -1;
		}
		else if (audit_rule_setup(optarg, &add, &action) != 0) {
			fprintf(stderr, "Append rule - bad keyword %s\n", 
				optarg);
			retval = -1;
		}
		else
			retval = 1; /* success - please send */
		break;
        case 'A': 
		if (strstr(optarg, "task") && audit_syscalladded) {
			fprintf(stderr, 
			   "Error: syscall auditing requested for task list\n");
			retval = -1;
		}
		else if (audit_rule_setup(optarg, &add, &action) == 0) {
			add |= AUDIT_FILTER_PREPEND;
			retval = 1; /* success - please send */
		}
		else {
			fprintf(stderr, "Add rule - bad keyword %s\n", optarg);
			retval = -1;
		}
		break;
        case 'd': 
		if (audit_rule_setup(optarg, &del, &action) != 0) {
			fprintf(stderr, "Delete rule - bad keyword %s\n", 
				optarg);
			retval = -1;
		}
		else
			retval = 1; /* success - please send */
		break;
        case 'S':
		/* Do some checking to make sure that we are not adding a
		 * syscall rule to a list that does not make sense. */
		if (((add & (AUDIT_FILTER_MASK|AUDIT_FILTER_UNSET)) ==
				AUDIT_FILTER_TASK || (del & 
				(AUDIT_FILTER_MASK|AUDIT_FILTER_UNSET)) == 
				AUDIT_FILTER_TASK)) {
			fprintf(stderr, 
			  "Error: syscall auditing being added to task list\n");
			return -1;
		} else if (((add & (AUDIT_FILTER_MASK|AUDIT_FILTER_UNSET)) ==
				AUDIT_FILTER_USER || (del &
				(AUDIT_FILTER_MASK|AUDIT_FILTER_UNSET)) ==
				AUDIT_FILTER_USER)) {
			fprintf(stderr, 
			  "Error: syscall auditing being added to user list\n");
			return -1;
		} else {
			if (!audit_elf) {
				int machine;
				unsigned int elf;
				machine = audit_detect_machine();
				if (machine < 0) {
					fprintf(stderr, 
					    "Error detecting machine type");
					return -1;
				}
				elf = audit_machine_to_elf(machine);
                                if (elf == 0) {
					fprintf(stderr, 
					    "Error looking up elf type");
					return -1;
				}
				audit_elf = elf;
			}
		}
		switch (audit_rule_syscallbyname(&rule, optarg))
		{
			case 0:
				audit_syscalladded = 1;
				break;
			case -1:
				fprintf(stderr, "Syscall name unknown: %s\n", 
							optarg);
				retval = -1;
				break;
			case -2:
				fprintf(stderr, "Elf type unknown: 0x%x\n", 
							audit_elf);
				retval = -1;
				break;
		}
		break;
        case 'F':
		if (add != AUDIT_FILTER_UNSET)
			flags = add & AUDIT_FILTER_MASK;
		else if (del != AUDIT_FILTER_UNSET)
			flags = del & AUDIT_FILTER_MASK;
		// if the field is arch & there is a -t option...we 
		// can allow it
		else if ((optind >= count) || (strstr(optarg, "arch=") == NULL)
				 || (strcmp(vars[optind], "-t") != 0)) {
			fprintf(stderr, "List must be given before field\n");
			retval = -1;
			break;
		}	
		switch (audit_rule_fieldpair(&rule, optarg, flags))
		{
			case 0:
				break;
			case -1:
				fprintf(stderr, "-F missing = for %s\n", 
					optarg);
				retval = -1;
				break;
			case -2:
				fprintf(stderr, "-F unknown field: %s\n", 
					optarg);
				retval = -1;
				break;
			case -3:
				fprintf(stderr, 
					"-F %s must be before -S\n", 
					optarg);
				retval = -1;
				break;
			case -4:
				fprintf(stderr, 
					"-F %s machine type not found\n", 
					optarg);
				retval = -1;
				break;
			case -5:
				fprintf(stderr, 
					"-F %s elf mapping not found\n", 
					optarg);
				retval = -1;
				break;
			case -6:
				fprintf(stderr, 
			"-F %s requested bit level not supported by machine\n", 
					optarg);
				retval = -1;
				break;
			case -7:
				fprintf(stderr,
			 "Field %s cannot be checked at syscall entry\n",
					 optarg);
				retval = -1;
				break;
			case -8:
				fprintf(stderr, 
					"-F unknown message type - %s\n",
					 optarg);
				retval = -1;
				break;
			case -9:
				fprintf(stderr,
		 "msgtype field can only be used with exclude filter list\n");
				retval = -1;
				break;
			default:
				retval = -1;
				break;
		}
		break;
        case 'm':
//		if (audit_send_message(fd, AUDIT_USER, optarg) <= 0)
		if (audit_log_user_message( fd, AUDIT_USER, optarg, NULL, 
				NULL, NULL, 1) <=0)
			retval = -1;
		else
			return -2;  // success - no reply for this
		break;
        case 't':
		if (optarg && isdigit(optarg[0])) {
			const char *ptr, *mstr;
			int machine;
			if (audit_elf) {
				machine = audit_elf_to_machine(audit_elf);
				if (machine < 0) {
					fprintf(stderr, 
					    "Error translating machine type");
					return -1;
				}
			} else {
				machine = audit_detect_machine();
				if (machine < 0) {
					fprintf(stderr, 
					    "Error detecting machine type");
					return -1;
				}
			}
			mstr = audit_machine_to_name(machine);
			if (mstr == NULL)
				mstr = "Unknown";
                        ptr = audit_syscall_to_name(strtol(optarg, NULL,0), machine);
			if (ptr)
				printf("arch:%s, syscall:%s\n", mstr, ptr);
			else
				printf("Translate unknown value: %s\n", optarg);
			retval = -2; /* Success - no reply */
			break;
		}
		else  
			fprintf(stderr, 
				"Translate needs a numeric value was %s\n",
				 optarg);
		retval = -1; /* No reply needed */
		break;
	case 'R':
		fprintf(stderr, "Error - nested rule files not supported\n");
		retval = -1;
		break;
	case 'D':
		if (count != 2) {
			fprintf(stderr,
			    "Delete all request should be given by itself\n");
			retval = -1;
			break;
		}
		{
			// make sure both returns are accounted for
			int rc1 = delete_all_rules();
			retval = delete_all_watches();
			if (rc1)
				retval = rc1;
			if (!retval)
				retval = -2; // success no reply
		}
		audit_request_both_lists(fd);
		break;
	case 'w':
		if (optarg) 
			retval = audit_setup_watch_name(&watch, optarg, &ins);
		else {
			fprintf(stderr, "watch option needs a path\n");	
			retval = -1;
		}
		break;
	case 'W':
		if (optarg) 
			retval = audit_setup_watch_name(&watch, optarg, &rem);
		else {
			fprintf(stderr, "watch option needs a path\n");	
			retval = -1;
		}
		break;
	case 'k':
		if (!ins && !rem) {
			fprintf(stderr, 
			"filterkey option needs a watch given prior to it\n");	
			retval = -1;
		}
		else if (!optarg) {
			fprintf(stderr, "filterkey option needs a filter\n");	
			retval = -1;
		}
		else 
			retval =  audit_setup_filterkey(&watch, optarg);
		break;
	case 'p':
		if (!ins && !rem) {
			fprintf(stderr, 
			"permission option needs a watch given prior to it\n");	
			retval = -1;
		}
		else if (!optarg) {
			fprintf(stderr, "permission option needs a filter\n");	
			retval = -1;
		}
		else
			retval = audit_setup_perms(&watch, optarg);
		break;
	case 'v':
			printf("auditctl version %s\n", VERSION);
			retval = -2;
			break;
        default: 
		usage();
		retval = -1;
		break;
        }
    }
    /* catch extra args or errors where the user types "- s" */
    if (optind == 1)
	retval = -1;
    else if ((optind < count) && (retval != -1)) {
	fprintf(stderr, "parameter passed without an option given\n");	
	retval = -1;
    }
    return retval;
}

static char *get_line(FILE *f, char *buf)
{
	if (fgets_unlocked(buf, LINE_SIZE, f)) {
		/* remove newline */
		char *ptr = strchr(buf, 0x0a);
		if (ptr)
			*ptr = 0;
		return buf;
	}
	return NULL;
}

/*
 * This function reads the given file line by line and executes the rule.
 * It returns 0 if everything went OK, 1 if there are problems before reading
 * the file and -1 on error conditions after executing some of the rules.
 * It will abort reading the file if it encounters any problems.
 */
static int fileopt(const char *file)
{
	int i, tfd, rc, lineno = 1;
	struct stat st;
        FILE *f;
        char buf[LINE_SIZE];

	/* Does the file exist? */
	rc = open(file, O_RDONLY);
	if (rc < 0) {
		if (errno != ENOENT) {
			fprintf(stderr,"Error opening %s (%s)\n", 
				file, strerror(errno));
                        return 1;
                }
                fprintf(stderr, "file %s doesn't exist, skipping\n", file);
                return 0;
        }
        tfd = rc;

	/* Is the file permissions sane? */
	if (fstat(tfd, &st) < 0) {
		fprintf(stderr, "Error fstat'ing %s (%s)\n",
			file, strerror(errno));
		close(tfd);
		return 1;
	}
	if (st.st_uid != 0) {
		fprintf(stderr, "Error - %s isn't owned by root\n", file);
		close(tfd);
		return 1;
	} 
	if ((st.st_mode & S_IWOTH) == S_IWOTH) {
		fprintf(stderr, "Error - %s is world writable\n", file);
		close(tfd);
		return 1;
	}
	if (!S_ISREG(st.st_mode)) {
		fprintf(stderr, "Error - %s is not a regular file\n", file);
		close(tfd);
		return 1;
	}

        f = fdopen(tfd, "r");
        if (f == NULL) {
                fprintf(stderr, "Error - fdopen failed (%s)\n",
                        strerror(errno));
		close(tfd);
                return 1;
        }

	/* Read until eof, lineno starts as 1 */
	while (get_line(f, buf)) {
		char *options[NUM_OPTIONS];
		char *ptr;
		int idx=0;

		/* Weed out blank lines */
		while (buf[idx] == ' ')
			idx++;
		if (buf[idx] == 0) {
			lineno++;
			continue;
		}
		ptr = strtok(buf, " ");
		if (ptr == NULL)
			break;
		/* allow comments */
		if (ptr[0] == '#') {
			lineno++;
			continue;
		}
		i = 0;
		options[i++] = "auditctl";
		options[i++] = ptr;
		while( (ptr=strtok(NULL, " ")) && i<NUM_OPTIONS-1 ) {
			options[i++] = ptr;
		}
		options[i] = NULL;

		/* Parse it */
		if (reset_vars()) {
			fclose(f);
			return -1;
		}
		rc = setopt(i, options);

		/* handle reply or send rule */
		if (handle_request(rc) == -1) {
			fprintf(stderr, "There was an error in line %d of %s\n",
				lineno, file);
			if (!ignore) {
				fclose(f);
				return -1;
			}
		}
		lineno++;
	}
	fclose(f);
	return 0;
}

int main(int argc, char *argv[])
{
	int retval = 1;

	set_aumessage_mode(MSG_STDERR, DBG_NO);

	/* Check where the rules are coming from: commandline or file */
	if ((argc == 3) && (strcmp(argv[1], "-R") == 0)) {
#ifndef DEBUG
		/* Make sure we are root */
		if (getuid() != 0) {
			fprintf(stderr, 
				"You must be root to run this program.\n");
			return 4;
		}
#endif
		if (fileopt(argv[2]))
			return 1;
		else
			return 0;
	} else {
		if (argc == 1) {
			usage();
			return 1;
		}
#ifndef DEBUG
		/* Make sure we are root */
		if (getuid() != 0) {
			fprintf(stderr, 
				"You must be root to run this program.\n");
			return 4;
		}
#endif
		if (reset_vars())
			return 1;
		retval = setopt(argc, argv);
	}
	return handle_request(retval);
}

/*
 * This function is called after setopt to handle the return code.
 * On entry, status = 0 means just get the reply. Greater than 0 means we
 * are adding or deleting a rule or watch. -1 means an error occurred.
 * -2 means everything is OK and no reply needed. Even if there's an 
 * error, we need to call this routine to close up the audit fd.
 * The return code from this function is 0 success and -1 error.
 */
static int handle_request(int status)
{
	if (status == 0) {
		if (audit_syscalladded) {
			fprintf(stderr, "Error - no list specified\n");
			return -1;
		}
		get_reply();
	} else if (status == -2)
		status = 0;  // report success 
	else if (status > 0) {
		int rc;
		if (add != AUDIT_FILTER_UNSET) {
			// if !task add syscall any if not specified
			if ((add & AUDIT_FILTER_MASK) != AUDIT_FILTER_TASK && 
					audit_syscalladded != 1)
				audit_rule_syscallbyname(&rule, "all");
			rc = audit_add_rule(fd, &rule, add, action);
		}
		else if (del != AUDIT_FILTER_UNSET) {
			if ((del & AUDIT_FILTER_MASK) != AUDIT_FILTER_TASK && 
					audit_syscalladded != 1)
				audit_rule_syscallbyname(&rule, "all");
			rc = audit_delete_rule(fd, &rule, del, action);
		}
		else if (ins && !rem) 
			rc = audit_insert_watch(fd, &watch);
		else if (rem && !ins)
			rc = audit_remove_watch(fd, &watch);
		else {
        		usage();
	    		audit_close(fd);
			exit(1);
	    	}
		if (rc <= 0) 
			status = -1;
		else
			status = 0;
	} else 
		status = -1;

	audit_close(fd);
	fd = -1;
	return status;
}

/*
 * A reply from the kernel is expected. Get and display it.
 */
static void get_reply(void)
{
	int i, retval;
	int timeout = 40; /* loop has delay of .1 - so this is 4 seconds */
	struct audit_reply rep;
	fd_set read_mask;
	FD_ZERO(&read_mask);
	FD_SET(fd, &read_mask);

	for (i = 0; i < timeout; i++) {
		struct timeval t;

		t.tv_sec  = 0;
		t.tv_usec = 100000; /* .1 second */
		do {
			retval=select(fd+1, &read_mask, NULL, NULL, &t);
		} while (retval < 0 && errno == EINTR);
		// We'll try to read just in case
		retval = audit_get_reply(fd, &rep, GET_REPLY_NONBLOCKING, 0);
		if (retval > 0) {
			if (rep.type == NLMSG_ERROR && rep.error->error == 0) {
				i = 0;    /* reset timeout */
				continue; /* This was an ack */
			}
			
			if ((retval = audit_print_reply(&rep)) == 0) 
				break;
			if (retval == 1)
				i = 0; /* If getting more, reset timeout */
		}
	}
}

struct audit_watch *transport_to_watch(void *memblk)
{
	unsigned int offset;
	struct watch_transport *r = (struct watch_transport *)memblk;
	struct audit_watch *req = (struct audit_watch *)malloc(
					sizeof(struct audit_watch));
	if (req == NULL)
		return NULL;

	req->dev_major = r->dev_major;
	req->dev_minor = r->dev_minor;
	req->perms = r->perms;
	req->valid = r->valid;
	req->namelen = r->pathlen;
	req->fklen =  r->fklen;

	offset = sizeof(struct watch_transport);
	req->filterkey = strndup(memblk + offset, req->fklen);
	offset += req->fklen;
	req->name = strndup(memblk + offset, req->namelen);

	return req;
}

/*
 * This function interprets the reply and prints it to stdout. It returns
 * 0 if no more should be read and 1 to indicate that more messages of this
 * type may need to be read. 
 */
static int audit_print_reply(struct audit_reply *rep)
{
	unsigned int i;
	int first;
	int sparse;
	int machine = audit_detect_machine();

	audit_elf = 0; 
	switch (rep->type) {
		case NLMSG_NOOP:
			return 1;
		case NLMSG_DONE:
			if (list_requested == 1)
				printf("No rules\n");
			else if (list_requested == 2)
				printf("No watches\n");
			return 0;
		case NLMSG_ERROR: 
		        printf("NLMSG_ERROR %d (%s)\n",
				-rep->error->error, 
				strerror(-rep->error->error));
			return 0;
		case AUDIT_GET:
			printf("AUDIT_STATUS: enabled=%d flag=%d pid=%d"
			" rate_limit=%d backlog_limit=%d lost=%d backlog=%d\n",
			rep->status->enabled, rep->status->failure,
			rep->status->pid, rep->status->rate_limit,
			rep->status->backlog_limit, rep->status->lost,
			rep->status->backlog);
			return 0;
		case AUDIT_LIST:
			list_requested = 0;
			printf("AUDIT_LIST: %s,%s",
				audit_flag_to_name((int)rep->rule->flags),
				audit_action_to_name(rep->rule->action));
			for (i = 0; i < rep->rule->field_count; i++) {
				int field = rep->rule->fields[i];
				int negated = 0;

				if (field & AUDIT_NEGATE) {
					field &= ~AUDIT_NEGATE;
					negated  = 1;
				}
                
				const char *name = audit_field_to_name(field);
				if (name) {
					if (strcmp(name, "arch") == 0) { 
						audit_elf =rep->rule->values[i];
						printf(" %s%s%u", name, 
							negated ? "!=" : "=",
							(unsigned)rep->rule->values[i]);
					}
					else 
						printf(" %s%s%d", name, 
							negated ? "!=" : "=",
							rep->rule->values[i]);
				}
				else 
					printf(" f%d%s%d", rep->rule->fields[i],
						negated ? "!=" : "=",
						rep->rule->values[i]);
				if (rep->rule->values[i])
					printf(" (0x%x)", rep->rule->values[i]);
			}
			if ((rep->rule->flags & AUDIT_FILTER_MASK) != 
						AUDIT_FILTER_USER) {
				printf(" syscall=");
				for (sparse = 0, i = 0; i < AUDIT_BITMASK_SIZE;
					 i++)
					if (rep->rule->mask[i] !=
						 (unsigned int)~0)
						sparse = 1;
				if (!sparse) {
					printf("all");
				} else for (first = 1, i = 0;
					i < AUDIT_BITMASK_SIZE * 32; i++) {
					int word = AUDIT_WORD(i);
					int bit  = AUDIT_BIT(i);
					if (rep->rule->mask[word] & bit) {
						const char *ptr;
						if (audit_elf)
							machine = 
							audit_elf_to_machine(
								audit_elf);
						if (machine < 0)
							ptr = 0;
						else
							ptr = 
							audit_syscall_to_name(i, 
							machine);
						if (ptr)
							printf("%s%s", 
							first ? "" : ",", ptr);
						else
							printf("%s%d", 
							first ? "" : ",", i);
						first = 0;
					}
				}
			}
			printf("\n");
			return 1; /* get more messages until NLMSG_DONE */
		case AUDIT_WATCH_LIST:
		{
			char perms[5];
			struct audit_watch *req=transport_to_watch(rep->watch);
			list_requested = 0;
			if (!req) {
				fputs("No memory receiving watch list\n", 
						stderr);
				return 0;
			}
			perms[0] = 0;
		        if (req->perms & WATCH_MAY_READ)
				strcat(perms, "r");
		        if (req->perms & WATCH_MAY_WRITE)
				strcat(perms, "w");
		        if (req->perms & WATCH_MAY_EXEC)
				strcat(perms, "x");
		        if (req->perms & WATCH_MAY_APPEND)
				strcat(perms, "a");
			printf("AUDIT_WATCH_LIST: dev=%u:%u, path=%s, "
				"filterkey=%s, perms=%s, valid=%d\n",
				req->dev_major, req->dev_minor, req->name,
				req->filterkey, perms, req->valid);
			free(req->name);
			free(req->filterkey);
			free(req);
			return 1; /* get more messages until NLMSG_DONE */
		}
		default:
			printf("Unknown: type=%d, len=%d\n", rep->type, 
				rep->nlh->nlmsg_len);
			return 0;
	}
}

/* Returns 0 for success and -1 for failure */
static int delete_all_rules(void)
{
	int seq, i;
	int timeout = 40; /* tenths of seconds */
	struct audit_reply rep;
	fd_set read_mask;

	/* list the rules */
	seq = audit_request_rules_list(fd);
	if (seq <= 0) 
		return -1;

	FD_ZERO(&read_mask);
	FD_SET(fd, &read_mask);

	for (i = 0; i < timeout; i++) {
		struct timeval t;
		int rc;

		t.tv_sec  = 0;
		t.tv_usec = 100000; /* .1 second */
		do {
			rc = select(fd+1, &read_mask, NULL, NULL, &t);
		} while (rc < 0 && errno == EINTR);
		// We'll try to read just in case
		rc = audit_get_reply(fd, &rep, GET_REPLY_NONBLOCKING, 0);
		if (rc > 0) {
			/* Reset timeout */
			i = 0;

			/* Don't make decisions based on wrong packet */
			if (rep.nlh->nlmsg_seq != seq)
				continue;

			/* If we get done or error, break out */
			if (rep.type == NLMSG_DONE)
				break;

			if (rep.type == NLMSG_ERROR && rep.error->error) {
				fprintf(stderr, 
					"Error receiving rules list (%s)\n", 
					strerror(-rep.error->error));
				return -1;
			}

			/* If its not what we are expecting, keep looping */
			if (rep.type != AUDIT_LIST)
				continue;

			/* Found it, bounce it right back with delete */
			rc = audit_send(fd, AUDIT_DEL, rep.rule, 
					sizeof(struct audit_rule));
			if (rc < 0) {
				fprintf(stderr, "Error deleting rule (%s)\n",
					strerror(-rc)); 
				return -1;
			}
		}
	}

	return 0;
}

/* This function deletes all watches. It returns: -2 if watches are not 
 * supported, -1 if there was an error, and 0 for success */
static int delete_all_watches(void)
{
	int i, seq;
	int timeout = 40; /* tenths of seconds */
	struct audit_reply rep;
	llist entries;	/* list of watches */
	const lnode *nptr;
	fd_set read_mask;

	/* list the rules */
	seq = audit_request_watch_list(fd);
	if (seq <= 0) {
		// EINVAL means that it does have file audit support
		if (seq == -EINVAL)
			return -2;
		else
			return -1;
	}

	FD_ZERO(&read_mask);
	FD_SET(fd, &read_mask);	
	list_create(&entries);

	for (i = 0; i < timeout; i++) {
		struct timeval t;
		int rc;

		t.tv_sec  = 0;
		t.tv_usec = 100000; /* .1 second */
		do {
			rc = select(fd+1, &read_mask, NULL, NULL, &t);
		} while (rc < 0 && errno == EINTR);
		// We'll try to read just in case
		rc = audit_get_reply(fd, &rep, GET_REPLY_NONBLOCKING, 0);
		if (rc > 0) {
			lnode n;
		
			/* Reset timeout */
			i = 0;

			/* Don't make decisions based on wrong packet */
			if (rep.nlh->nlmsg_seq != seq)
				continue;

			/* If we get done or error, break out */
			if (rep.type == NLMSG_DONE )
				break;

			if (rep.type == NLMSG_ERROR && rep.error->error) {
				/* If its the first one & EINVAL assume we
				   are on a kernel without watch support */
				if (rep.error->error == -EINVAL && 
							entries.cnt == 0)
					return -2;
				fprintf(stderr,
					"Error receiving watch list (%s)\n", 
					strerror(-rep.error->error));
				return -1;
			}

			/* If its not what we are expecting, keep looping */
			if (rep.type != AUDIT_WATCH_LIST)
				continue;

			/* Add to linked list */
			n.w = malloc(rep.len);
			if (n.w == NULL) {
				fprintf(stderr, "No memory\n");
				return -1;
			}
			memcpy(n.w, rep.watch, rep.len);
			n.size = rep.len;
			list_append(&entries, &n);
		}
	}

	/* loop through linked list and send */
	if (entries.cnt) {
		list_first(&entries);
		nptr = list_get_cur(&entries);
		do {
			int rc = audit_send(fd, AUDIT_WATCH_REM, nptr->w, 
					nptr->size);
			if (rc < 0) {
				fprintf(stderr, "Error deleting watch (%s)\n", 
					strerror(-rc));
				list_clear(&entries);
				return -1;
			}
		} while ((nptr=list_next(&entries)));

		/* Done with list - delete it */	
		list_clear(&entries);
	}

	return 0;
}

