/* auditd-config.c -- 
 * Copyright 2004-06 Red Hat Inc., Durham, North Carolina.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *   Steve Grubb <sgrubb@redhat.com>
 * 
 */

#include "config.h"
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>
#include <ctype.h>
#include <stdlib.h>
#include <netdb.h>
#include <fcntl.h>	/* O_NOFOLLOW needs gnu defined */
#include <libgen.h>
#include "auditd-config.h"
#include "libaudit.h"
#include "private.h"

/* Local prototypes */
struct nv_pair
{
	const char *name;
	const char *value;
};

struct kw_pair 
{
	const char *name;
	int (*parser)(const char *, int, struct daemon_conf *);
};

struct nv_list
{ 
	const char *name;
	int option;
};

static char *get_line(FILE *f, char *buf);
static int nv_split(char *buf, struct nv_pair *nv);
static const struct kw_pair *kw_lookup(const char *val);
static int log_file_parser(const char *val, int line, 
		struct daemon_conf *config);
static int num_logs_parser(const char *val, int line, 
		struct daemon_conf *config);
static int qos_parser(const char *val, int line,
		struct daemon_conf *config);
static int dispatch_parser(const char *val, int line,
		struct daemon_conf *config);
static int max_log_size_parser(const char *val, int line, 
		struct daemon_conf *config);
static int max_log_size_action_parser(const char *val, int line, 
		struct daemon_conf *config);
static int log_format_parser(const char *val, int line, 
		struct daemon_conf *config);
static int flush_parser(const char *val, int line, struct daemon_conf *config);
static int freq_parser(const char *val, int line, struct daemon_conf *config);
static int space_left_parser(const char *val, int line, 
		struct daemon_conf *config);
static int space_action_parser(const char *val, int line, 
		struct daemon_conf *config);
static int action_mail_acct_parser(const char *val, int line, 
		struct daemon_conf *config);
static int admin_space_left_parser(const char *val, int line, 
		struct daemon_conf *config);
static int admin_space_left_action_parser(const char *val, int line, 
		struct daemon_conf *config);
static int disk_full_action_parser(const char *val, int line, 
		struct daemon_conf *config);
static int disk_error_action_parser(const char *val, int line, 
		struct daemon_conf *config);
static int priority_boost_parser(const char *val, int line, struct daemon_conf *config);
static int sanity_check(struct daemon_conf *config);

static const struct kw_pair keywords[] = 
{
  {"log_file",                 log_file_parser },
  {"log_format",               log_format_parser },
  {"flush",                    flush_parser },
  {"freq",                     freq_parser },
  {"num_logs",                 num_logs_parser },
  {"dispatcher",               dispatch_parser },
  {"disp_qos",                 qos_parser },
  {"max_log_file",             max_log_size_parser },
  {"max_log_file_action",      max_log_size_action_parser },
  {"space_left",               space_left_parser },
  {"space_left_action",        space_action_parser },
  {"action_mail_acct",         action_mail_acct_parser },
  {"admin_space_left",         admin_space_left_parser },
  {"admin_space_left_action",  admin_space_left_action_parser },
  {"disk_full_action",         disk_full_action_parser },
  {"disk_error_action",        disk_error_action_parser },
  {"priority_boost",           priority_boost_parser },
  { NULL,                      NULL }
};

static const struct nv_list log_formats[] =
{
  {"raw",  LF_RAW },
  {"nolog", LF_NOLOG },
  { NULL,  0 }
};

static const struct nv_list flush_techniques[] =
{
  {"none",        FT_NONE },
  {"incremental", FT_INCREMENTAL },
  {"data",        FT_DATA },
  {"sync",        FT_SYNC },
  { NULL,         0 }
};

// Ideas: e-mail, run command
static const struct nv_list failure_actions[] =
{
  {"ignore",  FA_IGNORE },
  {"syslog",  FA_SYSLOG },
  {"email",   FA_EMAIL },
  {"suspend", FA_SUSPEND },
  {"single",  FA_SINGLE },
  {"halt",    FA_HALT },
  { NULL,     0 }
};

// Future ideas: e-mail, run command
static const struct nv_list size_actions[] =
{
  {"ignore",  SZ_IGNORE },
  {"syslog",  SZ_SYSLOG },
  {"suspend", SZ_SUSPEND },
  {"rotate",  SZ_ROTATE },
  {"keep_logs", SZ_KEEP_LOGS},
  { NULL,     0 }
};

static const struct nv_list qos_options[] =
{
  {"lossy",     QOS_NON_BLOCKING },
  {"lossless",  QOS_BLOCKING },
  { NULL,     0 }
};

const char *email_command = "/usr/lib/sendmail";

/*
 * Set everything to its default value
*/
static void clear_config(struct daemon_conf *config)
{
	config->qos = QOS_NON_BLOCKING;
	config->sender_uid = 0;
	config->sender_pid = 0;
	config->log_file = strdup("/var/log/audit/audit.log");
	config->log_format = LF_RAW;
	config->priority_boost = 3;
	config->flush =  FT_NONE;
	config->freq = 0;
	config->num_logs = 0L;
	config->dispatcher = NULL;
	config->max_log_size = 0L;
	config->max_log_size_action = SZ_IGNORE;
	config->space_left = 0L;
	config->space_left_action = FA_IGNORE;
	config->action_mail_acct = strdup("root");
	config->admin_space_left= 0L;
	config->admin_space_left_action = FA_IGNORE;
	config->disk_full_action = FA_IGNORE;
	config->disk_error_action = FA_SYSLOG;
}

static log_test_t log_test = TEST_AUDITD;
int load_config(struct daemon_conf *config, log_test_t lt)
{
	int fd, rc, lineno = 1;
	struct stat st;
	FILE *f;
	char buf[128];

	clear_config(config);
	log_test = lt;

	/* open the file */
	rc = open(CONFIG_FILE, O_NOFOLLOW|O_RDONLY);
	if (rc < 0) {
		if (errno != ENOENT) {
			audit_msg(LOG_ERR, "Error opening config file (%s)", 
				strerror(errno));
			return 1;
		}
		audit_msg(LOG_WARNING,
			"Config file %s doesn't exist, skipping", CONFIG_FILE);
		return 0;
	}
	fd = rc;

	/* check the file's permissions: owned by root, not world writable,
	 * not symlink.
	 */
	audit_msg(LOG_DEBUG, "Config file %s opened for parsing", 
			CONFIG_FILE);
	if (fstat(fd, &st) < 0) {
		audit_msg(LOG_ERR, "Error fstat'ing config file (%s)", 
			strerror(errno));
		close(fd);
		return 1;
	}
	if (st.st_uid != 0) {
		audit_msg(LOG_ERR, "Error - %s isn't owned by root", 
			CONFIG_FILE);
		close(fd);
		return 1;
	}
	if ((st.st_mode & S_IWOTH) == S_IWOTH) {
		audit_msg(LOG_ERR, "Error - %s is world writable", 
			CONFIG_FILE);
		close(fd);
		return 1;
	}
	if (!S_ISREG(st.st_mode)) {
		audit_msg(LOG_ERR, "Error - %s is not a regular file", 
			CONFIG_FILE);
		close(fd);
		return 1;
	}

	/* it's ok, read line by line */
	f = fdopen(fd, "r");
	if (f == NULL) {
		audit_msg(LOG_ERR, "Error - fdopen failed (%s)", 
			strerror(errno));
		close(fd);
		return 1;
	}

	while (get_line(f, buf)) {
		// convert line into name-value pair
		const struct kw_pair *kw;
		struct nv_pair nv;
		rc = nv_split(buf, &nv);
		switch (rc) {
			case 0: // fine
				break;
			case 1: // not the right number of tokens.
				audit_msg(LOG_ERR, 
				"Wrong number of arguments for line %d in %s", 
					lineno, CONFIG_FILE);
				break;
			case 2: // no '=' sign
				audit_msg(LOG_ERR, 
					"Missing equal sign for line %d in %s", 
					lineno, CONFIG_FILE);
				break;
			default: // something else went wrong... 
				audit_msg(LOG_ERR, 
					"Unknown error for line %d in %s", 
					lineno, CONFIG_FILE);
				break;
		}
		if (nv.name == NULL) {
			lineno++;
			continue;
		}
		if (nv.value == NULL) {
			fclose(f);
			return 1;
		}

		/* identify keyword or error */
		kw = kw_lookup(nv.name);
		if (kw->name == NULL) {
			audit_msg(LOG_ERR, 
				"Unknown keyword \"%s\" in line %d of %s", 
				nv.name, lineno, CONFIG_FILE);
			fclose(f);
			return 1;
		}

		/* dispatch to keyword's local parser */
		rc = kw->parser(nv.value, lineno, config);
		if (rc != 0) {
			fclose(f);
			return 1; // local parser puts message out
		}

		lineno++;
	}

	fclose(f);
	if (lineno > 1)
		return sanity_check(config);
	return 0;
}

static char *get_line(FILE *f, char *buf)
{
	if (fgets_unlocked(buf, 128, f)) {
		/* remove newline */
		char *ptr = strchr(buf, 0x0a);
		if (ptr)
			*ptr = 0;
		return buf;
	}
	return NULL;
}

static int nv_split(char *buf, struct nv_pair *nv)
{
	/* Get the name part */
	char *ptr;

	nv->name = NULL;
	nv->value = NULL;
	ptr = strtok(buf, " ");
	if (ptr == NULL)
		return 0; /* If there's nothing, go to next line */
	if (ptr[0] == '#')
		return 0; /* If there's a comment, go to next line */
	nv->name = ptr;

	/* Check for a '=' */
	ptr = strtok(NULL, " ");
	if (ptr == NULL)
		return 1;
	if (strcmp(ptr, "=") != 0)
		return 2;

	/* get the value */
	ptr = strtok(NULL, " ");
	if (ptr == NULL)
		return 1;
	nv->value = ptr;

	/* Make sure there's nothing else */
	ptr = strtok(NULL, " ");
	if (ptr)
		return 1;

	/* Everything is OK */
	return 0;
}

static const struct kw_pair *kw_lookup(const char *val)
{
	int i = 0;
	while (keywords[i].name != NULL) {
		if (strcasecmp(keywords[i].name, val) == 0)
			break;
		i++;
	}
	return &keywords[i];
}
 
static int log_file_parser(const char *val, int line,struct daemon_conf *config)
{
	char *dir = NULL, *tdir, *base;
	DIR *d;
	int fd, mode;
	struct stat buf;

	audit_msg(LOG_DEBUG, "log_file_parser called with: %s", val);

	/* split name into dir and basename. */
	tdir = strdup(val);
	if (tdir)
		dir = dirname(tdir);
	if (dir == NULL || strlen(dir) < 4) { //  '/var' is shortest dirname
		audit_msg(LOG_ERR, 
			"The directory name: %s is too short - line %d", 
			dir, line);
		free((void *)tdir);
		return 1;
	}

	base = basename((char *)val);
	if (base == 0 || strlen(base) == 0) {
		audit_msg(LOG_ERR, "The file name: %s is too short - line %d", 
			base, line);
		free((void *)tdir);
		return 1;
	}
	
	/* verify the directory path exists */
	d = opendir(dir);
	if (d == NULL) {
		audit_msg(LOG_ERR, "Could not open dir %s (%s)", dir, 
			strerror(errno));
		free((void *)tdir);
		return 1;
	}
	free((void *)tdir);
	closedir(d);

	/* if the file exists, see that its regular, owned by root, 
	 * and not world anything */
	if (log_test == TEST_AUDITD)
		mode = O_APPEND;
	else
		mode = O_RDONLY;

	fd = open(val, mode);
	if (fd < 0) {
		if (errno == ENOENT) {
			fd = create_log_file(val);
			if (fd < 0) 
				return 1;
		} else {
			audit_msg(LOG_ERR, "Unable to open %s (%s)", val, 
					strerror(errno));
			return 1;
		}
	}
	if (fstat(fd, &buf) < 0) {
		audit_msg(LOG_ERR, "Unable to stat %s (%s)",
					val, strerror(errno));
		close(fd);
		return 1;
	}
	close(fd);
	if (!S_ISREG(buf.st_mode)) {
		audit_msg(LOG_ERR, "%s is not a regular file", val);
		return 1;
	}
	if (buf.st_uid != 0) {
		audit_msg(LOG_ERR, "%s is not owned by root", val);
		return 1;
	}
	if ((buf.st_mode & (S_IRUSR|S_IWUSR|S_IRGRP)) != 
			   (S_IRUSR|S_IWUSR|S_IRGRP)) {
		audit_msg(LOG_ERR, "%s permissions should be 0640", val);
		return 1;
	}
	free((void *)config->log_file);
	config->log_file = strdup(val);
	if (config->log_file == NULL)
		return 1;
	return 0;
}

static int num_logs_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	const char *ptr = val;
	unsigned long i;

	audit_msg(LOG_DEBUG, "num_logs_parser called with: %s", val);

	/* check that all chars are numbers */
	for (i=0; ptr[i]; i++) {
		if (!isdigit(ptr[i])) {
			audit_msg(LOG_ERR, 
				"Value %s should only be numbers - line %d",
				val, line);
			return 1;
		}
	}

	/* convert to unsigned long */
	errno = 0;
	i = strtoul(val, NULL, 10);
	if (errno) {
		audit_msg(LOG_ERR, 
			"Error converting string to a number (%s) - line %d",
			strerror(errno), line);
		return 1;
	}
	if (i > 99) {
		audit_msg(LOG_ERR, "num_logs must be 99 or less");
		return 1;
	}
	config->num_logs = i;
	return 0;
}

static int qos_parser(const char *val, int line,
		struct daemon_conf *config)
{
	int i;

	audit_msg(LOG_DEBUG, "qos_parser called with: %s", val);
	for (i=0; qos_options[i].name != NULL; i++) {
		if (strcasecmp(val, qos_options[i].name) == 0) {
			config->qos = qos_options[i].option;
			return 0;
		}
	}
	audit_msg(LOG_ERR, "Option %s not found - line %d", val, line);
	return 1;
}


static int dispatch_parser(const char *val, int line,struct daemon_conf *config)
{
	char *dir = NULL, *tdir, *base;
	int fd;
	struct stat buf;

	audit_msg(LOG_DEBUG, "dispatch_parser called with: %s", val);
	if (val == NULL) {
		config->dispatcher = NULL;
		return 0;
	}

	/* split name into dir and basename. */
	tdir = strdup(val);
	if (tdir)
		dir = dirname(tdir);
	if (dir == NULL || strlen(dir) < 4) { //  '/var' is shortest dirname
		free(tdir);
		audit_msg(LOG_ERR,
			"The directory name: %s is too short - line %d",
			dir, line);
		return 1;
	}

	free((void *)tdir);
	base = basename((char *)val);
	if (base == 0 || strlen(base) == 0) {
		audit_msg(LOG_ERR, "The file name: %s is too short - line %d",
			base, line);
		return 1;
	}
	/* if the file exists, see that its regular, owned by root,
	 * and not world anything */
	fd = open(val, O_RDONLY);
	if (fd < 0) {
		audit_msg(LOG_ERR, "Unable to open %s (%s)", val,
			strerror(errno));
		return 1;
	}
	if (fstat(fd, &buf) < 0) {
		audit_msg(LOG_ERR, "Unable to stat %s (%s)", val,
			strerror(errno));
		close(fd);
		return 1;
	}
	close(fd);
	if (!S_ISREG(buf.st_mode)) {
		audit_msg(LOG_ERR, "%s is not a regular file", val);
		return 1;
	}
	if (buf.st_uid != 0) {
		audit_msg(LOG_ERR, "%s is not owned by root", val);
		return 1;
	}
	if ((buf.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP)) !=
			   (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP)) {
		audit_msg(LOG_ERR, "%s permissions should be 0750", val);
		return 1;
	}
	free((void *)config->dispatcher);
	config->dispatcher = strdup(val);
	if (config->dispatcher == NULL)
		return 1;
	return 0;
}

static int max_log_size_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	const char *ptr = val;
	unsigned long i;

	audit_msg(LOG_DEBUG, "max_log_size_parser called with: %s", val);

	/* check that all chars are numbers */
	for (i=0; ptr[i]; i++) {
		if (!isdigit(ptr[i])) {
			audit_msg(LOG_ERR, 
				"Value %s should only be numbers - line %d",
				val, line);
			return 1;
		}
	}

	/* convert to unsigned long */
	errno = 0;
	i = strtoul(val, NULL, 10);
	if (errno) {
		audit_msg(LOG_ERR, 
			"Error converting string to a number (%s) - line %d",
			strerror(errno), line);
		return 1;
	}
	config->max_log_size = i;
	return 0;
}

static int max_log_size_action_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	int i;

	audit_msg(LOG_DEBUG, "max_log_size_action_parser called with: %s", val);
	for (i=0; size_actions[i].name != NULL; i++) {
		if (strcasecmp(val, size_actions[i].name) == 0) {
			config->max_log_size_action = size_actions[i].option;
			return 0;
		}
	}
	audit_msg(LOG_ERR, "Option %s not found - line %d", val, line);
	return 1;
}

static int log_format_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	int i;

	audit_msg(LOG_DEBUG, "log_format_parser called with: %s", val);
	for (i=0; log_formats[i].name != NULL; i++) {
		if (strcasecmp(val, log_formats[i].name) == 0) {
			config->log_format = log_formats[i].option;
			return 0;
		}
	}
	audit_msg(LOG_ERR, "Option %s not found - line %d", val, line);
	return 1;
}

static int flush_parser(const char *val, int line, struct daemon_conf *config)
{
	int i;

	audit_msg(LOG_DEBUG, "flush_parser called with: %s", val);
	for (i=0; flush_techniques[i].name != NULL; i++) {
		if (strcasecmp(val, flush_techniques[i].name) == 0) {
			config->flush = flush_techniques[i].option;
			return 0;
		}
	}
	audit_msg(LOG_ERR, "Option %s not found - line %d", val, line);
	return 1;
}

static int freq_parser(const char *val, int line, struct daemon_conf *config)
{
	const char *ptr = val;
	int i;

	audit_msg(LOG_DEBUG, "freq_parser called with: %s", val);

	/* check that all chars are numbers */
	for (i=0; ptr[i]; i++) {
		if (!isdigit(ptr[i])) {
			audit_msg(LOG_ERR, 
				"Value %s should only be numbers - line %d",
				val, line);
			return 1;
		}
	}

	/* convert to unsigned int */
	errno = 0;
	i = strtoul(val, NULL, 10);
	if (errno) {
		audit_msg(LOG_ERR, 
			"Error converting string to a number (%s) - line %d",
			strerror(errno), line);
		return 1;
	}
	/* Check its range */
	if (i > INT_MAX) {
		audit_msg(LOG_ERR, 
			"Error - converted number (%s) is too large - line %d",
			val, line);
		return 1;
	}
	config->freq = (unsigned int)i;
	return 0;
}

static int space_left_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	const char *ptr = val;
	unsigned long i;

	audit_msg(LOG_DEBUG, "space_left_parser called with: %s", val);

	/* check that all chars are numbers */
	for (i=0; ptr[i]; i++) {
		if (!isdigit(ptr[i])) {
			audit_msg(LOG_ERR, 
				"Value %s should only be numbers - line %d",
				val, line);
			return 1;
		}
	}

	/* convert to unsigned long */
	errno = 0;
	i = strtoul(val, NULL, 10);
	if (errno) {
		audit_msg(LOG_ERR, 
			"Error converting string to a number (%s) - line %d",
			strerror(errno), line);
		return 1;
	}
	config->space_left = i;
	return 0;
}

static int space_action_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	int i;

	audit_msg(LOG_DEBUG, "space_action_parser called with: %s", val);
	for (i=0; failure_actions[i].name != NULL; i++) {
		if (strcasecmp(val, failure_actions[i].name) == 0) {
			if (failure_actions[i].option == FA_EMAIL) {
				if (access(email_command, X_OK)) {
					audit_msg(LOG_ERR,
		"Email option is specified but %s doesn't seem executable.",
						 email_command);
				}
			}
			config->space_left_action = failure_actions[i].option;
			return 0;
		}
	}
	audit_msg(LOG_ERR, "Option %s not found - line %d", val, line);
	return 1;
}

// returns 1 on error & 0 if OK
int validate_email(const char *acct)
{
	int i, len;
	char *ptr1;

	if (acct == NULL)
		return 1;

	len = strlen(acct);
	if (len < 2)
		return 1;

	// look for illegal char
	for (i=0; i<len; i++) {
		if (! (isalnum(acct[i]) || (acct[i] == '@') ||
				(acct[i]=='.') || (acct[i]=='-') ||
				(acct[i] == '_')) )
			return 1;
	}

	if ((ptr1 = strchr(acct, '@'))) {
		char *ptr2;
		struct hostent *t_addr;

		ptr2 = strrchr(acct, '.');        // get last dot - sb after @
		if ((ptr2 == NULL) || (ptr1 > ptr2))
			return 1;

		t_addr = gethostbyname(ptr1+1);
		if (t_addr == 0) {
			if ((h_errno == HOST_NOT_FOUND) ||
					(h_errno == NO_RECOVERY))
				return 1;
			else if (h_errno == TRY_AGAIN)
				return 2;
		}
	}
	return 0;
}

static int action_mail_acct_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	char *tmail;
	
	audit_msg(LOG_DEBUG, "action_mail_acct_parser called with: %s", val);
	tmail = strdup(val);
	if (tmail == NULL)
		return 1;

	if (validate_email(tmail)) {
		free(tmail);
		return 1;
	}


	if (config->action_mail_acct)
		free((void *)config->action_mail_acct);
	config->action_mail_acct = tmail;
	return 0;
}

static int admin_space_left_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	const char *ptr = val;
	unsigned long i;

	audit_msg(LOG_DEBUG, "admin_space_left_parser called with: %s", val);

	/* check that all chars are numbers */
	for (i=0; ptr[i]; i++) {
		if (!isdigit(ptr[i])) {
			audit_msg(LOG_ERR, 
				"Value %s should only be numbers - line %d",
				val, line);
			return 1;
		}
	}

	/* convert to unsigned long */
	errno = 0;
	i = strtoul(val, NULL, 10);
	if (errno) {
		audit_msg(LOG_ERR, 
			"Error converting string to a number (%s) - line %d",
			strerror(errno), line);
		return 1;
	}
	config->admin_space_left = i;
	return 0;
}

static int admin_space_left_action_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	int i;

	audit_msg(LOG_DEBUG, "admin_space_left_action_parser called with: %s", val);
	for (i=0; failure_actions[i].name != NULL; i++) {
		if (strcasecmp(val, failure_actions[i].name) == 0) {
			if (failure_actions[i].option == FA_EMAIL) {
				if (access(email_command, X_OK)) {
					audit_msg(LOG_ERR,
		"Email option is specified but %s doesn't seem executable.",
						 email_command);
				}
			}
			config->admin_space_left_action =
						failure_actions[i].option;
			return 0;
		}
	}
	audit_msg(LOG_ERR, "Option %s not found - line %d", val, line);
	return 1;
}

static int disk_full_action_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	int i;

	audit_msg(LOG_DEBUG, "disk_full_action_parser called with: %s", val);
	for (i=0; failure_actions[i].name != NULL; i++) {
		if (strcasecmp(val, failure_actions[i].name) == 0) {
			if (failure_actions[i].option != FA_EMAIL) {
				config->disk_full_action =
						failure_actions[i].option;
				return 0;
			} else {
				audit_msg(LOG_ERR, 
			"Illegal option %s for disk_full_action - line %d",
					 val, line);
				return 1;
			}
		}
	}
	audit_msg(LOG_ERR, "Option %s not found - line %d", val, line);
	return 1;
}

static int disk_error_action_parser(const char *val, int line, 
		struct daemon_conf *config)
{
	int i;

	audit_msg(LOG_DEBUG, "disk_error_action_parser called with: %s", val);
	for (i=0; failure_actions[i].name != NULL; i++) {
		if (strcasecmp(val, failure_actions[i].name) == 0) {
			if (failure_actions[i].option != FA_EMAIL) {
				config->disk_error_action = 
						failure_actions[i].option;
				return 0;
			} else {
				audit_msg(LOG_ERR, 
			"Illegal option %s for disk_error_action - line %d",
					 val, line);
				return 1;
			}
		}
	}
	audit_msg(LOG_ERR, "Option %s not found - line %d", val, line);
	return 1;
}

static int priority_boost_parser(const char *val, int line, struct daemon_conf *config)
{
	const char *ptr = val;
	int i;

	audit_msg(LOG_DEBUG, "priority_boost_parser called with: %s", val);

	/* check that all chars are numbers */
	for (i=0; ptr[i]; i++) {
		if (!isdigit(ptr[i])) {
			audit_msg(LOG_ERR, 
				"Value %s should only be numbers - line %d",
				val, line);
			return 1;
		}
	}

	/* convert to unsigned int */
	errno = 0;
	i = strtoul(val, NULL, 10);
	if (errno) {
		audit_msg(LOG_ERR, 
			"Error converting string to a number (%s) - line %d",
			strerror(errno), line);
		return 1;
	}
	/* Check its range */
	if (i > INT_MAX) {
		audit_msg(LOG_ERR, 
			"Error - converted number (%s) is too large - line %d",
			val, line);
		return 1;
	}
	config->priority_boost = (unsigned int)i;
	return 0;
}

/*
 * This function is where we do the integrated check of the audit config
 * options. At this point, all fields have been read. Returns 0 if no
 * problems and 1 if problems detected.
 */
static int sanity_check(struct daemon_conf *config)
{
	/* Error checking */
	if (config->space_left <= config->admin_space_left) {
		audit_msg(LOG_ERR, 
		    "Error - space_left(%lu) must be larger than admin_space_left(%lu)",
		    config->space_left, config->admin_space_left);
		return 1;
	}
	if (config->flush == FT_INCREMENTAL && config->freq == 0) {
		audit_msg(LOG_ERR, 
		"Error - incremental flushing chosen, but 0 selected for freq");
		return 1;
	}
	/* Warnings */
	if (config->flush > FT_INCREMENTAL && config->freq != 0) {
		audit_msg(LOG_WARNING, 
		    "Warning - freq is non-zero and incremental flushing not selected.");
	}
	return 0;
}

const char *audit_lookup_format(int fmt)
{
	int i;

	for (i=0; log_formats[i].name != NULL; i++) {
                if (log_formats[i].option == fmt)
			return log_formats[i].name;
	}
	return NULL;
}

int create_log_file(const char *val)
{
	int fd;

	umask(S_IRWXO);
	fd = open(val, O_CREAT|O_EXCL|O_APPEND, S_IRUSR|S_IWUSR|S_IRGRP);
	if (fd < 0) 
		audit_msg(LOG_ERR, "Unable to create %s (%s)", val,
			strerror(errno));
	return fd;
}

void free_config(struct daemon_conf *config)
{
	free((void*)config->log_file);
	free((void*)config->dispatcher);
	free((void *)config->action_mail_acct);
}

