/* aureport-options.c - parse commandline options and configure aureport
 * Copyright 2005-06 Red Hat Inc., Durham, North Carolina.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *     Steve Grubb <sgrubb@redhat.com>
 */

#include "config.h"
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <time.h>
#include "aureport-options.h"
#include "ausearch-time.h"
#include "libaudit.h"


/* Global vars that will be accessed by the main program */
char *user_file = NULL;

/* These are for compatibility with parser */
unsigned int event_id = -1;
const char *event_key = NULL;
const char *event_filename = NULL;
const char *event_exe = NULL;
const char *event_comm = NULL;
const char *event_hostname = NULL;
const char *event_terminal = NULL;
const char *event_subject = NULL;
const char *event_object = NULL;

/* These are used by aureport */
const char *dummy = "dummy";
report_type_t report_type = RPT_UNSET;
report_det_t report_detail = D_UNSET;
report_t report_format = RPT_RAW;
failed_t event_failed = F_BOTH;

struct nv_pair {
    int        value;
    const char *name;
};

enum {  R_INFILE, R_TIME_END, R_TIME_START, R_VERSION, R_SUMMARY, R_LOG_TIMES,
	R_CONFIGS, R_LOGINS, R_USERS, R_TERMINALS, R_HOSTS, R_EXES, R_FILES,
	R_AVCS, R_SYSCALLS, R_PIDS, R_EVENTS, R_ACCT_MODS, R_WATCHES, 
	R_INTERPRET, R_HELP, R_ANOMALY, R_RESPONSE, R_SUMMARY_DET, R_FAILED,
	R_SUCCESS };

static struct nv_pair optiontab[] = {
	{ R_AVCS, "-a" },
	{ R_CONFIGS, "-c" },
	{ R_EVENTS, "-e" },
	{ R_FILES, "-f" },
	{ R_FAILED, "--failed" },
	{ R_HOSTS, "-h" },
	{ R_HELP, "--help" },
	{ R_INTERPRET, "-i" },
	{ R_INFILE, "-if" },
	{ R_LOGINS, "-l" },
	{ R_ACCT_MODS, "-m" },
	{ R_ANOMALY, "-n" },
	{ R_PIDS, "-p" },
	{ R_RESPONSE, "-r" },
	{ R_SYSCALLS, "-s" },
	{ R_SUCCESS, "--success" },
	{ R_SUMMARY_DET, "--summary" },
	{ R_LOG_TIMES, "-t" },
	{ R_TIME_END, "-te"},
	{ R_TERMINALS, "-tm"}, // don't like this
	{ R_TIME_START, "-ts" },
	{ R_USERS, "-u" },
	{ R_VERSION, "-v" },
	{ R_WATCHES, "-w" },
	{ R_EXES, "-x" }
};
#define OPTION_NAMES (sizeof(optiontab)/sizeof(optiontab[0]))


static int audit_lookup_option(const char *name)
{
        int i;

        for (i = 0; i < OPTION_NAMES; i++)
                if (!strcmp(optiontab[i].name, name))
			return optiontab[i].value;
        return -1;
}

static void usage(void)
{
	printf("usage: aureport [options]\n"
	"\t-a\t\t\t\tAvc report\n"
	"\t-c\t\t\t\tConfig change report\n"
	"\t-e\t\t\t\tEvent report\n"
	"\t-f\t\t\t\tFile name report\n"
	"\t--failed\t\t\tonly failed events in report\n"
	"\t-h\t\t\t\tHost name report\n"
	"\t--help\t\t\t\thelp\n"
	"\t-i\t\t\t\tInterpretive mode\n"
	"\t-if <Input File name>\t\tuse this file as input\n"
	"\t-l\t\t\t\tLogin report\n"
	"\t-m\t\t\t\tModification to accounts report\n"
	"\t-n\t\t\t\taNomaly report\n"
	"\t-p\t\t\t\tPid report\n"
	"\t-r\t\t\t\tResponse to anomaly report\n"
	"\t-s\t\t\t\tSyscall report\n"
	"\t--success\t\t\tonly success events in report\n"
	"\t--summary\t\t\tsorted totals for main object in report\n"
	"\t-t\t\t\t\tTime range report\n"
	"\t-te [end date] [end time]\tending date & time for reports\n"
	"\t-tm\t\t\t\tTerMinal name report\n"
	"\t-ts [start date] [start time]\tstarting data & time for reports\n"
	"\t-u\t\t\t\tUser name report\n"
	"\t-v\t\t\t\tVersion\n"
	"\t-w\t\t\t\tWatched files report\n"
	"\t-x\t\t\t\teXecutable name report\n"
	"\tIf no report is given, the summary report will be displayed\n"
	);
}

static int set_report(report_type_t r)
{
	if (report_type == RPT_UNSET) {
		report_type = r;
		return 0;
	} else {
		fprintf(stderr, "Error - only one report can be specified");
		return 1;
	}
}

static int set_detail(report_det_t d)
{
	if (report_detail == D_UNSET) {
		report_detail = d;
		return 0;
	} else if (d == D_SUM) {
		report_detail = d;
		return 0;
	} else {
		return 1;
	}
}

/*
 * This function examines the commandline parameters and sets various
 * search options. It returns a 0 on success and < 0 on failure
 */
int check_params(int count, char *vars[])
{
	int c = 1;
	int retval = 0;
	const char *optarg;

	while (c < count && retval == 0) {
		// Go ahead and point to the next argument
		if (c+1 < count) {
			if (vars[c+1][0] != '-')
				optarg = vars[c+1];
			else
				optarg = NULL;
		} else
			optarg = NULL;

		switch (audit_lookup_option(vars[c])) {
		case R_INFILE:
			if (!optarg) {
				fprintf(stderr, 
					"Argument is required for -if\n");
				retval = -1;
			} else {
				user_file = strdup(optarg);
				if (user_file == NULL)
					retval = -1;
				c++;
			}
			break;
		case R_LOG_TIMES:
			if (set_report(RPT_TIME))
				retval = -1;
			else
				set_detail(D_DETAILED);
			break;
		case R_AVCS:
			if (set_report(RPT_AVC))
				retval = -1;
			else { 
				set_detail(D_DETAILED);
				event_comm = dummy;
				event_subject = dummy;
				event_object = dummy;
			}
			break;
		case R_CONFIGS:
			if (set_report(RPT_CONFIG))
				retval = -1;
			else 
				set_detail(D_DETAILED);
			break;
		case R_LOGINS:
			if (set_report(RPT_LOGIN))
				retval = -1;
			else {
				set_detail(D_DETAILED);
				event_exe = dummy;
				event_hostname = dummy;
				event_terminal = dummy;
			}
			break;
		case R_ACCT_MODS:
			if (set_report(RPT_ACCT_MOD))
				retval = -1;
			else { 
				set_detail(D_DETAILED);
				event_exe = dummy;
				event_hostname = dummy;
				event_terminal = dummy;
			}
			break;
		case R_EVENTS:
			if (set_report(RPT_EVENT))
				retval = -1;
			else {
//				if (!optarg)
					set_detail(D_DETAILED);
//				else {
//					UNIMPLEMENTED;
//					set_detail(D_SPECIFIC);
//					if (isdigit(optarg[0])) {
//						errno = 0;
//						event_id = strtoul(optarg,
//							NULL, 10);
//						if (errno) {
//							fprintf(stderr,
//					"Illegal value for audit event ID");
//							retval = -1;
//						}
//						c++;
//					} else {
//						fprintf(stderr,
//			"Audit event id must be a numeric value, was %s\n",
//						optarg);
//						retval = -1;
//					}
//				}
			}
			break;
		case R_FILES:
			if (set_report(RPT_FILE))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_filename = dummy;
					event_exe = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_HOSTS:
			if (set_report(RPT_HOST))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_hostname = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_INTERPRET:
			report_format = RPT_INTERP;
			if (optarg) {
				fprintf(stderr,
					"Argument is NOT required for -i\n");
				retval = -1;
			}
			break;
		case R_PIDS:
			if (set_report(RPT_PID))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_exe = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_SYSCALLS:
			if (set_report(RPT_SYSCALL))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_comm = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_TERMINALS:
			if (set_report(RPT_TERM))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_terminal = dummy;
					event_hostname = dummy;
					event_exe = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_USERS:
			if (set_report(RPT_USER))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_terminal = dummy;
					event_hostname = dummy;
					event_exe = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_WATCHES:
			if (set_report(RPT_WATCH))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_terminal = dummy;
					event_hostname = dummy;
					event_exe = dummy;
					event_filename = dummy;
					event_key = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_EXES:
			if (set_report(RPT_EXE))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_terminal = dummy;
					event_hostname = dummy;
					event_exe = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_ANOMALY:
			if (set_report(RPT_ANOMALY))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
					event_terminal = dummy;
					event_hostname = dummy;
					event_exe = dummy;
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_RESPONSE:
			if (set_report(RPT_RESPONSE))
				retval = -1;
			else {
				if (!optarg) {
					set_detail(D_DETAILED);
				} else {
					UNIMPLEMENTED;
				}
			}
			break;
		case R_TIME_END:
			if (optarg) {
				if ( (c+2 < count) && vars[c+2] && 
					(vars[c+2][0] != '-') ) {
				/* Have both date and time - check order*/
					if (strchr(optarg, ':')) {
						if (ausearch_time_end(vars[c+2],
								 optarg) != 0) 
							retval = -1;
					} else {
						if (ausearch_time_end(optarg, 
								vars[c+2]) != 0)
							retval = -1;
					}
					c++;			
				} else {
					// Check against recognized words
					int t = lookup_time(optarg);
					if (t >= 0) {
						struct tm d;
						switch (t)
						{
							case T_NOW:
							    set_tm_now(&d);
								break;
							case T_TODAY:
							    set_tm_today(&d);
								break;
							case T_YESTERDAY:
							   set_tm_yesterday(&d);
								break;
						}
						end_time = mktime(&d);
					} else if ( (strchr(optarg, ':')) == NULL) {
						/* Only have date */
						if (ausearch_time_end(optarg,
								NULL) != 0)
							retval = -1;
					} else {
						/* Only have time */
						if (ausearch_time_end(NULL,
								optarg) != 0)
							retval = -1;
					}
				}
				c++;			
				break;
			}
			fprintf(stderr,
				"-te requires either date and/or time\n");
			retval = -1;
			break;
		case R_TIME_START:
			if (optarg) {
				if ( (c+2 < count) && vars[c+2] && 
					(vars[c+2][0] != '-') ) {
				/* Have both date and time - check order */
					if (strchr(optarg, ':')) {
						if (ausearch_time_start(
							vars[c+2], optarg) != 0)
							retval = -1;
					} else {
						if (ausearch_time_start(optarg, 
								vars[c+2]) != 0)
							retval = -1;
					}
					c++;
				} else {
					// Check against recognized words
					int t = lookup_time(optarg);
					if (t >= 0) {
						struct tm d;
						switch (t)
						{
							case T_NOW:
							    set_tm_now(&d);
								break;
							case T_TODAY:
							     set_tm_today(&d);
								break;
							case T_YESTERDAY:
							   set_tm_yesterday(&d);
								break;
						}
						start_time = mktime(&d);
					} else if ( strchr(optarg, ':') == NULL) {
						/* Only have date */
						if (ausearch_time_start(optarg,
							"00:00:01") != 0)
							retval = -1;
					} else {
						/* Only have time */
						if (ausearch_time_start(NULL,
								optarg) != 0)
							retval = -1;
					}
				}
				c++;
				break;
			}
			fprintf(stderr, 
				"-ts requires either date and/or time\n");
			retval = -1;
			break;
		case R_SUMMARY_DET:
			set_detail(D_SUM);
			break;
		case R_FAILED:
			event_failed = F_FAILED;
			break;
		case R_SUCCESS:
			event_failed = F_SUCCESS;
			break;
		case R_VERSION:
	                printf("aureport version %s\n", VERSION);
			exit(0);
			break;
		case R_HELP:
			usage();
			exit(0);
			break;
		default:
			fprintf(stderr, "%s is an unsupported option\n", 
				vars[c]);
			retval = -1;
			break;
		}
		c++;
	}

	if (retval >= 0) {
		if (report_type == RPT_UNSET) {
			if (set_report(RPT_SUMMARY))
				retval = -1;
			else {
				set_detail(D_SUM);
				event_filename = dummy;
				event_hostname = dummy;
				event_terminal = dummy;
				event_exe = dummy;
			}
		}
	} else
		usage();

	return retval;
}

