/**********************************************************
 *
 * Snare Epilog for UNIX (Linux and Solaris) version 1.1
 *
 * Author: InterSect Alliance Pty Ltd
 *
 * Copyright 2001-2006 InterSect Alliance Pty Ltd
 *
 * Last Modified: 13/06/2006
 *
 **********************************************************
 *
 * Snare Epilog for UNIX is a cross platform agent designed
 * to monitor any given text file and report to one or more
 * SNARE servers.
 *
 *
 **********************************************************
 *
 * History:
 *      05/06/2006  Initial working version
 *      05/07/2006  Added support for SnareApache and SnareSquid
 *
 **********************************************************
 *
 * Compilation Instructions:
 *    gcc -o epilog epilog.c webserver.c WebPages.c -lsocket -lnsl -lintl -lpthread
 *
 **********************************************************/

#include <sys/types.h> 
#include <sys/stat.h> 
#include <sys/wait.h> 
#include <sys/errno.h>
#include <sys/fcntl.h>
#include <sys/termios.h>
//#include <asm/ioctls.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <stdlib.h>
#include <strings.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <limits.h>
#include <dirent.h>
#include <netdb.h>
#include <values.h>

#include <regex.h>

#include "epilog.h"
#include "webserver.h"

// Linked List Functions
static Node *head, *currentnode;
static int AuditDestination = AUDIT_TO_NETWORK;		// Send to NETWORK unless otherwise stated.
static int SyslogDestination = 13;			// user.notice

char USER_CONFIG_FILENAME[MAX_AUDIT_CONFIG_LINE] = "\0";

char hostid[MAX_HOSTID] = "";
int remote_allow = 0;
int remote_webport = 6161;
char remote_restrictip[16] = "";
char remote_password[256] = "";

//int AuditSocket = 0;
//struct sockaddr_in AuditSocketName;
pthread_mutex_t log_mutex = PTHREAD_MUTEX_INITIALIZER; 
pthread_mutex_t send_mutex = PTHREAD_MUTEX_INITIALIZER; 
pthread_mutex_t	debug_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t  log_data_available = PTHREAD_COND_INITIALIZER;

int LCCount=0;
MsgCache *LCHead=NULL;
MsgCache *LCTail=NULL;
MsgCache *LCCurrent=NULL;

HostNode host;

LogNode *lhead=NULL;
LogNode *ltail=NULL;
LogNode *lcurrent=NULL;

int SocketType = SOCKETTYPE_UDP;

int HNodeCounter=0;

// {{{ Signal functions 
/* Define an error variable so we can communicate with auditsvc */
int caught_pipe=0;                /* variable that lets us know when a SIGPIPE has been received */
int caught_kill=0;                /* variable that lets us know when a SIGKILL has been received */
int caught_usr1=0;                /* variable that lets us know when a SIGUSR1 has been received */

void pipe_signal(sig)
{
	caught_pipe++;
	return;
}

void kill_signal(sig)
{
	caught_kill++;
	//DebugMsg("Caught SIGTERM\n");
	return;
}

void usr1_signal(sig) {
	caught_usr1++;
	//DebugMsg("Caught SIGUSR\n");
	return;
}
// }}}

int main(int argc, char *argv[])
{
	// General program related variables
	char logbuffer[LOGBUFSIZE];		/* Holds the data from the text file */
	int continueloop=1;			/* Make sure we restart the process if the pipe fails */
	struct timespec timeout;		/* time between reads */
	char pid_file_name[MAX_AUDIT_CONFIG_LINE];

	/* Signal related variables */
	sigset_t signalset;

	int ppid=0;			/* process id */
	int webpid = 0;			/* Web server process id */

	pthread_t thread[1]; 

	/* The only person who can run auditsvc is root, so lets
	   make sure that we discourage other users */
	if (getuid() != 0) {
		printf("This program can only be run by root.\n");
		exit(1);
	}

	// If any command line arguments are provided, make sure they are legitimate
	// before divorcing from the parent process
	if (argc == 3 && strcmp(argv[1],"-c") == 0) {
		if(argc != 3) {
			usage(argv[0]);
			exit(1);
		}
		strncpy(USER_CONFIG_FILENAME, argv[2], MAX_AUDIT_CONFIG_LINE);
	} else if (argc != 1) {
		usage(argv[0]);
		exit(1);
	}

	/* Make sure that we divorce ourselves from the parent process
	   as we may do not want to inherit signals
	   Note that we will definately be divorced from the parent tty
	   after this, so error messages to stdout will not be particularly
	   useful. */

	if (fork())	{
		/* This is the parent process - exit, and ditch the parent.
		   We will be inherited by process 1 (init) */
		exit(0);
	}

	// Read our configuration file.
	read_config_file();
		
	if (!hostid) {
		getfqdn(hostid);
	} else {
		if (strlen(hostid) == 0) {
			getfqdn(hostid);
		}
	}


	/* Work out the current process ID */
	ppid=getpid();
	if (!ppid)
	{
		DebugMsg("Cannot find current process ID. Exiting.\n");
		exit(1);
	} else {
		/* Write out process ID */
		FILE *pidfile;
		if (USER_CONFIG_FILENAME[0] == '\0') {
			strncpy(pid_file_name, "/var/run/epilog.pid", MAX_AUDIT_CONFIG_LINE);
		} else {
			if (!strncmp(USER_CONFIG_FILENAME, CONFIG_FILENAME_SQUID, MAX_AUDIT_CONFIG_LINE)) {
				strncpy(pid_file_name, "/var/run/snaresquid.pid", MAX_AUDIT_CONFIG_LINE);
			} else if (!strncmp(USER_CONFIG_FILENAME, CONFIG_FILENAME_APACHE, MAX_AUDIT_CONFIG_LINE)) {
				strncpy(pid_file_name, "/var/run/snareapache.pid", MAX_AUDIT_CONFIG_LINE);
			} else {
				strncpy(pid_file_name, "/var/run/epilog_user.pid", MAX_AUDIT_CONFIG_LINE);
			}
		}
		if ((pidfile = fopen(pid_file_name,"w"))) {
			fprintf(pidfile,"%d",ppid);
			fclose(pidfile);
		}
	}

	
	if (remote_allow) {
		webpid=fork();
		if (webpid==0) {
			// Child process - Run our web server
			while(InitWebServer(remote_webport,remote_restrictip,remote_password) == -1) {
				// Problem binding. Sleep for a moment.
				sleep(5);
			}
			while(StartThread() == 1) {
				// Problem binding. Keep trying.
				sleep(5);
			}
			exit(0);
		}
	}

	// setup the timeout values
	timeout.tv_sec = 0;
	timeout.tv_nsec = TIMEOUT;
	// as the messages are cached, this thread will purge them
	pthread_create(&thread[1], NULL, (void *)send_logs, NULL);
	// This is effectively a while(true)
	while(continueloop)
	{
		// Configure / Reinstate signals
		// Trap signals relating to PIPE failures,
		//  and Child process termination
		// Reset this each time through the loop, just in case
		// a signal handler resets our values
		
		if (!setsignals(&signalset)) {
			DebugMsg("Cannot set important signals - exiting\n");
			kill(ppid,SIGTERM); waitpid(ppid,NULL,0);
			continueloop=0;
			break;
		}

		// For each of the log file identified
		lcurrent = lhead;
		while (lcurrent) {
			int ret=file_has_changed(lcurrent);
			int big_msg=0;
			if (ret > 0) {
				while (fgets(logbuffer, LOGBUFSIZE, lcurrent->fs)) {
					if (big_msg) {
						if (logbuffer[strlen(logbuffer) - 1] == '\n') {
							big_msg = 0;
						}
						continue;
					}
					if (lcurrent->pmsg && strlen(lcurrent->pmsg)) {
						strncat(lcurrent->pmsg, logbuffer, LOGBUFSIZE - strlen(lcurrent->pmsg) - 1);
						strncpy(logbuffer, lcurrent->pmsg, LOGBUFSIZE);
						lcurrent->pmsg[0]='\0';
					}
					if (logbuffer[strlen(logbuffer) - 1] != '\n') {
						if (strlen(logbuffer) >= LOGBUFSIZE -1) {
							logbuffer[LOGBUFSIZE]='\0';
							big_msg = 1;
						} else {
							strncpy(lcurrent->pmsg, logbuffer, LOGBUFSIZE);
							continue;
						}
					}
					trim(logbuffer);
					if (logbuffer && CheckObjective(logbuffer)) {
						//#######################################
						pthread_mutex_lock(&log_mutex);
						if (LCCount > DEFAULT_CACHE) {
							DebugMsg("Log Cache FULL - deleting oldest message\n");
							LCCurrent = LCHead;
							LCHead = LCCurrent->next;
							free(LCCurrent);
						}
						LCCurrent = (MsgCache *)malloc(sizeof(MsgCache));
						if (LCCurrent) {
							LCCurrent->msglen=strlen(logbuffer);
							strncpy(LCCurrent->msg, logbuffer, LCCurrent->msglen);
							LCCurrent->msg[LCCurrent->msglen]='\0';
							LCCurrent->next=NULL;
							strncpy(LCCurrent->type, lcurrent->type, MAX_AUDIT_CONFIG_LINE);
						} else {
							DebugMsg("Unable to allocate cache\n");
							pthread_mutex_unlock(&log_mutex);
							return(0);
						}
						LCCount++;
						if (LCTail) {
							LCTail->next=LCCurrent;
						}
						LCTail=LCCurrent;
						if (!LCHead) {
							LCHead=LCCurrent;
						}
						pthread_mutex_unlock(&log_mutex);
						pthread_cond_signal(&log_data_available);
						//#######################################
					}
				}
			} else {
				// The file hasn't changed OR
				// there was an error that we can't do anything about
			}
			lcurrent = lcurrent->next;
		}
		nanosleep(&timeout,NULL);
	}

	DebugMsg("Epilog - Exiting!");

	if (webpid) {
		kill(webpid, SIGTERM);
		waitpid(webpid, NULL, 0);
	}

	
	DestroyList();
	unlink(pid_file_name);
			
	/* All done */
	return 0;
}

int setsignals(sigset_t *signalset)
{
	sigfillset(signalset);
	sigdelset(signalset,SIGTERM);
	sigdelset(signalset,SIGALRM);
	sigdelset(signalset,SIGINT);
	sigdelset(signalset,SIGPIPE);
	sigdelset(signalset,SIGUSR1);
	sigdelset(signalset,SIGCHLD);
	sigprocmask(SIG_BLOCK,signalset, (void *) NULL);

	if (signal(SIGUSR1,usr1_signal) == SIG_ERR)
	{
		DebugMsg("Cannot set signal SIGUSR1\n");
		return(0);
	}

	if (signal(SIGPIPE,pipe_signal) == SIG_ERR)
	{
		DebugMsg("Cannot set signal SIGPIPE\n");
		return(0);
	}

	if (signal(SIGCHLD,pipe_signal) == SIG_ERR)
	{
		DebugMsg("Cannot set signal SIGCHLD\n");
		return(0);
	}

	return(1);
}

int read_config_file()
{
	FILE *configfile = (FILE *)NULL;
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];	// Should be enough for most config lines.
							// Would love to use gchar here instead, but need to keep this simple.
	// Config file header.
	int headertype = 0;

	// Clear some default variables.
	remote_allow = 0;
	remote_webport = 6161;
	strncpy(remote_restrictip, "", sizeof(remote_restrictip));
	strncpy(remote_password, "", sizeof(remote_password));

	// Clear the audit destination
	AuditDestination=0;

	if (strlen(USER_CONFIG_FILENAME))
		configfile = fopen(USER_CONFIG_FILENAME,"r");
	else
		configfile = fopen(CONFIG_FILENAME,"r");

	if (configfile == (FILE *)NULL) {
		printf("Cannot open audit configuration file.");
		return(0);
	}

	while (fgets(inputbuffer, MAX_AUDIT_CONFIG_LINE, configfile)) {

		// Kill whitespace from start and end of line.
		trim(inputbuffer);

		if (!iscomment(inputbuffer)) {
			// Is this line a header?
			if (isheader(inputbuffer)) {
				headertype = getheader(inputbuffer);
			} else {
				if (headertype == CONFIG_OBJECTIVES) {
					// Save off enough space to store the data we need.
					char path[MAX_AUDIT_CONFIG_LINE];
					int excludematchflag = 0;

					if (splitobjective(inputbuffer, path, &excludematchflag) > -1) {
						AddToList(path, excludematchflag);
					} else {
						DebugMsg("WARNING: Cannot process objective - please ensure configuration line\n\tcontains valid match elements\n\t%s\n", inputbuffer);
					}
				} else if (headertype == CONFIG_OUTPUT) {
					if (isnetwork(inputbuffer)) {
						if (!open_audit_network(inputbuffer)) {
							DebugMsg("WARNING: Could not open the network specified in the audit configuration: %s\n", inputbuffer);
						}
					} else if (issyslog(inputbuffer)) {
						SyslogDestination = get_syslog_dest(inputbuffer);
						if (!SyslogDestination) {
							SyslogDestination = 13;
							DebugMsg("WARNING: Could not establish the correct syslog destination from the audit configuration: %s - Sending to user.notice\n", inputbuffer);
						}
					}
				} else if (headertype == CONFIG_INPUT) {
					if (islog(inputbuffer)) {
						if (!add_log_watch(inputbuffer)) {
							DebugMsg("WARNING: Could not open the log file specified in the audit configuration: %s\n", inputbuffer);
						}
					}
				} else if (headertype == CONFIG_HOSTID) {
					getconfstring(inputbuffer, hostid, MAX_HOSTID);
				} else if (headertype == CONFIG_REMOTE) {
					// Grab the remote control stuff here
					if (regmatchi(inputbuffer, "^allow=")) {
						if (regmatchi(inputbuffer, "=1")) {
							remote_allow = 1;
						}
					} else if (regmatchi(inputbuffer, "^listen_port=")) {
						remote_webport = getport(inputbuffer);
					} else if (regmatchi(inputbuffer, "^restrict_ip=")) {
						if (!getconfstring(inputbuffer, remote_restrictip, sizeof(remote_restrictip))) {
							strncpy(remote_restrictip, "", sizeof(remote_restrictip));
						}
					} else if (regmatchi(inputbuffer, "^accesskey=")) {
						if (!getconfstring(inputbuffer, remote_password, sizeof(remote_password))) {
							strncpy(remote_password, "", sizeof(remote_password));
						}
					}
				} else {
					DebugMsg("WARNING: Configuration file line does not fit in any recognised header.\n\t%s\n", inputbuffer);
				}
			}
		}
	}

	fclose((FILE *)configfile);
	return(1);
}

// {{{ Config File "is" functions
int iscomment(char *line)
{
	// Verify that there is something to check.
	if (line == (char *) NULL) return(1);

	// And that there is some data within.
	if (strlen(line) == 0 || strlen(line) > MAX_AUDIT_CONFIG_LINE) return(1);

	// If the first non-whitespace character is a hash, this is a comment line.
	while (*line) {
		// Space or tab or newline / formfeed
		if (*line == ' ' || *line == 9 || *line == 10 || *line == 12) {
			line++;
		} else if (*line == '#') {
			return(1);
		} else {
			// Ahh. A non-whitespace, non hash character.
			return(0);
		}
	}

	// If we are here, then the whole line must have been whitespace
	return(1);
}

int isheader(char *string)
{
	if (string[0] == '[' && string[strlen(string) - 1] == ']') {
		return(1);
	}
	return(0);
}

// Network output
int isnetwork(char *string)
{
	if (regmatch(string, "^network=")) {
		return(1);
	} else {
		return(0);
	}
}

int islog(char *string)
{
	if (regmatch(string, "^log=")) {
		return(1);
	} else {
		return(0);
	}
}

int issyslog(char *string)
{
	if (regmatch(string, "^syslog=")) {
		return(1);
	}
	return(0);
}
// }}}

// Remove start / end whitespace, including the newline
void trim(char *string)
{
	char *pointer;
	char *pointer2;

	// Verify that there is something to check.
	if (string == (char *) NULL) return;

	// And that there is some data within.
	if (strlen(string) == 0) return;

	// Start from the end, work backwards.
	pointer = &string[strlen(string) - 1];

	while ((*pointer == ' ' || *pointer == 9 || *pointer == 10 || *pointer == 12) && pointer >= string) {
		*pointer = '\0';
		pointer--;
	}

	// Are we back at the start of the string? If so, this line must be null.
	if (pointer == string) return;
	// Pointer is now at the last non-whitespace character of the string.

	pointer2 = string;
	while ((*pointer2 == ' ' || *pointer2 == 9 || *pointer2 == 10 || *pointer2 == 12) && pointer2 < pointer) {
		pointer2++;
	}
	// pointer2 will now point to the start of the first non-null character
	// Copy the truncated string back to the original.
	strcpy(string, pointer2);
}

int getheader(char *string)
{
	char temp[256];
	char *stringp;
	strncpy(temp, string, 256);

	stringp = temp;

	// Remove the first and last bracket.
	stringp++;
	stringp[strlen(stringp) - 1] = '\0';

	if (regmatchi(stringp, "^objectives$")) {
		return(CONFIG_OBJECTIVES);
	} else if (regmatchi(stringp, "^input$")) {
		return(CONFIG_INPUT);
	} else if (regmatchi(stringp, "^output$")) {
		return(CONFIG_OUTPUT);
	} else if (regmatchi(stringp, "^hostid$")) {
		return(CONFIG_HOSTID);
	} else if (regmatchi(stringp, "^remote$")) {
		return(CONFIG_REMOTE);
	}

	DebugMsg("Unknown header in configuration file: %s\n", stringp);
	return(0);
}

// Open the network - destination host and port
int open_audit_network(char *string)
{
	int AuditSocket=0;
	char desthost[MAX_HOSTID];
	int port = 6161, protocol=SOCKETTYPE_UDP;	// Default port, protocol
	char *pos, *pos2;
	struct hostent *hp;
	pthread_t thread[1]; 

	// Initialise desthost.
	strncpy(desthost, "localhost", MAX_HOSTID);

	string += strlen("network=");
	if (strlen(string)) {
		pos = strstr(string, ":");
		if (pos) {
			if ((pos - string) < MAX_HOSTID) {
				strncpy(desthost, string, (pos - string));
				desthost[pos - string] = '\0';
			} else {
				strncpy(desthost, string, MAX_HOSTID);
				desthost[MAX_HOSTID - 1] = '\0';
			}
			pos++;
			// As long as the user has a number there.
			if (strlen(pos)) {
				port = atoi(pos);
			}
		} else {
			// No colon. Assume the default port to be 6161
			strncpy(desthost, string, MAX_HOSTID);
		}

		AuditSocket = socket(AF_INET,SOCK_DGRAM,0);
		if (AuditSocket < 0) {
			DebugMsg("Cannot open network socket\n");
			return(0);
		}

		// Just in case the user has entered a blank field
		if (strlen(desthost) == 0) {
			strncpy(desthost, "localhost", MAX_HOSTID);
		}

		hp = gethostbyname(desthost);
		if (hp == 0) {
			DebugMsg("Cannot resolve host %s\n", hostid);
			close(AuditSocket);
			return(0);
		}
		bzero((char *) &host.AuditSocketName, sizeof(host.AuditSocketName));
		memcpy(&host.AuditSocketName.sin_addr, hp->h_addr, hp->h_length);
		host.AuditSocketName.sin_family = AF_INET;
		host.AuditSocketName.sin_port = htons(port);
		host.dest_addr_size=sizeof(host.AuditSocketName);
		host.AuditSocket=AuditSocket;
		strncpy(host.desthost, desthost,MAX_HOSTID);
		host.port=port;
		host.protocol=protocol;
		host.next=NULL;

		HNodeCounter++;

		AuditDestination |= AUDIT_TO_NETWORK;
		if (port == 514) {
			AuditDestination |= AUDIT_TO_SYSLOG;
		}
		return(1);
	}

	return(0);
}

// For the moment, require the web server to do the work of converting
// destinations to numerics.
int get_syslog_dest(char *string)
{
	int destination;
	string += strlen("syslog=");
	destination = atoi(string);
	if (destination > 0) {
		return(destination);
	}
	return(0);
}

// Return the host identifier
char *getconfstring(char *string, char *file, int length)
{
	char *stringp = string;

	stringp = strstr(string, "=");
	if (stringp != (char *) NULL) {
		stringp++;
		if (strlen(stringp)) {
			strncpy(file, stringp, length - 1);
		} else {
			return((char *)NULL);
		}
	} else {
		return((char *)NULL);
	}
	return((char *)file);
}

int getport(char *string)
{
	char *stringp = string;
	char strPort[10];

	stringp = strstr(string, "=");

	if (stringp != (char *)NULL) {
		stringp++;
		if (strlen(stringp)) {
			strncpy(strPort, stringp, 10);
			return(atoi((char *)strPort));
		} else {
			return(0);
		}
	} else {
		return(0);
	}
}

void trimallwhitespace(char *string)
{
	char *readpointer;
	char *writepointer;

	// Verify that there is something to check.
	if (string == (char *) NULL) return;

	// And that there is some data within.
	if (strlen(string) == 0) return;

	// Start from the end, work backwards.
	readpointer = string;
	writepointer = string;
	while (readpointer < (string + strlen(string))) {
		if (*readpointer == ' ' || *readpointer == 9 || *readpointer == 10 || *readpointer == 12) {
			readpointer++;
		} else {
			if (writepointer != readpointer) {
				*writepointer = *readpointer;
			}
			readpointer++;
			writepointer++;
		}

	}
}

// Take an identified objective line
// NOTE: string and match must all be the same buffer size!
int splitobjective(char *string, char *match, int *excludematchflag)
{
	char *startmatch;
	char *stringpointer, *endstring;
	char *matchp;

	// Do some basic sanity checks.
	if (string == (char *) NULL || match == (char *) NULL) {
		return(-1);
	}
	if (!strlen(string)) {
		return(-1);
	}

	startmatch = strstr(string, "match=");
	if (startmatch == (char *)NULL) {
		startmatch = strstr(string, "match!=");
		if (startmatch != (char *)NULL) {
			// EXCLUDE rather than include.
			*excludematchflag = 1;
		}
	}

	// string pointers for iteration.
	stringpointer = string;
	matchp = match;

	// Pointer to the last character in the string.
	endstring = &string[strlen(string)];

	if (startmatch == (char *)NULL) {
		// Problem, this line is malformed. We really cannot proceed with this line.
		DebugMsg("The following line does not contain the match element: %s", string);
		return(-1);
	}

	if (*excludematchflag) {
		stringpointer = startmatch + strlen("match!=");
	} else {
		stringpointer = startmatch + strlen("match=");
	}

	while (*stringpointer && stringpointer != endstring) {
		*matchp = *stringpointer;
		stringpointer++;
		matchp++;
	}
	*matchp = '\0';
	// Remove extra whitespace at start and end.
	trim(match);

	return(1);
}

// {{{ Regular expression functions
// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
int regmatch(const char *string, const char *pattern)
{
	int status;
	regex_t re;

	if (regcomp(&re, pattern, REG_EXTENDED | REG_NOSUB) != 0) {
		return(0);
	}
	status = regexec(&re, string, (size_t) 0, NULL, 0);
	regfree(&re);
	if (status != 0) {
		return(0);
	}
	return(1);
}

// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
// COMPILED REGULAR EXPRESSION version
int regmatchC(const char *string, regex_t re)
{
	int status;

	status = regexec(&re, string, (size_t) 0, NULL, 0);
	if (status != 0) {
		return(0);
	}
	return(1);
}

// Match string against an extended regular expression. Case insensitive.
// Return 1 for match, 0 for no-match or error.
int regmatchi(const char *string, const char *pattern)
{
	int status;
	regex_t re;

	if (regcomp(&re, pattern, REG_EXTENDED | REG_NOSUB | REG_ICASE) != 0) {
		return(0);
	}

	status = regexec(&re, string, (size_t) 0, NULL, 0);
	regfree(&re);
	if (status != 0) {
		return(0);
	}
	return(1);
}
// }}}

Node *AddToList(char *path, int excludematchflag)
{
	Node *newNode = NULL;

	newNode = (Node *)malloc(sizeof(Node));

	if (newNode == NULL) {
		DebugMsg("AddToList(): error in dynamic memory allocation\nCould not add a new objective into our linked list. You may be low on memory.\n");
		return((Node *)NULL);
	}
	newNode->excludematchflag = excludematchflag;

	strncpy(newNode->path, path, PATH_MAX);
	// REDRED: NOTE: Must free regexp in objective list at end!
	regcomp(&newNode->pathRE, path, REG_EXTENDED | REG_NOSUB);

	if (head == NULL) {
		head = newNode;
		newNode->next = NULL;
	} else {
		newNode->next = head;
		head = newNode;
	}

	return newNode;
}

void ResetCurrentNode(void)
{
	currentnode = head;
}

int IsValidItem(void)
{
	return(NULL == currentnode) ? 0 : 1;
}

void NextItemInList(void)
{
	if (NULL == currentnode)
		return;

	currentnode = currentnode->next;
}

int CheckObjective(char *searchterm)
{
	ResetCurrentNode();
	//If there are no filter terms, match everything
	if (!IsValidItem()) {
		return(1);
	}

	while (IsValidItem()) {
		if (regmatchC(searchterm, currentnode->pathRE)) {
			if (currentnode->excludematchflag) {
				return(0);
			} else {
				return(1);
			}
		}
		NextItemInList();
	}

	ResetCurrentNode();
	return(1);
}

void DestroyList(void)
{
	if (NULL == head) {
		return;
	}

	while (NULL != head) {
		Node *tempPtr = head;
		head = head->next;

		// Free regular expressions.
		regfree(&tempPtr->pathRE);

		free(tempPtr);
	}
	close(host.AuditSocket);
	LCCurrent=LCHead;
	while(LCCurrent) {
		LCHead=LCCurrent->next;
		free(LCCurrent);
		LCCurrent=LCHead;
	}
	lcurrent=lhead;
	while(lcurrent) {
		lhead=lcurrent->next;
		if (lcurrent->fs) fclose(lcurrent->fs);
		free(lcurrent);
		lcurrent=lhead;
	}
}

// Pull out the fully qualified domain name, if possible.
char *getfqdn(char *FQDN)
{
	char hname[MAX_HOSTID];
	struct hostent *hp;

	if (gethostname(hname, MAX_HOSTID)) {
		strncpy(FQDN, "localhost.unknown", MAX_HOSTID);
		return(FQDN);
	}

	strncpy(FQDN, hname, MAX_HOSTID);

	hp = gethostbyname(hname);
	if (hp) {
		while (hp->h_aliases && *hp->h_aliases) {
			if (strlen(*(hp->h_aliases)) > strlen(hname) && !strncmp(hname, *(hp->h_aliases), strlen(hname))) {
				strncpy(FQDN, *(hp->h_aliases), MAX_HOSTID);
			}
			hp->h_aliases++;
		}
		if (strlen(hp->h_name) > strlen(hname) && !strncmp(hname, hp->h_name, strlen(hname))) {
			strncpy(FQDN, hp->h_name, MAX_HOSTID);
		}
	}

	return(FQDN);
}

int fix_last_error(HostNode *myhostcurrent) {
	time_t now;
	time(&now);
	if (now - myhostcurrent->last_error >= WAITTIME) {
		//try to reestablish the connection
		if (myhostcurrent->AuditSocket) close(myhostcurrent->AuditSocket);
		myhostcurrent->AuditSocket=0;
		myhostcurrent->AuditSocket = socket(AF_INET,SOCK_DGRAM,0);
		if (myhostcurrent->AuditSocket < 0) {
			// failed to grab a socket, go back to sleep
			DebugMsg("Cannot open network socket - continuing to wait\n");
			myhostcurrent->last_error=time(&myhostcurrent->last_error);
			myhostcurrent->AuditSocket = 0;
			return(1);
		}
		myhostcurrent->last_error = 0;
	} else {
		//still waiting
		return(1);
	}
	return(0);
}

void send_logs () {
	struct timespec timeout;
	time_t now;
	MsgCache *mymsg;

	pthread_mutex_lock(&send_mutex);
	while (!caught_kill) {
		// Waiting for head to be valid
		time(&now);
		// Run this thread every second, or when signaled.
		timeout.tv_sec = now + 1;
		timeout.tv_nsec = 0;
		// Ok, check for more data

		pthread_cond_timedwait(&log_data_available,&send_mutex,&timeout);
		mymsg = LCHead;
		while (mymsg && !caught_kill) {
			sendevent(mymsg->msg, mymsg->type);
			pthread_mutex_lock(&log_mutex);
			LCHead = mymsg->next;
			free(mymsg);
			LCCount--;
			mymsg = LCHead;
			if (!mymsg) {
				LCTail=NULL;
			}
			pthread_mutex_unlock(&log_mutex);
		}
	}
	pthread_mutex_unlock(&send_mutex);
}

////////////////////////////////////////////////////////////////////////////////
// This routine will send an event to the user-selected output device.
////////////////////////////////////////////////////////////////////////////////
int sendevent(char *string, char *type)
{
	// SEQ //static int snareseq = 1;
	char stringout[MAX_AUDITREC] = "";
	// SEQ //char stringoutend[MAX_AUDITREC] = "";
	

	// Audit record format, including host and Epilog identifier string:
	if (!strncmp(type,"GenericLog", 10))
		snprintf(stringout,MAX_AUDITREC,"%s\t%s\t0\t0\t0\t0\t",hostid, type);
	else
		snprintf(stringout,MAX_AUDITREC,"%s\t%s\t0\t",hostid, type);
	// SEQ //snprintf(stringoutend,MAX_AUDITREC,"\tsnareseq,%d",snareseq);
	// SEQ //strncat(stringout,string,MAX_AUDITREC - strlen(stringout) - strlen(stringoutend) - 1);
	strncat(stringout,string,MAX_AUDITREC - strlen(stringout) - 1);
	// SEQ //strncat(stringout,stringoutend,MAX_AUDITREC - strlen(stringout) - 1);

	// increment our snare sequence number
	// SEQ //snareseq++;

	//if (snareseq >= MAXINT) {
	//	snareseq=1;
	//}
	
	if (AuditDestination & AUDIT_TO_STDOUT) {
		// Send to STDOUT.
		printf("%s\n", stringout);
	}

	if (AuditDestination & AUDIT_TO_NETWORK) {
		if (AuditDestination & AUDIT_TO_SYSLOG) {
			// Do Syslog stuff here.
			time_t currenttime;
			struct tm *newtime;
			char CurrentDate[16] = "";

			time(&currenttime);
			newtime = localtime(&currenttime);
			syslogdate(CurrentDate, newtime);

			if (!strncmp(type,"GenericLog", 10))
				snprintf(stringout, MAX_AUDITREC, "<%d> %s %s %s\t0\t0\t0\t0\t", SyslogDestination, CurrentDate, hostid, type);
			else
				snprintf(stringout, MAX_AUDITREC, "<%d> %s %s %s\t0\t", SyslogDestination, CurrentDate, hostid, type);
			// SEQ //snprintf(stringoutend,MAX_AUDITREC,"\tsnareseq,%d",snareseq);
			// SEQ //strncat(stringout,string,MAX_AUDITREC - strlen(stringout) - strlen(stringoutend) - 1);
			strncat(stringout,string,MAX_AUDITREC - strlen(stringout) - 1);
			// SEQ //strncat(stringout,stringoutend,MAX_AUDITREC - strlen(stringout) - 1);
		}
		SendToSocket(stringout);
	}

	return(0);
}

int SendToSocket(char *message)
{
	int bytessent;

	if (!host.AuditSocket && !host.last_error) {
		// The socket is broken, flag an error and let last_error deal with it
		host.last_error=time(&host.last_error);
	}
	if (host.last_error) {
		fix_last_error(&host);
	}
	if (host.AuditSocket && !host.last_error) {
		// Send the packet to each of the destinations
		// defined in the linked list.
		bytessent = sendto(host.AuditSocket, message, strlen(message), 0, (const struct sockaddr *)&host.AuditSocketName, host.dest_addr_size);
		//DebugMsg("Bytes sent to %s: %d\n", host.desthost, bytessent);
		if (bytessent < 0) {
			DebugMsg("Error sending to server %s. Retrying in %d seconds\n", host.desthost, WAITTIME);
			// Create another cache entry
			host.last_error=time(&host.last_error);
			caught_pipe = 0;
		}
	}
	return(0);
}

int IsSocketValid(int sock) {
	fd_set connectionSet;
	struct timeval timeout;
	int result;

	//set timeout values
	timeout.tv_sec =0;
	timeout.tv_usec =250;

	FD_ZERO(&connectionSet);
	FD_SET(sock, &connectionSet);

	//call select to see if data is waiting on the socket
	result = select(sock+1, &connectionSet, NULL, NULL, &timeout);

	//check results
	//DebugMsg("SELECT result: %d\n", result);
	return (result == 0);
}

void syslogdate(char *sdate, struct tm *cdate)
{
	char Month[4];
	char Date[3];
	char Hour[3];
	char Min[3];
	char Sec[3];

	if (!sdate || !cdate) return;

	switch (cdate->tm_mon) {
		case 0: strcpy(Month, "Jan"); break;
		case 1: strcpy(Month, "Feb"); break;
		case 2: strcpy(Month, "Mar"); break;
		case 3: strcpy(Month, "Apr"); break;
		case 4: strcpy(Month, "May"); break;
		case 5: strcpy(Month, "Jun"); break;
		case 6: strcpy(Month, "Jul"); break;
		case 7: strcpy(Month, "Aug"); break;
		case 8: strcpy(Month, "Sep"); break;
		case 9: strcpy(Month, "Oct"); break;
		case 10: strcpy(Month, "Nov"); break;
		default: strcpy(Month, "Dec"); break;
	}

	if (cdate->tm_mday < 10) {
		snprintf(Date, 3, " %d\0", cdate->tm_mday);
	} else {
		snprintf(Date, 3, "%d\0", cdate->tm_mday);
	}

	if (cdate->tm_hour < 10) {
		snprintf(Hour, 3, "0%d\0", cdate->tm_hour);
	} else {
		snprintf(Hour, 3, "%d\0", cdate->tm_hour);
	}

	if (cdate->tm_min < 10) {
		snprintf(Min, 3, "0%d\0", cdate->tm_min);
	} else {
		snprintf(Min, 3, "%d\0", cdate->tm_min);
	}

	if (cdate->tm_sec < 10) {
		snprintf(Sec, 3, "0%d\0", cdate->tm_sec);
	} else {
		snprintf(Sec, 3, "%d\0", cdate->tm_sec);
	}

	snprintf(sdate, 16, "%s %s %s:%s:%s\0", Month, Date, Hour, Min, Sec);
}

int add_log_watch(char *string)
{
	char *pos, *pos2;
	string += strlen("log=");
	if (strlen(string)) {
		struct stat stats;
		lcurrent = (LogNode *)malloc(sizeof(LogNode));
		if (lcurrent) {
			pos = strstr(string, ":");
			pos2 = strstr(string, "/");
			if (pos && pos2 && pos < pos2) {
				pos2 = string;
				string = pos + 1;
				*pos='\0';
				strncpy(lcurrent->type, pos2, MAX_AUDIT_CONFIG_LINE);
			} else {
				//assume the default Epilog
				strncpy(lcurrent->type, "GenericLog", MAX_AUDIT_CONFIG_LINE);
			}
			// Record the name and open the file for reading
			strncpy(lcurrent->name, string, MAX_AUDIT_CONFIG_LINE);
			lcurrent->fs = fopen(lcurrent->name, "r");
			if (lcurrent->fs == NULL) {
				DebugMsg("Failed to grab file stream for %s\n", lcurrent->name);
				lcurrent->last_error=time(&lcurrent->last_error);
			} else {

				// Grab and record the current stats
				if (stat(lcurrent->name, &stats) < 0) {
					DebugMsg("Failed to find stats for %s\n", lcurrent->name);
					lcurrent->last_error=time(&lcurrent->last_error);
					return(0);
				}
				lcurrent->mtime = stats.st_mtim;
				lcurrent->size = stats.st_size;
				lcurrent->dev = stats.st_dev;
				lcurrent->ino = stats.st_ino;
				lcurrent->mode = stats.st_mode;
				lcurrent->last_error=0;
				lcurrent->next=NULL;
				lcurrent->pmsg[0]='\0';
				// Now that the file is ready, seek to the end of the file
				// and prepare to capture further output
				fseek(lcurrent->fs, lcurrent->size, SEEK_SET);
			}
			if (ltail) {
				ltail->next=lcurrent;
				ltail=lcurrent;
			}
			if (!lhead) {
				lhead=lcurrent;
				ltail=lcurrent;
			}
			return(1);
		} else {
			DebugMsg("Failed to allocate memory for log file\n");
			return(0);
		}
	}
	return(0);
}

void DebugMsg(const char *pszFormat, ...)
{
#ifdef DEBUG
	char buf[LOGBUFSIZE] = "";
	FILE *fp;
	va_list arglist;
	va_start(arglist, pszFormat);
	vsnprintf(&buf[strlen(buf)], LOGBUFSIZE - strlen(buf) - 1, pszFormat, arglist);
	va_end(arglist);


	pthread_mutex_lock(&debug_mutex);
	fp=fopen("/tmp/SNARE-OUT", "a");
	if (fp) {
		fputs(buf,fp);
		fclose(fp);
	} else {
		printf("\nERROR: There was an error opening the log file\n");
	}
	pthread_mutex_unlock(&debug_mutex);
#endif
}

//#################################

int file_has_changed (LogNode *f) {
	// Grab the new set of stats
	struct stat stats;

	if (stat(f->name, &stats) < 0) {
		if (!f->last_error) DebugMsg("Failed to find log file: %s\n", f->name);
		f->last_error=time(&f->last_error);
		return(-1);
	}
	// Firstly, check if the file has been rotated
	if (f->dev != stats.st_dev || f->ino != stats.st_ino || f->last_error) {
		FILE *fs = fopen(f->name, "r");
		// change detected, close the old file,
		// reset the stats and start from the begining
		DebugMsg("rotation detected\n");
		if (f->fs) fclose(f->fs);
		f->fs = fs;
		f->size = stats.st_size;
		f->mtime = stats.st_mtim;
		f->dev = stats.st_dev;
		f->ino = stats.st_ino;
		f->mode = stats.st_mode;
		f->last_error = 0;
		// seek to the begining of the file to grab anything we might have missed
		// fseek(fs, 0, SEEK_SET);
		return(1);
	}

	if (f->size == stats.st_size &&
	    f->mtime.tv_sec == stats.st_mtim.tv_sec &&
	    f->mtime.tv_nsec == stats.st_mtim.tv_nsec) {
		// Everything else is exactly the same
		// so do nothing for now
		return(0);
	} else {
		int change=0;
		// Something has changed, find out what and take action
		if (stats.st_size < f->size) {
			DebugMsg("%s: File truncated\n", f->name);
			change = 1;
		}
		if (f->mtime.tv_sec > stats.st_mtim.tv_sec) {
			DebugMsg("%s: Modified in the past\n");
			change = 1;
		} else if (f->mtime.tv_sec == stats.st_mtim.tv_sec && f->mtime.tv_nsec > stats.st_mtim.tv_nsec) {
			DebugMsg("%s: Modified in the past\n");
			change = 1;
		}

		// Reset the descriptors
		f->size = stats.st_size;
		f->mtime = stats.st_mtim;
		f->dev = stats.st_dev;
		f->ino = stats.st_ino;
		f->mode = stats.st_mode;
		if (change) {
			// Something went wrong, so ditch the old descriptor and
			// keep the new one.  However, since it was not a rotation,
			// i.e. we still have the same dev and inode numbers, then
			// seek to the beginning of the file and only return a change
			// if the file is larger than zero.
			FILE *fs = fopen(f->name, "r");
			if (f->fs) fclose(f->fs);
			f->fs = fs;
			//fseek(fs, f->size, SEEK_SET);
			if (f->size == (off_t) 0)
				return(0);
			else
				return(1);
		}
		return(1);
	}
}

void usage(char * exe) {
	if(!exe) { return; }
	printf("usage : %s [-c /path/to/config]\n", exe);
	printf("   eg : %s -c /etc/snare/epilog/squid.conf\n", exe);
}

