///////////////////////////////////////////////////////////////////////////////////////////
//
// System iNtrusion Analysis and Reporting Environment (SNARE) - Audit Daemon
//
// Audit Daemon
//
///////////////////////////////////////////////////////////////////////////////////////////
// Copyright 1999-2002 InterSect Alliance Pty Ltd - http://www.intersectalliance.com/
///////////////////////////////////////////////////////////////////////////////////////////


// #include <ctype.h>
#include <errno.h>		// perror()
#include <fcntl.h>
#include <signal.h>
#include <stdint.h>		// uint32_t
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>		// pause(), unlink()
#include <pwd.h>
#include <grp.h>
#include <time.h>
#include <regex.h>		// Regular expression matches.

#include <netdb.h>		// gethostbyname
#include <sys/socket.h>

#include <sys/mman.h>	// mmap()
#include <sys/stat.h>	// file modes
#include <sys/types.h>
//#include <sys/time.h>
//#include <sys/ioctl.h>

#include <sys/syscall.h>

#include <linux/kdev_t.h> 	// MAJOR and MINOR macros
#include <linux/limits.h>	// PATH_MAX
#include <linux/net.h>		// SYS_CONNECT and SYS_ACCEPT

#include "audit.h"
#include "auditd.h"


const char device_name[] = AUDITDEV_FILE;	// device name
int device_fd;								// File descriptor for /proc/audit
int process_id;								// Daemon's process id
int AuditMode=AUDIT_BY_EVENT;				// By Objective, or By Event.
int AuditDestination=AUDIT_TO_STDOUT;		// Send to STDOUT unless otherwise stated.
FILE * AuditFile=(FILE *) NULL;
int AuditSocket=0;
struct sockaddr_in	AuditSocketName;

// Linked List Functions
static Node *head, *currentnode;

// Host name - hopefully fully qualified
char hostid[MAX_HOSTID]="";


////////////////////////////////////////////////////////////////////////////////
// Set everything up, then wait for data to write out
////////////////////////////////////////////////////////////////////////////////
int main(int argc, char* argv[])
{
	struct sigaction act; /* For sigaction() */

	printf("SNARE audit daemon: version %d.%d%d starting up\nInter[01;31mS[0mect Alliance Pty Ltd\n[01;34mhttp://www.intersectalliance.com/[0m\n", AUDITMODULE_MAJOR_VERSION,
			AUDITMODULE_MINOR_VERSION, AUDITMODULE_PATCH_VERSION);
	
	// become daemon, die if it fails
	// Only do this if we are not sending data to stdout.
	
//	if((pid = fork()) < 0)
//		error_sys("auditd: can't become daemon, exiting \n");
//	else if(pid) // I'm the parent, suicide
//		exit(0);
		
//	printf("auditd: set up as daemon...\n");
	
	// start a new session and set up signal handling
	process_id = setsid();
	  
	act.sa_handler = sig_handler;
	sigfillset(&(act.sa_mask));	// block all signals while handling
	act.sa_flags = 0;			// no flags
	if(sigaction(SIGIO, &act, NULL) ||
	   sigaction(SIGTERM, &act, NULL) || sigaction(SIGHUP, &act, NULL) || sigaction(SIGINT, &act, NULL) || sigaction(SIGQUIT, &act, NULL) ||
	   sigaction(SIGALRM, &act, NULL) ||
	   sigaction(SIGUSR1, &act, NULL) ||									// Display statistics and information
	   sigaction(SIGTSTP, &act, NULL) || sigaction(SIGCONT, &act, NULL))	// Suspend
	{
		perror("auditd: problem setting signals, exiting");
		exit(1);
	}

	// Read configuration file.
	if((device_fd = open(device_name, O_RDONLY, 0)) < 0)
	{
		perror("SNARE audit daemon: I cannot connect to the audit module via the file /proc/audit. Audit module not yet loaded?\nTry installing the audit module using: insmod auditmodule");
		exit(1);
	}

	// Initialise our linked list.
	CreateLinkedList();

	// First, flush existing audit events that may have been left over
	// From a previous auditd incarnation
	if(ioctl(device_fd, AUDIT_FLUSH, 0) < 0)
	{
		close(device_fd);
		if(AuditDestination & AUDIT_TO_FILE) {
			if(AuditFile != (FILE *) NULL) {
				fclose(AuditFile);
			}
		}
		if(AuditDestination & AUDIT_TO_NETWORK) {
			if(AuditSocket) {
				close(AuditSocket);
			}
		}

		perror("audit daemon: can't flush audit events. Exiting.");
		DestroyList();
		exit(1);
	}

	// Here: read config file, set audit events, etc.
	// Auditmode is either by event, or by objective, or -1 for a failure
	AuditMode=read_config_file();
	
	// Fill in the hostid with the fully qualified domain name,
	// if it hasn't already been grabbed from the configuration file.
	if(!hostid) {
		getfqdn(hostid);
	} else {
		if(strlen(hostid)==0) {
			getfqdn(hostid);
		}
	}
	
	ResetCurrentNode();
	
	printf("SNARE audit daemon: driver open, starting audit\n");

	if(ioctl(device_fd, AUDIT_START, 0) < 0)
	{
		int a;
		a=ioctl(device_fd,AUDIT_START,0);
		
		close(device_fd);
		DestroyList();
		perror("auditd: can't start auditing, exiting");
		exit(1);
	}

	// sleep until we're signaled to get some data by the audit module
	while(1)
		pause();

	// if we're here, something went very wrong

	DestroyList();
	perror("Audit Daemon has terminated unexpectedly");
	
	exit(1);
}

////////////////////////////////////////////////////////////////////////////////
// Process signals sent by the audit kernel module
////////////////////////////////////////////////////////////////////////////////


/// Could ioctl to the auditmodule to start/stop events.
/// AUDIT_open + 1000 = start auditing this event
/// AUDIT_open + 2000 = stop auditing this event

void sig_handler(int signal_id)
{
	static int AUDITD_BUSY=0;
	static header_token header;
	static char tempbuffer[MAX_AUDITREC]; // useless data in event of corruption

	if(AUDITD_BUSY && signal_id == SIGIO)
	{
		// We have received a SIGIO, but we're in the middle of reading stuff
		// already. Ignore the signal.
		// printf("go, I say, go away boy, yer botherin me\n");
		return;
	}

	// Changed - 18/7/02 - Try and flush buffers, regardless of what signal we have received.	
	if(signal_id == SIGIO || signal_id == SIGINT)
	{
		AUDITD_BUSY=1;

		// Read the header data from the device, and determine the size of the remaining record.
		while(read(device_fd,&header,sizeof(header_token)) >0)
		{
			// Work out what type of event it is.
			if(header.event_class == AUDIT_CLASS_IO) {
				process_io_class(&header);
			} else if (header.event_class == AUDIT_CLASS_EXEC) {
				process_ex_class(&header);
			} else if (header.event_class == AUDIT_CLASS_PC) {
				process_pc_class(&header);
			} else if (header.event_class == AUDIT_CLASS_CH) {
				process_ch_class(&header);
			} else if (header.event_class == AUDIT_CLASS_CP) {
				process_cp_class(&header);
			} else if (header.event_class == AUDIT_CLASS_SU) {
				process_su_class(&header);
			} else if (header.event_class == AUDIT_CLASS_NET) {
				process_nt_class(&header);
			} else if (header.event_class == AUDIT_CLASS_AD) {
				process_ad_class(&header);
			} else {
				printf("Auditd: Problem - Data read by daemon is not a recognised audit class - %d\n",header.event_class);
				// Note: Only read to a max of 4096 here. There is an outside chance that
				// something may have been corrupted, and I don't want any overflows.
				if(header.event_size > MAX_AUDITREC) header.event_size=MAX_AUDITREC;
				read(device_fd,&tempbuffer,header.event_size);
			}
		}
		AUDITD_BUSY=0;
	} // else
	if(signal_id == SIGTERM || signal_id == SIGALRM || signal_id == SIGHUP
	                               || signal_id == SIGINT || signal_id == SIGQUIT ) {
		// stop the module from auditing events
		ioctl(device_fd, AUDIT_STOP, 0);

		close(device_fd);

		if(AuditDestination & AUDIT_TO_FILE) {
			if(AuditFile != (FILE *) NULL) {
				fclose(AuditFile);
			}
		} 
		if(AuditDestination & AUDIT_TO_NETWORK) {
			if(AuditSocket) {
				close(AuditSocket);
			}
		}
		
		DestroyList();
		printf("auditd: finished\n");
		exit(0);
	} else if(signal_id == SIGTSTP ) {
		// CTRL-Z received. Tell the audit daemon to stop sending data.
		// IGNORE suspend signals.
		// ioctl(device_fd, AUDIT_STOP, 0);
	} else if(signal_id == SIGCONT) {
		// And continue signals.
		// ioctl(device_fd, AUDIT_START, 0);
	} else if(signal_id == SIGUSR1) {
		// Could use HUP here instead...
		// Re-read the configuration file.
		// First, flush existing audit events.
		if(ioctl(device_fd, AUDIT_FLUSH, 0) < 0)
		{
			close(device_fd);
			if(AuditDestination & AUDIT_TO_FILE) {
				if(AuditFile != (FILE *) NULL) {
					fclose(AuditFile);
				}
			}
			if(AuditDestination & AUDIT_TO_NETWORK) {
				if(AuditSocket) {
					close(AuditSocket);
				}
			}

			perror("audit daemon: can't flush audit events. Exiting.");
			DestroyList();
			exit(1);
		}

		// Free the old linked list.
		DestroyList();
		CreateLinkedList();
		strcpy(hostid,"");

		if(AuditDestination & AUDIT_TO_FILE) {
			// And close the old audit file.
			if(AuditFile != (FILE *) NULL) {
				fclose(AuditFile);
			}
		}
		if(AuditDestination & AUDIT_TO_NETWORK) {
			if(AuditSocket) {
				close(AuditSocket);
			}
		}

		// Reset the event count and the event lost count.
		
		AuditMode=read_config_file();

		// Fill in the hostid with the fully qualified domain name,
		// if it hasn't already been grabbed from the configuration file.
		if(!hostid) {
			getfqdn(hostid);
		} else {
			if(strlen(hostid)==0) {
				getfqdn(hostid);
			}
		}
		
		ResetCurrentNode();
	}

}

int turn_event_on(int eventnumber)
{
	return(ioctl(device_fd,AUDIT_EVENT_ON,eventnumber));
}

////////////////////////////////////////////////////////////////////////////////
// Process events that relate to input/output
////////////////////////////////////////////////////////////////////////////////
int process_io_class(header_token *header)
{
	io_class	io_event;
	char eventdata [MAX_AUDITREC]="";		// Event data storage.
	char tempdata [MAX_AUDITREC]="";		// for temporary storage if required.
	char objectivetext[MAX_AUDITREC]="";	// For the objective text, if required.
	int ObjectiveReturn = RETURNCODE_SUCCESS;
	char options[MAX_OPTIONS]="";

	io_event.t_header = *header;
	
	// Note that this assumes that the return_token will ALWAYS be the second element in each structure.
	read(device_fd,&io_event.t_return,io_event.t_header.event_size);

	strcpy(eventdata,"event");
	strcpy(objectivetext,"");
	eventcat(eventdata,IDELIMITER);	// usually comma.

	if(io_event.t_header.event_id == SYS_open || io_event.t_header.event_id == SYS_creat)
	{
		if(io_event.t_header.event_id == SYS_open) {
			eventcat(eventdata,"open(");
			if(io_event.t_return.returncode >= 0) {
				eventcat(objectivetext,"The file %s has been opened ");
				ObjectiveReturn=RETURNCODE_SUCCESS;
			} else {
				eventcat(objectivetext,"A failed attempt was made to open the file %s ");
				ObjectiveReturn=RETURNCODE_FAILURE;
			}
		} else if (io_event.t_header.event_id == SYS_creat) {
			eventcat(eventdata,"creat(");
		}

		// GNU allows O_RDONLY|O_WRONLY as a valid alternative to O_RDWR
		if(io_event.t_attributes.mode & O_WRONLY && (!(io_event.t_attributes.mode & O_RDONLY))) {
			eventcat(tempdata,"O_WRONLY");
			eventcat(objectivetext,"(write-only)");
		} else if(io_event.t_attributes.mode & O_RDWR || ((io_event.t_attributes.mode & O_WRONLY) && (io_event.t_attributes.mode & O_RDONLY))) {
			eventcat(tempdata,"O_RDWR");
			eventcat(objectivetext,"(for both read and write)");
		} else {
			eventcat(tempdata,"O_RDONLY");
			eventcat(objectivetext,"(read only)");
		}
		eventcat(objectivetext," by the user %s");

		// Now for all the additional flags
		if(io_event.t_attributes.mode & O_CREAT) {		eventcat(tempdata,"|O_CREAT");		}
		if(io_event.t_attributes.mode & O_EXCL) {		eventcat(tempdata,"|O_EXCL");		}
		if(io_event.t_attributes.mode & O_NOCTTY) {		eventcat(tempdata,"|O_NOCTTY");		}
		if(io_event.t_attributes.mode & O_TRUNC) {		eventcat(tempdata,"|O_TRUNC");		}
		if(io_event.t_attributes.mode & O_APPEND) {		eventcat(tempdata,"|O_APPEND");		}
		if(io_event.t_attributes.mode & O_NONBLOCK) {	eventcat(tempdata,"|O_NONBLOCK");	}
		if(io_event.t_attributes.mode & O_SYNC) {		eventcat(tempdata,"|O_SYNC");		}
#ifdef __USE_GNU
		if(io_event.t_attributes.mode & O_NOFOLLOW) {	eventcat(tempdata,"|O_NOFOLLOW");	}
		if(io_event.t_attributes.mode & O_DIRECTORY) {	eventcat(tempdata,"|O_DIRECTORY");	}
#endif
#ifdef __USE_LARGEFILE64
		if(io_event.t_attributes.mode & O_LARGEFILE) {	eventcat(tempdata,"|O_LARGEFILE");	}
#endif

		strncat(options,tempdata,MAX_OPTIONS);
		eventcat(eventdata,tempdata);

		eventcat(eventdata,")");
	} else if(io_event.t_header.event_id == SYS_mkdir) {
		eventcat(eventdata,"mkdir()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The directory %s has been created by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to create the directory %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(io_event.t_header.event_id == SYS_unlink) {
		eventcat(eventdata,"unlink()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The file %s has been removed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to remove the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(io_event.t_header.event_id == SYS_mknod) {
		int delim=0;	// temporary variable to indicate whether a | is required.
		eventcat(eventdata,"mknod(");
		if(io_event.t_attributes.mode & S_IFREG) {
			eventcat(eventdata,"S_IFREG");
			delim=1;
		} else if(io_event.t_attributes.mode & S_IFCHR) {
			if(delim) eventcat(eventdata,"|");
			eventcat(eventdata,"S_IFCHR");
			delim=1;
		} else if(io_event.t_attributes.mode & S_IFBLK) {
			if(delim) eventcat(eventdata,"|");
			eventcat(eventdata,"S_IFBLK");
			delim=1;
		} else if(io_event.t_attributes.mode & S_IFBLK) {
			if(delim) eventcat(eventdata,"|");
			eventcat(eventdata,"S_IFIFO");
		}
		// Otherwise, it is a normal file.
		eventcat(eventdata,")");
		
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The special file %s has been created by the user %s using mknod");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to create the special file %s by the user %s using mknod");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
		
	} else if(io_event.t_header.event_id == SYS_rmdir) {
		eventcat(eventdata,"rmdir()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The directory %s has been removed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to remove the directory %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(io_event.t_header.event_id == SYS_chmod) {
		eventcat(eventdata,"chmod()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The permissions associated with the file %s have been changed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the permissions of the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(io_event.t_header.event_id == SYS_truncate) {
		eventcat(eventdata,"truncate()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The file %s was truncated by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to truncate the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}

	} else if(io_event.t_header.event_id == SYS_chroot) {
		eventcat(eventdata,"chroot()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The directory %s has now been set to the system root by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to set the system root to the directory %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(io_event.t_header.event_id == SYS_ftruncate) {
		eventcat(eventdata,"ftruncate()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The file %s was truncated by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to truncate the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(io_event.t_header.event_id == SYS_fchmod) {
		eventcat(eventdata,"fchmod()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The permissions associated with the file %s have been changed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the permissions of the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(io_event.t_header.event_id == SYS_ftruncate64) {
		eventcat(eventdata,"ftruncate64()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The file %s was truncated by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to truncate the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(io_event.t_header.event_id == SYS_truncate64) {
		eventcat(eventdata,"truncate64()");
		if(io_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The file %s was truncated by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to truncate the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else {
		printf("Unknown event type..\n");
		return(0);
	}

	// Date/time
	eventcat(eventdata,IDELIMITER);
	strncpy(tempdata,ctime(&io_event.t_header.time.tv_sec),MAX_AUDITREC);
	// Remove the newline character at the end
	tempdata[strlen(tempdata)-1]='\0';
	eventcat(eventdata,tempdata);

	eventcat(eventdata,DELIMITER);

	// User
	// UID

	// Only problem with the stuff below is that for each file open (for example),
	// another 4 file opens will be generated as the getpwuid/getgrgid tasks
	// interrogate the password file. Note that events generated by the audit daemon are ignored though.
	// Still, I'll try to keep these calls to a minimum.
	add_user_token(eventdata, &(io_event.t_header));
	eventcat(eventdata,DELIMITER);

	add_process_token(eventdata, &(io_event.t_process));
	eventcat(eventdata,DELIMITER);

	// Use realpath to work out the correct path for the file.
	eventcat(eventdata,"path");
	eventcat(eventdata,IDELIMITER);
	eventcat(eventdata,resolved_path(io_event.t_pwd.path,io_event.t_path.path,tempdata));
	eventcat(eventdata,DELIMITER);

	// NOTE: this is the requested file write mode.
	// It will be modified by the processes UMASK
	// Thus, the permissions shown may not represent the final file.
	// open events - only available if file created
	if((io_event.t_header.event_id == SYS_open && (io_event.t_attributes.mode & O_CREAT)) ||
	   io_event.t_header.event_id == SYS_mkdir)
	{
		eventcat(eventdata,"attributes");
		eventcat(eventdata,IDELIMITER);
		eventcat(eventdata,textmode(io_event.t_attributes.createmode,tempdata,sizeof(tempdata)));
		eventcat(eventdata,DELIMITER);

	} else if (io_event.t_header.event_id == SYS_mknod && (io_event.t_attributes.mode & S_IFBLK || io_event.t_attributes.mode & S_IFCHR)) {

		// Mknod
		eventcat(eventdata,"attributes");
		eventcat(eventdata,IDELIMITER);

		snprintf(tempdata,MAX_AUDITREC,"%d %d",(unsigned short)MAJOR(io_event.t_attributes.createmode), (unsigned short)MINOR(io_event.t_attributes.createmode));
		eventcat(eventdata,tempdata);
		eventcat(eventdata,DELIMITER);
	} else if(io_event.t_header.event_id == SYS_chmod || io_event.t_header.event_id == SYS_fchmod) {
		eventcat(eventdata,"attributes");
		eventcat(eventdata,IDELIMITER);
		eventcat(eventdata,textmode(io_event.t_attributes.createmode,tempdata,sizeof(tempdata)));
		eventcat(eventdata,DELIMITER);	
	} else if(io_event.t_header.event_id == SYS_truncate || io_event.t_header.event_id == SYS_ftruncate
	          || io_event.t_header.event_id == SYS_truncate64 || io_event.t_header.event_id == SYS_ftruncate64) {
		eventcat(eventdata,"attributes");
		eventcat(eventdata,IDELIMITER);
		snprintf(tempdata,MAX_AUDITREC,"%ld",io_event.t_attributes.createmode);
		eventcat(eventdata,tempdata);
		eventcat(eventdata,DELIMITER);	
	}

	// Return code
	add_return_token(eventdata,&(io_event.t_return));
	eventcat(eventdata,DELIMITER);
	add_sequence_token(eventdata);
	
	// If we are in objective mode, check the path and userid against our objectives.
	if(AuditMode == AUDIT_BY_OBJECTIVE) {
		char objectivedata [MAX_AUDITREC]="";
		char pathstore [PATH_MAX];
		char userstore [MAX_USERNAME];	// User name - effective userid
		int returncode;
		int matchcount=0;
		Node * nodematch;
		
		// ObjectiveReturn is the returncode mangled into SUCCESS or FAILURE.
		returncode=ObjectiveReturn;

		strncpy(pathstore,resolved_path(io_event.t_pwd.path,io_event.t_path.path,tempdata),PATH_MAX);
		strncpy(userstore,passname(io_event.t_header.euser_id,tempdata,MAX_AUDITREC),MAX_USERNAME);

		// Check this data against each defined objective		
		while((nodematch=CheckObjective(io_event.t_header.event_id,userstore,
						   pathstore,options,returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&io_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			// passname causes a file open remember... might be a better way?
			eventcat(objectivedata,IDELIMITER);

			snprintf(tempdata,MAX_AUDITREC,objectivetext,pathstore,userstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}
		if(matchcount==0) {
			// We are auditing by objective, but no objectives match.
			// Dump eventdata and return.
			return(0);
		}
	} else {
		sendevent(eventdata);
	}

	return(0);

}


////////////////////////////////////////////////////////////////////////////////
// Process events that relate to input/output CHMODs
////////////////////////////////////////////////////////////////////////////////
int process_ch_class(header_token *header)
{
	ch_class	ch_event;
	char eventdata [MAX_AUDITREC];	// will probably need to make this bigger.
	char tempdata [MAX_AUDITREC];	// for temporary storage if required.
	char objectivetext[MAX_AUDITREC]="";	// For the objective text, if required.
	int ObjectiveReturn = RETURNCODE_SUCCESS;

	ch_event.t_header = *header;

	// Note that this assumes that the path_token will ALWAYS be the second element in each structure.

	read(device_fd,&ch_event.t_return,ch_event.t_header.event_size);

	strcpy(objectivetext,"");
	strcpy(eventdata,"event");
	eventcat(eventdata,IDELIMITER);	// usually comma.

	if(ch_event.t_header.event_id == SYS_chown) {
		eventcat(eventdata,"chown()");
		if(ch_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The owner associated with the file %s have been changed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the owner of the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(ch_event.t_header.event_id == SYS_lchown) {
		eventcat(eventdata,"lchown()");
		if(ch_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The owner associated with the file %s have been changed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the owner of the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(ch_event.t_header.event_id == SYS_chown32) {
		eventcat(eventdata,"chown32()");
		if(ch_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The owner associated with the file %s have been changed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the owner of the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(ch_event.t_header.event_id == SYS_lchown32) {
		eventcat(eventdata,"lchown32()");
		if(ch_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The owner associated with the file %s have been changed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the owner of the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(ch_event.t_header.event_id == SYS_fchown) {
		eventcat(eventdata,"fchown()");
		if(ch_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The owner associated with the file %s have been changed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the owner of the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else {
		printf("Unknown event type..\n");
		return(0);
	}

	// Date/time
	eventcat(eventdata,IDELIMITER);
	strcpy(tempdata,ctime(&ch_event.t_header.time.tv_sec));
	// Remove the newline character at the end
	tempdata[strlen(tempdata)-1]='\0';
	eventcat(eventdata,tempdata);

	eventcat(eventdata,DELIMITER);

	add_user_token(eventdata, &(ch_event.t_header));
	eventcat(eventdata,DELIMITER);

	add_process_token(eventdata, &(ch_event.t_process));
	eventcat(eventdata,DELIMITER);

	// Use realpath to work out the correct path for the file.
	eventcat(eventdata,"path");
	eventcat(eventdata,IDELIMITER);
	eventcat(eventdata,resolved_path(ch_event.t_pwd.path,ch_event.t_path.path,tempdata));
	eventcat(eventdata,DELIMITER);

	if(ch_event.t_header.event_id == SYS_chown || ch_event.t_header.event_id == SYS_lchown
	   || ch_event.t_header.event_id == SYS_chown32 || ch_event.t_header.event_id == SYS_lchown32
	   || ch_event.t_header.event_id == SYS_fchown) {
		// owner
		eventcat(eventdata,"owner");
		eventcat(eventdata,IDELIMITER);
		eventcat(eventdata,passname(ch_event.t_owner.owner,tempdata,MAX_AUDITREC));

		eventcat(eventdata,"(");
		snprintf(tempdata,MAX_AUDITREC,"%d",ch_event.t_owner.owner);
		eventcat(eventdata,tempdata);

		// I was getting a segfault here under 2.2 for some reason if user or group id
		// are set to uint rather than ushort.
		// Not really sure why. The structure is not going out of scope or anything.
		eventcat(eventdata,")");
		eventcat(eventdata,IDELIMITER);
		eventcat(eventdata,groupname(ch_event.t_owner.group,tempdata,MAX_AUDITREC));

		eventcat(eventdata,"(");
		snprintf(tempdata,MAX_AUDITREC,"%d",(gid_t)ch_event.t_owner.group);
		eventcat(eventdata,tempdata);

		eventcat(eventdata,")");
		eventcat(eventdata,DELIMITER);
	}

	// Return code
	add_return_token(eventdata,&(ch_event.t_return));
	eventcat(eventdata,DELIMITER);
	add_sequence_token(eventdata);

	// If we are in objective mode, check the path and userid against our objectives.
	if(AuditMode == AUDIT_BY_OBJECTIVE) {
		char objectivedata [MAX_AUDITREC]="";
		char pathstore [PATH_MAX];
		char userstore [MAX_USERNAME];	// User name - effective userid
		int returncode;
		int matchcount=0;
		Node * nodematch;

		// ObjectiveReturn is the returncode mangled into SUCCESS or FAILURE.
		returncode=ObjectiveReturn;

		strncpy(pathstore,resolved_path(ch_event.t_pwd.path,ch_event.t_path.path,tempdata),PATH_MAX);
		strncpy(userstore,passname(ch_event.t_header.euser_id,tempdata,MAX_AUDITREC),MAX_USERNAME);

		// Check this data against each defined objective		
		while((nodematch=CheckObjective(ch_event.t_header.event_id,userstore,
						   pathstore,"",returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&ch_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,IDELIMITER);
			snprintf(tempdata,MAX_AUDITREC,objectivetext,pathstore,userstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}
		if(matchcount==0) {
			// We are auditing by objective, but no objectives match.
			// Dump eventdata and return.
			return(0);
		}
	} else {
		sendevent(eventdata);
	}

	return(0);

}


////////////////////////////////////////////////////////////////////////////////
// Process execution events
////////////////////////////////////////////////////////////////////////////////
int process_ex_class(header_token *header)
{
	ex_class	ex_event;
	char eventdata [MAX_AUDITREC];	// will probably need to make this bigger.
	char tempdata [MAX_AUDITREC];	// for temporary storage if required.
	char objectivetext[MAX_AUDITREC]="";	// For the objective text, if required.
	int ObjectiveReturn = RETURNCODE_SUCCESS;

	ex_event.t_header = *header;
	
	read(device_fd,&ex_event.t_return,ex_event.t_header.event_size);
	strcpy(objectivetext,"");
	strcpy(eventdata,"event");
	eventcat(eventdata,IDELIMITER);	// usually comma.

	if(ex_event.t_header.event_id == SYS_execve)
	{
		eventcat(eventdata,"execve()");
		if(ex_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The program %s has been executed by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to execute the program %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}

	} else {
		printf("Unknown event type..\n");
		return(0);
	}

	// Date/time
	eventcat(eventdata,IDELIMITER);
	strcpy(tempdata,ctime(&ex_event.t_header.time.tv_sec));
	// Remove the newline character at the end
	tempdata[strlen(tempdata)-1]='\0';
	// Need to kill the newline off the end of the ctime return value.
	eventcat(eventdata,tempdata);

	eventcat(eventdata,DELIMITER);

	add_user_token(eventdata, &(ex_event.t_header));
	eventcat(eventdata,DELIMITER);

	add_process_token(eventdata, &(ex_event.t_process));
	eventcat(eventdata,DELIMITER);

	// Use realpath to work out the correct path for the file.
	eventcat(eventdata,"path");
	eventcat(eventdata,IDELIMITER);
	eventcat(eventdata,resolved_path(ex_event.t_pwd.path,ex_event.t_path.path,tempdata));
	eventcat(eventdata,DELIMITER);

	eventcat(eventdata,"arguments");
	eventcat(eventdata,IDELIMITER);
	eventcat(eventdata,strip(ex_event.t_execargs.args));

	eventcat(eventdata,DELIMITER);

	add_return_token(eventdata,&(ex_event.t_return));
	eventcat(eventdata,DELIMITER);
	add_sequence_token(eventdata);

	// If we are in objective mode, check the path and userid against our objectives.
	if(AuditMode == AUDIT_BY_OBJECTIVE) {
		char objectivedata [MAX_AUDITREC]="";
		char pathstore [PATH_MAX];
		char userstore [MAX_USERNAME];	// User name - effective userid
		int returncode;		
		int matchcount=0;
		Node * nodematch;

		// ObjectiveReturn is the returncode mangled into SUCCESS or FAILURE.
		returncode=ObjectiveReturn;

		strncpy(pathstore,resolved_path(ex_event.t_pwd.path,ex_event.t_path.path,tempdata),PATH_MAX);
		strncpy(userstore,passname(ex_event.t_header.euser_id,tempdata,MAX_AUDITREC),MAX_USERNAME);
		
		// Check this data against each defined objective		
		while((nodematch=CheckObjective(ex_event.t_header.event_id,userstore,
						   pathstore,"",returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&ex_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			// passname causes a file open remember... might be a better way?
			eventcat(objectivedata,IDELIMITER);
			snprintf(tempdata,MAX_AUDITREC,objectivetext,pathstore,userstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}
		if(matchcount==0) {
			// We are auditing by objective, but no objectives match.
			// Dump eventdata and return.
			return(0);
		}
	} else {
		sendevent(eventdata);
	}
	
	return(0);

}

////////////////////////////////////////////////////////////////////////////////
// Process process-control events.
////////////////////////////////////////////////////////////////////////////////
int process_pc_class(header_token *header)
{
	pc_class	pc_event;
	char eventdata [MAX_AUDITREC];	// will probably need to make this bigger.
	char tempdata [MAX_AUDITREC];	// for temporary storage if required.
	char objectivetext[MAX_AUDITREC]="";	// For the objective text, if required.
	int ObjectiveReturn = RETURNCODE_SUCCESS;

	pc_event.t_header = *header;

	read(device_fd,&pc_event.t_return,pc_event.t_header.event_size);

	strcpy(objectivetext,"");
	strcpy(eventdata,"event");
	eventcat(eventdata,IDELIMITER);	// usually comma.

	if(pc_event.t_header.event_id == SYS_exit)
	{
		eventcat(eventdata,"exit()");
		// Note: exit does not return.. put this in for completeness though.
		if(pc_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s, owned by the user %s, has exited");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to exit the process %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if (pc_event.t_header.event_id == SYS_reboot) {
		eventcat(eventdata,"reboot()");
		if(pc_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s, owned by the user %s, has issued a reboot command");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to reboot the system from the process %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else {
		printf("Unknown event type..\n");
		return(0);
	}

	// Date/time
	eventcat(eventdata,IDELIMITER);
	strcpy(tempdata,ctime(&pc_event.t_header.time.tv_sec));
	// Remove the newline character at the end
	tempdata[strlen(tempdata)-1]='\0';
	eventcat(eventdata,tempdata);

	eventcat(eventdata,DELIMITER);

	add_user_token(eventdata, &(pc_event.t_header));
	eventcat(eventdata,DELIMITER);

	add_process_token(eventdata, &(pc_event.t_process));
	eventcat(eventdata,DELIMITER);
	add_return_token(eventdata,&(pc_event.t_return));
	eventcat(eventdata,DELIMITER);
	add_sequence_token(eventdata);

	// If we are in objective mode, check the path and userid against our objectives.
	if(AuditMode == AUDIT_BY_OBJECTIVE) {
		char objectivedata [MAX_AUDITREC]="";
		char userstore [MAX_USERNAME];	// User name - effective userid
		char commandstore[MAXCOMMAND];
		int returncode;		
		int matchcount=0;
		Node * nodematch;

		// ObjectiveReturn is the returncode mangled into SUCCESS or FAILURE.
		returncode=ObjectiveReturn;

		strncpy(userstore,passname(pc_event.t_header.euser_id,tempdata,MAX_AUDITREC),MAX_USERNAME);
		strncpy(commandstore,pc_event.t_process.name,MAXCOMMAND);
	
		// Check this data against each defined objective
		// the match term is the command that performed the process control function.		
		while((nodematch=CheckObjective(pc_event.t_header.event_id,userstore,
						   commandstore,"",returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&pc_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			// passname causes a file open remember... might be a better way?
			eventcat(objectivedata,IDELIMITER);
			// Should probably change this so that the text is more representative of the actual event.
			snprintf(tempdata,MAX_AUDITREC,objectivetext,commandstore,userstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}
		if(matchcount==0) {
			// We are auditing by objective, but no objectives match.
			// Dump eventdata and return.
			return(0);
		}
	} else {
		sendevent(eventdata);
	}

	return(0);

}

////////////////////////////////////////////////////////////////////////////////
// Process network related events.
////////////////////////////////////////////////////////////////////////////////
int process_nt_class(header_token *header)
{
	nt_class	nt_event;
	char eventdata [MAX_AUDITREC];	// will probably need to make this bigger.
	char tempdata [MAX_AUDITREC];	// for temporary storage if required.
	char objectivetext[MAX_AUDITREC]="";	// For the objective text, if required.
	int ObjectiveReturn = RETURNCODE_SUCCESS;
	char options[MAX_OPTIONS]="";

	nt_event.t_header = *header;

	read(device_fd,&nt_event.t_return,nt_event.t_header.event_size);

	strcpy(objectivetext,"");
	strcpy(eventdata,"event");
	eventcat(eventdata,IDELIMITER);	// usually comma.

	if(nt_event.t_header.event_id == SYS_socketcall) {
		eventcat(eventdata,"socketcall(");
		
		if(nt_event.syscall==SYS_ACCEPT) {
			eventcat(eventdata,"ACCEPT)");
			strcpy(options,"ACCEPT");		// This will allow users to specify something like socketcall(ACCEPT) .. or socketcall(.*)
			
			if(nt_event.t_return.returncode >= 0) {
				eventcat(objectivetext,"The process %s, owned by the user %s, has accepted a connection from port ");
				snprintf(tempdata,MAX_AUDITREC,"%d on remote system %s",nt_event.t_connection.src_port,nt_event.t_connection.src_ip);
				eventcat(objectivetext,tempdata);
				ObjectiveReturn=RETURNCODE_SUCCESS;
			} else {
				eventcat(objectivetext,"A network accept failed from the process %s by the user %s from port ");
				snprintf(tempdata,MAX_AUDITREC,"%d on %s",nt_event.t_connection.src_port,nt_event.t_connection.src_ip);
				eventcat(objectivetext,tempdata);
				ObjectiveReturn=RETURNCODE_FAILURE;
			}
		} else if(nt_event.syscall==SYS_CONNECT) {
			eventcat(eventdata,"CONNECT)");
			strcpy(options,"CONNECT");
			if(nt_event.t_return.returncode >= 0) {
				eventcat(objectivetext,"The process %s, owned by the user %s, has connected to port ");
				snprintf(tempdata,MAX_AUDITREC,"%d on remote system %s",nt_event.t_connection.dst_port,nt_event.t_connection.dst_ip);
				eventcat(objectivetext,tempdata);
				ObjectiveReturn=RETURNCODE_SUCCESS;
			} else {
				eventcat(objectivetext,"A network connect failed from the process %s by the user %s to port ");
				snprintf(tempdata,MAX_AUDITREC,"%d on %s",nt_event.t_connection.dst_port,nt_event.t_connection.dst_ip);
				eventcat(objectivetext,tempdata);
				ObjectiveReturn=RETURNCODE_FAILURE;
			}
		} else {
			eventcat(eventdata,"UNKNOWN)");
			if(nt_event.t_return.returncode >= 0) {
				eventcat(objectivetext,"The process %s, owned by the user %s, has issued a socketcall command");
				ObjectiveReturn=RETURNCODE_SUCCESS;
			} else {
				eventcat(objectivetext,"A failed attempt was made from the process %s by the user %s to issue a socketcall command");
				ObjectiveReturn=RETURNCODE_FAILURE;
			}
		}
	} else {
		printf("Unknown event type..\n");
		return(0);
	}

	// Date/time
	eventcat(eventdata,IDELIMITER);
	strcpy(tempdata,ctime(&nt_event.t_header.time.tv_sec));
	// Remove the newline character at the end
	tempdata[strlen(tempdata)-1]='\0';
	eventcat(eventdata,tempdata);

	eventcat(eventdata,DELIMITER);

	add_user_token(eventdata, &(nt_event.t_header));
	eventcat(eventdata,DELIMITER);

	
	add_process_token(eventdata, &(nt_event.t_process));
	eventcat(eventdata,DELIMITER);


	// Use realpath to work out the correct path for the file.
	eventcat(eventdata,"socket");
	eventcat(eventdata,IDELIMITER);
	snprintf(tempdata,MAX_AUDITREC,"%s%s%s%s%d%s%d",nt_event.t_connection.src_ip,IDELIMITER,nt_event.t_connection.dst_ip,IDELIMITER,
					   nt_event.t_connection.src_port,IDELIMITER,nt_event.t_connection.dst_port);
	eventcat(eventdata,tempdata);
	
	eventcat(eventdata,DELIMITER);


	add_return_token(eventdata,&(nt_event.t_return));	
	eventcat(eventdata,DELIMITER);
	add_sequence_token(eventdata);


	// If we are in objective mode, check the path and userid against our objectives.
	if(AuditMode == AUDIT_BY_OBJECTIVE) {
		char objectivedata [MAX_AUDITREC]="";
		char userstore [MAX_USERNAME];	// User name - effective userid
		char commandstore[MAXCOMMAND];
		int returncode;		
		int matchcount=0;
		Node * nodematch;

		// ObjectiveReturn is the returncode mangled into SUCCESS or FAILURE.
		returncode=ObjectiveReturn;

		strncpy(userstore,passname(nt_event.t_header.euser_id,tempdata,MAX_AUDITREC),MAX_USERNAME);
		strncpy(commandstore,nt_event.t_process.name,MAXCOMMAND);
	
		// Check this data against each defined objective
		// the match term is the command that performed the network function.		
		// Options: remote ip address? port?
		while((nodematch=CheckObjective(nt_event.t_header.event_id,userstore,
						   commandstore,options,returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&nt_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			// passname causes a file open remember... might be a better way?
			eventcat(objectivedata,IDELIMITER);
			// Should probably change this so that the text is more representative of the actual event.
			snprintf(tempdata,MAX_AUDITREC,objectivetext,commandstore,userstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}
		if(matchcount==0) {
			// We are auditing by objective, but no objectives match.
			// Dump eventdata and return.
			return(0);
		}
	} else {
		sendevent(eventdata);
	}

	return(0);

}


////////////////////////////////////////////////////////////////////////////////
// Process events that relate to file duplications
////////////////////////////////////////////////////////////////////////////////
int process_cp_class(header_token *header)
{
	cp_class	cp_event;
	char eventdata [MAX_AUDITREC];	// will probably need to make this bigger.
	char tempdata [MAX_AUDITREC];	// for temporary storage if required.
	char objectivetext[MAX_AUDITREC]="";	// For the objective text, if required.
	int ObjectiveReturn = RETURNCODE_SUCCESS;


	cp_event.t_header = *header;

	read(device_fd,&cp_event.t_return,cp_event.t_header.event_size);

	strcpy(objectivetext,"");
	strcpy(eventdata,"event");
	eventcat(eventdata,IDELIMITER);	// usually comma.

	if(cp_event.t_header.event_id == SYS_symlink) {
		eventcat(eventdata,"symlink()");
		if(cp_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The file %s has been symbolically linked to the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to symbolically link the file %s to the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(cp_event.t_header.event_id == SYS_link) {
		eventcat(eventdata,"link()");
		if(cp_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The file %s has been linked to the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to link the file %s to the file %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(cp_event.t_header.event_id == SYS_rename) {
		eventcat(eventdata,"rename()");
		if(cp_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The file %s has been renamed to %s by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to rename the file %s to %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(cp_event.t_header.event_id == SYS_mount) {
		eventcat(eventdata,"mount()");
		if(cp_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The directory %s has been mounted at the path %s by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to mount the directory %s at the path %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(cp_event.t_header.event_id == SYS_umount || cp_event.t_header.event_id == SYS_umount2) {
		// TODO
		// May change umount back to a different class, since it only has a single path entry.
		eventcat(eventdata,"umount()");
		if(cp_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The directory %s has been unmounted%s by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to unmount the directory %s%s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else {
		printf("Unknown event type..\n");
		return(0);
	}

	// Date/time
	eventcat(eventdata,IDELIMITER);
	strncpy(tempdata,ctime(&cp_event.t_header.time.tv_sec),MAX_AUDITREC);
	// Remove the newline character at the end
	tempdata[strlen(tempdata)-1]='\0';
	eventcat(eventdata,tempdata);

	eventcat(eventdata,DELIMITER);

	add_user_token(eventdata, &(cp_event.t_header));
	eventcat(eventdata,DELIMITER);

	// Process
	add_process_token(eventdata, &(cp_event.t_process));
	eventcat(eventdata,DELIMITER);

	// Use realpath to work out the correct path for the file.
	eventcat(eventdata,"path");
	eventcat(eventdata,IDELIMITER);
	eventcat(eventdata,resolved_path(cp_event.t_pwd.path,cp_event.t_sourcepath.path,tempdata));
	eventcat(eventdata,DELIMITER);

	// Path
	eventcat(eventdata,"destpath");
	eventcat(eventdata,IDELIMITER);
	eventcat(eventdata,resolved_path(cp_event.t_pwd.path,cp_event.t_destpath.path,tempdata));
	eventcat(eventdata,DELIMITER);

	add_return_token(eventdata,&(cp_event.t_return));
	eventcat(eventdata,DELIMITER);
	add_sequence_token(eventdata);

	// If we are in objective mode, check the path and userid against our objectives.
	if(AuditMode == AUDIT_BY_OBJECTIVE) {
		char objectivedata [MAX_AUDITREC]="";
		char spathstore [PATH_MAX];
		char dpathstore [PATH_MAX];
		char userstore [MAX_USERNAME];	// User name - effective userid
		int returncode;		
		int matchcount=0;
		Node * nodematch;

		// ObjectiveReturn is the returncode mangled into SUCCESS or FAILURE.
		returncode=ObjectiveReturn;

		strncpy(spathstore,resolved_path(cp_event.t_pwd.path,cp_event.t_sourcepath.path,tempdata),PATH_MAX);
		strncpy(dpathstore,resolved_path(cp_event.t_pwd.path,cp_event.t_destpath.path,tempdata),PATH_MAX);
		strncpy(userstore,passname(cp_event.t_header.euser_id,tempdata,MAX_AUDITREC),MAX_USERNAME);
	
		// Check this data against each defined objective		
		while((nodematch=CheckObjective(cp_event.t_header.event_id,userstore,
						   spathstore,"",returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&cp_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,IDELIMITER);
			snprintf(tempdata,MAX_AUDITREC,objectivetext,spathstore,dpathstore,userstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}
		// Check this data against each defined objective - for the destination path	
		while((nodematch=CheckObjective(cp_event.t_header.event_id,userstore,
						   dpathstore,"",returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&cp_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,IDELIMITER);
			snprintf(tempdata,MAX_AUDITREC,objectivetext,spathstore,dpathstore,userstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}

		if(matchcount==0) {
			// We are auditing by objective, but no objectives match.
			// Dump eventdata and return.
			return(0);
		}
	} else {
		sendevent(eventdata);
	}

	return(0);

}

////////////////////////////////////////////////////////////////////////////////
// Process setuid/setgid
////////////////////////////////////////////////////////////////////////////////
int process_su_class(header_token *header)
{
	su_class	su_event;
	char eventdata [MAX_AUDITREC];	// will probably need to make this bigger.
	char tempdata [MAX_AUDITREC];	// for temporary storage if required.
	char objectivetext[MAX_AUDITREC]="";	// For the objective text, if required.
	int ObjectiveReturn = RETURNCODE_SUCCESS;
	int TargetID = TARGET_EID;

	su_event.t_header = *header;
	// Note that this assumes that the return_token will ALWAYS be the second element in each structure.
	read(device_fd,&su_event.t_return,su_event.t_header.event_size);

	strcpy(objectivetext,"");
	strcpy(eventdata,"event");
	eventcat(eventdata,IDELIMITER);	// usually comma.

	if(su_event.t_header.event_id == SYS_setuid)
	{
		eventcat(eventdata,"setuid()");
		TargetID=TARGET_EID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed effective userid from the user %s to %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the effective user id of the process %s by the user %s to %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setreuid) {
		eventcat(eventdata,"setreuid()");
		TargetID=TARGET_EID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed real and effective user id from the user %s to %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the real and effective user id of the process %s by the user %s to the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setresuid) {
		eventcat(eventdata,"setresuid()");
		TargetID=TARGET_EID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed real, effective and saved user id from the user %s to %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the real, effective and saved user id of the process %s by the user %s to the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setuid32) {
		eventcat(eventdata,"setuid32()");
		TargetID=TARGET_EID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed effective userid from the user %s to %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the effective user id of the process %s by the user %s to %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setreuid32) {
		eventcat(eventdata,"setreuid32()");
		TargetID=TARGET_EID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed real and effective user id from the user %s to %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the real and effective user id of the process %s by the user %s to the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setresuid32) {
		eventcat(eventdata,"setresuid32()");
		TargetID=TARGET_EID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed real, effective and saved user id from the user %s to %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the real, effective and saved user id of the process %s by the user %s to the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else 	if(su_event.t_header.event_id == SYS_setgid)	{
		eventcat(eventdata,"setgid()");
		TargetID=TARGET_GID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed effective group id, called by the user %s, to the group %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the effective group id of the process %s by the user %s to the group %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setregid) {
		eventcat(eventdata,"setregid()");
		TargetID=TARGET_GID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed real and effective group id, called by the user %s, to the group %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the real and effective group id of the process %s by the user %s to the group %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setresgid) {
		eventcat(eventdata,"setresgid()");
		TargetID=TARGET_GID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed real, effective and saved group id, called by the user %s, to the group %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the real, effective and saved group id of the process %s by the user %s to the group %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setgid32) {
		eventcat(eventdata,"setgid32()");
		TargetID=TARGET_GID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed effective group id, called by the user %s, to the group %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the effective group id of the process %s by the user %s to the group %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setregid32) {
		eventcat(eventdata,"setregid32()");
		TargetID=TARGET_GID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed real and effective group id, called by the user %s, to the group %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the real and effective group id of the process %s by the user %s to the group %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(su_event.t_header.event_id == SYS_setresgid32) {
		eventcat(eventdata,"setresgid32()");
		TargetID=TARGET_GID;
		if(su_event.t_return.returncode >= 0) {
			eventcat(objectivetext,"The process %s has changed real, effective and saved group id, called by the user %s, to the group %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to change the real, effective and saved group id of the process %s by the user %s to the group %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else {
		printf("Unknown event type..\n");
		return(0);
	}

	// Date/time
	eventcat(eventdata,IDELIMITER);
	strcpy(tempdata,ctime(&su_event.t_header.time.tv_sec));
	// Remove the newline character at the end
	tempdata[strlen(tempdata)-1]='\0';
	eventcat(eventdata,tempdata);

	eventcat(eventdata,DELIMITER);

	add_user_token(eventdata, &(su_event.t_header));
	eventcat(eventdata,DELIMITER);

	add_process_token(eventdata, &(su_event.t_process));
	eventcat(eventdata,DELIMITER);
	if(su_event.t_header.event_id == SYS_setuid || su_event.t_header.event_id == SYS_setreuid ||
	   su_event.t_header.event_id == SYS_setresuid || su_event.t_header.event_id == SYS_setuid32 ||
	   su_event.t_header.event_id == SYS_setreuid32 || su_event.t_header.event_id == SYS_setresuid32)
	{
		// Target user
		eventcat(eventdata,"target");
		eventcat(eventdata,IDELIMITER);
		if(su_event.t_target.id != -1) {
			eventcat(eventdata,passname(su_event.t_target.id,tempdata,MAX_AUDITREC));
		}
		eventcat(eventdata,"(");
		snprintf(tempdata,MAX_AUDITREC,"%d",su_event.t_target.id);
		eventcat(eventdata,tempdata);
		eventcat(eventdata,")");
	} else 	if(su_event.t_header.event_id == SYS_setgid || su_event.t_header.event_id == SYS_setregid ||
	   su_event.t_header.event_id == SYS_setresgid || su_event.t_header.event_id == SYS_setgid32 ||
	   su_event.t_header.event_id == SYS_setregid32 || su_event.t_header.event_id == SYS_setresgid32)
	{
		// Target user
		eventcat(eventdata,"target");
		eventcat(eventdata,IDELIMITER);
		if(su_event.t_target.id != -1) {
			eventcat(eventdata,groupname(su_event.t_target.id,tempdata,MAX_AUDITREC));
		}
		eventcat(eventdata,"(");
		snprintf(tempdata,MAX_AUDITREC,"%d",su_event.t_target.id);
		eventcat(eventdata,tempdata);
		eventcat(eventdata,")");
	}
	

	if(su_event.t_header.event_id == SYS_setreuid || su_event.t_header.event_id == SYS_setresuid ||
	   su_event.t_header.event_id == SYS_setreuid32 || su_event.t_header.event_id == SYS_setresuid32)
	{
		// Target user real id
		eventcat(eventdata,IDELIMITER);
		if(su_event.t_target.rid != -1) {
			eventcat(eventdata,passname(su_event.t_target.rid,tempdata,MAX_AUDITREC));
		}
		eventcat(eventdata,"(");
		snprintf(tempdata,MAX_AUDITREC,"%d",su_event.t_target.rid);
		eventcat(eventdata,tempdata);
		eventcat(eventdata,")");
	} else if(su_event.t_header.event_id == SYS_setregid || su_event.t_header.event_id == SYS_setresgid ||
	   su_event.t_header.event_id == SYS_setregid32 || su_event.t_header.event_id == SYS_setresgid32)
	{
		// Target user real id
		eventcat(eventdata,IDELIMITER);
		if(su_event.t_target.rid != -1) {
			eventcat(eventdata,groupname(su_event.t_target.rid,tempdata,MAX_AUDITREC));
		}
		eventcat(eventdata,"(");
		snprintf(tempdata,MAX_AUDITREC,"%d",su_event.t_target.rid);
		eventcat(eventdata,tempdata);
		eventcat(eventdata,")");
	}
	
	if(su_event.t_header.event_id == SYS_setresuid || su_event.t_header.event_id == SYS_setresuid32)
	{
		// Target user saved id
		eventcat(eventdata,IDELIMITER);
		if(su_event.t_target.sid != -1) {
			eventcat(eventdata,passname(su_event.t_target.sid,tempdata,MAX_AUDITREC));
		}
		eventcat(eventdata,"(");
		snprintf(tempdata,MAX_AUDITREC,"%d",su_event.t_target.sid);
		eventcat(eventdata,tempdata);
		eventcat(eventdata,")");
	} else if(su_event.t_header.event_id == SYS_setresgid || su_event.t_header.event_id == SYS_setresgid32)
	{
		// Target user saved id
		eventcat(eventdata,IDELIMITER);
		if(su_event.t_target.sid != -1) {
			eventcat(eventdata,groupname(su_event.t_target.sid,tempdata,MAX_AUDITREC));
		}
		eventcat(eventdata,"(");
		snprintf(tempdata,MAX_AUDITREC,"%d",su_event.t_target.sid);
		eventcat(eventdata,tempdata);
		eventcat(eventdata,")");
	}
	eventcat(eventdata,DELIMITER);

	add_return_token(eventdata,&(su_event.t_return));
	eventcat(eventdata,DELIMITER);
	add_sequence_token(eventdata);

	// If we are in objective mode, check the path and userid against our objectives.
	if(AuditMode == AUDIT_BY_OBJECTIVE) {
		char objectivedata [MAX_AUDITREC]="";
		char userstore [MAX_USERNAME];	    // User name - effective userid
		char targetstore [MAX_USERNAME];	// Target user
		char commandstore[MAXCOMMAND];
		int returncode;		
		int matchcount=0;
		Node * nodematch;

		// ObjectiveReturn is the returncode mangled into SUCCESS or FAILURE.
		returncode=ObjectiveReturn;

		strncpy(userstore,passname(su_event.t_header.euser_id,tempdata,MAX_AUDITREC),MAX_USERNAME);
		if(TargetID==TARGET_EID) {
			strncpy(targetstore,passname(su_event.t_target.id,tempdata,MAX_AUDITREC),MAX_USERNAME);
		} else {
			strncpy(targetstore,groupname(su_event.t_target.id,tempdata,MAX_AUDITREC),MAX_USERNAME);
		}
		strncpy(commandstore,su_event.t_process.name,MAXCOMMAND);

		// Check this data against each defined objective
		// the match term is the command that performed the process control function.		
		while((nodematch=CheckObjective(su_event.t_header.event_id,userstore,
						   commandstore,"",returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&su_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,IDELIMITER);

			snprintf(tempdata,MAX_AUDITREC,objectivetext,commandstore,userstore,targetstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}
		if(matchcount==0) {
			// We are auditing by objective, but no objectives match.
			// Dump eventdata and return.
			return(0);
		}
	} else {
		sendevent(eventdata);
	}

	return(0);

}

////////////////////////////////////////////////////////////////////////////////
// Process execution events
////////////////////////////////////////////////////////////////////////////////
int process_ad_class(header_token *header)
{
	ad_class	ad_event;
	char eventdata [MAX_AUDITREC];	// will probably need to make this bigger.
	char tempdata [MAX_AUDITREC];	// for temporary storage if required.
	char objectivetext[MAX_AUDITREC]="";	// For the objective text, if required.
	int ObjectiveReturn = RETURNCODE_SUCCESS;

	ad_event.t_header = *header;

	read(device_fd,&ad_event.t_return,ad_event.t_header.event_size);
	strcpy(objectivetext,"");
	strcpy(eventdata,"event");
	eventcat(eventdata,IDELIMITER);	// usually comma.

	if(ad_event.t_header.event_id == SYS_create_module)
	{
		eventcat(eventdata,"create_module()");
		if(ad_event.t_return.returncode != -1) {
			eventcat(objectivetext,"The module %s was created by process %s, owned by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to create the module %s from the process %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else if(ad_event.t_header.event_id == SYS_delete_module) {
		eventcat(eventdata,"delete_module()");
		if(ad_event.t_return.returncode != -1) {
			eventcat(objectivetext,"The module %s was deleted by process %s, owned by the user %s");
			ObjectiveReturn=RETURNCODE_SUCCESS;
		} else {
			eventcat(objectivetext,"A failed attempt was made to delete the module %s from the process %s by the user %s");
			ObjectiveReturn=RETURNCODE_FAILURE;
		}
	} else {
		printf("Unknown event type..\n");
		return(0);
	}

	// Date/time
	eventcat(eventdata,IDELIMITER);
	strcpy(tempdata,ctime(&ad_event.t_header.time.tv_sec));
	// Remove the newline character at the end
	tempdata[strlen(tempdata)-1]='\0';
	// Need to kill the newline off the end of the ctime return value.
	eventcat(eventdata,tempdata);

	eventcat(eventdata,DELIMITER);

	add_user_token(eventdata, &(ad_event.t_header));
	eventcat(eventdata,DELIMITER);

	add_process_token(eventdata, &(ad_event.t_process));
	eventcat(eventdata,DELIMITER);

	// Use realpath to work out the correct module name.
	eventcat(eventdata,"path");
	eventcat(eventdata,IDELIMITER);
	eventcat(eventdata,ad_event.t_name.path);
	eventcat(eventdata,DELIMITER);

	eventcat(eventdata,DELIMITER);

	add_return_token(eventdata,&(ad_event.t_return));
	eventcat(eventdata,DELIMITER);
	add_sequence_token(eventdata);

	// If we are in objective mode, check the path and userid against our objectives.
	if(AuditMode == AUDIT_BY_OBJECTIVE) {
		char objectivedata [MAX_AUDITREC]="";
		char pathstore [PATH_MAX];
		char userstore [MAX_USERNAME];	// User name - effective userid
		int returncode;		
		int matchcount=0;
		Node * nodematch;
		char commandstore[MAXCOMMAND];
				
		// ObjectiveReturn is the returncode mangled into SUCCESS or FAILURE.
		returncode=ObjectiveReturn;

		strncpy(pathstore,ad_event.t_name.path,PATH_MAX);
		strncpy(userstore,passname(ad_event.t_header.euser_id,tempdata,MAX_AUDITREC),MAX_USERNAME);
		strncpy(commandstore,ad_event.t_process.name,MAXCOMMAND);
		
		// Check this data against each defined objective		
		while((nodematch=CheckObjective(ad_event.t_header.event_id,userstore,
						   pathstore,"",returncode)) != (Node *) NULL) {
			// Keep a tally of matched objectives.
			matchcount++;
			strcpy(objectivedata,"");			
			eventcat(objectivedata,"objective");

			eventcat(objectivedata,IDELIMITER);
			eventcat(objectivedata,criticalitystring(nodematch->criticality,tempdata));
			eventcat(objectivedata,IDELIMITER);
			
			strncpy(tempdata,ctime(&ad_event.t_header.time.tv_sec),MAX_AUDITREC);
			// Remove the newline character at the end
			tempdata[strlen(tempdata)-1]='\0';
			eventcat(objectivedata,tempdata);

			// passname causes a file open remember... might be a better way?
			eventcat(objectivedata,IDELIMITER);
			snprintf(tempdata,MAX_AUDITREC,objectivetext,pathstore,commandstore,userstore);
			eventcat(objectivedata,tempdata);

			eventcat(objectivedata,DELIMITER);
			// Add the normal event data
			eventcat(objectivedata,eventdata);

			sendevent(objectivedata);
		}
		if(matchcount==0) {
			// We are auditing by objective, but no objectives match.
			// Dump eventdata and return.
			return(0);
		}
	} else {
		sendevent(eventdata);
	}

	return(0);

}


int add_user_token(char *eventdata, header_token *header)
{
	char tempdata[MAX_AUDITREC];

	eventcat(eventdata,"user");
	eventcat(eventdata,IDELIMITER);
	eventcat(eventdata,passname(header->user_id,tempdata,MAX_AUDITREC));
	eventcat(eventdata,"(");
	snprintf(tempdata,MAX_AUDITREC,"%d",header->user_id);
	eventcat(eventdata,tempdata);
	eventcat(eventdata,")");
	eventcat(eventdata,IDELIMITER);
	// GID
	eventcat(eventdata,groupname(header->group_id,tempdata,MAX_AUDITREC));
	eventcat(eventdata,"(");
	snprintf(tempdata,MAX_AUDITREC,"%d",header->group_id);
	eventcat(eventdata,tempdata);
	eventcat(eventdata,")");
	eventcat(eventdata,IDELIMITER);
	// EUID
	eventcat(eventdata,passname(header->euser_id,tempdata,MAX_AUDITREC));
	eventcat(eventdata,"(");
	snprintf(tempdata,MAX_AUDITREC,"%d",header->euser_id);
	eventcat(eventdata,tempdata);
	eventcat(eventdata,")");
	eventcat(eventdata,IDELIMITER);
	// EGID
	eventcat(eventdata,groupname(header->egroup_id,tempdata,MAX_AUDITREC));
	eventcat(eventdata,"(");
	snprintf(tempdata,MAX_AUDITREC,"%d",header->egroup_id);
	eventcat(eventdata,tempdata);
	eventcat(eventdata,")");

	return(0);
}

int add_process_token(char *eventdata, process_token *process)
{
	char tempdata[MAX_AUDITREC];

	eventcat(eventdata,"process");
	eventcat(eventdata,IDELIMITER);
	// PID
	snprintf(tempdata,MAX_AUDITREC,"%d",process->pid);
	eventcat(eventdata,tempdata);
	eventcat(eventdata,IDELIMITER);
	// Name
	eventcat(eventdata,process->name);
	return(0);
}

int add_return_token(char *eventdata, return_token *returncode)
{
	char tempdata[MAX_AUDITREC];
	
	// Return code
	eventcat(eventdata,"return");
	eventcat(eventdata,IDELIMITER);

	snprintf(tempdata,MAX_AUDITREC,"%d",returncode->returncode);
	eventcat(eventdata,tempdata);
	return(0);
}

int add_sequence_token(char *eventdata)
{
	static unsigned int sequence=1;
	char tempdata[10];	// Enough for MAXINT (65535)
	
	// Return code
	eventcat(eventdata,"sequence");
	eventcat(eventdata,IDELIMITER);

	snprintf(tempdata,MAX_AUDITREC,"%d",sequence);
	eventcat(eventdata,tempdata);
	sequence++;
	if(sequence >= INT_MAX) {
		sequence=1;
	}
	
	return(0);
}




////////////////////////////////////////////////////////////////////////////////
// strcat that respects the maximum size of an audit record
////////////////////////////////////////////////////////////////////////////////
char * eventcat (char * string, char * append)
{
	if((strlen(string) + strlen(append)) < MAX_AUDITREC)
	{
		return((char *)strcat(string,append));
	} else {
		return(string);
	}
}

// Return a text-type mode
char * textmode (int mode, char *t_mode, int size)
{
	// Need at least 9 characters
	if(size<9) return NULL;

	t_mode[0]='\0';

	if(mode & S_IRUSR) {	strcat(t_mode,"r");	} else {	strcat(t_mode,"-");	}
	if(mode & S_IWUSR) {	strcat(t_mode,"w");	} else {	strcat(t_mode,"-");	}
	if(mode & S_IXUSR) {	strcat(t_mode,"x");	} else {	strcat(t_mode,"-");	}
		
	if(mode & S_IRGRP) {	strcat(t_mode,"r");	} else {	strcat(t_mode,"-");	}
	if(mode & S_IWGRP) {	strcat(t_mode,"w");	} else {	strcat(t_mode,"-");	}
	if(mode & S_IXGRP) {	strcat(t_mode,"x");	} else {	strcat(t_mode,"-");	}
		
	if(mode & S_IROTH) {	strcat(t_mode,"r");	} else {	strcat(t_mode,"-");	}
	if(mode & S_IWOTH) {	strcat(t_mode,"w");	} else {	strcat(t_mode,"-");	}
	if(mode & S_IXOTH) {	strcat(t_mode,"x");	} else {	strcat(t_mode,"-");	}

	return(t_mode);
}

////////////////////////////////////////////////////////////////////////////////
// This routine will send an event to the user-selected output device.
////////////////////////////////////////////////////////////////////////////////
int sendevent(char *string)
{
	char stringout[MAX_AUDITREC]="";
	
	// NEW audit record format, including host and LinuxAudit identifier string:
	snprintf(stringout,MAX_AUDITREC,"%s\tLinuxAudit\t%s",hostid,string);
	
	if(AuditDestination & AUDIT_TO_STDOUT) {
		// Send to STDOUT.
		printf("%s\n",stringout);
	}
	
	if(AuditDestination & AUDIT_TO_FILE) {
		// Write to file
		if(AuditFile != (FILE *) NULL) {
			fprintf(AuditFile,"%s\n",stringout);
			fflush(AuditFile);
		}
	}
	
	if(AuditDestination & AUDIT_TO_NETWORK) {
		if(AuditSocket) {		
			SendToSocket(stringout);
		}
	}
	
	return(0);
}


char * passname(int uid,char *buffer,int buffersize)
{
	struct passwd *passent;

	buffer[0]='\0';
	passent=getpwuid((uid_t)uid);
	if(passent != (struct passwd *)NULL)
	{
		strncpy(buffer,passent->pw_name,buffersize);
	}
	
	return(buffer);
}

char * groupname(int gid, char *buffer, int buffersize)
{
	struct group *groupent;

	buffer[0]='\0';
	groupent=getgrgid((uid_t)gid);
	if(groupent != (struct group *)NULL)
	{
		strncpy(buffer,groupent->gr_name,buffersize);
	}
	return(buffer);
}

// If we want to use a config file that uses text-type event names to enable/disable events
int	eventname2number(char * eventname)
{
	if(!strcmp(eventname,"open"))			{ return SYS_open; }
	if(!strcmp(eventname,"creat"))			{ return SYS_creat; }
	if(!strcmp(eventname,"execve"))			{ return SYS_execve; }
	if(!strcmp(eventname,"exit"))			{ return SYS_exit; }
	if(!strcmp(eventname,"mkdir"))			{ return SYS_mkdir; }
	if(!strcmp(eventname,"unlink"))			{ return SYS_unlink; }
	if(!strcmp(eventname,"mknod"))			{ return SYS_mknod; }
	if(!strcmp(eventname,"rmdir"))			{ return SYS_rmdir; }
	if(!strcmp(eventname,"chown"))			{ return SYS_chown; }
	if(!strcmp(eventname,"lchown"))			{ return SYS_lchown; }
	if(!strcmp(eventname,"chown32"))		{ return SYS_chown32; }
	if(!strcmp(eventname,"lchown32"))		{ return SYS_lchown32; }
	if(!strcmp(eventname,"chmod"))			{ return SYS_chmod; }
	if(!strcmp(eventname,"symlink"))		{ return SYS_symlink; }
	if(!strcmp(eventname,"link"))			{ return SYS_link; }
	if(!strcmp(eventname,"rename"))			{ return SYS_rename; }
	if(!strcmp(eventname,"reboot"))			{ return SYS_reboot; }
	if(!strcmp(eventname,"truncate"))		{ return SYS_truncate; }
	if(!strcmp(eventname,"chroot"))			{ return SYS_chroot; }
	if(!strcmp(eventname,"setuid"))			{ return SYS_setuid; }
	if(!strcmp(eventname,"setreuid"))		{ return SYS_setreuid; }
	if(!strcmp(eventname,"setresuid"))		{ return SYS_setresuid; }
	if(!strcmp(eventname,"setuid32"))		{ return SYS_setuid32; }
	if(!strcmp(eventname,"setreuid32"))		{ return SYS_setreuid32; }
	if(!strcmp(eventname,"setresuid32"))	{ return SYS_setresuid32; }
	if(!strcmp(eventname,"setgid"))			{ return SYS_setgid; }
	if(!strcmp(eventname,"setregid"))		{ return SYS_setregid; }
	if(!strcmp(eventname,"setresgid"))		{ return SYS_setresgid; }
	if(!strcmp(eventname,"setgid32"))		{ return SYS_setgid32; }
	if(!strcmp(eventname,"setregid32"))		{ return SYS_setregid32; }
	if(!strcmp(eventname,"setresgid32"))	{ return SYS_setresgid32; }
	if(!strcmp(eventname,"truncate64"))		{ return SYS_truncate64; }
	if(!strcmp(eventname,"socketcall"))		{ return SYS_socketcall; }
	if(!strcmp(eventname,"create_module"))	{ return SYS_create_module; }
	if(!strcmp(eventname,"delete_module"))	{ return SYS_delete_module; }
	if(!strcmp(eventname,"mount"))			{ return SYS_mount; }
	if(!strcmp(eventname,"umount"))			{ return SYS_umount; }
	if(!strcmp(eventname,"umount2"))		{ return SYS_umount2; }

	// We should not get here.
	return(-1);
}

// If we want to use a config file that uses text-type event names to enable/disable events
char * eventnumber2name(int number, char * eventname, int buffer)
{
	switch(number) {
		case SYS_open:			strncpy(eventname,"open",buffer);			break;
		case SYS_creat:			strncpy(eventname,"creat",buffer);			break;
		case SYS_execve:		strncpy(eventname,"execve",buffer);			break;
		case SYS_exit:			strncpy(eventname,"exit",buffer);			break;
		case SYS_mkdir:			strncpy(eventname,"mkdir",buffer);			break;
		case SYS_unlink:		strncpy(eventname,"unlink",buffer);			break;
		case SYS_mknod:			strncpy(eventname,"mknod",buffer);			break;
		case SYS_rmdir:			strncpy(eventname,"rmdir",buffer);			break;
		case SYS_chown:			strncpy(eventname,"chown",buffer);			break;
		case SYS_lchown:		strncpy(eventname,"lchown",buffer);			break;
		case SYS_chown32:		strncpy(eventname,"chown32",buffer);		break;
		case SYS_lchown32:		strncpy(eventname,"lchown32",buffer);		break;
		case SYS_chmod:			strncpy(eventname,"chmod",buffer);			break;
		case SYS_symlink:		strncpy(eventname,"symlink",buffer);		break;
		case SYS_link:			strncpy(eventname,"link",buffer);			break;
		case SYS_rename:		strncpy(eventname,"rename",buffer);			break;
		case SYS_reboot:		strncpy(eventname,"reboot",buffer);			break;
		case SYS_truncate:		strncpy(eventname,"truncate",buffer);		break;
		case SYS_chroot:		strncpy(eventname,"chroot",buffer);			break;
		case SYS_setuid:		strncpy(eventname,"setuid",buffer);			break;
		case SYS_setreuid:		strncpy(eventname,"setreuid",buffer);		break;
		case SYS_setresuid:		strncpy(eventname,"setresuid",buffer);		break;
		case SYS_setuid32:		strncpy(eventname,"setuid32",buffer);		break;
		case SYS_setreuid32:	strncpy(eventname,"setreuid32",buffer);		break;
		case SYS_setresuid32:	strncpy(eventname,"setresuid32",buffer);	break;
		case SYS_setgid:		strncpy(eventname,"setgid",buffer);			break;
		case SYS_setregid:		strncpy(eventname,"setregid",buffer);		break;
		case SYS_setresgid:		strncpy(eventname,"setresgid",buffer);		break;
		case SYS_setgid32:		strncpy(eventname,"setgid32",buffer);		break;
		case SYS_setregid32:	strncpy(eventname,"setregid32",buffer);		break;
		case SYS_setresgid32:	strncpy(eventname,"setresgid32",buffer);	break;
		case SYS_ftruncate:		strncpy(eventname,"ftruncate",buffer);		break;
		case SYS_fchmod:		strncpy(eventname,"fchmod",buffer);			break;
		case SYS_ftruncate64:	strncpy(eventname,"ftruncate64",buffer);	break;
		case SYS_truncate64:	strncpy(eventname,"truncate64",buffer);		break;
		case SYS_fchown:		strncpy(eventname,"fchown",buffer);			break;
		case SYS_socketcall:	strncpy(eventname,"socketcall",buffer);		break;
		case SYS_create_module:	strncpy(eventname,"create_module",buffer);	break;
		case SYS_delete_module:	strncpy(eventname,"delete_module",buffer);	break;
		case SYS_mount:			strncpy(eventname,"mount",buffer);			break;
		case SYS_umount:		strncpy(eventname,"umount",buffer);			break;
		case SYS_umount2:		strncpy(eventname,"umount2",buffer);		break;
		default:				strncpy(eventname,"",buffer);				break;
	}		
	return(eventname);
}

char * resolved_path(char *pwd,char *path,char *resolved)
{
	char temppath[PATH_MAX + PATH_MAX + 1];
	if(path[0] != '/')
	{
		strcpy(temppath,pwd);
		strcat(temppath,"/");
		strcat(temppath,path);
		if((resolved = realpath(temppath,resolved)) == (char *) NULL)
		{
			// Could not resolve for some reason.
			// Return the path. It will be the most useful filename.
			return(path);
		} else {
			return(resolved);
		}
	} else {
		// Path is already absolute. return it.
		if(realpath(path,resolved) == (char *) NULL)
		{
			// Could not resolve for some reason.
			// Return the path. It will be the most useful filename.
			return(path);
		} else {
			return(resolved);
		}
	}
}


// Here: add in options
Node * CheckObjective(int eventid,const char *username,const char *searchterm,char *options,int returncode)
{
	static int firstcall=0;
	Node *currentnode;

	if(firstcall==0) {
		ResetCurrentNode();
		firstcall=1;
	}
	
	while(IsValidItem()) {
		currentnode=GetCurrentItem();
		if((eventid == currentnode->event_number || currentnode->event_number == AUDIT_ALL) && (returncode == currentnode->returncode || currentnode->returncode == RETURNCODE_ANY)) {
			// Are we including users, or excluding.
			if(currentnode->excludeflag) {
				if(!(regmatch(username,currentnode->username)) && regmatch(searchterm,currentnode->path)) {
					if(regmatchi(options,currentnode->options)) {
						NextItemInList();
						return(currentnode);
					}
				}
			} else {
				if(regmatch(username,currentnode->username) && regmatch(searchterm,currentnode->path)) {
					if(regmatchi(options,currentnode->options)) {
						NextItemInList();
						return(currentnode);
					}
				}
			}
		}
		NextItemInList();
	}

	ResetCurrentNode();
	return((Node *) NULL);
}


// Open, read and act on the configuration file.
// Return AUDIT_BY_OBJECTIVE or AUDIT_BY_EVENT or -1 for error.
int read_config_file()
{
	FILE * configfile = (FILE *)NULL;
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];	// Should be enough for most config lines.
							// Would live to use gchar here instead, but need to keep this simple.
	// Config file header.
	int headertype=0;
	int audittype=AUDIT_BY_EVENT;			// Default.

	// Clear the audit destination
	AuditDestination=0;
	
	configfile=fopen(CONFIG_FILENAME,"r");

	if(configfile==(FILE *)NULL) {
		return(AUDIT_BY_EVENT);
	}

	while(fgets(inputbuffer,MAX_AUDIT_CONFIG_LINE,configfile)) {
		// Kill whitespace from start and end of line.
		trim(inputbuffer);
		
		if(!iscomment(inputbuffer))	{
			// Is this line a header?
			if(isheader(inputbuffer)) {
				headertype=getheader(inputbuffer);
			} else {
				if(headertype == CONFIG_AUDITTYPE) {
					// Audit by objective, or not?
					if(isaudittype(inputbuffer)) {
						audittype=getaudittype(inputbuffer);
					}
				} else if(headertype == CONFIG_OBJECTIVES) {
					// Save off enough space to store the data we need.
					char event[MAX_AUDIT_CONFIG_LINE], user[MAX_AUDIT_CONFIG_LINE], path[MAX_AUDIT_CONFIG_LINE];
					char event2[MAX_AUDIT_CONFIG_LINE], options[MAX_OPTIONS]="";
					char *eventpointer=NULL;
					char *eventpointer2=NULL;
					int criticality;
					int eventnumber;
					int returncode;
					int excludeflag=0;

					if((criticality = splitobjective(inputbuffer,event,user,path,&returncode,&excludeflag)) > -1) {
						// add the objective to the linked list.
						trimallwhitespace(event);

						// HERE: Split event into comma separated list of events				
						eventpointer=event;
						eventpointer2=event;
						
						// While there are no more commas
						while(eventpointer2) {
							eventpointer2=strstr(eventpointer,",");
							if(eventpointer2 == (char *)NULL) {
								// No commas left. Just copy to the end of the line.
								strncpy(event2,eventpointer,MAX_AUDIT_CONFIG_LINE);
							} else {
								strncpy(event2,eventpointer,(eventpointer2-eventpointer));
								// Make sure we have a null on the end of the line.
								event2[eventpointer2-eventpointer]='\0';
							}
							if(eventpointer2) {
								// Skip the comma
								eventpointer=eventpointer2+1;
							}
							// Clear out any options strings
							strcpy(options,"");
							if(regmatch(event2,"\\(.*\\)$")) {
								char *position;

								// Bonus, we have something. Grab it, add to options.
								// First, trim off the last bracket.
								event2[strlen(event2)-1]='\0';
								position=strstr(event2,"(");
								
								strncpy(options,position+1,MAX_OPTIONS);
								// truncate the event string at the bracket.
								*position='\0';
							}

							if((eventnumber = eventname2number(event2)) > -1) {
								AddToList(eventnumber,user,path,criticality,options,excludeflag,returncode);

								// Also, ensure this event is turned on if we are auditing by objective.
								// Note that this may not have been in the config file yet...
								// Best to assume objective auditing is NOT on though.
								if(audittype == AUDIT_BY_OBJECTIVE) {
									turn_event_on(eventnumber);
								}
							} else {
								if(!strcmp(event,"*")) {
									AddToList(AUDIT_ALL,user,path,criticality,"",excludeflag,returncode);
								}
							}
						}
					} else {
						fprintf(stderr,"WARNING: Cannot process objective - please ensure configuration line\n\tcontains valid criticality, event, criticality, return, user and match elements\n\t%s\n",inputbuffer);
					}
				} else if(headertype == CONFIG_EVENTS) {
					int eventnumber=0;
					
					// Do not turn events on if we are auditing by objectives at the moment.
					// Note that this will stop event="*" from working on any events that have not been explicitly defined
					// within an objective. However, it is appropriate from a usability perspective.
					if(audittype != AUDIT_BY_OBJECTIVE) {
						eventnumber=geteventswitch(inputbuffer);
						if(eventnumber > -1) {
							// Turn it on.
							turn_event_on(eventnumber);
						}
					}
				} else if(headertype == CONFIG_OUTPUT) {
					if(isfilename(inputbuffer)) {
						if(! open_audit_output(inputbuffer)) {
							fprintf(stderr,"WARNING: Could not open the file specified in the audit configuration: %s - Auditing to STDOUT\n",inputbuffer);
						}
					} else if(isnetwork(inputbuffer)) {
						if(! open_audit_network(inputbuffer)) {
							fprintf(stderr,"WARNING: Could not open the network specified in the audit configuration: %s - Auditing to STDOUT\n",inputbuffer);
						}
					}
				} else if(headertype == CONFIG_HOSTID) {
					gethostident(inputbuffer,hostid,MAX_HOSTID);
				} else {
					fprintf(stderr,"WARNING: Configuration file line does not fit in any recognised header.\n\t%s\n",inputbuffer);
				}
			}
		}
	}

	fclose((FILE *)configfile);
	return(audittype);
}

int iscomment(char * line)
{
	// Verify that there is something to check.
	if(line == (char *) NULL) return(1);
	
	// And that there is some data within.
	if(strlen(line) == 0 || strlen(line) > MAX_AUDIT_CONFIG_LINE) return(1);
	
	// If the first non-whitespace character is a hash, this is a comment line.
	while(*line) {
		// Space or tab or newline / formfeed
		if(*line == ' ' || *line == 9 || *line == 10 || *line == 12) {
			line++;
		} else if(*line == '#') {
				return(1);
		} else {
			// Ahh. A non-whitespace, non hash character.
			return(0);
		}
	}
	
	// If we are here, then the whole line must have been whitespace
	return(1);
}

int isheader(char *string)
{
	if(string[0] == '[' && string[strlen(string)-1] == ']') {
		return(1);
	}
	return(0);
}

int isaudittype(char *string)
{
	if(regmatch(string,"^type=")) {
		return(1);
	}
	return(0);
}

int isfilename(char *string)
{
	if(regmatch(string,"^file=")) {
		return(1);
	} else {
		return(0);
	}	
}

// Network output
int isnetwork(char *string)
{
	if(regmatch(string,"^network=")) {
		return(1);
	} else {
		return(0);
	}	
}

int open_audit_output(char *string)
{
	string+=strlen("file=");
	if(strlen(string)) {
		if(regmatchi(string,"stdout")) {
			AuditDestination |= AUDIT_TO_STDOUT;
			return(1);
		}
		AuditFile=fopen(string,"a");
		if(AuditFile != (FILE *) NULL) {
			AuditDestination |= AUDIT_TO_FILE;
			return(1);
		} else {
			AuditDestination |= AUDIT_TO_STDOUT;
			return(0);
		}
	}

	return(0);
}


// Open the network - destination host and port
int open_audit_network(char *string)
{
	char desthost[MAX_HOSTID];
	int port=6161;	// Default port
	char *pos;
	struct hostent *hp;

	// Initialise desthost.
	strncpy(desthost,"localhost",MAX_HOSTID);
	
	string+=strlen("network=");
	if(strlen(string)) {
		if(regmatchi(string,"stdout")) {
			AuditDestination |= AUDIT_TO_STDOUT;
			return(1);
		}

		AuditSocket = socket(AF_INET,SOCK_DGRAM,0);
		if(AuditSocket < 0)
		{
			fprintf(stderr,"Cannot open network socket - auditing to STDOUT\n");
			AuditDestination |=AUDIT_TO_STDOUT;
			return(0);
		}

		pos=strstr(string,":");
		if(pos) {
			if((pos-string) < MAX_HOSTID) {
				strncpy(desthost,string,(pos-string));
				desthost[pos-string]='\0';
			} else {
				strncpy(desthost,string,MAX_HOSTID);
				desthost[MAX_HOSTID-1]='\0';
			}
			pos++;
			// As long as the user has a number there.
			if(strlen(pos)) {
				port=atoi(pos);
			}
		} else {
			// No colon. Assume the default port to be 6161
			strncpy(desthost,string,MAX_HOSTID);
		}
		
		// Just in case the user has entered a blank field
		if(strlen(desthost) == 0) {
			strncpy(desthost,"localhost",MAX_HOSTID);
		}
		
		hp = gethostbyname(desthost);
		if(hp == 0) {
			fprintf(stderr,"Cannot resolve host %s - auditing to STDOUT\n",hostid);
			close(AuditSocket);
			AuditDestination |= AUDIT_TO_STDOUT;
			return(0);
		}
		bcopy(hp->h_addr, &AuditSocketName.sin_addr, hp->h_length);
		AuditSocketName.sin_family = AF_INET;
		AuditSocketName.sin_port = htons(port);
		
		AuditDestination |= AUDIT_TO_NETWORK;
		return(1);
	}

	AuditDestination |= AUDIT_TO_STDOUT;
	return(0);
}


int getheader(char *string)
{
	char *stringp=string;
	
	// Remove the first and last bracket.
	stringp++;
	stringp[strlen(stringp)-1]='\0';
	
	if(regmatchi(stringp,"^audittype$")){
		return(CONFIG_AUDITTYPE);
	} else if(regmatchi(stringp,"^objectives$")) {
		return(CONFIG_OBJECTIVES);
	} else if(regmatchi(stringp,"^events$")) {
		return(CONFIG_EVENTS);
	} else if(regmatchi(stringp,"^file$") || regmatchi(stringp,"^output$")) {
		return(CONFIG_OUTPUT);
	} else if(regmatchi(stringp,"^hostid$")) {
		return(CONFIG_HOSTID);
	}

	fprintf(stderr,"Unknown header in configuration file: %s\n",stringp);
	return(0);
}

int getaudittype(char *string)
{
	char *stringp=string;
	// Get rid of the type= part.
	stringp+=strlen("type=");
	if(regmatchi(stringp,"^objective$")) {
		return(AUDIT_BY_OBJECTIVE);
	} else {
		return(AUDIT_BY_EVENT);
	}
}

int geteventswitch(char *string)
{
	char *stringp=string;
	char *endname=string;

	stringp=strstr(string,"=");
	if(stringp != (char *) NULL) {
		endname=stringp;
		stringp++;
		if(!strcmp(stringp,"1")) {
			// Set this event.
			// Grab the event name. Insert a null into the string in place of the = character.
			*endname='\0';
			return(eventname2number(string));
		}
	} else {
		return(-1);
	}
	return(-1);
}

// Return the host identifier
char * gethostident(char *string, char *host,int length)
{
	char *stringp=string;

	stringp=strstr(string,"=");
	if(stringp != (char *) NULL) {
		stringp++;
		if(strlen(stringp)) {
			strncpy(host,stringp,length-1);
		} else {
			return((char *)NULL);
		}
	} else {
		return((char *)NULL);
	}
	return((char *)host);
}

// Return a string that contains the criticality
// Note that string is assumed to be MAX_AUDITREC in size.
char * criticalitystring(int criticality, char * string)
{
	switch(criticality)
	{
		case CRITICALITY_CRITICAL:	return(strncpy(string,"critical",MAX_AUDITREC));
		case CRITICALITY_PRIORITY:	return(strncpy(string,"priority",MAX_AUDITREC));
		case CRITICALITY_WARNING:	return(strncpy(string,"warning",MAX_AUDITREC));
		case CRITICALITY_INFO:		return(strncpy(string,"information",MAX_AUDITREC));
		default:					return(strncpy(string,"clear",MAX_AUDITREC));
	}
}


// Remove start / end whitespace, including the newline
void trim(char *string)
{
	char * pointer;
	char * pointer2;
	
	// Verify that there is something to check.
	if(string == (char *) NULL) return;
	
	// And that there is some data within.
	if(strlen(string) == 0 ) return;

	// Start from the end, work backwards.
	pointer=&string[strlen(string)-1];

	while((*pointer == ' ' || *pointer == 9 || *pointer == 10 || *pointer == 12) && pointer >= string) {
		*pointer='\0';
		pointer--;
	}
	
	// Are we back at the start of the string? If so, this line must be null.
	if(pointer==string) return;
	// Pointer is now at the last non-whitespace character of the string.
	
	pointer2=string;
	while((*pointer2 == ' ' || *pointer2 == 9 || *pointer2 == 10 || *pointer2 == 12) && pointer2 < pointer) {
		pointer2++;
	}
	// pointer2 will now point to the start of the first non-null character
	// Copy the truncated string back to the original.
	strcpy(string,pointer2);
}

void trimallwhitespace(char *string)
{
	char * readpointer;
	char * writepointer;
	
	// Verify that there is something to check.
	if(string == (char *) NULL) return;
	
	// And that there is some data within.
	if(strlen(string) == 0 ) return;

	// Start from the end, work backwards.
	readpointer=string;
	writepointer=string;
	while(readpointer < (string + strlen(string))) {
		if(*readpointer == ' ' || *readpointer == 9 || *readpointer == 10 || *readpointer == 12) {
			readpointer++;
		} else {
			if(writepointer != readpointer) {
				*writepointer=*readpointer;
			}
			readpointer++;
			writepointer++;
		}
			
	}
}

// Strip tabs and newlines out of a string.
char * strip (char * string)
{
	char *stringp=string;
		
	while(*stringp != '\0') {
		if(*stringp == '\t' || *stringp == '\n') {
			*stringp = ' ';
		}
		stringp++;
	}
	return(string);
}

// Take an identified objective line
// NOTE: string, event, user and match must all be the same buffer size!
int splitobjective(char *string, char *event, char *user, char *match, int *returncode, int *excludeflag)
{
	char *startevent, *startuser, *startmatch, *startcrit, *startreturn;
	char *stringpointer, *endstring;
	char *eventp, *userp, *matchp, *critp;
	char criticality[MAX_AUDIT_CONFIG_LINE];
	int crit=0;

	*returncode=RETURNCODE_ANY;

	// Do some basic sanity checks.
	if(string == (char *) NULL || event == (char *) NULL || user == (char *) NULL || match == (char *) NULL) {
		return(-1);
	}

	startcrit=strstr(string,"criticality=");
	startevent=strstr(string,"event=");
	startreturn=strstr(string,"return=");
	startuser=strstr(string,"user=");
	if(startuser==(char *)NULL) {
		startuser=strstr(string,"user!=");
		if(startuser!=(char *)NULL) {
			// EXCLUDE USERS rather than include.
			*excludeflag=1;
		}
	}
	startmatch=strstr(string,"match=");
	
	// string pointers for iteration.
	stringpointer=string;
	eventp=event; userp=user; matchp=match; critp=criticality;
	
	// Pointer to the last character in the string.
	endstring=&string[strlen(string)];
	
	if(startevent == (char *)NULL || startuser == (char *)NULL ||
	   startmatch == (char *)NULL || startcrit == (char *)NULL || startreturn == (char *)NULL) {
		// Problem, this line is malformed. We really cannot proceed with this line.
		fprintf(stderr,"The following line does not contain the criticality, event, return, user and match elements: %s",string);
		return(-1);
	}
	
	// Start with the event
	stringpointer=startcrit+strlen("criticality=");
	while(*stringpointer && (stringpointer!=startcrit && stringpointer != startreturn && stringpointer!=startuser && stringpointer!=startmatch && stringpointer != endstring)) {
		*critp=*stringpointer;
		critp++;
		stringpointer++;
	}
	*critp='\0';
	trim(criticality);
	crit=atoi(criticality);
	if(crit < CRITICALITY_CLEAR) crit=CRITICALITY_CLEAR;
	if(crit > CRITICALITY_CRITICAL) crit=CRITICALITY_CRITICAL;
	
	// Start with the event
	stringpointer=startevent+strlen("event=");
	while(*stringpointer && (stringpointer!=startcrit && stringpointer != startreturn && stringpointer!=startuser && stringpointer!=startmatch && stringpointer != endstring)) {
		*eventp=*stringpointer;
		eventp++;
		stringpointer++;
	}
	*eventp='\0';
	// Remove extra whitespace at start and end.
	trim(event);
	
	stringpointer=startreturn+strlen("return=");
	if(regmatchi(stringpointer,"^success")) {
		*returncode=RETURNCODE_SUCCESS;
	} else if (regmatchi(stringpointer,"^failure")) {
		*returncode=RETURNCODE_FAILURE;
	} else {
		// Any thing else is either success or failure.
		*returncode=RETURNCODE_ANY;
	}

	if(*excludeflag) {
		stringpointer=startuser+strlen("user!=");
	} else {
		stringpointer=startuser+strlen("user=");
	}

	while(*stringpointer && (stringpointer!=startcrit && stringpointer != startreturn && stringpointer!=startmatch && stringpointer!=startevent && stringpointer != endstring)) {
		*userp=*stringpointer;
		stringpointer++;
		userp++;
	}
	*userp='\0';
	// Remove extra whitespace at start and end.
	trim(user);
	
	stringpointer=startmatch+strlen("match=");
	while(*stringpointer && (stringpointer!=startcrit && stringpointer != startreturn && stringpointer!=startevent && stringpointer!=startuser && stringpointer != endstring)) {
		*matchp=*stringpointer;
		stringpointer++;
		matchp++;
	}
	*matchp='\0';
	// Remove extra whitespace at start and end.
	trim(match);

	// Return the criticality value.
	return(crit);
}


// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
int regmatch(const char *string, const char *pattern)
{
	int status;
	regex_t re;
	
	if(regcomp(&re,pattern,REG_EXTENDED|REG_NOSUB) != 0) {
		return(0);
	}
	status=regexec(&re,string,(size_t) 0, NULL, 0);
	regfree(&re);
	if(status != 0) {
		return(0);
	}
	return(1);
}	

// Match string against an extended regular expression. Case insensitive.
// Return 1 for match, 0 for no-match or error.
int regmatchi(const char *string, const char *pattern)
{
	int status;
	regex_t re;
	
	if(regcomp(&re,pattern,REG_EXTENDED|REG_NOSUB|REG_ICASE) != 0) {
		return(0);
	}
	status=regexec(&re,string,(size_t) 0, NULL, 0);
	regfree(&re);
	if(status != 0) {
		return(0);
	}
	return(1);
}	


// Linked List Functions

void CreateLinkedList(void)
{
    head = currentnode = NULL;
}

int IsListEmpty(void)
{
    if (NULL == head)
        return 1;
    else
        return 0;
}

Node * AddToList(int eventnumber, char *username, char *path, int criticality, char *options, int excludeflag, int returncode)
{
    Node *newNode=NULL;
    // FIXME: If auditd is started, then the audit.conf file moved out of the way to audit.conf.bak,
    //        a USR1 signal sent to auditd, then the audit.conf.bak file is returned to audit.conf
    //        and the USR1 signal sent once more, the malloc will result in a seg fault for some reason!
    //        Not sure why at the moment.
	//
	// NOTE: Think I may have fixed this with copy_from_user in auditmodule.
	//
    newNode = (Node *) malloc(sizeof(Node));

    if (newNode == NULL)
    {
        fprintf(stderr, "AddToList(): error in dynamic memory allocation\nCould not add a new objective into our linked list. You may be low on memory.\n");
        return((Node *)NULL);
    }

	newNode->event_number=eventnumber;
	newNode->criticality=criticality;
	newNode->returncode=returncode;
	newNode->excludeflag=excludeflag;

	strcpy(newNode->username,username);
	strcpy(newNode->path,path);
	strncpy(newNode->options,options,MAX_OPTIONS);

    if (head == NULL)
    {
        head = newNode;
        newNode->next = NULL;
    }
    else
    {
        newNode->next = head;
        head = newNode;
    }

    return newNode;
}

void RemoveFromListHead(void)
{
    Node *tempPtr;

    if (NULL == head)
        return;

    tempPtr = head;
    head = head->next;

    free(tempPtr);
}

void RemoveFromList(Node* node)
{
    Node *tempPtr, *previousPtr;

    if (NULL == node)
        return;

    if (head == node)
    {
        RemoveFromListHead();
        return;
    }

    tempPtr = head;

    while (NULL != tempPtr)
    {
        previousPtr = tempPtr;
        tempPtr = tempPtr ->next;

        if (tempPtr == node)
        {
            previousPtr->next = tempPtr->next;
            free(tempPtr);
            break;
        }
    }
}

void ResetCurrentNode(void)
{
    currentnode = head;
}

int IsValidItem(void)
{
    return (NULL == currentnode) ? 0 : 1;
}

Node * GetCurrentItem()
{
    return (currentnode);
}

void NextItemInList(void)
{
    if (NULL == currentnode)
        return;

    currentnode = currentnode->next;
}

void DestroyList(void)
{
    if (NULL == head)
    {
        return;
    }

    while (NULL != head)
    {
        Node *tempPtr = head;
        head = head->next;

        free(tempPtr);
    }
}


// Pull out the fully qualified domain name, if possible.
char * getfqdn(char * FQDN)
{
	char hname[MAX_HOSTID];
	struct hostent *hp;

	if (gethostname(hname, MAX_HOSTID)) {
		strncpy(FQDN,"localhost.unknown",MAX_HOSTID);
		return(FQDN);
	}
	
	strncpy(FQDN,hname,MAX_HOSTID);
 
	hp = gethostbyname(hname);
	if (hp) {
		while (hp->h_aliases && *hp->h_aliases) {
			if (strlen(*(hp->h_aliases)) > strlen(hname) && !strncmp(hname, *(hp->h_aliases), strlen(hname))) {
				strncpy(FQDN, *(hp->h_aliases),MAX_HOSTID);
			}
			hp->h_aliases++;
		}
		if (strlen(hp->h_name) > strlen(hname) && !strncmp(hname, hp->h_name, strlen(hname))) {
				strncpy(FQDN, hp->h_name,MAX_HOSTID);
		}
	}

     return(FQDN);
}

int SendToSocket(char * message)
{
	if(AuditSocket) {
		if(sendto(AuditSocket, message,
			strlen(message), 0, 
			(const struct sockaddr *)&AuditSocketName,
			sizeof(struct sockaddr))
					< 0) {
			puts(message);
			return(1);
		}
	} else {
		puts(message);
		return(1);
	}
	return(0);
}
