//////////////////////////////////////////////////////////////////////////////////////////
//
// System iNtrusion Analysis and Reporting Environment (SNARE) - Audit Module
//
// This is the core of the SNARE audit subsystem. The audit module intercepts
// system calls, and saves off details associated with the call to a circular memory
// buffer, which is then read by the audit daemon.
//
///////////////////////////////////////////////////////////////////////////////////////////
// Copyright 1999-2002 InterSect Alliance Pty Ltd - http://www.intersectalliance.com/
// Copyright 2002 Redhat Pty Ltd - http://www.redhat.com/
// - Many thanks to the Redhat Kernel team for providing MUCH valued assistance!
//
///////////////////////////////////////////////////////////////////////////////////////////
//
// 0.9.1 - implementation of redhat kernel team suggestions, including:
// - Changing execve to look more like arch/i386/kernel/process.c
// - updated info open/info read/info close as per redhat recommendations.
// - removed audit_device_open
// - retooled audit_info
// - Semaphore locking, rather than spin locks
//
//
//////////////////////////////////////////////////////////////////////////////////////////

#define HIDDEN_SYS_CALL_TABLE 1

// Includes
#include <linux/module.h>
#include <linux/reboot.h>
#include <linux/proc_fs.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/timer.h>

#include <asm/uaccess.h>

#include <sys/syscall.h>

#include <stdio.h>

#ifdef HIDDEN_SYS_CALL_TABLE
#include <asm/processor.h>
#endif


#include "audit.h"
#include "auditmodule.h"

// Would really prefer to use snprintf for ALL kernel versions
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,10)
#define snprintf(buf,count,fmt, args...) sprintf( buf, fmt, ## args )
#endif

// Global static variables
static void *read_position = (void *)NULL; // How far have we read through the node at the head of the list?

// /proc/auditinfo read position.
static char *	info_read_position;		// read position in the information proc entry.

static DECLARE_MUTEX(audit_lock);						// for info_read at the moment.

static struct task_struct *	auditdaemon_task_struct;	// Daemon interaction
unsigned long int flags; 								// CPU flags when using spin_lock_irqsave

#if !defined(HIDDEN_SYS_CALL_TABLE)
extern void *sys_call_table[];
#else
void **sys_call_table;
#endif

static int AUDIT_IS_RUNNING=0;	// Running flag

// How many packets have we lost this session due to low buffer size?
static int lost_events=0;
static int total_events=0;

// This is a little like 'module use count' - the variable is used to make sure that
// we don't free memory while it's still being used by an auditing routine.
// NOTE: Removed this - have migrated the memory free routines to the point at which the module unloads instead.
//static unsigned long Audit_Log_Write_Count=0;

// Create an array to store the active events
static short active_audit[MAXAUDIT];

struct proc_dir_entry *ProcEntry;
struct proc_dir_entry *ProcInfoEntry;

// Kernel 2.4 and above only
static struct file_operations file_ops =
{
	read:	auditmodule_read,		// read
	ioctl:	auditmodule_ioctl,		// ioctl
	open:	auditmodule_open,		// open
	release:	auditmodule_close,	// release
};

static struct file_operations info_ops =
{
	read:	info_read,				// read
	open:	info_open,
	release:	info_close,
};

MODULE_AUTHOR("InterSect Alliance - www.intersectalliance.com");
MODULE_DESCRIPTION("Linux Audit Module");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
MODULE_LICENSE("GPL");
#endif

#ifdef HIDDEN_SYS_CALL_TABLE
extern asmlinkage long sys_exit(int error_code);
#endif


Node * list_head=(Node *)NULL;
Node * list_tail=(Node *)NULL;


////////////////////////////////////////////////////////////////////////////////
// Initialise the module
// Set up our event overlay functions.
////////////////////////////////////////////////////////////////////////////////
// static int __init init_module(void)
// static int init_module(void)
int init_module(void)
{
	long newbuffersize=0;
	int count=0;
	
	AUDIT_IS_RUNNING = FALSE;
	auditdaemon_task_struct=NULL;

#ifdef HIDDEN_SYS_CALL_TABLE
	// The following code is from StJude v0.22
	// Timothy Lalwess (lawless@wwjh.net)
	//
	// In the infinite wisdom of someone, the sys_call_table
	// varable is no longer exported. I suspect this was
	// an attempt to foil the developers of kernel rootkits.
	// Unfortunitely, this is an example of when obscurit
	// does not infact do anything to improve security. The
	// fact of the matter is that the sys_call_table is easily
	// findable. All it does it make more work. Bah.
	//
	{
	        unsigned long ptr;
	        extern int loops_per_jiffy;

	        // sys_call_table = NULL;
	        for (ptr = (unsigned long) &loops_per_jiffy;
                ptr < (unsigned long) &boot_cpu_data; ptr += sizeof(void *))
	        {
	                unsigned long *p;
	                p = (unsigned long *) ptr;
	                if (p[1] == (unsigned long) sys_exit)
	                {
																		                        sys_call_table = (void **) p;
																		                        break;
																		                }
	        }

	        if (!sys_call_table)
	        {
	                printk("Cant find the sys_call_table. Aborting load. Feel safe to rmmod.\n");
	                return 0;
		}
	}
#endif

	// Initialise our active event storage array.
	for(count=0;count <= MAXAUDIT; count++) {
		active_audit[count]=0;
	}

	// From here on in, the read routine could be signaled.
	// Therefore, wherever we write to the audit buffer, or we rely on
	// any of the variables that the read routine changes,
	// we should lock the kernel first.

printk("DEBUG: Creating proc entry\n");
	// Would do this using proc_register in kernel 2.2 - see http://linuxkernel.to//port-2.4/eng/lkp-all.html
	if ((ProcEntry = create_proc_entry(AUDITDEV_NAME, S_IRUSR | S_IWUSR, NULL)) == NULL)
		return(1);
printk("DEBUG: Created proc entry\n");
	ProcEntry->proc_fops = &file_ops;

	if ((ProcInfoEntry = create_proc_entry(AUDITINFO_NAME, S_IRUGO | S_IWUSR, NULL)) == NULL)
		return(1);
		
	ProcInfoEntry->proc_fops = &info_ops;


	// Now: Verify what audit events the user wants, and insert ourselves into the
	// relevant system calls.

	printk("SNARE Audit Module: Version %d.%d%d initialized\n", AUDITMODULE_MAJOR_VERSION,
			AUDITMODULE_MINOR_VERSION, AUDITMODULE_PATCH_VERSION);

	return 0;
}

////////////////////////////////////////////////////////////////////////////////
// This routine handles ioctl messages from the audit daemon.
////////////////////////////////////////////////////////////////////////////////

int auditmodule_ioctl(struct inode* node, struct file*  the_file, unsigned int  command, unsigned long arg)
{
	static int counter=0;

	// If the audit module is running, there is not much point starting again!
	if((AUDIT_IS_RUNNING == TRUE) && (command == AUDIT_START))
		return -EBUSY;

	if(command == AUDIT_START) {
		down(&audit_lock);
		AUDIT_IS_RUNNING = TRUE;
		up(&audit_lock);
	} else if(command == AUDIT_STOP) {
		down(&audit_lock);
		AUDIT_IS_RUNNING = FALSE;
		up(&audit_lock);
	} else if(command == AUDIT_LOSTEVENTS) {
		return(lost_events);
	} else if(command == AUDIT_TOTALEVENTS) {
		return(total_events);
	} else if (command == AUDIT_EVENT_ON) {
		audit_on(arg);
	} else if (command == AUDIT_EVENT_OFF) {
		audit_off(arg);
		// Eg: ioctl(x,SYS_exit);
	} else if (command == AUDIT_FLUSH) {
			// Turn off all auditing.
			// This is generally sent when /etc/audit/audit.conf has been updated
			// and the user wishes to establish a new audit policy.
			int counter;
			for(counter=1; counter<=MAXAUDIT; counter++) {
				if(active_audit[counter]) audit_off(active_audit[counter]);
			}		

			// Also reset the lost/total event counters here.
			lost_events=0;
			total_events=0;
	} else {
		return -ENOSYS;	// Redhat kernel team suggest ENOTTY ? Clarify this.
	}

	return 0;
}


////////////////////////////////////////////////////////////////////////////////
// Here we give the user the data.
// NOTE that this module may be executed at any time - even in the middle
// of the execution of something else. Hence, any variables that are changed
// by this routine should be surrounded by a kernel lock if they are used
// elsewhere!
//
// In particular:
//  read_position
//  list_head
//  list_tail
//
// Let me just repeat this in another way:
// WARNING: DO NOT CHANGE OR RELY ON THE VARIABLES ABOVE UNLESS YOU ARE
//		  IN A LOCKED STATE - THEY MAY BE CORRUPTED BY THE
//		  AUDIT_READ MODULE, WHICH IS CALLED BY SIGNAL/INTERRUPT
//
// Unfortunately, we also need to spin-lock within this routine also.
// Otherwise, an interrupt might occur when we were half way through modifying
// something like read_position - and if the interrupt happens to generate an
// audit event, then we may be copying data to the wrong location in the array,
// leading to data corruption.
// Spin locking too much code slows down the system unfortunately. I'll try
// and keep it minimal.
////////////////////////////////////////////////////////////////////////////////

static ssize_t auditmodule_read(struct file *file, char *filebuffer, size_t length, loff_t *ppos)
{
	// Number of bytes actually written to the buffer
	int bytes_avail_to_send=0;
	int bytes_to_write=0;

	// Lock the kernel
	down(&audit_lock);

	// Just in case we have received a terminate request.
	if(AUDIT_IS_RUNNING == 0 || auditdaemon_task_struct == NULL) {
		up(&audit_lock);
		return(0);
	}

	// Do we have any data?
	if(list_head == (Node *)NULL) {
		up(&audit_lock);
		return(0);
	}
	
	if(read_position==(void *)NULL) {
		// Our first time into this routine, or first time back after all data has been read.
		read_position=list_head->location;
	}
	
	bytes_avail_to_send = (int)(((void *)list_head->location + list_head->size) - (void *)read_position);
	
	if(bytes_avail_to_send < length) {
		bytes_to_write=bytes_avail_to_send;
	} else {
		bytes_to_write=length;
	}
	
	if(copy_to_user(filebuffer,read_position,bytes_to_write))
	{
		up(&audit_lock);
		return -EFAULT;
	}

	read_position += bytes_to_write;

	// Have we reached the end of the current nodes data?
	if((void *)read_position >= ((void *)list_head->location + list_head->size)) {
		Node *tempnode;

		tempnode = list_head;
		list_head = list_head->next;
		
		// Excellent, we're at the end of the current audit event.
		// Remove this from the linked list, and set the read_position to the next record.
		if(list_head != (Node *)NULL) {
			read_position = list_head->location;
		} else {
			read_position = (Node *)NULL;
		}

		// No events remaining? Set list_tail to null also.
		if(tempnode->next == (Node *)NULL) {
			list_tail=(Node *)NULL;
		}
	
		// NOTE: If you look at /proc/meminfo before and after these calls, the memory will not SEEM to be released.
		//       However, it's just that the kernel does not release the memory IMMEDIATELY. It seems to wait for
		//		 an appropriate opportunity before actually giving the RAM back to the system.
		kfree(tempnode->location);
		kfree(tempnode);
	}
	
	up(&audit_lock);

	return(bytes_to_write);
}


////////////////////////////////////////////////////////////////////////////////
// open the device
////////////////////////////////////////////////////////////////////////////////
static int auditmodule_open(struct inode* node, struct file*  the_file)
{
	int device_minor = MINOR(node->i_rdev) & 0xf;  // Device minor number

	// Can only have one audit device, and it can only be open once
	if(device_minor)
		return -ENODEV;
	
	down(&audit_lock);
	
	if(auditdaemon_task_struct != NULL) {
		up(&audit_lock);
		return -EBUSY;
	}

	// reset the lost packets indicator
	lost_events=0;
	total_events=0;
	
	// Fetch the task structure of the process that opened the device
	auditdaemon_task_struct = current;

	up(&audit_lock);
	
	printk("Auditing: Process %d has opened the device\n", auditdaemon_task_struct->pid);

	MOD_INC_USE_COUNT; // Increment module count

	return 0;
}

////////////////////////////////////////////////////////////////////////////////
// close the device and release resources
////////////////////////////////////////////////////////////////////////////////
static int auditmodule_close(struct inode* node, struct file*  the_file)
{
	down(&audit_lock);
	AUDIT_IS_RUNNING = FALSE;
	
	auditdaemon_task_struct=NULL;

	printk("AUDITMODULE: Audit daemon has closed dev/audit.\n");
	printk("AUDITMODULE: Events lost due low memory this session: %d\n",lost_events);
	printk("AUDITMODULE: Total Events processed this session: %d\n",total_events);
	lost_events=0;
	total_events=0;

	up(&audit_lock);
	
	MOD_DEC_USE_COUNT; // Decrement module usage
	
	return 0;
}


// Prepare the data string for reading.

static char *info_makedata(void)
{
	enum { DATA_LEN = 150 };
	char *data;
	char processid[20];

	if ((data = kmalloc(DATA_LEN, GFP_KERNEL)) == NULL)
		return NULL;

	if (auditdaemon_task_struct != NULL) {
		snprintf(processid, 20, "%d",
			(int)auditdaemon_task_struct->pid);
	} else {
		snprintf(processid, 20, "No Daemon Running");
	}

	snprintf(data, DATA_LEN,
		"SNARE Version: %d.%d.%d\n"
		"Audit Active: %d\n"
		"Audit Process ID: %s\n"
		"Events Lost This Session: %d\n"
		"Events Processed This Session: %d\n",
			AUDITMODULE_MAJOR_VERSION, AUDITMODULE_MINOR_VERSION, AUDITMODULE_PATCH_VERSION,
			AUDIT_IS_RUNNING, processid, lost_events, total_events);

	return data;
}

// /proc/auditinfo open.
static int info_open(struct inode *inode, struct file *file)
{
	char *data;

	down(&audit_lock);
	data = info_makedata();
	up(&audit_lock);

	if (data == NULL)
		return -ENODEV;
	file->private_data = data;

	return 0;
}

static int info_close(struct inode *inode, struct file *file)
{
	kfree(file->private_data);
	return 0;
}

static ssize_t
info_read(struct file *file, char *ubuf, size_t length, loff_t *ppos)
{
	int bytes_to_write;
	int pos = *ppos;
	char *data = file->private_data;

	if (length == 0)		// Some users are so twisted.
		return 0;
	if ((bytes_to_write = strlen(data) - pos) <= 0)
		return 0;			// EOF
	if (bytes_to_write >= length)
		bytes_to_write = length;
	if (copy_to_user(ubuf, data + pos, bytes_to_write))
		return -EFAULT;
	*ppos = pos + bytes_to_write;
	return bytes_to_write;
}



////////////////////////////////////////////////////////////////////////////////
// Signal the audit daemon that is reading from /dev/audit (if it is there).
// Note that this is called from within audit_event, but is not IRQ locked.
////////////////////////////////////////////////////////////////////////////////
void signal_auditd()
{
	static struct siginfo sig_info;	/* Signal information */

	// Not much use sending signals to a non-existant process.
	if((auditdaemon_task_struct != NULL) && AUDIT_IS_RUNNING)
	{
		// Setup signal information and send it
		sig_info.si_signo = SIGIO;
		sig_info.si_errno = 0;
		sig_info.si_code  = SI_KERNEL;
		send_sig_info(SIGIO, &sig_info, auditdaemon_task_struct);
	}
}


////////////////////////////////////////////////////////////////////////////////
// Free buffers, release our overlay functions and exit.
////////////////////////////////////////////////////////////////////////////////
void cleanup_module(void)
{
	int counter=0;
	Node *currentnode;
	Node *tempnode;

	// printk("Auditing: Cleaning up module\n");

	// Lock down the kernel, so no new events can occur.
	down(&audit_lock);
	AUDIT_IS_RUNNING=0;

	// remove the proc entry, so the daemon won't try to get more data
	remove_proc_entry(AUDITDEV_NAME,NULL);
	remove_proc_entry(AUDITINFO_NAME,NULL);

	// Release the audited functions - skip AUDIT_NULL though.
	for(counter=1; counter<=MAXAUDIT; counter++) {
		if(active_audit[counter]) audit_off(active_audit[counter]);
	}
	
	// Reset all our buffers here.

	// We may have to wait for a few seconds, just in case we are half way through a write to one of these buffers.
	// Otherwise, we're stealing the memory location out from under the nose of some routine.
	// Not the nicest way to time...
	
	// NOTE: I don't think we need this any more, due to the semaphore locking improvements.
	// Will keep this here until after alpha/beta testing.
	
//	while(Audit_Log_Write_Count) {
//		int currenttime;
//				
//		currenttime=jiffies;
//		while(jiffies < currenttime + (1 * HZ)) {
//			schedule();
//		}
//	}
//
//	down(&audit_lock);
		
	// Free our linked list, just in case there are any left over.
	currentnode=list_head;
	while(currentnode != (Node *)NULL) {
		tempnode=currentnode;
		currentnode=currentnode->next;
		kfree(tempnode->location);
		kfree(tempnode);
	}

	up(&audit_lock);

	printk("SNARE Audit Module: Exiting\n");
}

// Don't turn it on twice.
void audit_on(int auditnum)
{
	// printk("Debug: Turning on event %d\n",auditnum);
	switch (auditnum)
	{
		case SYS_open:
			if(active_audit[AUDIT_open]) return;
			orig_open=sys_call_table[SYS_open];
			sys_call_table[SYS_open]=audit_open;
			active_audit[AUDIT_open]=SYS_open;
			break;
		case SYS_creat:
			if(active_audit[AUDIT_creat]) return;
			orig_creat=sys_call_table[SYS_creat];
			sys_call_table[SYS_creat]=audit_creat;
			active_audit[AUDIT_creat]=SYS_creat;
			break;
		case SYS_execve:
			if(active_audit[AUDIT_execve]) return;
			orig_execve=sys_call_table[SYS_execve];
			sys_call_table[SYS_execve]=audit_execve;
			active_audit[AUDIT_execve]=SYS_execve;
			break;
		case SYS_exit:
			if(active_audit[AUDIT_exit]) return;
			orig_exit=sys_call_table[SYS_exit];
			sys_call_table[SYS_exit]=audit_exit;
			active_audit[AUDIT_exit]=SYS_exit;
			break;
		case SYS_mkdir:
			if(active_audit[AUDIT_mkdir]) return;
			orig_mkdir=sys_call_table[SYS_mkdir];
			sys_call_table[SYS_mkdir]=audit_mkdir;
			active_audit[AUDIT_mkdir]=SYS_mkdir;
			break;
		case SYS_unlink:
			if(active_audit[AUDIT_unlink]) return;
			orig_unlink=sys_call_table[SYS_unlink];
			sys_call_table[SYS_unlink]=audit_unlink;
			active_audit[AUDIT_unlink]=SYS_unlink;
			break;
		case SYS_mknod:
			if(active_audit[AUDIT_mknod]) return;
			orig_mknod=sys_call_table[SYS_mknod];
			sys_call_table[SYS_mknod]=audit_mknod;
			active_audit[AUDIT_mknod]=SYS_mknod;
			break;
		case SYS_rmdir:
			if(active_audit[AUDIT_rmdir]) return;
			orig_rmdir=sys_call_table[SYS_rmdir];
			sys_call_table[SYS_rmdir]=audit_rmdir;
			active_audit[AUDIT_rmdir]=SYS_rmdir;
			break;
		case SYS_chown:
			if(active_audit[AUDIT_chown]) return;
			orig_chown=sys_call_table[SYS_chown];
			sys_call_table[SYS_chown]=audit_chown;
			active_audit[AUDIT_chown]=SYS_chown;
			break;
		case SYS_lchown:
			if(active_audit[AUDIT_lchown]) return;
			orig_lchown=sys_call_table[SYS_lchown];
			sys_call_table[SYS_lchown]=audit_lchown;
			active_audit[AUDIT_lchown]=SYS_lchown;
			break;
		case SYS_chown32:
			if(active_audit[AUDIT_chown32]) return;
			orig_chown32=sys_call_table[SYS_chown32];
			sys_call_table[SYS_chown32]=audit_chown32;
			active_audit[AUDIT_chown32]=SYS_chown32;
			break;
		case SYS_lchown32:
			if(active_audit[AUDIT_lchown32]) return;
			orig_lchown32=sys_call_table[SYS_lchown32];
			sys_call_table[SYS_lchown32]=audit_lchown32;
			active_audit[AUDIT_lchown32]=SYS_lchown32;
			break;
		case SYS_chmod:
			if(active_audit[AUDIT_chmod]) return;
			orig_chmod=sys_call_table[SYS_chmod];
			sys_call_table[SYS_chmod]=audit_chmod;
			active_audit[AUDIT_chmod]=SYS_chmod;
			break;
		case SYS_symlink:
			if(active_audit[AUDIT_symlink]) return;
			orig_symlink=sys_call_table[SYS_symlink];
			sys_call_table[SYS_symlink]=audit_symlink;
			active_audit[AUDIT_symlink]=SYS_symlink;
			break;
		case SYS_link:
			if(active_audit[AUDIT_link]) return;
			orig_link=sys_call_table[SYS_link];
			sys_call_table[SYS_link]=audit_link;
			active_audit[AUDIT_link]=SYS_link;
			break;
		case SYS_rename:
			if(active_audit[AUDIT_rename]) return;
			orig_rename=sys_call_table[SYS_rename];
			sys_call_table[SYS_rename]=audit_rename;
			active_audit[AUDIT_rename]=SYS_rename;
			break;
		case SYS_reboot:
			if(active_audit[AUDIT_reboot]) return;
			orig_reboot=sys_call_table[SYS_reboot];
			sys_call_table[SYS_reboot]=audit_reboot;
			active_audit[AUDIT_reboot]=SYS_reboot;
			break;
		case SYS_truncate:
			if(active_audit[AUDIT_truncate]) return;
			orig_truncate=sys_call_table[SYS_truncate];
			sys_call_table[SYS_truncate]=audit_truncate;
			active_audit[AUDIT_truncate]=SYS_truncate;
			break;
		case SYS_chroot:
			if(active_audit[AUDIT_chroot]) return;
			orig_chroot=sys_call_table[SYS_chroot];
			sys_call_table[SYS_chroot]=audit_chroot;
			active_audit[AUDIT_chroot]=SYS_chroot;
			break;
		case SYS_setuid:
			if(active_audit[AUDIT_setuid]) return;
			orig_setuid=sys_call_table[SYS_setuid];
			sys_call_table[SYS_setuid]=audit_setuid;
			active_audit[AUDIT_setuid]=SYS_setuid;
			break;
		case SYS_setreuid:
			if(active_audit[AUDIT_setreuid]) return;
			orig_setreuid=sys_call_table[SYS_setreuid];
			sys_call_table[SYS_setreuid]=audit_setreuid;
			active_audit[AUDIT_setreuid]=SYS_setreuid;
			break;
		case SYS_setresuid:
			if(active_audit[AUDIT_setresuid]) return;
			orig_setresuid=sys_call_table[SYS_setresuid];
			sys_call_table[SYS_setresuid]=audit_setresuid;
			active_audit[AUDIT_setresuid]=SYS_setresuid;
			break;
		case SYS_setuid32:
			if(active_audit[AUDIT_setuid32]) return;
			orig_setuid32=sys_call_table[SYS_setuid32];
			sys_call_table[SYS_setuid32]=audit_setuid32;
			active_audit[AUDIT_setuid32]=SYS_setuid32;
			break;
		case SYS_setreuid32:
			if(active_audit[AUDIT_setreuid32]) return;
			orig_setreuid32=sys_call_table[SYS_setreuid32];
			sys_call_table[SYS_setreuid32]=audit_setreuid32;
			active_audit[AUDIT_setreuid32]=SYS_setreuid32;
			break;
		case SYS_setresuid32:
			if(active_audit[AUDIT_setresuid32]) return;
			orig_setresuid32=sys_call_table[SYS_setresuid32];
			sys_call_table[SYS_setresuid32]=audit_setresuid32;
			active_audit[AUDIT_setresuid32]=SYS_setresuid32;
			break;
		case SYS_setgid:
			if(active_audit[AUDIT_setgid]) return;
			orig_setgid=sys_call_table[SYS_setgid];
			sys_call_table[SYS_setgid]=audit_setgid;
			active_audit[AUDIT_setgid]=SYS_setgid;
			break;
		case SYS_setregid:
			if(active_audit[AUDIT_setregid]) return;
			orig_setregid=sys_call_table[SYS_setregid];
			sys_call_table[SYS_setregid]=audit_setregid;
			active_audit[AUDIT_setregid]=SYS_setregid;
			break;
		case SYS_setresgid:
			if(active_audit[AUDIT_setresgid]) return;
			orig_setresgid=sys_call_table[SYS_setresgid];
			sys_call_table[SYS_setresgid]=audit_setresgid;
			active_audit[AUDIT_setresgid]=SYS_setresgid;
			break;
		case SYS_setgid32:
			if(active_audit[AUDIT_setgid32]) return;
			orig_setgid32=sys_call_table[SYS_setgid32];
			sys_call_table[SYS_setgid32]=audit_setgid32;
			active_audit[AUDIT_setgid32]=SYS_setgid32;
			break;
		case SYS_setregid32:
			if(active_audit[AUDIT_setregid32]) return;
			orig_setregid32=sys_call_table[SYS_setregid32];
			sys_call_table[SYS_setregid32]=audit_setregid32;
			active_audit[AUDIT_setregid32]=SYS_setregid32;
			break;
		case SYS_setresgid32:
			if(active_audit[AUDIT_setresgid32]) return;
			orig_setresgid32=sys_call_table[SYS_setresgid32];
			sys_call_table[SYS_setresgid32]=audit_setresgid32;
			active_audit[AUDIT_setresgid32]=SYS_setresgid32;
			break;
		case SYS_truncate64:
			if(active_audit[AUDIT_truncate64]) return;
			orig_truncate64=sys_call_table[SYS_truncate64];
			sys_call_table[SYS_truncate64]=audit_truncate64;
			active_audit[AUDIT_truncate64]=SYS_truncate64;
			break;
		case SYS_socketcall:
			if(active_audit[AUDIT_socketcall]) return;
			orig_socketcall=sys_call_table[SYS_socketcall];
			sys_call_table[SYS_socketcall]=audit_socketcall;
			active_audit[AUDIT_socketcall]=SYS_socketcall;
			break;
		case SYS_create_module:
			if(active_audit[AUDIT_create_module]) return;
			orig_create_module=sys_call_table[SYS_create_module];
			sys_call_table[SYS_create_module]=audit_create_module;
			active_audit[AUDIT_create_module]=SYS_create_module;
			break;
		case SYS_mount:
			if(active_audit[AUDIT_mount]) return;
			orig_mount=sys_call_table[SYS_mount];
			sys_call_table[SYS_mount]=audit_mount;
			active_audit[AUDIT_mount]=SYS_mount;
			break;			
		case SYS_umount:
			if(active_audit[AUDIT_umount]) return;
			orig_umount=sys_call_table[SYS_umount];
			sys_call_table[SYS_umount]=audit_umount;
			active_audit[AUDIT_umount]=SYS_umount;
			break;
		case SYS_umount2:
			if(active_audit[AUDIT_umount2]) return;
			orig_umount2=sys_call_table[SYS_umount2];
			sys_call_table[SYS_umount2]=audit_umount2;
			active_audit[AUDIT_umount2]=SYS_umount2;
			break;
//		case SYS_delete_module:
//			if(active_audit[AUDIT_delete_module]) return;
//			orig_delete_module=sys_call_table[SYS_delete_module];
//			sys_call_table[SYS_delete_module]=audit_delete_module;
//			active_audit[AUDIT_delete_module]=SYS_delete_module;
//			break;
			
// No need for the calls below, they are covered by their non f* calls (eg: ftruncate calls truncate64 on my system).
//		case SYS_ftruncate:
//			if(active_audit[AUDIT_ftruncate]) return;
//			orig_ftruncate=sys_call_table[SYS_ftruncate];
//			sys_call_table[SYS_ftruncate]=audit_ftruncate;
//			active_audit[AUDIT_ftruncate]=SYS_ftruncate;
//			break;
//		case SYS_fchmod:
//			if(active_audit[AUDIT_fchmod]) return;
//			orig_fchmod=sys_call_table[SYS_fchmod];
//			sys_call_table[SYS_fchmod]=audit_fchmod;
//			active_audit[AUDIT_fchmod]=SYS_fchmod;
//			break;
//		case SYS_ftruncate64:
//			if(active_audit[AUDIT_ftruncate64]) return;
//			orig_ftruncate64=sys_call_table[SYS_ftruncate64];
//			sys_call_table[SYS_ftruncate64]=audit_ftruncate64;
//			active_audit[AUDIT_ftruncate64]=SYS_ftruncate64;
//			break;
//		case SYS_fchown:
//			if(active_audit[AUDIT_fchown]) return;
//			orig_fchown=sys_call_table[SYS_fchown];
//			sys_call_table[SYS_fchown]=audit_fchown;
//			active_audit[AUDIT_fchown]=SYS_fchown;
//			break;
	}
}

void audit_off(int auditnum)
{
	switch (auditnum)
	{
		case SYS_open:
			if(active_audit[AUDIT_open]==0) return;
			sys_call_table[SYS_open]=orig_open;
			active_audit[AUDIT_open]=0;
			break;
		case SYS_creat:
			if(active_audit[AUDIT_creat]==0) return;
			sys_call_table[SYS_creat]=orig_creat;
			active_audit[AUDIT_creat]=0;
			break;
		case SYS_execve:
			if(active_audit[AUDIT_execve]==0) return;
			sys_call_table[SYS_execve]=orig_execve;
			active_audit[AUDIT_execve]=0;
			break;
		case SYS_exit:
			if(active_audit[AUDIT_exit]==0) return;
			sys_call_table[SYS_exit]=orig_exit;
			active_audit[AUDIT_exit]=0;
			break;
		case SYS_mkdir:
			if(active_audit[AUDIT_mkdir]==0) return;
			sys_call_table[SYS_mkdir]=orig_mkdir;
			active_audit[AUDIT_mkdir]=0;
			break;
		case SYS_unlink:
			if(active_audit[AUDIT_unlink]==0) return;
			sys_call_table[SYS_unlink]=orig_unlink;
			active_audit[AUDIT_unlink]=0;
			break;
		case SYS_mknod:
			if(active_audit[AUDIT_mknod]==0) return;
			sys_call_table[SYS_mknod]=orig_mknod;
			active_audit[AUDIT_mknod]=0;
			break;
		case SYS_rmdir:
			if(active_audit[AUDIT_rmdir]==0) return;
			sys_call_table[SYS_rmdir]=orig_rmdir;
			active_audit[AUDIT_rmdir]=0;
			break;
		case SYS_chown:
			if(active_audit[AUDIT_chown]==0) return;
			sys_call_table[SYS_chown]=orig_chown;
			active_audit[AUDIT_chown]=0;
			break;
		case SYS_lchown:
			if(active_audit[AUDIT_lchown]==0) return;
			sys_call_table[SYS_lchown]=orig_lchown;
			active_audit[AUDIT_lchown]=0;
			break;
		case SYS_chown32:
			if(active_audit[AUDIT_chown32]==0) return;
			sys_call_table[SYS_chown32]=orig_chown32;
			active_audit[AUDIT_chown32]=0;
			break;
		case SYS_lchown32:
			if(active_audit[AUDIT_lchown32]==0) return;
			sys_call_table[SYS_lchown32]=orig_lchown32;
			active_audit[AUDIT_lchown32]=0;
			break;
		case SYS_chmod:
			if(active_audit[AUDIT_chmod]==0) return;
			sys_call_table[SYS_chmod]=orig_chmod;
			active_audit[AUDIT_chmod]=0;
			break;
		case SYS_symlink:
			if(active_audit[AUDIT_symlink]==0) return;
			sys_call_table[SYS_symlink]=orig_symlink;
			active_audit[AUDIT_symlink]=0;
			break;
		case SYS_link:
			if(active_audit[AUDIT_link]==0) return;
			sys_call_table[SYS_link]=orig_link;
			active_audit[AUDIT_link]=0;
			break;
		case SYS_rename:
			if(active_audit[AUDIT_rename]==0) return;
			sys_call_table[SYS_rename]=orig_rename;
			active_audit[AUDIT_rename]=0;
			break;
		case SYS_reboot:
			if(active_audit[AUDIT_reboot]==0) return;
			sys_call_table[SYS_reboot]=orig_reboot;
			active_audit[AUDIT_reboot]=0;
			break;
		case SYS_truncate:
			if(active_audit[AUDIT_truncate]==0) return;
			sys_call_table[SYS_truncate]=orig_truncate;
			active_audit[AUDIT_truncate]=0;
			break;
		case SYS_chroot:
			if(active_audit[AUDIT_chroot]==0) return;
			sys_call_table[SYS_chroot]=orig_chroot;
			active_audit[AUDIT_chroot]=0;
			break;
		case SYS_setuid:
			if(active_audit[AUDIT_setuid]==0) return;
			sys_call_table[SYS_setuid]=orig_setuid;
			active_audit[AUDIT_setuid]=0;
			break;
		case SYS_setreuid:
			if(active_audit[AUDIT_setreuid]==0) return;
			sys_call_table[SYS_setreuid]=orig_setreuid;
			active_audit[AUDIT_setreuid]=0;
			break;
		case SYS_setresuid:
			if(active_audit[AUDIT_setresuid]==0) return;
			sys_call_table[SYS_setresuid]=orig_setresuid;
			active_audit[AUDIT_setresuid]=0;
			break;
		case SYS_setuid32:
			if(active_audit[AUDIT_setuid32]==0) return;
			sys_call_table[SYS_setuid32]=orig_setuid32;
			active_audit[AUDIT_setuid32]=0;
			break;
		case SYS_setreuid32:
			if(active_audit[AUDIT_setreuid32]==0) return;
			sys_call_table[SYS_setreuid32]=orig_setreuid32;
			active_audit[AUDIT_setreuid32]=0;
			break;
		case SYS_setresuid32:
			if(active_audit[AUDIT_setresuid32]==0) return;
			sys_call_table[SYS_setresuid32]=orig_setresuid32;
			active_audit[AUDIT_setresuid32]=0;
			break;
		case SYS_setgid:
			if(active_audit[AUDIT_setgid]==0) return;
			sys_call_table[SYS_setgid]=orig_setgid;
			active_audit[AUDIT_setgid]=0;
			break;
		case SYS_setregid:
			if(active_audit[AUDIT_setregid]==0) return;
			sys_call_table[SYS_setregid]=orig_setregid;
			active_audit[AUDIT_setregid]=0;
			break;
		case SYS_setresgid:
			if(active_audit[AUDIT_setresgid]==0) return;
			sys_call_table[SYS_setresgid]=orig_setresgid;
			active_audit[AUDIT_setresgid]=0;
			break;
		case SYS_setgid32:
			if(active_audit[AUDIT_setgid32]==0) return;
			sys_call_table[SYS_setgid32]=orig_setgid32;
			active_audit[AUDIT_setgid32]=0;
			break;
		case SYS_setregid32:
			if(active_audit[AUDIT_setregid32]==0) return;
			sys_call_table[SYS_setregid32]=orig_setregid32;
			active_audit[AUDIT_setregid32]=0;
			break;
		case SYS_setresgid32:
			if(active_audit[AUDIT_setresgid32]==0) return;
			sys_call_table[SYS_setresgid32]=orig_setresgid32;
			active_audit[AUDIT_setresgid32]=0;
			break;
		case SYS_truncate64:
			if(active_audit[AUDIT_truncate64]==0) return;
			sys_call_table[SYS_truncate64]=orig_truncate64;
			active_audit[AUDIT_truncate64]=0;
			break;
		case SYS_socketcall:
			if(active_audit[AUDIT_socketcall]==0) return;
			sys_call_table[SYS_socketcall]=orig_socketcall;
			active_audit[AUDIT_socketcall]=0;
			break;
		case SYS_create_module:
			if(active_audit[AUDIT_create_module]==0) return;
			sys_call_table[SYS_create_module]=orig_create_module;
			active_audit[AUDIT_create_module]=0;
			break;
		case SYS_mount:
			if(active_audit[AUDIT_mount]==0) return;
			sys_call_table[SYS_mount]=orig_mount;
			active_audit[AUDIT_mount]=0;
			break;
		case SYS_umount:
			if(active_audit[AUDIT_umount]==0) return;
			sys_call_table[SYS_umount]=orig_umount;
			active_audit[AUDIT_umount]=0;
			break;
		case SYS_umount2:
			if(active_audit[AUDIT_umount2]==0) return;
			sys_call_table[SYS_umount2]=orig_umount2;
			active_audit[AUDIT_umount2]=0;
			break;
//		case SYS_delete_module:
//			if(active_audit[AUDIT_delete_module]==0) return;
//			sys_call_table[SYS_delete_module]=orig_delete_module;
//			active_audit[AUDIT_delete_module]=0;
//			break;
//		case SYS_ftruncate:
//			if(active_audit[AUDIT_ftruncate]==0) return;
//			sys_call_table[SYS_ftruncate]=orig_ftruncate;
//			active_audit[AUDIT_ftruncate]=0;
//			break;
//		case SYS_fchmod:
//			if(active_audit[AUDIT_fchmod]==0) return;
//			sys_call_table[SYS_fchmod]=orig_fchmod;
//			active_audit[AUDIT_fchmod]=0;
//			break;
//		case SYS_ftruncate64:
//			if(active_audit[AUDIT_ftruncate64]==0) return;
//			sys_call_table[SYS_ftruncate64]=orig_ftruncate64;
//			active_audit[AUDIT_ftruncate64]=0;
//			break;
//		case SYS_fchown:
//			if(active_audit[AUDIT_fchown]==0) return;
//			sys_call_table[SYS_fchown]=orig_fchown;
//			active_audit[AUDIT_fchown]=0;
//			break;
	}
}



Node *alloc_event(int token_size)
{
	void *eventpointer;
	Node * nodepointer;

	eventpointer=kmalloc(token_size,GFP_KERNEL);
	if(eventpointer == (void *)NULL) {
		// Not enough RAM available aparently.
		return(0);
	}

	nodepointer=(Node *)kmalloc(sizeof(Node),GFP_KERNEL);
	if(nodepointer == (Node *)NULL) {
		// Not enough RAM available aparently.
		// Free the event pointer if we've got this far.
		kfree(eventpointer);
		return(0);
	}

	nodepointer->location=eventpointer;
	nodepointer->size=token_size;
	nodepointer->next=(Node *) NULL;	// This is the end.

	return nodepointer;
}

void append_event(Node *nodepointer)
{

	if(list_tail != (Node *)NULL) {
		list_tail->next=nodepointer;
	}
	list_tail=nodepointer;

	// Is this the first node in our linked list?
	if(list_head == (Node *) NULL) {
		list_head=nodepointer;
	}
}




////////////////////////////////////////////////////////////////////////////////
// Control the reporting of the audit event.
////////////////////////////////////////////////////////////////////////////////

//int audit_event(void * event, int userset)
int audit_event(Node * node, int userset)
{
	int class=0;
	static struct timeval time_temp;	// to get the current time
	static null_class * any_event;
	int signal;
	char tempbuffer[MAX_PATH];
	
	any_event = node->location;
	
	// Increment our write count - this is used to make sure module shutdown does
	// not free our memory prematurely.
//	down(&audit_lock);
//	Audit_Log_Write_Count++;
//	up(&audit_lock);
	
	signal=1;							// Signal the daemon

	down(&audit_lock);
	
	// No use auditing anything unless the auditd is listening.
	//if(auditdaemon_task_struct == NULL || audit_device_open == 0) {
	if(auditdaemon_task_struct == NULL) {
		// Decrement our write count - this is used to make sure module shutdown does
		// not free our memory prematurely.		
//		Audit_Log_Write_Count--;
		
		up(&audit_lock);

		// New: Free our node, since there is no audit daemon to receive it.
		kfree(node->location);
		kfree(node);

		return -ENODEV;
	}

	// No use auditing if we're in the process of shutting down.
	if(AUDIT_IS_RUNNING == FALSE) {
		// Decrement our write count - this is used to make sure module shutdown does
		// not free our memory prematurely.
//		Audit_Log_Write_Count--;
	
		up(&audit_lock);
		kfree(node->location);
		kfree(node);
		return(1);
	}
	
	
	// Reset both counts if we are close to maxint, otherwise we will have strange statistics.
	if(total_events>=(INT_MAX - 1)) {
		total_events=0;
		lost_events=0;
	}
	total_events++;

	up(&audit_lock);
	
	// The routine that sends data to us should have filled in
	// the event_id in the header_token.
	// And the returncode should also be known.
	class = any_event->t_header.event_class;

	// fill in the time of the event
	// NOTE: Redhat kernel team removed the spin_locks - perhaps this down/up is not required. Retain for the moment.
	down(&audit_lock);
	do_gettimeofday(&time_temp);
	up(&audit_lock);

	any_event->t_header.time.tv_sec = time_temp.tv_sec;
	any_event->t_header.time.tv_usec = time_temp.tv_usec;

	// If userid is already set, don't grab it again.
	if(!userset) {
		// Get the current user data
		// This will be the same for any event.
		any_event->t_header.user_id=current->uid;
		any_event->t_header.group_id=current->gid;
		any_event->t_header.euser_id=current->euid;
		any_event->t_header.egroup_id=current->egid;
	}

	any_event->t_header.event_size = node->size - sizeof(header_token);
	
	if(class == AUDIT_CLASS_IO)	{
		io_class *event;
		event = (io_class *)any_event;
		strncpy(event->t_pwd.path,d_path(current->fs->pwd,current->fs->rootmnt,tempbuffer,MAX_PATH),MAX_PATH);
	} else if (class == AUDIT_CLASS_EXEC) {
		ex_class *event;
		event = (ex_class *)any_event;
		strncpy(event->t_pwd.path,d_path(current->fs->pwd,current->fs->rootmnt,tempbuffer,MAX_PATH),MAX_PATH);
	} else if (class == AUDIT_CLASS_PC) {
		;
	} else if (class == AUDIT_CLASS_CH) {
		ch_class *event;
		event = (ch_class *)any_event;
		strncpy(event->t_pwd.path,d_path(current->fs->pwd,current->fs->rootmnt,tempbuffer,MAX_PATH),MAX_PATH);
	} else if (class == AUDIT_CLASS_CP) {
		cp_class *event;
		event = (cp_class *)any_event;
		strncpy(event->t_pwd.path,d_path(current->fs->pwd,current->fs->rootmnt,tempbuffer,MAX_PATH),MAX_PATH);
	} else if (class == AUDIT_CLASS_SU) {
		;
	} else if (class == AUDIT_CLASS_NET) {
		;
	} else if (class == AUDIT_CLASS_AD) {
		;
	} else {
		printk(KERN_WARNING "audit: Cannot handle class 0x%x\n", class);
		// Add 1 to the lost events tally
		down(&audit_lock); lost_events++; up(&audit_lock);
	}

	down(&audit_lock);
	append_event(node);

//	Audit_Log_Write_Count--;
	
	up(&audit_lock);

	signal_auditd();
	
	return(1); // no event handler
}





////////////////////////////////////////////////////////////////////////////////
// The following section contains the modified system calls that include
// the auditing that we require.
// Note that the kernel will be IRQ spin-locked once the audit_event routine
// is called.
////////////////////////////////////////////////////////////////////////////////

// Audit file opens.
asmlinkage int audit_open(const char *pathname, int flag, mode_t mode)
{
	io_class *	event;
	Node *		node;
	int			returncode;

	
	returncode=orig_open(pathname, flag, mode);

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_open;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_path.path[0]='\0';
	if(pathname) {
		// Only grab the first MAX_PATH characters to guard against overflows.
		strncpy_from_user(event->t_path.path,pathname,MAX_PATH);
	}
	event->t_attributes.mode=flag;
	event->t_attributes.createmode=mode;

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Audit file opens (create only).
asmlinkage int audit_creat(const char *pathname, mode_t mode)
{
	io_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_creat(pathname, mode);

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_creat;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_path.path[0]='\0';
	if(pathname) {
		strncpy_from_user(event->t_path.path,pathname,MAX_PATH);
	}
	event->t_attributes.mode=O_CREAT|O_WRONLY|O_TRUNC;
	event->t_attributes.createmode=mode;

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Audit execve command executions
// Execve is not handled quite the same as other audit events.
// See implementation of sys_execve() in process.c
int audit_execve(struct pt_regs regs)
{
	ex_class *	event;
	Node *		node;
	int			returncode;
	int error;

	char *filename=NULL;
	char **argv=NULL;

	unsigned int delimiter_required=0;
	char tempstring[MAX_PATH];
	
	filename = getname((char*)regs.ebx);
	error = PTR_ERR(filename);
	if(IS_ERR(filename)) {
		return error;
	}
		
	argv = (char **)regs.ecx;

    // If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		goto bail;
	}

	if ((node = alloc_event(sizeof(ex_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		goto bail;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	// Construct the argument list
	// and shoehorn it into exec_args

	event->t_execargs.args[0]='\0';
	while(argv && *argv)
	{
		strncpy_from_user(tempstring,*argv,MAX_PATH-1);
		tempstring[MAX_PATH]='\0';

		// Allow for the delimiter
		// if((strlen(ex_event.t_execargs.args) + strlen(*argv)) < MAX_PATH)
		// changed - argv is currently in userspace. We need to make a copy of the data first.
		if((strlen(event->t_execargs.args) + strlen(tempstring) +1) < (MAX_PATH-1))
		{
			if(!delimiter_required)
			{
				delimiter_required=1;
			} else {
				strcat(event->t_execargs.args," ");
			}
			// strcat(event->t_execargs.args,*argv);
			strcat(event->t_execargs.args,tempstring);
		} else {
			// If the argument size is greater than MAX_PATH, then ignore this argument, continue on to the next.
			// break;
			argv++;
			continue;
		}
		argv++;
	}

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);
	
	returncode = do_execve(filename, (char**)regs.ecx, (char**)regs.edx, &regs);
	if(returncode == 0) {
		current->ptrace &= ~PT_DTRACE;
	}
	event->t_return.returncode = returncode;
	event->t_path.path[0]='\0';
	strncpy(event->t_path.path,filename,MAX_PATH);

	putname(filename);

	event->t_header.event_class=AUDIT_CLASS_EXEC;
	event->t_header.event_id=SYS_execve;

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);
	
	return(returncode);

bail:
	returncode=do_execve(filename, (char**)regs.ecx, (char**)regs.edx, &regs);
		
	if(returncode == 0) {
		current->ptrace &= ~PT_DTRACE;
	}
	putname(filename);
	
	return(returncode);
}


// Audit process exits.
// This may be useful if we want to audit login/logouts - because
// as far as I can work out, the only way we can do it is to watch for
// execve/exit of /bin/login (without hacking the kernel).
//
// Note: current->comm is a 16 character representation of the executed command.
asmlinkage int audit_exit(int status)
{
	pc_class *	event;
	Node *		node;
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || (AUDIT_IS_RUNNING && (current->pid == auditdaemon_task_struct->pid)))
	{
		goto bail;
	}

	if ((node = alloc_event(sizeof(pc_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		goto bail;
	}
	event = node->location;
	
	event->t_return.returncode=status;

	event->t_header.event_class=AUDIT_CLASS_PC;
	event->t_header.event_id=SYS_exit;

	event->t_process.pid=current->pid;
	// Zero the string just in case this fails.
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

bail:
	return orig_exit(status);
}

// Audit mkdir directory creations.
asmlinkage int audit_mkdir(const char *path, mode_t mode)
{
	io_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_mkdir(path,mode);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}
	event->t_path.path[MAX_PATH-1]='\0';

	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_mkdir;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_attributes.mode=0;
	event->t_attributes.createmode=(unsigned long)mode;

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return(returncode);
}

// audit file unlinks
asmlinkage int	audit_unlink (const char *path)
{
	io_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_unlink(path);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_unlink;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	// Store the path
	// zero the path, just in case strncpy fails.
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return returncode;
}

// Create a special file/directory
asmlinkage int audit_mknod (const char *pathname, mode_t mode, dev_t dev)
{
	io_class *	event;
	Node *		node;
	int			returncode;
	
	returncode=orig_mknod(pathname, mode, dev);

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_mknod;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_path.path[0]='\0';
	if(pathname) {
		strncpy_from_user(event->t_path.path,pathname,MAX_PATH);
	}
	event->t_attributes.mode=(int)mode;
	event->t_attributes.createmode=(unsigned long)dev;

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Audit rmdir directory removals.
asmlinkage int audit_rmdir(const char *path)
{
	io_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_rmdir(path);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_rmdir;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	// Store the path
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);
	
	return(returncode);
}

// Audit change ownership
asmlinkage int audit_chown(const char *path, uid_t owner, gid_t group)
{
	ch_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_chown(path,owner,group);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(ch_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_CH;
	event->t_header.event_id=SYS_chown;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_owner.owner=(int) owner;
	event->t_owner.group=(int) group;

	// Store the path
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return(returncode);

}

// Same as chown, but does not follow symlinks.
asmlinkage int audit_lchown(const char *path, uid_t owner, gid_t group)
{
	ch_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_lchown(path,owner,group);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(ch_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_CH;
	event->t_header.event_id=SYS_lchown;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_owner.owner=(int) owner;
	event->t_owner.group=(int) group;

	// Store the path
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return(returncode);
}

// Change ownership
asmlinkage int audit_chown32(const char *path, uid_t owner, gid_t group)
{
	ch_class *	event;
	Node *		node;
	int			returncode;
	
	returncode=orig_chown32(path,owner,group);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(ch_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_CH;
	event->t_header.event_id=SYS_chown32;

	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_owner.owner=(int) owner;
	event->t_owner.group=(int) group;

	// Store the path
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return(returncode);
}

// Same as chown, but does not follow symlinks.
asmlinkage int audit_lchown32(const char *path, uid_t owner, gid_t group)
{
	ch_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_lchown32(path,owner,group);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(ch_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_CH;
	event->t_header.event_id=SYS_lchown32;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_owner.owner=(int) owner;
	event->t_owner.group=(int) group;

	// Store the path
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return(returncode);
}


// Change the mode of a file or directory
asmlinkage int audit_chmod (const char *pathname, mode_t mode)
{
	io_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_chmod(pathname, mode);

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_chmod;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_path.path[0]='\0';
	if(pathname) {
		strncpy_from_user(event->t_path.path,pathname,MAX_PATH);
	}
	event->t_attributes.mode=(int)NULL;
	event->t_attributes.createmode=(unsigned long)mode;

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Create a symbolic link
asmlinkage int audit_symlink (const char *oldpath, const char *newpath)
{
	cp_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_symlink(oldpath, newpath);

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(cp_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_CP;
	event->t_header.event_id=SYS_symlink;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_sourcepath.path[0]='\0';
	if(oldpath) {
		strncpy_from_user(event->t_sourcepath.path,oldpath,MAX_PATH);
	}
	event->t_destpath.path[0]='\0';
	if(newpath) {
		strncpy_from_user(event->t_destpath.path,newpath,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Create a hard link
asmlinkage int audit_link (const char *oldpath, const char *newpath)
{
	cp_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_link(oldpath, newpath);

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(cp_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_CP;
	event->t_header.event_id=SYS_link;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_sourcepath.path[0]='\0';
	if(oldpath) {
		strncpy_from_user(event->t_sourcepath.path,oldpath,MAX_PATH);
	}
	event->t_destpath.path[0]='\0';
	if(newpath) {
		strncpy_from_user(event->t_destpath.path,newpath,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// rename a file/directory
asmlinkage int audit_rename (const char *oldpath, const char *newpath)
{
	cp_class *	event;
	Node *		node;
	int			returncode;
	
	returncode=orig_rename(oldpath, newpath);

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(cp_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_CP;
	event->t_header.event_id=SYS_rename;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_sourcepath.path[0]='\0';
	if(oldpath) {
		strncpy_from_user(event->t_sourcepath.path,oldpath,MAX_PATH);
	}
	event->t_destpath.path[0]='\0';
	if(newpath) {
		strncpy_from_user(event->t_destpath.path,newpath,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Would like to audit reboots here.
// NOTE: this only audits when the reboot() system call is issued. There are many other
//       ways to reboot a system (eg: kill -9 1) - this will NOT catch these.
//       In fact, a normal reboot from the gdm login manager (for example), does not
//       seem to call the reboot() call. So this call may not be particularly useful.
// Only question is: does reboot return only after the system goes down?
// (in which case, I'll have to do the audit, fake the return code, and
// return with the orig_reboot().
// After all... we don't want to try and write stuff after filesystems have been unmounted.
asmlinkage int audit_reboot(int magic, int magic2, int flag, void *arg)
{
	pc_class *	event;
	Node *		node;
	int			returncode;
	
	returncode=orig_reboot(magic, magic2, flag, arg);

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct || flag == LINUX_REBOOT_CMD_CAD_ON || flag == LINUX_REBOOT_CMD_CAD_OFF)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(pc_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_PC;
	event->t_header.event_id=SYS_reboot;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// audit file truncates.
// Note that off_t can be shoehorned into a unsigned long for auditing.
asmlinkage int	audit_truncate (const char *path, off_t length)
{
	io_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_truncate(path,length);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_truncate;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	// Store the path
	// zero the path, just in case strncpy fails.
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}
	
	// Save off the offset.
	event->t_attributes.createmode=(unsigned long)length;

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return(returncode);
}

// audit root changes.
asmlinkage int	audit_chroot (const char *path)
{
	io_class *	event;
	Node *		node;
	int			returncode;

	returncode=orig_chroot(path);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return(returncode);
	}
	event = node->location;
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_chroot;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	// Store the path
	// zero the path, just in case strncpy fails.
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return(returncode);
}

// Setuid system calls
asmlinkage int	audit_setuid (uid_t uid)
{
	su_class *	event;
	Node *		node;
	int			returncode;
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setuid(uid);
	}
	
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setuid(uid);
	}
	event = node->location;

	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setuid(uid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setuid;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=uid;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setreuid (uid_t ruid, uid_t euid)
{
	su_class *	event;
	Node *		node;
	int			returncode;

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setreuid(ruid,euid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setreuid(ruid,euid);
	}
	event = node->location;

	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setreuid(ruid,euid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setreuid;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=euid;
	event->t_target.rid=ruid;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

// Set real, effective and saved user ID
asmlinkage int	audit_setresuid (uid_t ruid, uid_t euid, uid_t suid)
{
	su_class *	event;
	Node *		node;
	int			returncode;

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setresuid(ruid,euid,suid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setresuid(ruid,euid,suid);
	}
	event = node->location;

	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setresuid(ruid,euid,suid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setresuid;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=euid;
	event->t_target.rid=ruid;
	event->t_target.sid=suid;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setuid32 (uid_t uid)
{
	su_class *	event;
	Node *		node;
	int			returncode;

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setuid32(uid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setuid32(uid);
	}
	event = node->location;

	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setuid32(uid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setuid32;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=uid;
	event->t_target.rid=0;
	event->t_target.sid=0;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setreuid32 (uid_t ruid, uid_t euid)
{
	su_class *	event;
	Node *		node;
	int			returncode;

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setreuid32(ruid,euid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setreuid32(ruid,euid);
	}
	event = node->location;

	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setreuid32(ruid,euid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setreuid32;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=euid;
	event->t_target.rid=ruid;
	event->t_target.sid=0;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setresuid32 (uid_t ruid, uid_t euid, uid_t suid)
{
	su_class *	event;
	Node *		node;
	int			returncode;

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setresuid32(ruid,euid,suid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setresuid32(ruid,euid,suid);
	}
	event = node->location;

	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setresuid32(ruid,euid,suid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setresuid32;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=euid;
	event->t_target.rid=ruid;
	event->t_target.sid=suid;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setgid (gid_t gid)
{
	su_class *	event;
	Node *		node;
	int			returncode;
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setgid(gid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setgid(gid);
	}
	event = node->location;

	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setgid(gid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setgid;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=gid;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setregid (gid_t rgid, gid_t egid)
{
	su_class *	event;
	Node *		node;
	int			returncode;
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setregid(rgid,egid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setregid(rgid,egid);
	}
	event = node->location;
	
	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setregid(rgid,egid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setregid;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=egid;
	event->t_target.rid=rgid;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setresgid (gid_t rgid, gid_t egid, gid_t sgid)
{
	su_class *	event;
	Node *		node;
	int			returncode;

	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setresgid(rgid,egid,sgid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setresgid(rgid,egid,sgid);
	}
	event = node->location;

	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setresgid(rgid,egid,sgid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setresgid;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=egid;
	event->t_target.rid=rgid;
	event->t_target.sid=sgid;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setgid32 (gid_t gid)
{
	su_class *	event;
	Node *		node;
	int			returncode;
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setgid32(gid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setgid32(gid);
	}
	event = node->location;
	
	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setgid32(gid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setgid32;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=gid;
	event->t_target.rid=0;
	event->t_target.sid=0;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setregid32 (gid_t rgid, gid_t egid)
{
	su_class *	event;
	Node *		node;
	int			returncode;
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setregid32(rgid,egid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setregid32(rgid,egid);
	}
	event = node->location;
	
	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setregid32(rgid,egid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setregid32;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=egid;
	event->t_target.rid=rgid;
	event->t_target.sid=0;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}

asmlinkage int	audit_setresgid32 (gid_t rgid, gid_t egid, gid_t sgid)
{
	su_class *	event;
	Node *		node;
	int			returncode;
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return orig_setresgid32(rgid,egid,sgid);
	}
	if ((node = alloc_event(sizeof(su_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return orig_setresgid32(rgid,egid,sgid);
	}
	event = node->location;
	
	// Set the current id data from the pre-system call user data
	event->t_header.user_id=current->uid;
	event->t_header.group_id=current->gid;
	event->t_header.euser_id=current->euid;
	event->t_header.egroup_id=current->egid;

	returncode=orig_setresgid32(rgid,egid,sgid);
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_SU;
	event->t_header.event_id=SYS_setresgid32;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_target.id=egid;
	event->t_target.rid=rgid;
	event->t_target.sid=sgid;

	// Everything else can be handled by the generic event handler.
	audit_event(node,1);

	// Now continue on with the normal exit.
	return(returncode);
}


// audit file truncates.
asmlinkage int	audit_truncate64 (const char *path, off_t length)
{
	io_class *	event;
	Node *		node;
	int			returncode;
	
	returncode=orig_truncate64(path,length);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}

	if ((node = alloc_event(sizeof(io_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return returncode;
	}
	event = node->location;
	event->t_return.returncode=returncode;
	
	event->t_header.event_class=AUDIT_CLASS_IO;
	event->t_header.event_id=SYS_truncate64;
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	// Store the path
	// zero the path, just in case strncpy fails.
	event->t_path.path[0]='\0';
	if(path) {
		strncpy_from_user(event->t_path.path,path,MAX_PATH);
	}
	
	// Save off the offset.
	event->t_attributes.createmode=(unsigned long)length;

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	// Now continue on with the normal exit.
	return(returncode);
}

// Change this so we have an idea of what sort of socket is being opened.
// NOTE: report in the format 10.0.0.2:1234 (so match can look at ip and/or port!)
//
asmlinkage int audit_socketcall(int call, unsigned long *args)
{
	nt_class *	event;
	Node *		node;
	int			returncode;
	
	returncode=orig_socketcall(call, args);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct || (call != SYS_CONNECT && call != SYS_ACCEPT))
	{
		return(returncode);
	}
	
	if ((node = alloc_event(sizeof(nt_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return(returncode);
	}
	event = node->location;
	event->t_return.returncode=returncode;

	event->t_header.event_class=AUDIT_CLASS_NET;
	event->t_header.event_id=SYS_socketcall;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	if(call == SYS_CONNECT) {
		struct sockaddr *auditsockaddr;
		struct sockaddr_in *audit_in;
				
		auditsockaddr=(struct sockaddr *)args[1];

		event->syscall=SYS_CONNECT;
		
		if(auditsockaddr->sa_family == AF_INET) {
			char *p;
			audit_in=(struct sockaddr_in *)auditsockaddr;
			p=(char*)&audit_in->sin_addr.s_addr;
			
			// Work out whether the ip address (including dots) would fit in our buffer
			if((unsigned char)*(p+0) <=255 && (unsigned char)*(p+1) <=255 &&  (unsigned char)*(p+2) <= 255 && (unsigned char)*(p+3) <= 255) {
				snprintf(event->t_connection.dst_ip,16,"%d.%d.%d.%d",(unsigned char)*(p+0),(unsigned char)*(p+1),(unsigned char)*(p+2),(unsigned char)*(p+3));
			} else {
				snprintf(event->t_connection.dst_ip,16,"Unknown");
			}
			strncpy(event->t_connection.src_ip,"127.0.0.1",16);
			event->t_connection.dst_port=ntohs(audit_in->sin_port);
			event->t_connection.src_port=0; // Unknown
			
		} else {
			// Not inet? Not interested.
			// Note though: we may need to play with IPv6 here.
			return(returncode);
		}			
	} else if(call == SYS_ACCEPT) {
		struct sockaddr *auditsockaddr;
		struct sockaddr_in *audit_in;
		
		auditsockaddr=(struct sockaddr *)args[1];
			
		event->syscall=SYS_ACCEPT;
		if(auditsockaddr->sa_family == AF_INET) {
			char *p;
			audit_in=(struct sockaddr_in *)auditsockaddr;
			p=(char*)&audit_in->sin_addr.s_addr;

			// Work out whether the ip address (including dots) would fit in our buffer
			if((unsigned char)*(p+0) <=255 && (unsigned char)*(p+1) <=255 &&  (unsigned char)*(p+2) <= 255 && (unsigned char)*(p+3) <= 255) {
				snprintf(event->t_connection.src_ip,16,"%d.%d.%d.%d",(unsigned char)*(p+0),(unsigned char)*(p+1),(unsigned char)*(p+2),(unsigned char)*(p+3));
			} else {
				snprintf(event->t_connection.src_ip,16,"Unknown");
			}
			event->t_connection.src_port=ntohs(audit_in->sin_port);
			
			strncpy(event->t_connection.dst_ip,"127.0.0.1",16);
			event->t_connection.dst_port=0; // Unknown. I really would like to know though, since this is the port on the local machine that they are connecting to!
		} else {
			// Not inet? Not interested.
			// Note: We may need to cope with IPv6 here.
			return(returncode);
		}			

	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Change this so we can grab the name. Will use a different class.
asmlinkage caddr_t audit_create_module(const char *name, size_t size)
{
	ad_class *	event;
	caddr_t		returncode;
	Node *		node;
		
	returncode=orig_create_module(name,size);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}
		
	if ((node = alloc_event(sizeof(ad_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return(returncode);
	}
	event = node->location;
	event->t_return.returncode=(int)returncode;

	event->t_header.event_class=AUDIT_CLASS_AD;
	event->t_header.event_id=SYS_create_module;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_name.path[0]='\0';
	if(name) {
		strncpy_from_user(event->t_name.path,name,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Change this so we can grab the name. Will use a different class.
asmlinkage int audit_mount(char *dev_name, char *dir_name, char *type, unsigned long flags, void *data)
{
	cp_class *	event;
	int		returncode;
	Node *		node;
		
	returncode=orig_mount(dev_name,dir_name,type,flags,data);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}
		
	if ((node = alloc_event(sizeof(cp_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return(returncode);
	}
	event = node->location;
	event->t_return.returncode=(int)returncode;

	event->t_header.event_class=AUDIT_CLASS_CP;
	event->t_header.event_id=SYS_mount;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_sourcepath.path[0]='\0';
	if(dev_name) {
		strncpy_from_user(event->t_sourcepath.path,dev_name,MAX_PATH);
	}
		
	event->t_destpath.path[0]='\0';
	if(dir_name) {
		strncpy_from_user(event->t_destpath.path,dir_name,MAX_PATH);
	}

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}

// Change this so we can grab the name. Will use a different class.
asmlinkage int audit_umount(char *name, int flags)
{
	cp_class *	event;
	int		returncode;
	Node *		node;
		
	returncode=orig_umount(name,flags);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}
		
	if ((node = alloc_event(sizeof(cp_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return(returncode);
	}
	event = node->location;
	event->t_return.returncode=(int)returncode;

	event->t_header.event_class=AUDIT_CLASS_CP;
	event->t_header.event_id=SYS_umount;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_sourcepath.path[0]='\0';
	if(name) {
		strncpy_from_user(event->t_sourcepath.path,name,MAX_PATH);
	}

	event->t_destpath.path[0]='\0';

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}


// Change this so we can grab the name. Will use a different class.
asmlinkage int audit_umount2(char *name, int flags)
{
	cp_class *	event;
	int		returncode;
	Node *		node;
		
	returncode=orig_umount2(name,flags);
	
	// If no daemon, or the daemon has generated the event.
	if((!AUDIT_IS_RUNNING) || current == auditdaemon_task_struct)
	{
		return(returncode);
	}
		
	if ((node = alloc_event(sizeof(cp_class))) == NULL) {
		down(&audit_lock); lost_events++; up(&audit_lock);
		return(returncode);
	}
	event = node->location;
	event->t_return.returncode=(int)returncode;

	event->t_header.event_class=AUDIT_CLASS_CP;
	event->t_header.event_id=SYS_umount2;

	// Process data
	event->t_process.pid=current->pid;
	event->t_process.name[0]='\0';
	strncpy(event->t_process.name,current->comm,MAXCOMMAND);

	event->t_sourcepath.path[0]='\0';
	if(name) {
		strncpy_from_user(event->t_sourcepath.path,name,MAX_PATH);
	}

	event->t_destpath.path[0]='\0';

	// Everything else can be handled by the generic event handler.
	audit_event(node,0);

	return(returncode);
}


// Change this so we can grab the name. Will use a different class.
// NOTE: Have removed the capability to audit delete_module, since auditmodule will segfault if it tries to remove itself!
/*asmlinkage caddr_t audit_delete_module(const char *name)
{

}*/

