/************************************************************
 *
 * 1999 - 2004 (c) Intersect Alliance Pty Ltd.
 *
 * These routines read the audit config file, and populates
 * the audit configuration window, when called.
 *
 *
 * SNARE GUI Version 2.4
 *
 ************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <gnome.h>
#include <regex.h>		// Regular expression matches.

#include "read_config.h"
#include "support.h"
#include "write_config.h"


gint read_configuration(GtkWidget * config_window)
{
	GtkWidget *audittype_button, *logging_file, *objectives_clist;
	GtkWidget *hostid_entry, *desthost_entry, *port_entry,
	    *network_radio, *log_radio;
	GtkWidget *network_and_log_radio, *access_key, *restrict_ip,
	    *web_port;
	GtkWidget *remote_control_allow;

	FILE *configfile;
	int headertype = 0, criticality, returncode;
	int audittype = AUDIT_BY_EVENT;
	int eventnumber, user_match_result, filter_match_result;
	int excludeflag = 0, audit_to_file = FALSE, audit_to_network =
	    FALSE;
	gint current_row;

	char event[MAX_AUDIT_CONFIG_LINE], user[MAX_AUDIT_CONFIG_LINE],
	    path[MAX_AUDIT_CONFIG_LINE];
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];
	char hostid[MAX_HOSTID] =
	    "", *pos, desthost[MAX_HOSTID], port[MAX_HOSTID];

	gchar alert[TOKEN_SIZE] = "NULL", succ[TOKEN_SIZE] =
	    "NULL", objective[TOKEN_SIZE] = "NULL";
	gchar user_unreg[TOKEN_SIZE] = "NULL", filter_match[TOKEN_SIZE] =
	    "NULL";
	gchar user_match[MAX_AUDIT_CONFIG_LINE] =
	    "NULL", filter[MAX_AUDIT_CONFIG_LINE] = "NULL";
	gchar *line[7] =
	    { alert, succ, user_match, user_unreg, filter_match, filter,
       objective };
	gchar *filename, *network, *accesskey, *allow_remote, *restrictip,
	    *webport;
	GtkTreeIter iter;
	GtkListStore *list_store;


	objectives_clist =
	    lookup_widget(config_window, "objectives_clist");
	//list_store = gtk_tree_view_get_model(GTK_LIST_STORE(objectives_clist));
	list_store =
	    GTK_LIST_STORE(gtk_tree_view_get_model
			   (GTK_TREE_VIEW(objectives_clist)));


	configfile = fopen(CONFIG_FILENAME, "r");

	if (configfile == (FILE *) NULL)
		return FALSE;

	while (fgets(inputbuffer, MAX_AUDIT_CONFIG_LINE, configfile)) {
		trim(inputbuffer);
		if (!iscomment(inputbuffer)) {
			/* A HEADER IS SOMETHING LIKE: [Event] */
			if (isheader(inputbuffer)) {
				headertype = getheader(inputbuffer);
			} else if (headertype == CONFIG_OBJECTIVES) {
				excludeflag = 0;
				if ((criticality =
				     splitobjective(inputbuffer, event,
						    user, path,
						    &returncode,
						    &excludeflag)) > -1) {

					trimallwhitespace(event);

					if (g_strcasecmp(event, OPEN_FILE)
					    == 0)
						strcpy(objective,
						       "Read, write or create a file or directory");
					else if (g_strcasecmp
						 (event,
						  OPEN_READ_FILE) == 0)
						strcpy(objective,
						       "Open a file for reading only");
					else if (g_strcasecmp
						 (event,
						  REMOVE_FILE) == 0)
						strcpy(objective,
						       "Remove a file or directory");
					else if (g_strcasecmp
						 (event,
						  LOGON_LOGOFF) == 0)
						strcpy(objective,
						       "User logon or logoff");
					else if (g_strcasecmp
						 (event, ATTRIBUTES) == 0)
						strcpy(objective,
						       "Modify system, file or directory attributes");
					else if (g_strcasecmp
						 (event, PROG_EXEC) == 0)
						strcpy(objective,
						       "Start or stop a program execution");
					else if (g_strcasecmp
						 (event, USER_IDENT) == 0)
						strcpy(objective,
						       "Change user or group identity");
					else if (g_strcasecmp
						 (event,
						  ESTABLISH_SOCKET) == 0)
						strcpy(objective,
						       "Establish an outgoing network connection");
					else
						strcpy(objective, event);


					if (criticality ==
					    CRITICALITY_CRITICAL) {
						strcpy(alert, "Critical");
					} else if (criticality ==
						   CRITICALITY_PRIORITY) {
						strcpy(alert, "Priority");
					} else if (criticality ==
						   CRITICALITY_WARNING) {
						strcpy(alert, "Warning");
					} else if (criticality ==
						   CRITICALITY_INFO) {
						strcpy(alert,
						       "Information");
					} else {
						strcpy(alert, "Clear");
					}

					if (returncode ==
					    RETURNCODE_FAILURE) {
						strcpy(succ, "Failure");
					} else if (returncode ==
						   RETURNCODE_SUCCESS) {
						strcpy(succ, "Success");
					} else if (returncode ==
						   RETURNCODE_ANY) {
						strcpy(succ,
						       "Success & Failure");
					}

					user_match_result =
					    GetMatchUser(user, user_unreg);


					if (excludeflag == 1) {
						strcpy(user_match,
						       "Exclude");
					} else if (user_match_result == MATCH_ALL) {
						strcpy(user_match, "All");
					} else if (user_match_result ==
						   MATCH_SELECTED) {
						strcpy(user_match,
						       "Selected");
					}

					filter_match_result =
					    GetMatch(path, filter);

					if (filter_match_result ==
					    MATCH_PARTIAL) {
						strcpy(filter_match,
						       "Partial");
					} else if (filter_match_result ==
						   MATCH_EXACT) {
						strcpy(filter_match,
						       "Full");
					} else if (filter_match_result ==
						   MATCH_REGEXP) {
						strcpy(filter_match,
						       "Regular Exp");
					}

					gtk_list_store_append(list_store,
							      &iter);
					gtk_list_store_set(list_store,
							   &iter, 0, alert,
							   1, succ, 2,
							   user_match, 3,
							   user_unreg, 4,
							   filter_match, 5,
							   filter, 6,
							   objective, -1);
				}
			} else if (headertype == CONFIG_OUTPUT) {
				if (isfilename(inputbuffer)) {
					filename = inputbuffer;
					filename += strlen("file=");
					logging_file =
					    lookup_widget(config_window,
							  "log_file");
					gtk_entry_set_text(GTK_ENTRY
							   (logging_file),
							   filename);
					audit_to_file = TRUE;
				} else if (isnetwork(inputbuffer)) {
					network = inputbuffer;
					network += strlen("network=");
					pos = strstr(network, ":");
					if (pos) {
						*pos='\0';
						strncpy(desthost, network,
							MAX_HOSTID);
						pos++;
						strncpy(port, pos, 7);
					} else {
						strncpy(desthost, network,
							MAX_HOSTID);
					}
					desthost_entry =
					    lookup_widget(config_window,
							  "desthost_entry");
					port_entry =
					    lookup_widget(config_window,
							  "port_entry");
					gtk_entry_set_text(GTK_ENTRY
							   (desthost_entry),
							   desthost);
					gtk_entry_set_text(GTK_ENTRY
							   (port_entry),
							   port);
					audit_to_network = TRUE;
				}
			} else if (headertype == CONFIG_REMOTE) {
				if (isallowremote(inputbuffer)) {
					allow_remote = inputbuffer;
					allow_remote += strlen("allow=");
					remote_control_allow =
					    lookup_widget(config_window,
							  "remote_control_check");
					if (strstr(allow_remote, "1"))
						gtk_toggle_button_set_active
						    (GTK_TOGGLE_BUTTON
						     (remote_control_allow),
						     TRUE);
					else
						gtk_toggle_button_set_active
						    (GTK_TOGGLE_BUTTON
						     (remote_control_allow),
						     FALSE);

				} else if (isaccesskey(inputbuffer)) {
					accesskey = inputbuffer;
					accesskey += strlen("accesskey=");
					access_key =
					    lookup_widget(config_window,
							  "access_entry");
					gtk_entry_set_text(GTK_ENTRY
							   (access_key),
							   accesskey);
					gtk_toggle_button_set_active
					    (GTK_TOGGLE_BUTTON
					     (lookup_widget
					      (config_window,
					       "access_check")), TRUE);
				} else if (isrestrictip(inputbuffer)) {
					restrictip = inputbuffer;
					restrictip +=
					    strlen("restrict_ip=");
					restrict_ip =
					    lookup_widget(config_window,
							  "ip_entry");
					gtk_entry_set_text(GTK_ENTRY
							   (restrict_ip),
							   restrictip);
					gtk_toggle_button_set_active
					    (GTK_TOGGLE_BUTTON
					     (lookup_widget
					      (config_window, "ip_check")),
					     TRUE);
				} else if (iswebport(inputbuffer)) {
					webport = inputbuffer;
					webport += strlen("listen_port=");
					web_port =
					    lookup_widget(config_window,
							  "web_entry");
					gtk_entry_set_text(GTK_ENTRY
							   (web_port),
							   webport);
					gtk_toggle_button_set_active
					    (GTK_TOGGLE_BUTTON
					     (lookup_widget
					      (config_window,
					       "web_check")), TRUE);
				}
			} else if (headertype == CONFIG_HOSTID) {
				gethostident(inputbuffer, hostid,
					     MAX_HOSTID);
				hostid_entry =
				    lookup_widget(config_window,
						  "hostid_entry");
				gtk_entry_set_text(GTK_ENTRY(hostid_entry),
						   hostid);
			}

		}
	}

	network_radio = lookup_widget(config_window, "network_radio");
	log_radio = lookup_widget(config_window, "log_radio");
	network_and_log_radio =
	    lookup_widget(config_window, "network_and_log_radio");
	if ((audit_to_file == TRUE) && (audit_to_network == TRUE))
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
					     (network_and_log_radio),
					     TRUE);
	else if (audit_to_file == TRUE)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(log_radio),
					     TRUE);
	else
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
					     (network_radio), TRUE);

	fclose(configfile);
	return TRUE;
}


// *********************************   
// THE SUPPORTING ROUTINES ARE BELOW
// *********************************

// Remove start / end whitespace, including the newline
void trim(char *string)
{
	char *pointer;
	char *pointer2;

	// Verify that there is something to check.
	if (string == (char *) NULL)
		return;

	// And that there is some data within.
	if (strlen(string) == 0)
		return;

	// Start from the end, work backwards.
	pointer = &string[strlen(string) - 1];

	while ((*pointer == ' ' || *pointer == 9 || *pointer == 10
		|| *pointer == 12) && pointer >= string) {
		*pointer = '\0';
		pointer--;
	}

	// Are we back at the start of the string? If so, this line must be null.
	if (pointer == string)
		return;
	// Pointer is now at the last non-whitespace character of the string.

	pointer2 = string;
	while ((*pointer2 == ' ' || *pointer2 == 9 || *pointer2 == 10
		|| *pointer2 == 12) && pointer2 < pointer) {
		pointer2++;
	}
	// pointer2 will now point to the start of the first non-null character
	// Copy the truncated string back to the original.
	strcpy(string, pointer2);
}

void trimallwhitespace(char *string)
{
	char *readpointer;
	char *writepointer;

	// Verify that there is something to check.
	if (string == (char *) NULL)
		return;

	// And that there is some data within.
	if (strlen(string) == 0)
		return;

	// Start from the end, work backwards.
	readpointer = string;
	writepointer = string;
	while (readpointer < (string + strlen(string))) {
		if (*readpointer == ' ' || *readpointer == 9
		    || *readpointer == 10 || *readpointer == 12) {
			readpointer++;
		} else {
			if (writepointer != readpointer) {
				*writepointer = *readpointer;
			}
			readpointer++;
			writepointer++;
		}

	}
}

// Take an identified objective line
// NOTE: string, event, user and match must all be the same buffer size!
int splitobjective(char *string, char *event, char *user, char *match,
		   int *returncode, int *excludeflag)
{
	char *startevent, *startuser, *startmatch, *startcrit,
	    *startreturn;
	char *stringpointer, *endstring;
	char *eventp, *userp, *matchp, *critp;
	char criticality[MAX_AUDIT_CONFIG_LINE];
	int crit = 0;

	*returncode = RETURNCODE_ANY;

	// Do some basic sanity checks.
	if (string == (char *) NULL || event == (char *) NULL
	    || user == (char *) NULL || match == (char *) NULL) {
		return (-1);
	}

	startcrit = strstr(string, "criticality=");
	startevent = strstr(string, "event=");
	startreturn = strstr(string, "return=");
	startuser = strstr(string, "user=");
	if (startuser == (char *) NULL) {
		startuser = strstr(string, "user!=");
		if (startuser != (char *) NULL) {
			// EXCLUDE USERS rather than include.
			*excludeflag = 1;
		}
	}
	startmatch = strstr(string, "match=");

	// string pointers for iteration.
	stringpointer = string;
	eventp = event;
	userp = user;
	matchp = match;
	critp = criticality;

	// Pointer to the last character in the string.
	endstring = &string[strlen(string)];

	if (startevent == (char *) NULL || startuser == (char *) NULL ||
	    startmatch == (char *) NULL || startcrit == (char *) NULL
	    || startreturn == (char *) NULL) {
		// Problem, this line is malformed. We really cannot proceed with this line.
		// fprintf(stderr,"The following line does not contain the criticality, event, return, user and match elements: %s",string);
		return (-1);
	}
	// Start with the event
	stringpointer = startcrit + strlen("criticality=");
	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startuser
		   && stringpointer != startmatch
		   && stringpointer != endstring)) {
		*critp = *stringpointer;
		critp++;
		stringpointer++;
	}
	*critp = '\0';
	trim(criticality);
	crit = atoi(criticality);
	if (crit < CRITICALITY_CLEAR)
		crit = CRITICALITY_CLEAR;
	if (crit > CRITICALITY_CRITICAL)
		crit = CRITICALITY_CRITICAL;

	// Start with the event
	stringpointer = startevent + strlen("event=");
	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startuser
		   && stringpointer != startmatch
		   && stringpointer != endstring)) {
		*eventp = *stringpointer;
		eventp++;
		stringpointer++;
	}
	*eventp = '\0';
	// Remove extra whitespace at start and end.
	trim(event);

	stringpointer = startreturn + strlen("return=");
	if (regmatchi(stringpointer, "^success")) {
		*returncode = RETURNCODE_SUCCESS;
	} else if (regmatchi(stringpointer, "^failure")) {
		*returncode = RETURNCODE_FAILURE;
	} else {
		// Any thing else is either success or failure.
		*returncode = RETURNCODE_ANY;
	}

	if (*excludeflag) {
		stringpointer = startuser + strlen("user!=");
	} else {
		stringpointer = startuser + strlen("user=");
	}

	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startmatch
		   && stringpointer != startevent
		   && stringpointer != endstring)) {
		*userp = *stringpointer;
		stringpointer++;
		userp++;
	}
	*userp = '\0';
	// Remove extra whitespace at start and end.
	trim(user);

	stringpointer = startmatch + strlen("match=");
	while (*stringpointer
	       && (stringpointer != startcrit
		   && stringpointer != startreturn
		   && stringpointer != startevent
		   && stringpointer != startuser
		   && stringpointer != endstring)) {
		*matchp = *stringpointer;
		stringpointer++;
		matchp++;
	}
	*matchp = '\0';
	// Remove extra whitespace at start and end.
	trim(match);

	// Return the criticality value.
	return (crit);
}


int getaudittype(char *string)
{
	char *stringp = string;
	// Get rid of the type= part.
	stringp += strlen("type=");
	if (regmatchi(stringp, "^objective$")) {
		return (AUDIT_BY_OBJECTIVE);
	} else {
		return (AUDIT_BY_EVENT);
	}
}


int getheader(char *string)
{
	char *stringp = string;

	// Remove the first and last bracket.
	stringp++;
	stringp[strlen(stringp) - 1] = '\0';

	if (regmatchi(stringp, "^objectives$")) {
		return (CONFIG_OBJECTIVES);
	} else if (regmatchi(stringp, "^file$")
		   || regmatchi(stringp, "^output$")) {
		return (CONFIG_OUTPUT);
	} else if (regmatchi(stringp, "^hostid$")) {
		return (CONFIG_HOSTID);
	} else if (regmatchi(stringp, "^remote$")) {
		return (CONFIG_REMOTE);
	}
	//fprintf(stderr,"Unknown header in configuration file: %s\n",stringp);
	return (0);
}

int isaudittype(char *string)
{
	if (regmatch(string, "^type=")) {
		return (1);
	}
	return (0);
}

int isfilename(char *string)
{
	if (regmatch(string, "^file=")) {
		return (1);
	} else {
		return (0);
	}
}


int isheader(char *string)
{
	if (string[0] == '[' && string[strlen(string) - 1] == ']') {
		return (1);
	}
	return (0);
}

int iscomment(char *line)
{
	// Verify that there is something to check.
	if (line == (char *) NULL)
		return (1);

	// And that there is some data within.
	if (strlen(line) == 0 || strlen(line) > MAX_AUDIT_CONFIG_LINE)
		return (1);

	// If the first non-whitespace character is a hash, this is a comment line.
	while (*line) {
		// Space or tab or newline / formfeed
		if (*line == ' ' || *line == 9 || *line == 10
		    || *line == 12) {
			line++;
		} else if (*line == '#') {
			return (1);
		} else {
			// Ahh. A non-whitespace, non hash character.
			return (0);
		}
	}

	// If we are here, then the whole line must have been whitespace
	return (1);
}


// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
int regmatch(const char *string, const char *pattern)
{
	int status;
	regex_t re;

	if (regcomp(&re, pattern, REG_EXTENDED | REG_NOSUB) != 0) {
		return (0);
	}
	status = regexec(&re, string, (size_t) 0, NULL, 0);
	regfree(&re);
	if (status != 0) {
		return (0);
	}
	return (1);
}

// Match string against an extended regular expression. Case insensitive.
// Return 1 for match, 0 for no-match or error.
int regmatchi(const char *string, const char *pattern)
{
	int status;
	regex_t re;

	if (regcomp(&re, pattern, REG_EXTENDED | REG_NOSUB | REG_ICASE) !=
	    0) {
		return (0);
	}
	status = regexec(&re, string, (size_t) 0, NULL, 0);
	regfree(&re);
	if (status != 0) {
		return (0);
	}
	return (1);
}

int GetMatch(const char *string, char *buffer)
{
	int match = 0;
	int length;
	char *stringp;

	stringp = (char *) string;
	length = strlen(string);

	// Regular expression characters are:   .[]$()|*+?{}

	// If the string has a ^ at start, and $ at end, and there are no other unescaped regular expression characters
	// Then it is probably a exact match.

	if (stringp[0] == '^' && stringp[length - 1] == '$') {
		match = 1;
		while (*stringp) {
			if ((*stringp == '.' || *stringp == '['
			     || *stringp == ']' || *stringp == '$'
			     || *stringp == '(' || *stringp == ')'
			     || *stringp == '|' || *stringp == '*'
			     || *stringp == '+' || *stringp == '?'
			     || *stringp == '{' || *stringp == '}')
			    && stringp < &string[length - 1]) {
				if (stringp != string) {
					if (*(stringp - 1) != '\\') {
						match = 0;
					}
				} else {
					match = 0;
				}
			}
			stringp++;
		}
		if (match) {
			// Trim the first and last characters, cut \ characters (except \\) and return.
			// strncpy(buffer,string+1,length - 2);
			stringp = ((char *) string) + 1;
			while (stringp < string + (length - 1)) {
				if (*stringp == '\\') {
					if (stringp != string) {
						if (*(stringp - 1) == '\\') {
							*buffer = *stringp;
							buffer++;
						}
					}
				} else {
					*buffer = *stringp;
					buffer++;
				}
				stringp++;
			}

			*buffer = '\0';
			return (MATCH_EXACT);
		}
	}

	stringp = (char *) string;
	match = 0;
	if (stringp[0] == '.' && stringp[1] == '*'
	    && stringp[length - 2] == '.' && stringp[length - 1] == '*'
	    && length > 4) {
		stringp += 2;
		// First and last characters are .*
		// probably a partial match.
		match = 1;
		while (*stringp) {
			if ((*stringp == '.' || *stringp == '['
			     || *stringp == ']' || *stringp == '$'
			     || *stringp == '(' || *stringp == ')'
			     || *stringp == '|' || *stringp == '*'
			     || *stringp == '+' || *stringp == '?'
			     || *stringp == '{' || *stringp == '}')
			    && stringp > &string[1]
			    && stringp < &string[length - 2]) {
				if (stringp != string) {
					if (*(stringp - 1) != '\\') {
						match = 0;
					}
				} else {
					match = 0;
				}
			}
			stringp++;
		}

		if (match) {
			// Trim the first and last characters, and return.
			//strncpy(buffer,string+1,length - 2);
			//buffer[length-2]='\0';

			// Trim the first and last characters, cut \ characters (except \\) and return.
			// strncpy(buffer,string+1,length - 2);
			stringp = (char *) string + 2;
			while (stringp < string + (length - 2)) {
				if (*stringp == '\\') {
					if (stringp != string) {
						if (*(stringp - 1) == '\\') {
							*buffer = *stringp;
							buffer++;
						}
					}
				} else {
					*buffer = *stringp;
					buffer++;
				}
				stringp++;
			}
			*buffer = '\0';
			return (MATCH_PARTIAL);
		}
	}
	// Return the regular expression as is.
	stringp = (char *) string;
	while (stringp < string + length) {
		*buffer = *stringp;
		buffer++;
		stringp++;
	}

	*buffer = '\0';
	return (MATCH_REGEXP);
}

// This is a different routine, because what is considered a regexp in MATCH may be SELECTED in user.
int GetMatchUser(const char *string, char *buffer)
{
	int match = 0;
	int length;
	char *stringp;

	stringp = (char *) string;
	length = strlen(string);

	// Regular expression characters are:   .[]$()|*+?{}

	// If the string is '.*' then it's ALL
	if (!strcmp(string, ".*")) {
		strcpy(buffer, "");
		return (MATCH_ALL);
	}
	// If the string has a ^( or ^ at start, and )$ or $ at end, and there are no other unescaped regular expression characters
	// Then it is probably a selected match.

	if (stringp[0] == '^' && stringp[length - 1] == '$') {
		match = 1;
		while (*stringp) {
			if ((*stringp == '.' || *stringp == '['
			     || *stringp == ']' || *stringp == '$'
			     || (*stringp == '('
				 && (stringp != &string[1]))
			     || (*stringp == ')'
				 && (stringp != &string[length - 2]))
			     || *stringp == '*' || *stringp == '+'
			     || *stringp == '?' || *stringp == '{'
			     || *stringp == '}')
			    && stringp < &string[length - 1]) {
				if (stringp != string) {
					if (*(stringp - 1) != '\\') {
						match = 0;
					}
				} else {
					match = 0;
				}
			}
			stringp++;
		}
		if (match) {
			// Trim the first and last characters, cut \ characters (except \\) and return.
			// strncpy(buffer,string+1,length - 2);
			stringp = ((char *) string) + 1;
			if (string[1] == '(') {
				stringp = ((char *) string) + 2;
			}
			while (stringp < string + (length - 2)) {
				if (*stringp == '\\') {
					if (stringp != string) {
						if (*(stringp - 1) == '\\') {
							*buffer = *stringp;
							buffer++;
						}
					}
				} else {
					if (*stringp == '|') {
						*buffer = ',';
					} else {
						*buffer = *stringp;
					}
					buffer++;
				}
				stringp++;
			}

			if (*stringp != ')') {
				*buffer = *stringp;
				buffer++;
			}

			*buffer = '\0';
			return (MATCH_SELECTED);
		}
	}
	// Return the regular expression as is.
	// Cut \ characters (except \\) and return.
	stringp = (char *) string;
	while (stringp < string + length) {
		*buffer = *stringp;
		buffer++;
		stringp++;
	}

	*buffer = '\0';
	return (MATCH_REGEXP);
}

// Return the host identifier
char *gethostident(char *string, char *host, int length)
{
	char *stringp = string;

	stringp = strstr(string, "=");
	if (stringp != (char *) NULL) {
		stringp++;
		strncpy(host, stringp, length);
	} else {
		return ((char *) NULL);
	}
	return ((char *) host);
}

// Network output
int isnetwork(char *string)
{
	if (regmatch(string, "^network=")) {
		return (1);
	} else {
		return (0);
	}
}

// "Web Port" output
int iswebport(char *string)
{
	if (regmatch(string, "^listen_port=")) {
		return (1);
	} else {
		return (0);
	}
}

// "Retrict IP" output
int isrestrictip(char *string)
{
	if (regmatch(string, "^restrict_ip=")) {
		return (1);
	} else {
		return (0);
	}
}

// "Access Key" output
int isaccesskey(char *string)
{
	if (regmatch(string, "^accesskey=")) {
		return (1);
	} else {
		return (0);
	}
}

// "Allow Remote Control" output
int isallowremote(char *string)
{
	if (regmatch(string, "^allow=")) {
		return (1);
	} else {
		return (0);
	}
}
