/************************************************************
 *
 * 1999 - 2004 (c) Intersect Alliance Pty Ltd.
 *
 * These routines write the audit configuration to the 
 * audit.conf file
 *
 *
 * SNARE GUI Version 2.4 Dated 1 March 2004
 *
 ************************************************************/


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <gnome.h>
#include <regex.h>		// Regular expression matches.
#include <crypt.h>

#include "write_config.h"
#include "read_config.h"
#include "support.h"
#include "read_events.h"

gint write_configuration(GtkWidget * config_window)
{
	GtkWidget *objectives_clist, *error_dialog;
	FILE *configfile;

	objectives_clist =
	    lookup_widget(config_window, "objectives_clist");

	configfile = fopen(CONFIG_FILENAME, "w");

	// If it can't open the file for writing, display an error window
	if (configfile == (FILE *) NULL) {
		error_dialog =
		    gnome_message_box_new
		    ("Unable to open /etc/security/snare.conf for writing. \nCheck the whether the file exists, and has \nthe allowable permissions.",
		     GNOME_MESSAGE_BOX_ERROR, GNOME_STOCK_BUTTON_OK, NULL);
		gnome_dialog_run(GNOME_DIALOG(error_dialog));
		return FALSE;
	}


	write_header(config_window, configfile);
	write_log_or_network_parameters(config_window, configfile);
	write_remote_control_parameters(config_window, configfile);
	write_objectives(config_window, configfile);

	fclose(configfile);
	return TRUE;
}


gint write_header(GtkWidget * config_window, FILE * configfile)
{

	fprintf(configfile,
		"# WARNING: MANUAL EDITING OF THIS FILE IS NOT RECOMMENDED\n");

	return TRUE;
}

gint write_log_or_network_parameters(GtkWidget * config_window,
				     FILE * configfile)
{
	GtkWidget *logging_file_text_box, *hostid_entry, *desthost_entry,
	    *port_entry, *network_radio;
	GtkWidget *log_radio;
	G_CONST_RETURN gchar *logging_file_entry, *hostid, *desthost,
	    *port;

	network_radio = lookup_widget(config_window, "network_radio");
	log_radio = lookup_widget(config_window, "log_radio");

	hostid_entry = lookup_widget(config_window, "hostid_entry");
	hostid = gtk_entry_get_text(GTK_ENTRY(hostid_entry));
	fprintf(configfile, "\n[HostID]\n");
	fprintf(configfile, "\tname=%s\n", hostid);
	desthost_entry = lookup_widget(config_window, "desthost_entry");
	port_entry = lookup_widget(config_window, "port_entry");
	desthost = gtk_entry_get_text(GTK_ENTRY(desthost_entry));
	port = gtk_entry_get_text(GTK_ENTRY(port_entry));
	logging_file_text_box = lookup_widget(config_window, "log_file");
	logging_file_entry =
	    gtk_entry_get_text(GTK_ENTRY(logging_file_text_box));

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(network_radio))) {

		fprintf(configfile, "\n[Output]\n");
		fprintf(configfile, "\tnetwork=%s:%s\n", desthost, port);
		set_network_logging_variable(AUDIT_TO_NETWORK);
	} else
	    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(log_radio)))
	{
		fprintf(configfile, "\n[Output]\n");
		fprintf(configfile, "\tfile=%s\n", logging_file_entry);

		declare_new_file((char *) logging_file_entry);
		set_network_logging_variable(AUDIT_BY_OBJECTIVE);

	} else {
		fprintf(configfile, "\n[Output]\n");
		fprintf(configfile, "\tnetwork=%s:%s\n", desthost, port);
		fprintf(configfile, "\tfile=%s\n", logging_file_entry);

		declare_new_file((char *) logging_file_entry);

		// When sending audit to the network and a file, display the "objective" pixmap
		set_network_logging_variable(AUDIT_BY_OBJECTIVE);

	}

	return TRUE;
}

gint write_remote_control_parameters(GtkWidget * config_window,
				     FILE * configfile)
{
	GtkWidget *remote_control_check, *ip_check, *ip_entry,
	    *access_check, *access_entry;
	GtkWidget *web_check, *web_entry;
	G_CONST_RETURN gchar *ip_address, *access, *web_port;

	remote_control_check =
	    lookup_widget(config_window, "remote_control_check");
	ip_check = lookup_widget(config_window, "ip_check");
	ip_entry = lookup_widget(config_window, "ip_entry");
	ip_address = gtk_entry_get_text(GTK_ENTRY(ip_entry));
	access_check = lookup_widget(config_window, "access_check");
	access_entry = lookup_widget(config_window, "access_entry");
	access = gtk_entry_get_text(GTK_ENTRY(access_entry));
	web_check = lookup_widget(config_window, "web_check");
	web_entry = lookup_widget(config_window, "web_entry");
	web_port = gtk_entry_get_text(GTK_ENTRY(web_entry));

	fprintf(configfile, "\n[Remote]\n");
	if (gtk_toggle_button_get_active
	    (GTK_TOGGLE_BUTTON(remote_control_check)))
		fprintf(configfile, "\tallow=1\n");
	else
		fprintf(configfile, "\tallow=0\n");

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ip_check))) {
		fprintf(configfile, "\trestrict_ip=%s\n", ip_address);
	}
	// If the password is $1$.. for solaris or $1$Snare$.. for linux, then
	// it's already md5'd. pass it through without encryption.
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(access_check))) {
#ifdef __linux__
		if (access[0] == '$' && access[1] == '1'
		    && access[2] == '$' && access[3] == 'S'
		    && access[4] == 'n' && access[5] == 'a'
		    && access[6] == 'r' && access[7] == 'e'
		    && access[8] == '$') {
			fprintf(configfile, "\taccesskey=%s\n", access);
		} else {
			fprintf(configfile, "\taccesskey=%s\n",
				crypt(access, "$1$Snare"));
		}
#else				// solaris
		if (access[0] == '$' && access[1] == '1'
		    && access[2] == '$') {
			fprintf(configfile, "\taccesskey=%s\n", access);
		} else {
			char *tcrypt;
			tcrypt = crypt(access, "$1");
			if (tcrypt) {
				fprintf(configfile, "\taccesskey=%s\n",
					tcrypt);
			} else {
				fprintf(stderr,
					"MD5 encryption failed!\n");
			}
		}
#endif
	}

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(web_check))) {
		fprintf(configfile, "\tlisten_port=%s\n", web_port);
	}

	return TRUE;
}



gint write_objectives(GtkWidget * config_window, FILE * configfile)
{
	GtkWidget *obj_clist;
	gchar *alert, *success, *user_match, *users, *filter_type, *filter,
	    *objective;
	gint criticality;
	int user_match_type, filter_match_type;
	char filter_exp[TOKEN_SIZE], user_exp[TOKEN_SIZE],
	    events[TOKEN_SIZE];
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean iter_result = FALSE;
	char *tuser;

	fprintf(configfile, "\n[Objectives]\n");

	obj_clist = lookup_widget(config_window, "objectives_clist");
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(obj_clist));
	iter_result = gtk_tree_model_get_iter_first(model, &iter);
	if (iter_result) {
		gtk_tree_model_get(model, &iter, 0, &alert, 1, &success, 2,
				   &user_match, 3, &users, 4, &filter_type,
				   5, &filter, 6, &objective, -1);
	} else
		return TRUE;

	while (iter_result) {

		if (g_strcasecmp(alert, "Critical") == 0)
			criticality = CRITICALITY_CRITICAL;
		else if (g_strcasecmp(alert, "Priority") == 0)
			criticality = CRITICALITY_PRIORITY;
		else if (g_strcasecmp(alert, "Warning") == 0)
			criticality = CRITICALITY_WARNING;
		else if (g_strcasecmp(alert, "Information") == 0)
			criticality = CRITICALITY_INFO;
		else
			criticality = CRITICALITY_CLEAR;

		if (g_strcasecmp(success, "Success & Failure") == 0)
			strcpy(success, "*");
		else if (g_strcasecmp(success, "Success") == 0)
			strcpy(success, "Success");
		else
			strcpy(success, "Failure");

		if (g_strcasecmp
		    (objective,
		     "Read, write or create a file or directory") == 0)
			strcpy(events, OPEN_FILE);
		else if (g_strcasecmp
			 (objective, "Open a file for reading only") == 0)
			strcpy(events, OPEN_READ_FILE);
		else if (g_strcasecmp
			 (objective, "Remove a file or directory") == 0)
			strcpy(events, REMOVE_FILE);
		else if (g_strcasecmp(objective, "User logon or logoff") ==
			 0)
			strcpy(events, LOGON_LOGOFF);
		else if (g_strcasecmp
			 (objective,
			  "Modify system, file or directory attributes") ==
			 0)
			strcpy(events, ATTRIBUTES);
		else if (g_strcasecmp
			 (objective,
			  "Start or stop a program execution") == 0)
			strcpy(events, PROG_EXEC);
		else if (g_strcasecmp
			 (objective, "Change user or group identity") == 0)
			strcpy(events, USER_IDENT);
		else if (g_strcasecmp
			 (objective,
			  "Establish an outgoing network connection") == 0)
			strcpy(events, ESTABLISH_SOCKET);
		else {
			strcpy(events, objective);
		}

		if (g_strcasecmp(user_match, "All") == 0) {
			user_match_type = MATCH_ALL;
		} else if (g_strcasecmp(user_match, "Exclude") == 0) {
			user_match_type = MATCH_EXCLUDE;
		} else {
			user_match_type = MATCH_SELECTED;
		}

		SetMatch(users, user_exp, user_match_type);

		if (g_strcasecmp(filter_type, "Full") == 0)
			filter_match_type = MATCH_EXACT;
		else if (g_strcasecmp(filter_type, "Partial") == 0)
			filter_match_type = MATCH_PARTIAL;
		else
			filter_match_type = MATCH_REGEXP;
		SetMatch(filter, filter_exp, filter_match_type);

		// PRINTING TO THE CONFIG FILE STARTS HERE
		fprintf(configfile, "\tcriticality=%d", criticality);
		fprintf(configfile, "\tevent=%s", events);
		fprintf(configfile, "\treturn=%s", success);

		tuser = user_exp;
		while (*tuser) {
			if (*tuser == ',') {
				*tuser = '|';
			}
			tuser++;
		}

		if (user_match_type == MATCH_EXCLUDE) {
			fprintf(configfile, "\tuser!=%s", user_exp);
		} else {
			fprintf(configfile, "\tuser=%s", user_exp);
		}
		fprintf(configfile, "\tmatch=%s\n", filter_exp);

		iter_result = gtk_tree_model_iter_next(model, &iter);
		if (iter_result)
			gtk_tree_model_get(model, &iter, 0, &alert, 1,
					   &success, 2, &user_match, 3,
					   &users, 4, &filter_type, 5,
					   &filter, 6, &objective, -1);
	}

	return TRUE;
}


void SetMatch(const char *string, char *buffer, int type)
{

	if (type == MATCH_PARTIAL) {
		*buffer = '.';
		buffer++;
		*buffer = '*';
		buffer++;
	} else if (type == MATCH_EXACT) {
		*buffer = '^';
		buffer++;
	} else if (type == MATCH_SELECTED || type == MATCH_EXCLUDE) {
		*buffer = '^';
		buffer++;
		*buffer = '(';
		buffer++;
	} else if (type == MATCH_ALL) {
		strcpy(buffer, ".*");
		return;
	}
	// Expand regular expression characters.
	while (*string) {
		if (type != MATCH_REGEXP && type != MATCH_ALL) {
			if (*string == '.' || *string == '['
			    || *string == ']' || *string == '$'
			    || *string == '(' || *string == ')'
			    || *string == '|' || *string == '*'
			    || *string == '+' || *string == '?'
			    || *string == '{' || *string == '}'
			    || *string == '\\') {
				*buffer = '\\';
				buffer++;
			}
		}
		if (type == MATCH_SELECTED || type == MATCH_EXCLUDE) {
			// Replace pipes with commas
			if (*string == ',') {
				*buffer = '|';
				buffer++;
			} else {
				*buffer = *string;
				buffer++;
			}
		} else {
			*buffer = *string;
			buffer++;
		}
		string++;
	}
	if (type == MATCH_PARTIAL) {
		*buffer = '.';
		buffer++;
		*buffer = '*';
		buffer++;
	} else if (type == MATCH_EXACT) {
		*buffer = '$';
		buffer++;
	} else if (type == MATCH_SELECTED || type == MATCH_EXCLUDE) {
		*buffer = ')';
		buffer++;
		*buffer = '$';
		buffer++;
	} else if (type == MATCH_ALL) {
		*buffer = '.';
		buffer++;
		*buffer = '*';
		buffer++;
	}
	*buffer = '\0';
}
